"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.contentFingerprint = exports.fingerprint = void 0;
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const ignore_1 = require("./ignore");
const options_1 = require("./options");
const utils_1 = require("./utils");
const BUFFER_SIZE = 8 * 1024;
const CTRL_SOH = '\x01';
const CTRL_SOT = '\x02';
const CTRL_ETX = '\x03';
const CR = '\r';
const LF = '\n';
const CRLF = `${CR}${LF}`;
/**
 * Produces fingerprint based on the contents of a single file or an entire directory tree.
 *
 * Line endings are converted from CRLF to LF.
 *
 * The fingerprint will also include:
 * 1. An extra string if defined in `options.extra`.
 * 2. The symlink follow mode value.
 *
 * @param fileOrDirectory The directory or file to fingerprint
 * @param options Fingerprinting options
 */
function fingerprint(fileOrDirectory, options = {}) {
    const hash = crypto.createHash('sha256');
    _hashField(hash, 'options.extra', options.extraHash || '');
    const follow = options.follow || options_1.SymlinkFollowMode.EXTERNAL;
    _hashField(hash, 'options.follow', follow);
    // Resolve symlinks in the initial path (for example, the root directory
    // might be symlinked). It's important that we know the absolute path, so we
    // can judge if further symlinks inside the target directory are within the
    // target or not (if we don't resolve, we would test w.r.t. the wrong path).
    fileOrDirectory = fs.realpathSync(fileOrDirectory);
    const isDir = fs.statSync(fileOrDirectory).isDirectory();
    const rootDirectory = isDir
        ? fileOrDirectory
        : path.dirname(fileOrDirectory);
    const ignoreMode = options.ignoreMode || options_1.IgnoreMode.GLOB;
    if (ignoreMode != options_1.IgnoreMode.GLOB) {
        _hashField(hash, 'options.ignoreMode', ignoreMode);
    }
    const ignoreStrategy = ignore_1.IgnoreStrategy.fromCopyOptions(options, fileOrDirectory);
    _processFileOrDirectory(fileOrDirectory, isDir);
    return hash.digest('hex');
    function _processFileOrDirectory(symbolicPath, isRootDir = false, realPath = symbolicPath) {
        const relativePath = path.relative(fileOrDirectory, symbolicPath);
        if (!isRootDir && ignoreStrategy.ignores(symbolicPath)) {
            return;
        }
        const stat = fs.lstatSync(realPath);
        if (stat.isSymbolicLink()) {
            const linkTarget = fs.readlinkSync(realPath);
            const resolvedLinkTarget = path.resolve(path.dirname(realPath), linkTarget);
            if (utils_1.shouldFollow(follow, rootDirectory, resolvedLinkTarget)) {
                _processFileOrDirectory(symbolicPath, false, resolvedLinkTarget);
            }
            else {
                _hashField(hash, `link:${relativePath}`, linkTarget);
            }
        }
        else if (stat.isFile()) {
            _hashField(hash, `file:${relativePath}`, contentFingerprint(realPath));
        }
        else if (stat.isDirectory()) {
            for (const item of fs.readdirSync(realPath).sort()) {
                _processFileOrDirectory(path.join(symbolicPath, item), false, path.join(realPath, item));
            }
        }
        else {
            throw new Error(`Unable to hash ${symbolicPath}: it is neither a file nor a directory`);
        }
    }
}
exports.fingerprint = fingerprint;
function contentFingerprint(file) {
    const hash = crypto.createHash('sha256');
    const buffer = Buffer.alloc(BUFFER_SIZE);
    // eslint-disable-next-line no-bitwise
    const fd = fs.openSync(file, fs.constants.O_DSYNC | fs.constants.O_RDONLY | fs.constants.O_SYNC);
    let size = 0;
    let isBinary = false;
    let lastStr = '';
    let read = 0;
    try {
        while ((read = fs.readSync(fd, buffer, 0, BUFFER_SIZE, null)) !== 0) {
            const slicedBuffer = buffer.slice(0, read);
            // Detect if file is binary by checking the first 8k bytes for the
            // null character (git like implementation)
            if (size === 0) {
                isBinary = slicedBuffer.indexOf(0) !== -1;
            }
            let dataBuffer = slicedBuffer;
            if (!isBinary) { // Line endings normalization (CRLF -> LF)
                const str = buffer.slice(0, read).toString();
                // We are going to normalize line endings to LF. So if the current
                // buffer ends with CR, it could be that the next one starts with
                // LF so we need to save it for later use.
                if (new RegExp(`${CR}$`).test(str)) {
                    lastStr += str;
                    continue;
                }
                const data = lastStr + str;
                const normalizedData = data.replace(new RegExp(CRLF, 'g'), LF);
                dataBuffer = Buffer.from(normalizedData);
                lastStr = '';
            }
            size += dataBuffer.length;
            hash.update(dataBuffer);
        }
        if (lastStr) {
            hash.update(Buffer.from(lastStr));
        }
    }
    finally {
        fs.closeSync(fd);
    }
    return `${size}:${hash.digest('hex')}`;
}
exports.contentFingerprint = contentFingerprint;
function _hashField(hash, header, value) {
    hash.update(CTRL_SOH).update(header).update(CTRL_SOT).update(value).update(CTRL_ETX);
}
//# sourceMappingURL=data:application/json;base64,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