"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedCodeDeployedFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_synthetics_alpha_1 = require("@aws-cdk/aws-synthetics-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_codedeploy_1 = require("aws-cdk-lib/aws-codedeploy");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_ecs_patterns_1 = require("aws-cdk-lib/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const api_canary_1 = require("../api-canary");
const ecs_appspec_1 = require("../ecs-appspec");
const ecs_deployment_1 = require("../ecs-deployment");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer and deployed by CodeDeploy.
 */
class ApplicationLoadBalancedCodeDeployedFargateService extends aws_ecs_patterns_1.ApplicationLoadBalancedFargateService {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedCodeDeployedFargateService class.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            deploymentController: {
                type: aws_ecs_1.DeploymentControllerType.CODE_DEPLOY,
            },
        });
        if (props.deregistrationDelay) {
            this.targetGroup.setAttribute('deregistration_delay.timeout_seconds', props.deregistrationDelay.toSeconds().toString());
        }
        if (props.healthCheck) {
            this.targetGroup.configureHealthCheck(props.healthCheck);
        }
        this.accessLogBucket = props.accessLogBucket ??
            new aws_s3_1.Bucket(this, 'AccessLogBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                enforceSSL: true,
            });
        this.loadBalancer.logAccessLogs(this.accessLogBucket, props.accessLogPrefix);
        const alarms = [];
        if (props.responseTimeAlarmThreshold) {
            const responseTimeAlarm = new aws_cloudwatch_1.Alarm(this, 'ResponseTimeAlarm', {
                metric: this.loadBalancer.metricTargetResponseTime({
                    period: aws_cdk_lib_1.Duration.minutes(1),
                    statistic: 'p95',
                }),
                evaluationPeriods: 2,
                threshold: props.responseTimeAlarmThreshold.toSeconds(),
                comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            });
            alarms.push(responseTimeAlarm);
        }
        const protocol = props.protocol ?? (props.certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
        if (props.apiTestSteps?.length) {
            this.apiCanary = new api_canary_1.ApiCanary(this, 'Canary', {
                baseUrl: `${protocol.toLowerCase()}://${this.loadBalancer.loadBalancerDnsName}`,
                durationAlarmThreshold: props.apiCanaryTimeout,
                schedule: aws_synthetics_alpha_1.Schedule.rate(props.apiCanarySchedule ?? aws_cdk_lib_1.Duration.minutes(5)),
                threadCount: props.apiCanaryThreadCount,
                steps: props.apiTestSteps,
            });
            this.apiCanary.node.addDependency(this.service);
            alarms.push(this.apiCanary.successAlarm);
            if (this.apiCanary.durationAlarm) {
                alarms.push(this.apiCanary.durationAlarm);
            }
        }
        this.healthAlarm = new aws_cloudwatch_1.CompositeAlarm(this, 'HealthAlarm', {
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...alarms),
        });
        let testPort;
        if (props.listenerPort) {
            testPort = props.listenerPort + 1;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            testPort = 8080;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            testPort = 8443;
        }
        else {
            throw new Error('Unable to determine port for test listener');
        }
        this.testListener = this.loadBalancer.addListener('TestListener', {
            protocol,
            port: testPort,
            open: props.openListener ?? true,
            sslPolicy: props.sslPolicy,
        });
        this.greenTargetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(this, 'GreenTargetGroup', {
            vpc: this.cluster.vpc,
            port: testPort,
            protocol: props.targetProtocol ?? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            protocolVersion: props.protocolVersion,
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            targetType: aws_elasticloadbalancingv2_1.TargetType.IP,
        });
        this.testListener.addTargetGroups('ECS', {
            targetGroups: [this.greenTargetGroup],
        });
        this.application = new aws_codedeploy_1.EcsApplication(this, 'Application');
        this.deploymentGroup = new aws_codedeploy_1.EcsDeploymentGroup(this, 'DeploymentGroup', {
            application: this.application,
            alarms: [this.healthAlarm],
            service: this.service,
            blueGreenDeploymentConfig: {
                blueTargetGroup: this.targetGroup,
                greenTargetGroup: this.greenTargetGroup,
                listener: this.listener,
                testListener: this.testListener,
                terminationWaitTime: props.terminationWaitTime ?? aws_cdk_lib_1.Duration.minutes(10),
            },
            deploymentConfig: props.deploymentConfig ?? aws_codedeploy_1.EcsDeploymentConfig.ALL_AT_ONCE,
            autoRollback: {
                stoppedDeployment: true,
            },
        });
        this.deployment = new ecs_deployment_1.EcsDeployment({
            deploymentGroup: this.deploymentGroup,
            timeout: props.deploymentTimeout ?? aws_cdk_lib_1.Duration.minutes(60),
            appspec: new ecs_appspec_1.EcsAppSpec({
                taskDefinition: this.taskDefinition,
                containerName: this.taskDefinition.defaultContainer.containerName,
                containerPort: this.taskDefinition.defaultContainer.containerPort,
            }),
        });
    }
    addServiceAsTarget(service) {
        super.addServiceAsTarget(service);
    }
}
exports.ApplicationLoadBalancedCodeDeployedFargateService = ApplicationLoadBalancedCodeDeployedFargateService;
_a = JSII_RTTI_SYMBOL_1;
ApplicationLoadBalancedCodeDeployedFargateService[_a] = { fqn: "@cdklabs/cdk-ecs-codedeploy.ApplicationLoadBalancedCodeDeployedFargateService", version: "0.0.16" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbG9hZC1iYWxhbmNlZC1jb2RlZGVwbG95ZWQtZmFyZ2F0ZS1zZXJ2aWNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2Vjcy1wYXR0ZXJucy9hcHBsaWNhdGlvbi1sb2FkLWJhbGFuY2VkLWNvZGVkZXBsb3llZC1mYXJnYXRlLXNlcnZpY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSx3RUFBeUQ7QUFDekQsNkNBQXVDO0FBQ3ZDLCtEQUEwRztBQUMxRywrREFBMkg7QUFDM0gsaURBQTRFO0FBQzVFLG1FQUFpSTtBQUNqSSx1RkFBbUo7QUFDbkosK0NBQTBGO0FBRTFGLDhDQUF1RDtBQUN2RCxnREFBNEM7QUFDNUMsc0RBQWtEO0FBNEZsRDs7R0FFRztBQUNILE1BQWEsaURBQWtELFNBQVEsd0RBQXFDO0lBeUMxRzs7T0FFRztJQUNILFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBNkQ7UUFDckcsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixHQUFHLEtBQUs7WUFDUixvQkFBb0IsRUFBRTtnQkFDcEIsSUFBSSxFQUFFLGtDQUF3QixDQUFDLFdBQVc7YUFDM0M7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsRUFBRTtZQUM3QixJQUFJLENBQUMsV0FBVyxDQUFDLFlBQVksQ0FBQyxzQ0FBc0MsRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsU0FBUyxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztTQUN6SDtRQUNELElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRTtZQUNyQixJQUFJLENBQUMsV0FBVyxDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUMxRDtRQUNELElBQUksQ0FBQyxlQUFlLEdBQUcsS0FBSyxDQUFDLGVBQWU7WUFDMUMsSUFBSSxlQUFNLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO2dCQUNsQyxVQUFVLEVBQUUseUJBQWdCLENBQUMsVUFBVTtnQkFDdkMsaUJBQWlCLEVBQUUsMEJBQWlCLENBQUMsU0FBUztnQkFDOUMsVUFBVSxFQUFFLElBQUk7YUFDakIsQ0FBQyxDQUFDO1FBRUwsSUFBSSxDQUFDLFlBQVksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUM7UUFFN0UsTUFBTSxNQUFNLEdBQWEsRUFBRSxDQUFDO1FBQzVCLElBQUksS0FBSyxDQUFDLDBCQUEwQixFQUFFO1lBQ3BDLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxzQkFBSyxDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRTtnQkFDN0QsTUFBTSxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsd0JBQXdCLENBQUM7b0JBQ2pELE1BQU0sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7b0JBQzNCLFNBQVMsRUFBRSxLQUFLO2lCQUNqQixDQUFDO2dCQUNGLGlCQUFpQixFQUFFLENBQUM7Z0JBQ3BCLFNBQVMsRUFBRSxLQUFLLENBQUMsMEJBQTBCLENBQUMsU0FBUyxFQUFFO2dCQUN2RCxrQkFBa0IsRUFBRSxtQ0FBa0IsQ0FBQyxzQkFBc0I7YUFDOUQsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1NBQ2hDO1FBRUQsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLGdEQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsZ0RBQW1CLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFOUcsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFLE1BQU0sRUFBRTtZQUM5QixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksc0JBQVMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFO2dCQUM3QyxPQUFPLEVBQUUsR0FBRyxRQUFRLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsRUFBRTtnQkFDL0Usc0JBQXNCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtnQkFDOUMsUUFBUSxFQUFFLCtCQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDdkUsV0FBVyxFQUFFLEtBQUssQ0FBQyxvQkFBb0I7Z0JBQ3ZDLEtBQUssRUFBRSxLQUFLLENBQUMsWUFBWTthQUMxQixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBRWhELE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUN6QyxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsYUFBYSxFQUFFO2dCQUNoQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLENBQUM7YUFDM0M7U0FDRjtRQUNELElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSwrQkFBYyxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDekQsU0FBUyxFQUFFLDBCQUFTLENBQUMsS0FBSyxDQUFDLEdBQUcsTUFBTSxDQUFDO1NBQ3RDLENBQUMsQ0FBQztRQUVILElBQUksUUFBaUIsQ0FBQztRQUN0QixJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDdEIsUUFBUSxHQUFHLEtBQUssQ0FBQyxZQUFZLEdBQUcsQ0FBQyxDQUFDO1NBQ25DO2FBQU0sSUFBSSxRQUFRLEtBQUssZ0RBQW1CLENBQUMsSUFBSSxFQUFFO1lBQ2hELFFBQVEsR0FBRyxJQUFJLENBQUM7U0FDakI7YUFBTSxJQUFJLFFBQVEsS0FBSyxnREFBbUIsQ0FBQyxLQUFLLEVBQUU7WUFDakQsUUFBUSxHQUFHLElBQUksQ0FBQztTQUNqQjthQUFNO1lBQ0wsTUFBTSxJQUFJLEtBQUssQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1NBQy9EO1FBRUQsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxjQUFjLEVBQUU7WUFDaEUsUUFBUTtZQUNSLElBQUksRUFBRSxRQUFRO1lBQ2QsSUFBSSxFQUFFLEtBQUssQ0FBQyxZQUFZLElBQUksSUFBSTtZQUNoQyxTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7U0FDM0IsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksbURBQXNCLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQzNFLEdBQUcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUc7WUFDckIsSUFBSSxFQUFFLFFBQVE7WUFDZCxRQUFRLEVBQUUsS0FBSyxDQUFDLGNBQWMsSUFBSSxnREFBbUIsQ0FBQyxJQUFJO1lBQzFELGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZTtZQUN0QyxtQkFBbUIsRUFBRSxLQUFLLENBQUMsbUJBQW1CO1lBQzlDLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixVQUFVLEVBQUUsdUNBQVUsQ0FBQyxFQUFFO1NBQzFCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxZQUFZLENBQUMsZUFBZSxDQUFDLEtBQUssRUFBRTtZQUN2QyxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUM7U0FDdEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLCtCQUFjLENBQUMsSUFBSSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQzNELElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxtQ0FBa0IsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDckUsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXO1lBQzdCLE1BQU0sRUFBRSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUM7WUFDMUIsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPO1lBQ3JCLHlCQUF5QixFQUFFO2dCQUN6QixlQUFlLEVBQUUsSUFBSSxDQUFDLFdBQVc7Z0JBQ2pDLGdCQUFnQixFQUFFLElBQUksQ0FBQyxnQkFBZ0I7Z0JBQ3ZDLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtnQkFDdkIsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO2dCQUMvQixtQkFBbUIsRUFBRSxLQUFLLENBQUMsbUJBQW1CLElBQUksc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO2FBQ3ZFO1lBQ0QsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixJQUFJLG9DQUFtQixDQUFDLFdBQVc7WUFDM0UsWUFBWSxFQUFFO2dCQUNaLGlCQUFpQixFQUFFLElBQUk7YUFDeEI7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksOEJBQWEsQ0FBQztZQUNsQyxlQUFlLEVBQUUsSUFBSSxDQUFDLGVBQWU7WUFDckMsT0FBTyxFQUFFLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDeEQsT0FBTyxFQUFFLElBQUksd0JBQVUsQ0FBQztnQkFDdEIsY0FBYyxFQUFFLElBQUksQ0FBQyxjQUFjO2dCQUNuQyxhQUFhLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBaUIsQ0FBQyxhQUFhO2dCQUNsRSxhQUFhLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBaUIsQ0FBQyxhQUFhO2FBQ25FLENBQUM7U0FDSCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRVMsa0JBQWtCLENBQUMsT0FBb0I7UUFDL0MsS0FBSyxDQUFDLGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3BDLENBQUM7O0FBcktILDhHQXNLQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IFNjaGVkdWxlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLXN5bnRoZXRpY3MtYWxwaGEnO1xuaW1wb3J0IHsgRHVyYXRpb24gfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBBbGFybSwgQWxhcm1SdWxlLCBDb21wYXJpc29uT3BlcmF0b3IsIENvbXBvc2l0ZUFsYXJtLCBJQWxhcm0gfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgeyBFY3NBcHBsaWNhdGlvbiwgRWNzRGVwbG95bWVudENvbmZpZywgRWNzRGVwbG95bWVudEdyb3VwLCBJRWNzRGVwbG95bWVudENvbmZpZyB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1jb2RlZGVwbG95JztcbmltcG9ydCB7IEJhc2VTZXJ2aWNlLCBEZXBsb3ltZW50Q29udHJvbGxlclR5cGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWNzJztcbmltcG9ydCB7IEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VkRmFyZ2F0ZVNlcnZpY2UsIEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VkRmFyZ2F0ZVNlcnZpY2VQcm9wcyB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lY3MtcGF0dGVybnMnO1xuaW1wb3J0IHsgQXBwbGljYXRpb25MaXN0ZW5lciwgQXBwbGljYXRpb25Qcm90b2NvbCwgQXBwbGljYXRpb25UYXJnZXRHcm91cCwgSGVhbHRoQ2hlY2ssIFRhcmdldFR5cGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mic7XG5pbXBvcnQgeyBCbG9ja1B1YmxpY0FjY2VzcywgQnVja2V0LCBCdWNrZXRFbmNyeXB0aW9uLCBJQnVja2V0IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQXBpVGVzdFN0ZXAsIEFwaUNhbmFyeSB9IGZyb20gJy4uL2FwaS1jYW5hcnknO1xuaW1wb3J0IHsgRWNzQXBwU3BlYyB9IGZyb20gJy4uL2Vjcy1hcHBzcGVjJztcbmltcG9ydCB7IEVjc0RlcGxveW1lbnQgfSBmcm9tICcuLi9lY3MtZGVwbG95bWVudCc7XG5cbi8qKlxuICogVGhlIHByb3BlcnRpZXMgZm9yIHRoZSBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlZENvZGVEZXBsb3llZEZhcmdhdGVTZXJ2aWNlIHNlcnZpY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRDb2RlRGVwbG95ZWRGYXJnYXRlU2VydmljZVByb3BzIGV4dGVuZHMgQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRGYXJnYXRlU2VydmljZVByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSB0aW1lb3V0IGZvciBhIENvZGVEZXBsb3kgZGVwbG95bWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSA2MCBtaW51dGVzXG4gICAqL1xuICByZWFkb25seSBkZXBsb3ltZW50VGltZW91dD86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgdGltZSB0byB3YWl0IGJlZm9yZSB0ZXJtaW5hdGluZyB0aGUgb3JpZ2luYWwgKGJsdWUpIHRhc2sgc2V0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDEwIG1pbnV0ZXNcbiAgICovXG4gIHJlYWRvbmx5IHRlcm1pbmF0aW9uV2FpdFRpbWU/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIGRlcGxveW1lbnQgY29uZmlndXJhdGlvbiB0byB1c2UgZm9yIHRoZSBkZXBsb3ltZW50IGdyb3VwLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEVjc0RlcGxveW1lbnRDb25maWcuQUxMX0FUX09OQ0VcbiAgICovXG4gIHJlYWRvbmx5IGRlcGxveW1lbnRDb25maWc/OiBJRWNzRGVwbG95bWVudENvbmZpZztcblxuICAvKipcbiAgICogVGhlIGFtb3VudCBvZiB0aW1lIGZvciBFTEIgdG8gd2FpdCBiZWZvcmUgY2hhbmdpbmcgdGhlIHN0YXRlIG9mIGEgZGVyZWdpc3RlcmluZyB0YXJnZXRcbiAgICogZnJvbSAnZHJhaW5pbmcnIHRvICd1bnVzZWQnLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDMwMCBzZWNvbmRzXG4gICAqL1xuICByZWFkb25seSBkZXJlZ2lzdHJhdGlvbkRlbGF5PzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBoZWFsdGhjaGVjayB0byBjb25maWd1cmUgb24gdGhlIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZXIgdGFyZ2V0IGdyb3Vwcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBoZWFsdGggY2hlY2sgaXMgY29uZmlndXJlZFxuICAgKi9cbiAgcmVhZG9ubHkgaGVhbHRoQ2hlY2s/OiBIZWFsdGhDaGVjaztcblxuICAvKipcbiAgICogVGhlIGJ1Y2tldCB0byB1c2UgZm9yIGFjY2VzcyBsb2dzIGZyb20gdGhlIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYSBuZXcgUzMgYnVja2V0IHdpbGwgYmUgY3JlYXRlZFxuICAgKi9cbiAgcmVhZG9ubHkgYWNjZXNzTG9nQnVja2V0PzogSUJ1Y2tldDtcblxuICAvKipcbiAgICogVGhlIHByZWZpeCB0byB1c2UgZm9yIGFjY2VzcyBsb2dzIGZyb20gdGhlIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgYWNjZXNzTG9nUHJlZml4Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgdGhyZXNob2xkIGZvciByZXNwb25zZSB0aW1lIGFsYXJtLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGFsYXJtIHdpbGwgYmUgY3JlYXRlZFxuICAgKi9cbiAgcmVhZG9ubHkgcmVzcG9uc2VUaW1lQWxhcm1UaHJlc2hvbGQ/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiB0aHJlYWRzIHRvIHJ1biBjb25jdXJyZW50bHkgZm9yIHRoZSBzeW50aGV0aWMgdGVzdC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSAyMFxuICAgKi9cbiAgcmVhZG9ubHkgYXBpQ2FuYXJ5VGhyZWFkQ291bnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBmcmVxdWVuY3kgZm9yIHJ1bm5pbmcgdGhlIGFwaSBjYW5hcmllcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSA1IG1pbnV0ZXNcbiAgICovXG4gIHJlYWRvbmx5IGFwaUNhbmFyeVNjaGVkdWxlPzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSB0aHJlc2hvbGQgZm9yIGhvdyBsb25nIGEgYXBpIGNhbmFyeSBjYW4gdGFrZSB0byBydW4uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gYWxhcm0gaXMgY3JlYXRlZCBmb3IgdGVzdCBkdXJhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgYXBpQ2FuYXJ5VGltZW91dD86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgc3RlcHMgdG8gcnVuIGluIHRoZSBjYW5hcnkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gc3ludGhldGljIHRlc3Qgd2lsbCBiZSBjcmVhdGVkXG4gICAqL1xuICByZWFkb25seSBhcGlUZXN0U3RlcHM/OiBBcGlUZXN0U3RlcFtdO1xufVxuXG4vKipcbiAqIEEgRmFyZ2F0ZSBzZXJ2aWNlIHJ1bm5pbmcgb24gYW4gRUNTIGNsdXN0ZXIgZnJvbnRlZCBieSBhbiBhcHBsaWNhdGlvbiBsb2FkIGJhbGFuY2VyIGFuZCBkZXBsb3llZCBieSBDb2RlRGVwbG95LlxuICovXG5leHBvcnQgY2xhc3MgQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRDb2RlRGVwbG95ZWRGYXJnYXRlU2VydmljZSBleHRlbmRzIEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VkRmFyZ2F0ZVNlcnZpY2Uge1xuICAvKipcbiAgICogQ29tcG9zaXRlIGFsYXJtIGZvciBtb25pdG9yaW5nIGhlYWx0aCBvZiBzZXJ2aWNlLlxuICAgKi9cbiAgaGVhbHRoQWxhcm06IElBbGFybTtcblxuICAvKipcbiAgICogQVBJIENhbmFyeSBmb3IgdGhlIHNlcnZpY2UuXG4gICAqL1xuICBhcGlDYW5hcnk/OiBBcGlDYW5hcnk7XG5cbiAgLyoqXG4gICAqIFRlc3QgbGlzdGVuZXIgdG8gdXNlIGZvciBDb2RlRGVwbG95IGRlcGxveW1lbnRzLlxuICAgKi9cbiAgdGVzdExpc3RlbmVyOiBBcHBsaWNhdGlvbkxpc3RlbmVyO1xuXG4gIC8qKlxuICAgKiBUZXN0IHRhcmdldCBncm91cCB0byB1c2UgZm9yIENvZGVEZXBsb3kgZGVwbG95bWVudHMuXG4gICAqL1xuICBncmVlblRhcmdldEdyb3VwOiBBcHBsaWNhdGlvblRhcmdldEdyb3VwO1xuXG4gIC8qKlxuICAgKiBTMyBCdWNrZXQgdXNlZCBmb3IgYWNjZXNzIGxvZ3MuXG4gICAqL1xuICBhY2Nlc3NMb2dCdWNrZXQ6IElCdWNrZXQ7XG5cbiAgLyoqXG4gICAqIENvZGVEZXBsb3kgYXBwbGljYXRpb24gZm9yIHRoaXMgc2VydmljZS5cbiAgICovXG4gIGFwcGxpY2F0aW9uOiBFY3NBcHBsaWNhdGlvbjtcblxuICAvKipcbiAgICogQ29kZURlcGxveSBkZXBsb3ltZW50IGdyb3VwIGZvciB0aGlzIHNlcnZpY2UuXG4gICAqL1xuICBkZXBsb3ltZW50R3JvdXA6IEVjc0RlcGxveW1lbnRHcm91cDtcblxuICAvKipcbiAgICogQ29kZURlcGxveSBkZXBsb3ltZW50IGZvciB0aGlzIHNlcnZpY2UuXG4gICAqL1xuICBkZXBsb3ltZW50OiBFY3NEZXBsb3ltZW50O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlZENvZGVEZXBsb3llZEZhcmdhdGVTZXJ2aWNlIGNsYXNzLlxuICAgKi9cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VkQ29kZURlcGxveWVkRmFyZ2F0ZVNlcnZpY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgLi4ucHJvcHMsXG4gICAgICBkZXBsb3ltZW50Q29udHJvbGxlcjoge1xuICAgICAgICB0eXBlOiBEZXBsb3ltZW50Q29udHJvbGxlclR5cGUuQ09ERV9ERVBMT1ksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgaWYgKHByb3BzLmRlcmVnaXN0cmF0aW9uRGVsYXkpIHtcbiAgICAgIHRoaXMudGFyZ2V0R3JvdXAuc2V0QXR0cmlidXRlKCdkZXJlZ2lzdHJhdGlvbl9kZWxheS50aW1lb3V0X3NlY29uZHMnLCBwcm9wcy5kZXJlZ2lzdHJhdGlvbkRlbGF5LnRvU2Vjb25kcygpLnRvU3RyaW5nKCkpO1xuICAgIH1cbiAgICBpZiAocHJvcHMuaGVhbHRoQ2hlY2spIHtcbiAgICAgIHRoaXMudGFyZ2V0R3JvdXAuY29uZmlndXJlSGVhbHRoQ2hlY2socHJvcHMuaGVhbHRoQ2hlY2spO1xuICAgIH1cbiAgICB0aGlzLmFjY2Vzc0xvZ0J1Y2tldCA9IHByb3BzLmFjY2Vzc0xvZ0J1Y2tldCA/P1xuICAgICAgbmV3IEJ1Y2tldCh0aGlzLCAnQWNjZXNzTG9nQnVja2V0Jywge1xuICAgICAgICBlbmNyeXB0aW9uOiBCdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gICAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBCbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgICAgIGVuZm9yY2VTU0w6IHRydWUsXG4gICAgICB9KTtcblxuICAgIHRoaXMubG9hZEJhbGFuY2VyLmxvZ0FjY2Vzc0xvZ3ModGhpcy5hY2Nlc3NMb2dCdWNrZXQsIHByb3BzLmFjY2Vzc0xvZ1ByZWZpeCk7XG5cbiAgICBjb25zdCBhbGFybXM6IElBbGFybVtdID0gW107XG4gICAgaWYgKHByb3BzLnJlc3BvbnNlVGltZUFsYXJtVGhyZXNob2xkKSB7XG4gICAgICBjb25zdCByZXNwb25zZVRpbWVBbGFybSA9IG5ldyBBbGFybSh0aGlzLCAnUmVzcG9uc2VUaW1lQWxhcm0nLCB7XG4gICAgICAgIG1ldHJpYzogdGhpcy5sb2FkQmFsYW5jZXIubWV0cmljVGFyZ2V0UmVzcG9uc2VUaW1lKHtcbiAgICAgICAgICBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMSksXG4gICAgICAgICAgc3RhdGlzdGljOiAncDk1JyxcbiAgICAgICAgfSksXG4gICAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAyLFxuICAgICAgICB0aHJlc2hvbGQ6IHByb3BzLnJlc3BvbnNlVGltZUFsYXJtVGhyZXNob2xkLnRvU2Vjb25kcygpLFxuICAgICAgICBjb21wYXJpc29uT3BlcmF0b3I6IENvbXBhcmlzb25PcGVyYXRvci5HUkVBVEVSX1RIQU5fVEhSRVNIT0xELFxuICAgICAgfSk7XG4gICAgICBhbGFybXMucHVzaChyZXNwb25zZVRpbWVBbGFybSk7XG4gICAgfVxuXG4gICAgY29uc3QgcHJvdG9jb2wgPSBwcm9wcy5wcm90b2NvbCA/PyAocHJvcHMuY2VydGlmaWNhdGUgPyBBcHBsaWNhdGlvblByb3RvY29sLkhUVFBTIDogQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQKTtcblxuICAgIGlmIChwcm9wcy5hcGlUZXN0U3RlcHM/Lmxlbmd0aCkge1xuICAgICAgdGhpcy5hcGlDYW5hcnkgPSBuZXcgQXBpQ2FuYXJ5KHRoaXMsICdDYW5hcnknLCB7XG4gICAgICAgIGJhc2VVcmw6IGAke3Byb3RvY29sLnRvTG93ZXJDYXNlKCl9Oi8vJHt0aGlzLmxvYWRCYWxhbmNlci5sb2FkQmFsYW5jZXJEbnNOYW1lfWAsXG4gICAgICAgIGR1cmF0aW9uQWxhcm1UaHJlc2hvbGQ6IHByb3BzLmFwaUNhbmFyeVRpbWVvdXQsXG4gICAgICAgIHNjaGVkdWxlOiBTY2hlZHVsZS5yYXRlKHByb3BzLmFwaUNhbmFyeVNjaGVkdWxlID8/IER1cmF0aW9uLm1pbnV0ZXMoNSkpLFxuICAgICAgICB0aHJlYWRDb3VudDogcHJvcHMuYXBpQ2FuYXJ5VGhyZWFkQ291bnQsXG4gICAgICAgIHN0ZXBzOiBwcm9wcy5hcGlUZXN0U3RlcHMsXG4gICAgICB9KTtcblxuICAgICAgdGhpcy5hcGlDYW5hcnkubm9kZS5hZGREZXBlbmRlbmN5KHRoaXMuc2VydmljZSk7XG5cbiAgICAgIGFsYXJtcy5wdXNoKHRoaXMuYXBpQ2FuYXJ5LnN1Y2Nlc3NBbGFybSk7XG4gICAgICBpZiAodGhpcy5hcGlDYW5hcnkuZHVyYXRpb25BbGFybSkge1xuICAgICAgICBhbGFybXMucHVzaCh0aGlzLmFwaUNhbmFyeS5kdXJhdGlvbkFsYXJtKTtcbiAgICAgIH1cbiAgICB9XG4gICAgdGhpcy5oZWFsdGhBbGFybSA9IG5ldyBDb21wb3NpdGVBbGFybSh0aGlzLCAnSGVhbHRoQWxhcm0nLCB7XG4gICAgICBhbGFybVJ1bGU6IEFsYXJtUnVsZS5hbnlPZiguLi5hbGFybXMpLFxuICAgIH0pO1xuXG4gICAgbGV0IHRlc3RQb3J0IDogbnVtYmVyO1xuICAgIGlmIChwcm9wcy5saXN0ZW5lclBvcnQpIHtcbiAgICAgIHRlc3RQb3J0ID0gcHJvcHMubGlzdGVuZXJQb3J0ICsgMTtcbiAgICB9IGVsc2UgaWYgKHByb3RvY29sID09PSBBcHBsaWNhdGlvblByb3RvY29sLkhUVFApIHtcbiAgICAgIHRlc3RQb3J0ID0gODA4MDtcbiAgICB9IGVsc2UgaWYgKHByb3RvY29sID09PSBBcHBsaWNhdGlvblByb3RvY29sLkhUVFBTKSB7XG4gICAgICB0ZXN0UG9ydCA9IDg0NDM7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVW5hYmxlIHRvIGRldGVybWluZSBwb3J0IGZvciB0ZXN0IGxpc3RlbmVyJyk7XG4gICAgfVxuXG4gICAgdGhpcy50ZXN0TGlzdGVuZXIgPSB0aGlzLmxvYWRCYWxhbmNlci5hZGRMaXN0ZW5lcignVGVzdExpc3RlbmVyJywge1xuICAgICAgcHJvdG9jb2wsXG4gICAgICBwb3J0OiB0ZXN0UG9ydCxcbiAgICAgIG9wZW46IHByb3BzLm9wZW5MaXN0ZW5lciA/PyB0cnVlLFxuICAgICAgc3NsUG9saWN5OiBwcm9wcy5zc2xQb2xpY3ksXG4gICAgfSk7XG5cbiAgICB0aGlzLmdyZWVuVGFyZ2V0R3JvdXAgPSBuZXcgQXBwbGljYXRpb25UYXJnZXRHcm91cCh0aGlzLCAnR3JlZW5UYXJnZXRHcm91cCcsIHtcbiAgICAgIHZwYzogdGhpcy5jbHVzdGVyLnZwYyxcbiAgICAgIHBvcnQ6IHRlc3RQb3J0LFxuICAgICAgcHJvdG9jb2w6IHByb3BzLnRhcmdldFByb3RvY29sID8/IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUCxcbiAgICAgIHByb3RvY29sVmVyc2lvbjogcHJvcHMucHJvdG9jb2xWZXJzaW9uLFxuICAgICAgZGVyZWdpc3RyYXRpb25EZWxheTogcHJvcHMuZGVyZWdpc3RyYXRpb25EZWxheSxcbiAgICAgIGhlYWx0aENoZWNrOiBwcm9wcy5oZWFsdGhDaGVjayxcbiAgICAgIHRhcmdldFR5cGU6IFRhcmdldFR5cGUuSVAsXG4gICAgfSk7XG5cbiAgICB0aGlzLnRlc3RMaXN0ZW5lci5hZGRUYXJnZXRHcm91cHMoJ0VDUycsIHtcbiAgICAgIHRhcmdldEdyb3VwczogW3RoaXMuZ3JlZW5UYXJnZXRHcm91cF0sXG4gICAgfSk7XG5cbiAgICB0aGlzLmFwcGxpY2F0aW9uID0gbmV3IEVjc0FwcGxpY2F0aW9uKHRoaXMsICdBcHBsaWNhdGlvbicpO1xuICAgIHRoaXMuZGVwbG95bWVudEdyb3VwID0gbmV3IEVjc0RlcGxveW1lbnRHcm91cCh0aGlzLCAnRGVwbG95bWVudEdyb3VwJywge1xuICAgICAgYXBwbGljYXRpb246IHRoaXMuYXBwbGljYXRpb24sXG4gICAgICBhbGFybXM6IFt0aGlzLmhlYWx0aEFsYXJtXSxcbiAgICAgIHNlcnZpY2U6IHRoaXMuc2VydmljZSxcbiAgICAgIGJsdWVHcmVlbkRlcGxveW1lbnRDb25maWc6IHtcbiAgICAgICAgYmx1ZVRhcmdldEdyb3VwOiB0aGlzLnRhcmdldEdyb3VwLFxuICAgICAgICBncmVlblRhcmdldEdyb3VwOiB0aGlzLmdyZWVuVGFyZ2V0R3JvdXAsXG4gICAgICAgIGxpc3RlbmVyOiB0aGlzLmxpc3RlbmVyLFxuICAgICAgICB0ZXN0TGlzdGVuZXI6IHRoaXMudGVzdExpc3RlbmVyLFxuICAgICAgICB0ZXJtaW5hdGlvbldhaXRUaW1lOiBwcm9wcy50ZXJtaW5hdGlvbldhaXRUaW1lID8/IER1cmF0aW9uLm1pbnV0ZXMoMTApLFxuICAgICAgfSxcbiAgICAgIGRlcGxveW1lbnRDb25maWc6IHByb3BzLmRlcGxveW1lbnRDb25maWcgPz8gRWNzRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgIGF1dG9Sb2xsYmFjazoge1xuICAgICAgICBzdG9wcGVkRGVwbG95bWVudDogdHJ1ZSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgdGhpcy5kZXBsb3ltZW50ID0gbmV3IEVjc0RlcGxveW1lbnQoe1xuICAgICAgZGVwbG95bWVudEdyb3VwOiB0aGlzLmRlcGxveW1lbnRHcm91cCxcbiAgICAgIHRpbWVvdXQ6IHByb3BzLmRlcGxveW1lbnRUaW1lb3V0ID8/IER1cmF0aW9uLm1pbnV0ZXMoNjApLFxuICAgICAgYXBwc3BlYzogbmV3IEVjc0FwcFNwZWMoe1xuICAgICAgICB0YXNrRGVmaW5pdGlvbjogdGhpcy50YXNrRGVmaW5pdGlvbixcbiAgICAgICAgY29udGFpbmVyTmFtZTogdGhpcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyIS5jb250YWluZXJOYW1lLFxuICAgICAgICBjb250YWluZXJQb3J0OiB0aGlzLnRhc2tEZWZpbml0aW9uLmRlZmF1bHRDb250YWluZXIhLmNvbnRhaW5lclBvcnQsXG4gICAgICB9KSxcbiAgICB9KTtcbiAgfVxuXG4gIHByb3RlY3RlZCBhZGRTZXJ2aWNlQXNUYXJnZXQoc2VydmljZTogQmFzZVNlcnZpY2UpIHtcbiAgICBzdXBlci5hZGRTZXJ2aWNlQXNUYXJnZXQoc2VydmljZSk7XG4gIH1cbn1cbiJdfQ==