"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const aws_kinesisanalytics_1 = require("aws-cdk-lib/aws-kinesisanalytics");
const logs = require("aws-cdk-lib/aws-logs");
const core = require("aws-cdk-lib");
const environment_properties_1 = require("./private/environment-properties");
const flink_application_configuration_1 = require("./private/flink-application-configuration");
const validation_1 = require("./private/validation");
/**
 * Implements the functionality shared between CDK created and imported
 * IApplications.
 */
class ApplicationBase extends core.Resource {
    /** Implement the convenience `IApplication.addToPrincipalPolicy` method. */
    addToRolePolicy(policyStatement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(policyStatement);
            return true;
        }
        return false;
    }
    get connections() {
        if (!this._connections) {
            throw new Error('This Application isn\'t associated with a VPC. Provide a "vpc" prop when creating the Application or "securityGroups" when importing it.');
        }
        return this._connections;
    }
    /**
     * Return a CloudWatch metric associated with this Flink application.
     *
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/KinesisAnalytics',
            metricName,
            dimensionsMap: { Application: this.applicationName },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of Kinesis Processing Units that are used to run your stream
     * processing application. The average number of KPUs used each hour
     * determines the billing for your application.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricKpus(props) {
        return this.metric('KPUs', { statistic: 'Average', ...props });
    }
    /**
     * The time elapsed during an outage for failing/recovering jobs.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricDowntime(props) {
        return this.metric('downtime', { statistic: 'Average', ...props });
    }
    /**
     * The time that the job has been running without interruption.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricUptime(props) {
        return this.metric('uptime', { statistic: 'Average', ...props });
    }
    /**
     * The total number of times this job has fully restarted since it was
     * submitted. This metric does not measure fine-grained restarts.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricFullRestarts(props) {
        return this.metric('fullRestarts', { statistic: 'Sum', ...props });
    }
    /**
     * The number of times checkpointing has failed.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricNumberOfFailedCheckpoints(props) {
        return this.metric('numberOfFailedCheckpoints', { statistic: 'Sum', ...props });
    }
    /**
     * The time it took to complete the last checkpoint.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - maximum over 5 minutes
     */
    metricLastCheckpointDuration(props) {
        return this.metric('lastCheckpointDuration', { statistic: 'Maximum', ...props });
    }
    /**
     * The total size of the last checkpoint.
     *
     * Units: Bytes
     *
     * Reporting Level: Application
     *
     * @default - maximum over 5 minutes
     */
    metricLastCheckpointSize(props) {
        return this.metric('lastCheckpointSize', { statistic: 'Maximum', ...props });
    }
    /**
     * The overall percentage of CPU utilization across task managers. For
     * example, if there are five task managers, Kinesis Data Analytics publishes
     * five samples of this metric per reporting interval.
     *
     * Units: Percentage
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricCpuUtilization(props) {
        return this.metric('cpuUtilization', { statistic: 'Average', ...props });
    }
    /**
     * Overall heap memory utilization across task managers. For example, if there
     * are five task managers, Kinesis Data Analytics publishes five samples of
     * this metric per reporting interval.
     *
     * Units: Percentage
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricHeapMemoryUtilization(props) {
        return this.metric('heapMemoryUtilization', { statistic: 'Average', ...props });
    }
    /**
     * The total time spent performing old garbage collection operations.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricOldGenerationGCTime(props) {
        return this.metric('oldGenerationGCTime', { statistic: 'Sum', ...props });
    }
    /**
     * The total number of old garbage collection operations that have occurred
     * across all task managers.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricOldGenerationGCCount(props) {
        return this.metric('oldGenerationGCCount', { statistic: 'Sum', ...props });
    }
    /**
     * The total number of live threads used by the application.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricThreadsCount(props) {
        return this.metric('threadsCount', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator, or task has
     * received.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsIn(props) {
        return this.metric('numRecordsIn', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has received
     * per second.
     *
     * Units: Count/Second
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsInPerSecond(props) {
        return this.metric('numRecordsInPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has emitted.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsOut(props) {
        return this.metric('numRecordsOut', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has emitted
     * per second.
     *
     * Units: Count/Second
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsOutPerSecond(props) {
        return this.metric('numRecordsOutPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The number of records this operator or task has dropped due to arriving
     * late.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - sum over 5 minutes
     */
    metricNumLateRecordsDropped(props) {
        return this.metric('numLateRecordsDropped', { statistic: 'Sum', ...props });
    }
    /**
     * The last watermark this application/operator/task/thread has received.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - maximum over 5 minutes
     */
    metricCurrentInputWatermark(props) {
        return this.metric('currentInputWatermark', { statistic: 'Maximum', ...props });
    }
    /**
     * The last watermark this application/operator/task/thread has received.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - maximum over 5 minutes
     */
    metricCurrentOutputWatermark(props) {
        return this.metric('currentOutputWatermark', { statistic: 'Maximum', ...props });
    }
    /**
     * The amount of managed memory currently used.
     *
     * Units: Bytes
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryUsed(props) {
        return this.metric('managedMemoryUsed', { statistic: 'Average', ...props });
    }
    /**
     * The total amount of managed memory.
     *
     * Units: Bytes
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryTotal(props) {
        return this.metric('managedMemoryTotal', { statistic: 'Average', ...props });
    }
    /**
     * Derived from managedMemoryUsed/managedMemoryTotal.
     *
     * Units: Percentage
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryUtilization(props) {
        return this.metric('managedMemoryUtilization', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is idle (has no data to
     * process) per second. Idle time excludes back pressured time, so if the task
     * is back pressured it is not idle.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricIdleTimeMsPerSecond(props) {
        return this.metric('idleTimeMsPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is back pressured per
     * second.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricBackPressuredTimeMsPerSecond(props) {
        return this.metric('backPressuredTimeMsPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is busy (neither idle nor
     * back pressured) per second. Can be NaN, if the value could not be
     * calculated.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricBusyTimePerMsPerSecond(props) {
        return this.metric('busyTimePerMsPerSecond', { statistic: 'Average', ...props });
    }
}
/**
 * An imported Flink application.
 */
class Import extends ApplicationBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        // Imported applications have no associated role or grantPrincipal
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
        this.role = undefined;
        this.applicationArn = attrs.applicationArn;
        const applicationName = core.Stack.of(scope).splitArn(attrs.applicationArn, core.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!applicationName) {
            throw new Error(`applicationArn for fromApplicationArn (${attrs.applicationArn}) must include resource name`);
        }
        this.applicationName = applicationName;
        const securityGroups = attrs.securityGroups ?? [];
        if (securityGroups.length > 0) {
            this._connections = new ec2.Connections({ securityGroups });
        }
    }
}
/**
 * The L2 construct for Flink Kinesis Data Applications.
 *
 * @resource AWS::KinesisAnalyticsV2::Application
 *
 */
class Application extends ApplicationBase {
    /**
     * Import an existing Flink application defined outside of CDK code by
     * applicationName.
     */
    static fromApplicationName(scope, id, applicationName) {
        const applicationArn = core.Stack.of(scope).formatArn(applicationArnComponents(applicationName));
        return new Import(scope, id, { applicationArn });
    }
    /**
     * Import an existing application defined outside of CDK code by
     * applicationArn.
     */
    static fromApplicationArn(scope, id, applicationArn) {
        return new Import(scope, id, { applicationArn });
    }
    /**
     * Import an existing application defined outside of CDK code.
     */
    static fromApplicationAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_kinesisanalytics_flink_alpha_ApplicationAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromApplicationAttributes);
            }
            throw error;
        }
        return new Import(scope, id, {
            applicationArn: attrs.applicationArn,
            securityGroups: attrs.securityGroups,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, { physicalName: props.applicationName });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_kinesisanalytics_flink_alpha_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Application);
            }
            throw error;
        }
        (0, validation_1.validateFlinkApplicationProps)(props);
        this.role = props.role ?? new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('kinesisanalytics.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        // Permit metric publishing to CloudWatch
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudwatch:PutMetricData'],
            resources: ['*'],
        }));
        const code = props.code.bind(this);
        code.bucket.grantRead(this);
        let vpcConfigurations;
        if (props.vpc) {
            const securityGroups = props.securityGroups ?? [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                }),
            ];
            this._connections = new ec2.Connections({ securityGroups });
            const subnetSelection = props.vpcSubnets ?? {
                subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
            };
            vpcConfigurations = [{
                    securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
                    subnetIds: props.vpc.selectSubnets(subnetSelection).subnetIds,
                }];
        }
        const resource = new aws_kinesisanalytics_1.CfnApplicationV2(this, 'Resource', {
            applicationName: props.applicationName,
            runtimeEnvironment: props.runtime.value,
            serviceExecutionRole: this.role.roleArn,
            applicationConfiguration: {
                ...code.applicationCodeConfigurationProperty,
                environmentProperties: (0, environment_properties_1.environmentProperties)(props.propertyGroups),
                flinkApplicationConfiguration: (0, flink_application_configuration_1.flinkApplicationConfiguration)({
                    checkpointingEnabled: props.checkpointingEnabled,
                    checkpointInterval: props.checkpointInterval,
                    minPauseBetweenCheckpoints: props.minPauseBetweenCheckpoints,
                    logLevel: props.logLevel,
                    metricsLevel: props.metricsLevel,
                    autoScalingEnabled: props.autoScalingEnabled,
                    parallelism: props.parallelism,
                    parallelismPerKpu: props.parallelismPerKpu,
                }),
                applicationSnapshotConfiguration: {
                    snapshotsEnabled: props.snapshotsEnabled ?? true,
                },
                vpcConfigurations,
            },
        });
        resource.node.addDependency(this.role);
        const logGroup = props.logGroup ?? new logs.LogGroup(this, 'LogGroup');
        const logStream = new logs.LogStream(this, 'LogStream', { logGroup });
        /* Permit logging */
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogGroups'],
            resources: [
                core.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    arnFormat: core.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: '*',
                }),
            ],
        }));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogStreams'],
            resources: [logGroup.logGroupArn],
        }));
        const logStreamArn = `arn:${core.Aws.PARTITION}:logs:${core.Aws.REGION}:${core.Aws.ACCOUNT_ID}:log-group:${logGroup.logGroupName}:log-stream:${logStream.logStreamName}`;
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [logStreamArn],
        }));
        new aws_kinesisanalytics_1.CfnApplicationCloudWatchLoggingOptionV2(this, 'LoggingOption', {
            applicationName: resource.ref,
            cloudWatchLoggingOption: {
                logStreamArn,
            },
        });
        // Permissions required for VPC usage per:
        // https://docs.aws.amazon.com/kinesisanalytics/latest/java/vpc-permissions.html
        if (props.vpc) {
            this.role.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: [
                    'ec2:DescribeVpcs',
                    'ec2:DescribeSubnets',
                    'ec2:DescribeSecurityGroups',
                    'ec2:DescribeDhcpOptions',
                    'ec2:CreateNetworkInterface',
                    'ec2:CreateNetworkInterfacePermission',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DeleteNetworkInterface',
                ],
                resources: ['*'],
            }));
        }
        this.applicationName = this.getResourceNameAttribute(resource.ref);
        this.applicationArn = this.getResourceArnAttribute(core.Stack.of(this).formatArn(applicationArnComponents(resource.ref)), applicationArnComponents(this.physicalName));
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.DESTROY,
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-kinesisanalytics-flink-alpha.Application", version: "2.72.1-alpha.0" };
exports.Application = Application;
function applicationArnComponents(resourceName) {
    return {
        service: 'kinesisanalytics',
        resource: 'application',
        resourceName,
    };
}
//# sourceMappingURL=data:application/json;base64,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