"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateFlinkApplicationProps = void 0;
const core = require("aws-cdk-lib");
/**
 * Early validation for the props used to create FlinkApplications.
 */
function validateFlinkApplicationProps(props) {
    validateApplicationName(props.applicationName);
    validateParallelism(props.parallelism);
    validateParallelismPerKpu(props.parallelismPerKpu);
    validateVpcProps(props);
}
exports.validateFlinkApplicationProps = validateFlinkApplicationProps;
function validateApplicationName(applicationName) {
    if (applicationName === undefined || core.Token.isUnresolved(applicationName)) {
        return;
    }
    if (applicationName.length === 0) {
        throw new Error('applicationName cannot be empty. It must contain at least 1 character.');
    }
    if (!/^[a-zA-Z0-9_.-]+$/.test(applicationName)) {
        throw new Error(`applicationName may only contain letters, numbers, underscores, hyphens, and periods. Name: ${applicationName}`);
    }
    if (applicationName.length > 128) {
        throw new Error(`applicationName max length is 128. Name: ${applicationName} is ${applicationName.length} characters.`);
    }
}
function validateParallelism(parallelism) {
    if (parallelism === undefined || core.Token.isUnresolved(parallelism)) {
        return;
    }
    if (parallelism < 1) {
        throw new Error('parallelism must be at least 1');
    }
}
function validateParallelismPerKpu(parallelismPerKpu) {
    if (parallelismPerKpu === undefined || core.Token.isUnresolved(parallelismPerKpu)) {
        return;
    }
    if (parallelismPerKpu < 1) {
        throw new Error('parallelismPerKpu must be at least 1');
    }
}
function validateVpcProps({ vpc, securityGroups = [], vpcSubnets }) {
    if (!vpc) {
        if (vpcSubnets) {
            throw new Error('vpc prop required when passing vpcSubnets');
        }
        if (securityGroups.length > 0) {
            throw new Error('vpc prop required when passing securityGroups');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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