"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const policy_statement_1 = require("./policy-statement");
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
class Grant {
    constructor(props) {
        this.options = props.options;
        this.principalStatement = props.principalStatement;
        this.resourceStatement = props.resourceStatement;
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return props.policyDependable ? cdk.DependableTrait.get(props.policyDependable).dependencyRoots : [];
            },
        });
    }
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options) {
        var _a;
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        if (result.success) {
            return result;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        return new Grant({
            resourceStatement: statement,
            options,
            policyDependable: resourceResult.statementAdded ? (_a = resourceResult.policyDependable) !== null && _a !== void 0 ? _a : options.resource : undefined,
        });
    }
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options) {
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: options.resourceArns,
        });
        const addedToPrincipal = options.grantee.grantPrincipal.addToPrincipalPolicy(statement);
        if (!addedToPrincipal.statementAdded) {
            return new Grant({ principalStatement: undefined, options });
        }
        if (!addedToPrincipal.policyDependable) {
            throw new Error('Contract violation: when Principal returns statementAdded=true, it should return a dependable');
        }
        return new Grant({ principalStatement: statement, options, policyDependable: addedToPrincipal.policyDependable });
    }
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the principal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options) {
        var _a;
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.resourcePolicyPrincipal || options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        const resourceDependable = resourceResult.statementAdded ? (_a = resourceResult.policyDependable) !== null && _a !== void 0 ? _a : options.resource : undefined;
        return new Grant({
            principalStatement: statement,
            resourceStatement: result.resourceStatement,
            options,
            policyDependable: resourceDependable ? new CompositeDependable(result, resourceDependable) : result,
        });
    }
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee, _intent) {
        return new Grant({
            options: { grantee, actions: [], resourceArns: [] },
        });
    }
    /**
     * Whether the grant operation was successful
     */
    get success() {
        return this.principalStatement !== undefined || this.resourceStatement !== undefined;
    }
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess() {
        if (!this.success) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`${describeGrant(this.options)} could not be added on either identity or resource policy.`);
        }
    }
    /**
     * Make sure this grant is applied before the given constructs are deployed
     *
     * The same as construct.node.addDependency(grant), but slightly nicer to read.
     */
    applyBefore(...constructs) {
        for (const construct of constructs) {
            construct.node.addDependency(this);
        }
    }
}
exports.Grant = Grant;
function describeGrant(options) {
    return `Permissions for '${options.grantee}' to call '${options.actions}' on '${options.resourceArns}'`;
}
/**
 * Composite dependable
 *
 * Not as simple as eagerly getting the dependency roots from the
 * inner dependables, as they may be mutable so we need to defer
 * the query.
 */
class CompositeDependable {
    constructor(...dependables) {
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return Array.prototype.concat.apply([], dependables.map(d => cdk.DependableTrait.get(d).dependencyRoots));
            },
        });
    }
}
exports.CompositeDependable = CompositeDependable;
//# sourceMappingURL=data:application/json;base64,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