"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const path = require("path");
const RESOURCE_TYPE = 'Custom::AWSCDKOpenIdConnectProvider';
/**
 * IAM OIDC identity providers are entities in IAM that describe an external
 * identity provider (IdP) service that supports the OpenID Connect (OIDC)
 * standard, such as Google or Salesforce. You use an IAM OIDC identity provider
 * when you want to establish trust between an OIDC-compatible IdP and your AWS
 * account. This is useful when creating a mobile app or web application that
 * requires access to AWS resources, but you don't want to create custom sign-in
 * code or manage your own user identities.
 *
 * @see http://openid.net/connect
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc.html
 *
 * @experimental
 */
class OpenIdConnectProvider extends core_1.Construct {
    /**
     * Defines an OpenID Connect provider.
     * @param scope The definition scope
     * @param id Construct ID
     * @param props Initialization properties
     */
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: RESOURCE_TYPE,
            serviceToken: this.getOrCreateProvider(),
            properties: {
                ClientIDList: props.clientIds,
                ThumbprintList: props.thumbprints,
                Url: props.url,
            },
        });
        this.openIdConnectProviderArn = core_1.Token.asString(resource.ref);
    }
    /**
     * Imports an Open ID connect provider from an ARN.
     * @param scope The definition scope
     * @param id ID of the construct
     * @param openIdConnectProviderArn the ARN to import
     */
    static fromOpenIdConnectProviderArn(scope, id, openIdConnectProviderArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.openIdConnectProviderArn = openIdConnectProviderArn;
            }
        }
        return new Import(scope, id);
    }
    get stack() { return core_1.Stack.of(this); }
    getOrCreateProvider() {
        return core_1.CustomResourceProvider.getOrCreate(this, RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'oidc-provider'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                {
                    Effect: 'Allow',
                    Resource: '*',
                    Action: [
                        'iam:CreateOpenIDConnectProvider',
                        'iam:DeleteOpenIDConnectProvider',
                        'iam:UpdateOpenIDConnectProviderThumbprint',
                        'iam:AddClientIDToOpenIDConnectProvider',
                        'iam:RemoveClientIDFromOpenIDConnectProvider',
                    ],
                },
            ],
        });
    }
}
exports.OpenIdConnectProvider = OpenIdConnectProvider;
//# sourceMappingURL=data:application/json;base64,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