"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
const util_1 = require("./util");
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return { statementAdded: false };
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    /**
     * Returns a new PrincipalWithConditions using this principal as the base, with the
     * passed conditions added.
     *
     * When there is a value for the same operator and key in both the principal and the
     * conditions parameter, the value from the conditions parameter will be used.
     *
     * @returns a new PrincipalWithConditions object.
     */
    withConditions(conditions) {
        return new PrincipalWithConditions(this, conditions);
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * An IAM principal with additional conditions specifying when the policy is in effect.
 *
 * For more information about conditions, see:
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html
 */
class PrincipalWithConditions {
    constructor(principal, conditions) {
        this.principal = principal;
        this.grantPrincipal = this;
        this.assumeRoleAction = this.principal.assumeRoleAction;
        this.additionalConditions = conditions;
    }
    /**
     * Add a condition to the principal
     */
    addCondition(key, value) {
        const existingValue = this.conditions[key];
        this.conditions[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Adds multiple conditions to the principal
     *
     * Values from the conditions parameter will overwrite existing values with the same operator
     * and key.
     */
    addConditions(conditions) {
        Object.entries(conditions).forEach(([key, value]) => {
            this.addCondition(key, value);
        });
    }
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    get conditions() {
        return this.mergeConditions(this.principal.policyFragment.conditions, this.additionalConditions);
    }
    get policyFragment() {
        return new PrincipalPolicyFragment(this.principal.policyFragment.principalJson, this.conditions);
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(statement) {
        return this.principal.addToPrincipalPolicy(statement);
    }
    toString() {
        return this.principal.toString();
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    mergeConditions(principalConditions, additionalConditions) {
        const mergedConditions = {};
        Object.entries(principalConditions).forEach(([operator, condition]) => {
            mergedConditions[operator] = condition;
        });
        Object.entries(additionalConditions).forEach(([operator, condition]) => {
            // merge the conditions if one of the additional conditions uses an
            // operator that's already used by the principal's conditions merge the
            // inner structure.
            const existing = mergedConditions[operator];
            if (!existing) {
                mergedConditions[operator] = condition;
                return; // continue
            }
            // if either the existing condition or the new one contain unresolved
            // tokens, fail the merge. this is as far as we go at this point.
            if (cdk.Token.isUnresolved(condition) || cdk.Token.isUnresolved(existing)) {
                throw new Error(`multiple "${operator}" conditions cannot be merged if one of them contains an unresolved token`);
            }
            mergedConditions[operator] = { ...existing, ...condition };
        });
        return mergedConditions;
    }
}
exports.PrincipalWithConditions = PrincipalWithConditions;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    /**
     *
     * @param principalJson JSON of the "Principal" section in a policy statement
     * @param conditions conditions that need to be applied to this policy
     */
    constructor(principalJson, 
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
/**
 * Specify a principal by the Amazon Resource Name (ARN).
 * You can specify AWS accounts, IAM users, Federated SAML users, IAM roles, and specific assumed-role sessions.
 * You cannot specify IAM groups or instance profiles as principals
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
 */
class ArnPrincipal extends PrincipalBase {
    /**
     *
     * @param arn Amazon Resource Name (ARN) of the principal entity (i.e. arn:aws:iam::123456789012:user/user-name)
     */
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
/**
 * Specify AWS account ID as the principal entity in a policy to delegate authority to the account.
 */
class AccountPrincipal extends ArnPrincipal {
    /**
     *
     * @param accountId AWS account ID (i.e. 123456789012)
     */
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    /**
     *
     * @param service AWS service (i.e. sqs.amazonaws.com)
     */
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString(),
            ],
        }, this.opts.conditions);
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    /**
     *
     * @param organizationId The unique identifier (ID) of an organization (i.e. o-12345abcde)
     */
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy principal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    /**
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account.
     *   root user and IAM users for an account all see the same ID.
     *   (i.e. 79a59df900b949e55d96a1e698fbacedfd6e09d98eacf8f8d5218e7cd47ef2be)
     */
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
/**
 * Principal entity that represents a federated identity provider such as Amazon Cognito,
 * that can be used to provide temporary security credentials to users who have been authenticated.
 * Additional condition keys are available when the temporary security credentials are used to make a request.
 * You can use these keys to write policies that limit the access of federated users.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#condition-keys-wif
 */
class FederatedPrincipal extends PrincipalBase {
    /**
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito)
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
/**
 * A principal that represents a federated identity provider as Web Identity such as Cognito, Amazon,
 * Facebook, Google, etc.
 */
class WebIdentityPrincipal extends FederatedPrincipal {
    /**
     *
     * @param identityProvider identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito)
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(identityProvider, conditions = {}) {
        super(identityProvider, conditions !== null && conditions !== void 0 ? conditions : {}, 'sts:AssumeRoleWithWebIdentity');
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `WebIdentityPrincipal(${this.federated})`;
    }
}
exports.WebIdentityPrincipal = WebIdentityPrincipal;
/**
 * A principal that represents a federated identity provider as from a OpenID Connect provider.
 */
class OpenIdConnectPrincipal extends WebIdentityPrincipal {
    /**
     *
     * @param openIdConnectProvider OpenID Connect provider
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(openIdConnectProvider, conditions = {}) {
        super(openIdConnectProvider.openIdConnectProviderArn, conditions !== null && conditions !== void 0 ? conditions : {});
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `OpenIdConnectPrincipal(${this.federated})`;
    }
}
exports.OpenIdConnectPrincipal = OpenIdConnectPrincipal;
/**
 * Use the AWS account into which a stack is deployed as the principal entity in a policy
 */
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.account).toString());
    }
    toString() {
        return 'AccountRootPrincipal()';
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return 'AnyPrincipal()';
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
/**
 * Represents a principal that has multiple types of principals. A composite principal cannot
 * have conditions. i.e. multiple ServicePrincipals that form a composite principal
 */
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    /**
     * Adds IAM principals to the composite principal. Composite principals cannot have
     * conditions.
     *
     * @param principals IAM principals that will be added to the composite principal
     */
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error('Cannot add multiple principals with different "assumeRoleAction". ' +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error('Components of a CompositePrincipal must not have conditions. ' +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        return this.fn(cdk.Stack.of(context.scope));
    }
    toString() {
        return cdk.Token.asString(this);
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return '<unresolved-token>';
    }
}
class ServicePrincipalToken {
    constructor(service, opts) {
        this.service = service;
        this.opts = opts;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(ctx) {
        const region = this.opts.region || cdk.Stack.of(ctx.scope).region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, cdk.Aws.URL_SUFFIX);
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: this.service,
        });
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return `<${this.service}>`;
    }
}
//# sourceMappingURL=data:application/json;base64,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