"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const MAX_POLICY_NAME_LEN = 128;
function undefinedIfEmpty(f) {
    return core_1.Lazy.listValue({ produce: () => {
            const array = f();
            return (array && array.length > 0) ? array : undefined;
        } });
}
exports.undefinedIfEmpty = undefinedIfEmpty;
/**
 * Used to generate a unique policy name based on the policy resource construct.
 * The logical ID of the resource is a great candidate as long as it doesn't exceed
 * 128 characters, so we take the last 128 characters (in order to make sure the hash
 * is there).
 */
function generatePolicyName(scope, logicalId) {
    // as logicalId is itself a Token, resolve it first
    const resolvedLogicalId = core_1.Tokenization.resolve(logicalId, {
        scope,
        resolver: new core_1.DefaultTokenResolver(new core_1.StringConcat()),
    });
    return lastNCharacters(resolvedLogicalId, MAX_POLICY_NAME_LEN);
}
exports.generatePolicyName = generatePolicyName;
/**
 * Returns a string composed of the last n characters of str.
 * If str is shorter than n, returns str.
 *
 * @param str the string to return the last n characters of
 * @param n how many characters to return
 */
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex, str.length);
}
/**
 * Helper class that maintains the set of attached policies for a principal.
 */
class AttachedPolicies {
    constructor() {
        this.policies = new Array();
    }
    /**
     * Adds a policy to the list of attached policies.
     *
     * If this policy is already, attached, returns false.
     * If there is another policy attached with the same name, throws an exception.
     */
    attach(policy) {
        if (this.policies.find(p => p === policy)) {
            return; // already attached
        }
        if (this.policies.find(p => p.policyName === policy.policyName)) {
            throw new Error(`A policy named "${policy.policyName}" is already attached`);
        }
        this.policies.push(policy);
    }
}
exports.AttachedPolicies = AttachedPolicies;
/**
 * Merge two dictionaries that represent IAM principals
 */
function mergePrincipal(target, source) {
    for (const key of Object.keys(source)) {
        target[key] = target[key] || [];
        let value = source[key];
        if (!Array.isArray(value)) {
            value = [value];
        }
        target[key].push(...value);
    }
    return target;
}
exports.mergePrincipal = mergePrincipal;
//# sourceMappingURL=data:application/json;base64,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