"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const sinon = require("sinon");
const iam = require("../lib");
const diff_1 = require("../lib/oidc-provider/diff");
const external_1 = require("../lib/oidc-provider/external");
const handler = require("../lib/oidc-provider/index");
describe('OpenIdConnectProvider resource', () => {
    test('minimal configuration (no clients and no thumbprint)', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new iam.OpenIdConnectProvider(stack, 'MyProvider', {
            url: 'https://openid-endpoint',
        });
        // THEN
        expect(stack).toHaveResource('Custom::AWSCDKOpenIdConnectProvider', {
            Url: 'https://openid-endpoint',
        });
    });
    test('"openIdConnectProviderArn" resolves to the ref', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const provider = new iam.OpenIdConnectProvider(stack, 'MyProvider', {
            url: 'https://openid-endpoint',
        });
        // THEN
        expect(stack.resolve(provider.openIdConnectProviderArn)).toStrictEqual({ Ref: 'MyProvider730BA1C8' });
    });
    test('static fromOpenIdConnectProviderArn can be used to import a provider', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const provider = iam.OpenIdConnectProvider.fromOpenIdConnectProviderArn(stack, 'MyProvider', 'arn:of:provider');
        // THEN
        expect(stack.resolve(provider.openIdConnectProviderArn)).toStrictEqual('arn:of:provider');
    });
    test('thumbprint list and client ids can be specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new iam.OpenIdConnectProvider(stack, 'MyProvider', {
            url: 'https://my-url',
            clientIds: ['client1', 'client2'],
            thumbprints: ['thumb1'],
        });
        // THEN
        expect(stack).toHaveResource('Custom::AWSCDKOpenIdConnectProvider', {
            Url: 'https://my-url',
            ClientIDList: ['client1', 'client2'],
            ThumbprintList: ['thumb1'],
        });
    });
});
describe('custom resource provider infrastructure', () => {
    test('two resources share the same cr provider', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'stack');
        // WHEN
        new iam.OpenIdConnectProvider(stack, 'Provider1', { url: 'provider1' });
        new iam.OpenIdConnectProvider(stack, 'Provider2', { url: 'provider2' });
        // THEN
        const template = app.synth().getStackArtifact(stack.artifactId).template;
        const resourceTypes = Object.values(template.Resources).map((r) => r.Type).sort();
        expect(resourceTypes).toStrictEqual([
            // custom resource perovider resources
            'AWS::IAM::Role',
            'AWS::Lambda::Function',
            // open id connect resources
            'Custom::AWSCDKOpenIdConnectProvider',
            'Custom::AWSCDKOpenIdConnectProvider',
        ]);
    });
    test('iam policy', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new iam.OpenIdConnectProvider(stack, 'Provider1', { url: 'provider1' });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Role', {
            Policies: [
                {
                    PolicyName: 'Inline',
                    PolicyDocument: {
                        Version: '2012-10-17',
                        Statement: [
                            {
                                Effect: 'Allow',
                                Resource: '*',
                                Action: [
                                    'iam:CreateOpenIDConnectProvider',
                                    'iam:DeleteOpenIDConnectProvider',
                                    'iam:UpdateOpenIDConnectProviderThumbprint',
                                    'iam:AddClientIDToOpenIDConnectProvider',
                                    'iam:RemoveClientIDFromOpenIDConnectProvider',
                                ],
                            },
                        ],
                    },
                },
            ],
        });
    });
});
describe('custom resource provider handler', () => {
    external_1.external.log = () => { return; }; // disable verbosity for tests
    const downloadThumbprint = external_1.external.downloadThumbprint = sinon.fake.returns('FAKE-THUMBPRINT');
    const createOpenIDConnectProvider = external_1.external.createOpenIDConnectProvider = sinon.fake.resolves({ OpenIDConnectProviderArn: 'FAKE-ARN' });
    const deleteOpenIDConnectProvider = external_1.external.deleteOpenIDConnectProvider = sinon.fake.resolves({});
    const updateOpenIDConnectProviderThumbprint = external_1.external.updateOpenIDConnectProviderThumbprint = sinon.fake.resolves({});
    const addClientIDToOpenIDConnectProvider = external_1.external.addClientIDToOpenIDConnectProvider = sinon.fake.resolves({});
    const removeClientIDFromOpenIDConnectProvider = external_1.external.removeClientIDFromOpenIDConnectProvider = sinon.fake.resolves({});
    beforeEach(() => sinon.reset());
    test('create with url will download thumbprint from host', async () => {
        // WHEN
        const response = await invokeHandler({
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://my-urlx',
                ThumbprintList: ['MyThumbprint'],
            },
        });
        // THEN
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.calledWithExactly(createOpenIDConnectProvider, {
            ClientIDList: [],
            Url: 'https://my-urlx',
            ThumbprintList: ['MyThumbprint'],
        });
        expect(response).toStrictEqual({
            PhysicalResourceId: 'FAKE-ARN',
        });
    });
    test('create without thumbprint will download from host', async () => {
        // WHEN
        const response = await invokeHandler({
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://my-urlx',
            },
        });
        // THEN
        sinon.assert.calledWithExactly(downloadThumbprint, 'https://my-urlx');
        sinon.assert.calledWithExactly(createOpenIDConnectProvider, {
            ClientIDList: [],
            Url: 'https://my-urlx',
            ThumbprintList: ['FAKE-THUMBPRINT'],
        });
        expect(response).toStrictEqual({
            PhysicalResourceId: 'FAKE-ARN',
        });
    });
    test('delete', async () => {
        // WHEN
        await invokeHandler({
            RequestType: 'Delete',
            PhysicalResourceId: 'FAKE-ARN',
        });
        // THEN
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.notCalled(createOpenIDConnectProvider);
        sinon.assert.calledWithExactly(deleteOpenIDConnectProvider, {
            OpenIDConnectProviderArn: 'FAKE-ARN',
        });
    });
    test('update url with explicit thumbprints (replace)', async () => {
        // WHEN
        const response = await invokeHandler({
            RequestType: 'Update',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://new',
                ThumbprintList: ['THUMB1', 'THUMB2'],
            },
            OldResourceProperties: {
                Url: 'https://old',
            },
        });
        // THEN
        expect(response).toStrictEqual({
            PhysicalResourceId: 'FAKE-ARN',
        });
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.calledWithExactly(createOpenIDConnectProvider, {
            ClientIDList: [],
            Url: 'https://new',
            ThumbprintList: ['THUMB1', 'THUMB2'],
        });
    });
    test('update url with no thumbprint (replace)', async () => {
        // WHEN
        const response = await invokeHandler({
            RequestType: 'Update',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://new',
            },
            OldResourceProperties: {
                Url: 'https://old',
            },
        });
        // THEN
        expect(response).toStrictEqual({
            PhysicalResourceId: 'FAKE-ARN',
        });
        sinon.assert.calledOnceWithExactly(downloadThumbprint, 'https://new');
        sinon.assert.calledOnceWithExactly(createOpenIDConnectProvider, {
            ClientIDList: [],
            Url: 'https://new',
            ThumbprintList: ['FAKE-THUMBPRINT'],
        });
        sinon.assert.notCalled(deleteOpenIDConnectProvider);
    });
    test('update thumbprint list', async () => {
        // WHEN
        await invokeHandler({
            RequestType: 'Update',
            PhysicalResourceId: 'FAKE-PhysicalResourceId',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://url',
                ThumbprintList: ['Foo', 'Bar'],
            },
            OldResourceProperties: {
                Url: 'https://url',
                ThumbprintList: ['Foo'],
            },
        });
        // THEN
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.notCalled(createOpenIDConnectProvider);
        sinon.assert.notCalled(deleteOpenIDConnectProvider);
        sinon.assert.calledOnceWithExactly(updateOpenIDConnectProviderThumbprint, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId',
            ThumbprintList: ['Bar', 'Foo'],
        });
    });
    test('add/remove client ids', async () => {
        // WHEN
        await invokeHandler({
            RequestType: 'Update',
            PhysicalResourceId: 'FAKE-PhysicalResourceId',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://url',
                ClientIDList: ['A', 'B', 'C'],
            },
            OldResourceProperties: {
                Url: 'https://url',
                ClientIDList: ['A', 'D'],
            },
        });
        // THEN
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.notCalled(createOpenIDConnectProvider);
        sinon.assert.notCalled(deleteOpenIDConnectProvider);
        sinon.assert.notCalled(updateOpenIDConnectProviderThumbprint);
        sinon.assert.calledTwice(addClientIDToOpenIDConnectProvider);
        sinon.assert.calledWithExactly(addClientIDToOpenIDConnectProvider, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId', ClientID: 'B',
        });
        sinon.assert.calledWithExactly(addClientIDToOpenIDConnectProvider, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId', ClientID: 'C',
        });
        sinon.assert.calledOnceWithExactly(removeClientIDFromOpenIDConnectProvider, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId', ClientID: 'D',
        });
    });
    test('multiple in-place updates (no replace)', async () => {
        // WHEN
        await invokeHandler({
            RequestType: 'Update',
            PhysicalResourceId: 'FAKE-PhysicalResourceId',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://url',
                ThumbprintList: ['NEW-LIST'],
                ClientIDList: ['A'],
            },
            OldResourceProperties: {
                Url: 'https://url',
                ThumbprintList: ['OLD-LIST'],
                ClientIDList: [],
            },
        });
        // THEN
        sinon.assert.notCalled(downloadThumbprint);
        sinon.assert.notCalled(createOpenIDConnectProvider);
        sinon.assert.notCalled(deleteOpenIDConnectProvider);
        sinon.assert.notCalled(removeClientIDFromOpenIDConnectProvider);
        sinon.assert.calledOnceWithExactly(updateOpenIDConnectProviderThumbprint, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId',
            ThumbprintList: ['NEW-LIST'],
        });
        sinon.assert.calledOnceWithExactly(addClientIDToOpenIDConnectProvider, {
            OpenIDConnectProviderArn: 'FAKE-PhysicalResourceId',
            ClientID: 'A',
        });
    });
    test('multiple updates that include a url update, which means replacement', async () => {
        // WHEN
        await invokeHandler({
            RequestType: 'Update',
            PhysicalResourceId: 'FAKE-PhysicalResourceId',
            ResourceProperties: {
                ServiceToken: 'Foo',
                Url: 'https://new-url',
                ClientIDList: ['A'],
            },
            OldResourceProperties: {
                Url: 'https://old-url',
                ThumbprintList: ['OLD-LIST'],
                ClientIDList: [],
            },
        });
        // THEN
        sinon.assert.notCalled(deleteOpenIDConnectProvider);
        sinon.assert.notCalled(removeClientIDFromOpenIDConnectProvider);
        sinon.assert.notCalled(updateOpenIDConnectProviderThumbprint);
        sinon.assert.notCalled(addClientIDToOpenIDConnectProvider);
        sinon.assert.calledOnceWithExactly(downloadThumbprint, 'https://new-url'); // since thumbprint list is empty
        sinon.assert.calledOnceWithExactly(createOpenIDConnectProvider, {
            ClientIDList: ['A'],
            ThumbprintList: ['FAKE-THUMBPRINT'],
            Url: 'https://new-url',
        });
    });
});
describe('arrayDiff', () => {
    test('calculates the difference between two arrays', () => {
        expect(diff_1.arrayDiff(['a', 'b', 'c'], ['a', 'd'])).toStrictEqual({ adds: ['d'], deletes: ['b', 'c'] });
        expect(diff_1.arrayDiff(['a', 'b', 'c'], [])).toStrictEqual({ adds: [], deletes: ['a', 'b', 'c'] });
        expect(diff_1.arrayDiff(['a', 'b', 'c'], ['a', 'c', 'b'])).toStrictEqual({ adds: [], deletes: [] });
        expect(diff_1.arrayDiff([], ['a', 'c', 'b'])).toStrictEqual({ adds: ['a', 'c', 'b'], deletes: [] });
        expect(diff_1.arrayDiff(['x', 'y'], ['a', 'c', 'b'])).toStrictEqual({ adds: ['a', 'c', 'b'], deletes: ['x', 'y'] });
        expect(diff_1.arrayDiff([], [])).toStrictEqual({ adds: [], deletes: [] });
        expect(diff_1.arrayDiff(['a', 'a'], ['a', 'b', 'a', 'b', 'b'])).toStrictEqual({ adds: ['b'], deletes: [] });
    });
});
async function invokeHandler(event) {
    return await handler.handler(event);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib2lkYy1wcm92aWRlci50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsib2lkYy1wcm92aWRlci50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsZ0NBQThCO0FBQzlCLHdDQUEyQztBQUMzQywrQkFBK0I7QUFDL0IsOEJBQThCO0FBQzlCLG9EQUFzRDtBQUN0RCw0REFBeUQ7QUFDekQsc0RBQXNEO0FBRXRELFFBQVEsQ0FBQyxnQ0FBZ0MsRUFBRSxHQUFHLEVBQUU7SUFFOUMsSUFBSSxDQUFDLHNEQUFzRCxFQUFFLEdBQUcsRUFBRTtRQUNoRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixPQUFPO1FBQ1AsSUFBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtZQUNqRCxHQUFHLEVBQUUseUJBQXlCO1NBQy9CLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHFDQUFxQyxFQUFFO1lBQ2xFLEdBQUcsRUFBRSx5QkFBeUI7U0FDL0IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsZ0RBQWdELEVBQUUsR0FBRyxFQUFFO1FBQzFELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ2xFLEdBQUcsRUFBRSx5QkFBeUI7U0FDL0IsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLEVBQUUsR0FBRyxFQUFFLG9CQUFvQixFQUFFLENBQUMsQ0FBQztJQUN4RyxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxzRUFBc0UsRUFBRSxHQUFHLEVBQUU7UUFDaEYsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFFMUIsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyw0QkFBNEIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFFaEgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDNUYsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsaURBQWlELEVBQUUsR0FBRyxFQUFFO1FBQzNELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE9BQU87UUFDUCxJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ2pELEdBQUcsRUFBRSxnQkFBZ0I7WUFDckIsU0FBUyxFQUFFLENBQUUsU0FBUyxFQUFFLFNBQVMsQ0FBRTtZQUNuQyxXQUFXLEVBQUUsQ0FBRSxRQUFRLENBQUU7U0FDMUIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMscUNBQXFDLEVBQUU7WUFDbEUsR0FBRyxFQUFFLGdCQUFnQjtZQUNyQixZQUFZLEVBQUUsQ0FBRSxTQUFTLEVBQUUsU0FBUyxDQUFFO1lBQ3RDLGNBQWMsRUFBRSxDQUFFLFFBQVEsQ0FBRTtTQUM3QixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUVMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLHlDQUF5QyxFQUFFLEdBQUcsRUFBRTtJQUV2RCxJQUFJLENBQUMsMENBQTBDLEVBQUUsR0FBRyxFQUFFO1FBQ3BELFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUV0QyxPQUFPO1FBQ1AsSUFBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRSxFQUFFLEdBQUcsRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDO1FBQ3hFLElBQUksR0FBRyxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUUsRUFBRSxHQUFHLEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQztRQUV4RSxPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxRQUFRLENBQUM7UUFDekUsTUFBTSxhQUFhLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBTSxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDdkYsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGFBQWEsQ0FBQztZQUNsQyxzQ0FBc0M7WUFDdEMsZ0JBQWdCO1lBQ2hCLHVCQUF1QjtZQUV2Qiw0QkFBNEI7WUFDNUIscUNBQXFDO1lBQ3JDLHFDQUFxQztTQUN0QyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxZQUFZLEVBQUUsR0FBRyxFQUFFO1FBQ3RCLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE9BQU87UUFDUCxJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFLEVBQUUsR0FBRyxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUM7UUFFeEUsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUU7WUFDN0MsUUFBUSxFQUFFO2dCQUNSO29CQUNFLFVBQVUsRUFBRSxRQUFRO29CQUNwQixjQUFjLEVBQUU7d0JBQ2QsT0FBTyxFQUFFLFlBQVk7d0JBQ3JCLFNBQVMsRUFBRTs0QkFDVDtnQ0FDRSxNQUFNLEVBQUUsT0FBTztnQ0FDZixRQUFRLEVBQUUsR0FBRztnQ0FDYixNQUFNLEVBQUU7b0NBQ04saUNBQWlDO29DQUNqQyxpQ0FBaUM7b0NBQ2pDLDJDQUEyQztvQ0FDM0Msd0NBQXdDO29DQUN4Qyw2Q0FBNkM7aUNBQzlDOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLGtDQUFrQyxFQUFFLEdBQUcsRUFBRTtJQUNoRCxtQkFBUSxDQUFDLEdBQUcsR0FBRyxHQUFHLEVBQUUsR0FBRyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsOEJBQThCO0lBQ2hFLE1BQU0sa0JBQWtCLEdBQUcsbUJBQVEsQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQy9GLE1BQU0sMkJBQTJCLEdBQUcsbUJBQVEsQ0FBQywyQkFBMkIsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLHdCQUF3QixFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUM7SUFDekksTUFBTSwyQkFBMkIsR0FBRyxtQkFBUSxDQUFDLDJCQUEyQixHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUcsQ0FBQyxDQUFDO0lBQ3BHLE1BQU0scUNBQXFDLEdBQUcsbUJBQVEsQ0FBQyxxQ0FBcUMsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFHLENBQUMsQ0FBQztJQUN4SCxNQUFNLGtDQUFrQyxHQUFHLG1CQUFRLENBQUMsa0NBQWtDLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRyxDQUFDLENBQUM7SUFDbEgsTUFBTSx1Q0FBdUMsR0FBRyxtQkFBUSxDQUFDLHVDQUF1QyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUcsQ0FBQyxDQUFDO0lBRTVILFVBQVUsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQztJQUVoQyxJQUFJLENBQUMsb0RBQW9ELEVBQUUsS0FBSyxJQUFJLEVBQUU7UUFDcEUsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLE1BQU0sYUFBYSxDQUFDO1lBQ25DLFdBQVcsRUFBRSxRQUFRO1lBQ3JCLGtCQUFrQixFQUFFO2dCQUNsQixZQUFZLEVBQUUsS0FBSztnQkFDbkIsR0FBRyxFQUFFLGlCQUFpQjtnQkFDdEIsY0FBYyxFQUFFLENBQUUsY0FBYyxDQUFFO2FBQ25DO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDM0MsS0FBSyxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQywyQkFBMkIsRUFBRTtZQUMxRCxZQUFZLEVBQUUsRUFBRTtZQUNoQixHQUFHLEVBQUUsaUJBQWlCO1lBQ3RCLGNBQWMsRUFBRSxDQUFFLGNBQWMsQ0FBRTtTQUNuQyxDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsYUFBYSxDQUFDO1lBQzdCLGtCQUFrQixFQUFFLFVBQVU7U0FDL0IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsbURBQW1ELEVBQUUsS0FBSyxJQUFJLEVBQUU7UUFDbkUsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLE1BQU0sYUFBYSxDQUFDO1lBQ25DLFdBQVcsRUFBRSxRQUFRO1lBQ3JCLGtCQUFrQixFQUFFO2dCQUNsQixZQUFZLEVBQUUsS0FBSztnQkFDbkIsR0FBRyxFQUFFLGlCQUFpQjthQUN2QjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxLQUFLLENBQUMsTUFBTSxDQUFDLGlCQUFpQixDQUFDLGtCQUFrQixFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDdEUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQywyQkFBMkIsRUFBRTtZQUMxRCxZQUFZLEVBQUUsRUFBRTtZQUNoQixHQUFHLEVBQUUsaUJBQWlCO1lBQ3RCLGNBQWMsRUFBRSxDQUFFLGlCQUFpQixDQUFFO1NBQ3RDLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxhQUFhLENBQUM7WUFDN0Isa0JBQWtCLEVBQUUsVUFBVTtTQUMvQixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxRQUFRLEVBQUUsS0FBSyxJQUFJLEVBQUU7UUFDeEIsT0FBTztRQUNQLE1BQU0sYUFBYSxDQUFDO1lBQ2xCLFdBQVcsRUFBRSxRQUFRO1lBQ3JCLGtCQUFrQixFQUFFLFVBQVU7U0FDL0IsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDM0MsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUNwRCxLQUFLLENBQUMsTUFBTSxDQUFDLGlCQUFpQixDQUFDLDJCQUEyQixFQUFFO1lBQzFELHdCQUF3QixFQUFFLFVBQVU7U0FDckMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsZ0RBQWdELEVBQUUsS0FBSyxJQUFJLEVBQUU7UUFDaEUsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLE1BQU0sYUFBYSxDQUFDO1lBQ25DLFdBQVcsRUFBRSxRQUFRO1lBQ3JCLGtCQUFrQixFQUFFO2dCQUNsQixZQUFZLEVBQUUsS0FBSztnQkFDbkIsR0FBRyxFQUFFLGFBQWE7Z0JBQ2xCLGNBQWMsRUFBRSxDQUFFLFFBQVEsRUFBRSxRQUFRLENBQUU7YUFDdkM7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsR0FBRyxFQUFFLGFBQWE7YUFDbkI7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLGFBQWEsQ0FBQztZQUM3QixrQkFBa0IsRUFBRSxVQUFVO1NBQy9CLENBQUMsQ0FBQztRQUNILEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDM0MsS0FBSyxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQywyQkFBMkIsRUFBRTtZQUMxRCxZQUFZLEVBQUUsRUFBRTtZQUNoQixHQUFHLEVBQUUsYUFBYTtZQUNsQixjQUFjLEVBQUUsQ0FBRSxRQUFRLEVBQUUsUUFBUSxDQUFFO1NBQ3ZDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHlDQUF5QyxFQUFFLEtBQUssSUFBSSxFQUFFO1FBQ3pELE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxNQUFNLGFBQWEsQ0FBQztZQUNuQyxXQUFXLEVBQUUsUUFBUTtZQUNyQixrQkFBa0IsRUFBRTtnQkFDbEIsWUFBWSxFQUFFLEtBQUs7Z0JBQ25CLEdBQUcsRUFBRSxhQUFhO2FBQ25CO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEdBQUcsRUFBRSxhQUFhO2FBQ25CO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxhQUFhLENBQUM7WUFDN0Isa0JBQWtCLEVBQUUsVUFBVTtTQUMvQixDQUFDLENBQUM7UUFDSCxLQUFLLENBQUMsTUFBTSxDQUFDLHFCQUFxQixDQUFDLGtCQUFrQixFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQ3RFLEtBQUssQ0FBQyxNQUFNLENBQUMscUJBQXFCLENBQUMsMkJBQTJCLEVBQUU7WUFDOUQsWUFBWSxFQUFFLEVBQUU7WUFDaEIsR0FBRyxFQUFFLGFBQWE7WUFDbEIsY0FBYyxFQUFFLENBQUUsaUJBQWlCLENBQUU7U0FDdEMsQ0FBQyxDQUFDO1FBQ0gsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUN0RCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx3QkFBd0IsRUFBRSxLQUFLLElBQUksRUFBRTtRQUN4QyxPQUFPO1FBQ1AsTUFBTSxhQUFhLENBQUM7WUFDbEIsV0FBVyxFQUFFLFFBQVE7WUFDckIsa0JBQWtCLEVBQUUseUJBQXlCO1lBQzdDLGtCQUFrQixFQUFFO2dCQUNsQixZQUFZLEVBQUUsS0FBSztnQkFDbkIsR0FBRyxFQUFFLGFBQWE7Z0JBQ2xCLGNBQWMsRUFBRSxDQUFFLEtBQUssRUFBRSxLQUFLLENBQUU7YUFDakM7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsR0FBRyxFQUFFLGFBQWE7Z0JBQ2xCLGNBQWMsRUFBRSxDQUFFLEtBQUssQ0FBRTthQUMxQjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzNDLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDcEQsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUNwRCxLQUFLLENBQUMsTUFBTSxDQUFDLHFCQUFxQixDQUFDLHFDQUFxQyxFQUFFO1lBQ3hFLHdCQUF3QixFQUFFLHlCQUF5QjtZQUNuRCxjQUFjLEVBQUUsQ0FBRSxLQUFLLEVBQUUsS0FBSyxDQUFFO1NBQ2pDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHVCQUF1QixFQUFFLEtBQUssSUFBSSxFQUFFO1FBQ3ZDLE9BQU87UUFDUCxNQUFNLGFBQWEsQ0FBQztZQUNsQixXQUFXLEVBQUUsUUFBUTtZQUNyQixrQkFBa0IsRUFBRSx5QkFBeUI7WUFDN0Msa0JBQWtCLEVBQUU7Z0JBQ2xCLFlBQVksRUFBRSxLQUFLO2dCQUNuQixHQUFHLEVBQUUsYUFBYTtnQkFDbEIsWUFBWSxFQUFFLENBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUU7YUFDaEM7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsR0FBRyxFQUFFLGFBQWE7Z0JBQ2xCLFlBQVksRUFBRSxDQUFFLEdBQUcsRUFBRSxHQUFHLENBQUU7YUFDM0I7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUMzQyxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3BELEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDcEQsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUM5RCxLQUFLLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzdELEtBQUssQ0FBQyxNQUFNLENBQUMsaUJBQWlCLENBQUMsa0NBQWtDLEVBQUU7WUFDakUsd0JBQXdCLEVBQUUseUJBQXlCLEVBQUUsUUFBUSxFQUFFLEdBQUc7U0FDbkUsQ0FBQyxDQUFDO1FBQ0gsS0FBSyxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxrQ0FBa0MsRUFBRTtZQUNqRSx3QkFBd0IsRUFBRSx5QkFBeUIsRUFBRSxRQUFRLEVBQUUsR0FBRztTQUNuRSxDQUFDLENBQUM7UUFDSCxLQUFLLENBQUMsTUFBTSxDQUFDLHFCQUFxQixDQUFDLHVDQUF1QyxFQUFFO1lBQzFFLHdCQUF3QixFQUFFLHlCQUF5QixFQUFFLFFBQVEsRUFBRSxHQUFHO1NBQ25FLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEtBQUssSUFBSSxFQUFFO1FBQ3hELE9BQU87UUFDUCxNQUFNLGFBQWEsQ0FBQztZQUNsQixXQUFXLEVBQUUsUUFBUTtZQUNyQixrQkFBa0IsRUFBRSx5QkFBeUI7WUFDN0Msa0JBQWtCLEVBQUU7Z0JBQ2xCLFlBQVksRUFBRSxLQUFLO2dCQUNuQixHQUFHLEVBQUUsYUFBYTtnQkFDbEIsY0FBYyxFQUFFLENBQUUsVUFBVSxDQUFFO2dCQUM5QixZQUFZLEVBQUUsQ0FBRSxHQUFHLENBQUU7YUFDdEI7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsR0FBRyxFQUFFLGFBQWE7Z0JBQ2xCLGNBQWMsRUFBRSxDQUFFLFVBQVUsQ0FBRTtnQkFDOUIsWUFBWSxFQUFFLEVBQUc7YUFDbEI7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUMzQyxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3BELEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDcEQsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNoRSxLQUFLLENBQUMsTUFBTSxDQUFDLHFCQUFxQixDQUFDLHFDQUFxQyxFQUFFO1lBQ3hFLHdCQUF3QixFQUFFLHlCQUF5QjtZQUNuRCxjQUFjLEVBQUUsQ0FBRSxVQUFVLENBQUU7U0FDL0IsQ0FBQyxDQUFDO1FBQ0gsS0FBSyxDQUFDLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxrQ0FBa0MsRUFBRTtZQUNyRSx3QkFBd0IsRUFBRSx5QkFBeUI7WUFDbkQsUUFBUSxFQUFFLEdBQUc7U0FDZCxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxxRUFBcUUsRUFBRSxLQUFLLElBQUksRUFBRTtRQUNyRixPQUFPO1FBQ1AsTUFBTSxhQUFhLENBQUM7WUFDbEIsV0FBVyxFQUFFLFFBQVE7WUFDckIsa0JBQWtCLEVBQUUseUJBQXlCO1lBQzdDLGtCQUFrQixFQUFFO2dCQUNsQixZQUFZLEVBQUUsS0FBSztnQkFDbkIsR0FBRyxFQUFFLGlCQUFpQjtnQkFDdEIsWUFBWSxFQUFFLENBQUUsR0FBRyxDQUFFO2FBQ3RCO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEdBQUcsRUFBRSxpQkFBaUI7Z0JBQ3RCLGNBQWMsRUFBRSxDQUFFLFVBQVUsQ0FBRTtnQkFDOUIsWUFBWSxFQUFFLEVBQUc7YUFDbEI7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUNwRCxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1FBQ2hFLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDOUQsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUMzRCxLQUFLLENBQUMsTUFBTSxDQUFDLHFCQUFxQixDQUFDLGtCQUFrQixFQUFFLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxpQ0FBaUM7UUFDNUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQywyQkFBMkIsRUFBRTtZQUM5RCxZQUFZLEVBQUUsQ0FBRSxHQUFHLENBQUU7WUFDckIsY0FBYyxFQUFFLENBQUUsaUJBQWlCLENBQUU7WUFDckMsR0FBRyxFQUFFLGlCQUFpQjtTQUN2QixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUVMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLFdBQVcsRUFBRSxHQUFHLEVBQUU7SUFDekIsSUFBSSxDQUFDLDhDQUE4QyxFQUFFLEdBQUcsRUFBRTtRQUN4RCxNQUFNLENBQUMsZ0JBQVMsQ0FBQyxDQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFFLEVBQUUsQ0FBRSxHQUFHLEVBQUUsR0FBRyxDQUFFLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxFQUFFLElBQUksRUFBRSxDQUFFLEdBQUcsQ0FBRSxFQUFFLE9BQU8sRUFBRSxDQUFFLEdBQUcsRUFBRSxHQUFHLENBQUUsRUFBQyxDQUFDLENBQUM7UUFDMUcsTUFBTSxDQUFDLGdCQUFTLENBQUMsQ0FBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBRSxFQUFFLEVBQUcsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLEVBQUUsSUFBSSxFQUFFLEVBQUUsRUFBRSxPQUFPLEVBQUUsQ0FBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBRSxFQUFFLENBQUMsQ0FBQztRQUNsRyxNQUFNLENBQUMsZ0JBQVMsQ0FBQyxDQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFFLEVBQUUsQ0FBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBRSxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsRUFBRSxJQUFJLEVBQUUsRUFBRSxFQUFFLE9BQU8sRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pHLE1BQU0sQ0FBQyxnQkFBUyxDQUFDLEVBQUcsRUFBRSxDQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFFLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxFQUFFLElBQUksRUFBRSxDQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFFLEVBQUUsT0FBTyxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDbEcsTUFBTSxDQUFDLGdCQUFTLENBQUMsQ0FBRSxHQUFHLEVBQUUsR0FBRyxDQUFFLEVBQUUsQ0FBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBRSxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsRUFBRSxJQUFJLEVBQUUsQ0FBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBRSxFQUFFLE9BQU8sRUFBRSxDQUFFLEdBQUcsRUFBRSxHQUFHLENBQUUsRUFBQyxDQUFDLENBQUM7UUFDcEgsTUFBTSxDQUFDLGdCQUFTLENBQUMsRUFBRyxFQUFFLEVBQUcsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLEVBQUUsSUFBSSxFQUFFLEVBQUUsRUFBRSxPQUFPLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNyRSxNQUFNLENBQUMsZ0JBQVMsQ0FBQyxDQUFFLEdBQUcsRUFBRSxHQUFHLENBQUUsRUFBRSxDQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUUsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLEVBQUUsSUFBSSxFQUFFLENBQUUsR0FBRyxDQUFFLEVBQUUsT0FBTyxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDN0csQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILEtBQUssVUFBVSxhQUFhLENBQUMsS0FBMkQ7SUFDdEYsT0FBTyxNQUFNLE9BQU8sQ0FBQyxPQUFPLENBQUMsS0FBWSxDQUFDLENBQUM7QUFDN0MsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0IHsgQXBwLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgc2lub24gZnJvbSAnc2lub24nO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uL2xpYic7XG5pbXBvcnQgeyBhcnJheURpZmYgfSBmcm9tICcuLi9saWIvb2lkYy1wcm92aWRlci9kaWZmJztcbmltcG9ydCB7IGV4dGVybmFsIH0gZnJvbSAnLi4vbGliL29pZGMtcHJvdmlkZXIvZXh0ZXJuYWwnO1xuaW1wb3J0ICogYXMgaGFuZGxlciBmcm9tICcuLi9saWIvb2lkYy1wcm92aWRlci9pbmRleCc7XG5cbmRlc2NyaWJlKCdPcGVuSWRDb25uZWN0UHJvdmlkZXIgcmVzb3VyY2UnLCAoKSA9PiB7XG5cbiAgdGVzdCgnbWluaW1hbCBjb25maWd1cmF0aW9uIChubyBjbGllbnRzIGFuZCBubyB0aHVtYnByaW50KScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IGlhbS5PcGVuSWRDb25uZWN0UHJvdmlkZXIoc3RhY2ssICdNeVByb3ZpZGVyJywge1xuICAgICAgdXJsOiAnaHR0cHM6Ly9vcGVuaWQtZW5kcG9pbnQnLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0N1c3RvbTo6QVdTQ0RLT3BlbklkQ29ubmVjdFByb3ZpZGVyJywge1xuICAgICAgVXJsOiAnaHR0cHM6Ly9vcGVuaWQtZW5kcG9pbnQnLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdcIm9wZW5JZENvbm5lY3RQcm92aWRlckFyblwiIHJlc29sdmVzIHRvIHRoZSByZWYnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHByb3ZpZGVyID0gbmV3IGlhbS5PcGVuSWRDb25uZWN0UHJvdmlkZXIoc3RhY2ssICdNeVByb3ZpZGVyJywge1xuICAgICAgdXJsOiAnaHR0cHM6Ly9vcGVuaWQtZW5kcG9pbnQnLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjay5yZXNvbHZlKHByb3ZpZGVyLm9wZW5JZENvbm5lY3RQcm92aWRlckFybikpLnRvU3RyaWN0RXF1YWwoeyBSZWY6ICdNeVByb3ZpZGVyNzMwQkExQzgnIH0pO1xuICB9KTtcblxuICB0ZXN0KCdzdGF0aWMgZnJvbU9wZW5JZENvbm5lY3RQcm92aWRlckFybiBjYW4gYmUgdXNlZCB0byBpbXBvcnQgYSBwcm92aWRlcicsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcHJvdmlkZXIgPSBpYW0uT3BlbklkQ29ubmVjdFByb3ZpZGVyLmZyb21PcGVuSWRDb25uZWN0UHJvdmlkZXJBcm4oc3RhY2ssICdNeVByb3ZpZGVyJywgJ2FybjpvZjpwcm92aWRlcicpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjay5yZXNvbHZlKHByb3ZpZGVyLm9wZW5JZENvbm5lY3RQcm92aWRlckFybikpLnRvU3RyaWN0RXF1YWwoJ2FybjpvZjpwcm92aWRlcicpO1xuICB9KTtcblxuICB0ZXN0KCd0aHVtYnByaW50IGxpc3QgYW5kIGNsaWVudCBpZHMgY2FuIGJlIHNwZWNpZmllZCcsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IGlhbS5PcGVuSWRDb25uZWN0UHJvdmlkZXIoc3RhY2ssICdNeVByb3ZpZGVyJywge1xuICAgICAgdXJsOiAnaHR0cHM6Ly9teS11cmwnLFxuICAgICAgY2xpZW50SWRzOiBbICdjbGllbnQxJywgJ2NsaWVudDInIF0sXG4gICAgICB0aHVtYnByaW50czogWyAndGh1bWIxJyBdLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0N1c3RvbTo6QVdTQ0RLT3BlbklkQ29ubmVjdFByb3ZpZGVyJywge1xuICAgICAgVXJsOiAnaHR0cHM6Ly9teS11cmwnLFxuICAgICAgQ2xpZW50SURMaXN0OiBbICdjbGllbnQxJywgJ2NsaWVudDInIF0sXG4gICAgICBUaHVtYnByaW50TGlzdDogWyAndGh1bWIxJyBdLFxuICAgIH0pO1xuICB9KTtcblxufSk7XG5cbmRlc2NyaWJlKCdjdXN0b20gcmVzb3VyY2UgcHJvdmlkZXIgaW5mcmFzdHJ1Y3R1cmUnLCAoKSA9PiB7XG5cbiAgdGVzdCgndHdvIHJlc291cmNlcyBzaGFyZSB0aGUgc2FtZSBjciBwcm92aWRlcicsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdzdGFjaycpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBpYW0uT3BlbklkQ29ubmVjdFByb3ZpZGVyKHN0YWNrLCAnUHJvdmlkZXIxJywgeyB1cmw6ICdwcm92aWRlcjEnIH0pO1xuICAgIG5ldyBpYW0uT3BlbklkQ29ubmVjdFByb3ZpZGVyKHN0YWNrLCAnUHJvdmlkZXIyJywgeyB1cmw6ICdwcm92aWRlcjInIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHRlbXBsYXRlID0gYXBwLnN5bnRoKCkuZ2V0U3RhY2tBcnRpZmFjdChzdGFjay5hcnRpZmFjdElkKS50ZW1wbGF0ZTtcbiAgICBjb25zdCByZXNvdXJjZVR5cGVzID0gT2JqZWN0LnZhbHVlcyh0ZW1wbGF0ZS5SZXNvdXJjZXMpLm1hcCgocjogYW55KSA9PiByLlR5cGUpLnNvcnQoKTtcbiAgICBleHBlY3QocmVzb3VyY2VUeXBlcykudG9TdHJpY3RFcXVhbChbXG4gICAgICAvLyBjdXN0b20gcmVzb3VyY2UgcGVyb3ZpZGVyIHJlc291cmNlc1xuICAgICAgJ0FXUzo6SUFNOjpSb2xlJyxcbiAgICAgICdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLFxuXG4gICAgICAvLyBvcGVuIGlkIGNvbm5lY3QgcmVzb3VyY2VzXG4gICAgICAnQ3VzdG9tOjpBV1NDREtPcGVuSWRDb25uZWN0UHJvdmlkZXInLFxuICAgICAgJ0N1c3RvbTo6QVdTQ0RLT3BlbklkQ29ubmVjdFByb3ZpZGVyJyxcbiAgICBdKTtcbiAgfSk7XG5cbiAgdGVzdCgnaWFtIHBvbGljeScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IGlhbS5PcGVuSWRDb25uZWN0UHJvdmlkZXIoc3RhY2ssICdQcm92aWRlcjEnLCB7IHVybDogJ3Byb3ZpZGVyMScgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICBQb2xpY2llczogW1xuICAgICAgICB7XG4gICAgICAgICAgUG9saWN5TmFtZTogJ0lubGluZScsXG4gICAgICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgICAgIFJlc291cmNlOiAnKicsXG4gICAgICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICAgICAnaWFtOkNyZWF0ZU9wZW5JRENvbm5lY3RQcm92aWRlcicsXG4gICAgICAgICAgICAgICAgICAnaWFtOkRlbGV0ZU9wZW5JRENvbm5lY3RQcm92aWRlcicsXG4gICAgICAgICAgICAgICAgICAnaWFtOlVwZGF0ZU9wZW5JRENvbm5lY3RQcm92aWRlclRodW1icHJpbnQnLFxuICAgICAgICAgICAgICAgICAgJ2lhbTpBZGRDbGllbnRJRFRvT3BlbklEQ29ubmVjdFByb3ZpZGVyJyxcbiAgICAgICAgICAgICAgICAgICdpYW06UmVtb3ZlQ2xpZW50SURGcm9tT3BlbklEQ29ubmVjdFByb3ZpZGVyJyxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcbiAgfSk7XG59KTtcblxuZGVzY3JpYmUoJ2N1c3RvbSByZXNvdXJjZSBwcm92aWRlciBoYW5kbGVyJywgKCkgPT4ge1xuICBleHRlcm5hbC5sb2cgPSAoKSA9PiB7IHJldHVybjsgfTsgLy8gZGlzYWJsZSB2ZXJib3NpdHkgZm9yIHRlc3RzXG4gIGNvbnN0IGRvd25sb2FkVGh1bWJwcmludCA9IGV4dGVybmFsLmRvd25sb2FkVGh1bWJwcmludCA9IHNpbm9uLmZha2UucmV0dXJucygnRkFLRS1USFVNQlBSSU5UJyk7XG4gIGNvbnN0IGNyZWF0ZU9wZW5JRENvbm5lY3RQcm92aWRlciA9IGV4dGVybmFsLmNyZWF0ZU9wZW5JRENvbm5lY3RQcm92aWRlciA9IHNpbm9uLmZha2UucmVzb2x2ZXMoeyBPcGVuSURDb25uZWN0UHJvdmlkZXJBcm46ICdGQUtFLUFSTicgfSk7XG4gIGNvbnN0IGRlbGV0ZU9wZW5JRENvbm5lY3RQcm92aWRlciA9IGV4dGVybmFsLmRlbGV0ZU9wZW5JRENvbm5lY3RQcm92aWRlciA9IHNpbm9uLmZha2UucmVzb2x2ZXMoeyB9KTtcbiAgY29uc3QgdXBkYXRlT3BlbklEQ29ubmVjdFByb3ZpZGVyVGh1bWJwcmludCA9IGV4dGVybmFsLnVwZGF0ZU9wZW5JRENvbm5lY3RQcm92aWRlclRodW1icHJpbnQgPSBzaW5vbi5mYWtlLnJlc29sdmVzKHsgfSk7XG4gIGNvbnN0IGFkZENsaWVudElEVG9PcGVuSURDb25uZWN0UHJvdmlkZXIgPSBleHRlcm5hbC5hZGRDbGllbnRJRFRvT3BlbklEQ29ubmVjdFByb3ZpZGVyID0gc2lub24uZmFrZS5yZXNvbHZlcyh7IH0pO1xuICBjb25zdCByZW1vdmVDbGllbnRJREZyb21PcGVuSURDb25uZWN0UHJvdmlkZXIgPSBleHRlcm5hbC5yZW1vdmVDbGllbnRJREZyb21PcGVuSURDb25uZWN0UHJvdmlkZXIgPSBzaW5vbi5mYWtlLnJlc29sdmVzKHsgfSk7XG5cbiAgYmVmb3JlRWFjaCgoKSA9PiBzaW5vbi5yZXNldCgpKTtcblxuICB0ZXN0KCdjcmVhdGUgd2l0aCB1cmwgd2lsbCBkb3dubG9hZCB0aHVtYnByaW50IGZyb20gaG9zdCcsIGFzeW5jICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBpbnZva2VIYW5kbGVyKHtcbiAgICAgIFJlcXVlc3RUeXBlOiAnQ3JlYXRlJyxcbiAgICAgIFJlc291cmNlUHJvcGVydGllczoge1xuICAgICAgICBTZXJ2aWNlVG9rZW46ICdGb28nLFxuICAgICAgICBVcmw6ICdodHRwczovL215LXVybHgnLFxuICAgICAgICBUaHVtYnByaW50TGlzdDogWyAnTXlUaHVtYnByaW50JyBdLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBzaW5vbi5hc3NlcnQubm90Q2FsbGVkKGRvd25sb2FkVGh1bWJwcmludCk7XG4gICAgc2lub24uYXNzZXJ0LmNhbGxlZFdpdGhFeGFjdGx5KGNyZWF0ZU9wZW5JRENvbm5lY3RQcm92aWRlciwge1xuICAgICAgQ2xpZW50SURMaXN0OiBbXSxcbiAgICAgIFVybDogJ2h0dHBzOi8vbXktdXJseCcsXG4gICAgICBUaHVtYnByaW50TGlzdDogWyAnTXlUaHVtYnByaW50JyBdLFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHJlc3BvbnNlKS50b1N0cmljdEVxdWFsKHtcbiAgICAgIFBoeXNpY2FsUmVzb3VyY2VJZDogJ0ZBS0UtQVJOJyxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnY3JlYXRlIHdpdGhvdXQgdGh1bWJwcmludCB3aWxsIGRvd25sb2FkIGZyb20gaG9zdCcsIGFzeW5jICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBpbnZva2VIYW5kbGVyKHtcbiAgICAgIFJlcXVlc3RUeXBlOiAnQ3JlYXRlJyxcbiAgICAgIFJlc291cmNlUHJvcGVydGllczoge1xuICAgICAgICBTZXJ2aWNlVG9rZW46ICdGb28nLFxuICAgICAgICBVcmw6ICdodHRwczovL215LXVybHgnLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBzaW5vbi5hc3NlcnQuY2FsbGVkV2l0aEV4YWN0bHkoZG93bmxvYWRUaHVtYnByaW50LCAnaHR0cHM6Ly9teS11cmx4Jyk7XG4gICAgc2lub24uYXNzZXJ0LmNhbGxlZFdpdGhFeGFjdGx5KGNyZWF0ZU9wZW5JRENvbm5lY3RQcm92aWRlciwge1xuICAgICAgQ2xpZW50SURMaXN0OiBbXSxcbiAgICAgIFVybDogJ2h0dHBzOi8vbXktdXJseCcsXG4gICAgICBUaHVtYnByaW50TGlzdDogWyAnRkFLRS1USFVNQlBSSU5UJyBdLFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHJlc3BvbnNlKS50b1N0cmljdEVxdWFsKHtcbiAgICAgIFBoeXNpY2FsUmVzb3VyY2VJZDogJ0ZBS0UtQVJOJyxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnZGVsZXRlJywgYXN5bmMgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBhd2FpdCBpbnZva2VIYW5kbGVyKHtcbiAgICAgIFJlcXVlc3RUeXBlOiAnRGVsZXRlJyxcbiAgICAgIFBoeXNpY2FsUmVzb3VyY2VJZDogJ0ZBS0UtQVJOJyxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBzaW5vbi5hc3NlcnQubm90Q2FsbGVkKGRvd25sb2FkVGh1bWJwcmludCk7XG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChjcmVhdGVPcGVuSURDb25uZWN0UHJvdmlkZXIpO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRXaXRoRXhhY3RseShkZWxldGVPcGVuSURDb25uZWN0UHJvdmlkZXIsIHtcbiAgICAgIE9wZW5JRENvbm5lY3RQcm92aWRlckFybjogJ0ZBS0UtQVJOJyxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgndXBkYXRlIHVybCB3aXRoIGV4cGxpY2l0IHRodW1icHJpbnRzIChyZXBsYWNlKScsIGFzeW5jICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBpbnZva2VIYW5kbGVyKHtcbiAgICAgIFJlcXVlc3RUeXBlOiAnVXBkYXRlJyxcbiAgICAgIFJlc291cmNlUHJvcGVydGllczoge1xuICAgICAgICBTZXJ2aWNlVG9rZW46ICdGb28nLFxuICAgICAgICBVcmw6ICdodHRwczovL25ldycsXG4gICAgICAgIFRodW1icHJpbnRMaXN0OiBbICdUSFVNQjEnLCAnVEhVTUIyJyBdLFxuICAgICAgfSxcbiAgICAgIE9sZFJlc291cmNlUHJvcGVydGllczoge1xuICAgICAgICBVcmw6ICdodHRwczovL29sZCcsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChyZXNwb25zZSkudG9TdHJpY3RFcXVhbCh7XG4gICAgICBQaHlzaWNhbFJlc291cmNlSWQ6ICdGQUtFLUFSTicsXG4gICAgfSk7XG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChkb3dubG9hZFRodW1icHJpbnQpO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRXaXRoRXhhY3RseShjcmVhdGVPcGVuSURDb25uZWN0UHJvdmlkZXIsIHtcbiAgICAgIENsaWVudElETGlzdDogW10sXG4gICAgICBVcmw6ICdodHRwczovL25ldycsXG4gICAgICBUaHVtYnByaW50TGlzdDogWyAnVEhVTUIxJywgJ1RIVU1CMicgXSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgndXBkYXRlIHVybCB3aXRoIG5vIHRodW1icHJpbnQgKHJlcGxhY2UpJywgYXN5bmMgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCByZXNwb25zZSA9IGF3YWl0IGludm9rZUhhbmRsZXIoe1xuICAgICAgUmVxdWVzdFR5cGU6ICdVcGRhdGUnLFxuICAgICAgUmVzb3VyY2VQcm9wZXJ0aWVzOiB7XG4gICAgICAgIFNlcnZpY2VUb2tlbjogJ0ZvbycsXG4gICAgICAgIFVybDogJ2h0dHBzOi8vbmV3JyxcbiAgICAgIH0sXG4gICAgICBPbGRSZXNvdXJjZVByb3BlcnRpZXM6IHtcbiAgICAgICAgVXJsOiAnaHR0cHM6Ly9vbGQnLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QocmVzcG9uc2UpLnRvU3RyaWN0RXF1YWwoe1xuICAgICAgUGh5c2ljYWxSZXNvdXJjZUlkOiAnRkFLRS1BUk4nLFxuICAgIH0pO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRPbmNlV2l0aEV4YWN0bHkoZG93bmxvYWRUaHVtYnByaW50LCAnaHR0cHM6Ly9uZXcnKTtcbiAgICBzaW5vbi5hc3NlcnQuY2FsbGVkT25jZVdpdGhFeGFjdGx5KGNyZWF0ZU9wZW5JRENvbm5lY3RQcm92aWRlciwge1xuICAgICAgQ2xpZW50SURMaXN0OiBbXSxcbiAgICAgIFVybDogJ2h0dHBzOi8vbmV3JyxcbiAgICAgIFRodW1icHJpbnRMaXN0OiBbICdGQUtFLVRIVU1CUFJJTlQnIF0sXG4gICAgfSk7XG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChkZWxldGVPcGVuSURDb25uZWN0UHJvdmlkZXIpO1xuICB9KTtcblxuICB0ZXN0KCd1cGRhdGUgdGh1bWJwcmludCBsaXN0JywgYXN5bmMgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBhd2FpdCBpbnZva2VIYW5kbGVyKHtcbiAgICAgIFJlcXVlc3RUeXBlOiAnVXBkYXRlJyxcbiAgICAgIFBoeXNpY2FsUmVzb3VyY2VJZDogJ0ZBS0UtUGh5c2ljYWxSZXNvdXJjZUlkJyxcbiAgICAgIFJlc291cmNlUHJvcGVydGllczoge1xuICAgICAgICBTZXJ2aWNlVG9rZW46ICdGb28nLFxuICAgICAgICBVcmw6ICdodHRwczovL3VybCcsXG4gICAgICAgIFRodW1icHJpbnRMaXN0OiBbICdGb28nLCAnQmFyJyBdLFxuICAgICAgfSxcbiAgICAgIE9sZFJlc291cmNlUHJvcGVydGllczoge1xuICAgICAgICBVcmw6ICdodHRwczovL3VybCcsXG4gICAgICAgIFRodW1icHJpbnRMaXN0OiBbICdGb28nIF0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHNpbm9uLmFzc2VydC5ub3RDYWxsZWQoZG93bmxvYWRUaHVtYnByaW50KTtcbiAgICBzaW5vbi5hc3NlcnQubm90Q2FsbGVkKGNyZWF0ZU9wZW5JRENvbm5lY3RQcm92aWRlcik7XG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChkZWxldGVPcGVuSURDb25uZWN0UHJvdmlkZXIpO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRPbmNlV2l0aEV4YWN0bHkodXBkYXRlT3BlbklEQ29ubmVjdFByb3ZpZGVyVGh1bWJwcmludCwge1xuICAgICAgT3BlbklEQ29ubmVjdFByb3ZpZGVyQXJuOiAnRkFLRS1QaHlzaWNhbFJlc291cmNlSWQnLFxuICAgICAgVGh1bWJwcmludExpc3Q6IFsgJ0JhcicsICdGb28nIF0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2FkZC9yZW1vdmUgY2xpZW50IGlkcycsIGFzeW5jICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgYXdhaXQgaW52b2tlSGFuZGxlcih7XG4gICAgICBSZXF1ZXN0VHlwZTogJ1VwZGF0ZScsXG4gICAgICBQaHlzaWNhbFJlc291cmNlSWQ6ICdGQUtFLVBoeXNpY2FsUmVzb3VyY2VJZCcsXG4gICAgICBSZXNvdXJjZVByb3BlcnRpZXM6IHtcbiAgICAgICAgU2VydmljZVRva2VuOiAnRm9vJyxcbiAgICAgICAgVXJsOiAnaHR0cHM6Ly91cmwnLFxuICAgICAgICBDbGllbnRJRExpc3Q6IFsgJ0EnLCAnQicsICdDJyBdLFxuICAgICAgfSxcbiAgICAgIE9sZFJlc291cmNlUHJvcGVydGllczoge1xuICAgICAgICBVcmw6ICdodHRwczovL3VybCcsXG4gICAgICAgIENsaWVudElETGlzdDogWyAnQScsICdEJyBdLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBzaW5vbi5hc3NlcnQubm90Q2FsbGVkKGRvd25sb2FkVGh1bWJwcmludCk7XG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChjcmVhdGVPcGVuSURDb25uZWN0UHJvdmlkZXIpO1xuICAgIHNpbm9uLmFzc2VydC5ub3RDYWxsZWQoZGVsZXRlT3BlbklEQ29ubmVjdFByb3ZpZGVyKTtcbiAgICBzaW5vbi5hc3NlcnQubm90Q2FsbGVkKHVwZGF0ZU9wZW5JRENvbm5lY3RQcm92aWRlclRodW1icHJpbnQpO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRUd2ljZShhZGRDbGllbnRJRFRvT3BlbklEQ29ubmVjdFByb3ZpZGVyKTtcbiAgICBzaW5vbi5hc3NlcnQuY2FsbGVkV2l0aEV4YWN0bHkoYWRkQ2xpZW50SURUb09wZW5JRENvbm5lY3RQcm92aWRlciwge1xuICAgICAgT3BlbklEQ29ubmVjdFByb3ZpZGVyQXJuOiAnRkFLRS1QaHlzaWNhbFJlc291cmNlSWQnLCBDbGllbnRJRDogJ0InLFxuICAgIH0pO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRXaXRoRXhhY3RseShhZGRDbGllbnRJRFRvT3BlbklEQ29ubmVjdFByb3ZpZGVyLCB7XG4gICAgICBPcGVuSURDb25uZWN0UHJvdmlkZXJBcm46ICdGQUtFLVBoeXNpY2FsUmVzb3VyY2VJZCcsIENsaWVudElEOiAnQycsXG4gICAgfSk7XG4gICAgc2lub24uYXNzZXJ0LmNhbGxlZE9uY2VXaXRoRXhhY3RseShyZW1vdmVDbGllbnRJREZyb21PcGVuSURDb25uZWN0UHJvdmlkZXIsIHtcbiAgICAgIE9wZW5JRENvbm5lY3RQcm92aWRlckFybjogJ0ZBS0UtUGh5c2ljYWxSZXNvdXJjZUlkJywgQ2xpZW50SUQ6ICdEJyxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnbXVsdGlwbGUgaW4tcGxhY2UgdXBkYXRlcyAobm8gcmVwbGFjZSknLCBhc3luYyAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIGF3YWl0IGludm9rZUhhbmRsZXIoe1xuICAgICAgUmVxdWVzdFR5cGU6ICdVcGRhdGUnLFxuICAgICAgUGh5c2ljYWxSZXNvdXJjZUlkOiAnRkFLRS1QaHlzaWNhbFJlc291cmNlSWQnLFxuICAgICAgUmVzb3VyY2VQcm9wZXJ0aWVzOiB7XG4gICAgICAgIFNlcnZpY2VUb2tlbjogJ0ZvbycsXG4gICAgICAgIFVybDogJ2h0dHBzOi8vdXJsJyxcbiAgICAgICAgVGh1bWJwcmludExpc3Q6IFsgJ05FVy1MSVNUJyBdLFxuICAgICAgICBDbGllbnRJRExpc3Q6IFsgJ0EnIF0sXG4gICAgICB9LFxuICAgICAgT2xkUmVzb3VyY2VQcm9wZXJ0aWVzOiB7XG4gICAgICAgIFVybDogJ2h0dHBzOi8vdXJsJyxcbiAgICAgICAgVGh1bWJwcmludExpc3Q6IFsgJ09MRC1MSVNUJyBdLFxuICAgICAgICBDbGllbnRJRExpc3Q6IFsgXSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChkb3dubG9hZFRodW1icHJpbnQpO1xuICAgIHNpbm9uLmFzc2VydC5ub3RDYWxsZWQoY3JlYXRlT3BlbklEQ29ubmVjdFByb3ZpZGVyKTtcbiAgICBzaW5vbi5hc3NlcnQubm90Q2FsbGVkKGRlbGV0ZU9wZW5JRENvbm5lY3RQcm92aWRlcik7XG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChyZW1vdmVDbGllbnRJREZyb21PcGVuSURDb25uZWN0UHJvdmlkZXIpO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRPbmNlV2l0aEV4YWN0bHkodXBkYXRlT3BlbklEQ29ubmVjdFByb3ZpZGVyVGh1bWJwcmludCwge1xuICAgICAgT3BlbklEQ29ubmVjdFByb3ZpZGVyQXJuOiAnRkFLRS1QaHlzaWNhbFJlc291cmNlSWQnLFxuICAgICAgVGh1bWJwcmludExpc3Q6IFsgJ05FVy1MSVNUJyBdLFxuICAgIH0pO1xuICAgIHNpbm9uLmFzc2VydC5jYWxsZWRPbmNlV2l0aEV4YWN0bHkoYWRkQ2xpZW50SURUb09wZW5JRENvbm5lY3RQcm92aWRlciwge1xuICAgICAgT3BlbklEQ29ubmVjdFByb3ZpZGVyQXJuOiAnRkFLRS1QaHlzaWNhbFJlc291cmNlSWQnLFxuICAgICAgQ2xpZW50SUQ6ICdBJyxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnbXVsdGlwbGUgdXBkYXRlcyB0aGF0IGluY2x1ZGUgYSB1cmwgdXBkYXRlLCB3aGljaCBtZWFucyByZXBsYWNlbWVudCcsIGFzeW5jICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgYXdhaXQgaW52b2tlSGFuZGxlcih7XG4gICAgICBSZXF1ZXN0VHlwZTogJ1VwZGF0ZScsXG4gICAgICBQaHlzaWNhbFJlc291cmNlSWQ6ICdGQUtFLVBoeXNpY2FsUmVzb3VyY2VJZCcsXG4gICAgICBSZXNvdXJjZVByb3BlcnRpZXM6IHtcbiAgICAgICAgU2VydmljZVRva2VuOiAnRm9vJyxcbiAgICAgICAgVXJsOiAnaHR0cHM6Ly9uZXctdXJsJyxcbiAgICAgICAgQ2xpZW50SURMaXN0OiBbICdBJyBdLFxuICAgICAgfSxcbiAgICAgIE9sZFJlc291cmNlUHJvcGVydGllczoge1xuICAgICAgICBVcmw6ICdodHRwczovL29sZC11cmwnLFxuICAgICAgICBUaHVtYnByaW50TGlzdDogWyAnT0xELUxJU1QnIF0sXG4gICAgICAgIENsaWVudElETGlzdDogWyBdLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBzaW5vbi5hc3NlcnQubm90Q2FsbGVkKGRlbGV0ZU9wZW5JRENvbm5lY3RQcm92aWRlcik7XG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChyZW1vdmVDbGllbnRJREZyb21PcGVuSURDb25uZWN0UHJvdmlkZXIpO1xuICAgIHNpbm9uLmFzc2VydC5ub3RDYWxsZWQodXBkYXRlT3BlbklEQ29ubmVjdFByb3ZpZGVyVGh1bWJwcmludCk7XG4gICAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZChhZGRDbGllbnRJRFRvT3BlbklEQ29ubmVjdFByb3ZpZGVyKTtcbiAgICBzaW5vbi5hc3NlcnQuY2FsbGVkT25jZVdpdGhFeGFjdGx5KGRvd25sb2FkVGh1bWJwcmludCwgJ2h0dHBzOi8vbmV3LXVybCcpOyAvLyBzaW5jZSB0aHVtYnByaW50IGxpc3QgaXMgZW1wdHlcbiAgICBzaW5vbi5hc3NlcnQuY2FsbGVkT25jZVdpdGhFeGFjdGx5KGNyZWF0ZU9wZW5JRENvbm5lY3RQcm92aWRlciwge1xuICAgICAgQ2xpZW50SURMaXN0OiBbICdBJyBdLFxuICAgICAgVGh1bWJwcmludExpc3Q6IFsgJ0ZBS0UtVEhVTUJQUklOVCcgXSxcbiAgICAgIFVybDogJ2h0dHBzOi8vbmV3LXVybCcsXG4gICAgfSk7XG4gIH0pO1xuXG59KTtcblxuZGVzY3JpYmUoJ2FycmF5RGlmZicsICgpID0+IHtcbiAgdGVzdCgnY2FsY3VsYXRlcyB0aGUgZGlmZmVyZW5jZSBiZXR3ZWVuIHR3byBhcnJheXMnLCAoKSA9PiB7XG4gICAgZXhwZWN0KGFycmF5RGlmZihbICdhJywgJ2InLCAnYycgXSwgWyAnYScsICdkJyBdKSkudG9TdHJpY3RFcXVhbCh7IGFkZHM6IFsgJ2QnIF0sIGRlbGV0ZXM6IFsgJ2InLCAnYycgXX0pO1xuICAgIGV4cGVjdChhcnJheURpZmYoWyAnYScsICdiJywgJ2MnIF0sIFsgXSkpLnRvU3RyaWN0RXF1YWwoeyBhZGRzOiBbXSwgZGVsZXRlczogWyAnYScsICdiJywgJ2MnIF0gfSk7XG4gICAgZXhwZWN0KGFycmF5RGlmZihbICdhJywgJ2InLCAnYycgXSwgWyAnYScsICdjJywgJ2InIF0pKS50b1N0cmljdEVxdWFsKHsgYWRkczogW10sIGRlbGV0ZXM6IFtdIH0pO1xuICAgIGV4cGVjdChhcnJheURpZmYoWyBdLCBbICdhJywgJ2MnLCAnYicgXSkpLnRvU3RyaWN0RXF1YWwoeyBhZGRzOiBbICdhJywgJ2MnLCAnYicgXSwgZGVsZXRlczogW10gfSk7XG4gICAgZXhwZWN0KGFycmF5RGlmZihbICd4JywgJ3knIF0sIFsgJ2EnLCAnYycsICdiJyBdKSkudG9TdHJpY3RFcXVhbCh7IGFkZHM6IFsgJ2EnLCAnYycsICdiJyBdLCBkZWxldGVzOiBbICd4JywgJ3knIF19KTtcbiAgICBleHBlY3QoYXJyYXlEaWZmKFsgXSwgWyBdKSkudG9TdHJpY3RFcXVhbCh7IGFkZHM6IFtdLCBkZWxldGVzOiBbXSB9KTtcbiAgICBleHBlY3QoYXJyYXlEaWZmKFsgJ2EnLCAnYScgXSwgWyAnYScsICdiJywgJ2EnLCAnYicsICdiJyBdKSkudG9TdHJpY3RFcXVhbCh7IGFkZHM6IFsgJ2InIF0sIGRlbGV0ZXM6IFtdIH0pO1xuICB9KTtcbn0pO1xuXG5hc3luYyBmdW5jdGlvbiBpbnZva2VIYW5kbGVyKGV2ZW50OiBQYXJ0aWFsPEFXU0xhbWJkYS5DbG91ZEZvcm1hdGlvbkN1c3RvbVJlc291cmNlRXZlbnQ+KSB7XG4gIHJldHVybiBhd2FpdCBoYW5kbGVyLmhhbmRsZXIoZXZlbnQgYXMgYW55KTtcbn0iXX0=