"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.POLICY_DESCRIPTION_ANNOTATION = void 0;
exports.checkParseSchema = checkParseSchema;
exports.checkParsePolicy = checkParsePolicy;
exports.getPolicyDescription = getPolicyDescription;
exports.checkParseTemplate = checkParseTemplate;
exports.validatePolicy = validatePolicy;
exports.cleanUpApiNameForNamespace = cleanUpApiNameForNamespace;
exports.buildSchema = buildSchema;
const cedar = require("@cedar-policy/cedar-wasm/nodejs");
exports.POLICY_DESCRIPTION_ANNOTATION = '@cdkDescription';
function checkParseSchema(schemaStr) {
    const schemaParseResult = cedar.checkParseSchema(schemaStr);
    if (schemaParseResult.type === 'error') {
        throw new Error(`Schema is invalid: ${schemaParseResult.errors}`);
    }
}
function checkParsePolicy(policyStatement) {
    const parsePolicyResult = cedar.checkParsePolicySet(policyStatement);
    if (parsePolicyResult.type == 'error') {
        throw new Error(`Invalid policy statement: ${policyStatement}. Errors: ${parsePolicyResult.errors.join(', ')}`);
    }
}
/**
 * Extracts the Description of the Policy searching for the @see POLICY_DESCRIPTION_ANNOTATION annotation on top of policy contents (before effect)
 * @param policyStatement The policy statement in string format
 * @returns Returns the description if found or null
 */
function getPolicyDescription(policyStatement) {
    const regex = new RegExp(String.raw `^${exports.POLICY_DESCRIPTION_ANNOTATION}\(("|')(.*)("|')(\))([\r\n|\r|\n| ]*)(permit|forbid)`);
    let matches = policyStatement.match(regex);
    return (matches) ? matches[2] : null;
}
function checkParseTemplate(templateStatement) {
    const templateParseResult = cedar.checkParseTemplate(templateStatement);
    if (templateParseResult.type === 'error') {
        throw new Error(`Error parsing template: ${templateStatement}. Errors: ${templateParseResult.errors.join(', ')}`);
    }
}
function validatePolicy(policyStatement, schemaStr) {
    const validationResult = cedar.validate({
        validationSettings: { enabled: 'on' },
        schema: {
            json: JSON.parse(schemaStr),
        },
        policySet: policyStatement,
    });
    if (validationResult.type === 'failure') {
        throw new Error(`Policy (contents: ${policyStatement}) could not be parsed: ${validationResult.errors.join('; ')}`);
    }
    if (validationResult.type === 'success' && validationResult.validationErrors.length > 0) {
        throw new Error(`Policy (contents: ${policyStatement}) could not be validated against the schema:
      ${validationResult.validationErrors.join('; ')}`);
    }
}
function cleanUpApiNameForNamespace(apiName) {
    const validCedarName = apiName.replace(/[^a-zA-Z0-9_]/g, '').trim();
    if (validCedarName.length === 0) {
        return 'ImportedApi';
    }
    if (/[0-9_]/.exec(validCedarName[0])) {
        return `Api${validCedarName}`;
    }
    return validCedarName;
}
function buildSchema(namespace, actionNames, principalGroupType) {
    const additionalEntities = {};
    if (principalGroupType) {
        additionalEntities[principalGroupType] = {
            shape: {
                type: 'Record',
                attributes: {},
            },
        };
    }
    const actions = actionNames.reduce((acc, actionName) => {
        return {
            ...acc,
            [actionName]: {
                appliesTo: {
                    context: { type: 'Record', attributes: {} },
                    principalTypes: ['User'],
                    resourceTypes: ['Application'],
                },
            },
        };
    }, {});
    return {
        [namespace]: {
            entityTypes: {
                ...additionalEntities,
                User: {
                    shape: {
                        type: 'Record',
                        attributes: {},
                    },
                    memberOfTypes: principalGroupType ? [principalGroupType] : [],
                },
                Application: {
                    shape: {
                        type: 'Record',
                        attributes: {},
                    },
                },
            },
            actions,
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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