# Copyright (C) 2019 The python-bitcointx developers
#
# This file is part of python-bitcointx.
#
# It is subject to the license terms in the LICENSE file found in the top-level
# directory of this distribution.
#
# No part of python-bitcointx, including this file, may be copied, modified,
# propagated, or distributed except according to the terms contained in the
# LICENSE file.

# pylama:ignore=E501,C901

import unittest
import base64

from bitcointx import ChainParams
from bitcointx.wallet import CCoinExtKey, CCoinExtPubKey
from bitcointx.core import (
    x, lx, b2x, CTransaction, CTxOut, CMutableTxOut, CTxIn, COutPoint,
    coins_to_satoshi
)
from bitcointx.core.key import CPubKey, KeyStore, BIP32Path
from bitcointx.core.script import (
    SIGHASH_ALL, CScript, OP_CHECKMULTISIG,
    parse_standard_multisig_redeem_script, standard_witness_v0_scriptpubkey,
    CScriptWitness
)
from bitcointx.core.serialize import (
    SerializationError, SerializationTruncationError
)
from bitcointx.core.psbt import (
    PartiallySignedTransaction, PSBT_KeyDerivationInfo, PSBT_ProprietaryTypeData
)


class Test_PSBT(unittest.TestCase):

    def test_invalid_psbt(self) -> None:
        def deserialize(hex_data: str) -> PartiallySignedTransaction:
            return PartiallySignedTransaction.deserialize(x(hex_data))

        # Network transaction, not PSBT format
        with self.assertRaisesRegex(SerializationError,
                                    'Invalid PSBT magic bytes'):
            deserialize('0200000001268171371edff285e937adeea4b37b78000c0566cbb3ad64641713ca42171bf6000000006a473044022070b2245123e6bf474d60c5b50c043d4c691a5d2435f09a34a7662a9dc251790a022001329ca9dacf280bdf30740ec0390422422c81cb45839457aeb76fc12edd95b3012102657d118d3357b8e0f4c2cd46db7b39f6d9c38d9a70abcb9b2de5dc8dbfe4ce31feffffff02d3dff505000000001976a914d0c59903c5bac2868760e90fd521a4665aa7652088ac00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787b32e1300')

        # PSBT missing outputs
        with self.assertRaises(SerializationTruncationError):
            deserialize('70736274ff0100750200000001268171371edff285e937adeea4b37b78000c0566cbb3ad64641713ca42171bf60000000000feffffff02d3dff505000000001976a914d0c59903c5bac2868760e90fd521a4665aa7652088ac00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787b32e1300000100fda5010100000000010289a3c71eab4d20e0371bbba4cc698fa295c9463afa2e397f8533ccb62f9567e50100000017160014be18d152a9b012039daf3da7de4f53349eecb985ffffffff86f8aa43a71dff1448893a530a7237ef6b4608bbb2dd2d0171e63aec6a4890b40100000017160014fe3e9ef1a745e974d902c4355943abcb34bd5353ffffffff0200c2eb0b000000001976a91485cff1097fd9e008bb34af709c62197b38978a4888ac72fef84e2c00000017a914339725ba21efd62ac753a9bcd067d6c7a6a39d05870247304402202712be22e0270f394f568311dc7ca9a68970b8025fdd3b240229f07f8a5f3a240220018b38d7dcd314e734c9276bd6fb40f673325bc4baa144c800d2f2f02db2765c012103d2e15674941bad4a996372cb87e1856d3652606d98562fe39c5e9e7e413f210502483045022100d12b852d85dcd961d2f5f4ab660654df6eedcc794c0c33ce5cc309ffb5fce58d022067338a8e0e1725c197fb1a88af59f51e44e4255b20167c8684031c05d1f2592a01210223b72beef0965d10be0778efecd61fcac6f79a4ea169393380734464f84f2ab30000000000')

        # PSBT where one input has a filled scriptSig in the unsigned tx
        with self.assertRaisesRegex(ValueError,
                                    'Unsigned transaction contains non-empty '
                                    'scriptSigs'):
            deserialize('70736274ff0100fd0a010200000002ab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be4000000006a47304402204759661797c01b036b25928948686218347d89864b719e1f7fcf57d1e511658702205309eabf56aa4d8891ffd111fdf1336f3a29da866d7f8486d75546ceedaf93190121035cdc61fc7ba971c0b501a646a2a83b102cb43881217ca682dc86e2d73fa88292feffffffab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40100000000feffffff02603bea0b000000001976a914768a40bbd740cbe81d988e71de2a4d5c71396b1d88ac8e240000000000001976a9146f4620b553fa095e721b9ee0efe9fa039cca459788ac00000000000001012000e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787010416001485d13537f2e265405a34dbafa9e3dda01fb82308000000')

        # PSBT where inputs and outputs are provided but without an unsigned tx
        with self.assertRaisesRegex(ValueError,
                                    '.* does not contain unsigned transaction'):
            deserialize('70736274ff000100fda5010100000000010289a3c71eab4d20e0371bbba4cc698fa295c9463afa2e397f8533ccb62f9567e50100000017160014be18d152a9b012039daf3da7de4f53349eecb985ffffffff86f8aa43a71dff1448893a530a7237ef6b4608bbb2dd2d0171e63aec6a4890b40100000017160014fe3e9ef1a745e974d902c4355943abcb34bd5353ffffffff0200c2eb0b000000001976a91485cff1097fd9e008bb34af709c62197b38978a4888ac72fef84e2c00000017a914339725ba21efd62ac753a9bcd067d6c7a6a39d05870247304402202712be22e0270f394f568311dc7ca9a68970b8025fdd3b240229f07f8a5f3a240220018b38d7dcd314e734c9276bd6fb40f673325bc4baa144c800d2f2f02db2765c012103d2e15674941bad4a996372cb87e1856d3652606d98562fe39c5e9e7e413f210502483045022100d12b852d85dcd961d2f5f4ab660654df6eedcc794c0c33ce5cc309ffb5fce58d022067338a8e0e1725c197fb1a88af59f51e44e4255b20167c8684031c05d1f2592a01210223b72beef0965d10be0778efecd61fcac6f79a4ea169393380734464f84f2ab30000000000')

        # PSBT with duplicate keys in an input
        with self.assertRaisesRegex(SerializationError,
                                    'Duplicate key encountered'):
            deserialize('70736274ff0100750200000001268171371edff285e937adeea4b37b78000c0566cbb3ad64641713ca42171bf60000000000feffffff02d3dff505000000001976a914d0c59903c5bac2868760e90fd521a4665aa7652088ac00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787b32e1300000100fda5010100000000010289a3c71eab4d20e0371bbba4cc698fa295c9463afa2e397f8533ccb62f9567e50100000017160014be18d152a9b012039daf3da7de4f53349eecb985ffffffff86f8aa43a71dff1448893a530a7237ef6b4608bbb2dd2d0171e63aec6a4890b40100000017160014fe3e9ef1a745e974d902c4355943abcb34bd5353ffffffff0200c2eb0b000000001976a91485cff1097fd9e008bb34af709c62197b38978a4888ac72fef84e2c00000017a914339725ba21efd62ac753a9bcd067d6c7a6a39d05870247304402202712be22e0270f394f568311dc7ca9a68970b8025fdd3b240229f07f8a5f3a240220018b38d7dcd314e734c9276bd6fb40f673325bc4baa144c800d2f2f02db2765c012103d2e15674941bad4a996372cb87e1856d3652606d98562fe39c5e9e7e413f210502483045022100d12b852d85dcd961d2f5f4ab660654df6eedcc794c0c33ce5cc309ffb5fce58d022067338a8e0e1725c197fb1a88af59f51e44e4255b20167c8684031c05d1f2592a01210223b72beef0965d10be0778efecd61fcac6f79a4ea169393380734464f84f2ab30000000001003f0200000001ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000000ffffffff010000000000000000036a010000000000000000')

        # PSBT With invalid global transaction typed key (extra key data)
        with self.assertRaisesRegex(SerializationError,
                                    'Unexpected data after key type UNSIGNED_TX'):
            deserialize('70736274ff020001550200000001279a2323a5dfb51fc45f220fa58b0fc13e1e3342792a85d7e36cd6333b5cbc390000000000ffffffff01a05aea0b000000001976a914ffe9c0061097cc3b636f2cb0460fa4fc427d2b4588ac0000000000010120955eea0b0000000017a9146345200f68d189e1adc0df1c4d16ea8f14c0dbeb87220203b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4646304302200424b58effaaa694e1559ea5c93bbfd4a89064224055cdf070b6771469442d07021f5c8eb0fea6516d60b8acb33ad64ede60e8785bfb3aa94b99bdf86151db9a9a010104220020771fd18ad459666dd49f3d564e3dbc42f4c84774e360ada16816a8ed488d5681010547522103b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd462103de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd52ae220603b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4610b4a6ba67000000800000008004000080220603de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd10b4a6ba670000008000000080050000800000')

        # PSBT With invalid input witness utxo typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Unexpected data after key type WITNESS_UTXO'):
            deserialize('70736274ff0100550200000001279a2323a5dfb51fc45f220fa58b0fc13e1e3342792a85d7e36cd6333b5cbc390000000000ffffffff01a05aea0b000000001976a914ffe9c0061097cc3b636f2cb0460fa4fc427d2b4588ac000000000002010020955eea0b0000000017a9146345200f68d189e1adc0df1c4d16ea8f14c0dbeb87220203b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4646304302200424b58effaaa694e1559ea5c93bbfd4a89064224055cdf070b6771469442d07021f5c8eb0fea6516d60b8acb33ad64ede60e8785bfb3aa94b99bdf86151db9a9a010104220020771fd18ad459666dd49f3d564e3dbc42f4c84774e360ada16816a8ed488d5681010547522103b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd462103de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd52ae220603b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4610b4a6ba67000000800000008004000080220603de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd10b4a6ba670000008000000080050000800000')

        # PSBT With invalid pubkey length for input partial signature typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Invalid pubkey encountered in PARTIAL_SIG'):
            deserialize('70736274ff0100550200000001279a2323a5dfb51fc45f220fa58b0fc13e1e3342792a85d7e36cd6333b5cbc390000000000ffffffff01a05aea0b000000001976a914ffe9c0061097cc3b636f2cb0460fa4fc427d2b4588ac0000000000010120955eea0b0000000017a9146345200f68d189e1adc0df1c4d16ea8f14c0dbeb87210203b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd46304302200424b58effaaa694e1559ea5c93bbfd4a89064224055cdf070b6771469442d07021f5c8eb0fea6516d60b8acb33ad64ede60e8785bfb3aa94b99bdf86151db9a9a010104220020771fd18ad459666dd49f3d564e3dbc42f4c84774e360ada16816a8ed488d5681010547522103b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd462103de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd52ae220603b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4610b4a6ba67000000800000008004000080220603de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd10b4a6ba670000008000000080050000800000')

        # PSBT With invalid redeemscript typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Unexpected data after key type REDEEM_SCRIPT'):
            deserialize('70736274ff0100550200000001279a2323a5dfb51fc45f220fa58b0fc13e1e3342792a85d7e36cd6333b5cbc390000000000ffffffff01a05aea0b000000001976a914ffe9c0061097cc3b636f2cb0460fa4fc427d2b4588ac0000000000010120955eea0b0000000017a9146345200f68d189e1adc0df1c4d16ea8f14c0dbeb87220203b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4646304302200424b58effaaa694e1559ea5c93bbfd4a89064224055cdf070b6771469442d07021f5c8eb0fea6516d60b8acb33ad64ede60e8785bfb3aa94b99bdf86151db9a9a01020400220020771fd18ad459666dd49f3d564e3dbc42f4c84774e360ada16816a8ed488d5681010547522103b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd462103de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd52ae220603b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4610b4a6ba67000000800000008004000080220603de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd10b4a6ba670000008000000080050000800000')

        # PSBT With invalid witnessscript typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Unexpected data after key type WITNESS_SCRIPT'):
            deserialize('70736274ff0100550200000001279a2323a5dfb51fc45f220fa58b0fc13e1e3342792a85d7e36cd6333b5cbc390000000000ffffffff01a05aea0b000000001976a914ffe9c0061097cc3b636f2cb0460fa4fc427d2b4588ac0000000000010120955eea0b0000000017a9146345200f68d189e1adc0df1c4d16ea8f14c0dbeb87220203b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4646304302200424b58effaaa694e1559ea5c93bbfd4a89064224055cdf070b6771469442d07021f5c8eb0fea6516d60b8acb33ad64ede60e8785bfb3aa94b99bdf86151db9a9a010104220020771fd18ad459666dd49f3d564e3dbc42f4c84774e360ada16816a8ed488d568102050047522103b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd462103de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd52ae220603b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4610b4a6ba67000000800000008004000080220603de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd10b4a6ba670000008000000080050000800000')

        # PSBT With invalid bip32 typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Invalid pubkey encountered in BIP32_DERIVATION'):
            deserialize('70736274ff0100550200000001279a2323a5dfb51fc45f220fa58b0fc13e1e3342792a85d7e36cd6333b5cbc390000000000ffffffff01a05aea0b000000001976a914ffe9c0061097cc3b636f2cb0460fa4fc427d2b4588ac0000000000010120955eea0b0000000017a9146345200f68d189e1adc0df1c4d16ea8f14c0dbeb87220203b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4646304302200424b58effaaa694e1559ea5c93bbfd4a89064224055cdf070b6771469442d07021f5c8eb0fea6516d60b8acb33ad64ede60e8785bfb3aa94b99bdf86151db9a9a010104220020771fd18ad459666dd49f3d564e3dbc42f4c84774e360ada16816a8ed488d5681010547522103b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd462103de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd52ae210603b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd10b4a6ba67000000800000008004000080220603de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd10b4a6ba670000008000000080050000800000')

        # PSBT With invalid non-witness utxo typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Unexpected data after key type NON_WITNESS_UTXO'):
            deserialize('70736274ff01009a020000000258e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd750000000000ffffffff838d0427d0ec650a68aa46bb0b098aea4422c071b2ca78352a077959d07cea1d0100000000ffffffff0270aaf00800000000160014d85c2b71d0060b09c9886aeb815e50991dda124d00e1f5050000000016001400aea9a2e5f0f876a588df5546e8742d1d87008f0000000000020000bb0200000001aad73931018bd25f84ae400b68848be09db706eac2ac18298babee71ab656f8b0000000048473044022058f6fc7c6a33e1b31548d481c826c015bd30135aad42cd67790dab66d2ad243b02204a1ced2604c6735b6393e5b41691dd78b00f0c5942fb9f751856faa938157dba01feffffff0280f0fa020000000017a9140fb9463421696b82c833af241c78c17ddbde493487d0f20a270100000017a91429ca74f8a08f81999428185c97b5d852e4063f6187650000000107da00473044022074018ad4180097b873323c0015720b3684cc8123891048e7dbcd9b55ad679c99022073d369b740e3eb53dcefa33823c8070514ca55a7dd9544f157c167913261118c01483045022100f61038b308dc1da865a34852746f015772934208c6d24454393cd99bdf2217770220056e675a675a6d0a02b85b14e5e29074d8a25a9b5760bea2816f661910a006ea01475221029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f2102dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d752ae0001012000c2eb0b0000000017a914b7f5faf40e3d40a5a459b1db3535f2b72fa921e8870107232200208c2353173743b595dfb4a07b72ba8e42e3797da74e87fe7d9d7497e3b20289030108da0400473044022062eb7a556107a7c73f45ac4ab5a1dddf6f7075fb1275969a7f383efff784bcb202200c05dbb7470dbf2f08557dd356c7325c1ed30913e996cd3840945db12228da5f01473044022065f45ba5998b59a27ffe1a7bed016af1f1f90d54b3aa8f7450aa5f56a25103bd02207f724703ad1edb96680b284b56d4ffcb88f7fb759eabbe08aa30f29b851383d20147522103089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc21023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7352ae00220203a9a4c37f5996d3aa25dbac6b570af0650394492942460b354753ed9eeca5877110d90c6a4f000000800000008004000080002202027f6399757d2eff55a136ad02c684b1838b6556e5f1b6b34282a94b6b5005109610d90c6a4f00000080000000800500008000')

        # PSBT With invalid final scriptsig typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Unexpected data after key type FINAL_SCRIPTSIG'):
            deserialize('70736274ff01009a020000000258e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd750000000000ffffffff838d0427d0ec650a68aa46bb0b098aea4422c071b2ca78352a077959d07cea1d0100000000ffffffff0270aaf00800000000160014d85c2b71d0060b09c9886aeb815e50991dda124d00e1f5050000000016001400aea9a2e5f0f876a588df5546e8742d1d87008f00000000000100bb0200000001aad73931018bd25f84ae400b68848be09db706eac2ac18298babee71ab656f8b0000000048473044022058f6fc7c6a33e1b31548d481c826c015bd30135aad42cd67790dab66d2ad243b02204a1ced2604c6735b6393e5b41691dd78b00f0c5942fb9f751856faa938157dba01feffffff0280f0fa020000000017a9140fb9463421696b82c833af241c78c17ddbde493487d0f20a270100000017a91429ca74f8a08f81999428185c97b5d852e4063f618765000000020700da00473044022074018ad4180097b873323c0015720b3684cc8123891048e7dbcd9b55ad679c99022073d369b740e3eb53dcefa33823c8070514ca55a7dd9544f157c167913261118c01483045022100f61038b308dc1da865a34852746f015772934208c6d24454393cd99bdf2217770220056e675a675a6d0a02b85b14e5e29074d8a25a9b5760bea2816f661910a006ea01475221029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f2102dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d752ae0001012000c2eb0b0000000017a914b7f5faf40e3d40a5a459b1db3535f2b72fa921e8870107232200208c2353173743b595dfb4a07b72ba8e42e3797da74e87fe7d9d7497e3b20289030108da0400473044022062eb7a556107a7c73f45ac4ab5a1dddf6f7075fb1275969a7f383efff784bcb202200c05dbb7470dbf2f08557dd356c7325c1ed30913e996cd3840945db12228da5f01473044022065f45ba5998b59a27ffe1a7bed016af1f1f90d54b3aa8f7450aa5f56a25103bd02207f724703ad1edb96680b284b56d4ffcb88f7fb759eabbe08aa30f29b851383d20147522103089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc21023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7352ae00220203a9a4c37f5996d3aa25dbac6b570af0650394492942460b354753ed9eeca5877110d90c6a4f000000800000008004000080002202027f6399757d2eff55a136ad02c684b1838b6556e5f1b6b34282a94b6b5005109610d90c6a4f00000080000000800500008000')

        # PSBT With invalid final script witness typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Unexpected data after key type FINAL_SCRIPTWITNESS'):
            deserialize('70736274ff01009a020000000258e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd750000000000ffffffff838d0427d0ec650a68aa46bb0b098aea4422c071b2ca78352a077959d07cea1d0100000000ffffffff0270aaf00800000000160014d85c2b71d0060b09c9886aeb815e50991dda124d00e1f5050000000016001400aea9a2e5f0f876a588df5546e8742d1d87008f00000000000100bb0200000001aad73931018bd25f84ae400b68848be09db706eac2ac18298babee71ab656f8b0000000048473044022058f6fc7c6a33e1b31548d481c826c015bd30135aad42cd67790dab66d2ad243b02204a1ced2604c6735b6393e5b41691dd78b00f0c5942fb9f751856faa938157dba01feffffff0280f0fa020000000017a9140fb9463421696b82c833af241c78c17ddbde493487d0f20a270100000017a91429ca74f8a08f81999428185c97b5d852e4063f6187650000000107da00473044022074018ad4180097b873323c0015720b3684cc8123891048e7dbcd9b55ad679c99022073d369b740e3eb53dcefa33823c8070514ca55a7dd9544f157c167913261118c01483045022100f61038b308dc1da865a34852746f015772934208c6d24454393cd99bdf2217770220056e675a675a6d0a02b85b14e5e29074d8a25a9b5760bea2816f661910a006ea01475221029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f2102dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d752ae0001012000c2eb0b0000000017a914b7f5faf40e3d40a5a459b1db3535f2b72fa921e8870107232200208c2353173743b595dfb4a07b72ba8e42e3797da74e87fe7d9d7497e3b2028903020800da0400473044022062eb7a556107a7c73f45ac4ab5a1dddf6f7075fb1275969a7f383efff784bcb202200c05dbb7470dbf2f08557dd356c7325c1ed30913e996cd3840945db12228da5f01473044022065f45ba5998b59a27ffe1a7bed016af1f1f90d54b3aa8f7450aa5f56a25103bd02207f724703ad1edb96680b284b56d4ffcb88f7fb759eabbe08aa30f29b851383d20147522103089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc21023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7352ae00220203a9a4c37f5996d3aa25dbac6b570af0650394492942460b354753ed9eeca5877110d90c6a4f000000800000008004000080002202027f6399757d2eff55a136ad02c684b1838b6556e5f1b6b34282a94b6b5005109610d90c6a4f00000080000000800500008000')

        # PSBT With invalid pubkey in output BIP 32 derivation paths typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Invalid pubkey encountered in BIP32_DERIVATION'):
            deserialize('70736274ff01009a020000000258e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd750000000000ffffffff838d0427d0ec650a68aa46bb0b098aea4422c071b2ca78352a077959d07cea1d0100000000ffffffff0270aaf00800000000160014d85c2b71d0060b09c9886aeb815e50991dda124d00e1f5050000000016001400aea9a2e5f0f876a588df5546e8742d1d87008f00000000000100bb0200000001aad73931018bd25f84ae400b68848be09db706eac2ac18298babee71ab656f8b0000000048473044022058f6fc7c6a33e1b31548d481c826c015bd30135aad42cd67790dab66d2ad243b02204a1ced2604c6735b6393e5b41691dd78b00f0c5942fb9f751856faa938157dba01feffffff0280f0fa020000000017a9140fb9463421696b82c833af241c78c17ddbde493487d0f20a270100000017a91429ca74f8a08f81999428185c97b5d852e4063f6187650000000107da00473044022074018ad4180097b873323c0015720b3684cc8123891048e7dbcd9b55ad679c99022073d369b740e3eb53dcefa33823c8070514ca55a7dd9544f157c167913261118c01483045022100f61038b308dc1da865a34852746f015772934208c6d24454393cd99bdf2217770220056e675a675a6d0a02b85b14e5e29074d8a25a9b5760bea2816f661910a006ea01475221029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f2102dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d752ae0001012000c2eb0b0000000017a914b7f5faf40e3d40a5a459b1db3535f2b72fa921e8870107232200208c2353173743b595dfb4a07b72ba8e42e3797da74e87fe7d9d7497e3b20289030108da0400473044022062eb7a556107a7c73f45ac4ab5a1dddf6f7075fb1275969a7f383efff784bcb202200c05dbb7470dbf2f08557dd356c7325c1ed30913e996cd3840945db12228da5f01473044022065f45ba5998b59a27ffe1a7bed016af1f1f90d54b3aa8f7450aa5f56a25103bd02207f724703ad1edb96680b284b56d4ffcb88f7fb759eabbe08aa30f29b851383d20147522103089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc21023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7352ae00210203a9a4c37f5996d3aa25dbac6b570af0650394492942460b354753ed9eeca58710d90c6a4f000000800000008004000080002202027f6399757d2eff55a136ad02c684b1838b6556e5f1b6b34282a94b6b5005109610d90c6a4f00000080000000800500008000')

        # PSBT With invalid input sighash type typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Unexpected data after key type SIGHASH_TYPE'):
            deserialize('70736274ff0100730200000001301ae986e516a1ec8ac5b4bc6573d32f83b465e23ad76167d68b38e730b4dbdb0000000000ffffffff02747b01000000000017a91403aa17ae882b5d0d54b25d63104e4ffece7b9ea2876043993b0000000017a914b921b1ba6f722e4bfa83b6557a3139986a42ec8387000000000001011f00ca9a3b00000000160014d2d94b64ae08587eefc8eeb187c601e939f9037c0203000100000000010016001462e9e982fff34dd8239610316b090cd2a3b747cb000100220020876bad832f1d168015ed41232a9ea65a1815d9ef13c0ef8759f64b5b2b278a65010125512103b7ce23a01c5b4bf00a642537cdfabb315b668332867478ef51309d2bd57f8a8751ae00')

        # PSBT With invalid output redeemScript typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Unexpected data after key type REDEEM_SCRIPT'):
            deserialize('70736274ff0100730200000001301ae986e516a1ec8ac5b4bc6573d32f83b465e23ad76167d68b38e730b4dbdb0000000000ffffffff02747b01000000000017a91403aa17ae882b5d0d54b25d63104e4ffece7b9ea2876043993b0000000017a914b921b1ba6f722e4bfa83b6557a3139986a42ec8387000000000001011f00ca9a3b00000000160014d2d94b64ae08587eefc8eeb187c601e939f9037c0002000016001462e9e982fff34dd8239610316b090cd2a3b747cb000100220020876bad832f1d168015ed41232a9ea65a1815d9ef13c0ef8759f64b5b2b278a65010125512103b7ce23a01c5b4bf00a642537cdfabb315b668332867478ef51309d2bd57f8a8751ae00')

        # PSBT With invalid output witnessScript typed key
        with self.assertRaisesRegex(SerializationError,
                                    'Unexpected data after key type WITNESS_SCRIPT'):
            deserialize('70736274ff0100730200000001301ae986e516a1ec8ac5b4bc6573d32f83b465e23ad76167d68b38e730b4dbdb0000000000ffffffff02747b01000000000017a91403aa17ae882b5d0d54b25d63104e4ffece7b9ea2876043993b0000000017a914b921b1ba6f722e4bfa83b6557a3139986a42ec8387000000000001011f00ca9a3b00000000160014d2d94b64ae08587eefc8eeb187c601e939f9037c00010016001462e9e982fff34dd8239610316b090cd2a3b747cb000100220020876bad832f1d168015ed41232a9ea65a1815d9ef13c0ef8759f64b5b2b278a6521010025512103b7ce23a01c5b4bf00a642537cdfabb315b668332867478ef51309d06d57f8a8751ae00')

        with self.assertRaisesRegex(ValueError, 'Invalid CTxOut provided for utxo'):
            PartiallySignedTransaction.from_base64('cHNidP8BAHECAAAAAfA00BFgAm6tp86RowwH6BMImQNL5zXUcTT97XoLGz0BAAAAAAD/////AgD5ApUAAAAAFgAUKNw0x8HRctAgmvoevm4u1SbN7XL87QKVAAAAABYAFPck4gF7iL4NL4wtfRAKgQbghiTUAAAAAAABAR8AgIFq49AHABYAFJUDtxf2PHo641HEOBOAIvFMNTr2AAAA')

        with self.assertRaisesRegex(ValueError, 'txout.nValue too high'):
            PartiallySignedTransaction.from_base64('cHNidP8BAHECAAAAAfA00BFgAm6tp86RowwH6BMImQNL5zXUcTT97XoLGz0BAAAAAAD/////AgCAgWrj0AcAFgAUKNw0x8HRctAgmvoevm4u1SbN7XL87QKVAAAAABYAFPck4gF7iL4NL4wtfRAKgQbghiTUAAAAAAABAR8A8gUqAQAAABYAFJUDtxf2PHo641HEOBOAIvFMNTr2AAAA')

        with self.assertRaisesRegex(ValueError, 'prevout index in unsigned_tx is beyond the length of utxo.vout'):
            PartiallySignedTransaction.from_base64('cHNidP8BAJoCAAAAAkvEW8NnDtdNtDpsmze+Ht2LH35IJcKv00jKAlUs21RrAwAAAAD/////S8Rbw2cO1020OmybN74e3Ysffkglwq/TSMoCVSzbVGsBAAAAAP7///8CwLYClQAAAAAWABSNJKzjaUb3uOxixsvh1GGE3fW7zQD5ApUAAAAAFgAUKNw0x8HRctAgmvoevm4u1SbN7XIAAAAAAAEAnQIAAAACczMa321tVHuN4GKWKRncycI22aX3uXgwSFUKM2orjRsBAAAAAP7///9zMxrfbW1Ue43gYpYpGdzJwjbZpfe5eDBIVQozaiuNGwAAAAAA/v///wIA+QKVAAAAABl2qRT9zXUVA8Ls5iVqynLHe5/vSe1XyYisQM0ClQAAAAAWABRmWQUcjSjghQ8/uH4Bn/zkakwLtAAAAAAAAQEfQM0ClQAAAAAWABRmWQUcjSjghQ8/uH4Bn/zkakwLtAAAAA==')

        with self.assertRaisesRegex(ValueError, 'sum of input amounts is out of valid range'):
            PartiallySignedTransaction.from_base64('cHNidP8BAJoCAAAAAvA00BFgAm6tp86RowwH6BMImQNL5zXUcTT97XoLGz0BAAAAAAD/////8DTQEWACbq2nzpGjDAfoEwiZA0vnNdRxNP3tegsbPQEBAAAAAP////8CAPkClQAAAAAWABQo3DTHwdFy0CCa+h6+bi7VJs3tcvztApUAAAAAFgAU9yTiAXuIvg0vjC19EAqBBuCGJNQAAAAAAAEBHwAAu8VkQwQAFgAU1EyyBQKdNymMjX5teAIUYQ+kIc0AAQEfAAC7xWRDBAAWABSSuJ3ylxfvTwYlM3tgxCBvDoOVXgAAAA==')

        with self.assertRaisesRegex(ValueError, 'CheckTransaction\\(\\) : txout total out of range'):
            PartiallySignedTransaction.from_base64('cHNidP8BAHECAAAAAfA00BFgAm6tp86RowwH6BMImQNL5zXUcTT97XoLGz0BAAAAAAD/////AgAAu8VkQwQAFgAUKNw0x8HRctAgmvoevm4u1SbN7XIAALvFZEMEABYAFPck4gF7iL4NL4wtfRAKgQbghiTUAAAAAAABAR+AsU8BAAAAABYAFJUDtxf2PHo641HEOBOAIvFMNTr2AAAA')

    def test_valid_psbt(self) -> None:
        def deserialize(hex_data: str) -> PartiallySignedTransaction:
            data = x(hex_data)
            psbt = PartiallySignedTransaction.deserialize(data)

            self.assertEqual(len(psbt.unsigned_tx.vin), len(psbt.inputs))
            self.assertEqual(len(psbt.unsigned_tx.vout), len(psbt.outputs))

            # Note: this check will not pass with _any_ PSBT,
            # but have to pass with our test PSBTs
            self.assertEqual(psbt.serialize(), data)

            return psbt

        tx_data = x('0100000000010289a3c71eab4d20e0371bbba4cc698fa295c9463afa2e397f8533ccb62f9567e50100000017160014be18d152a9b012039daf3da7de4f53349eecb985ffffffff86f8aa43a71dff1448893a530a7237ef6b4608bbb2dd2d0171e63aec6a4890b40100000017160014fe3e9ef1a745e974d902c4355943abcb34bd5353ffffffff0200c2eb0b000000001976a91485cff1097fd9e008bb34af709c62197b38978a4888ac72fef84e2c00000017a914339725ba21efd62ac753a9bcd067d6c7a6a39d05870247304402202712be22e0270f394f568311dc7ca9a68970b8025fdd3b240229f07f8a5f3a240220018b38d7dcd314e734c9276bd6fb40f673325bc4baa144c800d2f2f02db2765c012103d2e15674941bad4a996372cb87e1856d3652606d98562fe39c5e9e7e413f210502483045022100d12b852d85dcd961d2f5f4ab660654df6eedcc794c0c33ce5cc309ffb5fce58d022067338a8e0e1725c197fb1a88af59f51e44e4255b20167c8684031c05d1f2592a01210223b72beef0965d10be0778efecd61fcac6f79a4ea169393380734464f84f2ab300000000')
        txout_data = x('00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787')
        redeem_script_data = x('001485d13537f2e265405a34dbafa9e3dda01fb82308')
        scriptsig_data = x('47304402204759661797c01b036b25928948686218347d89864b719e1f7fcf57d1e511658702205309eabf56aa4d8891ffd111fdf1336f3a29da866d7f8486d75546ceedaf93190121035cdc61fc7ba971c0b501a646a2a83b102cb43881217ca682dc86e2d73fa88292')

        # PSBT with one P2PKH input. Outputs are empty
        psbt = deserialize('70736274ff0100750200000001268171371edff285e937adeea4b37b78000c0566cbb3ad64641713ca42171bf60000000000feffffff02d3dff505000000001976a914d0c59903c5bac2868760e90fd521a4665aa7652088ac00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787b32e1300000100fda5010100000000010289a3c71eab4d20e0371bbba4cc698fa295c9463afa2e397f8533ccb62f9567e50100000017160014be18d152a9b012039daf3da7de4f53349eecb985ffffffff86f8aa43a71dff1448893a530a7237ef6b4608bbb2dd2d0171e63aec6a4890b40100000017160014fe3e9ef1a745e974d902c4355943abcb34bd5353ffffffff0200c2eb0b000000001976a91485cff1097fd9e008bb34af709c62197b38978a4888ac72fef84e2c00000017a914339725ba21efd62ac753a9bcd067d6c7a6a39d05870247304402202712be22e0270f394f568311dc7ca9a68970b8025fdd3b240229f07f8a5f3a240220018b38d7dcd314e734c9276bd6fb40f673325bc4baa144c800d2f2f02db2765c012103d2e15674941bad4a996372cb87e1856d3652606d98562fe39c5e9e7e413f210502483045022100d12b852d85dcd961d2f5f4ab660654df6eedcc794c0c33ce5cc309ffb5fce58d022067338a8e0e1725c197fb1a88af59f51e44e4255b20167c8684031c05d1f2592a01210223b72beef0965d10be0778efecd61fcac6f79a4ea169393380734464f84f2ab300000000000000')
        self.assertEqual(len(psbt.inputs), 1)
        self.assertEqual(len(psbt.outputs), 2)
        assert isinstance(psbt.inputs[0].utxo, CTransaction)
        self.assertIsInstance(psbt.inputs[0].non_witness_utxo, CTransaction)
        self.assertEqual(psbt.inputs[0].witness_utxo, None)
        self.assertEqual(psbt.inputs[0].utxo.serialize(), tx_data)
        self.assertTrue(psbt.outputs[0].is_null())
        self.assertTrue(psbt.outputs[1].is_null())

        # PSBT with one P2PKH input and one P2SH-P2WPKH input.
        # First input is signed and finalized. Outputs are empty
        psbt = deserialize('70736274ff0100a00200000002ab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40000000000feffffffab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40100000000feffffff02603bea0b000000001976a914768a40bbd740cbe81d988e71de2a4d5c71396b1d88ac8e240000000000001976a9146f4620b553fa095e721b9ee0efe9fa039cca459788ac000000000001076a47304402204759661797c01b036b25928948686218347d89864b719e1f7fcf57d1e511658702205309eabf56aa4d8891ffd111fdf1336f3a29da866d7f8486d75546ceedaf93190121035cdc61fc7ba971c0b501a646a2a83b102cb43881217ca682dc86e2d73fa882920001012000e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787010416001485d13537f2e265405a34dbafa9e3dda01fb82308000000')
        self.assertEqual(len(psbt.inputs), 2)
        self.assertEqual(len(psbt.outputs), 2)
        self.assertEqual(psbt.inputs[0].final_script_sig, scriptsig_data)
        assert isinstance(psbt.inputs[1].utxo, CTxOut)
        self.assertEqual(psbt.inputs[1].utxo.serialize(), txout_data)
        self.assertEqual(psbt.inputs[1].redeem_script, redeem_script_data)
        self.assertTrue(psbt.outputs[0].is_null())
        self.assertTrue(psbt.outputs[1].is_null())
        self.assertTrue(psbt.inputs[0].sighash_type is None)
        self.assertTrue(psbt.inputs[1].sighash_type is None)

        # PSBT with one P2PKH input which has a non-final scriptSig
        # and has a sighash type specified. Outputs are empty
        psbt = deserialize('70736274ff0100750200000001268171371edff285e937adeea4b37b78000c0566cbb3ad64641713ca42171bf60000000000feffffff02d3dff505000000001976a914d0c59903c5bac2868760e90fd521a4665aa7652088ac00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787b32e1300000100fda5010100000000010289a3c71eab4d20e0371bbba4cc698fa295c9463afa2e397f8533ccb62f9567e50100000017160014be18d152a9b012039daf3da7de4f53349eecb985ffffffff86f8aa43a71dff1448893a530a7237ef6b4608bbb2dd2d0171e63aec6a4890b40100000017160014fe3e9ef1a745e974d902c4355943abcb34bd5353ffffffff0200c2eb0b000000001976a91485cff1097fd9e008bb34af709c62197b38978a4888ac72fef84e2c00000017a914339725ba21efd62ac753a9bcd067d6c7a6a39d05870247304402202712be22e0270f394f568311dc7ca9a68970b8025fdd3b240229f07f8a5f3a240220018b38d7dcd314e734c9276bd6fb40f673325bc4baa144c800d2f2f02db2765c012103d2e15674941bad4a996372cb87e1856d3652606d98562fe39c5e9e7e413f210502483045022100d12b852d85dcd961d2f5f4ab660654df6eedcc794c0c33ce5cc309ffb5fce58d022067338a8e0e1725c197fb1a88af59f51e44e4255b20167c8684031c05d1f2592a01210223b72beef0965d10be0778efecd61fcac6f79a4ea169393380734464f84f2ab30000000001030401000000000000')
        self.assertEqual(len(psbt.inputs), 1)
        self.assertEqual(len(psbt.outputs), 2)
        self.assertEqual(psbt.inputs[0].sighash_type, SIGHASH_ALL)
        assert isinstance(psbt.inputs[0].utxo, CTransaction)
        self.assertEqual(psbt.inputs[0].utxo.serialize(), tx_data)
        self.assertTrue(psbt.outputs[0].is_null())
        self.assertTrue(psbt.outputs[1].is_null())

        # PSBT with one P2PKH input and one P2SH-P2WPKH input both with
        # non-final scriptSigs. P2SH-P2WPKH input's redeemScript is available.
        # Outputs filled.
        psbt = deserialize('70736274ff0100a00200000002ab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40000000000feffffffab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40100000000feffffff02603bea0b000000001976a914768a40bbd740cbe81d988e71de2a4d5c71396b1d88ac8e240000000000001976a9146f4620b553fa095e721b9ee0efe9fa039cca459788ac00000000000100df0200000001268171371edff285e937adeea4b37b78000c0566cbb3ad64641713ca42171bf6000000006a473044022070b2245123e6bf474d60c5b50c043d4c691a5d2435f09a34a7662a9dc251790a022001329ca9dacf280bdf30740ec0390422422c81cb45839457aeb76fc12edd95b3012102657d118d3357b8e0f4c2cd46db7b39f6d9c38d9a70abcb9b2de5dc8dbfe4ce31feffffff02d3dff505000000001976a914d0c59903c5bac2868760e90fd521a4665aa7652088ac00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787b32e13000001012000e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787010416001485d13537f2e265405a34dbafa9e3dda01fb8230800220202ead596687ca806043edc3de116cdf29d5e9257c196cd055cf698c8d02bf24e9910b4a6ba670000008000000080020000800022020394f62be9df19952c5587768aeb7698061ad2c4a25c894f47d8c162b4d7213d0510b4a6ba6700000080010000800200008000')

        tx_data = x('0200000001268171371edff285e937adeea4b37b78000c0566cbb3ad64641713ca42171bf6000000006a473044022070b2245123e6bf474d60c5b50c043d4c691a5d2435f09a34a7662a9dc251790a022001329ca9dacf280bdf30740ec0390422422c81cb45839457aeb76fc12edd95b3012102657d118d3357b8e0f4c2cd46db7b39f6d9c38d9a70abcb9b2de5dc8dbfe4ce31feffffff02d3dff505000000001976a914d0c59903c5bac2868760e90fd521a4665aa7652088ac00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787b32e1300')
        txout_data = x('00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787')
        pub1 = CPubKey(x('02ead596687ca806043edc3de116cdf29d5e9257c196cd055cf698c8d02bf24e99'))
        pub2 = CPubKey(x('0394f62be9df19952c5587768aeb7698061ad2c4a25c894f47d8c162b4d7213d05'))

        self.assertEqual(len(psbt.inputs), 2)
        self.assertEqual(len(psbt.outputs), 2)
        assert isinstance(psbt.inputs[0].utxo, CTransaction)
        assert isinstance(psbt.inputs[1].utxo, CTxOut)
        self.assertEqual(psbt.inputs[0].utxo.serialize(), tx_data)
        self.assertEqual(psbt.inputs[1].utxo.serialize(), txout_data)
        self.assertEqual(
            psbt.outputs[0].derivation_map[pub1].master_fp, x('b4a6ba67'))
        self.assertEqual(
            str(psbt.outputs[0].derivation_map[pub1].path), "m/0'/0'/2'")
        self.assertEqual(
            psbt.outputs[1].derivation_map[pub2].master_fp,
            x('b4a6ba67'))
        self.assertEqual(
            str(psbt.outputs[1].derivation_map[pub2].path), "m/0'/1'/2'")
        self.assertEqual(psbt.inputs[1].utxo.serialize(), txout_data)

        # PSBT with one P2SH-P2WSH input of a 2-of-2 multisig, redeemScript,
        # witnessScript, and keypaths are available. Contains one signature.
        psbt = deserialize('70736274ff0100550200000001279a2323a5dfb51fc45f220fa58b0fc13e1e3342792a85d7e36cd6333b5cbc390000000000ffffffff01a05aea0b000000001976a914ffe9c0061097cc3b636f2cb0460fa4fc427d2b4588ac0000000000010120955eea0b0000000017a9146345200f68d189e1adc0df1c4d16ea8f14c0dbeb87220203b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4646304302200424b58effaaa694e1559ea5c93bbfd4a89064224055cdf070b6771469442d07021f5c8eb0fea6516d60b8acb33ad64ede60e8785bfb3aa94b99bdf86151db9a9a010104220020771fd18ad459666dd49f3d564e3dbc42f4c84774e360ada16816a8ed488d5681010547522103b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd462103de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd52ae220603b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd4610b4a6ba67000000800000008004000080220603de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd10b4a6ba670000008000000080050000800000')

        txout_data = x('955eea0b0000000017a9146345200f68d189e1adc0df1c4d16ea8f14c0dbeb87')
        pubkey = CPubKey(x('03b1341ccba7683b6af4f1238cd6e97e7167d569fac47f1e48d47541844355bd46'))
        witness_script = CScript([
            2,
            pubkey,
            x('03de55d1e1dac805e3f8a58c1fbf9b94c02f3dbaafe127fefca4995f26f82083bd'),
            2, OP_CHECKMULTISIG
        ])
        redeem_script_data = witness_script.to_p2wsh_scriptPubKey()
        signature_data = x('304302200424b58effaaa694e1559ea5c93bbfd4a89064224055cdf070b6771469442d07021f5c8eb0fea6516d60b8acb33ad64ede60e8785bfb3aa94b99bdf86151db9a9a01')

        self.assertEqual(len(psbt.inputs), 1)
        self.assertEqual(len(psbt.outputs), 1)
        self.assertTrue(psbt.outputs[0].is_null())
        assert isinstance(psbt.inputs[0].utxo, CTxOut)
        self.assertEqual(psbt.inputs[0].utxo.serialize(), txout_data)
        self.assertEqual(psbt.inputs[0].redeem_script, redeem_script_data)
        self.assertEqual(psbt.inputs[0].witness_script, witness_script)
        self.assertEqual(psbt.inputs[0].partial_sigs[pubkey],
                         signature_data)

        # test data has testnet xpubs in this case
        with ChainParams('bitcoin/testnet'):
            # PSBT with one P2WSH input of a 2-of-2 multisig. witnessScript,
            # keypaths, and global xpubs are available.
            # Contains no signatures. Outputs filled.
            pubkey1 = CPubKey(x('029da12cdb5b235692b91536afefe5c91c3ab9473d8e43b533836ab456299c8871'))
            pubkey2 = CPubKey(x('03372b34234ed7cf9c1fea5d05d441557927be9542b162eb02e1ab2ce80224c00b'))
            pubkey3 = CPubKey(x('039eff1f547a1d5f92dfa2ba7af6ac971a4bd03ba4a734b03156a256b8ad3a1ef9'))
            xpub1 = CCoinExtPubKey('tpubDBkJeJo2X94Yq3RVz65DoUgyLUkaDrkfyrn2VcgyCRSKCRonvKvCF2FpYDGJWDkdRHBajXJGpc63GnumUt63ySvqCu2XaTRGVTKMYGuFk9H')
            xpub2 = CCoinExtPubKey('tpubDBkJeJo2X94YsvtBEU1eKoibEWiNv51nW5iHhs6VZp59jsE6nen8KZMFyGHuGbCvqjRqirgeMcfpVBkttpUUT6brm4duzSGoZeTbhqCNUu6')
            witness_script = CScript([2, pubkey1, pubkey2, 2, OP_CHECKMULTISIG])
            redeem_script_data = witness_script.to_p2wsh_scriptPubKey()
            psbt = deserialize('70736274ff01005202000000019dfc6628c26c5899fe1bd3dc338665bfd55d7ada10f6220973df2d386dec12760100000000ffffffff01f03dcd1d000000001600147b3a00bfdc14d27795c2b74901d09da6ef133579000000004f01043587cf02da3fd0088000000097048b1ad0445b1ec8275517727c87b4e4ebc18a203ffa0f94c01566bd38e9000351b743887ee1d40dc32a6043724f2d6459b3b5a4d73daec8fbae0472f3bc43e20cd90c6a4fae000080000000804f01043587cf02da3fd00880000001b90452427139cd78c2cff2444be353cd58605e3e513285e528b407fae3f6173503d30a5e97c8adbc557dac2ad9a7e39c1722ebac69e668b6f2667cc1d671c83cab0cd90c6a4fae000080010000800001012b0065cd1d000000002200202c5486126c4978079a814e13715d65f36459e4d6ccaded266d0508645bafa6320105475221029da12cdb5b235692b91536afefe5c91c3ab9473d8e43b533836ab456299c88712103372b34234ed7cf9c1fea5d05d441557927be9542b162eb02e1ab2ce80224c00b52ae2206029da12cdb5b235692b91536afefe5c91c3ab9473d8e43b533836ab456299c887110d90c6a4fae0000800000008000000000220603372b34234ed7cf9c1fea5d05d441557927be9542b162eb02e1ab2ce80224c00b10d90c6a4fae0000800100008000000000002202039eff1f547a1d5f92dfa2ba7af6ac971a4bd03ba4a734b03156a256b8ad3a1ef910ede45cc500000080000000800100008000')
            self.assertEqual(len(psbt.inputs), 1)
            self.assertEqual(len(psbt.outputs), 1)
            assert isinstance(psbt.inputs[0].utxo, CTxOut)
            self.assertEqual(psbt.inputs[0].utxo.scriptPubKey,
                             witness_script.to_p2wsh_scriptPubKey())
            self.assertEqual(psbt.inputs[0].witness_script, witness_script)
            self.assertEqual(psbt.inputs[0].redeem_script, CScript([]))
            self.assertEqual(
                psbt.inputs[0].derivation_map[pubkey1].master_fp,
                x('d90c6a4f'))
            self.assertEqual(
                str(psbt.inputs[0].derivation_map[pubkey1].path),
                "m/174'/0'/0")
            self.assertEqual(
                psbt.inputs[0].derivation_map[pubkey2].master_fp,
                x('d90c6a4f'))
            self.assertEqual(
                str(psbt.inputs[0].derivation_map[pubkey2].path),
                "m/174'/1'/0")
            self.assertFalse(psbt.outputs[0].is_null())
            self.assertEqual(
                psbt.outputs[0].derivation_map[pubkey3].master_fp,
                x('ede45cc5'))
            self.assertEqual(
                str(psbt.outputs[0].derivation_map[pubkey3].path),
                "m/0'/0'/1'")
            self.assertEqual(len(psbt.xpubs), 2)
            self.assertTrue(xpub1 in psbt.xpubs)
            self.assertTrue(xpub2 in psbt.xpubs)
            self.assertEqual(psbt.xpubs[xpub1].master_fp, x('d90c6a4f'))
            self.assertEqual(str(psbt.xpubs[xpub1].path), "m/174'/0'")
            self.assertEqual(psbt.xpubs[xpub2].master_fp, x('d90c6a4f'))
            self.assertEqual(str(psbt.xpubs[xpub2].path), "m/174'/1'")

        # PSBT with unknown types in the inputs
        psbt = deserialize('70736274ff01003f0200000001ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000000ffffffff010000000000000000036a010000000000000a0f0102030405060708090f0102030405060708090a0b0c0d0e0f0000')
        self.assertEqual(len(psbt.inputs), 1)
        self.assertEqual(len(psbt.outputs), 1)
        self.assertEqual(len(psbt.inputs[0].unknown_fields), 1)
        self.assertEqual(psbt.inputs[0].unknown_fields[0].key_type, 15)
        self.assertEqual(psbt.inputs[0].unknown_fields[0].key_data, x('010203040506070809'))
        self.assertEqual(psbt.inputs[0].unknown_fields[0].value, x('0102030405060708090a0b0c0d0e0f'))

        # PSBT with `PSBT_GLOBAL_XPUB`
        psbt = deserialize('70736274ff01009d0100000002710ea76ab45c5cb6438e607e59cc037626981805ae9e0dfd9089012abb0be5350100000000ffffffff190994d6a8b3c8c82ccbcfb2fba4106aa06639b872a8d447465c0d42588d6d670000000000ffffffff0200e1f505000000001976a914b6bc2c0ee5655a843d79afedd0ccc3f7dd64340988ac605af405000000001600141188ef8e4ce0449eaac8fb141cbf5a1176e6a088000000004f010488b21e039e530cac800000003dbc8a5c9769f031b17e77fea1518603221a18fd18f2b9a54c6c8c1ac75cbc3502f230584b155d1c7f1cd45120a653c48d650b431b67c5b2c13f27d7142037c1691027569c503100008000000080000000800001011f00e1f5050000000016001433b982f91b28f160c920b4ab95e58ce50dda3a4a220203309680f33c7de38ea6a47cd4ecd66f1f5a49747c6ffb8808ed09039243e3ad5c47304402202d704ced830c56a909344bd742b6852dccd103e963bae92d38e75254d2bb424502202d86c437195df46c0ceda084f2a291c3da2d64070f76bf9b90b195e7ef28f77201220603309680f33c7de38ea6a47cd4ecd66f1f5a49747c6ffb8808ed09039243e3ad5c1827569c5031000080000000800000008000000000010000000001011f00e1f50500000000160014388fb944307eb77ef45197d0b0b245e079f011de220202c777161f73d0b7c72b9ee7bde650293d13f095bc7656ad1f525da5fd2e10b11047304402204cb1fb5f869c942e0e26100576125439179ae88dca8a9dc3ba08f7953988faa60220521f49ca791c27d70e273c9b14616985909361e25be274ea200d7e08827e514d01220602c777161f73d0b7c72b9ee7bde650293d13f095bc7656ad1f525da5fd2e10b1101827569c5031000080000000800000008000000000000000000000220202d20ca502ee289686d21815bd43a80637b0698e1fbcdbe4caed445f6c1a0a90ef1827569c50310000800000008000000080000000000400000000')
        xpub = CCoinExtPubKey('xpub6CpGH79LXVkeiux2ZPWMpEubBrRfgcGCgy2HiagyN6NW3qdioJaqFYyD1fG6LDfxWEhMXJqcDuU5VneKt5UQYUGPa5Mfxdw2D2NArwX5TBm')
        self.assertEqual(len(psbt.inputs), 2)
        self.assertEqual(len(psbt.outputs), 2)
        self.assertEqual(psbt.xpubs[xpub].master_fp, x('27569c50'))
        self.assertEqual(str(psbt.xpubs[xpub].path), "m/49'/0'/0'")

        # PSBT with proprietary values
        psbt = deserialize('70736274ff0100550200000001ab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40100000000feffffff018e240000000000001976a9146f4620b553fa095e721b9ee0efe9fa039cca459788ac0000000015fc0a676c6f62616c5f706678016d756c7469706c790563686965660001012000e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787010416001485d13537f2e265405a34dbafa9e3dda01fb823080ffc06696e5f706678fde80377686174056672616d650afc00fe40420f0061736b077361746f7368690012fc076f75745f706678feffffff01636f726e05746967657217fc076f75745f706678ffffffffffffffffff707570707905647269766500')
        self.assertEqual(len(psbt.inputs), 1)
        self.assertEqual(len(psbt.outputs), 1)
        self.assertEqual(len(psbt.proprietary_fields), 1)
        self.assertEqual(len(psbt.proprietary_fields[b'global_pfx']), 1)
        self.assertEqual(len(psbt.inputs[0].proprietary_fields), 2)
        self.assertEqual(len(psbt.inputs[0].proprietary_fields[b'in_pfx']), 1)
        self.assertEqual(len(psbt.inputs[0].proprietary_fields[b'']), 1)
        self.assertEqual(len(psbt.outputs[0].proprietary_fields), 1)
        self.assertEqual(len(psbt.outputs[0].proprietary_fields[b'out_pfx']), 2)
        self.assertEqual(psbt.proprietary_fields[b'global_pfx'][0].subtype, 1)
        self.assertEqual(psbt.proprietary_fields[b'global_pfx'][0].key_data, b'multiply')
        self.assertEqual(psbt.proprietary_fields[b'global_pfx'][0].value, b'chief')
        self.assertEqual(psbt.inputs[0].proprietary_fields[b'in_pfx'][0].subtype, 1000)
        self.assertEqual(psbt.inputs[0].proprietary_fields[b'in_pfx'][0].key_data, b'what')
        self.assertEqual(psbt.inputs[0].proprietary_fields[b'in_pfx'][0].value, b'frame')
        self.assertEqual(psbt.inputs[0].proprietary_fields[b''][0].subtype, 1000000)
        self.assertEqual(psbt.inputs[0].proprietary_fields[b''][0].key_data, b'ask')
        self.assertEqual(psbt.inputs[0].proprietary_fields[b''][0].value, b'satoshi')
        self.assertEqual(psbt.outputs[0].proprietary_fields[b'out_pfx'][0].subtype, 0x2000000-1)
        self.assertEqual(psbt.outputs[0].proprietary_fields[b'out_pfx'][0].key_data, b'corn')
        self.assertEqual(psbt.outputs[0].proprietary_fields[b'out_pfx'][0].value, b'tiger')
        self.assertEqual(psbt.outputs[0].proprietary_fields[b'out_pfx'][1].subtype, 0xFFFFFFFFFFFFFFFF)
        self.assertEqual(psbt.outputs[0].proprietary_fields[b'out_pfx'][1].key_data, b'puppy')
        self.assertEqual(psbt.outputs[0].proprietary_fields[b'out_pfx'][1].value, b'drive')

    def test_signer_checks(self) -> None:
        def T(hex_data: str, expected_regex: str) -> None:
            data = x(hex_data)

            with self.assertRaises(ValueError):
                # should fail without relaxed_sanity_checks
                psbt = PartiallySignedTransaction.deserialize(data)

            psbt = PartiallySignedTransaction.deserialize(
                data, relaxed_sanity_checks=True)

            with self.assertRaisesRegex(ValueError, expected_regex):
                psbt.sign(KeyStore())

        T('70736274ff0100a00200000002ab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40000000000feffffffab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40100000000feffffff02603bea0b000000001976a914768a40bbd740cbe81d988e71de2a4d5c71396b1d88ac8e240000000000001976a9146f4620b553fa095e721b9ee0efe9fa039cca459788ac0000000000010122d3dff505000000001976a914d48ed3110b94014cb114bd32d6f4d066dc74256b88ac0001012000e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787010416001485d13537f2e265405a34dbafa9e3dda01fb8230800220202ead596687ca806043edc3de116cdf29d5e9257c196cd055cf698c8d02bf24e9910b4a6ba670000008000000080020000800022020394f62be9df19952c5587768aeb7698061ad2c4a25c894f47d8c162b4d7213d0510b4a6ba6700000080010000800200008000',
          'input at index 0 specified as witness UTXO, but has non-witness scriptPubKey')

        T('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',
          'redeem script for input at index 0 does not match scriptPubKey in UTXO')

        T('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',
          'redeem script for p2sh-wrapped segwit input at index 1 does not match the scriptPubKey')

        T('70736274ff01009a020000000258e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd750000000000ffffffff838d0427d0ec650a68aa46bb0b098aea4422c071b2ca78352a077959d07cea1d0100000000ffffffff0270aaf00800000000160014d85c2b71d0060b09c9886aeb815e50991dda124d00e1f5050000000016001400aea9a2e5f0f876a588df5546e8742d1d87008f00000000000100bb0200000001aad73931018bd25f84ae400b68848be09db706eac2ac18298babee71ab656f8b0000000048473044022058f6fc7c6a33e1b31548d481c826c015bd30135aad42cd67790dab66d2ad243b02204a1ced2604c6735b6393e5b41691dd78b00f0c5942fb9f751856faa938157dba01feffffff0280f0fa020000000017a9140fb9463421696b82c833af241c78c17ddbde493487d0f20a270100000017a91429ca74f8a08f81999428185c97b5d852e4063f618765000000220202dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d7483045022100f61038b308dc1da865a34852746f015772934208c6d24454393cd99bdf2217770220056e675a675a6d0a02b85b14e5e29074d8a25a9b5760bea2816f661910a006ea01010304010000000104475221029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f2102dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d752ae2206029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f10d90c6a4f000000800000008000000080220602dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d710d90c6a4f0000008000000080010000800001012000c2eb0b0000000017a914b7f5faf40e3d40a5a459b1db3535f2b72fa921e8872202023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e73473044022065f45ba5998b59a27ffe1a7bed016af1f1f90d54b3aa8f7450aa5f56a25103bd02207f724703ad1edb96680b284b56d4ffcb88f7fb759eabbe08aa30f29b851383d2010103040100000001042200208c2353173743b595dfb4a07b72ba8e42e3797da74e87fe7d9d7497e3b2028903010547522103089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc21023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7352ad2206023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7310d90c6a4f000000800000008003000080220603089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc10d90c6a4f00000080000000800200008000220203a9a4c37f5996d3aa25dbac6b570af0650394492942460b354753ed9eeca5877110d90c6a4f000000800000008004000080002202027f6399757d2eff55a136ad02c684b1838b6556e5f1b6b34282a94b6b5005109610d90c6a4f00000080000000800500008000',
          'witness script for p2sh-wrapped segwit p2wpkh input at index 1 does not match the redeem script')

    def test_convert_to_witness_utxo(self) -> None:
        # PSBT with inputs[0].utxo given as CTransaction, but it has
        # final_script_witness set
        psbt_data = x('70736274ff0100530200000001dd0f87b0f831f6a0ab1dff6237ecaf75ddc92732dac0a13bfc21af64f251aec40100000000ffffffff01b0bae4000000000017a9143e671da50a3a36d43bb3853f29e207fb48d44c738700000000000100f9020000000001017d65d832d5b1851b80b715160854baac04e21ad1aeefce921fb39b916f5ad6590100000017160014239bb89dbd96d490b58cf18efcf02958f3055b1dffffffff02c0e1e400000000001976a914c4b614bcb6ea61aaf96d51729c34d430b5dc820e88aca0780f050000000017a9143e671da50a3a36d43bb3853f29e207fb48d44c7387024730440220327afb49285b8b6aa1dfa8827bfecd5e8545d9b6e41a45dcc84e3b3439c23c71022018abb15599eadad23a197490b242873a9be93b9b208e91e62b5349e1b47e6a290121037abd691a39c0c0f1a7d71a66f13cb6c38a98d2a7aa2a73c3a0b7509d197839980000000001086c02483045022100a294dd2256fdea194a562fa52323ce0c0b14becf114340f477536c3f21002cda022016e471407800f93ecc90c9f46c9f738dd26bd9c4871f4bd38eb314bd1bb77bf4012103ffc7c28a070af755a0ae991cf86b8d29741f47941ad83df69f1c8a62993a8f890000')

        psbt = PartiallySignedTransaction.deserialize(psbt_data,
                                                      relaxed_sanity_checks=True)

        self.assertIsInstance(psbt.inputs[0].utxo, CTransaction)
        self.assertIsInstance(psbt.inputs[0].witness_utxo, CTxOut)
        self.assertEqual(psbt.inputs[0].non_witness_utxo, None)

        psbt = PartiallySignedTransaction.deserialize(psbt_data,
                                                      allow_convert_to_witness_utxo=True,
                                                      relaxed_sanity_checks=True)
        # check that non-witness utxo was converted to witness utxo
        self.assertIsInstance(psbt.inputs[0].utxo, CTxOut)
        self.assertEqual(psbt.inputs[0].utxo, psbt.inputs[0].witness_utxo)

        # but it is still not strictly valid, because it must supply
        # final_script_sig, too, if it is a p2sh-wrapped segwit input
        with self.assertRaisesRegex(ValueError, 'final_script_sig is not present for p2sh-wrapped segwit input'):
            PartiallySignedTransaction.deserialize(psbt_data, allow_convert_to_witness_utxo=True)

    def test_flow(self) -> None:
        # Create

        tx = CTransaction(
            vin=[
                CTxIn(COutPoint(lx('75ddabb27b8845f5247975c8a5ba7c6f336c4570708ebe230caf6db5217ae858'),
                                0)),
                CTxIn(COutPoint(lx('1dea7cd05979072a3578cab271c02244ea8a090bbb46aa680a65ecd027048d83'),
                                1))
            ],
            vout=[
                CTxOut(coins_to_satoshi(1.49990000),
                       CScript(x('0014d85c2b71d0060b09c9886aeb815e50991dda124d'))),
                CTxOut(coins_to_satoshi(1.00000000),
                       CScript(x('001400aea9a2e5f0f876a588df5546e8742d1d87008f')))
            ]
        )
        psbt = PartiallySignedTransaction(unsigned_tx=tx)
        self.assertEqual(b2x(psbt.serialize()),
                         '70736274ff01009a020000000258e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd750000000000ffffffff838d0427d0ec650a68aa46bb0b098aea4422c071b2ca78352a077959d07cea1d0100000000ffffffff0270aaf00800000000160014d85c2b71d0060b09c9886aeb815e50991dda124d00e1f5050000000016001400aea9a2e5f0f876a588df5546e8742d1d87008f000000000000000000')

        # Update
        with ChainParams('bitcoin/testnet'):
            xpriv = CCoinExtKey('tprv8ZgxMBicQKsPd9TeAdPADNnSyH9SSUUbTVeFszDE23Ki6TBB5nCefAdHkK8Fm3qMQR6sHwA56zqRmKmxnHk37JkiFzvncDqoKmPWubu7hDF')
            psbt.inputs[0].redeem_script = CScript(x('5221029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f2102dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d752ae'))
            msig0 = parse_standard_multisig_redeem_script(
                psbt.inputs[0].redeem_script)
            psbt.inputs[1].redeem_script = CScript(x('00208c2353173743b595dfb4a07b72ba8e42e3797da74e87fe7d9d7497e3b2028903'))
            psbt.inputs[1].witness_script = CScript(x('522103089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc21023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7352ae'))
            msig1 = parse_standard_multisig_redeem_script(
                psbt.inputs[1].witness_script)
            prev_tx0 = CTransaction.deserialize(x('0200000000010158e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd7501000000171600145f275f436b09a8cc9a2eb2a2f528485c68a56323feffffff02d8231f1b0100000017a914aed962d6654f9a2b36608eb9d64d2b260db4f1118700c2eb0b0000000017a914b7f5faf40e3d40a5a459b1db3535f2b72fa921e88702483045022100a22edcc6e5bc511af4cc4ae0de0fcd75c7e04d8c1c3a8aa9d820ed4b967384ec02200642963597b9b1bc22c75e9f3e117284a962188bf5e8a74c895089046a20ad770121035509a48eb623e10aace8bfd0212fdb8a8e5af3c94b0b133b95e114cab89e4f7965000000'))
            prev_tx1 = CTransaction.deserialize(x('0200000001aad73931018bd25f84ae400b68848be09db706eac2ac18298babee71ab656f8b0000000048473044022058f6fc7c6a33e1b31548d481c826c015bd30135aad42cd67790dab66d2ad243b02204a1ced2604c6735b6393e5b41691dd78b00f0c5942fb9f751856faa938157dba01feffffff0280f0fa020000000017a9140fb9463421696b82c833af241c78c17ddbde493487d0f20a270100000017a91429ca74f8a08f81999428185c97b5d852e4063f618765000000'))
            psbt.inputs[0].set_utxo(prev_tx1, psbt.unsigned_tx)
            psbt.inputs[1].set_utxo(prev_tx0.vout[tx.vin[1].prevout.n],
                                    psbt.unsigned_tx)

            self.assertEqual(msig0.pubkeys[0], xpriv.derive_path("m/0'/0'/0'").pub)
            self.assertEqual(msig0.pubkeys[1], xpriv.derive_path("m/0'/0'/1'").pub)
            psbt.inputs[0].derivation_map[msig0.pubkeys[0]] = \
                PSBT_KeyDerivationInfo(xpriv.fingerprint,
                                       BIP32Path("m/0'/0'/0'"))
            psbt.inputs[0].derivation_map[msig0.pubkeys[1]] = \
                PSBT_KeyDerivationInfo(xpriv.fingerprint,
                                       BIP32Path("m/0'/0'/1'"))

            self.assertEqual(msig1.pubkeys[0], xpriv.derive_path("m/0'/0'/2'").pub)
            self.assertEqual(msig1.pubkeys[1], xpriv.derive_path("m/0'/0'/3'").pub)
            psbt.inputs[1].derivation_map[msig1.pubkeys[1]] = \
                PSBT_KeyDerivationInfo(xpriv.fingerprint,
                                       BIP32Path("m/0'/0'/3'"))
            psbt.inputs[1].derivation_map[msig1.pubkeys[0]] = \
                PSBT_KeyDerivationInfo(xpriv.fingerprint,
                                       BIP32Path("m/0'/0'/2'"))

            outpub0 = xpriv.derive_path("m/0'/0'/4'").pub
            outpub1 = xpriv.derive_path("m/0'/0'/5'").pub
            self.assertEqual(psbt.unsigned_tx.vout[0].scriptPubKey,
                             standard_witness_v0_scriptpubkey(outpub0.key_id))
            self.assertEqual(psbt.unsigned_tx.vout[1].scriptPubKey,
                             standard_witness_v0_scriptpubkey(outpub1.key_id))
            psbt.outputs[0].derivation_map[outpub0] = \
                PSBT_KeyDerivationInfo(xpriv.fingerprint,
                                       BIP32Path("m/0'/0'/4'"))
            psbt.outputs[1].derivation_map[outpub1] = \
                PSBT_KeyDerivationInfo(xpriv.fingerprint,
                                       BIP32Path("m/0'/0'/5'"))

            self.assertEqual(b2x(psbt.serialize()),
                             '70736274ff01009a020000000258e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd750000000000ffffffff838d0427d0ec650a68aa46bb0b098aea4422c071b2ca78352a077959d07cea1d0100000000ffffffff0270aaf00800000000160014d85c2b71d0060b09c9886aeb815e50991dda124d00e1f5050000000016001400aea9a2e5f0f876a588df5546e8742d1d87008f00000000000100bb0200000001aad73931018bd25f84ae400b68848be09db706eac2ac18298babee71ab656f8b0000000048473044022058f6fc7c6a33e1b31548d481c826c015bd30135aad42cd67790dab66d2ad243b02204a1ced2604c6735b6393e5b41691dd78b00f0c5942fb9f751856faa938157dba01feffffff0280f0fa020000000017a9140fb9463421696b82c833af241c78c17ddbde493487d0f20a270100000017a91429ca74f8a08f81999428185c97b5d852e4063f6187650000000104475221029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f2102dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d752ae2206029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f10d90c6a4f000000800000008000000080220602dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d710d90c6a4f0000008000000080010000800001012000c2eb0b0000000017a914b7f5faf40e3d40a5a459b1db3535f2b72fa921e88701042200208c2353173743b595dfb4a07b72ba8e42e3797da74e87fe7d9d7497e3b2028903010547522103089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc21023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7352ae2206023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7310d90c6a4f000000800000008003000080220603089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc10d90c6a4f00000080000000800200008000220203a9a4c37f5996d3aa25dbac6b570af0650394492942460b354753ed9eeca5877110d90c6a4f000000800000008004000080002202027f6399757d2eff55a136ad02c684b1838b6556e5f1b6b34282a94b6b5005109610d90c6a4f00000080000000800500008000')

            # Updater which adds SIGHASH_ALL
            for inp in psbt.inputs:
                inp.sighash_type = SIGHASH_ALL

            self.assertEqual(b2x(psbt.serialize()),
                             '70736274ff01009a020000000258e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd750000000000ffffffff838d0427d0ec650a68aa46bb0b098aea4422c071b2ca78352a077959d07cea1d0100000000ffffffff0270aaf00800000000160014d85c2b71d0060b09c9886aeb815e50991dda124d00e1f5050000000016001400aea9a2e5f0f876a588df5546e8742d1d87008f00000000000100bb0200000001aad73931018bd25f84ae400b68848be09db706eac2ac18298babee71ab656f8b0000000048473044022058f6fc7c6a33e1b31548d481c826c015bd30135aad42cd67790dab66d2ad243b02204a1ced2604c6735b6393e5b41691dd78b00f0c5942fb9f751856faa938157dba01feffffff0280f0fa020000000017a9140fb9463421696b82c833af241c78c17ddbde493487d0f20a270100000017a91429ca74f8a08f81999428185c97b5d852e4063f618765000000010304010000000104475221029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f2102dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d752ae2206029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f10d90c6a4f000000800000008000000080220602dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d710d90c6a4f0000008000000080010000800001012000c2eb0b0000000017a914b7f5faf40e3d40a5a459b1db3535f2b72fa921e8870103040100000001042200208c2353173743b595dfb4a07b72ba8e42e3797da74e87fe7d9d7497e3b2028903010547522103089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc21023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7352ae2206023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7310d90c6a4f000000800000008003000080220603089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc10d90c6a4f00000080000000800200008000220203a9a4c37f5996d3aa25dbac6b570af0650394492942460b354753ed9eeca5877110d90c6a4f000000800000008004000080002202027f6399757d2eff55a136ad02c684b1838b6556e5f1b6b34282a94b6b5005109610d90c6a4f00000080000000800500008000')

            pclone1 = psbt.clone()
            pclone2 = psbt.clone()
            pclone3 = psbt.clone()

            # check signer with keystore that has only these non-extended privs
            kstore = KeyStore(xpriv.derive_path("m/0'/0'/0'").priv,
                              xpriv.derive_path("m/0'/0'/2'").priv)

            pclone1.sign(kstore, finalize=False)

            self.assertEqual(b2x(pclone1.serialize()),
                             '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')

            # check signer with keystore that has only these non-extended privs
            kstore = KeyStore(xpriv.derive_path("m/0'/0'/1'").priv,
                              xpriv.derive_path("m/0'/0'/3'").priv)

            pclone2.sign(kstore, finalize=False)

            self.assertEqual(b2x(pclone2.serialize()),
                             '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')

            combined_psbt = pclone1.combine(pclone2)
            self.assertEqual(b2x(combined_psbt.serialize()),
                             '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')

            # check signer with keystore that has xpriv, that should sign
            # all the inputs at once.
            kstore = KeyStore(xpriv, require_path_templates=False)
            pclone3.sign(kstore, finalize=False)
            self.assertEqual(b2x(pclone3.serialize()),
                             b2x(combined_psbt.serialize()))

            # Because all inputs should be already signed,
            # we can pass empty keystore.
            # We do not supply finalize=False, so sign() by default finalizes.
            combined_psbt.sign(KeyStore())

            # Check finalized PSBT
            self.assertEqual(b2x(combined_psbt.serialize()),
                             '70736274ff01009a020000000258e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd750000000000ffffffff838d0427d0ec650a68aa46bb0b098aea4422c071b2ca78352a077959d07cea1d0100000000ffffffff0270aaf00800000000160014d85c2b71d0060b09c9886aeb815e50991dda124d00e1f5050000000016001400aea9a2e5f0f876a588df5546e8742d1d87008f00000000000100bb0200000001aad73931018bd25f84ae400b68848be09db706eac2ac18298babee71ab656f8b0000000048473044022058f6fc7c6a33e1b31548d481c826c015bd30135aad42cd67790dab66d2ad243b02204a1ced2604c6735b6393e5b41691dd78b00f0c5942fb9f751856faa938157dba01feffffff0280f0fa020000000017a9140fb9463421696b82c833af241c78c17ddbde493487d0f20a270100000017a91429ca74f8a08f81999428185c97b5d852e4063f6187650000000107da00473044022074018ad4180097b873323c0015720b3684cc8123891048e7dbcd9b55ad679c99022073d369b740e3eb53dcefa33823c8070514ca55a7dd9544f157c167913261118c01483045022100f61038b308dc1da865a34852746f015772934208c6d24454393cd99bdf2217770220056e675a675a6d0a02b85b14e5e29074d8a25a9b5760bea2816f661910a006ea01475221029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f2102dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d752ae0001012000c2eb0b0000000017a914b7f5faf40e3d40a5a459b1db3535f2b72fa921e8870107232200208c2353173743b595dfb4a07b72ba8e42e3797da74e87fe7d9d7497e3b20289030108da0400473044022062eb7a556107a7c73f45ac4ab5a1dddf6f7075fb1275969a7f383efff784bcb202200c05dbb7470dbf2f08557dd356c7325c1ed30913e996cd3840945db12228da5f01473044022065f45ba5998b59a27ffe1a7bed016af1f1f90d54b3aa8f7450aa5f56a25103bd02207f724703ad1edb96680b284b56d4ffcb88f7fb759eabbe08aa30f29b851383d20147522103089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc21023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7352ae00220203a9a4c37f5996d3aa25dbac6b570af0650394492942460b354753ed9eeca5877110d90c6a4f000000800000008004000080002202027f6399757d2eff55a136ad02c684b1838b6556e5f1b6b34282a94b6b5005109610d90c6a4f00000080000000800500008000')

            # Check extracted CTransaction
            self.assertEqual(b2x(combined_psbt.extract_transaction().serialize()),
                             '0200000000010258e87a21b56daf0c23be8e7070456c336f7cbaa5c8757924f545887bb2abdd7500000000da00473044022074018ad4180097b873323c0015720b3684cc8123891048e7dbcd9b55ad679c99022073d369b740e3eb53dcefa33823c8070514ca55a7dd9544f157c167913261118c01483045022100f61038b308dc1da865a34852746f015772934208c6d24454393cd99bdf2217770220056e675a675a6d0a02b85b14e5e29074d8a25a9b5760bea2816f661910a006ea01475221029583bf39ae0a609747ad199addd634fa6108559d6c5cd39b4c2183f1ab96e07f2102dab61ff49a14db6a7d02b0cd1fbb78fc4b18312b5b4e54dae4dba2fbfef536d752aeffffffff838d0427d0ec650a68aa46bb0b098aea4422c071b2ca78352a077959d07cea1d01000000232200208c2353173743b595dfb4a07b72ba8e42e3797da74e87fe7d9d7497e3b2028903ffffffff0270aaf00800000000160014d85c2b71d0060b09c9886aeb815e50991dda124d00e1f5050000000016001400aea9a2e5f0f876a588df5546e8742d1d87008f000400473044022062eb7a556107a7c73f45ac4ab5a1dddf6f7075fb1275969a7f383efff784bcb202200c05dbb7470dbf2f08557dd356c7325c1ed30913e996cd3840945db12228da5f01473044022065f45ba5998b59a27ffe1a7bed016af1f1f90d54b3aa8f7450aa5f56a25103bd02207f724703ad1edb96680b284b56d4ffcb88f7fb759eabbe08aa30f29b851383d20147522103089dc10c7ac6db54f91329af617333db388cead0c231f723379d1b99030b02dc21023add904f3d6dcf59ddb906b0dee23529b7ffb9ed50e5e86151926860221f0e7352ae00000000')

        # Check finalization and extraction on another PSBT
        psbt = PartiallySignedTransaction.from_base64('cHNidP8BAFMCAAAAAd0Ph7D4Mfagqx3/Yjfsr3XdyScy2sChO/whr2TyUa7EAAAAAAD/////AbC65AAAAAAAF6kUPmcdpQo6NtQ7s4U/KeIH+0jUTHOHAAAAAAABAPkCAAAAAAEBfWXYMtWxhRuAtxUWCFS6rATiGtGu786SH7ObkW9a1lkBAAAAFxYAFCObuJ29ltSQtYzxjvzwKVjzBVsd/////wLA4eQAAAAAABl2qRTEthS8tuphqvltUXKcNNQwtdyCDoisoHgPBQAAAAAXqRQ+Zx2lCjo21DuzhT8p4gf7SNRMc4cCRzBEAiAyevtJKFuLaqHfqIJ7/s1ehUXZtuQaRdzITjs0OcI8cQIgGKuxVZnq2tI6GXSQskKHOpvpO5sgjpHmK1NJ4bR+aikBIQN6vWkaOcDA8afXGmbxPLbDipjSp6oqc8Ogt1CdGXg5mAAAAAAiAgP/x8KKBwr3VaCumRz4a40pdB9HlBrYPfafHIpimTqPiUgwRQIhAKKU3SJW/eoZSlYvpSMjzgwLFL7PEUNA9HdTbD8hACzaAiAW5HFAeAD5PsyQyfRsn3ON0mvZxIcfS9OOsxS9G7d79AEBAwQBAAAAIgYD/8fCigcK91Wgrpkc+GuNKXQfR5Qa2D32nxyKYpk6j4kYppD2ESwAAIAAAACAAQAAgAAAAAAAAAAAAAA=')
        psbt.sign(KeyStore())  # finalize=True by default
        self.assertEqual(b2x(psbt.extract_transaction().serialize()), '0200000001dd0f87b0f831f6a0ab1dff6237ecaf75ddc92732dac0a13bfc21af64f251aec4000000006b483045022100a294dd2256fdea194a562fa52323ce0c0b14becf114340f477536c3f21002cda022016e471407800f93ecc90c9f46c9f738dd26bd9c4871f4bd38eb314bd1bb77bf4012103ffc7c28a070af755a0ae991cf86b8d29741f47941ad83df69f1c8a62993a8f89ffffffff01b0bae4000000000017a9143e671da50a3a36d43bb3853f29e207fb48d44c738700000000')

    def test_unknown_proprietary_fields_merge(self) -> None:
        psbt1 = PartiallySignedTransaction.deserialize(x('70736274ff01003f0200000001ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000000ffffffff010000000000000000036a0100000000000a0f0102030405060708090f0102030405060708090a0b0c0d0e0f000a0f0102030405060708090f0102030405060708090a0b0c0d0e0f000a0f0102030405060708090f0102030405060708090a0b0c0d0e0f00'))
        psbt2 = PartiallySignedTransaction.deserialize(x('70736274ff01003f0200000001ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000000ffffffff010000000000000000036a0100000000000a0f0102030405060708100f0102030405060708090a0b0c0d0e0f000a0f0102030405060708100f0102030405060708090a0b0c0d0e0f000a0f0102030405060708100f0102030405060708090a0b0c0d0e0f00'))
        psbt3 = psbt1.clone()
        psbt_p = psbt1.clone()
        psbt4 = psbt1.clone()
        psbt3.merge(psbt1)
        self.assertEqual(b2x(psbt3.serialize()), b2x(psbt1.serialize()))
        psbt1.merge(psbt2)
        psbt4.merge(psbt2, allow_blob_duplicates=True)
        self.assertEqual(b2x(psbt4.serialize()),
                         '70736274ff01003f0200000001ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000000ffffffff010000000000000000036a0100000000000a0f0102030405060708090f0102030405060708090a0b0c0d0e0f0a0f0102030405060708100f0102030405060708090a0b0c0d0e0f000a0f0102030405060708090f0102030405060708090a0b0c0d0e0f0a0f0102030405060708100f0102030405060708090a0b0c0d0e0f000a0f0102030405060708090f0102030405060708090a0b0c0d0e0f0a0f0102030405060708100f0102030405060708090a0b0c0d0e0f00')
        prefix = b'test'
        psbt_p.proprietary_fields[prefix] = [
            PSBT_ProprietaryTypeData(subtype=ud.key_type,
                                     key_data=ud.key_data,
                                     value=ud.value)
            for ud in psbt_p.unknown_fields
        ]
        for inp in psbt_p.inputs:
            inp.proprietary_fields[prefix] = [
                PSBT_ProprietaryTypeData(subtype=ud.key_type,
                                         key_data=ud.key_data,
                                         value=ud.value)
                for ud in inp.unknown_fields
            ]
        for outp in psbt_p.outputs:
            outp.proprietary_fields[prefix] = [
                PSBT_ProprietaryTypeData(subtype=ud.key_type,
                                         key_data=ud.key_data,
                                         value=ud.value)
                for ud in outp.unknown_fields
            ]
        psbt_pclone = psbt_p.clone()
        psbt_p.merge(psbt_pclone)
        self.assertEqual(b2x(psbt_p.serialize()), b2x(psbt_pclone.serialize()))

        psbt_p.merge(psbt_pclone, allow_blob_duplicates=True)
        self.assertEqual(len(psbt_p.proprietary_fields[prefix]),
                         len(psbt_pclone.proprietary_fields[prefix])*2)
        self.assertEqual(psbt_p.proprietary_fields[prefix][:len(psbt_pclone.proprietary_fields[prefix])],
                         psbt_pclone.proprietary_fields[prefix])
        self.assertEqual(psbt_p.proprietary_fields[prefix][len(psbt_pclone.proprietary_fields[prefix]):],
                         psbt_pclone.proprietary_fields[prefix])
        for index, inp in enumerate(psbt_p.inputs):
            inp_clone = psbt_pclone.inputs[index]
            self.assertEqual(len(inp.proprietary_fields[prefix]),
                             len(inp_clone.proprietary_fields[prefix])*2)
            self.assertEqual(inp.proprietary_fields[prefix][:len(inp_clone.proprietary_fields[prefix])],
                             inp_clone.proprietary_fields[prefix])
            self.assertEqual(inp.proprietary_fields[prefix][len(inp_clone.proprietary_fields[prefix]):],
                             inp_clone.proprietary_fields[prefix])
        for index, outp in enumerate(psbt_p.outputs):
            outp_clone = psbt_pclone.outputs[index]
            self.assertEqual(len(outp.proprietary_fields[prefix]),
                             len(outp_clone.proprietary_fields[prefix])*2)
            self.assertEqual(outp.proprietary_fields[prefix][:len(outp_clone.proprietary_fields[prefix])],
                             outp_clone.proprietary_fields[prefix])
            self.assertEqual(outp.proprietary_fields[prefix][len(outp_clone.proprietary_fields[prefix]):],
                             outp_clone.proprietary_fields[prefix])

    def test_clone(self) -> None:
        # Take some PSBT, fill all the fields that we know exist,
        # and check that clone() creates the same PSBT
        psbt = PartiallySignedTransaction.deserialize(x('70736274ff0100550200000001ab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40100000000feffffff018e240000000000001976a9146f4620b553fa095e721b9ee0efe9fa039cca459788ac0000000015fc0a676c6f62616c5f706678016d756c7469706c79056368696566046f616263036465660001012000e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787010416001485d13537f2e265405a34dbafa9e3dda01fb823080ffc06696e5f706678fde80377686174056672616d650afc00fe40420f0061736b077361746f736869046f616263036465660012fc076f75745f706678feffffff01636f726e05746967657213fc076f75745f706678fe000000027075707079056472697665046f6162630364656600'))
        pub = CPubKey(x('039eff1f547a1d5f92dfa2ba7af6ac971a4bd03ba4a734b03156a256b8ad3a1ef9'))
        psbt.inputs[0].witness_script = CScript([1, 2, 3])
        psbt.inputs[0].redeem_script = psbt.inputs[0].witness_script.to_p2wsh_scriptPubKey()
        assert isinstance(psbt.inputs[0].utxo, CTxOut)
        psbt.set_utxo(psbt.inputs[0].utxo.to_mutable(), index=0)
        assert isinstance(psbt.inputs[0].utxo, CMutableTxOut)
        psbt.inputs[0].utxo.scriptPubKey = psbt.inputs[0].redeem_script.to_p2sh_scriptPubKey()
        psbt.inputs[0].partial_sigs[pub] = b'123'
        psbt.inputs[0].sighash_type = SIGHASH_ALL
        psbt.inputs[0].derivation_map[pub] = \
            PSBT_KeyDerivationInfo(x('d90c6a4f'), BIP32Path("m/0'/0'/5'"))
        psbt.inputs[0].proof_of_reserves_commitment = b'123'

        psbt.unsigned_tx = psbt.unsigned_tx.to_mutable()
        psbt.outputs[0].witness_script = CScript([7, 8, 9])
        psbt.outputs[0].redeem_script = psbt.outputs[0].witness_script.to_p2wsh_scriptPubKey()
        psbt.unsigned_tx.vout[0].scriptPubKey = psbt.outputs[0].redeem_script.to_p2sh_scriptPubKey()
        psbt.outputs[0].derivation_map[pub] = \
            PSBT_KeyDerivationInfo(x('ffffeeee'), BIP32Path("m/1'/2'/3'"))

        with ChainParams('bitcoin/testnet'):
            xpriv = CCoinExtKey('tprv8ZgxMBicQKsPd9TeAdPADNnSyH9SSUUbTVeFszDE23Ki6TBB5nCefAdHkK8Fm3qMQR6sHwA56zqRmKmxnHk37JkiFzvncDqoKmPWubu7hDF')

        psbt.xpubs[xpriv.neuter()] = \
            PSBT_KeyDerivationInfo(x('ddddcccc'), BIP32Path("m/5'/4'/8'"))

        self.assertEqual(b2x(psbt.serialize()), b2x(psbt.clone().serialize()))

        # Cannot check cloning of final fields along with non-final,
        # so need second check with final fields.
        psbt.inputs[0].final_script_sig = b'final_script_sig'
        psbt.inputs[0].final_script_witness = CScriptWitness([b'script', b'witness'])
        psbt.inputs[0]._clear_nonfinal_fields()
        self.assertEqual(b2x(psbt.serialize()), b2x(psbt.clone().serialize()))
        self.assertEqual(b2x(psbt.serialize()),
                         b2x(PartiallySignedTransaction.deserialize(psbt.serialize()).serialize()))

    def test_add_input_ouput(self) -> None:
        psbt = PartiallySignedTransaction.deserialize(x('70736274ff0100a00200000002ab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40000000000feffffffab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40100000000feffffff02603bea0b000000001976a914768a40bbd740cbe81d988e71de2a4d5c71396b1d88ac8e240000000000001976a9146f4620b553fa095e721b9ee0efe9fa039cca459788ac00000000000100df0200000001268171371edff285e937adeea4b37b78000c0566cbb3ad64641713ca42171bf6000000006a473044022070b2245123e6bf474d60c5b50c043d4c691a5d2435f09a34a7662a9dc251790a022001329ca9dacf280bdf30740ec0390422422c81cb45839457aeb76fc12edd95b3012102657d118d3357b8e0f4c2cd46db7b39f6d9c38d9a70abcb9b2de5dc8dbfe4ce31feffffff02d3dff505000000001976a914d0c59903c5bac2868760e90fd521a4665aa7652088ac00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787b32e13000001012000e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787010416001485d13537f2e265405a34dbafa9e3dda01fb8230800220202ead596687ca806043edc3de116cdf29d5e9257c196cd055cf698c8d02bf24e9910b4a6ba670000008000000080020000800022020394f62be9df19952c5587768aeb7698061ad2c4a25c894f47d8c162b4d7213d0510b4a6ba6700000080010000800200008000'))
        new_psbt = psbt.clone()
        new_psbt.inputs = []
        new_psbt.outputs = []
        new_psbt.unsigned_tx = new_psbt.unsigned_tx.to_mutable()
        new_psbt.unsigned_tx.vin = []
        new_psbt.unsigned_tx.vout = []
        new_psbt.unsigned_tx = new_psbt.unsigned_tx.to_immutable()

        for index, inp in enumerate(psbt.inputs):
            new_psbt.add_input(psbt.unsigned_tx.vin[index], inp)

        for index, outp in enumerate(psbt.outputs):
            new_psbt.add_output(psbt.unsigned_tx.vout[index], outp)

        self.assertEqual(b2x(new_psbt.serialize()), b2x(psbt.serialize()))

    def test_b64(self) -> None:
        data = x('70736274ff0100a00200000002ab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40000000000feffffffab0949a08c5af7c49b8212f417e2f15ab3f5c33dcf153821a8139f877a5b7be40100000000feffffff02603bea0b000000001976a914768a40bbd740cbe81d988e71de2a4d5c71396b1d88ac8e240000000000001976a9146f4620b553fa095e721b9ee0efe9fa039cca459788ac00000000000100df0200000001268171371edff285e937adeea4b37b78000c0566cbb3ad64641713ca42171bf6000000006a473044022070b2245123e6bf474d60c5b50c043d4c691a5d2435f09a34a7662a9dc251790a022001329ca9dacf280bdf30740ec0390422422c81cb45839457aeb76fc12edd95b3012102657d118d3357b8e0f4c2cd46db7b39f6d9c38d9a70abcb9b2de5dc8dbfe4ce31feffffff02d3dff505000000001976a914d0c59903c5bac2868760e90fd521a4665aa7652088ac00e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787b32e13000001012000e1f5050000000017a9143545e6e33b832c47050f24d3eeb93c9c03948bc787010416001485d13537f2e265405a34dbafa9e3dda01fb8230800220202ead596687ca806043edc3de116cdf29d5e9257c196cd055cf698c8d02bf24e9910b4a6ba670000008000000080020000800022020394f62be9df19952c5587768aeb7698061ad2c4a25c894f47d8c162b4d7213d0510b4a6ba6700000080010000800200008000')
        b64_data = base64.b64encode(data).decode('utf-8')
        psbt1 = PartiallySignedTransaction.deserialize(data)
        psbt2 = PartiallySignedTransaction.from_base64(b64_data)
        psbt3 = PartiallySignedTransaction.from_base64_or_binary(b64_data)
        psbt4 = PartiallySignedTransaction.from_base64_or_binary(data)
        with self.assertRaisesRegex(ValueError, 'magic bytes at the start do not match'):
            psbt4 = PartiallySignedTransaction.from_base64_or_binary('abcde')
        self.assertEqual(b2x(psbt3.serialize()), b2x(data))
        self.assertEqual(b2x(psbt4.serialize()), b2x(data))
        self.assertEqual(b2x(psbt2.serialize()), b2x(data))
        self.assertEqual(b2x(psbt1.serialize()), b2x(psbt2.serialize()))
        self.assertEqual(b64_data, psbt1.to_base64())
