#
# This code is auto-generated using the sklearn_wrapper_template.py_template template.
# Do not modify the auto-generated code(except automatic reformating by precommit hooks).
#
#
# Copyright (c) 2012-2022 Snowflake Computing Inc. All rights reserved.
#
import inspect
import os
import posixpath
from typing import Iterable, Optional, Union, List, Any, Dict, Callable, Set
from uuid import uuid4

import cloudpickle as cp
import pandas as pd
import numpy as np
import numpy
import xgboost
from sklearn.utils.metaestimators import available_if

from snowflake.ml.modeling.framework.base import BaseTransformer
from snowflake.ml.modeling.framework._utils import to_native_format
from snowflake.ml._internal import telemetry
from snowflake.ml._internal.utils.query_result_checker import SqlResultValidator
from snowflake.ml._internal.utils import pkg_version_utils, identifier
from snowflake.ml._internal.utils.temp_file_utils import cleanup_temp_files, get_temp_file_path
from snowflake.snowpark import DataFrame, Session
from snowflake.snowpark.functions import pandas_udf, sproc
from snowflake.snowpark.types import PandasSeries
from snowflake.snowpark._internal.type_utils import convert_sp_to_sf_type

from snowflake.ml.model.model_signature import (
    DataType,
    FeatureSpec,
    ModelSignature,
    _infer_signature,
    _rename_features,
)

_PROJECT = "ModelDevelopment"
# Derive subproject from module name by removing "sklearn"
# and converting module name from underscore to CamelCase
# e.g. sklearn.linear_model -> LinearModel.
_SUBPROJECT = "".join([s.capitalize() for s in "xgboost".replace("sklearn.", "").split("_")])


def _original_estimator_has_callable(attr : str) -> Callable[[Any], bool]:
    """ Checks that the original estimator has callable `attr`.

    Args:
        attr: Attribute to check for.

    Returns:
        A function which checks for the existance of callable `attr` on the given object.
    """
    def check(self) -> bool:
        """ Check for the existance of callable `attr` in self.

        Returns:
            True of the callable `attr` exists in self, False otherwise.
        """
        return callable(getattr(self._sklearn_object, attr, None))

    return check

def _gather_dependencies(obj: Any) -> Set[str]:
    """ Gethers dependencies from the SnowML Estimator and Transformer objects.

    Args:
        obj: Source object to collect dependencies from. Source object could of any type, example, lists, tuples, etc.

    Returns:
        A set of dependencies required to work with the object.
    """

    if isinstance(obj, list) or isinstance(obj, tuple):
        deps: Set[str] = set()
        for elem in obj:
            deps = deps | set(_gather_dependencies(elem))
        return deps
    elif isinstance(obj, BaseTransformer):
        return set(obj._get_dependencies())
    else:
        return set()

def _transform_snowml_obj_to_sklearn_obj(obj: Any) -> Any:
    """Converts SnowML Estimator and Transformer objects to equivalent SKLearn objects.

    Args:
        obj: Source object that needs to be converted. Source object could of any type, example, lists, tuples, etc.

    Returns:
        An equivalent object with SnowML estimators and transforms replaced with equivalent SKLearn objects.
    """

    if isinstance(obj, list):
        # Apply transform function to each element in the list
        return list(map(_transform_snowml_obj_to_sklearn_obj, obj))
    elif isinstance(obj, tuple):
        # Apply transform function to each element in the tuple
        return tuple(map(_transform_snowml_obj_to_sklearn_obj, obj))
    elif isinstance(obj, BaseTransformer):
        # Convert SnowML object to equivalent SKLearn object
        return to_native_format(obj)
    else:
        # Return all other objects as it is.
        return obj


def _validate_sklearn_args(args: Dict[str, Any], klass: type) -> Dict[str, Any]:
    """ Validate if all the keyword args are supported by current version of SKLearn/XGBoost object.

    Args:
        args: Dictionary of keyword args for the wrapper init method.
        klass: Underlying SKLearn/XGBoost class object.

    Raises:
        Raises an expception if a user specified arg is not supported by current version of sklearn/xgboost.
    """
    result = {}
    signature = inspect.signature(klass.__init__)
    for k, v in args.items():
        if k not in signature.parameters.keys():  # Arg is not supported.
            if (
                v[2]  # Arg doesn't have default value in the signature.
                or (
                    v[0] != v[1]  # Value is not same as default.
                    and not (isinstance(v[0], float) and np.isnan(v[0]) and np.isnan(v[1])))  # both are not NANs
            ):
                raise RuntimeError(f"Arg {k} is not supported by current version of SKLearn/XGBoost.")
        else:
            result[k] = v[0]
    return result


class XGBRFClassifier(BaseTransformer):
    r"""scikit-learn API for XGBoost random forest classification
    For more details on this class, see [xgboost.XGBRFClassifier]
    (https://xgboost.readthedocs.io/en/stable/python/python_api.html#xgboost.XGBRFClassifier)

    Parameters
    ----------

        n_estimators : int
            Number of trees in random forest to fit.

        max_depth :  Optional[int]
            Maximum tree depth for base learners.
        max_leaves :
            Maximum number of leaves; 0 indicates no limit.
        max_bin :
            If using histogram-based algorithm, maximum number of bins per feature
        grow_policy :
            Tree growing policy. 0: favor splitting at nodes closest to the node, i.e. grow
            depth-wise. 1: favor splitting at nodes with highest loss change.
        learning_rate : Optional[float]
            Boosting learning rate (xgb's "eta")
        verbosity : Optional[int]
            The degree of verbosity. Valid values are 0 (silent) - 3 (debug).
        objective : typing.Union[str, typing.Callable[[numpy.ndarray, numpy.ndarray], typing.Tuple[numpy.ndarray, numpy.ndarray]], NoneType]
            Specify the learning task and the corresponding learning objective or
            a custom objective function to be used (see note below).
        booster: Optional[str]
            Specify which booster to use: gbtree, gblinear or dart.
        tree_method: Optional[str]
            Specify which tree method to use.  Default to auto.  If this parameter is set to
            default, XGBoost will choose the most conservative option available.  It's
            recommended to study this option from the parameters document :doc:`tree method
            </treemethod>`
        n_jobs : Optional[int]
            Number of parallel threads used to run xgboost.  When used with other
            Scikit-Learn algorithms like grid search, you may choose which algorithm to
            parallelize and balance the threads.  Creating thread contention will
            significantly slow down both algorithms.
        gamma : Optional[float]
            (min_split_loss) Minimum loss reduction required to make a further partition on a
            leaf node of the tree.
        min_child_weight : Optional[float]
            Minimum sum of instance weight(hessian) needed in a child.
        max_delta_step : Optional[float]
            Maximum delta step we allow each tree's weight estimation to be.
        subsample : Optional[float]
            Subsample ratio of the training instance.
        sampling_method :
            Sampling method. Used only by `gpu_hist` tree method.
              - `uniform`: select random training instances uniformly.
              - `gradient_based` select random training instances with higher probability when
                the gradient and hessian are larger. (cf. CatBoost)
        colsample_bytree : Optional[float]
            Subsample ratio of columns when constructing each tree.
        colsample_bylevel : Optional[float]
            Subsample ratio of columns for each level.
        colsample_bynode : Optional[float]
            Subsample ratio of columns for each split.
        reg_alpha : Optional[float]
            L1 regularization term on weights (xgb's alpha).
        reg_lambda : Optional[float]
            L2 regularization term on weights (xgb's lambda).
        scale_pos_weight : Optional[float]
            Balancing of positive and negative weights.
        base_score : Optional[float]
            The initial prediction score of all instances, global bias.
        random_state : Optional[Union[numpy.random.RandomState, int]]
            Random number seed.

               Using gblinear booster with shotgun updater is nondeterministic as
               it uses Hogwild algorithm.

        missing : float, default np.nan
            Value in the data which needs to be present as a missing value.
        num_parallel_tree: Optional[int]
            Used for boosting random forest.
        monotone_constraints : Optional[Union[Dict[str, int], str]]
            Constraint of variable monotonicity.  See :doc:`tutorial </tutorials/monotonic>`
            for more information.
        interaction_constraints : Optional[Union[str, List[Tuple[str]]]]
            Constraints for interaction representing permitted interactions.  The
            constraints must be specified in the form of a nested list, e.g. ``[[0, 1], [2,
            3, 4]]``, where each inner list is a group of indices of features that are
            allowed to interact with each other.  See :doc:`tutorial
            </tutorials/feature_interaction_constraint>` for more information
        importance_type: Optional[str]
            The feature importance type for the feature_importances\_ property:

            * For tree model, it's either "gain", "weight", "cover", "total_gain" or
              "total_cover".
            * For linear model, only "weight" is defined and it's the normalized coefficients
              without bias.

        gpu_id : Optional[int]
            Device ordinal.
        validate_parameters : Optional[bool]
            Give warnings for unknown parameter.
        predictor : Optional[str]
            Force XGBoost to use specific predictor, available choices are [cpu_predictor,
            gpu_predictor].
        enable_categorical : bool

            Experimental support for categorical data.  When enabled, cudf/pandas.DataFrame
            should be used to specify categorical data type.  Also, JSON/UBJSON
            serialization format is required.

        feature_types : FeatureTypes

            Used for specifying feature types without constructing a dataframe. See
            :py:class:`DMatrix` for details.

        max_cat_to_onehot : Optional[int]

            A threshold for deciding whether XGBoost should use one-hot encoding based split
            for categorical data.  When number of categories is lesser than the threshold
            then one-hot encoding is chosen, otherwise the categories will be partitioned
            into children nodes. Also, `enable_categorical` needs to be set to have
            categorical feature support. See :doc:`Categorical Data
            </tutorials/categorical>` and :ref:`cat-param` for details.

        max_cat_threshold : Optional[int]

            Maximum number of categories considered for each split. Used only by
            partition-based splits for preventing over-fitting. Also, `enable_categorical`
            needs to be set to have categorical feature support. See :doc:`Categorical Data
            </tutorials/categorical>` and :ref:`cat-param` for details.

        eval_metric : Optional[Union[str, List[str], Callable]]

            Metric used for monitoring the training result and early stopping.  It can be a
            string or list of strings as names of predefined metric in XGBoost (See
            doc/parameter.rst), one of the metrics in :py:mod:`sklearn.metrics`, or any other
            user defined metric that looks like `sklearn.metrics`.

            If custom objective is also provided, then custom metric should implement the
            corresponding reverse link function.

            Unlike the `scoring` parameter commonly used in scikit-learn, when a callable
            object is provided, it's assumed to be a cost function and by default XGBoost will
            minimize the result during early stopping.

            For advanced usage on Early stopping like directly choosing to maximize instead of
            minimize, see :py:obj:`xgboost.callback.EarlyStopping`.

            See :doc:`Custom Objective and Evaluation Metric </tutorials/custom_metric_obj>`
            for more.

                 This parameter replaces `eval_metric` in :py:meth:`fit` method.  The old one
                 receives un-transformed prediction regardless of whether custom objective is
                 being used.

                from sklearn.datasets import load_diabetes
                from sklearn.metrics import mean_absolute_error
                X, y = load_diabetes(return_X_y=True)
                reg = xgb.XGBRegressor(
                    tree_method="hist",
                    eval_metric=mean_absolute_error,
                )
                reg.fit(X, y, eval_set=[(X, y)])

        early_stopping_rounds : Optional[int]

            Activates early stopping. Validation metric needs to improve at least once in
            every **early_stopping_rounds** round(s) to continue training.  Requires at least
            one item in **eval_set** in :py:meth:`fit`.

            The method returns the model from the last iteration (not the best one).  If
            there's more than one item in **eval_set**, the last entry will be used for early
            stopping.  If there's more than one metric in **eval_metric**, the last metric
            will be used for early stopping.

            If early stopping occurs, the model will have three additional fields:
            :py:attr:`best_score`, :py:attr:`best_iteration` and
            :py:attr:`best_ntree_limit`.

                This parameter replaces `early_stopping_rounds` in :py:meth:`fit` method.

        callbacks : Optional[List[TrainingCallback]]
            List of callback functions that are applied at end of each iteration.
            It is possible to use predefined callbacks by using
            :ref:`Callback API <callback_api>`.

               States in callback are not preserved during training, which means callback
               objects can not be reused for multiple training sessions without
               reinitialization or deepcopy.

                for params in parameters_grid:
                    # be sure to (re)initialize the callbacks before each run
                    callbacks = [xgb.callback.LearningRateScheduler(custom_rates)]
                    xgboost.train(params, Xy, callbacks=callbacks)

        kwargs : dict, optional
            Keyword arguments for XGBoost Booster object.  Full documentation of parameters
            can be found :doc:`here </parameter>`.
            Attempting to set a parameter via the constructor args and \*\*kwargs
            dict simultaneously will result in a TypeError.

                \*\*kwargs is unsupported by scikit-learn.  We do not guarantee
                that parameters passed via this argument will interact properly
                with scikit-learn.

                A custom objective function can be provided for the ``objective``
                parameter. In this case, it should have the signature
                ``objective(y_true, y_pred) -> grad, hess``:

                y_true: array_like of shape [n_samples]
                    The target values
                y_pred: array_like of shape [n_samples]
                    The predicted values

                grad: array_like of shape [n_samples]
                    The value of the gradient for each sample point.
                hess: array_like of shape [n_samples]
                    The value of the second derivative for each sample point

    input_cols : Optional[Union[str, List[str]]]
        A string or list of strings representing column names that contain features.
        If this parameter is not specified, all columns in the input DataFrame except
        the columns specified by label_cols and sample-weight_col parameters are
        considered input columns.

    label_cols : Optional[Union[str, List[str]]]
        A string or list of strings representing column names that contain labels.
        This is a required param for estimators, as there is no way to infer these
        columns. If this parameter is not specified, then object is fitted without
        labels(Like a transformer).

    output_cols: Optional[Union[str, List[str]]]
        A string or list of strings representing column names that will store the
        output of predict and transform operations. The length of output_cols mus
        match the expected number of output columns from the specific estimator or
        transformer class used.
        If this parameter is not specified, output column names are derived by
        adding an OUTPUT_ prefix to the label column names. These inferred output
        column names work for estimator's predict() method, but output_cols must
        be set explicitly for transformers.

    sample_weight_col: Optional[str]
        A string representing the column name containing the examples’ weights.
        This argument is only required when working with weighted datasets.

    drop_input_cols: Optional[bool], default=False
        If set, the response of predict(), transform() methods will not contain input columns.
    """

    def __init__(
        self,
        *,
        learning_rate=1.0,
        subsample=0.8,
        colsample_bynode=0.8,
        reg_lambda=1e-05,
        input_cols: Optional[Union[str, Iterable[str]]] = None,
        output_cols: Optional[Union[str, Iterable[str]]] = None,
        label_cols: Optional[Union[str, Iterable[str]]] = None,
        drop_input_cols: Optional[bool] = False,
        sample_weight_col: Optional[str] = None,
        **kwargs,
    ) -> None:
        super().__init__()
        deps: Set[str] = set([f'numpy=={np.__version__}', f'xgboost=={xgboost.__version__}', f'cloudpickle=={cp.__version__}'])
        
        self._deps = list(deps)
        
        init_args = {'learning_rate':(learning_rate, 1.0, False),
            'subsample':(subsample, 0.8, False),
            'colsample_bynode':(colsample_bynode, 0.8, False),
            'reg_lambda':(reg_lambda, 1e-05, False),}
        cleaned_up_init_args = _validate_sklearn_args(
            args=init_args,
            klass=xgboost.XGBRFClassifier
        )
        self._sklearn_object = xgboost.XGBRFClassifier(
            **cleaned_up_init_args,
            **kwargs,
        )
        self._model_signature_dict = None
        self.set_input_cols(input_cols)
        self.set_output_cols(output_cols)
        self.set_label_cols(label_cols)
        self.set_drop_input_cols(drop_input_cols)
        self.set_sample_weight_col(sample_weight_col)

    def _get_rand_id(self) -> str:
        """
        Generate random id to be used in sproc and stage names.

        Returns:
            Random id string usable in sproc, table, and stage names.
        """
        return str(uuid4()).replace("-", "_").upper()

    def _infer_input_output_cols(self, dataset: Union[DataFrame, pd.DataFrame]) -> None:
        """
        Infer `self.input_cols` and `self.output_cols` if they are not explicitly set.

        Args:
            dataset: Input dataset.
        """
        if not self.input_cols:
            cols = [
                c for c in dataset.columns 
                if c not in self.get_label_cols() and c != self.sample_weight_col
            ]
            self.set_input_cols(input_cols=cols)

        if not self.output_cols:
            cols = [identifier.concat_names(ids=['OUTPUT_', c]) for c in self.label_cols]
            self.set_output_cols(output_cols=cols)

    @telemetry.send_api_usage_telemetry(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    def fit(self, dataset: Union[DataFrame, pd.DataFrame]) -> "XGBRFClassifier":
        """Fit gradient boosting classifier
        For more details on this function, see [xgboost.XGBRFClassifier.fit]
        (https://xgboost.readthedocs.io/en/stable/python/python_api.html#xgboost.XGBRFClassifier.fit)


        Raises:
            TypeError: Supported dataset types: snowpark.DataFrame, pandas.DataFrame.

        Args:
            dataset: Union[snowflake.snowpark.DataFrame, pandas.DataFrame]
                Snowpark or Pandas DataFrame.

        Returns:
            self
        """
        self._infer_input_output_cols(dataset)
        if isinstance(dataset, pd.DataFrame):
            self._fit_pandas(dataset)
        elif isinstance(dataset, DataFrame):
            self._fit_snowpark(dataset)
        else:
            raise TypeError(
                f"Unexpected dataset type: {type(dataset)}."
                "Supported dataset types: snowpark.DataFrame, pandas.DataFrame."
            )
        self._is_fitted = True
        self._get_model_signatures(dataset)
        return self

    def _fit_snowpark(self, dataset: DataFrame) -> None:
        session = dataset._session
        # Validate that key package version in user workspace are supported in snowflake conda channel
        # If customer doesn't have package in conda channel, replace the ones have the closest versions
        self._deps = pkg_version_utils.get_valid_pkg_versions_supported_in_snowflake_conda_channel(
            pkg_versions=self._get_dependencies(), session=session, subproject=_SUBPROJECT)

        # Specify input columns so column pruing will be enforced
        selected_cols = (
            self.input_cols +
            self.label_cols +
            [self.sample_weight_col] if self.sample_weight_col is not None else []
        )
        if len(selected_cols) > 0:
            dataset = dataset.select(selected_cols)

        # Extract query that generated the datafrome. We will need to pass it to the fit procedure.
        query = str(dataset.queries["queries"][0])

        # Create a temp file and dump the transform to that file.
        local_transform_file_name = get_temp_file_path()
        with open(local_transform_file_name, mode="w+b") as local_transform_file:
            cp.dump(self._sklearn_object, local_transform_file)

        # Create temp stage to run fit.
        transform_stage_name = "SNOWML_TRANSFORM_{safe_id}".format(safe_id=self._get_rand_id())
        stage_creation_query = f"CREATE OR REPLACE TEMPORARY STAGE {transform_stage_name};"
        SqlResultValidator(
            session=session,
            query=stage_creation_query
        ).has_dimensions(
            expected_rows=1, expected_cols=1
        ).has_value_match(
            row_idx=0,
            col_idx=0,
            expected_value=f"Stage area {transform_stage_name} successfully created."
        ).validate()

        # Use posixpath to construct stage paths
        stage_transform_file_name = posixpath.join(transform_stage_name, os.path.basename(local_transform_file_name))
        stage_result_file_name = posixpath.join(transform_stage_name, os.path.basename(local_transform_file_name))
        local_result_file_name = get_temp_file_path()

        fit_sproc_name = "SNOWML_FIT_{safe_id}".format(safe_id=self._get_rand_id())
        statement_params = telemetry.get_function_usage_statement_params(
            project=_PROJECT,
            subproject=_SUBPROJECT,
            function_name=telemetry.get_statement_params_full_func_name(
                inspect.currentframe(), self.__class__.__name__
            ),
            api_calls=[sproc],
            custom_tags=dict([("autogen", True)]),
        )
        # Put locally serialized transform on stage.
        session.file.put(
            local_transform_file_name,
            stage_transform_file_name,
            auto_compress=False,
            overwrite=True,
            statement_params=statement_params
        )

        @sproc(
            is_permanent=False,
            name=fit_sproc_name,
            packages=["snowflake-snowpark-python"] + self._get_dependencies(),
            replace=True,
            session=session,
            statement_params=statement_params,
            anonymous=True
        )
        def fit_wrapper_sproc(
            session: Session,
            sql_query: str,
            stage_transform_file_name: str,
            stage_result_file_name: str,
            input_cols: List[str],
            label_cols: List[str],
            sample_weight_col: Optional[str],
            statement_params: Dict[str, str]
        ) -> str:
            import cloudpickle as cp
            import numpy as np
            import os
            import pandas
            import tempfile
            import inspect
            import xgboost

            # Execute snowpark query and obtain the results as pandas dataframe
            # NB: this implies that the result data must fit into memory.
            df = session.sql(sql_query).to_pandas(statement_params=statement_params)

            local_transform_file = tempfile.NamedTemporaryFile(delete=True)
            local_transform_file_name = local_transform_file.name
            local_transform_file.close()

            session.file.get(stage_transform_file_name, local_transform_file_name, statement_params=statement_params)

            local_transform_file_path = os.path.join(
                local_transform_file_name,
                os.listdir(local_transform_file_name)[0]
            )
            with open(local_transform_file_path, mode="r+b") as local_transform_file_obj:
                estimator = cp.load(local_transform_file_obj)

            argspec = inspect.getfullargspec(estimator.fit)
            args = {'X': df[input_cols]}
            if label_cols:
                label_arg_name = "Y" if "Y" in argspec.args else "y"
                args[label_arg_name] = df[label_cols].squeeze()

            if sample_weight_col is not None and "sample_weight" in argspec.args:
                args['sample_weight'] = df[sample_weight_col].squeeze()

            estimator.fit(**args)

            local_result_file = tempfile.NamedTemporaryFile(delete=True)
            local_result_file_name = local_result_file.name
            local_result_file.close()

            with open(local_result_file_name, mode="w+b") as local_result_file_obj:
                cp.dump(estimator, local_result_file_obj)

            session.file.put(
                local_result_file_name,
                stage_result_file_name,
                auto_compress = False,
                overwrite = True,
                statement_params=statement_params
            )

            # Note: you can add something like  + "|" + str(df) to the return string
            # to pass debug information to the caller.
            return str(os.path.basename(local_result_file_name))

        # Call fit sproc
        statement_params = telemetry.get_function_usage_statement_params(
            project=_PROJECT,
            subproject=_SUBPROJECT,
            function_name=telemetry.get_statement_params_full_func_name(
                inspect.currentframe(), self.__class__.__name__
            ),
            api_calls=[Session.call],
            custom_tags=dict([("autogen", True)]),
        )
        sproc_export_file_name = fit_wrapper_sproc(
            session,
            query,
            stage_transform_file_name,
            stage_result_file_name,
            identifier.get_unescaped_names(self.input_cols),
            identifier.get_unescaped_names(self.label_cols),
            identifier.get_unescaped_names(self.sample_weight_col),
            statement_params,
        )

        if "|" in sproc_export_file_name:
            fields = sproc_export_file_name.strip().split("|")
            sproc_export_file_name = fields[0]
            if len(fields) > 1:
                print("\n".join(fields[1:]))

        session.file.get(
            posixpath.join(stage_result_file_name, sproc_export_file_name),
            local_result_file_name,
            statement_params=statement_params
        )
        with open(os.path.join(local_result_file_name, sproc_export_file_name),mode="r+b") as result_file_obj:
            self._sklearn_object = cp.load(result_file_obj)

        cleanup_temp_files([local_transform_file_name, local_result_file_name])

    def _fit_pandas(self, dataset: pd.DataFrame) -> None:
        argspec = inspect.getfullargspec(self._sklearn_object.fit)
        args = {'X': dataset[self.input_cols]}
        if self.label_cols:
            label_arg_name = "Y" if "Y" in argspec.args else "y"
            args[label_arg_name] = dataset[self.label_cols].squeeze()

        if self.sample_weight_col is not None and "sample_weight" in argspec.args:
            args['sample_weight'] = dataset[self.sample_weight_col].squeeze()

        self._sklearn_object.fit(**args)

    def _get_pass_through_columns(self, dataset: DataFrame) -> List[str]:
        if self._drop_input_cols:
            return []
        else:
            return list(set(dataset.columns) - set(self.output_cols))

    def _batch_inference(
        self,
        dataset: DataFrame,
        inference_method: str,
        expected_output_cols_list: List[str],
        expected_output_cols_type: str = "",
    ) -> DataFrame:
        """Util method to create UDF and run batch inference.
        """
        if not self._is_fitted:
            raise RuntimeError(f"Estimator not fitted before calling {inference_method} method.")

        session = dataset._session
        # Validate that key package version in user workspace are supported in snowflake conda channel
        pkg_version_utils.get_valid_pkg_versions_supported_in_snowflake_conda_channel(
            pkg_versions=self._get_dependencies(), session=session, subproject=_SUBPROJECT)

        # Register vectorized UDF for batch inference
        batch_inference_udf_name = "SNOWML_BATCH_INFERENCE_{safe_id}_{method}".format(
                safe_id=self._get_rand_id(), method=inference_method)

        # Need to do this since if we use self._sklearn_object directly in the UDF, Snowpark
        # will try to pickle all of self which fails.
        estimator = self._sklearn_object

        # Input columns for UDF are sorted by column names.
        # We need actual order of input cols to reorder dataframe before calling inference methods.
        input_cols = self.input_cols
        unquoted_input_cols = identifier.get_unescaped_names(self.input_cols)

        statement_params = telemetry.get_function_usage_statement_params(
            project=_PROJECT,
            subproject=_SUBPROJECT,
            function_name=telemetry.get_statement_params_full_func_name(
                inspect.currentframe(), self.__class__.__name__
            ),
            api_calls=[pandas_udf],
            custom_tags=dict([("autogen", True)]),
        )

        @pandas_udf(
            is_permanent=False,
            name=batch_inference_udf_name,
            packages= self._get_dependencies(),
            replace=True,
            session=session,
            statement_params=statement_params,
        )
        def vec_batch_infer(ds: PandasSeries[dict]) -> PandasSeries[dict]:
            import pandas as pd
            import numpy as np

            input_df = pd.io.json.json_normalize(ds)

            # pd.io.json.json_normalize() doesn't remove quotes around quoted identifiers like snowpakr_df.to_pandas().
            # But trained models have unquoted input column names saved in internal state if trained using snowpark_df
            # or quoted input column names saved in internal state if trained using pandas_df.
            # Model expects exact same columns names in the input df for predict call.

            input_df = input_df[input_cols] # Select input columns with quoted column names.
            if hasattr(estimator, "feature_names_in_"):
                missing_features = []
                for i, f in enumerate(getattr(estimator, "feature_names_in_")):
                    if i >= len(input_cols) or (input_cols[i] != f and unquoted_input_cols[i] != f):
                        missing_features.append(f)

                if len(missing_features) > 0:
                    raise ValueError(
                        "The feature names should match with those that were passed during fit.\n"
                        f"Features seen during fit call but not present in the input: {missing_features}\n"
                        f"Features in the input dataframe : {input_cols}\n"
                    )
                input_df.columns = getattr(estimator, "feature_names_in_")
            else:
                # Just rename the column names to unquoted identifiers.
                input_df.columns = unquoted_input_cols # Replace the quoted columns identifier with unquoted column ids.
            transformed_numpy_array = getattr(estimator, inference_method)(input_df)
            if (
                isinstance(transformed_numpy_array, list)
                and len(transformed_numpy_array) > 0
                and isinstance(transformed_numpy_array[0], np.ndarray)
            ):
                # In case of multioutput estimators, predict_proba(), decision_function(), etc., functions return
                # a list of ndarrays. We need to concatenate them.
                transformed_numpy_array = np.concatenate(transformed_numpy_array, axis=1)

            if len(transformed_numpy_array.shape) == 3:
                # VotingClassifier will return results of shape (n_classifiers, n_samples, n_classes)
                # when voting = "soft" and flatten_transform = False. We can't handle unflatten transforms,
                # so we ignore flatten_transform flag and flatten the results.
                transformed_numpy_array = np.hstack(transformed_numpy_array)

            if (
                len(transformed_numpy_array.shape) > 1
                and transformed_numpy_array.shape[1] != len(expected_output_cols_list)
            ):
                # HeterogeneousEnsemble's transfrom method produce results with variying shapes
                # from (n_samples, n_estimators) to (n_samples, n_estimators * n_classes). 
                # It is hard to predict the response shape without using fragile introspection logic.
                # So, to avoid that we are packing the results into a dataframe of shape (n_samples, 1) with
                # each element being a list.
                if len(expected_output_cols_list) != 1:
                    raise TypeError("expected_output_cols_list must be same length as transformed array or "
                            "should be of length 1")
                series = pd.Series(transformed_numpy_array.tolist())
                transformed_pandas_df = pd.DataFrame(series, columns=expected_output_cols_list)
            else:
                transformed_pandas_df = pd.DataFrame(transformed_numpy_array, columns=expected_output_cols_list)
            return transformed_pandas_df.to_dict("records")

        batch_inference_table_name = "SNOWML_BATCH_INFERENCE_INPUT_TABLE_{safe_id}".format(
            safe_id=self._get_rand_id()
        )

        pass_through_columns = self._get_pass_through_columns(dataset)
        # Run Transform
        query_from_df = str(dataset.queries["queries"][0])

        outer_select_list = pass_through_columns[:]
        inner_select_list = pass_through_columns[:]

        outer_select_list.extend([
                "{object_name}:{column_name}{udf_datatype} as {column_name}".format(
                    object_name=batch_inference_udf_name,
                    column_name=c,
                    udf_datatype=(f"::{expected_output_cols_type}" if expected_output_cols_type else "")
                )
                for c in expected_output_cols_list
        ])

        inner_select_list.extend([
            "{udf_name}(object_construct_keep_null({input_cols_dict})) AS {udf_name}".format(
                udf_name=batch_inference_udf_name,
                input_cols_dict=", ".join([f"'{c}', {c}" for c in self.input_cols]),
            )
        ])

        sql = """WITH {input_table_name} AS ({query})
                    SELECT
                      {outer_select_stmt}
                    FROM (
                      SELECT
                        {inner_select_stmt}
                      FROM {input_table_name}
                    )
               """.format(
            input_table_name=batch_inference_table_name,
            query=query_from_df,
            outer_select_stmt=", ".join(outer_select_list),
            inner_select_stmt=", ".join(inner_select_list),
        )

        return session.sql(sql)

    def _sklearn_inference(
        self,
        dataset: pd.DataFrame,
        inference_method: str,
        expected_output_cols_list: List[str]
    ) -> pd.DataFrame:
        output_cols = expected_output_cols_list.copy()

        # Model expects exact same columns names in the input df for predict call.
        # Given the scenario that user use snowpark DataFrame in fit call, but pandas DataFrame in predict call
        # input cols need to match unquoted / quoted
        input_cols = self.input_cols
        unquoted_input_cols = identifier.get_unescaped_names(self.input_cols)
        quoted_input_cols = identifier.get_escaped_names(unquoted_input_cols)

        estimator = self._sklearn_object

        features_required_by_estimator =  getattr(estimator, "feature_names_in_") if hasattr(estimator, "feature_names_in_") else unquoted_input_cols
        missing_features = []
        features_in_dataset = set(dataset.columns)
        columns_to_select = []
        for i, f in enumerate(features_required_by_estimator):
            if (
                    i >= len(input_cols)
                or (input_cols[i] != f and unquoted_input_cols[i] != f and quoted_input_cols[i] != f)
                or (input_cols[i] not in features_in_dataset and unquoted_input_cols[i] not in features_in_dataset 
                    and quoted_input_cols[i] not in features_in_dataset)
                ):
                missing_features.append(f)
            elif input_cols[i] in features_in_dataset:
                columns_to_select.append(input_cols[i])
            elif unquoted_input_cols[i] in features_in_dataset:
                columns_to_select.append(unquoted_input_cols[i])
            else:
                columns_to_select.append(quoted_input_cols[i])

        if len(missing_features) > 0:
            raise ValueError(
                "The feature names should match with those that were passed during fit.\n"
                f"Features seen during fit call but not present in the input: {missing_features}\n"
                f"Features in the input dataframe : {input_cols}\n"
            )
        input_df = dataset[columns_to_select]
        input_df.columns = features_required_by_estimator

        transformed_numpy_array = getattr(estimator, inference_method)(
            input_df
        )

        if (
            isinstance(transformed_numpy_array, list)
            and len(transformed_numpy_array) > 0
            and isinstance(transformed_numpy_array[0], np.ndarray)
        ):
            # In case of multioutput estimators, predict_proba(), decision_function(), etc., functions return 
            # a list of ndarrays. We need to concatenate them.

            # First compute output column names
            if len(output_cols) == len(transformed_numpy_array):
                actual_output_cols = []
                for idx, np_arr in enumerate(transformed_numpy_array):
                    for i in range(1 if len(np_arr.shape) <= 1 else np_arr.shape[1]):
                        actual_output_cols.append(f"{output_cols[idx]}_{i}")
                output_cols = actual_output_cols

            # Concatenate np arrays
            transformed_numpy_array = np.concatenate(transformed_numpy_array, axis=1)

        if len(transformed_numpy_array.shape) == 3:
            # VotingClassifier will return results of shape (n_classifiers, n_samples, n_classes)
            # when voting = "soft" and flatten_transform = False. We can't handle unflatten transforms,
            # so we ignore flatten_transform flag and flatten the results.
            transformed_numpy_array = np.hstack(transformed_numpy_array)

        if len(transformed_numpy_array.shape) == 1:
            transformed_numpy_array = np.reshape(transformed_numpy_array, (-1, 1))

        shape = transformed_numpy_array.shape
        if shape[1] != len(output_cols):
            if len(output_cols) != 1:
                raise TypeError(
                    "expected_output_cols_list must be same length as transformed array or "
                    "should be of length 1 or should be of length number of label columns"
                )
            actual_output_cols = []
            for i in range(shape[1]):
                actual_output_cols.append(f"{output_cols[0]}_{i}")
            output_cols = actual_output_cols

        if self._drop_input_cols:
            dataset = pd.DataFrame(data=transformed_numpy_array, columns=output_cols)
        else:
            dataset = dataset.copy()
            dataset[output_cols] = transformed_numpy_array
        return dataset

    @available_if(_original_estimator_has_callable("predict"))
    @telemetry.send_api_usage_telemetry(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    @telemetry.add_stmt_params_to_df(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    def predict(self, dataset: Union[DataFrame, pd.DataFrame]) -> Union[DataFrame, pd.DataFrame]:
        """Predict with `X`
        For more details on this function, see [xgboost.XGBRFClassifier.predict]
        (https://xgboost.readthedocs.io/en/stable/python/python_api.html#xgboost.XGBRFClassifier.predict)


        Raises:
            TypeError: Supported dataset types: snowpark.DataFrame, pandas.DataFrame.

        Args:
            dataset: Union[snowflake.snowpark.DataFrame, pandas.DataFrame]
                Snowpark or Pandas DataFrame.

        Returns:
            Transformed dataset.
        """
        if isinstance(dataset, DataFrame):
            expected_type_inferred = ""
            # when it is classifier, infer the datatype from label columns
            if expected_type_inferred == "" and 'predict' in self.model_signatures:
                expected_type_inferred = convert_sp_to_sf_type(
                    self.model_signatures['predict'].outputs[0].as_snowpark_type()
                )
            
            output_df = self._batch_inference(
                dataset=dataset,
                inference_method="predict",
                expected_output_cols_list=self.output_cols,
                expected_output_cols_type=expected_type_inferred,
            )
        elif isinstance(dataset, pd.DataFrame):
            output_df = self._sklearn_inference(
                dataset=dataset,
                inference_method="predict",
                expected_output_cols_list=self.output_cols,)
        else:
            raise TypeError(
                f"Unexpected dataset type: {type(dataset)}."
                "Supported dataset types: snowpark.DataFrame, pandas.DataFrame."
            )

        return output_df

    @available_if(_original_estimator_has_callable("transform"))
    @telemetry.send_api_usage_telemetry(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    @telemetry.add_stmt_params_to_df(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    def transform(self, dataset: Union[DataFrame, pd.DataFrame]) -> Union[DataFrame, pd.DataFrame]:
        """Method not supported for this class.


        Raises:
            TypeError: Supported dataset types: snowpark.DataFrame, pandas.DataFrame.

        Args:
            dataset: Union[snowflake.snowpark.DataFrame, pandas.DataFrame]
                Snowpark or Pandas DataFrame.

        Returns:
            Transformed dataset.
        """
        if isinstance(dataset, DataFrame):
            expected_dtype = ""
            if False:  # is child of _BaseHeterogeneousEnsemble
                # transform() method of HeterogeneousEnsemble estimators return responses of varying shapes
                # from (n_samples, n_estimators) to (n_samples, n_estimators * n_classes) (and everything in between)
                # based on init param values. We will convert that to pandas dataframe of shape (n_samples, 1) with
                # each row containing a list of values.
                expected_dtype = "ARRAY"

            output_df = self._batch_inference(
                dataset=dataset,
                inference_method="transform",
                expected_output_cols_list=self.output_cols,
                expected_output_cols_type=expected_dtype,
            )
        elif isinstance(dataset, pd.DataFrame):
            output_df = self._sklearn_inference(
                dataset=dataset,
                inference_method="transform",
                expected_output_cols_list=self.output_cols,
            )
        else:
            raise TypeError(
                f"Unexpected dataset type: {type(dataset)}."
                "Supported dataset types: snowpark.DataFrame, pandas.DataFrame."
            )

        return output_df

    def _get_output_column_names(self, output_cols_prefix: str) -> List[str]:
        """ Returns the list of output columns for predict_proba(), decision_function(), etc.. functions.
        Returns a list with output_cols_prefix as the only element if the estimator is not a classifier.
        """
        if getattr(self._sklearn_object, "classes_", None) is None:
            return [output_cols_prefix]

        classes = self._sklearn_object.classes_
        if isinstance(classes, numpy.ndarray):
            return [f'{output_cols_prefix}{c}' for c in classes.tolist()]
        elif isinstance(classes, list) and len(classes) > 0 and isinstance(classes[0], numpy.ndarray):
            # If the estimator is a multioutput estimator, classes_ will be a list of ndarrays.
            output_cols = []
            for i, cl in enumerate(classes):
                # For binary classification, there is only one output column for each class
                # ndarray as the two classes are complementary.
                if len(cl) == 2:
                    output_cols.append(f'{output_cols_prefix}_{i}_{cl[0]}')
                else:
                    output_cols.extend([
                        f'{output_cols_prefix}_{i}_{c}' for c in cl.tolist()
                    ])
            return output_cols
        return []

    @available_if(_original_estimator_has_callable("predict_proba"))
    @telemetry.send_api_usage_telemetry(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    @telemetry.add_stmt_params_to_df(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    def predict_proba(
        self, dataset: Union[DataFrame, pd.DataFrame], output_cols_prefix: str = "predict_proba_"
    ) -> Union[DataFrame, pd.DataFrame]:
        """Predict the probability of each `X` example being of a given class
        For more details on this function, see [xgboost.XGBRFClassifier.predict_proba]
        (https://xgboost.readthedocs.io/en/stable/python/python_api.html#xgboost.XGBRFClassifier.predict_proba)


        Raises:
            TypeError: Supported dataset types: snowpark.DataFrame, pandas.DataFrame.

        Args:
            dataset: Union[snowflake.snowpark.DataFrame, pandas.DataFrame]
                Snowpark or Pandas DataFrame.
            output_cols_prefix: Prefix for the response columns

        Returns:
            Output dataset with probability of the sample for each class in the model.
        """
        if isinstance(dataset, DataFrame):
            output_df = self._batch_inference(
                dataset=dataset,
                inference_method="predict_proba",
                expected_output_cols_list=self._get_output_column_names(output_cols_prefix),
                expected_output_cols_type="float"
            )
        elif isinstance(dataset, pd.DataFrame):
            output_df = self._sklearn_inference(
                dataset=dataset,
                inference_method="predict_proba",
                expected_output_cols_list=self._get_output_column_names(output_cols_prefix),
            )
        else:
            raise TypeError(
                f"Unexpected dataset type: {type(dataset)}."
                "Supported dataset types: snowpark.DataFrame, pandas.DataFrame."
            )

        return output_df

    @available_if(_original_estimator_has_callable("predict_log_proba"))
    @telemetry.send_api_usage_telemetry(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    @telemetry.add_stmt_params_to_df(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    def predict_log_proba(
        self, dataset: Union[DataFrame, pd.DataFrame], output_cols_prefix: str = "predict_log_proba_"
    ) -> Union[DataFrame, pd.DataFrame]:
        """Predict the probability of each `X` example being of a given class
        For more details on this function, see [xgboost.XGBRFClassifier.predict_proba]
        (https://xgboost.readthedocs.io/en/stable/python/python_api.html#xgboost.XGBRFClassifier.predict_proba)


        Raises:
            TypeError: Supported dataset types: snowpark.DataFrame, pandas.DataFrame.

        Args:
            dataset: Union[snowflake.snowpark.DataFrame, pandas.DataFrame]
                Snowpark or Pandas DataFrame.
            output_cols_prefix: str
                Prefix for the response columns

        Returns:
            Output dataset with log probability of the sample for each class in the model.
        """
        if isinstance(dataset, DataFrame):
            output_df = self._batch_inference(
                dataset=dataset,
                inference_method="predict_log_proba",
                expected_output_cols_list=self._get_output_column_names(output_cols_prefix),
                expected_output_cols_type="float"
            )
        elif isinstance(dataset, pd.DataFrame):
            output_df = self._sklearn_inference(
                dataset=dataset,
                inference_method="predict_log_proba",
                expected_output_cols_list=self._get_output_column_names(output_cols_prefix),
            )
        else:
            raise TypeError(
                f"Unexpected dataset type: {type(dataset)}."
                "Supported dataset types: snowpark.DataFrame, pandas.DataFrame."
            )

        return output_df

    @available_if(_original_estimator_has_callable("decision_function"))
    @telemetry.send_api_usage_telemetry(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    @telemetry.add_stmt_params_to_df(
        project=_PROJECT,
        subproject=_SUBPROJECT,
        custom_tags=dict([("autogen", True)]),
    )
    def decision_function(
        self, dataset: Union[DataFrame, pd.DataFrame], output_cols_prefix: str = "decision_function_"
    ) -> Union[DataFrame, pd.DataFrame]:
        """Method not supported for this class.


        Raises:
            TypeError: Supported dataset types: snowpark.DataFrame, pandas.DataFrame.

        Args:
            dataset: Union[snowflake.snowpark.DataFrame, pandas.DataFrame]
                Snowpark or Pandas DataFrame.
            output_cols_prefix: str
                Prefix for the response columns

        Returns:
            Output dataset with results of the decision function for the samples in input dataset.
        """
        if isinstance(dataset, DataFrame):
            output_df = self._batch_inference(
                dataset=dataset,
                inference_method="decision_function",
                expected_output_cols_list=self._get_output_column_names(output_cols_prefix),
                expected_output_cols_type="float"
            )
        elif isinstance(dataset, pd.DataFrame):
            output_df = self._sklearn_inference(
                dataset=dataset,
                inference_method="decision_function",
                expected_output_cols_list=self._get_output_column_names(output_cols_prefix),
            )
        else:
            raise TypeError(
                f"Unexpected dataset type: {type(dataset)}."
                "Supported dataset types: snowpark.DataFrame, pandas.DataFrame."
            )

        return output_df

    @available_if(_original_estimator_has_callable("score"))
    def score(self, dataset: Union[DataFrame, pd.DataFrame]) -> float:
        """

        Returns:
            Score.
        """
        self._infer_input_output_cols(dataset)
        if isinstance(dataset, pd.DataFrame):
            return self._score_sklearn(dataset)
        elif isinstance(dataset, DataFrame):
            return self._score_snowpark(dataset)
        else:
            raise TypeError(
                f"Unexpected dataset type: {type(dataset)}."
                "Supported dataset types: snowpark.DataFrame, pandas.DataFrame."
            )

    def _score_sklearn(self, dataset: pd.DataFrame) -> float:
        argspec = inspect.getfullargspec(self._sklearn_object.score)
        if "X" in argspec.args:
            args = {'X': dataset[self.input_cols]}
        elif "X_test" in argspec.args:
            args = {'X_test': dataset[self.input_cols]}
        else:
            raise RuntimeError("Neither 'X' or 'X_test' exist in argument")
        
        if self.label_cols:
            label_arg_name = "Y" if "Y" in argspec.args else "y"
            args[label_arg_name] = dataset[self.label_cols].squeeze()

        if self.sample_weight_col is not None and "sample_weight" in argspec.args:
            args['sample_weight'] = dataset[self.sample_weight_col].squeeze()

        score = self._sklearn_object.score(**args)
        return score

    def _score_snowpark(self, dataset: DataFrame) -> float:
        # Specify input columns so column pruing will be enforced
        selected_cols = (
            self.input_cols + self.label_cols + [self.sample_weight_col]
            if self.sample_weight_col is not None
            else []
        )
        if len(selected_cols) > 0:
            dataset = dataset.select(selected_cols)

        # Extract query that generated the dataframe. We will need to pass it to score procedure.
        query = str(dataset.queries["queries"][0])

        # Create a temp file and dump the score to that file.
        local_score_file_name = get_temp_file_path()
        with open(local_score_file_name, mode="w+b") as local_score_file:
            cp.dump(self._sklearn_object, local_score_file)

        # Create temp stage to run score.
        score_stage_name = "SNOWML_SCORE_{safe_id}".format(safe_id=self._get_rand_id())
        session = dataset._session
        stage_creation_query = f"CREATE OR REPLACE TEMPORARY STAGE {score_stage_name};"
        SqlResultValidator(
            session=session,
            query=stage_creation_query
        ).has_dimensions(
            expected_rows=1, expected_cols=1
        ).has_value_match(
            row_idx=0,
            col_idx=0,
            expected_value=f"Stage area {score_stage_name} successfully created."
        ).validate()

        # Use posixpath to construct stage paths
        stage_score_file_name = posixpath.join(score_stage_name, os.path.basename(local_score_file_name))
        score_sproc_name = "SNOWML_SCORE_{safe_id}".format(safe_id=self._get_rand_id())
        statement_params = telemetry.get_function_usage_statement_params(
            project=_PROJECT,
            subproject=_SUBPROJECT,
            function_name=telemetry.get_statement_params_full_func_name(
                inspect.currentframe(), self.__class__.__name__
            ),
            api_calls=[sproc],
            custom_tags=dict([("autogen", True)]),
        )
        # Put locally serialized score on stage.
        session.file.put(
            local_score_file_name,
            stage_score_file_name,
            auto_compress=False,
            overwrite=True,
            statement_params=statement_params
        )

        @sproc(
            is_permanent=False,
            name=score_sproc_name,
            packages=["snowflake-snowpark-python"] + self._get_dependencies(),
            replace=True,
            session=session,
            statement_params=statement_params,
            anonymous=True
        )
        def score_wrapper_sproc(
            session: Session,
            sql_query: str,
            stage_score_file_name: str,
            input_cols: List[str],
            label_cols: List[str],
            sample_weight_col: Optional[str],
            statement_params: Dict[str, str]
        ) -> float:
            import cloudpickle as cp
            import numpy as np
            import os
            import pandas
            import tempfile
            import inspect
            import xgboost

            df = session.sql(sql_query).to_pandas(statement_params=statement_params)

            local_score_file = tempfile.NamedTemporaryFile(delete=True)
            local_score_file_name = local_score_file.name
            local_score_file.close()

            session.file.get(stage_score_file_name, local_score_file_name, statement_params=statement_params)

            local_score_file_name_path = os.path.join(local_score_file_name, os.listdir(local_score_file_name)[0])
            with open(local_score_file_name_path, mode="r+b") as local_score_file_obj:
                estimator = cp.load(local_score_file_obj)

            argspec = inspect.getfullargspec(estimator.score)
            if "X" in argspec.args:
                args = {'X': df[input_cols]}
            elif "X_test" in argspec.args:
                args = {'X_test': df[input_cols]}
            else:
                raise RuntimeError("Neither 'X' or 'X_test' exist in argument")
            
            if label_cols:
                label_arg_name = "Y" if "Y" in argspec.args else "y"
                args[label_arg_name] = df[label_cols].squeeze()

            if sample_weight_col is not None and "sample_weight" in argspec.args:
                args['sample_weight'] = df[sample_weight_col].squeeze()

            result = estimator.score(**args)
            return result

        # Call score sproc
        statement_params = telemetry.get_function_usage_statement_params(
            project=_PROJECT,
            subproject=_SUBPROJECT,
            function_name=telemetry.get_statement_params_full_func_name(
                inspect.currentframe(), self.__class__.__name__
            ),
            api_calls=[Session.call],
            custom_tags=dict([("autogen", True)]),
        )
        score = score_wrapper_sproc(
            session,
            query,
            stage_score_file_name,
            identifier.get_unescaped_names(self.input_cols),
            identifier.get_unescaped_names(self.label_cols),
            identifier.get_unescaped_names(self.sample_weight_col),
            statement_params,
        )

        cleanup_temp_files([local_score_file_name])

        return score

    def _get_model_signatures(self, dataset: Union[DataFrame, pd.DataFrame]) -> None:
        self._model_signature_dict: Dict[str, ModelSignature] = dict()

        PROB_FUNCTIONS = ["predict_log_proba", "predict_proba", "decision_function"]

        inputs = _infer_signature(dataset[self.input_cols], "input")
        if hasattr(self, "predict"):
            # For classifier, the type of predict is the same as the type of label
            if self._sklearn_object._estimator_type == 'classifier':
                outputs = _infer_signature(dataset[self.label_cols], "output")  # label columns is the desired type for output
                outputs = _rename_features(outputs, self.output_cols)  # rename the output columns
                self._model_signature_dict["predict"] = ModelSignature(inputs, 
                                                                       ([] if self._drop_input_cols else inputs) + outputs)
            # For regressor, the type of predict is float64
            elif self._sklearn_object._estimator_type == 'regressor':
                outputs = [FeatureSpec(dtype=DataType.DOUBLE, name=c) for c in self.output_cols]
                self._model_signature_dict["predict"] = ModelSignature(inputs, 
                                                                       ([] if self._drop_input_cols else inputs) + outputs)
        for prob_func in PROB_FUNCTIONS:
            if hasattr(self, prob_func):
                output_cols_prefix: str = f"{prob_func}_"
                output_column_names = self._get_output_column_names(output_cols_prefix)
                outputs = [FeatureSpec(dtype=DataType.DOUBLE, name=c) for c in output_column_names]
                self._model_signature_dict[prob_func] = ModelSignature(inputs, 
                                                                       ([] if self._drop_input_cols else inputs) + outputs)

    @property
    def model_signatures(self) -> Dict[str, ModelSignature]:
        if self._model_signature_dict is None:
            raise RuntimeError("Estimator not fitted before accessing property model_signatures! ")
        return self._model_signature_dict

    def to_xgboost(self) -> Any:
        if self._sklearn_object is None:
            self._sklearn_object = self._create_sklearn_object()
        return self._sklearn_object

    def to_sklearn(self) -> Any:
        raise AttributeError("Estimator doesn't support to_sklearn(). Please use to_xgboost()")

    def to_lightgbm(self) -> Any:
        raise AttributeError("Estimator doesn't support to_lightgbm(). Please use to_xgboost()")

    def _get_dependencies(self) -> List[str]:
        return self._deps
