"""
Hunk applicator module for applying specific hunks to the git staging area.
"""

import os
import subprocess
import tempfile
from typing import List, Dict, Optional, Tuple, Set
from .diff_parser import Hunk, validate_hunk_combination, create_dependency_groups


class HunkApplicatorError(Exception):
    """Custom exception for hunk application errors."""
    pass


def apply_hunks(hunk_ids: List[str], hunks_by_id: Dict[str, Hunk], base_diff: str) -> bool:
    """
    Apply specific hunks to the git staging area using dependency-aware grouping.

    Args:
        hunk_ids: List of hunk IDs to apply
        hunks_by_id: Dictionary mapping hunk IDs to Hunk objects
        base_diff: Original full diff output

    Returns:
        True if successful, False otherwise

    Raises:
        HunkApplicatorError: If hunk application fails
    """
    if not hunk_ids:
        return True

    # Get the hunks to apply
    hunks_to_apply = []
    for hunk_id in hunk_ids:
        if hunk_id not in hunks_by_id:
            raise HunkApplicatorError(f"Hunk ID not found: {hunk_id}")
        hunks_to_apply.append(hunks_by_id[hunk_id])

    # Validate that hunks can be applied together
    is_valid, error_msg = validate_hunk_combination(hunks_to_apply)
    if not is_valid:
        raise HunkApplicatorError(f"Invalid hunk combination: {error_msg}")

    # Use dependency-aware application for better handling of complex changes
    return _apply_hunks_with_dependencies(hunks_to_apply, base_diff)


def _apply_hunks_with_dependencies(hunks: List[Hunk], base_diff: str) -> bool:
    """
    Apply hunks using dependency-aware grouping for better handling of complex changes.

    Args:
        hunks: List of hunks to apply
        base_diff: Original full diff output

    Returns:
        True if all hunks applied successfully, False otherwise
    """
    # Create dependency groups
    dependency_groups = create_dependency_groups(hunks)

    print(f"Dependency analysis: {len(dependency_groups)} groups identified")
    for i, group in enumerate(dependency_groups):
        print(f"  Group {i+1}: {len(group)} hunks")
        for hunk in group:
            deps = len(hunk.dependencies)
            dependents = len(hunk.dependents)
            print(f"    - {hunk.id} ({hunk.change_type}, deps: {deps}, dependents: {dependents})")

    # Apply groups in order
    for i, group in enumerate(dependency_groups):
        print(f"Applying group {i+1}/{len(dependency_groups)} ({len(group)} hunks)...")

        # CRITICAL FIX: Save staging state before attempting group application
        # This prevents corrupt patch failures from leaving the repository in a broken state
        group_staging_state = _save_staging_state()

        if len(group) == 1:
            # Single hunk - apply individually for better error isolation
            success = _apply_hunks_sequentially(group, base_diff)
        else:
            # Multiple interdependent hunks - try atomic application first
            success = _apply_dependency_group_atomically(group, base_diff)

            if not success:
                print("  Atomic application failed, trying sequential with smart ordering...")
                # Fallback to sequential application with dependency ordering
                success = _apply_dependency_group_sequentially(group, base_diff)

        if not success:
            print(f"Failed to apply group {i+1}, restoring staging state...")
            # CRITICAL FIX: Restore staging state to prevent broken repository state
            _restore_staging_state(group_staging_state)
            return False

        print(f"✓ Group {i+1} applied successfully")

    return True


def _apply_dependency_group_atomically(hunks: List[Hunk], base_diff: str) -> bool:
    """
    Apply a dependency group using git's native patch application with proper line number calculation.

    Args:
        hunks: List of hunks in the dependency group
        base_diff: Original full diff output

    Returns:
        True if successful, False otherwise
    """
    try:
        # Generate valid patch with corrected line numbers
        from .diff_parser import _create_valid_git_patch
        patch_content = _create_valid_git_patch(hunks, base_diff)

        if not patch_content.strip():
            print("No valid patch content generated")
            return False

        # Apply using git's native mechanism
        return _apply_patch_with_git(patch_content)

    except Exception as e:
        print(f"Error in atomic application: {e}")
        return False


def _apply_dependency_group_sequentially(hunks: List[Hunk], base_diff: str) -> bool:
    """
    Apply hunks in a dependency group sequentially using git native mechanisms.

    Args:
        hunks: List of hunks in the dependency group
        base_diff: Original full diff output

    Returns:
        True if successful, False otherwise
    """
    # Order hunks by dependencies (topological sort)
    ordered_hunks = _topological_sort_hunks(hunks)

    if not ordered_hunks:
        # Fallback to simple ordering if topological sort fails
        ordered_hunks = sorted(hunks, key=lambda h: (h.file_path, h.start_line))

    # Apply hunks in dependency order using git native mechanisms
    for i, hunk in enumerate(ordered_hunks):
        try:
            success = _relocate_and_apply_hunk(hunk, base_diff)
            if not success:
                print(f"Failed to apply hunk {hunk.id} ({i+1}/{len(ordered_hunks)}) via git apply")
                return False

        except Exception as e:
            print(f"Error applying hunk {hunk.id}: {e}")
            return False

    return True


def _topological_sort_hunks(hunks: List[Hunk]) -> List[Hunk]:
    """
    Sort hunks based on their dependencies using topological sort.

    Args:
        hunks: List of hunks to sort

    Returns:
        List of hunks in dependency order, or empty list if cyclic dependencies
    """
    # Build hunk map for quick lookups
    hunk_map = {hunk.id: hunk for hunk in hunks}
    hunk_ids = set(hunk.id for hunk in hunks)

    # Calculate in-degrees (number of dependencies within this group)
    in_degree = {}
    for hunk in hunks:
        # Only count dependencies that are within this group
        local_deps = hunk.dependencies & hunk_ids
        in_degree[hunk.id] = len(local_deps)

    # Start with hunks that have no dependencies within the group
    queue = [hunk_id for hunk_id in hunk_ids if in_degree[hunk_id] == 0]
    result = []

    while queue:
        current_id = queue.pop(0)
        result.append(hunk_map[current_id])

        # Reduce in-degree for dependents
        current_hunk = hunk_map[current_id]
        for dependent_id in current_hunk.dependents:
            if dependent_id in hunk_ids:
                in_degree[dependent_id] -= 1
                if in_degree[dependent_id] == 0:
                    queue.append(dependent_id)

    # Check for cycles
    if len(result) != len(hunks):
        print("Warning: Cyclic dependencies detected, using fallback ordering")
        return []

    return result


def _apply_hunks_sequentially(hunks: List[Hunk], base_diff: str) -> bool:
    """
    Apply hunks one by one using git native mechanisms for better reliability.

    Args:
        hunks: List of hunks to apply
        base_diff: Original full diff output

    Returns:
        True if all hunks applied successfully, False otherwise
    """
    # Sort hunks by file and line number for consistent application order
    sorted_hunks = sorted(hunks, key=lambda h: (h.file_path, h.start_line))

    for i, hunk in enumerate(sorted_hunks):
        try:
            # Use git native patch application
            success = _relocate_and_apply_hunk(hunk, base_diff)
            if not success:
                print(f"Failed to apply hunk {hunk.id} ({i+1}/{len(sorted_hunks)}) via git apply")
                return False

        except Exception as e:
            print(f"Error applying hunk {hunk.id}: {e}")
            return False

    return True



def _extract_files_from_patch(patch_content: str) -> Set[str]:
    """
    Extract the list of files affected by a patch.
    
    Args:
        patch_content: The patch content
        
    Returns:
        Set of file paths affected by the patch
    """
    files = set()
    lines = patch_content.split('\n')
    
    for line in lines:
        if line.startswith('diff --git'):
            # Extract file path from diff header
            # Format: diff --git a/path/to/file b/path/to/file
            parts = line.split()
            if len(parts) >= 4:
                # Remove 'b/' prefix
                file_path = parts[3][2:] if parts[3].startswith('b/') else parts[3]
                files.add(file_path)
        elif line.startswith('+++'):
            # Alternative: extract from +++ header
            # Format: +++ b/path/to/file
            parts = line.split()
            if len(parts) >= 2 and parts[1] != '/dev/null':
                file_path = parts[1][2:] if parts[1].startswith('b/') else parts[1]
                files.add(file_path)
    
    return files


def _sync_files_from_staging(file_paths: Set[str]) -> bool:
    """
    Sync specific files from staging area to working directory.
    
    Args:
        file_paths: Set of file paths to sync
        
    Returns:
        True if all files synced successfully
    """
    if not file_paths:
        # If no files specified, sync all
        try:
            subprocess.run(['git', 'checkout-index', '-f', '-a'], check=True, capture_output=True)
            return True
        except subprocess.CalledProcessError:
            return False
    
    # Sync each file individually
    all_success = True
    for file_path in file_paths:
        try:
            # Use git checkout-index to sync specific file
            result = subprocess.run(
                ['git', 'checkout-index', '-f', '--', file_path],
                capture_output=True,
                text=True,
                check=False
            )
            
            if result.returncode != 0:
                print(f"Failed to sync {file_path}: {result.stderr}")
                all_success = False
                
        except Exception as e:
            print(f"Error syncing {file_path}: {e}")
            all_success = False
    
    return all_success


def _apply_patch_with_git(patch_content: str) -> bool:
    """
    Apply a patch using git's native mechanism with improved file-specific sync.

    Args:
        patch_content: The patch content to apply

    Returns:
        True if successfully applied, False otherwise
    """
    try:
        # Extract affected files from patch content
        affected_files = _extract_files_from_patch(patch_content)
        
        # Save current staging state for rollback
        staging_state = _save_staging_state()
        
        # CRITICAL FIX: Save working directory state for affected files BEFORE any patch operations
        working_dir_state = _save_working_dir_state(affected_files)
        
        # CRITICAL FIX: Also save the current staging state before applying patches
        original_staging_state = _save_staging_state()

        # Create temporary patch file with enhanced validation
        with tempfile.NamedTemporaryFile(mode='w', suffix='.patch', delete=False) as patch_file:
            patch_file.write(patch_content)
            patch_file.flush()  # CRITICAL FIX: Ensure content is written to disk before git reads it
            # ADDITIONAL FIX: Force OS to sync the file to disk
            os.fsync(patch_file.fileno())
            patch_file_path = patch_file.name

        try:
            # CRITICAL FIX: Validate patch file is readable before attempting to apply
            try:
                with open(patch_file_path, 'r') as test_read:
                    patch_verification = test_read.read()
                    if patch_verification != patch_content:
                        raise Exception("Patch file write verification failed")
            except Exception as e:
                print(f"Patch file validation failed: {e}")
                # Restore states and return failure
                _restore_working_dir_state(working_dir_state, affected_files)
                _restore_staging_state(original_staging_state)
                return False
            
            # Apply patch using git apply --index to update both staging area and working directory
            # This ensures that the working directory is immediately synchronized with the staging area
            result = subprocess.run(
                ['git', 'apply', '--index', '--whitespace=nowarn', patch_file_path],
                capture_output=True,
                text=True,
                cwd=os.getcwd()
            )

            if result.returncode == 0:
                print("✓ Patch applied successfully via git apply --index")
                # CRITICAL FIX: Verify working directory matches expected state after successful apply
                if _verify_working_dir_integrity(affected_files, patch_content):
                    return True
                else:
                    print("Warning: Working directory integrity check failed after successful patch apply")
                    # Don't fail here, but log the issue
                    return True
            else:
                print(f"Git apply --index failed: {result.stderr}")
                # CRITICAL FIX: Immediately restore BOTH working directory and staging states
                print("Restoring working directory and staging states after --index failure...")
                _restore_working_dir_state(working_dir_state, affected_files)
                _restore_staging_state(original_staging_state)
                
                # If --index fails, fallback to --cached and then sync working directory
                result_cached = subprocess.run(
                    ['git', 'apply', '--cached', '--whitespace=nowarn', patch_file_path],
                    capture_output=True,
                    text=True,
                    cwd=os.getcwd()
                )
                
                if result_cached.returncode == 0:
                    print("✓ Patch applied to staging area, syncing working directory...")
                    
                    # Sync only the affected files from staging to working directory
                    # This is more precise than syncing all files with checkout-index -a
                    sync_success = _sync_files_from_staging(affected_files)
                    
                    if sync_success:
                        print("✓ Working directory synchronized for affected files")
                        return True
                    else:
                        print("Failed to sync working directory")
                        # CRITICAL FIX: Restore both staging and working directory states
                        _restore_staging_state(staging_state)
                        _restore_working_dir_state(working_dir_state, affected_files)
                        return False
                else:
                    print(f"Git apply --cached also failed: {result_cached.stderr}")
                    # CRITICAL FIX: Restore both staging and working directory states
                    _restore_staging_state(staging_state)
                    _restore_working_dir_state(working_dir_state, affected_files)
                    return False

        finally:
            # Clean up temporary file
            os.unlink(patch_file_path)

    except Exception as e:
        print(f"Error applying patch with git: {e}")
        # CRITICAL FIX: Restore working directory state on any exception
        try:
            _restore_working_dir_state(working_dir_state, affected_files)
        except:
            pass
        return False


def _save_staging_state() -> Optional[str]:
    """
    Save current staging state for rollback.

    Returns:
        Staging state identifier or None if unable to save
    """
    try:
        # Get current staged diff
        result = subprocess.run(
            ['git', 'diff', '--cached'],
            capture_output=True,
            text=True,
            check=True
        )
        return result.stdout
    except:
        return None


def _restore_staging_state(saved_state: Optional[str]) -> bool:
    """
    Restore staging state from saved state.

    Args:
        saved_state: Previously saved staging state

    Returns:
        True if restoration successful
    """
    try:
        if saved_state is None:
            return True

        # Reset staging area
        subprocess.run(['git', 'reset', 'HEAD'], capture_output=True, check=True)

        # If there was staged content, reapply it
        if saved_state.strip():
            with tempfile.NamedTemporaryFile(mode='w', suffix='.patch', delete=False) as patch_file:
                patch_file.write(saved_state)
                patch_file_path = patch_file.name

            try:
                subprocess.run(
                    ['git', 'apply', '--cached', patch_file_path],
                    capture_output=True,
                    check=True
                )
            finally:
                os.unlink(patch_file_path)

        return True
    except:
        return False


def _save_working_dir_state(affected_files: Set[str]) -> Dict[str, str]:
    """
    Save current working directory state for affected files with enhanced reliability.

    Args:
        affected_files: Set of file paths to save state for

    Returns:
        Dictionary mapping file paths to their content, or empty dict if unable to save
    """
    file_states = {}
    try:
        for file_path in affected_files:
            try:
                if os.path.exists(file_path):
                    # CRITICAL FIX: Use binary mode first to handle any file type
                    try:
                        with open(file_path, 'rb') as f:
                            binary_content = f.read()
                        # Try to decode as UTF-8, fall back to latin-1 if needed
                        try:
                            file_states[file_path] = binary_content.decode('utf-8')
                        except UnicodeDecodeError:
                            file_states[file_path] = binary_content.decode('latin-1')
                    except Exception:
                        # Final fallback: read as text with ignore errors
                        with open(file_path, 'r', encoding='utf-8', errors='ignore') as f:
                            file_states[file_path] = f.read()
                else:
                    # Mark non-existent files as such
                    file_states[file_path] = None
                    
                # CRITICAL FIX: Verify the file was read correctly by checking length
                if file_states[file_path] is not None and os.path.exists(file_path):
                    expected_size = os.path.getsize(file_path)
                    if expected_size > 0 and len(file_states[file_path]) == 0:
                        print(f"Warning: File {file_path} appears non-empty but read as empty")
                        
            except Exception as e:
                print(f"Warning: Could not save state for {file_path}: {e}")
                # Continue with other files
        
        print(f"Successfully saved working directory state for {len(file_states)} files")
        return file_states
    except Exception as e:
        print(f"Error saving working directory state: {e}")
        return {}


def _restore_working_dir_state(saved_states: Dict[str, str], affected_files: Set[str]) -> bool:
    """
    Restore working directory state for affected files with enhanced reliability.

    Args:
        saved_states: Dictionary mapping file paths to their saved content
        affected_files: Set of file paths to restore

    Returns:
        True if restoration successful
    """
    try:
        if not saved_states:
            print("No saved states to restore")
            return True

        restored_count = 0
        failed_count = 0
        
        for file_path in affected_files:
            try:
                if file_path in saved_states:
                    saved_content = saved_states[file_path]
                    if saved_content is None:
                        # File didn't exist, remove it if it exists now
                        if os.path.exists(file_path):
                            os.remove(file_path)
                            print(f"Removed file that shouldn't exist: {file_path}")
                            restored_count += 1
                    else:
                        # CRITICAL FIX: Create directory if it doesn't exist
                        dir_path = os.path.dirname(file_path)
                        if dir_path and not os.path.exists(dir_path):
                            os.makedirs(dir_path, exist_ok=True)
                        
                        # CRITICAL FIX: Use same encoding strategy as saving
                        try:
                            # Try to encode as UTF-8 first
                            content_bytes = saved_content.encode('utf-8')
                            with open(file_path, 'wb') as f:
                                f.write(content_bytes)
                        except UnicodeEncodeError:
                            # Fallback to latin-1 if UTF-8 fails
                            content_bytes = saved_content.encode('latin-1')
                            with open(file_path, 'wb') as f:
                                f.write(content_bytes)
                        
                        # CRITICAL FIX: Verify the file was written correctly
                        try:
                            with open(file_path, 'r', encoding='utf-8', errors='ignore') as f:
                                written_content = f.read()
                            if len(written_content) != len(saved_content):
                                print(f"Warning: File {file_path} restoration size mismatch: expected {len(saved_content)}, got {len(written_content)}")
                        except Exception:
                            pass  # Verification failed, but file was written
                        
                        print(f"Restored working directory state for: {file_path}")
                        restored_count += 1
                else:
                    print(f"Warning: No saved state found for {file_path}")
                    failed_count += 1
                    
            except Exception as e:
                print(f"Error: Could not restore {file_path}: {e}")
                failed_count += 1
                # Continue with other files

        print(f"Working directory restoration: {restored_count} restored, {failed_count} failed out of {len(affected_files)} files")
        return failed_count == 0
    except Exception as e:
        print(f"Error during working directory restoration: {e}")
        return False


def _verify_working_dir_integrity(affected_files: Set[str], patch_content: str) -> bool:
    """
    Verify that working directory files are in a valid state after patch application.
    
    Args:
        affected_files: Set of file paths that were modified
        patch_content: The patch content that was applied
        
    Returns:
        True if working directory appears to be in good state
    """
    try:
        issues_found = 0
        total_files = len(affected_files)
        
        for file_path in affected_files:
            try:
                if os.path.exists(file_path):
                    with open(file_path, 'r', encoding='utf-8', errors='ignore') as f:
                        content = f.read()
                    
                    # Check for common corruption patterns
                    if file_path.endswith(('.js', '.ts', '.jsx', '.tsx', '.svelte', '.vue')):
                        # Check for missing closing braces in JavaScript-like files
                        if content.strip() and not content.rstrip().endswith(('}', ';', ')', ']', '>')):
                            # This might be a truncation issue
                            lines = content.splitlines()
                            if lines:
                                last_line = lines[-1].strip()
                                if last_line and not last_line.endswith(('}', ';', ')', ']', '>')):
                                    print(f"Warning: File {file_path} may be missing closing syntax")
                                    issues_found += 1
                    
                    # Check for completely empty files that shouldn't be empty
                    if os.path.getsize(file_path) == 0 and 'delete' not in patch_content.lower():
                        print(f"Warning: File {file_path} is unexpectedly empty")
                        issues_found += 1
                        
            except Exception as e:
                print(f"Warning: Could not verify integrity of {file_path}: {e}")
                issues_found += 1
        
        if issues_found > 0:
            print(f"Working directory integrity check: {issues_found} potential issues found out of {total_files} files")
            return False
        else:
            print(f"Working directory integrity check: All {total_files} files appear valid")
            return True
            
    except Exception as e:
        print(f"Error during working directory integrity check: {e}")
        return False


def _relocate_and_apply_hunk(hunk: Hunk, base_diff: str) -> bool:
    """
    Apply a hunk using git's native patch application instead of direct file modification.

    Args:
        hunk: The hunk to apply
        base_diff: Original full diff for context

    Returns:
        True if successfully applied, False otherwise
    """
    try:
        # Generate valid patch for single hunk
        from .diff_parser import _create_valid_git_patch
        patch_content = _create_valid_git_patch([hunk], base_diff)

        if not patch_content.strip():
            print(f"Could not generate valid patch for hunk {hunk.id}")
            return False

        # Apply using git's native mechanism
        return _apply_patch_with_git(patch_content)

    except Exception as e:
        print(f"Error applying hunk {hunk.id}: {e}")
        return False


def _parse_hunk_content(hunk: Hunk) -> Tuple[List[str], List[str], List[str]]:
    """
    Parse hunk content to extract additions, deletions, and context lines.

    Args:
        hunk: The hunk to parse

    Returns:
        Tuple of (additions, deletions, context_lines)
    """
    additions = []
    deletions = []
    context_lines = []

    for line in hunk.content.split('\n')[1:]:  # Skip header
        # CRITICAL FIX: Don't filter out empty lines - they're significant in git diffs
        if line.startswith('+') and not line.startswith('+++'):
            additions.append(line[1:])  # Remove + prefix
        elif line.startswith('-') and not line.startswith('---'):
            deletions.append(line[1:])  # Remove - prefix
        elif line.startswith(' '):
            context_lines.append(line[1:])  # Remove space prefix
        elif not line:
            # Empty lines are context lines (preserve file structure)
            context_lines.append('')

    return additions, deletions, context_lines


def _is_file_operation_hunk(hunk: Hunk) -> bool:
    """
    Determine if a hunk represents a file creation or deletion operation.

    Args:
        hunk: The hunk to check

    Returns:
        True if this is a file operation hunk
    """
    # Check if this is a file deletion (hunk ID shows 0-0 range)
    if hunk.start_line == 0 and hunk.end_line == 0:
        return True

    # Check if file doesn't exist (creation case)
    if not os.path.exists(hunk.file_path):
        return True

    # Check hunk content for file operation markers
    hunk_content = hunk.content
    if 'new file mode' in hunk_content or 'deleted file mode' in hunk_content:
        return True

    # Check if hunk contains only additions (likely file creation)
    additions, deletions, _ = _parse_hunk_content(hunk)
    if not deletions and len(additions) > 5:  # Threshold for "new file"
        return True

    return False


def _apply_file_operation_hunk(hunk: Hunk, additions: list, deletions: list) -> bool:
    """
    Handle file creation and deletion operations specially.

    Args:
        hunk: The hunk representing file operation
        additions: Lines being added
        deletions: Lines being deleted

    Returns:
        True if operation succeeded
    """
    try:
        # File deletion case (including 0-0 range hunks)
        if hunk.start_line == 0 and hunk.end_line == 0:
            if os.path.exists(hunk.file_path):
                print(f"Deleting file: {hunk.file_path}")
                result = subprocess.run(['git', 'rm', hunk.file_path], capture_output=True, text=True)
                return result.returncode == 0
            else:
                print(f"File {hunk.file_path} already deleted, staging deletion")
                # File is already deleted from filesystem, but we need to stage the deletion
                result = subprocess.run(['git', 'add', hunk.file_path], capture_output=True, text=True)
                return result.returncode == 0

        # File creation case
        elif not os.path.exists(hunk.file_path) and additions:
            print(f"Creating new file: {hunk.file_path}")
            os.makedirs(os.path.dirname(hunk.file_path), exist_ok=True)
            with open(hunk.file_path, 'w', encoding='utf-8') as f:
                f.write('\n'.join(additions) + '\n' if additions else '')

            # Stage the new file
            result = subprocess.run(['git', 'add', hunk.file_path], capture_output=True, text=True)
            return result.returncode == 0

        # Check if this is actually a content modification that should be handled differently
        elif os.path.exists(hunk.file_path):
            # Fall back to git native patch application
            from .diff_parser import _create_valid_git_patch
            patch_content = _create_valid_git_patch([hunk], "")
            if patch_content.strip():
                return _apply_patch_with_git(patch_content)
            return False

        else:
            print(f"Unclear file operation for {hunk.file_path}")
            return False

    except Exception as e:
        print(f"Error handling file operation for {hunk.file_path}: {e}")
        return False




def _lines_match_fuzzy(file_lines: list, target_lines: list, threshold: float = 0.7) -> bool:
    """
    Check if lines match with fuzzy matching (handles whitespace, etc.).

    Args:
        file_lines: Lines from the current file
        target_lines: Lines we're trying to match
        threshold: Minimum match ratio (0.0 to 1.0)

    Returns:
        True if lines match above threshold
    """
    if len(file_lines) != len(target_lines):
        return False

    if not target_lines:
        return True

    matches = 0
    for file_line, target_line in zip(file_lines, target_lines):
        # Normalize whitespace and compare
        if file_line.strip() == target_line.strip():
            matches += 1
        # Also allow partial matches for similar content
        elif target_line.strip() in file_line.strip() or file_line.strip() in target_line.strip():
            matches += 0.5

    match_ratio = matches / len(target_lines)
    return match_ratio >= threshold




def apply_hunks_with_fallback(hunk_ids: List[str], hunks_by_id: Dict[str, Hunk], base_diff: str) -> bool:
    """
    Apply hunks using the hunk-based approach only.

    Args:
        hunk_ids: List of hunk IDs to apply
        hunks_by_id: Dictionary mapping hunk IDs to Hunk objects
        base_diff: Original full diff output

    Returns:
        True if successful, False otherwise
    """
    return apply_hunks(hunk_ids, hunks_by_id, base_diff)


def _apply_files_fallback(hunk_ids: List[str], hunks_by_id: Dict[str, Hunk]) -> bool:
    """
    Fallback method: stage entire files that contain the specified hunks.

    Args:
        hunk_ids: List of hunk IDs
        hunks_by_id: Dictionary mapping hunk IDs to Hunk objects

    Returns:
        True if successful, False otherwise
    """
    try:
        # Get unique file paths from the hunks
        file_paths = set()
        for hunk_id in hunk_ids:
            if hunk_id in hunks_by_id:
                file_paths.add(hunks_by_id[hunk_id].file_path)

        if not file_paths:
            return True

        # Stage the files
        for file_path in file_paths:
            result = subprocess.run(
                ['git', 'add', file_path],
                capture_output=True,
                text=True,
                check=False
            )

            if result.returncode != 0:
                print(f"Failed to stage file {file_path}: {result.stderr}")
                return False

        return True

    except Exception as e:
        print(f"Error in file fallback: {e}")
        return False


def reset_staging_area():
    """Reset the staging area to match HEAD."""
    try:
        result = subprocess.run(
            ['git', 'reset', 'HEAD'],
            capture_output=True,
            text=True,
            check=False
        )
        return result.returncode == 0
    except Exception:
        return False




def preview_hunk_application(hunk_ids: List[str], hunks_by_id: Dict[str, Hunk]) -> str:
    """
    Generate a preview of what would be applied when staging these hunks.

    Args:
        hunk_ids: List of hunk IDs to preview
        hunks_by_id: Dictionary mapping hunk IDs to Hunk objects

    Returns:
        String description of what would be applied
    """
    if not hunk_ids:
        return "No hunks selected."

    # Group hunks by file
    files_affected = {}
    for hunk_id in hunk_ids:
        if hunk_id in hunks_by_id:
            hunk = hunks_by_id[hunk_id]
            if hunk.file_path not in files_affected:
                files_affected[hunk.file_path] = []
            files_affected[hunk.file_path].append(hunk)

    # Generate preview
    preview_lines = []
    for file_path, hunks in files_affected.items():
        preview_lines.append(f"File: {file_path}")
        for hunk in sorted(hunks, key=lambda h: h.start_line):
            line_range = f"lines {hunk.start_line}-{hunk.end_line}"
            preview_lines.append(f"  - {hunk.id} ({line_range})")
        preview_lines.append("")

    return "\n".join(preview_lines)


def get_staging_status() -> Dict[str, List[str]]:
    """
    Get the current staging status.

    Returns:
        Dictionary with 'staged' and 'modified' file lists
    """
    try:
        result = subprocess.run(
            ['git', 'status', '--porcelain'],
            capture_output=True,
            text=True,
            check=True
        )

        staged = []
        modified = []

        for line in result.stdout.strip().split('\n'):
            if len(line) >= 2:
                status = line[:2]
                file_path = line[3:]

                if status[0] != ' ' and status[0] != '?':  # Staged changes
                    staged.append(file_path)
                if status[1] != ' ' and status[1] != '?':  # Modified changes
                    modified.append(file_path)

        return {'staged': staged, 'modified': modified}

    except Exception:
        return {'staged': [], 'modified': []}
