"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const fs = require("fs-extra");
const path = require("path");
const lib_1 = require("../lib");
jest.mock('child_process', () => ({
    spawnSync: jest.fn((_cmd, args) => {
        require('fs-extra').ensureDirSync(args[3]); // eslint-disable-line @typescript-eslint/no-require-imports
        return { error: null, status: 0 };
    })
}));
let stack;
const buildDir = path.join(__dirname, '.build');
beforeEach(() => {
    stack = new core_1.Stack();
    fs.removeSync(buildDir);
});
afterEach(() => {
    fs.removeSync(buildDir);
});
test('NodejsFunction', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1');
    new lib_1.NodejsFunction(stack, 'handler2');
    // THEN
    const { spawnSync } = require('child_process'); // eslint-disable-line @typescript-eslint/no-require-imports
    expect(spawnSync).toHaveBeenCalledWith('parcel', expect.arrayContaining([
        'build',
        expect.stringContaining('function.test.handler1.ts'),
        '--out-dir',
        expect.stringContaining(buildDir),
        '--out-file',
        'index.js',
        '--global',
        'handler',
        '--target',
        'node',
        '--bundle-node-modules',
        '--log-level',
        '2',
        '--no-minify',
        '--no-source-maps'
    ]));
    // Automatically finds .js handler file
    expect(spawnSync).toHaveBeenCalledWith('parcel', expect.arrayContaining([
        expect.stringContaining('function.test.handler2.js'),
    ]));
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
    });
});
test('throws when entry is not js/ts', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'handler.py'
    })).toThrow(/Only JavaScript or TypeScript entry files are supported/);
});
test('throws when entry does not exist', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'notfound.ts'
    })).toThrow(/Cannot find entry file at/);
});
test('throws when entry cannot be automatically found', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn')).toThrow(/Cannot find entry file./);
});
test('throws with the wrong runtime family', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'handler1', {
        runtime: aws_lambda_1.Runtime.PYTHON_3_8
    })).toThrow(/Only `NODEJS` runtimes are supported/);
});
//# sourceMappingURL=data:application/json;base64,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