/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.in;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.zip.GZIPInputStream;
import loci.common.DateTools;
import loci.common.Location;
import loci.common.RandomAccessInputStream;
import loci.formats.CoreMetadata;
import loci.formats.FormatException;
import loci.formats.FormatReader;
import loci.formats.FormatTools;
import loci.formats.MetadataTools;
import loci.formats.in.MetadataLevel;
import loci.formats.meta.MetadataStore;
import ome.units.UNITS;
import ome.units.quantity.Frequency;
import ome.units.quantity.Length;
import ome.units.quantity.Power;
import ome.units.quantity.Time;
import ome.xml.model.primitives.Timestamp;

public class ICSReader
extends FormatReader {
    public static final String NL = "\r\n";
    public static final String[] DATE_FORMATS = new String[]{"EEEE, MMMM dd, yyyy HH:mm:ss", "EEE dd MMMM yyyy HH:mm:ss", "EEE MMM dd HH:mm:ss yyyy", "EE dd MMM yyyy HH:mm:ss z", "HH:mm:ss dd\\MM\\yy"};
    private static final String[][] DOCUMENT_KEYS = new String[][]{{"date"}, {"document", "average"}, {"document"}, {"gmtdate"}, {"label"}};
    private static final String[][] HISTORY_KEYS = new String[][]{{"a\\d"}, {"acquisition", "acquire\\..*."}, {"acquisition", "laserbox\\..*."}, {"acquisition", "modules\\(.*."}, {"acquisition", "objective", "position"}, {"adc", "resolution"}, {"atd_hardware", "ver"}, {"atd_libraries", "ver"}, {"atd_microscopy", "ver"}, {"author"}, {"averagecount"}, {"averagequality"}, {"beam", "zoom"}, {"binning"}, {"bits/pixel"}, {"black", "level"}, {"black", "level\\*"}, {"black_level"}, {"camera", "manufacturer"}, {"camera", "model"}, {"camera"}, {"cfd", "holdoff"}, {"cfd", "limit", "high"}, {"cfd", "limit", "low"}, {"cfd", "zc", "level"}, {"channel\\*"}, {"collection", "time"}, {"cols"}, {"company"}, {"count", "increment"}, {"created", "on"}, {"creation", "date"}, {"cube", "descriptio"}, {"cube", "description"}, {"cube", "emm", "nm"}, {"cube", "exc", "nm"}, {"cube"}, {"date"}, {"dategmt"}, {"dead", "time", "comp"}, {"desc", "exc", "turret"}, {"desc", "emm", "turret"}, {"detector", "type"}, {"detector"}, {"dimensions"}, {"direct", "turret"}, {"dither", "range"}, {"dwell"}, {"excitationfwhm"}, {"experiment"}, {"experimenter"}, {"expon.", "order"}, {"exposure"}, {"exposure_time"}, {"ext", "latch", "delay"}, {"extents"}, {"filterset", "dichroic", "name"}, {"filterset", "dichroic", "nm"}, {"filterset", "emm", "name"}, {"filterset", "emm", "nm"}, {"filterset", "exc", "name"}, {"filterset", "exc", "nm"}, {"filterset"}, {"filter\\*"}, {"firmware"}, {"fret", "backgr\\d"}, {"frametime"}, {"gain"}, {"gain\\d"}, {"gain\\*"}, {"gamma"}, {"icsviewer", "ver"}, {"ht\\*"}, {"id"}, {"illumination", "mode", "laser"}, {"illumination", "mode"}, {"image", "bigendian"}, {"image", "bpp"}, {"image", "form"}, {"image", "physical_sizex"}, {"image", "physical_sizey"}, {"image", "sizex"}, {"image", "sizey"}, {"labels"}, {"lamp", "manufacturer"}, {"lamp", "model"}, {"laser", "firmware"}, {"laser", "manufacturer"}, {"laser", "model"}, {"laser", "power"}, {"laser", "rep", "rate"}, {"laser", "type"}, {"laser\\d", "intensity"}, {"laser\\d", "name"}, {"laser\\d", "wavelength"}, {"left"}, {"length"}, {"line", "compressio"}, {"line", "compression"}, {"linetime"}, {"magnification"}, {"manufacturer"}, {"max", "photon", "coun"}, {"max", "photon", "count"}, {"memory", "bank"}, {"metadata", "format", "ver"}, {"microscope", "built", "on"}, {"microscope", "name"}, {"microscope"}, {"mirror", "\\d"}, {"mode"}, {"noiseval"}, {"no.", "frames"}, {"objective", "detail"}, {"objective", "immersion"}, {"objective", "mag"}, {"objective", "magnification"}, {"objective", "na"}, {"objective", "type"}, {"objective", "workingdistance"}, {"objective"}, {"offsets"}, {"other", "text"}, {"passcount"}, {"pinhole"}, {"pixel", "clock"}, {"pixel", "time"}, {"pmt"}, {"polarity"}, {"region"}, {"rep", "period"}, {"repeat", "time"}, {"revision"}, {"routing", "chan", "x"}, {"routing", "chan", "y"}, {"rows"}, {"scan", "borders"}, {"scan", "flyback"}, {"scan", "pattern"}, {"scan", "pixels", "x"}, {"scan", "pixels", "y"}, {"scan", "pos", "x"}, {"scan", "pos", "y"}, {"scan", "resolution"}, {"scan", "speed"}, {"scan", "zoom"}, {"scanner", "lag"}, {"scanner", "pixel", "time"}, {"scanner", "resolution"}, {"scanner", "speed"}, {"scanner", "xshift"}, {"scanner", "yshift"}, {"scanner", "zoom"}, {"shutter\\d"}, {"shutter", "type"}, {"software"}, {"spectral", "bin_definition"}, {"spectral", "calibration", "gain", "data"}, {"spectral", "calibration", "gain", "mode"}, {"spectral", "calibration", "offset", "data"}, {"spectral", "calibration", "offset", "mode"}, {"spectral", "calibration", "sensitivity", "mode"}, {"spectral", "central_wavelength"}, {"spectral", "laser_shield"}, {"spectral", "laser_shield_width"}, {"spectral", "resolution"}, {"stage", "controller"}, {"stage", "firmware"}, {"stage", "manufacturer"}, {"stage", "model"}, {"stage", "pos"}, {"stage", "positionx"}, {"stage", "positiony"}, {"stage", "positionz"}, {"stage_xyzum"}, {"step\\d", "channel", "\\d"}, {"step\\d", "gain", "\\d"}, {"step\\d", "laser"}, {"step\\d", "name"}, {"step\\d", "pinhole"}, {"step\\d", "pmt", "ch", "\\d"}, {"step\\d", "shutter", "\\d"}, {"step\\d"}, {"stop", "on", "o'flow"}, {"stop", "on", "time"}, {"study"}, {"sync", "freq", "div"}, {"sync", "holdoff"}, {"sync"}, {"tac", "gain"}, {"tac", "limit", "low"}, {"tac", "offset"}, {"tac", "range"}, {"tau\\d"}, {"tcspc", "adc", "res"}, {"tcspc", "adc", "resolution"}, {"tcspc", "approx", "adc", "rate"}, {"tcspc", "approx", "cfd", "rate"}, {"tcspc", "approx", "tac", "rate"}, {"tcspc", "bh"}, {"tcspc", "cfd", "holdoff"}, {"tcspc", "cfd", "limit", "high"}, {"tcspc", "cfd", "limit", "low"}, {"tcspc", "cfd", "zc", "level"}, {"tcspc", "clock", "polarity"}, {"tcspc", "collection", "time"}, {"tcspc", "count", "increment"}, {"tcspc", "dead", "time", "enabled"}, {"tcspc", "delay"}, {"tcspc", "dither", "range"}, {"tcspc", "left", "border"}, {"tcspc", "line", "compression"}, {"tcspc", "mem", "offset"}, {"tcspc", "operation", "mode"}, {"tcspc", "overflow"}, {"tcspc", "pixel", "clk", "divider"}, {"tcspc", "pixel", "clock"}, {"tcspc", "routing", "x"}, {"tcspc", "routing", "y"}, {"tcspc", "scan", "x"}, {"tcspc", "scan", "y"}, {"tcspc", "sync", "divider"}, {"tcspc", "sync", "holdoff"}, {"tcspc", "sync", "rate"}, {"tcspc", "sync", "threshold"}, {"tcspc", "sync", "zc", "level"}, {"tcspc", "tac", "gain"}, {"tcspc", "tac", "limit", "high"}, {"tcspc", "tac", "limit", "low"}, {"tcspc", "tac", "offset"}, {"tcspc", "tac", "range"}, {"tcspc", "time", "window"}, {"tcspc", "top", "border"}, {"tcspc", "total", "frames"}, {"tcspc", "total", "time"}, {"tcspc", "trigger"}, {"tcspc", "x", "sync", "polarity"}, {"tcspc", "y", "sync", "polarity"}, {"text"}, {"time"}, {"title"}, {"top"}, {"transmission"}, {"trigger"}, {"type"}, {"units"}, {"version"}, {"wavelength\\*"}, {"x", "amplitude"}, {"y", "amplitude"}, {"x", "delay"}, {"y", "delay"}, {"x", "offset"}, {"y", "offset"}, {"z", "\\(background\\)"}};
    private static final String[][] LAYOUT_KEYS = new String[][]{{"coordinates"}, {"order"}, {"parameters"}, {"real_significant_bits"}, {"significant_bits"}, {"significant_channels"}, {"sizes"}};
    private static final String[][] PARAMETER_KEYS = new String[][]{{"allowedlinemodes"}, {"ch"}, {"higher_limit"}, {"labels"}, {"lower_limit"}, {"origin"}, {"range"}, {"sample_width", "ch"}, {"sample_width"}, {"scale"}, {"units", "adc-units", "channels"}, {"units", "adc-units", "nm"}, {"units"}};
    private static final String[][] REPRESENTATION_KEYS = new String[][]{{"byte_order"}, {"compression"}, {"format"}, {"sign"}};
    private static final String[][] SENSOR_KEYS = new String[][]{{"model"}, {"s_params", "channels"}, {"s_params", "exphotoncnt"}, {"s_params", "lambdaem"}, {"s_params", "lambdaex"}, {"s_params", "numaperture"}, {"s_params", "pinholeradius"}, {"s_params", "pinholespacing"}, {"s_params", "refinxlensmedium"}, {"s_params", "refinxmedium"}, {"s_params", "refrinxlensmedium"}, {"s_params", "refrinxmedium"}, {"type"}};
    private static final String[][] VIEW_KEYS = new String[][]{{"view", "color", "lib", "lut"}, {"view", "color", "count"}, {"view", "color", "doc", "scale"}, {"view", "color", "mode", "rgb", "set"}, {"view", "color", "mode", "rgb"}, {"view", "color", "schemes"}, {"view", "color", "view", "active"}, {"view", "color"}, {"view\\d", "alpha"}, {"view\\d", "alphastate"}, {"view\\d", "annotation", "annellipse"}, {"view\\d", "annotation", "annpoint"}, {"view\\d", "autoresize"}, {"view\\d", "axis"}, {"view\\d", "blacklevel"}, {"view\\d", "color"}, {"view\\d", "cursor"}, {"view\\d", "dimviewoption"}, {"view\\d", "gamma"}, {"view\\d", "ignoreaspect"}, {"view\\d", "intzoom"}, {"view\\d", "live"}, {"view\\d", "order"}, {"view\\d", "port"}, {"view\\d", "position"}, {"view\\d", "saturation"}, {"view\\d", "scale"}, {"view\\d", "showall"}, {"view\\d", "showcursor"}, {"view\\d", "showindex"}, {"view\\d", "size"}, {"view\\d", "synchronize"}, {"view\\d", "tile"}, {"view\\d", "useunits"}, {"view\\d", "zoom"}, {"view\\d"}, {"view"}};
    private static String[][] OTHER_KEYS = new String[][]{{"cube", "descriptio"}, {"cube", "description"}, {"image", "form"}, {"refinxlensmedium"}, {"refinxmedium"}, {"scil_type"}, {"source"}};
    private String currentIcsId;
    private String currentIdsId;
    private boolean versionTwo;
    private byte[] data;
    private long offset;
    private boolean gzip;
    private GZIPInputStream gzipStream;
    private boolean invertY;
    private boolean lifetime;
    private String labels;
    private Vector<Integer> channelLengths;
    private Vector<String> channelTypes;
    private int prevImage;
    private boolean hasInstrumentData = false;
    private boolean storedRGB = false;

    public ICSReader() {
        super("Image Cytometry Standard", new String[]{"ics", "ids"});
        this.domains = new String[]{"Light Microscopy", "Fluorescence-Lifetime Imaging", "Unknown"};
        this.hasCompanionFiles = true;
        this.datasetDescription = "One .ics and possibly one .ids with a similar name";
    }

    @Override
    public boolean isSingleFile(String id) throws FormatException, IOException {
        try (RandomAccessInputStream f = new RandomAccessInputStream(id);){
            boolean bl = f.readString(17).trim().equals("ics_version\t2.0");
            return bl;
        }
    }

    @Override
    public String[] getDomains() {
        FormatTools.assertId(this.currentId, true, 1);
        String[] domain = new String[]{"Graphics"};
        if (this.getModuloC().length() > 1) {
            domain[0] = "Fluorescence-Lifetime Imaging";
        } else if (this.hasInstrumentData) {
            domain[0] = "Light Microscopy";
        }
        return domain;
    }

    @Override
    public boolean isInterleaved(int subC) {
        FormatTools.assertId(this.currentId, true, 1);
        return subC == 0 && ((CoreMetadata)this.core.get((int)0)).interleaved;
    }

    @Override
    public int fileGroupOption(String id) throws FormatException, IOException {
        return 0;
    }

    @Override
    public byte[] openBytes(int no, byte[] buf, int x, int y, int w, int h2) throws FormatException, IOException {
        int sizeC;
        FormatTools.checkPlaneParameters(this, no, buf.length, x, y, w, h2);
        int bpp = FormatTools.getBytesPerPixel(this.getPixelType());
        int len = FormatTools.getPlaneSize(this);
        int rowLen = FormatTools.getPlaneSize(this, w, 1);
        if (this.invertY) {
            y = this.getSizeY() - y - h2;
        }
        int[] coordinates = this.getZCTCoords(no);
        int[] prevCoordinates = this.getZCTCoords(this.prevImage);
        if (!this.gzip) {
            this.in.seek(this.offset + (long)no * (long)len);
        } else {
            long toSkip = (long)(no - this.prevImage - 1) * (long)len;
            if (this.gzipStream == null || no <= this.prevImage) {
                FileInputStream fis = null;
                toSkip = (long)no * (long)len;
                if (this.versionTwo) {
                    fis = new FileInputStream(this.currentIcsId);
                    fis.skip(this.offset);
                } else {
                    fis = new FileInputStream(this.currentIdsId);
                    toSkip += this.offset;
                }
                try {
                    if (this.gzipStream != null) {
                        this.gzipStream.close();
                    }
                    this.gzipStream = new GZIPInputStream(fis);
                }
                catch (IOException e) {
                    this.gzip = false;
                    this.in.seek(this.offset + (long)no * (long)len);
                    this.gzipStream = null;
                }
            }
            if (this.gzipStream != null) {
                while (toSkip > 0L) {
                    toSkip -= this.gzipStream.skip(toSkip);
                }
                this.data = new byte[len * (this.storedRGB ? this.getSizeC() : 1)];
                for (int toRead = this.data.length; toRead > 0; toRead -= this.gzipStream.read(this.data, this.data.length - toRead, toRead)) {
                }
            }
        }
        int n = sizeC = this.lifetime ? 1 : this.getSizeC();
        if (!this.isRGB() && this.channelLengths.size() == 1 && this.storedRGB) {
            this.in.seek(this.offset + (long)len * (long)this.getIndex(coordinates[0], 0, coordinates[2]));
            if (!this.gzip && this.data == null) {
                this.data = new byte[len * this.getSizeC()];
                this.in.read(this.data);
            } else if (!(this.gzip || coordinates[0] == prevCoordinates[0] && coordinates[2] == prevCoordinates[2])) {
                this.in.read(this.data);
            }
            for (int row = y; row < h2 + y; ++row) {
                for (int col = x; col < w + x; ++col) {
                    int src = bpp * (no % this.getSizeC() + sizeC * (row * this.getSizeX() + col));
                    int dest = bpp * ((row - y) * w + (col - x));
                    System.arraycopy(this.data, src, buf, dest, bpp);
                }
            }
        } else if (this.gzip) {
            try (RandomAccessInputStream s2 = new RandomAccessInputStream(this.data);){
                this.readPlane(s2, x, y, w, h2, buf);
            }
        } else {
            this.readPlane(this.in, x, y, w, h2, buf);
        }
        if (this.invertY) {
            byte[] row = new byte[rowLen];
            for (int r = 0; r < h2 / 2; ++r) {
                int topOffset = r * rowLen;
                int bottomOffset = (h2 - r - 1) * rowLen;
                System.arraycopy(buf, topOffset, row, 0, rowLen);
                System.arraycopy(buf, bottomOffset, buf, topOffset, rowLen);
                System.arraycopy(row, 0, buf, bottomOffset, rowLen);
            }
        }
        this.prevImage = no;
        return buf;
    }

    @Override
    public String[] getSeriesUsedFiles(boolean noPixels) {
        String[] stringArray;
        FormatTools.assertId(this.currentId, true, 1);
        if (this.versionTwo) {
            String[] stringArray2;
            if (noPixels) {
                stringArray2 = null;
            } else {
                String[] stringArray3 = new String[1];
                stringArray2 = stringArray3;
                stringArray3[0] = this.currentIcsId;
            }
            return stringArray2;
        }
        if (noPixels) {
            String[] stringArray4 = new String[1];
            stringArray = stringArray4;
            stringArray4[0] = this.currentIcsId;
        } else {
            String[] stringArray5 = new String[2];
            stringArray5[0] = this.currentIcsId;
            stringArray = stringArray5;
            stringArray5[1] = this.currentIdsId;
        }
        return stringArray;
    }

    @Override
    public void close(boolean fileOnly) throws IOException {
        super.close(fileOnly);
        if (!fileOnly) {
            this.currentIcsId = null;
            this.currentIdsId = null;
            this.data = null;
            this.versionTwo = false;
            this.gzip = false;
            this.invertY = false;
            this.lifetime = false;
            this.prevImage = 0;
            this.hasInstrumentData = false;
            this.storedRGB = false;
            if (this.gzipStream != null) {
                this.gzipStream.close();
            }
            this.gzipStream = null;
        }
    }

    @Override
    public void reopenFile() throws IOException {
        if (this.in != null) {
            this.in.close();
        }
        this.in = this.versionTwo ? new RandomAccessInputStream(this.currentIcsId) : new RandomAccessInputStream(this.currentIdsId);
        this.in.order(this.isLittleEndian());
    }

    @Override
    protected void initFile(String id) throws FormatException, IOException {
        char[] c;
        String ext;
        super.initFile(id);
        LOGGER.info("Finding companion file");
        String icsId = id;
        String idsId = id;
        int dot = id.lastIndexOf(".");
        String string = ext = dot < 0 ? "" : id.substring(dot + 1).toLowerCase();
        if (ext.equals("ics")) {
            c = idsId.toCharArray();
            int n = c.length - 2;
            c[n] = (char)(c[n] + '\u0001');
            idsId = new String(c);
        } else if (ext.equals("ids")) {
            c = icsId.toCharArray();
            int n = c.length - 2;
            c[n] = (char)(c[n] - '\u0001');
            icsId = new String(c);
        }
        if (icsId == null) {
            throw new FormatException("No ICS file found.");
        }
        Location icsFile = new Location(icsId);
        if (!icsFile.exists()) {
            throw new FormatException("ICS file not found.");
        }
        LOGGER.info("Checking file version");
        try (RandomAccessInputStream f = new RandomAccessInputStream(icsId);){
            if (f.readString(17).trim().equals("ics_version\t2.0")) {
                this.in = new RandomAccessInputStream(icsId);
                this.versionTwo = true;
            } else {
                if (idsId == null) {
                    throw new FormatException("No IDS file found.");
                }
                Location idsFile = new Location(idsId);
                if (!idsFile.exists()) {
                    throw new FormatException("IDS file not found.");
                }
                this.currentIdsId = idsId;
                this.in = new RandomAccessInputStream(this.currentIdsId);
            }
        }
        this.currentIcsId = icsId;
        LOGGER.info("Reading metadata");
        CoreMetadata m3 = (CoreMetadata)this.core.get(0);
        Double[] scales = null;
        Double[] timestamps = null;
        String[] units = null;
        String[] axes = null;
        int[] axisLengths = null;
        String byteOrder = null;
        String rFormat = null;
        String compression = null;
        Double[] emWaves = null;
        Double[] exWaves = null;
        Length[] stagePos = null;
        String imageName = null;
        String date = null;
        String description = null;
        Double magnification = null;
        Double lensNA = null;
        Double workingDistance = null;
        String objectiveModel = null;
        String immersion = null;
        String lastName = null;
        StringBuilder textBlock = new StringBuilder();
        String laserModel = null;
        String laserManufacturer = null;
        Double laserPower = null;
        Double laserRepetitionRate = null;
        String detectorManufacturer = null;
        String detectorModel = null;
        String microscopeModel = null;
        String microscopeManufacturer = null;
        String experimentType = null;
        Time exposureTime = null;
        String filterSetModel = null;
        String dichroicModel = null;
        String excitationModel = null;
        String emissionModel = null;
        Hashtable<Integer, Double> gains = new Hashtable<Integer, Double>();
        Hashtable<Integer, Double> pinholes = new Hashtable<Integer, Double>();
        Hashtable<Integer, Double> wavelengths = new Hashtable<Integer, Double>();
        Hashtable<Integer, String> channelNames = new Hashtable<Integer, String>();
        double[] sizes = null;
        boolean signed = false;
        try (RandomAccessInputStream reader = new RandomAccessInputStream(icsId);){
            reader.seek(0L);
            reader.readString(NL);
            String line = reader.readString(NL);
            while (line != null && !line.trim().equals("end") && reader.getFilePointer() < reader.length() - 1L) {
                if ((line = line.trim()).length() > 0) {
                    String value;
                    String key;
                    String[] tokens = this.tokenize(line);
                    Iterator<String> token0 = tokens[0].toLowerCase();
                    String[] keyValue = null;
                    if (((String)((Object)token0)).equals("ics_version")) {
                        String value2 = this.concatenateTokens(tokens, 1, tokens.length);
                        this.addGlobalMeta((String)((Object)token0), value2);
                    } else if (((String)((Object)token0)).equals("filename")) {
                        imageName = this.concatenateTokens(tokens, 1, tokens.length);
                        this.addGlobalMeta((String)((Object)token0), imageName);
                    } else if (((String)((Object)token0)).equals("layout")) {
                        keyValue = this.findKeyValue(tokens, LAYOUT_KEYS);
                        key = keyValue[0];
                        value = keyValue[1];
                        this.addGlobalMeta(key, value);
                        if (key.equalsIgnoreCase("layout sizes")) {
                            StringTokenizer t = new StringTokenizer(value);
                            axisLengths = new int[t.countTokens()];
                            for (int n = 0; n < axisLengths.length; ++n) {
                                try {
                                    axisLengths[n] = Integer.parseInt(t.nextToken().trim());
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse axis length", e);
                                }
                            }
                        } else if (key.equalsIgnoreCase("layout order")) {
                            StringTokenizer t = new StringTokenizer(value);
                            axes = new String[t.countTokens()];
                            for (int n = 0; n < axes.length; ++n) {
                                axes[n] = t.nextToken().trim();
                            }
                        } else if (key.equalsIgnoreCase("layout significant_bits")) {
                            m3.bitsPerPixel = Integer.parseInt(value);
                        }
                    } else if (((String)((Object)token0)).equals("representation")) {
                        keyValue = this.findKeyValue(tokens, REPRESENTATION_KEYS);
                        key = keyValue[0];
                        value = keyValue[1];
                        this.addGlobalMeta(key, value);
                        if (key.equalsIgnoreCase("representation byte_order")) {
                            byteOrder = value;
                        } else if (key.equalsIgnoreCase("representation format")) {
                            rFormat = value;
                        } else if (key.equalsIgnoreCase("representation compression")) {
                            compression = value;
                        } else if (key.equalsIgnoreCase("representation sign")) {
                            signed = value.equals("signed");
                        }
                    } else if (((String)((Object)token0)).equals("parameter")) {
                        keyValue = this.findKeyValue(tokens, PARAMETER_KEYS);
                        key = keyValue[0];
                        value = keyValue[1];
                        this.addGlobalMeta(key, value);
                        if (key.equalsIgnoreCase("parameter scale")) {
                            scales = this.splitDoubles(value);
                        } else if (key.equalsIgnoreCase("parameter t")) {
                            timestamps = this.splitDoubles(value);
                        } else if (key.equalsIgnoreCase("parameter units")) {
                            units = value.split("\\s+");
                        }
                        if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM && key.equalsIgnoreCase("parameter ch")) {
                            String[] names = value.split(" ");
                            for (int n = 0; n < names.length; ++n) {
                                channelNames.put(new Integer(n), names[n].trim());
                            }
                        }
                    } else if (((String)((Object)token0)).equals("history")) {
                        keyValue = this.findKeyValue(tokens, HISTORY_KEYS);
                        key = keyValue[0];
                        value = keyValue[1];
                        this.addGlobalMeta(key, value);
                        Double doubleValue = null;
                        try {
                            doubleValue = new Double(value);
                        }
                        catch (NumberFormatException e) {
                            LOGGER.debug("Could not parse double value '{}'", (Object)value, (Object)e);
                        }
                        if (key.equalsIgnoreCase("history software") && value.indexOf("SVI") != -1) {
                            this.invertY = true;
                        } else if (key.equalsIgnoreCase("history date") || key.equalsIgnoreCase("history created on")) {
                            if (value.indexOf(32) > 0) {
                                date = value.substring(0, value.lastIndexOf(" "));
                                date = DateTools.formatDate(date, DATE_FORMATS);
                            }
                        } else if (key.equalsIgnoreCase("history creation date")) {
                            date = DateTools.formatDate(value, DATE_FORMATS);
                        } else if (key.equalsIgnoreCase("history type")) {
                            if (value.equalsIgnoreCase("time resolved") || value.equalsIgnoreCase("FluorescenceLifetime")) {
                                this.lifetime = true;
                            }
                            experimentType = value;
                        } else if (key.equalsIgnoreCase("history labels")) {
                            this.labels = value;
                        } else if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
                            if (key.equalsIgnoreCase("history") || key.equalsIgnoreCase("history text")) {
                                textBlock.append(value);
                                textBlock.append("\n");
                                this.metadata.remove(key);
                            } else if (key.startsWith("history gain")) {
                                Integer n = 0;
                                try {
                                    n = new Integer(key.substring(12).trim());
                                    n = new Integer(n - 1);
                                }
                                catch (NumberFormatException e) {
                                    // empty catch block
                                }
                                if (doubleValue != null) {
                                    gains.put(n, doubleValue);
                                }
                            } else if (key.startsWith("history laser") && key.endsWith("wavelength")) {
                                int laser = Integer.parseInt(key.substring(13, key.indexOf(" ", 13))) - 1;
                                value = value.replaceAll("nm", "").trim();
                                try {
                                    wavelengths.put(new Integer(laser), new Double(value));
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse wavelength", e);
                                }
                            } else if (key.equalsIgnoreCase("history Wavelength*")) {
                                String[] waves = value.split(" ");
                                for (int i = 0; i < waves.length; ++i) {
                                    wavelengths.put(new Integer(i), new Double(waves[i]));
                                }
                            } else if (key.equalsIgnoreCase("history laser manufacturer")) {
                                laserManufacturer = value;
                            } else if (key.equalsIgnoreCase("history laser model")) {
                                laserModel = value;
                            } else if (key.equalsIgnoreCase("history laser power")) {
                                try {
                                    laserPower = new Double(value);
                                }
                                catch (NumberFormatException waves) {}
                            } else if (key.equalsIgnoreCase("history laser rep rate")) {
                                String repRate = value;
                                if (repRate.indexOf(32) != -1) {
                                    repRate = repRate.substring(0, repRate.lastIndexOf(" "));
                                }
                                laserRepetitionRate = new Double(repRate);
                            } else if (key.equalsIgnoreCase("history objective type") || key.equalsIgnoreCase("history objective")) {
                                objectiveModel = value;
                            } else if (key.equalsIgnoreCase("history objective immersion")) {
                                immersion = value;
                            } else if (key.equalsIgnoreCase("history objective NA")) {
                                lensNA = doubleValue;
                            } else if (key.equalsIgnoreCase("history objective WorkingDistance")) {
                                workingDistance = doubleValue;
                            } else if (key.equalsIgnoreCase("history objective magnification") || key.equalsIgnoreCase("history objective mag")) {
                                magnification = doubleValue;
                            } else if (key.equalsIgnoreCase("history camera manufacturer")) {
                                detectorManufacturer = value;
                            } else if (key.equalsIgnoreCase("history camera model")) {
                                detectorModel = value;
                            } else if (key.equalsIgnoreCase("history author") || key.equalsIgnoreCase("history experimenter")) {
                                lastName = value;
                            } else if (key.equalsIgnoreCase("history extents")) {
                                String[] lengths = value.split(" ");
                                sizes = new double[lengths.length];
                                for (int n = 0; n < sizes.length; ++n) {
                                    try {
                                        sizes[n] = Double.parseDouble(lengths[n].trim());
                                        continue;
                                    }
                                    catch (NumberFormatException e) {
                                        LOGGER.debug("Could not parse axis length", e);
                                    }
                                }
                            } else if (key.equalsIgnoreCase("history stage_xyzum")) {
                                String[] positions = value.split(" ");
                                stagePos = new Length[positions.length];
                                for (int n = 0; n < stagePos.length; ++n) {
                                    try {
                                        Double number = Double.valueOf(positions[n]);
                                        stagePos[n] = new Length(number, UNITS.REFERENCEFRAME);
                                        continue;
                                    }
                                    catch (NumberFormatException e) {
                                        LOGGER.debug("Could not parse stage position", e);
                                    }
                                }
                            } else if (key.equalsIgnoreCase("history stage positionx")) {
                                if (stagePos == null) {
                                    stagePos = new Length[3];
                                }
                                Double number = Double.valueOf(value);
                                stagePos[0] = new Length(number, UNITS.REFERENCEFRAME);
                            } else if (key.equalsIgnoreCase("history stage positiony")) {
                                if (stagePos == null) {
                                    stagePos = new Length[3];
                                }
                                Double number = Double.valueOf(value);
                                stagePos[1] = new Length(number, UNITS.REFERENCEFRAME);
                            } else if (key.equalsIgnoreCase("history stage positionz")) {
                                if (stagePos == null) {
                                    stagePos = new Length[3];
                                }
                                Double number = Double.valueOf(value);
                                stagePos[2] = new Length(number, UNITS.REFERENCEFRAME);
                            } else if (key.equalsIgnoreCase("history other text")) {
                                description = value;
                            } else if (key.startsWith("history step") && key.endsWith("name")) {
                                Integer n = new Integer(key.substring(12, key.indexOf(" ", 12)));
                                channelNames.put(n, value);
                            } else if (key.equalsIgnoreCase("history cube")) {
                                channelNames.put(new Integer(channelNames.size()), value);
                            } else if (key.equalsIgnoreCase("history cube emm nm")) {
                                if (emWaves == null) {
                                    emWaves = new Double[]{new Double(value.split(" ")[1].trim())};
                                }
                            } else if (key.equalsIgnoreCase("history cube exc nm")) {
                                if (exWaves == null) {
                                    exWaves = new Double[]{new Double(value.split(" ")[1].trim())};
                                }
                            } else if (key.equalsIgnoreCase("history microscope")) {
                                microscopeModel = value;
                            } else if (key.equalsIgnoreCase("history manufacturer")) {
                                microscopeManufacturer = value;
                            } else if (key.equalsIgnoreCase("history Exposure")) {
                                Double expDouble;
                                String expTime = value;
                                if (expTime.indexOf(32) != -1) {
                                    expTime = expTime.substring(0, expTime.indexOf(32));
                                }
                                if ((expDouble = new Double(expTime)) != null) {
                                    exposureTime = new Time(expDouble, UNITS.SECOND);
                                }
                            } else if (key.equalsIgnoreCase("history filterset")) {
                                filterSetModel = value;
                            } else if (key.equalsIgnoreCase("history filterset dichroic name")) {
                                dichroicModel = value;
                            } else if (key.equalsIgnoreCase("history filterset exc name")) {
                                excitationModel = value;
                            } else if (key.equalsIgnoreCase("history filterset emm name")) {
                                emissionModel = value;
                            }
                        }
                    } else if (((String)((Object)token0)).equals("document")) {
                        keyValue = this.findKeyValue(tokens, DOCUMENT_KEYS);
                        key = keyValue[0];
                        value = keyValue[1];
                        this.addGlobalMeta(key, value);
                    } else if (((String)((Object)token0)).equals("sensor")) {
                        keyValue = this.findKeyValue(tokens, SENSOR_KEYS);
                        key = keyValue[0];
                        value = keyValue[1];
                        this.addGlobalMeta(key, value);
                        if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
                            if (key.equalsIgnoreCase("sensor s_params LambdaEm")) {
                                String[] waves = value.split(" ");
                                emWaves = new Double[waves.length];
                                for (int n = 0; n < emWaves.length; ++n) {
                                    try {
                                        emWaves[n] = new Double(Double.parseDouble(waves[n]));
                                        continue;
                                    }
                                    catch (NumberFormatException e) {
                                        LOGGER.debug("Could not parse emission wavelength", e);
                                    }
                                }
                            } else if (key.equalsIgnoreCase("sensor s_params LambdaEx")) {
                                String[] waves = value.split(" ");
                                exWaves = new Double[waves.length];
                                for (int n = 0; n < exWaves.length; ++n) {
                                    try {
                                        exWaves[n] = new Double(Double.parseDouble(waves[n]));
                                        continue;
                                    }
                                    catch (NumberFormatException e) {
                                        LOGGER.debug("Could not parse excitation wavelength", e);
                                    }
                                }
                            } else if (key.equalsIgnoreCase("sensor s_params PinholeRadius")) {
                                String[] pins = value.split(" ");
                                int channel = 0;
                                for (int n = 0; n < pins.length; ++n) {
                                    if (pins[n].trim().equals("")) continue;
                                    try {
                                        pinholes.put(new Integer(channel++), new Double(pins[n]));
                                        continue;
                                    }
                                    catch (NumberFormatException e) {
                                        LOGGER.debug("Could not parse pinhole", e);
                                    }
                                }
                            }
                        }
                    } else if (((String)((Object)token0)).equals("view")) {
                        keyValue = this.findKeyValue(tokens, VIEW_KEYS);
                        key = keyValue[0];
                        value = keyValue[1];
                        if (key.equalsIgnoreCase("view view color lib lut")) {
                            int redIndex = value.toLowerCase().lastIndexOf("red");
                            int greenIndex = value.toLowerCase().lastIndexOf("green");
                            int blueIndex = value.toLowerCase().lastIndexOf("blue");
                            int index = redIndex > 0 && redIndex > greenIndex && redIndex > blueIndex ? redIndex + "red".length() : (greenIndex > 0 && greenIndex > redIndex && greenIndex > blueIndex ? greenIndex + "green".length() : (blueIndex > 0 && blueIndex > redIndex && blueIndex > greenIndex ? blueIndex + "blue".length() : value.indexOf(32)));
                            if (index > 0) {
                                key = key + ' ' + value.substring(0, index);
                                value = value.substring(index + 1);
                            }
                        } else if (key.equalsIgnoreCase("view view color mode rgb set")) {
                            int index = value.toLowerCase().lastIndexOf("colors");
                            index = index > 0 ? (index += "colors".length()) : value.indexOf(32);
                            if (index > 0) {
                                key = key + ' ' + value.substring(0, index);
                                value = value.substring(index + 1);
                            }
                        }
                        this.addGlobalMeta(key, value);
                    } else {
                        LOGGER.debug("Unknown category " + (String)((Object)token0));
                    }
                }
                line = reader.readString(NL);
            }
        }
        this.hasInstrumentData = emWaves != null || exWaves != null || lensNA != null || stagePos != null || magnification != null || workingDistance != null || objectiveModel != null || immersion != null;
        this.addGlobalMeta("history text", textBlock.toString());
        LOGGER.info("Populating core metadata");
        m3.rgb = false;
        m3.dimensionOrder = "XY";
        this.channelLengths = new Vector();
        this.channelTypes = new Vector();
        int bitsPerPixel = 0;
        for (int i = 0; i < axes.length && i < axisLengths.length; ++i) {
            if (axes[i].equals("bits")) {
                bitsPerPixel = axisLengths[i];
                while (bitsPerPixel % 8 != 0) {
                    ++bitsPerPixel;
                }
                if (bitsPerPixel != 24 && bitsPerPixel != 48) continue;
                bitsPerPixel /= 3;
                continue;
            }
            if (axes[i].equals("x")) {
                m3.sizeX = axisLengths[i];
                continue;
            }
            if (axes[i].equals("y")) {
                m3.sizeY = axisLengths[i];
                continue;
            }
            if (axes[i].equals("z")) {
                m3.sizeZ = axisLengths[i];
                if (this.getDimensionOrder().indexOf(90) != -1) continue;
                m3.dimensionOrder = m3.dimensionOrder + 'Z';
                continue;
            }
            if (axes[i].equals("t")) {
                m3.sizeT = this.getSizeT() == 0 ? axisLengths[i] : (m3.sizeT *= axisLengths[i]);
                if (this.getDimensionOrder().indexOf(84) != -1) continue;
                m3.dimensionOrder = m3.dimensionOrder + 'T';
                continue;
            }
            m3.sizeC = m3.sizeC == 0 ? axisLengths[i] : (m3.sizeC *= axisLengths[i]);
            this.channelLengths.add(new Integer(axisLengths[i]));
            this.storedRGB = this.getSizeX() == 0;
            boolean bl = m3.rgb = this.getSizeX() == 0 && this.getSizeC() <= 4 && this.getSizeC() > 1;
            if (this.getDimensionOrder().indexOf(67) == -1) {
                m3.dimensionOrder = m3.dimensionOrder + 'C';
            }
            if (axes[i].startsWith("c")) {
                this.channelTypes.add("Channel");
                continue;
            }
            if (axes[i].equals("p")) {
                this.channelTypes.add("Phase");
                continue;
            }
            if (axes[i].equals("f")) {
                this.channelTypes.add("Frequency");
                continue;
            }
            this.channelTypes.add("");
        }
        if (this.channelLengths.isEmpty()) {
            this.channelLengths.add(1);
            this.channelTypes.add("Channel");
        }
        if (this.isRGB() && emWaves != null && emWaves.length == this.getSizeC()) {
            m3.rgb = false;
            this.storedRGB = true;
        }
        m3.dimensionOrder = MetadataTools.makeSaneDimensionOrder(this.getDimensionOrder());
        if (this.getSizeZ() == 0) {
            m3.sizeZ = 1;
        }
        if (this.getSizeC() == 0) {
            m3.sizeC = 1;
        }
        if (this.getSizeT() == 0) {
            m3.sizeT = 1;
        }
        if (this.channelLengths.size() > 0) {
            int clen0 = this.channelLengths.get(0);
            String ctype0 = this.channelTypes.get(0);
            boolean same = true;
            for (Integer len : this.channelLengths) {
                if (clen0 == len) continue;
                same = false;
            }
            for (String type : this.channelTypes) {
                if (ctype0.equals(type)) continue;
                same = false;
            }
            if (same) {
                m3.moduloC.type = ctype0;
                if ("Lifetime".equals(ctype0)) {
                    m3.moduloC.parentType = "Spectra";
                }
                m3.moduloC.typeDescription = "TCSPC";
                m3.moduloC.start = 0.0;
                m3.moduloC.step = 1.0;
                m3.moduloC.end = clen0 - 1;
            }
        }
        m3.interleaved = this.isRGB();
        m3.indexed = false;
        m3.falseColor = false;
        m3.metadataComplete = true;
        m3.littleEndian = true;
        if (this.lifetime && this.labels != null) {
            int binCount = 0;
            String newOrder = null;
            if (this.labels.equalsIgnoreCase("t x y")) {
                newOrder = "XYCZT";
                m3.interleaved = true;
                binCount = m3.sizeX;
                m3.sizeX = m3.sizeY;
                m3.sizeY = m3.sizeZ;
                m3.sizeZ = 1;
            } else if (this.labels.equalsIgnoreCase("x y t")) {
                newOrder = "XYCZT";
                binCount = m3.sizeZ;
                m3.sizeZ = 1;
            } else {
                LOGGER.debug("Lifetime data, unexpected 'history labels' " + this.labels);
            }
            if (newOrder != null) {
                m3.dimensionOrder = newOrder;
                m3.sizeC = binCount;
                m3.moduloC.parentType = "Lifetime";
            }
        }
        m3.imageCount = this.getSizeZ() * this.getSizeT();
        if (!this.isRGB()) {
            m3.imageCount *= this.getSizeC();
        }
        if (byteOrder != null) {
            String firstByte = byteOrder.split(" ")[0];
            int first = Integer.parseInt(firstByte);
            m3.littleEndian = rFormat.equals("real") ? first == 1 : first != 1;
        }
        boolean bl = this.gzip = compression == null ? false : compression.equals("gzip");
        if (this.versionTwo) {
            String s2 = this.in.readString(NL);
            while (!s2.trim().equals("end")) {
                s2 = this.in.readString(NL);
            }
        }
        this.offset = this.in.getFilePointer();
        int bytes = bitsPerPixel / 8;
        if (bitsPerPixel < 32) {
            m3.littleEndian = !this.isLittleEndian();
        }
        boolean fp = rFormat.equals("real");
        m3.pixelType = FormatTools.pixelTypeFromBytes(bytes, signed, fp);
        LOGGER.info("Populating OME metadata");
        MetadataStore store = this.makeFilterMetadata();
        MetadataTools.populatePixels(store, this, true);
        imageName = imageName.replace('/', File.separatorChar);
        imageName = imageName.replace('\\', File.separatorChar);
        imageName = imageName.substring(imageName.lastIndexOf(File.separator) + 1);
        store.setImageName(imageName, 0);
        if (date != null) {
            store.setImageAcquisitionDate(new Timestamp(date), 0);
        }
        if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
            store.setImageDescription(description, 0);
            String instrumentID = MetadataTools.createLSID("Instrument", 0);
            store.setInstrumentID(instrumentID, 0);
            store.setMicroscopeModel(microscopeModel, 0);
            store.setMicroscopeManufacturer(microscopeManufacturer, 0);
            store.setImageInstrumentRef(instrumentID, 0);
            store.setExperimentID(MetadataTools.createLSID("Experiment", 0), 0);
            store.setExperimentType(MetadataTools.getExperimentType(experimentType), 0);
            if (scales != null) {
                if (units != null && units.length == scales.length - 1) {
                    ArrayList<String> realUnits = new ArrayList<String>();
                    int unitIndex = 0;
                    for (int i = 0; i < axes.length; ++i) {
                        if (axes[i].toLowerCase().equals("ch") || unitIndex >= units.length) {
                            realUnits.add("nm");
                            continue;
                        }
                        realUnits.add(units[unitIndex++]);
                    }
                    units = realUnits.toArray(new String[realUnits.size()]);
                }
                for (int i = 0; i < scales.length; ++i) {
                    String unit;
                    Double scale = scales[i];
                    if (scale == null) continue;
                    String axis = axes != null && axes.length > i ? axes[i] : "";
                    String string2 = unit = units != null && units.length > i ? units[i] : "";
                    if (axis.equals("x")) {
                        Length x;
                        if (!this.checkUnit(unit, "um", "microns", "micrometers") || (x = FormatTools.getPhysicalSizeX(scale)) == null) continue;
                        store.setPixelsPhysicalSizeX(x, 0);
                        continue;
                    }
                    if (axis.equals("y")) {
                        Length y;
                        if (!this.checkUnit(unit, "um", "microns", "micrometers") || (y = FormatTools.getPhysicalSizeY(scale)) == null) continue;
                        store.setPixelsPhysicalSizeY(y, 0);
                        continue;
                    }
                    if (axis.equals("z")) {
                        Length z;
                        if (!this.checkUnit(unit, "um", "microns", "micrometers") || (z = FormatTools.getPhysicalSizeZ(scale)) == null) continue;
                        store.setPixelsPhysicalSizeZ(z, 0);
                        continue;
                    }
                    if (!axis.equals("t") || scale == null) continue;
                    if (this.checkUnit(unit, "ms")) {
                        store.setPixelsTimeIncrement(new Time(scale, UNITS.MILLISECOND), 0);
                        continue;
                    }
                    if (!this.checkUnit(unit, "seconds") && !this.checkUnit(unit, "s")) continue;
                    store.setPixelsTimeIncrement(new Time(scale, UNITS.SECOND), 0);
                }
            } else if (sizes != null) {
                Length x;
                if (sizes.length > 0 && (x = FormatTools.getPhysicalSizeX(sizes[0])) != null) {
                    store.setPixelsPhysicalSizeX(x, 0);
                }
                if (sizes.length > 1) {
                    sizes[1] = sizes[1] / (double)this.getSizeY();
                    Length y = FormatTools.getPhysicalSizeY(sizes[1]);
                    if (y != null) {
                        store.setPixelsPhysicalSizeY(y, 0);
                    }
                }
            }
            if (timestamps != null) {
                for (int t = 0; t < timestamps.length && t < this.getSizeT(); ++t) {
                    Time deltaT;
                    if (timestamps[t] == null || Double.isNaN((deltaT = new Time(timestamps[t], UNITS.SECOND)).value().doubleValue())) continue;
                    for (int z = 0; z < this.getSizeZ(); ++z) {
                        for (int c2 = 0; c2 < this.getEffectiveSizeC(); ++c2) {
                            int index = this.getIndex(z, c2, t);
                            store.setPlaneDeltaT(deltaT, 0, index);
                        }
                    }
                }
            }
            for (int i = 0; i < this.getEffectiveSizeC(); ++i) {
                Length ex;
                Length em;
                if (channelNames.containsKey(i)) {
                    store.setChannelName((String)channelNames.get(i), 0, i);
                }
                if (pinholes.containsKey(i)) {
                    store.setChannelPinholeSize(new Length((Number)pinholes.get(i), UNITS.MICROMETER), 0, i);
                }
                if (emWaves != null && i < emWaves.length && (em = FormatTools.getEmissionWavelength(emWaves[i])) != null) {
                    store.setChannelEmissionWavelength(em, 0, i);
                }
                if (exWaves == null || i >= exWaves.length || (ex = FormatTools.getExcitationWavelength(exWaves[i])) == null) continue;
                store.setChannelExcitationWavelength(ex, 0, i);
            }
            Object[] lasers = wavelengths.keySet().toArray(new Integer[0]);
            Arrays.sort(lasers);
            for (int i = 0; i < lasers.length; ++i) {
                Frequency theLaserRepetitionRate;
                store.setLaserID(MetadataTools.createLSID("LightSource", 0, i), 0, i);
                Length wave = FormatTools.getWavelength((Double)wavelengths.get(lasers[i]));
                if (wave != null) {
                    store.setLaserWavelength(wave, 0, i);
                }
                store.setLaserType(MetadataTools.getLaserType("Other"), 0, i);
                store.setLaserLaserMedium(MetadataTools.getLaserMedium("Other"), 0, i);
                store.setLaserManufacturer(laserManufacturer, 0, i);
                store.setLaserModel(laserModel, 0, i);
                Power theLaserPower = FormatTools.createPower(laserPower, UNITS.MILLIWATT);
                if (theLaserPower != null) {
                    store.setLaserPower(theLaserPower, 0, i);
                }
                if ((theLaserRepetitionRate = FormatTools.createFrequency(laserRepetitionRate, UNITS.HERTZ)) == null) continue;
                store.setLaserRepetitionRate(theLaserRepetitionRate, 0, i);
            }
            if (lasers.length == 0 && laserManufacturer != null) {
                Frequency theLaserRepetitionRate;
                store.setLaserID(MetadataTools.createLSID("LightSource", 0, 0), 0, 0);
                store.setLaserType(MetadataTools.getLaserType("Other"), 0, 0);
                store.setLaserLaserMedium(MetadataTools.getLaserMedium("Other"), 0, 0);
                store.setLaserManufacturer(laserManufacturer, 0, 0);
                store.setLaserModel(laserModel, 0, 0);
                Power theLaserPower = FormatTools.createPower(laserPower, UNITS.MILLIWATT);
                if (theLaserPower != null) {
                    store.setLaserPower(theLaserPower, 0, 0);
                }
                if ((theLaserRepetitionRate = FormatTools.createFrequency(laserRepetitionRate, UNITS.HERTZ)) != null) {
                    store.setLaserRepetitionRate(theLaserRepetitionRate, 0, 0);
                }
            }
            if (filterSetModel != null) {
                store.setFilterSetID(MetadataTools.createLSID("FilterSet", 0, 0), 0, 0);
                store.setFilterSetModel(filterSetModel, 0, 0);
                String dichroicID = MetadataTools.createLSID("Dichroic", 0, 0);
                String emFilterID = MetadataTools.createLSID("Filter", 0, 0);
                String exFilterID = MetadataTools.createLSID("Filter", 0, 1);
                store.setDichroicID(dichroicID, 0, 0);
                store.setDichroicModel(dichroicModel, 0, 0);
                store.setFilterSetDichroicRef(dichroicID, 0, 0);
                store.setFilterID(emFilterID, 0, 0);
                store.setFilterModel(emissionModel, 0, 0);
                store.setFilterSetEmissionFilterRef(emFilterID, 0, 0, 0);
                store.setFilterID(exFilterID, 0, 1);
                store.setFilterModel(excitationModel, 0, 1);
                store.setFilterSetExcitationFilterRef(exFilterID, 0, 0, 0);
            }
            if (objectiveModel != null) {
                store.setObjectiveModel(objectiveModel, 0, 0);
            }
            if (immersion == null) {
                immersion = "Other";
            }
            store.setObjectiveImmersion(MetadataTools.getImmersion(immersion), 0, 0);
            if (lensNA != null) {
                store.setObjectiveLensNA(lensNA, 0, 0);
            }
            if (workingDistance != null) {
                store.setObjectiveWorkingDistance(new Length(workingDistance, UNITS.MICROMETER), 0, 0);
            }
            if (magnification != null) {
                store.setObjectiveCalibratedMagnification(magnification, 0, 0);
            }
            store.setObjectiveCorrection(MetadataTools.getCorrection("Other"), 0, 0);
            String objectiveID = MetadataTools.createLSID("Objective", 0, 0);
            store.setObjectiveID(objectiveID, 0, 0);
            store.setObjectiveSettingsID(objectiveID, 0);
            String detectorID = MetadataTools.createLSID("Detector", 0, 0);
            store.setDetectorID(detectorID, 0, 0);
            store.setDetectorManufacturer(detectorManufacturer, 0, 0);
            store.setDetectorModel(detectorModel, 0, 0);
            store.setDetectorType(MetadataTools.getDetectorType("Other"), 0, 0);
            for (Integer key : gains.keySet()) {
                int index = key;
                if (index >= this.getEffectiveSizeC()) continue;
                store.setDetectorSettingsGain((Double)gains.get(key), 0, index);
                store.setDetectorSettingsID(detectorID, 0, index);
            }
            if (lastName != null) {
                String experimenterID = MetadataTools.createLSID("Experimenter", 0);
                store.setExperimenterID(experimenterID, 0);
                store.setExperimenterLastName(lastName, 0);
            }
            if (stagePos != null) {
                for (int i = 0; i < this.getImageCount(); ++i) {
                    if (stagePos.length > 0) {
                        store.setPlanePositionX(stagePos[0], 0, i);
                        this.addGlobalMeta("X position for position #1", stagePos[0]);
                    }
                    if (stagePos.length > 1) {
                        store.setPlanePositionY(stagePos[1], 0, i);
                        this.addGlobalMeta("Y position for position #1", stagePos[1]);
                    }
                    if (stagePos.length <= 2) continue;
                    store.setPlanePositionZ(stagePos[2], 0, i);
                    this.addGlobalMeta("Z position for position #1", stagePos[2]);
                }
            }
            if (exposureTime != null) {
                for (int i = 0; i < this.getImageCount(); ++i) {
                    store.setPlaneExposureTime(exposureTime, 0, i);
                }
            }
        }
    }

    private String[] tokenize(String line) {
        ArrayList<String> tokens = new ArrayList<String>();
        boolean inWhiteSpace = true;
        boolean withinQuotes = false;
        StringBuilder token = null;
        for (int i = 0; i < line.length(); ++i) {
            char c = line.charAt(i);
            if (Character.isWhitespace(c) || c == '\u0004') {
                if (withinQuotes) {
                    token.append(c);
                    continue;
                }
                if (inWhiteSpace) continue;
                inWhiteSpace = true;
                if (token.length() <= 0) continue;
                tokens.add(token.toString());
                token = null;
                continue;
            }
            if ('\"' == c) {
                boolean bl = withinQuotes = !withinQuotes;
            }
            if (inWhiteSpace) {
                inWhiteSpace = false;
                token = new StringBuilder();
            }
            token.append(c);
        }
        if (null != token && token.length() > 0) {
            tokens.add(token.toString());
        }
        return tokens.toArray(new String[0]);
    }

    String[] findKeyValue(String[] tokens, String[][] regexesArray) {
        String[] keyValue = this.findKeyValueForCategory(tokens, regexesArray);
        if (null == keyValue) {
            keyValue = this.findKeyValueOther(tokens, OTHER_KEYS);
        }
        if (null == keyValue) {
            String key = tokens[0];
            String value = this.concatenateTokens(tokens, 1, tokens.length);
            keyValue = new String[]{key, value};
        }
        return keyValue;
    }

    private String concatenateTokens(String[] tokens, int start, int stop) {
        StringBuilder returnValue = new StringBuilder();
        for (int i = start; i < tokens.length && i < stop; ++i) {
            returnValue.append(tokens[i]);
            if (i >= stop - 1) continue;
            returnValue.append(' ');
        }
        return returnValue.toString();
    }

    private String[] findKeyValueForCategory(String[] tokens, String[][] regexesArray) {
        String[] keyValue = null;
        for (String[] regexes : regexesArray) {
            if (!this.compareTokens(tokens, 1, regexes, 0)) continue;
            int splitIndex = 1 + regexes.length;
            String key = this.concatenateTokens(tokens, 0, splitIndex);
            String value = this.concatenateTokens(tokens, splitIndex, tokens.length);
            keyValue = new String[]{key, value};
            break;
        }
        return keyValue;
    }

    private String[] findKeyValueOther(String[] tokens, String[][] regexesArray) {
        String[] keyValue = null;
        for (String[] regexes : regexesArray) {
            for (int i = 1; i < tokens.length - regexes.length; ++i) {
                if (!tokens[i].toLowerCase().matches(regexes[0]) || 1 != regexes.length && !this.compareTokens(tokens, i + 1, regexes, 1)) continue;
                int splitIndex = i + regexes.length;
                String key = this.concatenateTokens(tokens, 0, splitIndex);
                String value = this.concatenateTokens(tokens, splitIndex, tokens.length);
                keyValue = new String[]{key, value};
                break;
            }
            if (null != keyValue) break;
        }
        return keyValue;
    }

    private boolean compareTokens(String[] tokens, int tokenIndex, String[] regexes, int regexesIndex) {
        boolean returnValue = true;
        int i = tokenIndex;
        for (int j = regexesIndex; j < regexes.length; ++j) {
            if (i >= tokens.length || !tokens[i].toLowerCase().matches(regexes[j])) {
                returnValue = false;
                break;
            }
            ++i;
        }
        return returnValue;
    }

    private Double[] splitDoubles(String v) {
        StringTokenizer t = new StringTokenizer(v);
        Double[] values = new Double[t.countTokens()];
        for (int n = 0; n < values.length; ++n) {
            String token = t.nextToken().trim();
            try {
                values[n] = new Double(token);
                continue;
            }
            catch (NumberFormatException e) {
                LOGGER.debug("Could not parse double value '{}'", (Object)token, (Object)e);
            }
        }
        return values;
    }

    private boolean checkUnit(String actual, String ... expected) {
        if (actual == null || actual.equals("")) {
            return true;
        }
        for (String exp : expected) {
            if (!actual.equals(exp)) continue;
            return true;
        }
        LOGGER.debug("Unexpected unit '{}'; expected '{}'", (Object)actual, (Object)expected);
        return false;
    }
}

