from functools import partial
import re

from cellacdc import html_utils, myutils, widgets

from . import issues_url

def warnTooManyItems(mainWin, numItems, qparent):
    from . import widgets, html_utils
    mainWin.logger.info(
        '[WARNING]: asking user what to do with too many graphical items...'
    )
    msg = widgets.myMessageBox()
    txt = html_utils.paragraph(f"""
        You loaded a segmentation mask that has <b>{numItems} objects</b>.<br><br>
        Creating <b>high resolution</b> text annotations 
        for these many objects could take a <b>long time</b>.<br><br>
        We recommend <b>switching to low resolution</b> annotations.<br><br>
        You can still try to switch to high resolution later.<br><br>
        What do you want to do?
    """)

    _, stayHighResButton, switchToLowResButton = msg.warning(
        qparent, 'Too many objects', txt,
        buttonsTexts=(
            'Cancel', 'Stay on high resolution', 
            widgets.reloadPushButton(' Switch to low resolution ')              
        )
    )
    return msg.cancel, msg.clickedButton==switchToLowResButton

def warnRestartCellACDCcolorModeToggled(scheme, app_name='Cell-ACDC', parent=None):
    from . import widgets, html_utils
    msg = widgets.myMessageBox(wrapText=False)
    txt = (
        'In order for the change to take effect, '
        f'<b>please restart {app_name}</b>'
    )
    if scheme == 'dark':
        issues_href = f'<a href="{issues_url}">GitHub page</a>'
        note_txt = (
            'NOTE: <b>dark mode</b> is a recent feature so if you see '
            'if you see anything odd,<br>'
            'please, <b>report it</b> by opening an issue '
            f'on our {issues_href}.<br><br>'
            'Thanks!'
        )
        txt = f'{txt}<br><br>{note_txt}'
    txt = html_utils.paragraph(txt)
    msg.information(parent, f'Restart {app_name}', txt)

class DataTypeWarning(RuntimeWarning):
    def __init__(self, message):
        self._message = message
    
    def __str__(self):
        return repr(self._message)

def warn_image_overflow_dtype(input_dtype, max_value, inferred_dtype):
    import warnings
    warnings.warn(
        f'The input image has data type {input_dtype}. Since it is neither '
        f'8-bit, 16-bit, nor 32-bit the data was inferred as {inferred_dtype} '
        f'from the max value of the image of {max_value}.', 
        DataTypeWarning
    )

def warn_cca_integrity(txt, category, qparent, go_to_frame_callback=None):
    from . import html_utils, widgets
    from qtpy.QtWidgets import QCheckBox
    
    preamble = html_utils.paragraph(
        'WARNING: <b>integrity of cell cycle annotations</b> '
        'might be <b>compromised</b>:'
    )
    
    msg_text = f'{preamble}{txt}'
    
    stopSpecificMessageCheckbox = QCheckBox(
        'Stop warning with this specific message'
    )
    stopCategoryCheckbox = QCheckBox(
        f'Stop warning about "{category}"'
    )
    disableAllWarningsCheckbox = QCheckBox(
        'Disable all warnings'
    )
    
    checkboxes = (
        stopSpecificMessageCheckbox, 
        stopCategoryCheckbox, 
        disableAllWarningsCheckbox
    )
    
    msg = widgets.myMessageBox(wrapText=False)
    if go_to_frame_callback is not None and txt.find('At frame n.') != -1:
        frame_n = re.findall(r'At frame n. (\d+)', txt)[0]
        goToFrameButton = widgets.NavigatePushButton(f'Go to frame n. {frame_n}')
        goToFrameButton = msg.addButton(goToFrameButton)
        goToFrameButton.disconnect()
        goToFrameButton.clicked.connect(
            partial(go_to_frame_callback, int(frame_n))
        )
        
    msg.warning(
        qparent, 'Annotations integrity warning', msg_text, 
        widgets=checkboxes
    )
    
    if stopSpecificMessageCheckbox.isChecked():
        return txt
    
    if stopCategoryCheckbox.isChecked():
        return category
    
    if disableAllWarningsCheckbox.isChecked():
        return 'disable_all'
    
    return ''

def warn_installing_different_cellpose_version(
        requested_version, installed_version
    ):
    if not myutils.is_gui_running():
        print(
            f'[WARNING]: You requested to install `Cellpose {requested_version}` '
            f'but you already have `Cellpose {installed_version}`.\n\n'
            f'If you proceed, Cell-ACDC will *uninstall* `{installed_version}` ' 
            f'and will install `{requested_version}`.'
        )
        return False
    
    note_text = """
    You can still proceed and let Cell-ACDC take care of 
    uninstalling/installing the right versions every time you request it.
    """
    txt = html_utils.paragraph(f"""
        [WARNING]: You requested to install 
        <code>Cellpose {requested_version}</code>, however you <b>already have</b> 
        <code>Cellpose {installed_version}</code>.<br><br>
        If you proceed, Cell-ACDC will <b>uninstall</b> <code>{installed_version}</code> 
        and will install <code>{requested_version}</code>.<br><br>
        If you plan to use both versions, we recommend installing Cell-ACDC 
        again in a <b>different environment</b><br> where you can keep one 
        version of Cellpose per environment.<br><br>
        Thank you for your patience!<br>
        {html_utils.to_note(note_text)}
    """)
    msg = widgets.myMessageBox(wrapText=False)
    msg.warning(
        None, 'Cellpose already installed', txt, 
        buttonsTexts=('Cancel', 'Ok')
    )
    return msg.cancel