import apsw, uuid, os, chromadb
from searchbibleai.utils.BibleBooks import BibleBooks
from searchbibleai.utils.AGBsubheadings import agbSubheadings
from searchbibleai.utils.AGBparagraphs_expanded import agbParagraphs
from searchbibleai.utils.health_check import HealthCheck
from pathlib import Path
from prompt_toolkit.shortcuts import ProgressBar


class ConvertBible:
    """
    Convert UniqueBible App bible files (*.bible) into Chromadb format
    """

    @staticmethod
    def convert_bible(database: str) -> None:
        if not os.path.isfile(database) or not database.endswith(".bible"):
            print("Invalid file path given!")
            return None
        else:
            version = os.path.basename(database)[:-6]
        
        def getAllVerses(bible=version):
            # get all verses from Unique Bible App bible file
            with apsw.Connection(database) as connection:
                cursor = connection.cursor()
                query = "SELECT * FROM Verses ORDER BY Book, Chapter, Verse"
                cursor.execute(query)
                allVerses = cursor.fetchall()
                return allVerses
            return ()

        # database path
        dbpath = os.path.join(HealthCheck.getFiles(), "bibles", version)
        if os.path.isdir(dbpath):
            # remove old database if it exists
            HealthCheck.print3(f"Removing old database: {dbpath}")
            shutil.rmtree(dbpath, ignore_errors=True)
        HealthCheck.print3(f"Creating database: {dbpath}")
        Path(dbpath).mkdir(parents=True, exist_ok=True)
        # client
        chroma_client = chromadb.PersistentClient(dbpath)
        # collection
        collectionVerse = chroma_client.get_or_create_collection(
            name="verses",
            metadata={"hnsw:space": "cosine"},
            embedding_function=HealthCheck.getEmbeddingFunction(embeddingModel="all-mpnet-base-v2"),
        )
        collectionParagraph = chroma_client.get_or_create_collection(
            name="paragraphs",
            metadata={"hnsw:space": "cosine"},
            embedding_function=HealthCheck.getEmbeddingFunction(embeddingModel="all-mpnet-base-v2"),
        )

        paragraphTitle = ""
        paragraphStart = ""
        paragraphStartB = ""
        paragraphStartC = ""
        paragraphStartV = ""
        paragraphEnd = ""
        paragraphEndB = ""
        paragraphEndC = ""
        paragraphEndV = ""
        paragraphContent = ""            

        with ProgressBar() as pb:
            for book, chapter, verse, scripture in pb(getAllVerses()):
                bcv = f"{book}.{chapter}.{verse}"

                abbrev = BibleBooks.abbrev["eng"]
                book_abbr = abbrev[str(book)][0]
                metadata = {
                    "book_abbr": book_abbr,
                    "book": book,
                    "chapter": chapter,
                    "verse": verse,
                    "reference": bcv,
                }
                id = str(uuid.uuid4())
                collectionVerse.add(
                    documents = [scripture],
                    metadatas = [metadata],
                    ids = [id]
                )

                if bcv in agbSubheadings:
                    if paragraphStart and paragraphEnd:
                        # save previous paragraph
                        metadata = {
                            "title": paragraphTitle,
                            "start": paragraphStart,
                            "book_start": paragraphStartB,
                            "chapter_start": paragraphStartC,
                            "verse_start": paragraphStartV,
                            "end": paragraphEnd,
                            "book_end": paragraphEndB,
                            "chapter_end": paragraphEndC,
                            "verse_end": paragraphEndV,
                        }
                        id = str(uuid.uuid4())
                        collectionParagraph.add(
                            documents = [paragraphContent],
                            metadatas = [metadata],
                            ids = [id]
                        )
                    paragraphTitle = agbSubheadings.get(bcv)
                    paragraphStart = bcv
                    paragraphStartB = str(book)
                    paragraphStartC = str(chapter)
                    paragraphStartV = str(verse)
                    paragraphContent = f"{paragraphTitle}\n{chapter}:{verse} {scripture}"
                else:
                    if (book, chapter, verse) in agbParagraphs:
                        paragraphContent += "\n"
                    paragraphContent += f"\n{chapter}:{verse} {scripture}"
                paragraphEnd = bcv
                paragraphEndB = str(book)
                paragraphEndC = str(chapter)
                paragraphEndV = str(verse)

            # save the last paragraph
            metadata = {
                "title": paragraphTitle,
                "start": paragraphStart,
                "book_start": paragraphStartB,
                "chapter_start": paragraphStartC,
                "verse_start": paragraphStartV,
                "end": paragraphEnd,
                "book_end": paragraphEndB,
                "chapter_end": paragraphEndC,
                "verse_end": paragraphEndV,
            }
            id = str(uuid.uuid4())
            collectionParagraph.add(
                documents = [paragraphContent],
                metadatas = [metadata],
                ids = [id]
            )

        return None
