"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebLoadBalancing = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
const secret_http_header_1 = require("./secret-http-header");
const util_1 = require("../util");
/**
 * A utility for creating a public-facing Application Load Balancer.
 */
class WebLoadBalancing extends constructs_1.Construct {
    /**
     * Creates a new WebLoadBalancing.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { vpc, certificates, securityGroup, secretHeaderName } = props;
        this.enableSecretHeader = Boolean(props.requireSecretHeader || secretHeaderName);
        this.enableKnownHostnames = Boolean(props.requireKnownHostname);
        this.idleTimeout = props.idleTimeout || aws_cdk_lib_1.Duration.seconds(59);
        this.priorities = new Set();
        const alb = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, "LoadBalancer", {
            vpc,
            vpcSubnets: { subnets: vpc.publicSubnets },
            securityGroup,
            internetFacing: true,
            idleTimeout: this.idleTimeout,
            http2Enabled: true,
            ipAddressType: props.ipAddressType,
        });
        this.loadBalancer = alb;
        const albLogs = new aws_s3_1.Bucket(this, "AccessLogs", {
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        alb.logAccessLogs(albLogs);
        if (this.enableSecretHeader) {
            this.secretHeader = new secret_http_header_1.SecretHttpHeader(this, "SecretHeader", {
                headerName: secretHeaderName,
            });
        }
        // If either the secret header or the Host header is incorrect we should
        // return a 421 Misdirected Request.
        const defaultAction = this.enableSecretHeader || this.enableKnownHostnames
            ? aws_elasticloadbalancingv2_1.ListenerAction.fixedResponse(421, { contentType: "text/plain" })
            : undefined;
        if (defaultAction) {
            this.defaultAction = defaultAction;
        }
        this.listener = alb.addListener("HttpsListener", {
            port: 443,
            open: false,
            certificates,
            sslPolicy: aws_elasticloadbalancingv2_1.SslPolicy.RECOMMENDED_TLS,
            defaultAction,
        });
    }
    /**
     * Adds a target to the listener.
     *
     * If the following options are left undefined, these defaults will be used.
     * - `port`: 443
     * - `protocol`: HTTPS
     * - `deregistrationDelay`: load balancer idle timeout
     * - `healthCheck.path`: /
     * - `healthCheck.healthyThresholdCount`: 2
     * - `healthCheck.interval`: 30 seconds
     * - `healthCheck.timeout`: 29 seconds
     *
     * @param id - The ID of the new target group.
     * @param target - The load balancing target to receive traffic.
     * @param options - The target group options.
     * @returns The new Application Target Group
     */
    addTarget(id, target, options) {
        const { priority, hostnames = [], healthCheck = {}, port = 443, protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS, deregistrationDelay = this.idleTimeout, ...others } = options || {};
        delete others.vpc;
        delete others.targets;
        if (this.enableKnownHostnames && hostnames.length === 0) {
            throw new Error("You must provide hostnames for this load balancer target");
        }
        const defaultHealthCheckProps = {
            path: "/",
            healthyThresholdCount: 2,
            interval: aws_cdk_lib_1.Duration.seconds(30),
            timeout: aws_cdk_lib_1.Duration.seconds(29),
        };
        const targetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(this, id, {
            port,
            protocol,
            deregistrationDelay,
            healthCheck: { ...defaultHealthCheckProps, ...healthCheck },
            vpc: this.loadBalancer.vpc,
            targets: [target],
            ...others,
        });
        const conditions = this.secretHeader
            ? [this.secretHeader.createListenerCondition()]
            : [];
        const nextSafePriority = this.priorities.size === 0
            ? 1
            : Math.max(...Array.from(this.priorities.values())) + 1;
        let rulePriority = priority ?? nextSafePriority;
        if (hostnames.length > 0) {
            // ListenerRule only supports 5 "Condition Values per Rule".
            // The number of hostnames per host header condition must consider this.
            const hostnameRules = (0, util_1.collate)(hostnames, 5 - conditions.length);
            for (const [index, names] of hostnameRules.entries()) {
                const groupPriority = rulePriority + index;
                this.listener.addTargetGroups(`${id}Hosts${index}`, {
                    targetGroups: [targetGroup],
                    priority: groupPriority,
                    conditions: [...conditions, aws_elasticloadbalancingv2_1.ListenerCondition.hostHeaders(names)],
                });
                this.priorities.add(groupPriority);
            }
        }
        else {
            const groupPriority = this.defaultAction ? rulePriority : undefined;
            this.listener.addTargetGroups(id, {
                targetGroups: [targetGroup],
                priority: groupPriority,
                conditions: conditions.length > 0 ? conditions : undefined,
            });
            if (groupPriority) {
                this.priorities.add(groupPriority);
            }
        }
        return targetGroup;
    }
}
exports.WebLoadBalancing = WebLoadBalancing;
_a = JSII_RTTI_SYMBOL_1;
WebLoadBalancing[_a] = { fqn: "shady-island.networking.WebLoadBalancing", version: "0.1.25" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibG9hZC1iYWxhbmNpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvbmV0d29ya2luZy9sb2FkLWJhbGFuY2luZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZDQUF1QztBQUd2Qyx1RkFhZ0Q7QUFDaEQsK0NBSTRCO0FBQzVCLDJDQUF1QztBQUN2Qyw2REFBMkU7QUFDM0Usa0NBQWtDO0FBMEZsQzs7R0FFRztBQUNILE1BQWEsZ0JBQWlCLFNBQVEsc0JBQVM7SUFzQjdDOzs7Ozs7T0FNRztJQUNILFlBQ0UsS0FBZ0IsRUFDaEIsRUFBVSxFQUNWLEtBQTRCO1FBRTVCLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxFQUFFLEdBQUcsRUFBRSxZQUFZLEVBQUUsYUFBYSxFQUFFLGdCQUFnQixFQUFFLEdBQUcsS0FBSyxDQUFDO1FBQ3JFLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxPQUFPLENBQy9CLEtBQUssQ0FBQyxtQkFBbUIsSUFBSSxnQkFBZ0IsQ0FDOUMsQ0FBQztRQUNGLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxPQUFPLENBQUMsS0FBSyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDaEUsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxJQUFJLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQzdELElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxHQUFHLEVBQUUsQ0FBQztRQUU1QixNQUFNLEdBQUcsR0FBRyxJQUFJLG9EQUF1QixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDNUQsR0FBRztZQUNILFVBQVUsRUFBRSxFQUFFLE9BQU8sRUFBRSxHQUFHLENBQUMsYUFBYSxFQUFFO1lBQzFDLGFBQWE7WUFDYixjQUFjLEVBQUUsSUFBSTtZQUNwQixXQUFXLEVBQUUsSUFBSSxDQUFDLFdBQVc7WUFDN0IsWUFBWSxFQUFFLElBQUk7WUFDbEIsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO1NBQ25DLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxZQUFZLEdBQUcsR0FBRyxDQUFDO1FBRXhCLE1BQU0sT0FBTyxHQUFHLElBQUksZUFBTSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDN0MsVUFBVSxFQUFFLHlCQUFnQixDQUFDLFVBQVU7WUFDdkMsaUJBQWlCLEVBQUUsMEJBQWlCLENBQUMsU0FBUztZQUM5QyxVQUFVLEVBQUUsSUFBSTtTQUNqQixDQUFDLENBQUM7UUFDSCxHQUFHLENBQUMsYUFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRTNCLElBQUksSUFBSSxDQUFDLGtCQUFrQixFQUFFLENBQUM7WUFDNUIsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLHFDQUFnQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7Z0JBQzdELFVBQVUsRUFBRSxnQkFBZ0I7YUFDN0IsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztRQUVELHdFQUF3RTtRQUN4RSxvQ0FBb0M7UUFDcEMsTUFBTSxhQUFhLEdBQ2pCLElBQUksQ0FBQyxrQkFBa0IsSUFBSSxJQUFJLENBQUMsb0JBQW9CO1lBQ2xELENBQUMsQ0FBQywyQ0FBYyxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUUsRUFBRSxXQUFXLEVBQUUsWUFBWSxFQUFFLENBQUM7WUFDbEUsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUNoQixJQUFJLGFBQWEsRUFBRSxDQUFDO1lBQ2xCLElBQUksQ0FBQyxhQUFhLEdBQUcsYUFBYSxDQUFDO1FBQ3JDLENBQUM7UUFFRCxJQUFJLENBQUMsUUFBUSxHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUMsZUFBZSxFQUFFO1lBQy9DLElBQUksRUFBRSxHQUFHO1lBQ1QsSUFBSSxFQUFFLEtBQUs7WUFDWCxZQUFZO1lBQ1osU0FBUyxFQUFFLHNDQUFTLENBQUMsZUFBZTtZQUNwQyxhQUFhO1NBQ2QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7O09BZ0JHO0lBQ0ksU0FBUyxDQUNkLEVBQVUsRUFDVixNQUFzQyxFQUN0QyxPQUF1QjtRQUV2QixNQUFNLEVBQ0osUUFBUSxFQUNSLFNBQVMsR0FBRyxFQUFFLEVBQ2QsV0FBVyxHQUFHLEVBQUUsRUFDaEIsSUFBSSxHQUFHLEdBQUcsRUFDVixRQUFRLEdBQUcsZ0RBQW1CLENBQUMsS0FBSyxFQUNwQyxtQkFBbUIsR0FBRyxJQUFJLENBQUMsV0FBVyxFQUN0QyxHQUFHLE1BQU0sRUFDVixHQUFHLE9BQU8sSUFBSSxFQUFFLENBQUM7UUFDbEIsT0FBTyxNQUFNLENBQUMsR0FBRyxDQUFDO1FBQ2xCLE9BQU8sTUFBTSxDQUFDLE9BQU8sQ0FBQztRQUV0QixJQUFJLElBQUksQ0FBQyxvQkFBb0IsSUFBSSxTQUFTLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQ3hELE1BQU0sSUFBSSxLQUFLLENBQ2IsMERBQTBELENBQzNELENBQUM7UUFDSixDQUFDO1FBRUQsTUFBTSx1QkFBdUIsR0FBRztZQUM5QixJQUFJLEVBQUUsR0FBRztZQUNULHFCQUFxQixFQUFFLENBQUM7WUFDeEIsUUFBUSxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM5QixPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1NBQzlCLENBQUM7UUFFRixNQUFNLFdBQVcsR0FBRyxJQUFJLG1EQUFzQixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDdkQsSUFBSTtZQUNKLFFBQVE7WUFDUixtQkFBbUI7WUFDbkIsV0FBVyxFQUFFLEVBQUUsR0FBRyx1QkFBdUIsRUFBRSxHQUFHLFdBQVcsRUFBRTtZQUMzRCxHQUFHLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHO1lBQzFCLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQztZQUNqQixHQUFHLE1BQU07U0FDVixDQUFDLENBQUM7UUFFSCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsWUFBWTtZQUNsQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLHVCQUF1QixFQUFFLENBQUM7WUFDL0MsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUVQLE1BQU0sZ0JBQWdCLEdBQ3BCLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxLQUFLLENBQUM7WUFDeEIsQ0FBQyxDQUFDLENBQUM7WUFDSCxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQzVELElBQUksWUFBWSxHQUFHLFFBQVEsSUFBSSxnQkFBZ0IsQ0FBQztRQUVoRCxJQUFJLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFLENBQUM7WUFDekIsNERBQTREO1lBQzVELHdFQUF3RTtZQUN4RSxNQUFNLGFBQWEsR0FBRyxJQUFBLGNBQU8sRUFBQyxTQUFTLEVBQUUsQ0FBQyxHQUFHLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNoRSxLQUFLLE1BQU0sQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLElBQUksYUFBYSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUM7Z0JBQ3JELE1BQU0sYUFBYSxHQUFHLFlBQVksR0FBRyxLQUFLLENBQUM7Z0JBQzNDLElBQUksQ0FBQyxRQUFRLENBQUMsZUFBZSxDQUFDLEdBQUcsRUFBRSxRQUFRLEtBQUssRUFBRSxFQUFFO29CQUNsRCxZQUFZLEVBQUUsQ0FBQyxXQUFXLENBQUM7b0JBQzNCLFFBQVEsRUFBRSxhQUFhO29CQUN2QixVQUFVLEVBQUUsQ0FBQyxHQUFHLFVBQVUsRUFBRSw4Q0FBaUIsQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUM7aUJBQ2xFLENBQUMsQ0FBQztnQkFDSCxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUNyQyxDQUFDO1FBQ0gsQ0FBQzthQUFNLENBQUM7WUFDTixNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUNwRSxJQUFJLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQyxFQUFFLEVBQUU7Z0JBQ2hDLFlBQVksRUFBRSxDQUFDLFdBQVcsQ0FBQztnQkFDM0IsUUFBUSxFQUFFLGFBQWE7Z0JBQ3ZCLFVBQVUsRUFBRSxVQUFVLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQzNELENBQUMsQ0FBQztZQUNILElBQUksYUFBYSxFQUFFLENBQUM7Z0JBQ2xCLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ3JDLENBQUM7UUFDSCxDQUFDO1FBRUQsT0FBTyxXQUFXLENBQUM7SUFDckIsQ0FBQzs7QUFwTEgsNENBcUxDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgRHVyYXRpb24gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7IElDZXJ0aWZpY2F0ZSB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtY2VydGlmaWNhdGVtYW5hZ2VyXCI7XG5pbXBvcnQgeyBJU2VjdXJpdHlHcm91cCwgSVZwYyB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtZWMyXCI7XG5pbXBvcnQge1xuICBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlcixcbiAgQXBwbGljYXRpb25Qcm90b2NvbCxcbiAgQXBwbGljYXRpb25UYXJnZXRHcm91cCxcbiAgQXBwbGljYXRpb25UYXJnZXRHcm91cFByb3BzLFxuICBJQXBwbGljYXRpb25MaXN0ZW5lcixcbiAgSUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyLFxuICBJQXBwbGljYXRpb25UYXJnZXRHcm91cCxcbiAgSUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyVGFyZ2V0LFxuICBJcEFkZHJlc3NUeXBlLFxuICBMaXN0ZW5lckFjdGlvbixcbiAgTGlzdGVuZXJDb25kaXRpb24sXG4gIFNzbFBvbGljeSxcbn0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1lbGFzdGljbG9hZGJhbGFuY2luZ3YyXCI7XG5pbXBvcnQge1xuICBCbG9ja1B1YmxpY0FjY2VzcyxcbiAgQnVja2V0LFxuICBCdWNrZXRFbmNyeXB0aW9uLFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLXMzXCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0IHsgSVNlY3JldEh0dHBIZWFkZXIsIFNlY3JldEh0dHBIZWFkZXIgfSBmcm9tIFwiLi9zZWNyZXQtaHR0cC1oZWFkZXJcIjtcbmltcG9ydCB7IGNvbGxhdGUgfSBmcm9tIFwiLi4vdXRpbFwiO1xuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGFkZGluZyBhIG5ldyB0YXJnZXQgZ3JvdXAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVGFyZ2V0T3B0aW9ucyBleHRlbmRzIEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXBQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgaG9zdG5hbWVzIG9uIHdoaWNoIHRyYWZmaWMgaXMgc2VydmVkLlxuICAgKi9cbiAgcmVhZG9ubHkgaG9zdG5hbWVzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFRoZSBwcmlvcml0eSBvZiB0aGUgbGlzdGVuZXIgcnVsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBdXRvbWF0aWNhbGx5IGRldGVybWluZWRcbiAgICovXG4gIHJlYWRvbmx5IHByaW9yaXR5PzogbnVtYmVyO1xufVxuXG4vKipcbiAqIENvbnN0cnVjdG9yIHByb3BlcnRpZXMgZm9yIFdlYkxvYWRCYWxhbmNpbmcuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgV2ViTG9hZEJhbGFuY2luZ1Byb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBWUEMgd2hlcmUgdGhlc2UgcmVzb3VyY2VzIHNob3VsZCBiZSBkZXBsb3llZC5cbiAgICovXG4gIHJlYWRvbmx5IHZwYzogSVZwYztcblxuICAvKipcbiAgICogQSBzZWN1cml0eSBncm91cCBmb3IgdGhlIGxvYWQgYmFsYW5jZXIgaXRzZWxmLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEEgbmV3IHNlY3VyaXR5IGdyb3VwIHdpbGwgYmUgY3JlYXRlZFxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IElTZWN1cml0eUdyb3VwO1xuXG4gIC8qKlxuICAgKiBUaGUgY2VydGlmaWNhdGUgdG8gYXR0YWNoIHRvIHRoZSBsb2FkIGJhbGFuY2VyIGFuZCBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGNlcnRpZmljYXRlczogSUNlcnRpZmljYXRlW107XG5cbiAgLyoqXG4gICAqIFRoZSBsb2FkIGJhbGFuY2VyIGlkbGUgdGltZW91dCwgaW4gc2Vjb25kcy5cbiAgICpcbiAgICogSWYgeW91IGhhdmUgYSByZXZlcnNlIHByb3h5IGluIGZyb250IG9mIHRoaXMgbG9hZCBiYWxhbmNlciwgc3VjaCBhc1xuICAgKiBDbG91ZEZyb250LCB0aGlzIG51bWJlciBzaG91bGQgYmUgbGVzcyB0aGFuIHRoZSByZXZlcnNlIHByb3h5J3MgcmVxdWVzdFxuICAgKiB0aW1lb3V0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDU5IHNlY29uZHNcbiAgICovXG4gIHJlYWRvbmx5IGlkbGVUaW1lb3V0PzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIElQIGFkZHJlc3NlcyB0byB1c2UgKElQdjQgb3IgRHVhbCBTdGFjaykuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSVB2NCBvbmx5XG4gICAqL1xuICByZWFkb25seSBpcEFkZHJlc3NUeXBlPzogSXBBZGRyZXNzVHlwZTtcblxuICAvKipcbiAgICogRm9yYmlkIHJlcXVlc3RzIHRoYXQgYXJlIG1pc3NpbmcgYW4gSFRUUCBoZWFkZXIgd2l0aCBhIHNwZWNpZmljIHZhbHVlLlxuICAgKlxuICAgKiBJZiB0aGlzIG9wdGlvbiBpcyBzZXQgdG8gYHRydWVgLCB0aGlzIGNvbnN0cnVjdCB3aWxsIHByb3ZpZGUgYSBuZXdcbiAgICogYFNlY3JldEh0dHBIZWFkZXJgIGFjY2Vzc2libGUgb24gdGhlIGBzZWNyZXRIZWFkZXJgIHByb3BlcnR5LlxuICAgKlxuICAgKiBSZXF1ZXN0cyB3aXRob3V0IHRoZSBjb3JyZWN0IGhlYWRlciBuYW1lIGFuZCB2YWx1ZSB3aWxsIHJlY2VpdmUgYW4gSFRUUCA0MjFcbiAgICogc3RhdHVzIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZXF1aXJlU2VjcmV0SGVhZGVyPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHNlY3JldCBIVFRQIGhlYWRlci5cbiAgICpcbiAgICogUHJvdmlkaW5nIHRoaXMgb3B0aW9uIGltcGxpZXMgdGhhdCBgcmVxdWlyZVNlY3JldEhlYWRlcmAgaXMgYHRydWVgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFgtU2VjcmV0LVBhc3NwaHJhc2VcbiAgICovXG4gIHJlYWRvbmx5IHNlY3JldEhlYWRlck5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEZvcmJpZCByZXF1ZXN0cyB0aGF0IGFzayBmb3IgYW4gdW5rbm93biBob3N0bmFtZS5cbiAgICpcbiAgICogUmVxdWVzdHMgZm9yIGFuIHVua25vd24gaG9zdG5hbWUgd2lsbCByZWNlaXZlIGFuIEhUVFAgNDIxIHN0YXR1cyByZXNwb25zZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVxdWlyZUtub3duSG9zdG5hbWU/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIEEgdXRpbGl0eSBmb3IgY3JlYXRpbmcgYSBwdWJsaWMtZmFjaW5nIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZXIuXG4gKi9cbmV4cG9ydCBjbGFzcyBXZWJMb2FkQmFsYW5jaW5nIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgLyoqXG4gICAqIFRoZSBsb2FkIGJhbGFuY2VyIGl0c2VsZi5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBsb2FkQmFsYW5jZXI6IElBcHBsaWNhdGlvbkxvYWRCYWxhbmNlcjtcblxuICAvKipcbiAgICogVGhlIEhUVFBTIGxpc3RlbmVyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGxpc3RlbmVyOiBJQXBwbGljYXRpb25MaXN0ZW5lcjtcblxuICAvKipcbiAgICogVGhlIHNlY3JldCBoZWFkZXIgKGlmIGByZXF1aXJlU2VjcmV0SGVhZGVyYCB3YXMgc2V0IHRvIGB0cnVlYCkuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc2VjcmV0SGVhZGVyPzogSVNlY3JldEh0dHBIZWFkZXI7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBlbmFibGVTZWNyZXRIZWFkZXI6IGJvb2xlYW47XG4gIHByaXZhdGUgcmVhZG9ubHkgZW5hYmxlS25vd25Ib3N0bmFtZXM6IGJvb2xlYW47XG4gIHByaXZhdGUgcmVhZG9ubHkgaWRsZVRpbWVvdXQ6IER1cmF0aW9uO1xuICBwcml2YXRlIHJlYWRvbmx5IHByaW9yaXRpZXM6IFNldDxudW1iZXI+O1xuICBwcml2YXRlIGRlZmF1bHRBY3Rpb24/OiBMaXN0ZW5lckFjdGlvbjtcblxuICAvKipcbiAgICogQ3JlYXRlcyBhIG5ldyBXZWJMb2FkQmFsYW5jaW5nLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgLSBUaGUgc2NvcGUgaW4gd2hpY2ggdG8gZGVmaW5lIHRoaXMgY29uc3RydWN0LlxuICAgKiBAcGFyYW0gaWQgLSBUaGUgc2NvcGVkIGNvbnN0cnVjdCBJRC5cbiAgICogQHBhcmFtIHByb3BzIC0gSW5pdGlhbGl6YXRpb24gcHJvcGVydGllcyBmb3IgdGhpcyBjb25zdHJ1Y3QuXG4gICAqL1xuICBwdWJsaWMgY29uc3RydWN0b3IoXG4gICAgc2NvcGU6IENvbnN0cnVjdCxcbiAgICBpZDogc3RyaW5nLFxuICAgIHByb3BzOiBXZWJMb2FkQmFsYW5jaW5nUHJvcHNcbiAgKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHsgdnBjLCBjZXJ0aWZpY2F0ZXMsIHNlY3VyaXR5R3JvdXAsIHNlY3JldEhlYWRlck5hbWUgfSA9IHByb3BzO1xuICAgIHRoaXMuZW5hYmxlU2VjcmV0SGVhZGVyID0gQm9vbGVhbihcbiAgICAgIHByb3BzLnJlcXVpcmVTZWNyZXRIZWFkZXIgfHwgc2VjcmV0SGVhZGVyTmFtZVxuICAgICk7XG4gICAgdGhpcy5lbmFibGVLbm93bkhvc3RuYW1lcyA9IEJvb2xlYW4ocHJvcHMucmVxdWlyZUtub3duSG9zdG5hbWUpO1xuICAgIHRoaXMuaWRsZVRpbWVvdXQgPSBwcm9wcy5pZGxlVGltZW91dCB8fCBEdXJhdGlvbi5zZWNvbmRzKDU5KTtcbiAgICB0aGlzLnByaW9yaXRpZXMgPSBuZXcgU2V0KCk7XG5cbiAgICBjb25zdCBhbGIgPSBuZXcgQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIodGhpcywgXCJMb2FkQmFsYW5jZXJcIiwge1xuICAgICAgdnBjLFxuICAgICAgdnBjU3VibmV0czogeyBzdWJuZXRzOiB2cGMucHVibGljU3VibmV0cyB9LFxuICAgICAgc2VjdXJpdHlHcm91cCxcbiAgICAgIGludGVybmV0RmFjaW5nOiB0cnVlLFxuICAgICAgaWRsZVRpbWVvdXQ6IHRoaXMuaWRsZVRpbWVvdXQsXG4gICAgICBodHRwMkVuYWJsZWQ6IHRydWUsXG4gICAgICBpcEFkZHJlc3NUeXBlOiBwcm9wcy5pcEFkZHJlc3NUeXBlLFxuICAgIH0pO1xuICAgIHRoaXMubG9hZEJhbGFuY2VyID0gYWxiO1xuXG4gICAgY29uc3QgYWxiTG9ncyA9IG5ldyBCdWNrZXQodGhpcywgXCJBY2Nlc3NMb2dzXCIsIHtcbiAgICAgIGVuY3J5cHRpb246IEJ1Y2tldEVuY3J5cHRpb24uUzNfTUFOQUdFRCxcbiAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBCbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgIH0pO1xuICAgIGFsYi5sb2dBY2Nlc3NMb2dzKGFsYkxvZ3MpO1xuXG4gICAgaWYgKHRoaXMuZW5hYmxlU2VjcmV0SGVhZGVyKSB7XG4gICAgICB0aGlzLnNlY3JldEhlYWRlciA9IG5ldyBTZWNyZXRIdHRwSGVhZGVyKHRoaXMsIFwiU2VjcmV0SGVhZGVyXCIsIHtcbiAgICAgICAgaGVhZGVyTmFtZTogc2VjcmV0SGVhZGVyTmFtZSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIC8vIElmIGVpdGhlciB0aGUgc2VjcmV0IGhlYWRlciBvciB0aGUgSG9zdCBoZWFkZXIgaXMgaW5jb3JyZWN0IHdlIHNob3VsZFxuICAgIC8vIHJldHVybiBhIDQyMSBNaXNkaXJlY3RlZCBSZXF1ZXN0LlxuICAgIGNvbnN0IGRlZmF1bHRBY3Rpb24gPVxuICAgICAgdGhpcy5lbmFibGVTZWNyZXRIZWFkZXIgfHwgdGhpcy5lbmFibGVLbm93bkhvc3RuYW1lc1xuICAgICAgICA/IExpc3RlbmVyQWN0aW9uLmZpeGVkUmVzcG9uc2UoNDIxLCB7IGNvbnRlbnRUeXBlOiBcInRleHQvcGxhaW5cIiB9KVxuICAgICAgICA6IHVuZGVmaW5lZDtcbiAgICBpZiAoZGVmYXVsdEFjdGlvbikge1xuICAgICAgdGhpcy5kZWZhdWx0QWN0aW9uID0gZGVmYXVsdEFjdGlvbjtcbiAgICB9XG5cbiAgICB0aGlzLmxpc3RlbmVyID0gYWxiLmFkZExpc3RlbmVyKFwiSHR0cHNMaXN0ZW5lclwiLCB7XG4gICAgICBwb3J0OiA0NDMsXG4gICAgICBvcGVuOiBmYWxzZSxcbiAgICAgIGNlcnRpZmljYXRlcyxcbiAgICAgIHNzbFBvbGljeTogU3NsUG9saWN5LlJFQ09NTUVOREVEX1RMUyxcbiAgICAgIGRlZmF1bHRBY3Rpb24sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHRhcmdldCB0byB0aGUgbGlzdGVuZXIuXG4gICAqXG4gICAqIElmIHRoZSBmb2xsb3dpbmcgb3B0aW9ucyBhcmUgbGVmdCB1bmRlZmluZWQsIHRoZXNlIGRlZmF1bHRzIHdpbGwgYmUgdXNlZC5cbiAgICogLSBgcG9ydGA6IDQ0M1xuICAgKiAtIGBwcm90b2NvbGA6IEhUVFBTXG4gICAqIC0gYGRlcmVnaXN0cmF0aW9uRGVsYXlgOiBsb2FkIGJhbGFuY2VyIGlkbGUgdGltZW91dFxuICAgKiAtIGBoZWFsdGhDaGVjay5wYXRoYDogL1xuICAgKiAtIGBoZWFsdGhDaGVjay5oZWFsdGh5VGhyZXNob2xkQ291bnRgOiAyXG4gICAqIC0gYGhlYWx0aENoZWNrLmludGVydmFsYDogMzAgc2Vjb25kc1xuICAgKiAtIGBoZWFsdGhDaGVjay50aW1lb3V0YDogMjkgc2Vjb25kc1xuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBUaGUgSUQgb2YgdGhlIG5ldyB0YXJnZXQgZ3JvdXAuXG4gICAqIEBwYXJhbSB0YXJnZXQgLSBUaGUgbG9hZCBiYWxhbmNpbmcgdGFyZ2V0IHRvIHJlY2VpdmUgdHJhZmZpYy5cbiAgICogQHBhcmFtIG9wdGlvbnMgLSBUaGUgdGFyZ2V0IGdyb3VwIG9wdGlvbnMuXG4gICAqIEByZXR1cm5zIFRoZSBuZXcgQXBwbGljYXRpb24gVGFyZ2V0IEdyb3VwXG4gICAqL1xuICBwdWJsaWMgYWRkVGFyZ2V0KFxuICAgIGlkOiBzdHJpbmcsXG4gICAgdGFyZ2V0OiBJQXBwbGljYXRpb25Mb2FkQmFsYW5jZXJUYXJnZXQsXG4gICAgb3B0aW9ucz86IFRhcmdldE9wdGlvbnNcbiAgKTogSUFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAge1xuICAgIGNvbnN0IHtcbiAgICAgIHByaW9yaXR5LFxuICAgICAgaG9zdG5hbWVzID0gW10sXG4gICAgICBoZWFsdGhDaGVjayA9IHt9LFxuICAgICAgcG9ydCA9IDQ0MyxcbiAgICAgIHByb3RvY29sID0gQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQUyxcbiAgICAgIGRlcmVnaXN0cmF0aW9uRGVsYXkgPSB0aGlzLmlkbGVUaW1lb3V0LFxuICAgICAgLi4ub3RoZXJzXG4gICAgfSA9IG9wdGlvbnMgfHwge307XG4gICAgZGVsZXRlIG90aGVycy52cGM7XG4gICAgZGVsZXRlIG90aGVycy50YXJnZXRzO1xuXG4gICAgaWYgKHRoaXMuZW5hYmxlS25vd25Ib3N0bmFtZXMgJiYgaG9zdG5hbWVzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIllvdSBtdXN0IHByb3ZpZGUgaG9zdG5hbWVzIGZvciB0aGlzIGxvYWQgYmFsYW5jZXIgdGFyZ2V0XCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgZGVmYXVsdEhlYWx0aENoZWNrUHJvcHMgPSB7XG4gICAgICBwYXRoOiBcIi9cIixcbiAgICAgIGhlYWx0aHlUaHJlc2hvbGRDb3VudDogMixcbiAgICAgIGludGVydmFsOiBEdXJhdGlvbi5zZWNvbmRzKDMwKSxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLnNlY29uZHMoMjkpLFxuICAgIH07XG5cbiAgICBjb25zdCB0YXJnZXRHcm91cCA9IG5ldyBBcHBsaWNhdGlvblRhcmdldEdyb3VwKHRoaXMsIGlkLCB7XG4gICAgICBwb3J0LFxuICAgICAgcHJvdG9jb2wsXG4gICAgICBkZXJlZ2lzdHJhdGlvbkRlbGF5LFxuICAgICAgaGVhbHRoQ2hlY2s6IHsgLi4uZGVmYXVsdEhlYWx0aENoZWNrUHJvcHMsIC4uLmhlYWx0aENoZWNrIH0sXG4gICAgICB2cGM6IHRoaXMubG9hZEJhbGFuY2VyLnZwYyxcbiAgICAgIHRhcmdldHM6IFt0YXJnZXRdLFxuICAgICAgLi4ub3RoZXJzLFxuICAgIH0pO1xuXG4gICAgY29uc3QgY29uZGl0aW9ucyA9IHRoaXMuc2VjcmV0SGVhZGVyXG4gICAgICA/IFt0aGlzLnNlY3JldEhlYWRlci5jcmVhdGVMaXN0ZW5lckNvbmRpdGlvbigpXVxuICAgICAgOiBbXTtcblxuICAgIGNvbnN0IG5leHRTYWZlUHJpb3JpdHkgPVxuICAgICAgdGhpcy5wcmlvcml0aWVzLnNpemUgPT09IDBcbiAgICAgICAgPyAxXG4gICAgICAgIDogTWF0aC5tYXgoLi4uQXJyYXkuZnJvbSh0aGlzLnByaW9yaXRpZXMudmFsdWVzKCkpKSArIDE7XG4gICAgbGV0IHJ1bGVQcmlvcml0eSA9IHByaW9yaXR5ID8/IG5leHRTYWZlUHJpb3JpdHk7XG5cbiAgICBpZiAoaG9zdG5hbWVzLmxlbmd0aCA+IDApIHtcbiAgICAgIC8vIExpc3RlbmVyUnVsZSBvbmx5IHN1cHBvcnRzIDUgXCJDb25kaXRpb24gVmFsdWVzIHBlciBSdWxlXCIuXG4gICAgICAvLyBUaGUgbnVtYmVyIG9mIGhvc3RuYW1lcyBwZXIgaG9zdCBoZWFkZXIgY29uZGl0aW9uIG11c3QgY29uc2lkZXIgdGhpcy5cbiAgICAgIGNvbnN0IGhvc3RuYW1lUnVsZXMgPSBjb2xsYXRlKGhvc3RuYW1lcywgNSAtIGNvbmRpdGlvbnMubGVuZ3RoKTtcbiAgICAgIGZvciAoY29uc3QgW2luZGV4LCBuYW1lc10gb2YgaG9zdG5hbWVSdWxlcy5lbnRyaWVzKCkpIHtcbiAgICAgICAgY29uc3QgZ3JvdXBQcmlvcml0eSA9IHJ1bGVQcmlvcml0eSArIGluZGV4O1xuICAgICAgICB0aGlzLmxpc3RlbmVyLmFkZFRhcmdldEdyb3VwcyhgJHtpZH1Ib3N0cyR7aW5kZXh9YCwge1xuICAgICAgICAgIHRhcmdldEdyb3VwczogW3RhcmdldEdyb3VwXSxcbiAgICAgICAgICBwcmlvcml0eTogZ3JvdXBQcmlvcml0eSxcbiAgICAgICAgICBjb25kaXRpb25zOiBbLi4uY29uZGl0aW9ucywgTGlzdGVuZXJDb25kaXRpb24uaG9zdEhlYWRlcnMobmFtZXMpXSxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMucHJpb3JpdGllcy5hZGQoZ3JvdXBQcmlvcml0eSk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IGdyb3VwUHJpb3JpdHkgPSB0aGlzLmRlZmF1bHRBY3Rpb24gPyBydWxlUHJpb3JpdHkgOiB1bmRlZmluZWQ7XG4gICAgICB0aGlzLmxpc3RlbmVyLmFkZFRhcmdldEdyb3VwcyhpZCwge1xuICAgICAgICB0YXJnZXRHcm91cHM6IFt0YXJnZXRHcm91cF0sXG4gICAgICAgIHByaW9yaXR5OiBncm91cFByaW9yaXR5LFxuICAgICAgICBjb25kaXRpb25zOiBjb25kaXRpb25zLmxlbmd0aCA+IDAgPyBjb25kaXRpb25zIDogdW5kZWZpbmVkLFxuICAgICAgfSk7XG4gICAgICBpZiAoZ3JvdXBQcmlvcml0eSkge1xuICAgICAgICB0aGlzLnByaW9yaXRpZXMuYWRkKGdyb3VwUHJpb3JpdHkpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiB0YXJnZXRHcm91cDtcbiAgfVxufVxuIl19