"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RawSha256 = void 0;
var constants_1 = require("./constants");
/**
 * @internal
 */
var RawSha256 = /** @class */ (function () {
    function RawSha256() {
        this.state = Int32Array.from(constants_1.INIT);
        this.temp = new Int32Array(64);
        this.buffer = new Uint8Array(64);
        this.bufferLength = 0;
        this.bytesHashed = 0;
        /**
         * @internal
         */
        this.finished = false;
    }
    RawSha256.prototype.update = function (data) {
        if (this.finished) {
            throw new Error("Attempted to update an already finished hash.");
        }
        var position = 0;
        var byteLength = data.byteLength;
        this.bytesHashed += byteLength;
        if (this.bytesHashed * 8 > constants_1.MAX_HASHABLE_LENGTH) {
            throw new Error("Cannot hash more than 2^53 - 1 bits");
        }
        while (byteLength > 0) {
            this.buffer[this.bufferLength++] = data[position++];
            byteLength--;
            if (this.bufferLength === constants_1.BLOCK_SIZE) {
                this.hashBuffer();
                this.bufferLength = 0;
            }
        }
    };
    RawSha256.prototype.digest = function () {
        if (!this.finished) {
            var bitsHashed = this.bytesHashed * 8;
            var bufferView = new DataView(this.buffer.buffer, this.buffer.byteOffset, this.buffer.byteLength);
            var undecoratedLength = this.bufferLength;
            bufferView.setUint8(this.bufferLength++, 0x80);
            // Ensure the final block has enough room for the hashed length
            if (undecoratedLength % constants_1.BLOCK_SIZE >= constants_1.BLOCK_SIZE - 8) {
                for (var i = this.bufferLength; i < constants_1.BLOCK_SIZE; i++) {
                    bufferView.setUint8(i, 0);
                }
                this.hashBuffer();
                this.bufferLength = 0;
            }
            for (var i = this.bufferLength; i < constants_1.BLOCK_SIZE - 8; i++) {
                bufferView.setUint8(i, 0);
            }
            bufferView.setUint32(constants_1.BLOCK_SIZE - 8, Math.floor(bitsHashed / 0x100000000), true);
            bufferView.setUint32(constants_1.BLOCK_SIZE - 4, bitsHashed);
            this.hashBuffer();
            this.finished = true;
        }
        // The value in state is little-endian rather than big-endian, so flip
        // each word into a new Uint8Array
        var out = new Uint8Array(constants_1.DIGEST_LENGTH);
        for (var i = 0; i < 8; i++) {
            out[i * 4] = (this.state[i] >>> 24) & 0xff;
            out[i * 4 + 1] = (this.state[i] >>> 16) & 0xff;
            out[i * 4 + 2] = (this.state[i] >>> 8) & 0xff;
            out[i * 4 + 3] = (this.state[i] >>> 0) & 0xff;
        }
        return out;
    };
    RawSha256.prototype.hashBuffer = function () {
        var _a = this, buffer = _a.buffer, state = _a.state;
        var state0 = state[0], state1 = state[1], state2 = state[2], state3 = state[3], state4 = state[4], state5 = state[5], state6 = state[6], state7 = state[7];
        for (var i = 0; i < constants_1.BLOCK_SIZE; i++) {
            if (i < 16) {
                this.temp[i] =
                    ((buffer[i * 4] & 0xff) << 24) |
                        ((buffer[i * 4 + 1] & 0xff) << 16) |
                        ((buffer[i * 4 + 2] & 0xff) << 8) |
                        (buffer[i * 4 + 3] & 0xff);
            }
            else {
                var u = this.temp[i - 2];
                var t1_1 = ((u >>> 17) | (u << 15)) ^ ((u >>> 19) | (u << 13)) ^ (u >>> 10);
                u = this.temp[i - 15];
                var t2_1 = ((u >>> 7) | (u << 25)) ^ ((u >>> 18) | (u << 14)) ^ (u >>> 3);
                this.temp[i] =
                    ((t1_1 + this.temp[i - 7]) | 0) + ((t2_1 + this.temp[i - 16]) | 0);
            }
            var t1 = ((((((state4 >>> 6) | (state4 << 26)) ^
                ((state4 >>> 11) | (state4 << 21)) ^
                ((state4 >>> 25) | (state4 << 7))) +
                ((state4 & state5) ^ (~state4 & state6))) |
                0) +
                ((state7 + ((constants_1.KEY[i] + this.temp[i]) | 0)) | 0)) |
                0;
            var t2 = ((((state0 >>> 2) | (state0 << 30)) ^
                ((state0 >>> 13) | (state0 << 19)) ^
                ((state0 >>> 22) | (state0 << 10))) +
                ((state0 & state1) ^ (state0 & state2) ^ (state1 & state2))) |
                0;
            state7 = state6;
            state6 = state5;
            state5 = state4;
            state4 = (state3 + t1) | 0;
            state3 = state2;
            state2 = state1;
            state1 = state0;
            state0 = (t1 + t2) | 0;
        }
        state[0] += state0;
        state[1] += state1;
        state[2] += state2;
        state[3] += state3;
        state[4] += state4;
        state[5] += state5;
        state[6] += state6;
        state[7] += state7;
    };
    return RawSha256;
}());
exports.RawSha256 = RawSha256;
//# sourceMappingURL=data:application/json;base64,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