"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.withResolved = exports.isResolvableObject = exports.Tokenization = exports.Token = exports.TokenComparison = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const lazy_1 = require("./lazy");
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
/**
 * An enum-like class that represents the result of comparing two Tokens.
 *
 * The return type of {@link Token.compareStrings}.
 *
 * @stability stable
 */
class TokenComparison {
    constructor() {
    }
}
exports.TokenComparison = TokenComparison;
_a = JSII_RTTI_SYMBOL_1;
TokenComparison[_a] = { fqn: "@aws-cdk/core.TokenComparison", version: "1.124.0" };
/**
 * This means we're certain the two components are NOT Tokens, and identical.
 *
 * @stability stable
 */
TokenComparison.SAME = new TokenComparison();
/**
 * This means we're certain the two components are NOT Tokens, and different.
 *
 * @stability stable
 */
TokenComparison.DIFFERENT = new TokenComparison();
/**
 * This means exactly one of the components is a Token.
 *
 * @stability stable
 */
TokenComparison.ONE_UNRESOLVED = new TokenComparison();
/**
 * This means both components are Tokens.
 *
 * @stability stable
 */
TokenComparison.BOTH_UNRESOLVED = new TokenComparison();
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 *
 * @stability stable
 */
class Token {
    constructor() {
    }
    /**
     * Returns true if obj represents an unresolved value.
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     * @stability stable
     */
    static isUnresolved(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * Return a reversible string representation of this token.
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     *
     * @stability stable
     */
    static asString(value, options = {}) {
        if (typeof value === 'string') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token.
     *
     * @stability stable
     */
    static asNumber(value) {
        if (typeof value === 'number') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token.
     *
     * @stability stable
     */
    static asList(value, options = {}) {
        if (Array.isArray(value) && value.every(x => typeof x === 'string')) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a resolvable representation of the given value.
     *
     * @stability stable
     */
    static asAny(value) {
        return isResolvableObject(value) ? value : new intrinsic_1.Intrinsic(value);
    }
    /**
     * Compare two strings that might contain Tokens with each other.
     *
     * @stability stable
     */
    static compareStrings(possibleToken1, possibleToken2) {
        const firstIsUnresolved = Token.isUnresolved(possibleToken1);
        const secondIsUnresolved = Token.isUnresolved(possibleToken2);
        if (firstIsUnresolved && secondIsUnresolved) {
            return TokenComparison.BOTH_UNRESOLVED;
        }
        if (firstIsUnresolved || secondIsUnresolved) {
            return TokenComparison.ONE_UNRESOLVED;
        }
        return possibleToken1 === possibleToken2 ? TokenComparison.SAME : TokenComparison.DIFFERENT;
    }
}
exports.Token = Token;
_b = JSII_RTTI_SYMBOL_1;
Token[_b] = { fqn: "@aws-cdk/core.Token", version: "1.124.0" };
/**
 * Less oft-needed functions to manipulate Tokens.
 *
 * @stability stable
 */
class Tokenization {
    constructor() {
    }
    /**
     * Un-encode a string potentially containing encoded tokens.
     *
     * @stability stable
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a string which is either a complete encoded token, or doesn't contain tokens at all.
     *
     * It's illegal for the string to be a concatenation of an encoded token and something else.
     *
     * @stability stable
     */
    static reverseCompleteString(s) {
        const fragments = Tokenization.reverseString(s);
        if (fragments.length !== 1) {
            throw new Error(`Tokenzation.reverseCompleteString: argument must not be a concatentation, got '${s}'`);
        }
        return fragments.firstToken;
    }
    /**
     * Un-encode a Tokenized value from a number.
     *
     * @stability stable
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list.
     *
     * @stability stable
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Reverse any value into a Resolvable, if possible.
     *
     * In case of a string, the string must not be a concatenation.
     *
     * @stability stable
     */
    static reverse(x, options = {}) {
        if (Tokenization.isResolvable(x)) {
            return x;
        }
        if (typeof x === 'string') {
            if (options.failConcat === false) {
                // Handle this specially because reverseCompleteString might fail
                const fragments = Tokenization.reverseString(x);
                return fragments.length === 1 ? fragments.firstToken : undefined;
            }
            return Tokenization.reverseCompleteString(x);
        }
        if (Array.isArray(x)) {
            return Tokenization.reverseList(x);
        }
        if (typeof x === 'number') {
            return Tokenization.reverseNumber(x);
        }
        return undefined;
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     *
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     * @stability stable
     */
    static resolve(obj, options) {
        var _d;
        return resolve_1.resolve(obj, {
            scope: options.scope,
            resolver: options.resolver,
            preparing: ((_d = options.preparing) !== null && _d !== void 0 ? _d : false),
            removeEmpty: options.removeEmpty,
        });
    }
    /**
     * Return whether the given object is an IResolvable object.
     *
     * This is different from Token.isUnresolved() which will also check for
     * encoded Tokens, whereas this method will only do a type check on the given
     * object.
     *
     * @stability stable
     */
    static isResolvable(obj) {
        return isResolvableObject(obj);
    }
    /**
     * Stringify a number directly or lazily if it's a Token.
     *
     * If it is an object (i.e., { Ref: 'SomeLogicalId' }), return it as-is.
     *
     * @stability stable
     */
    static stringifyNumber(x) {
        // only convert numbers to strings so that Refs, conditions, and other things don't end up synthesizing as [object object]
        if (Token.isUnresolved(x)) {
            return lazy_1.Lazy.uncachedString({
                produce: context => {
                    const resolved = context.resolve(x);
                    return typeof resolved !== 'number' ? resolved : `${resolved}`;
                },
            });
        }
        else {
            return typeof x !== 'number' ? x : `${x}`;
        }
    }
}
exports.Tokenization = Tokenization;
_c = JSII_RTTI_SYMBOL_1;
Tokenization[_c] = { fqn: "@aws-cdk/core.Tokenization", version: "1.124.0" };
function isResolvableObject(x) {
    return typeof (x) === 'object' && x !== null && typeof x.resolve === 'function';
}
exports.isResolvableObject = isResolvableObject;
function withResolved(...args) {
    if (args.length < 2) {
        return;
    }
    const argArray = args.slice(0, args.length - 1);
    if (argArray.some(Token.isUnresolved)) {
        return;
    }
    args[args.length - 1].apply(arguments, argArray);
}
exports.withResolved = withResolved;
//# sourceMappingURL=data:application/json;base64,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