from __future__ import annotations

from dataclasses import dataclass
from typing import Any

from psnawp_api import PSNAWP
from psnawp_api.models.user import User


@dataclass
class PlaystationNetworkData:
    """Dataclass representing data retrieved from the Playstation Network api."""

    presence: dict[str, Any]
    username: str
    account_id: str
    available: bool
    title_metadata: dict[str, Any]
    platform: dict[str, Any]

class PlaystationNetwork:
    """Helper Class to return playstation network data in an easy to use structure"""
    def __init__(self, npsso:str):
        self.psn = PSNAWP(npsso)
        self.user: User | None = None
        self.data: PlaystationNetworkData | None = None

    def get_user(self):
        self.user = self.psn.user(online_id='me')
        return self.user

    def get_data(self):
        data: PlaystationNetworkData = PlaystationNetworkData(
            {}, "", "", False, {}, {}
        )
        
        if not self.user:
            self.user = self.get_user()
            data.username = self.user.online_id
            data.account_id = self.user.account_id
            data.presence = self.user.get_presence()

        data.available = (
            data.presence.get("basicPresence", {}).get("availability")
            == "availableToPlay"
        )
        data.platform = data.presence.get("basicPresence", {}).get(
            "primaryPlatformInfo"
        )
        game_title_info_list = data.presence.get("basicPresence", {}).get(
            "gameTitleInfoList"
        )

        if game_title_info_list:
            data.title_metadata = game_title_info_list[0]

        self.data = data
        return self.data