import keyring
import typer
from coderflow.agents.fastapi import get_fastapi_agent
from coderflow.agents.react import get_react_agent
import subprocess

app = typer.Typer()

SERVICE_NAME = "coderflow"

def ensure_api_key():
    """
    Middleware to ensure an API key is set before executing any command.
    Prompts the user to set the API key if not found.
    """
    api_key = keyring.get_password(SERVICE_NAME, "api_key")
    if not api_key:
        typer.echo("No API key found. Please enter your API key:")
        api_key = typer.prompt("API Key")
        keyring.set_password(SERVICE_NAME, "api_key", api_key)
        typer.echo("API key has been set successfully.")
    return api_key


@app.command()
def update_api_key(new_api_key: str = typer.Option(..., prompt=True, help="The new API key")):
    """
    Update the API key to a new value.
    """
    keyring.set_password(SERVICE_NAME, "api_key", new_api_key)
    typer.echo("API key has been updated successfully.")


@app.command()
def delete_api_key():
    """
    Delete the stored API key.
    """
    keyring.delete_password(SERVICE_NAME, "api_key")
    typer.echo("API key has been deleted successfully.")


@app.command()
def fastapi(prompt: str):
    """
    Generate a FastAPI app based on a given prompt.
    """
    api_key = ensure_api_key()
    fastapi_agent = get_fastapi_agent(api_key)
    fastapi_agent.stream_and_parse(prompt)


@app.command()
def react(prompt: str):
    """
    Generate a React app based on a given prompt.
    """
    api_key = ensure_api_key()
    react_agent = get_react_agent(api_key)
    react_agent.stream_and_parse(prompt)

@app.command()
def undo():
    """
    Undo the last command.
    """
    # A commit happens before and after each command, so we can simply revert to the previous commit.
    subprocess.run("git reset --hard HEAD~1")


@app.command()
def redo():
    """
    Redo the last undone command.
    """
    # We can simply revert the revert commit to redo the last undone command.
    subprocess.run("git reset --hard snapshot-after-parsing-stream")

if __name__ == "__main__":
    app()

