import os
import json
import signal
import anthropic
from coderflow.core.parse import StreamParser

DEFAULT_MODEL = "claude-3-haiku-20240307"
DEFAULT_SYSTEM = "You are a helpful agent."
CONFIG_DIR = os.path.expanduser("~/.config/coderflow")

class Agent:
    def __init__(self, api_key: str, model: str = DEFAULT_MODEL, system: str = DEFAULT_SYSTEM, session: str = None):
        self.anthropic_client = anthropic.Anthropic(api_key=api_key)
        self.model = model
        self.system = system
        self.parser = StreamParser()
        self.session_id = session
        # Attach signal handler for SIGINT
        signal.signal(signal.SIGINT, self.signal_handler)
        # Ensure the config directory exists
        os.makedirs(CONFIG_DIR, exist_ok=True)
        # Load message history if a session is provided
        if session is not None:
            self._load_message_history()
        else:
            self.message_history = []

    def _get_session_file_path(self, session_id: str) -> str:
        """Returns the file path for storing the session's message history."""
        return os.path.join(CONFIG_DIR, f"{session_id}.json")

    def set_session(self, session_id: str):
        """Sets the current session ID and loads the message history from disk."""
        self.session_id = session_id
        self._load_message_history()

    def add_message(self, role: str, content: str):
        """Adds a message to the current session's history and saves it to disk."""
        if self.session_id is None:
            raise ValueError("Session ID is not set.")
        message = {"role": role, "content": content}
        self.message_history.append(message)
        self._save_message_history()

    def get_message_history(self):
        """Retrieves the message history for the current session."""
        if self.session_id is None:
            raise ValueError("Session ID is not set.")
        return self.message_history

    def _load_message_history(self):
        """Loads the message history from disk for the current session."""
        if self.session_id is None:
            raise ValueError("Session ID is not set.")
        session_file = self._get_session_file_path(self.session_id)
        if os.path.exists(session_file):
            with open(session_file, "r") as f:
                self.message_history = json.load(f)
        else:
            self.message_history = []

    def _save_message_history(self):
        """Saves the current message history to disk."""
        if self.session_id is None:
            raise ValueError("Session ID is not set.")
        session_file = self._get_session_file_path(self.session_id)
        with open(session_file, "w") as f:
            json.dump(self.message_history, f, indent=4)

    def stream(self, prompt: str):
        """Streams the response from the Anthropic model."""
        if self.session_id is None:
            raise ValueError("Session ID is not set.")

        # Add the new user message to the history
        self.add_message("user", prompt)

        # Retrieve the complete message history for the current session
        messages = [] + self.get_message_history()
        if len(messages) > 0:
            print("length of messages: ", len(messages))
            print("length of messages[0]: ", len(messages[0]))

        with self.anthropic_client.messages.stream(
            model=self.model,
            max_tokens=4096,
            system=self.system,
            messages=messages
        ) as stream_response:
            for chunk in stream_response.text_stream:
                yield chunk

    def stream_and_parse(self, prompt: str):
        """Streams the response from the Anthropic model and parses it."""
        stream = self.stream(prompt)
        self.parser.parse_stream(stream)

    def signal_handler(self, sig, frame):
        if self.parser.current_process:
            self.parser.current_process.terminate()
            self.parser.current_process.wait()
            print("\n[red]Subprocess terminated.[/red]")
        else:
            print("\n[red]No subprocess to terminate.[/red]")

    def reset_session(self):
        """Clears the message history for the current session."""
        if self.session_id is not None:
            self.message_history = []
            self._save_message_history()
