"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resource = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_resource_1 = require("./cfn-resource");
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const reference_1 = require("./reference");
const stack_1 = require("./stack");
const token_1 = require("./token");
const RESOURCE_SYMBOL = Symbol.for('@aws-cdk/core.Resource');
/**
 * A construct which represents an AWS resource.
 *
 * @stability stable
 */
class Resource extends construct_compat_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d, _e;
        super(scope, id);
        if ((props.account !== undefined || props.region !== undefined) && props.environmentFromArn !== undefined) {
            throw new Error(`Supply at most one of 'account'/'region' (${props.account}/${props.region}) and 'environmentFromArn' (${props.environmentFromArn})`);
        }
        Object.defineProperty(this, RESOURCE_SYMBOL, { value: true });
        this.stack = stack_1.Stack.of(this);
        const parsedArn = props.environmentFromArn ? this.stack.parseArn(props.environmentFromArn) : undefined;
        this.env = {
            account: (_c = (_b = props.account) !== null && _b !== void 0 ? _b : parsedArn === null || parsedArn === void 0 ? void 0 : parsedArn.account) !== null && _c !== void 0 ? _c : this.stack.account,
            region: (_e = (_d = props.region) !== null && _d !== void 0 ? _d : parsedArn === null || parsedArn === void 0 ? void 0 : parsedArn.region) !== null && _e !== void 0 ? _e : this.stack.region,
        };
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.string({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Check whether the given construct is a Resource.
     *
     * @stability stable
     */
    static isResource(construct) {
        return construct !== null && typeof (construct) === 'object' && RESOURCE_SYMBOL in construct;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    /**
     * Apply the given removal policy to this resource.
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be deleted (`RemovalPolicy.DELETE`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * @stability stable
     */
    applyRemovalPolicy(policy) {
        const child = this.node.defaultChild;
        if (!child || !cfn_resource_1.CfnResource.isCfnResource(child)) {
            throw new Error('Cannot apply RemovalPolicy: no child or not a CfnResource. Apply the removal policy on the CfnResource directly.');
        }
        child.applyRemovalPolicy(policy);
    }
    /**
     * @stability stable
     */
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * Returns an environment-sensitive token that should be used for the resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * @stability stable
     */
    getResourceNameAttribute(nameAttr) {
        return mimicReference(nameAttr, {
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            },
        });
    }
    /**
     * Returns an environment-sensitive token that should be used for the resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * @param arnComponents The format of the ARN of this resource.
     * @stability stable
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return mimicReference(arnAttr, {
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            },
        });
    }
}
exports.Resource = Resource;
_a = JSII_RTTI_SYMBOL_1;
Resource[_a] = { fqn: "@aws-cdk/core.Resource", version: "1.102.0" };
/**
 * Produce a Lazy that is also a Reference (if the base value is a Reference).
 *
 * If the given value is a Reference (or resolves to a Reference), return a new
 * Reference that mimics the same target and display name, but resolves using
 * the logic of the passed lazy.
 *
 * If the given value is NOT a Reference, just return a simple Lazy.
 */
function mimicReference(refSource, producer) {
    const reference = token_1.Tokenization.reverse(refSource, {
        // If this is an ARN concatenation, just fail to extract a reference.
        failConcat: false,
    });
    if (!reference_1.Reference.isReference(reference)) {
        return lazy_1.Lazy.uncachedString(producer);
    }
    return token_1.Token.asString(new class extends reference_1.Reference {
        resolve(context) {
            return producer.produce(context);
        }
    }(reference, reference.target, reference.displayName));
}
//# sourceMappingURL=data:application/json;base64,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