"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isNameOfCloudFormationIntrinsic = exports.minimalCloudFormationJoin = exports.CLOUDFORMATION_TOKEN_RESOLVER = exports.CloudFormationLang = void 0;
const lazy_1 = require("../lazy");
const resolvable_1 = require("../resolvable");
const stack_1 = require("../stack");
const token_1 = require("../token");
const cfn_utils_provider_1 = require("./cfn-utils-provider");
const resolve_1 = require("./resolve");
/**
 * Routines that know how to do operations at the CloudFormation document language level
 */
class CloudFormationLang {
    /**
     * Turn an arbitrary structure potentially containing Tokens into a JSON string.
     *
     * Returns a Token which will evaluate to CloudFormation expression that
     * will be evaluated by CloudFormation to the JSON representation of the
     * input structure.
     *
     * All Tokens substituted in this way must return strings, or the evaluation
     * in CloudFormation will fail.
     *
     * @param obj The object to stringify
     * @param space Indentation to use (default: no pretty-printing)
     */
    static toJSON(obj, space) {
        return lazy_1.Lazy.uncachedString({
            // We used to do this by hooking into `JSON.stringify()` by adding in objects
            // with custom `toJSON()` functions, but it's ultimately simpler just to
            // reimplement the `stringify()` function from scratch.
            produce: (ctx) => tokenAwareStringify(obj, space !== null && space !== void 0 ? space : 0, ctx),
        });
    }
    /**
     * Produce a CloudFormation expression to concat two arbitrary expressions when resolving
     */
    static concat(left, right) {
        if (left === undefined && right === undefined) {
            return '';
        }
        const parts = new Array();
        if (left !== undefined) {
            parts.push(left);
        }
        if (right !== undefined) {
            parts.push(right);
        }
        // Some case analysis to produce minimal expressions
        if (parts.length === 1) {
            return parts[0];
        }
        if (parts.length === 2 && typeof parts[0] === 'string' && typeof parts[1] === 'string') {
            return parts[0] + parts[1];
        }
        // Otherwise return a Join intrinsic (already in the target document language to avoid taking
        // circular dependencies on FnJoin & friends)
        return fnJoinConcat(parts);
    }
}
exports.CloudFormationLang = CloudFormationLang;
/**
 * Return a CFN intrinsic mass concatting any number of CloudFormation expressions
 */
function fnJoinConcat(parts) {
    return { 'Fn::Join': ['', minimalCloudFormationJoin('', parts)] };
}
/**
 * Perform a JSON.stringify()-like operation, except aware of Tokens and CloudFormation intrincics
 *
 * Tokens will be resolved and if any resolve to CloudFormation intrinsics, the intrinsics
 * will be lifted to the top of a giant `{ Fn::Join }` expression.
 *
 * If Tokens resolve to primitive types (for example, by using Lazies), we'll
 * use the primitive type to determine how to encode the value into the JSON.
 *
 * If Tokens resolve to CloudFormation intrinsics, we'll use the type of the encoded
 * value as a type hint to determine how to encode the value into the JSON. The difference
 * is that we add quotes (") around strings, and don't add anything around non-strings.
 *
 * The following structure:
 *
 *    { SomeAttr: resource.someAttr }
 *
 * Will JSONify to either:
 *
 *    '{ "SomeAttr": "' ++ { Fn::GetAtt: [Resource, SomeAttr] } ++ '" }'
 * or '{ "SomeAttr": ' ++ { Fn::GetAtt: [Resource, SomeAttr] } ++ ' }'
 *
 * Depending on whether `someAttr` is type-hinted to be a string or not.
 *
 * (Where ++ is the CloudFormation string-concat operation (`{ Fn::Join }`).
 *
 * -----------------------
 *
 * This work requires 2 features from the `resolve()` function:
 *
 * - INTRINSICS TYPE HINTS: intrinsics are represented by values like
 *   `{ Ref: 'XYZ' }`. These values can reference either a string or a list/number at
 *   deploy time, and from the value alone there's no way to know which. We need
 *   to know the type to know whether to JSONify this reference to:
 *
 *      '{ "referencedValue": "' ++ { Ref: XYZ } ++ '"}'
 *   or '{ "referencedValue": ' ++ { Ref: XYZ } ++ '}'
 *
 *   I.e., whether or not we need to enclose the reference in quotes or not.
 *
 *   We COULD have done this by resolving one token at a time, and looking at the
 *   type of the encoded token we were resolving to obtain a type hint. However,
 *   the `resolve()` and Token system resist a level-at-a-time resolve
 *   operation: because of the existence of post-processors, we must have done a
 *   complete recursive resolution of a token before we can look at its result
 *   (after which any type information about the sources of nested resolved
 *   values is lost).
 *
 *   To fix this, "type hints" have been added to the `resolve()` function,
 *   giving an idea of the type of the source value for compplex result values.
 *   This only works for objects (not strings and numbers) but fortunately
 *   we only care about the types of intrinsics, which are always complex values.
 *
 *   Type hinting could have been added to the `IResolvable` protocol as well,
 *   but for now we just use the type of an encoded value as a type hint. That way
 *   we don't need to annotate anything more at the L1 level--we will use the type
 *   encodings added by construct authors at the L2 levels. L1 users can escape the
 *   default decision of "string" by using `Token.asList()`.
 *
 * - COMPLEX KEYS: since tokens can be string-encoded, we can use string-encoded tokens
 *   as the keys in JavaScript objects. However, after resolution, those string-encoded
 *   tokens could resolve to intrinsics (`{ Ref: ... }`), which CANNOT be stored in
 *   JavaScript objects anymore.
 *
 *   We therefore need a protocol to store the resolved values somewhere in the JavaScript
 *   type model,  which can be returned by `resolve()`, and interpreted by `tokenAwareStringify()`
 *   to produce the correct JSON.
 *
 *   And example will quickly show the point:
 *
 *    User writes:
 *       { [resource.resourceName]: 'SomeValue' }
 *    ------ string actually looks like ------>
 *       { '${Token[1234]}': 'SomeValue' }
 *    ------ resolve ------->
 *       { '$IntrinsicKey$0': [ {Ref: Resource}, 'SomeValue' ] }
 *    ------ tokenAwareStringify ------->
 *       '{ "' ++ { Ref: Resource } ++ '": "SomeValue" }'
 */
function tokenAwareStringify(root, space, ctx) {
    let indent = 0;
    const ret = new Array();
    // First completely resolve the tree, then encode to JSON while respecting the type
    // hints we got for the resolved intrinsics.
    recurse(ctx.resolve(root, { allowIntrinsicKeys: true }));
    switch (ret.length) {
        case 0: return undefined;
        case 1: return renderSegment(ret[0]);
        default:
            return fnJoinConcat(ret.map(renderSegment));
    }
    /**
     * Stringify a JSON element
     */
    function recurse(obj) {
        if (obj === undefined) {
            return;
        }
        if (token_1.Token.isUnresolved(obj)) {
            throw new Error('This shouldnt happen anymore');
        }
        if (Array.isArray(obj)) {
            return renderCollection('[', ']', obj, recurse);
        }
        if (typeof obj === 'object' && obj != null && !(obj instanceof Date)) {
            // Treat as an intrinsic if this LOOKS like a CFN intrinsic (`{ Ref: ... }`)
            // AND it's the result of a token resolution. Otherwise, we just treat this
            // value as a regular old JSON object (that happens to look a lot like an intrinsic).
            if (isIntrinsic(obj) && resolve_1.resolvedTypeHint(obj)) {
                renderIntrinsic(obj);
                return;
            }
            return renderCollection('{', '}', definedEntries(obj), ([key, value]) => {
                if (key.startsWith(resolve_1.INTRINSIC_KEY_PREFIX)) {
                    [key, value] = value;
                }
                recurse(key);
                pushLiteral(prettyPunctuation(':'));
                recurse(value);
            });
        }
        // Otherwise we have a scalar, defer to JSON.stringify()s serialization
        pushLiteral(JSON.stringify(obj));
    }
    /**
     * Render an object or list
     */
    function renderCollection(pre, post, xs, each) {
        pushLiteral(pre);
        indent += space;
        let atLeastOne = false;
        for (const [comma, item] of sepIter(xs)) {
            if (comma) {
                pushLiteral(',');
            }
            pushLineBreak();
            each(item);
            atLeastOne = true;
        }
        indent -= space;
        if (atLeastOne) {
            pushLineBreak();
        }
        pushLiteral(post);
    }
    function renderIntrinsic(intrinsic) {
        switch (resolve_1.resolvedTypeHint(intrinsic)) {
            case resolve_1.ResolutionTypeHint.STRING:
                pushLiteral('"');
                pushIntrinsic(deepQuoteStringLiterals(intrinsic));
                pushLiteral('"');
                return;
            case resolve_1.ResolutionTypeHint.LIST:
                // We need this to look like:
                //
                //    '{"listValue":' ++ STRINGIFY(CFN_EVAL({ Ref: MyList })) ++ '}'
                //
                // However, STRINGIFY would need to execute at CloudFormation deployment time, and that doesn't exist.
                //
                // We could *ALMOST* use:
                //
                //   '{"listValue":["' ++ JOIN('","', { Ref: MyList }) ++ '"]}'
                //
                // But that has the unfortunate side effect that if `CFN_EVAL({ Ref: MyList }) == []`, then it would
                // evaluate to `[""]`, which is a different value. Since CloudFormation does not have arbitrary
                // conditionals there's no way to deal with this case properly.
                //
                // Therefore, if we encounter lists we need to defer to a custom resource to handle
                // them properly at deploy time.
                pushIntrinsic(cfn_utils_provider_1.CfnUtils.stringify(stack_1.Stack.of(ctx.scope), `CdkJsonStringify${stringifyCounter++}`, intrinsic));
                return;
            case resolve_1.ResolutionTypeHint.NUMBER:
                pushIntrinsic(intrinsic);
                return;
        }
        throw new Error(`Unexpected type hint: ${resolve_1.resolvedTypeHint(intrinsic)}`);
    }
    /**
     * Push a literal onto the current segment if it's also a literal, otherwise open a new Segment
     */
    function pushLiteral(lit) {
        let last = ret[ret.length - 1];
        if ((last === null || last === void 0 ? void 0 : last.type) !== 'literal') {
            last = { type: 'literal', parts: [] };
            ret.push(last);
        }
        last.parts.push(lit);
    }
    /**
     * Add a new intrinsic segment
     */
    function pushIntrinsic(intrinsic) {
        ret.push({ type: 'intrinsic', intrinsic });
    }
    /**
     * Push a line break if we are pretty-printing, otherwise don't
     */
    function pushLineBreak() {
        if (space > 0) {
            pushLiteral(`\n${' '.repeat(indent)}`);
        }
    }
    /**
     * Add a space after the punctuation if we are pretty-printing, no space if not
     */
    function prettyPunctuation(punc) {
        return space > 0 ? `${punc} ` : punc;
    }
}
/**
 * Render a segment
 */
function renderSegment(s) {
    switch (s.type) {
        case 'literal': return s.parts.join('');
        case 'intrinsic': return s.intrinsic;
    }
}
const CLOUDFORMATION_CONCAT = {
    join(left, right) {
        return CloudFormationLang.concat(left, right);
    },
};
/**
 * Default Token resolver for CloudFormation templates
 */
exports.CLOUDFORMATION_TOKEN_RESOLVER = new resolvable_1.DefaultTokenResolver(CLOUDFORMATION_CONCAT);
/**
 * Do an intelligent CloudFormation join on the given values, producing a minimal expression
 */
function minimalCloudFormationJoin(delimiter, values) {
    let i = 0;
    while (i < values.length) {
        const el = values[i];
        if (isSplicableFnJoinIntrinsic(el)) {
            values.splice(i, 1, ...el['Fn::Join'][1]);
        }
        else if (i > 0 && isPlainString(values[i - 1]) && isPlainString(values[i])) {
            values[i - 1] += delimiter + values[i];
            values.splice(i, 1);
        }
        else {
            i += 1;
        }
    }
    return values;
    function isPlainString(obj) {
        return typeof obj === 'string' && !token_1.Token.isUnresolved(obj);
    }
    function isSplicableFnJoinIntrinsic(obj) {
        if (!isIntrinsic(obj)) {
            return false;
        }
        if (Object.keys(obj)[0] !== 'Fn::Join') {
            return false;
        }
        const [delim, list] = obj['Fn::Join'];
        if (delim !== delimiter) {
            return false;
        }
        if (token_1.Token.isUnresolved(list)) {
            return false;
        }
        if (!Array.isArray(list)) {
            return false;
        }
        return true;
    }
}
exports.minimalCloudFormationJoin = minimalCloudFormationJoin;
/**
 * Return whether the given value represents a CloudFormation intrinsic
 */
function isIntrinsic(x) {
    if (Array.isArray(x) || x === null || typeof x !== 'object') {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === 'Ref' || isNameOfCloudFormationIntrinsic(keys[0]);
}
function isNameOfCloudFormationIntrinsic(name) {
    if (!name.startsWith('Fn::')) {
        return false;
    }
    // these are 'fake' intrinsics, only usable inside the parameter overrides of a CFN CodePipeline Action
    return name !== 'Fn::GetArtifactAtt' && name !== 'Fn::GetParam';
}
exports.isNameOfCloudFormationIntrinsic = isNameOfCloudFormationIntrinsic;
/**
 * Separated iterator
 */
function* sepIter(xs) {
    let comma = false;
    for (const item of xs) {
        yield [comma, item];
        comma = true;
    }
}
/**
 * Object.entries() but skipping undefined values
 */
function* definedEntries(xs) {
    for (const [key, value] of Object.entries(xs)) {
        if (value !== undefined) {
            yield [key, value];
        }
    }
}
/**
 * Quote string literals inside an intrinsic
 *
 * Formally, this should only match string literals that will be interpreted as
 * string literals. Fortunately, the strings that should NOT be quoted are
 * Logical IDs and attribute names, which cannot contain quotes anyway. Hence,
 * we can get away not caring about the distinction and just quoting everything.
 */
function deepQuoteStringLiterals(x) {
    if (Array.isArray(x)) {
        return x.map(deepQuoteStringLiterals);
    }
    if (typeof x === 'object' && x != null) {
        const ret = {};
        for (const [key, value] of Object.entries(x)) {
            ret[deepQuoteStringLiterals(key)] = deepQuoteStringLiterals(value);
        }
        return ret;
    }
    if (typeof x === 'string') {
        return quoteString(x);
    }
    return x;
}
/**
 * Quote the characters inside a string, for use inside toJSON
 */
function quoteString(s) {
    s = JSON.stringify(s);
    return s.substring(1, s.length - 1);
}
let stringifyCounter = 1;
//# sourceMappingURL=data:application/json;base64,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