'''
Core module for the API.
'''

from __future__ import annotations

import json
import requests

import time
from random import randint
from datetime import datetime

from typing import Self
from dataclasses import dataclass

from onlyfakes import consts

rand_string = lambda le: ''.join(map(str, (randint(0, 9) for _ in range(le))))


@dataclass
class Seed:
    '''
    Represents a prompt seed.
    '''
    
    api: API = None
    value: int = None
    freeze: bool = False
    
    def gen(self) -> int:
        '''
        Generate the seed.
        '''
        
        # Generate new seed
        if not self.freeze:
            self.value = int(rand_string(8))
        
        return self.value

@dataclass
class Image:
    '''
    Represents an image generated by the backend.
    '''
    
    api: API = None
    url: str = None
    
    def get_bytes(self) -> bytes:
        '''
        Download the image to ram.
        '''
        
        return self.api.session.get(self.url).content
    
    def download(self, path: str) -> None:
        '''
        Download the image locally.
        '''
        
        if '{ext}' in path:
            # Parse extension and format to path
            
            ext = self.url.split('.')[-1]
            path = path.format(ext = ext)
        
        with open(path, 'wb') as file:
            file.write(self.get_bytes())


@dataclass
class Prompt:
    '''
    Represents a prompt that can be sent to the backend.
    '''
    
    api: API = None
    engine: consts.Engine = None
    
    positive: str | tuple | list = ''
    negative: str | tuple | list = ''
    
    seed: Seed = None
    width: int = 512
    height: int = 640
    cfg: int | float = 7.5
    
    @classmethod
    def new(cls,
            prompt: str,
            negative_prompt: str,
            engine: consts.Engine,
            image_size: tuple[int] = (512, 640),
            classifer_free_guidance = 7.5) -> Self:
        '''
        Create a new prompt.
        '''
        
        return cls(
            api = None,
            engine = engine,
            positive = prompt,
            negative = negative_prompt,
            width = image_size[0],
            height = image_size[1],
            cfg = classifer_free_guidance
        )
    
    def __post_init__(self) -> None:
        '''
        Post prompt initialisation.
        '''
        
        # Generate seed if needed
        if self.seed is None:
            self.seed = Seed()
        
        if len(self.positive) > consts.char_limit or \
            len(self.negative) > consts.char_limit:
                
                raise Exception('Prompt length must be equal or lower to: ', consts.char_limit)
    
    def dictify(self) -> dict:
        '''
        Transform object to dict for the backend.
        '''
        
        return dict(cfg = self.cfg,
                    engine = self.engine,
                    fullPrompt = self.positive,
                    negativePrompt = self.negative,
                    seed = self.seed.gen(),
                    size = dict(width = str(self.width),
                                height = str(self.height)))

    def submit(self, api: API = None) -> dict:
        '''
        Shorthand for submiting a prompt
        from the API class.
        '''
        
        # Update api if given
        if api: self.api = api
        
        if self.api is None:
            raise ValueError('You need to plug this prompt to an API first.')
        
        return self.api.submit(self)

    def generate(self, api: API = None, **kwargs) -> Image:
        '''
        Shorthand for generating a prompt
        from the API class.
        '''
        
        # Update api if given
        if api: self.api = api
        
        if self.api is None:
            raise ValueError('You need to plug this prompt to an API first.')
        
        return self.api.generate(self, **kwargs)


class API:
    def __init__(self) -> None:
        '''
        Represent an API for the target.
        '''
        
        self.session = requests.Session()
        self.track_id = None
    
    def submit(self, prompt: Prompt) -> dict:
        '''
        Submit a prompt to the backend and return
        its position in queue and fetch token.
        '''
        
        data = prompt.dictify()
        
        # Create and inject track id
        nano = rand_string(5)
        self.track_id = datetime.now().strftime('%Y-%m-%d-%f') + nano
        data['trackId'] = self.track_id
        
        raw = json.dumps(data)
        
        response = self.session.post(
            url = consts.target + 'createImage',
            data = raw,
            timeout = 3000,
            headers = consts.headers
        )
        
        parsed = response.json()
        self.session.cookies.set('fetchToken', parsed.get('fetchToken'))
        return parsed
    
    def check(self, timeout = 3000) -> bool | dict:
        '''
        Try to catch the image and returns its data.
        Else, returns False.
        '''
        
        data = self.session.post(
            consts.target + 'getImageData',
            data = json.dumps({'trackId': self.track_id}),
            headers = consts.headers,
            timeout = timeout
        ).json()
        
        if data.get('status') == 'success':
            return data
        
        return False
    
    def generate(self,
                 prompt: Prompt,
                 interval: int = 2 * 60,
                 timeout: int = 4 * 60,
                 verbose: bool = False) -> Image:
        '''
        Submit a prompt to the backend. Returns the image when
        it has been generated.
        '''
        
        self.submit(prompt)
        start = time.time()
        
        if verbose:
            print('\033[94m[ GEN ]\033[0m Prompt submited')
        
        while 1:
            # Wait for interval
            time.sleep(interval)
            
            if verbose:
                print('\033[94m[ GEN ]\033[0m Checking image status')
            
            # Check for timeout
            if time.time() - start > timeout:
                raise TimeoutError('Max timeout passed')
            
            # Ask status to the backend
            data = self.check()
            if data == False: continue
            
            if verbose:
                print('\033[94m[ GEN ]\033[0m Operation succedeed')
            
            return Image(self, data['imageUrl'])

    def new(self,
            prompt: str,
            negative_prompt: str,
            engine: consts.Engine,
            image_size: tuple[int] = (512, 640),
            classifer_free_guidance = 7.5) -> Prompt:
        '''
        Create a new prompt.
        '''
        
        return Prompt(
            api = self,
            engine = engine,
            positive = prompt,
            negative = negative_prompt,
            width = image_size[0],
            height = image_size[1],
            cfg = classifer_free_guidance
        )

# EOF