"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsAppSpec = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Represents an AppSpec to be used for ECS services.
 *
 * see: https://docs.aws.amazon.com/codedeploy/latest/userguide/reference-appspec-file-structure-resources.html#reference-appspec-file-structure-resources-ecs
 */
class EcsAppSpec {
    constructor(targetService, hooks) {
        this.targetService = targetService;
        this.hooks = hooks;
    }
    /**
     * Render JSON string for this AppSpec to be used
     *
     * @returns string representation of this AppSpec
     */
    toString() {
        const appSpec = {
            version: '0.0',
            Resources: [{
                    TargetService: {
                        Type: 'AWS::ECS::Service',
                        Properties: {
                            TaskDefinition: this.targetService.taskDefinition.taskDefinitionArn,
                            LoadBalancerInfo: {
                                ContainerName: this.targetService.containerName,
                                ContainerPort: this.targetService.containerPort,
                            },
                            PlatformVersion: this.targetService.platformVersion,
                            NetworkConfiguration: this.configureAwsVpcNetworkingWithSecurityGroups(this.targetService.awsvpcConfiguration),
                            CapacityProviderStrategy: this.targetService.capacityProviderStrategy?.map(capacityProviderStrategy => {
                                return {
                                    Base: capacityProviderStrategy.base,
                                    CapacityProvider: capacityProviderStrategy.capacityProvider,
                                    Weight: capacityProviderStrategy.weight,
                                };
                            }),
                        },
                    },
                }],
            ...this.hooksSection(),
        };
        return JSON.stringify(appSpec);
    }
    hooksSection() {
        if (this.hooks == undefined) {
            return {};
        }
        const hooks = this.hooks;
        if (this.hooks.beforeInstall == undefined &&
            this.hooks.afterInstall == undefined &&
            this.hooks.afterAllowTestTraffic == undefined &&
            this.hooks.beforeAllowTraffic == undefined &&
            this.hooks.afterAllowTraffic == undefined) {
            return {};
        }
        const capitalize = (s) => s && s[0].toUpperCase() + s.slice(1);
        const hook = (name) => {
            const hookValue = hooks[name];
            if (hookValue == undefined) {
                return [];
            }
            else {
                return [{
                        [capitalize(name)]: typeof hookValue === 'string'
                            ? hookValue
                            : hookValue.functionArn,
                    }];
            }
        };
        const beforeInstall = hook('beforeInstall');
        const afterInstall = hook('afterInstall');
        const afterAllowTestTraffic = hook('afterAllowTestTraffic');
        const beforeAllowTraffic = hook('beforeAllowTraffic');
        const afterAllowTraffic = hook('afterAllowTraffic');
        return {
            Hooks: [
                ...beforeInstall,
                ...afterInstall,
                ...afterAllowTestTraffic,
                ...beforeAllowTraffic,
                ...afterAllowTraffic,
            ],
        };
    }
    configureAwsVpcNetworkingWithSecurityGroups(awsvpcConfiguration) {
        if (!awsvpcConfiguration) {
            return undefined;
        }
        return {
            awsvpcConfiguration: {
                assignPublicIp: awsvpcConfiguration.assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: awsvpcConfiguration.vpc.selectSubnets(awsvpcConfiguration.vpcSubnets).subnetIds,
                securityGroups: awsvpcConfiguration.securityGroups?.map((sg) => sg.securityGroupId),
            },
        };
    }
}
exports.EcsAppSpec = EcsAppSpec;
_a = JSII_RTTI_SYMBOL_1;
EcsAppSpec[_a] = { fqn: "@cdklabs/cdk-ecs-codedeploy.EcsAppSpec", version: "0.0.325" };
//# sourceMappingURL=data:application/json;base64,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