import os
import sys
import importlib.util
import inspect
import json
from typing import List, Dict, Any


def snake_case(name: str) -> str:
    import re
    s1 = re.sub('(.)([A-Z][a-z]+)', r'\1_\2', name)
    return re.sub('([a-z0-9])([A-Z])', r'\1_\2', s1).lower()


def get_tool_classes(module) -> List[type]:
    """Return all classes in module that inherit from BaseTool."""
    from agency_swarm.tools import BaseTool
    return [cls for name, cls in inspect.getmembers(module, inspect.isclass)
            if issubclass(cls, BaseTool) and cls is not BaseTool]


def extract_schema(tool_cls) -> Dict[str, Any]:
    # Name
    name = snake_case(tool_cls.__name__)
    # Description
    description = inspect.getdoc(tool_cls) or ""
    # Parameters
    properties = {}
    required = []
    # Support both Pydantic v1 and v2
    if hasattr(tool_cls, "model_fields"):  # Pydantic v2
        fields = tool_cls.model_fields.items()
        for field_name, model_field in fields:
            field_type = model_field.annotation
            field_type_name = getattr(field_type, "__name__", str(field_type))
            type_map = {"str": "string", "int": "integer", "float": "number", "bool": "boolean", "dict": "object", "list": "array"}
            json_type = type_map.get(field_type_name, "string")
            properties[field_name] = {
                "type": json_type,
                "description": model_field.description or ""
            }
            if model_field.is_required:
                required.append(field_name)
    else:  # Pydantic v1
        fields = getattr(tool_cls, "__fields__", {}).items()
        for field_name, model_field in fields:
            field_type = model_field.outer_type_
            field_type_name = getattr(field_type, "__name__", str(field_type))
            type_map = {"str": "string", "int": "integer", "float": "number", "bool": "boolean", "dict": "object", "list": "array"}
            json_type = type_map.get(field_type_name, "string")
            properties[field_name] = {
                "type": json_type,
                "description": model_field.field_info.description or ""
            }
            if model_field.required:
                required.append(field_name)
    return {
        "type": "function",
        "function": {
            "name": name,
            "description": description,
            "parameters": {
                "type": "object",
                "properties": properties,
                "required": required
            }
        }
    }


def load_module_from_path(path: str):
    module_name = os.path.splitext(os.path.basename(path))[0]
    spec = importlib.util.spec_from_file_location(module_name, path)
    module = importlib.util.module_from_spec(spec)
    sys.modules[module_name] = module
    spec.loader.exec_module(module)
    return module


def generate_schema_from_folder(folder: str) -> List[Dict[str, Any]]:
    schemas = []
    for fname in os.listdir(folder):
        if fname.endswith(".py") and not fname.startswith("__"):
            path = os.path.join(folder, fname)
            module = load_module_from_path(path)
            for tool_cls in get_tool_classes(module):
                schemas.append(extract_schema(tool_cls))
    return schemas


def generate_schema_from_files(files: List[str]) -> List[Dict[str, Any]]:
    schemas = []
    for path in files:
        module = load_module_from_path(path)
        for tool_cls in get_tool_classes(module):
            schemas.append(extract_schema(tool_cls))
    return schemas

def generate_schema(files: List[str] = None, folder: str = None):
    if files and folder:
        return "You should pass one of the two arguments"
    if files:
        return generate_schema_from_files(files)

    if folder:
        return generate_schema_from_folder(folder)
    