import unittest
from unittest.mock import Mock, patch, MagicMock

multiprocessing = __import__('multiprocessing').get_context('fork')

from genie.harness._commons_internal import pcall_configure


manager = multiprocessing.Manager()


def mock_configure(*args, device=None, **kwargs):
    device.configure.calls.append(kwargs)


class MockCopyFile:

    def copyfile(*args, device=None, **kwargs):
        device.calls.append(kwargs)


class FileUtilsMock:

    def from_device(self, *args, **kwargs):
        return MockCopyFile()


class TestGenieHarnessConnect(unittest.TestCase):
    
    @patch('genie.harness._commons_internal.connect_device')
    @patch('genie.harness._commons_internal.pcall')
    @patch('genie.harness._commons_internal.ThreadPoolExecutor')
    def test_connect_parallel(self, mock_thread_pool, mock_pcall, mock_connect_device):
        from genie.harness._commons_internal import connect
        
        # Configure thread pool mock
        mock_executor = MagicMock()
        mock_thread_pool.return_value.__enter__.return_value = mock_executor
        
        # Setup test data with proper structure
        mock_self = Mock()
        mock_self.parent = Mock()
        mock_self.parent.mapping_data = {'devices': {'device1': {}, 'device2': {}}}
        mock_self.parent.url = "https://test.com"
        
        # Create properly structured device mocks
        mock_device1 = Mock()
        mock_device1.name = 'device1'
        mock_device1.type = 'router'
        # Empty connections dict to support iteration
        mock_device1.connections = {}
        mock_device1.management_interface = None
        
        mock_device2 = Mock()
        mock_device2.name = 'device2'
        mock_device2.type = 'router'
        # Empty connections dict to support iteration
        mock_device2.connections = {}
        mock_device2.management_interface = None
        
        mock_testbed = Mock()
        mock_testbed.devices = {'device1': mock_device1, 'device2': mock_device2}
        
        mock_steps = Mock()
        
        # Make connect_device function properly update summary_information
        def side_effect_connect_device(device, pool_size, netconf_pool_size, summary_information):
            summary_information['connected'] = [('cli', 'default', None)]
        mock_connect_device.side_effect = side_effect_connect_device
        
        # Test with parallel=True (default)
        connect(mock_self, mock_testbed, mock_steps)
        mock_executor.map.assert_called_once()
        mock_connect_device.assert_not_called()
        
        # Reset mocks
        mock_executor.reset_mock()
        mock_connect_device.reset_mock()
        
        # Test with parallel=False
        connect(mock_self, mock_testbed, mock_steps, parallel=False)
        mock_executor.map.assert_not_called()
        self.assertTrue(mock_connect_device.called)


class TestGenieHarnessConfigure(unittest.TestCase):

    @patch('genie.harness._commons_internal._configure_on_device', new=mock_configure)
    def test_pcall_configure_jinja2(self):
        from genie.harness._commons_internal import pcall_configure
        device1 = Mock()
        device1.configure = Mock()
        device1.configure.calls = manager.list()
        device2 = Mock()
        device2.configure = Mock()
        device2.configure.calls = manager.list()
        device_dict = {
            "R1": [
                {
                    "device": device1,
                    "sleep": 1,
                    "rendered": "config text 1",
                    "type": "jinja",
                    "config_file": "cfg.j2",
                    "verify": None
                },
                {
                    "device": device1,
                    "sleep": 0,
                    "rendered": "config text 2",
                    "type": "jinja",
                    "config_file": "cfg.j2",
                    "configure_arguments": {"bulk": True},
                    "verify": None
                }
            ],
            "R2": [
                {
                    "device": device2,
                    "sleep": 0,
                    "rendered": "config text 3",
                    "type": "jinja",
                    "config_file": "cfg.j2",
                    "verify": None
                }
            ]
        }
        pcall_configure(device_dict)
        self.assertEqual(list(device1.configure.calls), [
            {'rendered': 'config text 1', 'config_file': 'cfg.j2', 'type': 'jinja'},
            {'rendered': 'config text 2', 'config_file': 'cfg.j2', 'type': 'jinja', 'configure_arguments': {'bulk': True}}
        ])
        self.assertEqual(list(device2.configure.calls), [
            {'rendered': 'config text 3', 'config_file': 'cfg.j2', 'type': 'jinja'}
        ])

    @patch('genie.harness._commons_internal.FileUtils', new=FileUtilsMock)
    def test_pcall_configure_copyfile(self):
        device1 = Mock()
        device1.calls = manager.list()
        device2 = Mock()
        device2.calls = manager.list()
        device_dict = {
            'R1': [{
                'device': device1,
                'source': 'http://server/file1.txt',
                'destination': 'running-config',
                'invalid': [],
                'sleep': 1,
                'config_file': 'file.txt',
                'verify': None,
            }],
            'R2': [{
                'device': device2,
                'source': 'http://server/file2.txt',
                'destination': 'running-config',
                'invalid': [],
                'sleep': 1,
                'config_file': 'file.txt',
                'verify': None,
            }]
        }
        pcall_configure(device_dict)
        self.assertEqual(list(device1.calls), [{
            'config_file': 'file.txt',
            'destination': 'running-config',
            'invalid': [],
            'source': 'http://server/file1.txt'
        }])
        self.assertEqual(list(device2.calls), [{
            'config_file': 'file.txt',
            'destination': 'running-config',
            'invalid': [],
            'source': 'http://server/file2.txt'
        }])


if __name__ == "__main__":
    unittest.main()
