import os
from datetime import datetime
from typing import Dict, Any, List
import xml.dom.minidom as minidom
import xml.etree.ElementTree as ET
from .exceptions import OutputGenerationError
from .utils.tree_generator import generate_tree_string

PLAIN_SEPARATOR = "=" * 16
PLAIN_LONG_SEPARATOR = "=" * 64


def generate_output(
    root_dir: str,
    config: Dict[str, Any],
    sanitized_files: List[Dict[str, str]],
    all_file_paths: List[str],
    output_path: str,
) -> None:
    """
    Generate the output file based on the specified configuration.

    Args:
        root_dir (str): The root directory of the repository.
        config (Dict[str, Any]): The configuration dictionary.
        sanitized_files (List[Dict[str, str]]): List of sanitized file contents.
        all_file_paths (List[str]): List of all file paths in the repository.
        output_path (str): The path to the output file.

    Raises:
        OutputGenerationError: If there's an error during output generation.
    """
    common_data = generate_common_data(config, all_file_paths, sanitized_files)

    try:
        if config["output"]["style"] == "xml":
            output = generate_xml_output(common_data)
        else:
            output = generate_plain_output(common_data)

        with open(output_path, "w", encoding="utf-8") as f:
            f.write(output)
    except Exception as e:
        raise OutputGenerationError(f"Error generating output: {str(e)}")


def generate_common_data(
    config: Dict[str, Any],
    all_file_paths: List[str],
    sanitized_files: List[Dict[str, str]],
) -> Dict[str, Any]:
    return {
        "generationDate": datetime.now().isoformat(),
        "treeString": generate_tree_string(all_file_paths),
        "sanitizedFiles": sanitized_files,
        "config": config,
    }


def generate_plain_output(data: Dict[str, Any]) -> str:
    generationDate = data["generationDate"]
    treeString = data["treeString"]
    sanitizedFiles = data["sanitizedFiles"]
    config = data["config"]

    output = f"{PLAIN_LONG_SEPARATOR}\n"
    output += "RepopackPy Output File\n"
    output += f"{PLAIN_LONG_SEPARATOR}\n\n"
    output += f"This file was generated by RepopackPy on: {generationDate}\n\n"
    output += "Purpose:\n"
    output += "--------\n"
    output += "This file contains a packed representation of the entire repository's contents.\n"
    output += "It is designed to be easily consumable by AI systems for analysis, code review,\n"
    output += "or other automated processes.\n\n"
    output += "File Format:\n"
    output += "------------\n"
    output += "The content is organized as follows:\n"
    output += "1. This header section\n"
    output += "2. Repository structure\n"
    output += "3. Multiple file entries, each consisting of:\n"
    output += "  a. A separator line (================)\n"
    output += "  b. The file path (File: path/to/file)\n"
    output += "  c. Another separator line\n"
    output += "  d. The full contents of the file\n"
    output += "  e. A blank line\n\n"
    output += "Usage Guidelines:\n"
    output += "-----------------\n"
    output += "1. This file should be treated as read-only. Any changes should be made to the\n"
    output += "  original repository files, not this packed version.\n"
    output += '2. When processing this file, use the separators and "File:" markers to\n'
    output += "  distinguish between different files in the repository.\n"
    output += "3. Be aware that this file may contain sensitive information. Handle it with\n"
    output += "  the same level of security as you would the original repository.\n\n"
    output += "Notes:\n"
    output += "------\n"
    output += "- Some files may have been excluded based on .gitignore rules and RepopackPy's\n"
    output += "  configuration.\n"
    output += "- Binary files are not included in this packed representation. Please refer to\n"
    output += "  the Repository Structure section for a complete list of file paths, including\n"
    output += "  binary files.\n"
    if config["output"]["remove_comments"]:
        output += "- Code comments have been removed.\n"
    if config["output"]["show_line_numbers"]:
        output += "- Line numbers have been added to the beginning of each line.\n"
    output += "\n"
    output += "For more information about RepopackPy, visit: https://github.com/abinthomasonline/repopack-py\n\n"

    if config["output"].get("header_text"):
        output += "Additional User-Provided Header:\n"
        output += "--------------------------------\n"
        output += f"{config['output']['header_text']}\n\n"

    output += f"{PLAIN_LONG_SEPARATOR}\n"
    output += "Repository Structure\n"
    output += f"{PLAIN_LONG_SEPARATOR}\n"
    output += f"{treeString}\n\n"
    output += f"{PLAIN_LONG_SEPARATOR}\n"
    output += "Repository Files\n"
    output += f"{PLAIN_LONG_SEPARATOR}\n\n"

    for file in sanitizedFiles:
        output += f"{PLAIN_SEPARATOR}\n"
        output += f"File: {file['path']}\n"
        output += f"{PLAIN_SEPARATOR}\n"
        output += f"{file['content']}\n\n"

    return output.strip() + "\n"


def generate_xml_output(data: Dict[str, Any]) -> str:
    generationDate = data["generationDate"]
    treeString = data["treeString"]
    sanitizedFiles = data["sanitizedFiles"]
    config = data["config"]

    xml = "<summary>\n\n"
    xml += "<header>\n"
    xml += "RepopackPy Output File\n"
    xml += f"This file was generated by RepopackPy on: {generationDate}\n"
    xml += "</header>\n\n"
    xml += "<purpose>\n"
    xml += "This file contains a packed representation of the entire repository's contents.\n"
    xml += "It is designed to be easily consumable by AI systems for analysis, code review,\n"
    xml += "or other automated processes.\n"
    xml += "</purpose>\n\n"
    xml += "<file_format>\n"
    xml += "The content is organized as follows:\n"
    xml += "1. This summary section\n"
    xml += "2. Repository structure\n"
    xml += "3. Repository files, each consisting of:\n"
    xml += "    - File path as an attribute\n"
    xml += "    - Full contents of the file\n"
    xml += "</file_format>\n\n"
    xml += "<usage_guidelines>\n"
    xml += "1. This file should be treated as read-only. Any changes should be made to the\n"
    xml += "    original repository files, not this packed version.\n"
    xml += "2. When processing this file, use the file path attributes to distinguish\n"
    xml += "    between different files in the repository.\n"
    xml += "3. Be aware that this file may contain sensitive information. Handle it with\n"
    xml += "    the same level of security as you would the original repository.\n"
    xml += "</usage_guidelines>\n\n"
    xml += "<notes>\n"
    xml += "- Some files may have been excluded based on .gitignore rules and RepopackPy's\n"
    xml += "  configuration.\n"
    xml += "- Binary files are not included in this packed representation.\n"
    if config["output"]["remove_comments"]:
        xml += "- Code comments have been removed.\n"
    if config["output"]["show_line_numbers"]:
        xml += "- Line numbers have been added to the beginning of each line.\n"
    xml += "</notes>\n\n"
    xml += "<additional_info>\n"
    xml += "For more information about RepopackPy, visit: https://github.com/abinthomasonline/repopack-py\n"
    xml += "</additional_info>\n"

    if config["output"].get("header_text"):
        xml += "\n<user_provided_header>\n"
        xml += f"{config['output']['header_text']}\n"
        xml += "</user_provided_header>\n"

    xml += "\n</summary>\n\n"
    xml += "<repository_structure>\n"
    xml += f"{treeString}\n"
    xml += "</repository_structure>\n\n"
    xml += "<repository_files>\n"

    for file in sanitizedFiles:
        xml += f"\n<file path=\"{file['path']}\">\n"
        xml += f"{file['content']}\n"
        xml += "</file>\n"

    xml += "\n</repository_files>\n"

    return xml.strip() + "\n"
