"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test minimal deployment
// --------------------------------------------------------------
test('Test minimal deployment', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 1: snapshot test
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2: test for an producer function
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        FunctionName: 'producer-function'
    });
    // Assertion 3: test for a consumer function
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        FunctionName: 'consumer-function'
    });
    // Assertion 4: test for a queue
    expect(stack).toHaveResource('AWS::SQS::Queue');
    // Assertion 5: test for send-message permissions (only) on the producer function
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:GetQueueUrl"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "lambdasqslambdalambdatosqsqueue49588D68",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
    // Assertion 6: test for consume-message permissions (only) on the consumer function
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "sqs:ReceiveMessage",
                        "sqs:ChangeMessageVisibility",
                        "sqs:GetQueueUrl",
                        "sqs:DeleteMessage",
                        "sqs:GetQueueAttributes"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "lambdasqslambdalambdatosqsqueue49588D68",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing producer function
// --------------------------------------------------------------
test('Test deployment w/ existing producer function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Define existing resources
    const existingProducerFn = defaults.buildLambdaFunction(stack, {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/producer-function`),
            functionName: 'existing-producer-function'
        }
    });
    // Helper declaration
    const props = {
        existingProducerLambdaObj: existingProducerFn,
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/consumer-function`),
            functionName: 'deployed-consumer-function'
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 1: snapshot test
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2: test for the existing producer function
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        FunctionName: 'existing-producer-function'
    });
    // Assertion 3: test for the deployed consumer function
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        FunctionName: 'deployed-consumer-function'
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing consumer function
// --------------------------------------------------------------
test('Test deployment w/ existing consumer function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Define existing resources
    const existingConsumerFn = defaults.buildLambdaFunction(stack, {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/consumer-function`),
            functionName: 'existing-consumer-function'
        }
    });
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/producer-function`),
            functionName: 'deployed-producer-function'
        },
        existingConsumerLambdaObj: existingConsumerFn
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 1: snapshot test
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2: test for the deployed producer function
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        FunctionName: 'deployed-producer-function'
    });
    // Assertion 3: test for the existing consumer function
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        FunctionName: 'existing-consumer-function'
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing queue
// --------------------------------------------------------------
test('Test deployment w/ existing queue', () => {
    // Stack
    const stack = new core_1.Stack();
    // Define existing resources
    const [existingQueue] = defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        }
    });
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        existingQueueObj: existingQueue
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 1: snapshot test
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2: test for the existing queue
    expect(stack).toHaveResource('AWS::SQS::Queue', {
        QueueName: 'existing-queue'
    });
});
// --------------------------------------------------------------
// Test deployment w/ DLQ explicitly disabled
// --------------------------------------------------------------
test('Test deployment w/ DLQ explicitly disabled', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/producer-function`)
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/consumer-function`)
        },
        deployDeadLetterQueue: false,
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 1: snapshot test
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2: test for a non-existing DLQ
    expect(!assert_1.haveResourceLike('AWS::SQS::Queue', {
        RedrivePolicy: {
            deadLetterTargetArn: "a-target-arn"
        }
    }));
});
// --------------------------------------------------------------
// Test deployment w/ DLQ explicitly enabled and w/ MRC override
// --------------------------------------------------------------
test('Test deployment w/ DLQ explicitly enabled and w/ MRC override', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/producer-function`)
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/consumer-function`)
        },
        deployDeadLetterQueue: true,
        maxReceiveCount: 6
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 1: snapshot test
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2: test for an existing DLQ
    expect(assert_1.haveResourceLike('AWS::SQS::Queue', {
        RedrivePolicy: {
            deadLetterTargetArn: "a-target-arn"
        }
    }));
    // Assertion 3: test for the overridden max receive count
    expect(assert_1.haveResourceLike('AWS::SQS::Queue', {
        RedrivePolicy: {
            maxReceiveCount: 6
        }
    }));
});
// --------------------------------------------------------------
// Test overrides for producer and consumer functions
// --------------------------------------------------------------
test('Test overrides for producer and consumer functions', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 1: snapshot test
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2: test for updated runtime on producer function
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Runtime: "nodejs12.x"
    });
    // Assertion 3: test for updated runtime on consumer function
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Runtime: "nodejs12.x"
    });
});
// --------------------------------------------------------------
// Test the public pattern props
// --------------------------------------------------------------
test('Test the public pattern props', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        }
    };
    const pattern = new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 1: get the producer function
    expect(pattern.producerLambdaFunction).toBeDefined();
    // Assertion 2: get the queue
    expect(pattern.sqsQueue).toBeDefined();
    // Assertion 3: get the dead letter queue
    expect(pattern.deadLetterQueue).toBeDefined();
    // Assertion 4: get the consumer function
    expect(pattern.consumerLambdaFunction).toBeDefined();
});
//# sourceMappingURL=data:application/json;base64,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