"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workload = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const context_1 = require("./context");
const WORKLOAD_SYMBOL = Symbol.for("@shady-island/core.Workload");
const IMPORTED_STACKS = new Map();
/**
 * A collection of Stacks in an Environment representing a deployment Tier.
 *
 * Consider deriving a subclass of `Workload` and creating your `Stack` objects
 * within its constructor.
 *
 * The difference between this class and a `Stage` is that a `Stage` is meant to
 * be deployed with CDK Pipelines. This class can be used with `cdk deploy`.
 * This class also provides context loading capabilities.
 *
 * It is an anti-pattern to provide a `Workload` instance as the parent scope to
 * the `aws-cdk-lib.Stack` constructor. You should either use the
 * `createStack()` method, create your own sub-class of `Stack` and provide a
 * `Workload` instance as the parent scope, or use the `import()` method to
 * essentially _import_ a `Stack` and its constructs into a `Workload` without
 * changing its scope.
 */
class Workload extends constructs_1.Construct {
    /**
     * Return the Workload the construct is contained within, fails if there is no
     * workload up the tree.
     *
     * @param construct - The construct whose parent nodes will be searched
     * @returns The Workload containing the construct
     * @throws Error - if none of the construct's parents are a workload
     */
    static of(construct) {
        const constructStack = aws_cdk_lib_1.Stack.of(construct);
        if (IMPORTED_STACKS.has(constructStack)) {
            return IMPORTED_STACKS.get(constructStack);
        }
        const workload = construct.node.scopes
            .reverse()
            .slice(1)
            .find(Workload.isWorkload);
        if (!workload) {
            throw new Error("No workload exists in the parentage of the provided construct");
        }
        return workload;
    }
    /**
     * Test whether the given construct is a Workload.
     *
     * @param x - The value to test
     * @returns Whether the value is a Workload object.
     */
    static isWorkload(x) {
        return x !== null && typeof x === "object" && WORKLOAD_SYMBOL in x;
    }
    /**
     * Creates a new Workload.
     *
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param props - The constructor options.
     */
    constructor(scope, id, props) {
        super(scope, id);
        Object.defineProperty(this, WORKLOAD_SYMBOL, { value: true });
        const { env: { region, account } = {}, tier, workloadName, contextFile, baseDomainName, } = props;
        this.workloadName = workloadName ?? `${id}`.toLowerCase();
        this.tier = tier;
        this.publicDomainName = baseDomainName
            ? `${tier.id}.${this.workloadName}.${baseDomainName}`.toLowerCase()
            : undefined;
        this._stacks = new Map();
        const { region: stageRegion, account: stageAccount } = aws_cdk_lib_1.Stage.of(this);
        this.region = region || stageRegion;
        this.account = account || stageAccount;
        if (contextFile) {
            context_1.ContextLoader.loadContext(contextFile, this.node);
        }
    }
    /**
     * @returns The stacks created by invoking `createStack`
     */
    get stacks() {
        return [...this._stacks.values()];
    }
    /**
     * Register the provided `Stack` as being part of this `Workload`.
     *
     * @param stack - The stack to register.
     * @returns The provided Stack
     */
    registerStack(stack) {
        this.tier.applyTags(stack);
        this._stacks.set(stack.stackName, stack);
        return stack;
    }
    /**
     * Forces a return value for `Workload.of` for one or more `Stack` objects.
     *
     * Normally, a construct must be within the scope of the `Workload` instance,
     * such as a construct that is a descendant of a `Stack` returned from
     * `createStack()`.
     *
     * That means that any `Stack` instances you created in your CDK application
     * _before_ installing the `shady-island` library would not be able to be part
     * of a `Workload` unless you changed the `scope` argument of the `Stack`
     * constructor from the `App` or `Stage` to the desired `Workload` instance.
     * However, that's bad news for a `Stack` that has already been deployed to
     * CloudFormation because the resource identifier of persistent child
     * constructs (e.g. RDS databases, S3 buckets) would change.
     *
     * A successful call to this method will register the provided `Stack` objects
     * and all their construct descendants as members of that `Workload` instance.
     * Calling `Workload.of()` with any of the provided `Stack` objects or their
     * descendant constructs will return that `Workload` instance.
     *
     * If any of the `Stack` objects provided to this method already belong to a
     * different `Workload` object, or whose parent scope is not identical to the
     * parent scope of this `Workload` (i.e. the `Stage` or the `App`), an error
     * will be thrown.
     *
     * @param stacks - The `Stack` instances to import to this `Workload`
     * @throws {Error} if any of the stacks are already part of another workload
     * @throws {Error} if any of the stacks have a different parent scope
     */
    import(...stacks) {
        for (const stack of stacks) {
            if (stack.node.scope === this) {
                continue;
            }
            if (Workload.isWorkload(stack.node.scope) && stack.node.scope !== this) {
                throw new Error("The Stack is already contained within a different Workload");
            }
            if (stack.node.scope !== this.node.scope) {
                throw new Error("The Stack must be contained within the same scope as this Workload");
            }
            IMPORTED_STACKS.set(this.registerStack(stack), this);
        }
    }
    /**
     * Adds a stack to the Workload.
     *
     * This method will return a `Stack` with this Workload as its scope. By
     * default, the `stackName` property provided to the `Stack` will be this
     * Workload's `workloadName`, its `tier`, and the value of the `id`
     * parameter separated by hyphens, all in lowercase.
     *
     * @example
     *
     * const exampleDev = new Workload(app, 'Example', {
     *   tier: Tier.DEVELOPMENT,
     *   env: { account: '123456789012', region: 'us-east-1' },
     * });
     * const networkStack = exampleDev.createStack('Network', {});
     * assert.strictEqual(networkStack.stackName, 'example-dev-network').
     *
     * You can override the `env` and `stackName` properties in the `props`
     * argument if desired.
     *
     * The stack will have a `DeploymentTier` tag added, set to the tier label.
     *
     * @param id - The Stack construct id (e.g. "Network")
     * @param props - The new Stack properties
     */
    createStack(id, props) {
        const { stackName, ...options } = props ?? {};
        const newStackName = stackName ?? `${this.workloadName}-${this.tier.id}-${id}`.toLowerCase();
        const stack = new aws_cdk_lib_1.Stack(this, id, {
            stackName: newStackName,
            env: this.account && this.region
                ? { account: this.account, region: this.region }
                : undefined,
            ...options,
        });
        return this.registerStack(stack);
    }
}
exports.Workload = Workload;
_a = JSII_RTTI_SYMBOL_1;
Workload[_a] = { fqn: "shady-island.Workload", version: "0.1.53" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid29ya2xvYWQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvd29ya2xvYWQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2Q0FBb0U7QUFDcEUsMkNBQW1EO0FBQ25ELHVDQUEwQztBQUcxQyxNQUFNLGVBQWUsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7QUFtRGxFLE1BQU0sZUFBZSxHQUFHLElBQUksR0FBRyxFQUFtQixDQUFDO0FBRW5EOzs7Ozs7Ozs7Ozs7Ozs7O0dBZ0JHO0FBQ0gsTUFBYSxRQUFTLFNBQVEsc0JBQVM7SUFDckM7Ozs7Ozs7T0FPRztJQUNJLE1BQU0sQ0FBQyxFQUFFLENBQUMsU0FBcUI7UUFDcEMsTUFBTSxjQUFjLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDM0MsSUFBSSxlQUFlLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUM7WUFDeEMsT0FBTyxlQUFlLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBRSxDQUFDO1FBQzlDLENBQUM7UUFDRCxNQUFNLFFBQVEsR0FBRyxTQUFTLENBQUMsSUFBSSxDQUFDLE1BQU07YUFDbkMsT0FBTyxFQUFFO2FBQ1QsS0FBSyxDQUFDLENBQUMsQ0FBQzthQUNSLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDN0IsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2QsTUFBTSxJQUFJLEtBQUssQ0FDYiwrREFBK0QsQ0FDaEUsQ0FBQztRQUNKLENBQUM7UUFDRCxPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUFDLENBQU07UUFDN0IsT0FBTyxDQUFDLEtBQUssSUFBSSxJQUFJLE9BQU8sQ0FBQyxLQUFLLFFBQVEsSUFBSSxlQUFlLElBQUksQ0FBQyxDQUFDO0lBQ3JFLENBQUM7SUFtREQ7Ozs7OztPQU1HO0lBQ0gsWUFBbUIsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBb0I7UUFDbkUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUU5RCxNQUFNLEVBQ0osR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsRUFDN0IsSUFBSSxFQUNKLFlBQVksRUFDWixXQUFXLEVBQ1gsY0FBYyxHQUNmLEdBQUcsS0FBSyxDQUFDO1FBRVYsSUFBSSxDQUFDLFlBQVksR0FBRyxZQUFZLElBQUksR0FBRyxFQUFFLEVBQUUsQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUMxRCxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztRQUNqQixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsY0FBYztZQUNwQyxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsRUFBRSxJQUFJLElBQUksQ0FBQyxZQUFZLElBQUksY0FBYyxFQUFFLENBQUMsV0FBVyxFQUFFO1lBQ25FLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDZCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksR0FBRyxFQUFFLENBQUM7UUFFekIsTUFBTSxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsT0FBTyxFQUFFLFlBQVksRUFBRSxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBRSxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxJQUFJLFdBQVcsQ0FBQztRQUNwQyxJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sSUFBSSxZQUFZLENBQUM7UUFFdkMsSUFBSSxXQUFXLEVBQUUsQ0FBQztZQUNoQix1QkFBYSxDQUFDLFdBQVcsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3BELENBQUM7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLE1BQU07UUFDZixPQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ08sYUFBYSxDQUFDLEtBQVk7UUFDbEMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDM0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN6QyxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQTRCRztJQUNJLE1BQU0sQ0FBQyxHQUFHLE1BQWU7UUFDOUIsS0FBSyxNQUFNLEtBQUssSUFBSSxNQUFNLEVBQUUsQ0FBQztZQUMzQixJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxLQUFLLElBQUksRUFBRSxDQUFDO2dCQUM5QixTQUFTO1lBQ1gsQ0FBQztZQUNELElBQUksUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxLQUFLLElBQUksRUFBRSxDQUFDO2dCQUN2RSxNQUFNLElBQUksS0FBSyxDQUNiLDREQUE0RCxDQUM3RCxDQUFDO1lBQ0osQ0FBQztZQUNELElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLEtBQUssSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFDekMsTUFBTSxJQUFJLEtBQUssQ0FDYixvRUFBb0UsQ0FDckUsQ0FBQztZQUNKLENBQUM7WUFDRCxlQUFlLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDdkQsQ0FBQztJQUNILENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bd0JHO0lBQ0ksV0FBVyxDQUFDLEVBQVUsRUFBRSxLQUFrQjtRQUMvQyxNQUFNLEVBQUUsU0FBUyxFQUFFLEdBQUcsT0FBTyxFQUFFLEdBQUcsS0FBSyxJQUFJLEVBQUUsQ0FBQztRQUM5QyxNQUFNLFlBQVksR0FDaEIsU0FBUyxJQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLEVBQUUsQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUMxRSxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUNoQyxTQUFTLEVBQUUsWUFBWTtZQUN2QixHQUFHLEVBQ0QsSUFBSSxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUMsTUFBTTtnQkFDekIsQ0FBQyxDQUFDLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPLEVBQUUsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNLEVBQUU7Z0JBQ2hELENBQUMsQ0FBQyxTQUFTO1lBQ2YsR0FBRyxPQUFPO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ25DLENBQUM7O0FBbE9ILDRCQW1PQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEVudmlyb25tZW50LCBTdGFnZSwgU3RhY2ssIFN0YWNrUHJvcHMgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7IENvbnN0cnVjdCwgSUNvbnN0cnVjdCB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5pbXBvcnQgeyBDb250ZXh0TG9hZGVyIH0gZnJvbSBcIi4vY29udGV4dFwiO1xuaW1wb3J0IHsgVGllciB9IGZyb20gXCIuL3RpZXJcIjtcblxuY29uc3QgV09SS0xPQURfU1lNQk9MID0gU3ltYm9sLmZvcihcIkBzaGFkeS1pc2xhbmQvY29yZS5Xb3JrbG9hZFwiKTtcblxuLyoqXG4gKiBDb25zdHJ1Y3RvciBwcm9wZXJ0aWVzIGZvciBhIFdvcmtsb2FkLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFdvcmtsb2FkUHJvcHMge1xuICAvKipcbiAgICogVGhlIEFXUyBlbnZpcm9ubWVudCAoYWNjb3VudC9yZWdpb24pIHdoZXJlIHRoaXMgc3RhY2sgd2lsbCBiZSBkZXBsb3llZC5cbiAgICovXG4gIHJlYWRvbmx5IGVudj86IEVudmlyb25tZW50O1xuICAvKipcbiAgICogVGhlIGRlcGxveW1lbnQgdGllci5cbiAgICovXG4gIHJlYWRvbmx5IHRpZXI6IFRpZXI7XG4gIC8qKlxuICAgKiBUaGUgbWFjaGluZSBpZGVudGlmaWVyIGZvciB0aGlzIHdvcmtsb2FkLlxuICAgKlxuICAgKiBUaGlzIHZhbHVlIHdpbGwgYmUgdXNlZCB0byBjcmVhdGUgdGhlIGBwdWJsaWNEb21haW5OYW1lYCBwcm9wZXJ0eS5cbiAgICpcbiAgICogQnkgZGVmYXVsdCwgdGhlIGBzdGFja05hbWVgIHByb3BlcnR5IHVzZWQgdG8gY3JlYXRlIGBTdGFja2AgY29uc3RydWN0cyBpblxuICAgKiB0aGUgYGNyZWF0ZVN0YWNrYCBtZXRob2Qgd2lsbCBiZWdpbiB3aXRoIHRoaXMgV29ya2xvYWQncyBgd29ya2xvYWROYW1lYCBhbmRcbiAgICogaXRzIGB0aWVyYCBzZXBhcmF0ZWQgYnkgaHlwaGVucy5cbiAgICpcbiAgICogQ29uc2lkZXIgcHJvdmlkaW5nIGEgY29uc3RhbnQgYHdvcmtsb2FkTmFtZWAgdmFsdWUgdG8gdGhlIHN1cGVyY2xhc3NcbiAgICogY29uc3RydWN0b3IgaW4geW91ciBkZXJpdmVkIGNsYXNzLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKlxuICAgKiBjbGFzcyBNeVdvcmtsb2FkIGV4dGVuZHMgV29ya2xvYWQge1xuICAgKiAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBXb3JrbG9hZFByb3BzKSB7XG4gICAqICAgICBzdXBlcihzY29wZSwgaWQsIHsgLi4ucHJvcHMsIHdvcmtsb2FkTmFtZTogJ215LXdvcmtsb2FkJyB9KTtcbiAgICogICB9XG4gICAqIH1cbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgaWQgcGFzc2VkIHRvIHRoZSBgV29ya2xvYWRgIGNvbnN0cnVjdG9yLCBidXQgaW4gbG93ZXJjYXNlXG4gICAqL1xuICByZWFkb25seSB3b3JrbG9hZE5hbWU/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgZmlsZXN5c3RlbSBwYXRoIHRvIGEgSlNPTiBmaWxlIHRoYXQgY29udGFpbnMgY29udGV4dCB2YWx1ZXMgdG8gbG9hZC5cbiAgICpcbiAgICogVXNpbmcgdGhpcyBwcm9wZXJ0eSBhbGxvd3MgeW91IHRvIGxvYWQgZGlmZmVyZW50IGNvbnRleHQgdmFsdWVzIHdpdGhpbiBlYWNoXG4gICAqIGluc3RhbnRpYXRlZCBgV29ya2xvYWRgLCBkaXJlY3RseSBmcm9tIGEgZmlsZSB5b3UgY2FuIGNoZWNrIGludG8gc291cmNlXG4gICAqIGNvbnRyb2wuXG4gICAqL1xuICByZWFkb25seSBjb250ZXh0RmlsZT86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBiYXNlIGRvbWFpbiBuYW1lIHVzZWQgdG8gY3JlYXRlIHRoZSBGUUROIGZvciBwdWJsaWMgcmVzb3VyY2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgYmFzZURvbWFpbk5hbWU/OiBzdHJpbmc7XG59XG5cbmNvbnN0IElNUE9SVEVEX1NUQUNLUyA9IG5ldyBNYXA8U3RhY2ssIFdvcmtsb2FkPigpO1xuXG4vKipcbiAqIEEgY29sbGVjdGlvbiBvZiBTdGFja3MgaW4gYW4gRW52aXJvbm1lbnQgcmVwcmVzZW50aW5nIGEgZGVwbG95bWVudCBUaWVyLlxuICpcbiAqIENvbnNpZGVyIGRlcml2aW5nIGEgc3ViY2xhc3Mgb2YgYFdvcmtsb2FkYCBhbmQgY3JlYXRpbmcgeW91ciBgU3RhY2tgIG9iamVjdHNcbiAqIHdpdGhpbiBpdHMgY29uc3RydWN0b3IuXG4gKlxuICogVGhlIGRpZmZlcmVuY2UgYmV0d2VlbiB0aGlzIGNsYXNzIGFuZCBhIGBTdGFnZWAgaXMgdGhhdCBhIGBTdGFnZWAgaXMgbWVhbnQgdG9cbiAqIGJlIGRlcGxveWVkIHdpdGggQ0RLIFBpcGVsaW5lcy4gVGhpcyBjbGFzcyBjYW4gYmUgdXNlZCB3aXRoIGBjZGsgZGVwbG95YC5cbiAqIFRoaXMgY2xhc3MgYWxzbyBwcm92aWRlcyBjb250ZXh0IGxvYWRpbmcgY2FwYWJpbGl0aWVzLlxuICpcbiAqIEl0IGlzIGFuIGFudGktcGF0dGVybiB0byBwcm92aWRlIGEgYFdvcmtsb2FkYCBpbnN0YW5jZSBhcyB0aGUgcGFyZW50IHNjb3BlIHRvXG4gKiB0aGUgYGF3cy1jZGstbGliLlN0YWNrYCBjb25zdHJ1Y3Rvci4gWW91IHNob3VsZCBlaXRoZXIgdXNlIHRoZVxuICogYGNyZWF0ZVN0YWNrKClgIG1ldGhvZCwgY3JlYXRlIHlvdXIgb3duIHN1Yi1jbGFzcyBvZiBgU3RhY2tgIGFuZCBwcm92aWRlIGFcbiAqIGBXb3JrbG9hZGAgaW5zdGFuY2UgYXMgdGhlIHBhcmVudCBzY29wZSwgb3IgdXNlIHRoZSBgaW1wb3J0KClgIG1ldGhvZCB0b1xuICogZXNzZW50aWFsbHkgX2ltcG9ydF8gYSBgU3RhY2tgIGFuZCBpdHMgY29uc3RydWN0cyBpbnRvIGEgYFdvcmtsb2FkYCB3aXRob3V0XG4gKiBjaGFuZ2luZyBpdHMgc2NvcGUuXG4gKi9cbmV4cG9ydCBjbGFzcyBXb3JrbG9hZCBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIFdvcmtsb2FkIHRoZSBjb25zdHJ1Y3QgaXMgY29udGFpbmVkIHdpdGhpbiwgZmFpbHMgaWYgdGhlcmUgaXMgbm9cbiAgICogd29ya2xvYWQgdXAgdGhlIHRyZWUuXG4gICAqXG4gICAqIEBwYXJhbSBjb25zdHJ1Y3QgLSBUaGUgY29uc3RydWN0IHdob3NlIHBhcmVudCBub2RlcyB3aWxsIGJlIHNlYXJjaGVkXG4gICAqIEByZXR1cm5zIFRoZSBXb3JrbG9hZCBjb250YWluaW5nIHRoZSBjb25zdHJ1Y3RcbiAgICogQHRocm93cyBFcnJvciAtIGlmIG5vbmUgb2YgdGhlIGNvbnN0cnVjdCdzIHBhcmVudHMgYXJlIGEgd29ya2xvYWRcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgb2YoY29uc3RydWN0OiBJQ29uc3RydWN0KTogV29ya2xvYWQge1xuICAgIGNvbnN0IGNvbnN0cnVjdFN0YWNrID0gU3RhY2sub2YoY29uc3RydWN0KTtcbiAgICBpZiAoSU1QT1JURURfU1RBQ0tTLmhhcyhjb25zdHJ1Y3RTdGFjaykpIHtcbiAgICAgIHJldHVybiBJTVBPUlRFRF9TVEFDS1MuZ2V0KGNvbnN0cnVjdFN0YWNrKSE7XG4gICAgfVxuICAgIGNvbnN0IHdvcmtsb2FkID0gY29uc3RydWN0Lm5vZGUuc2NvcGVzXG4gICAgICAucmV2ZXJzZSgpXG4gICAgICAuc2xpY2UoMSlcbiAgICAgIC5maW5kKFdvcmtsb2FkLmlzV29ya2xvYWQpO1xuICAgIGlmICghd29ya2xvYWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJObyB3b3JrbG9hZCBleGlzdHMgaW4gdGhlIHBhcmVudGFnZSBvZiB0aGUgcHJvdmlkZWQgY29uc3RydWN0XCJcbiAgICAgICk7XG4gICAgfVxuICAgIHJldHVybiB3b3JrbG9hZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBUZXN0IHdoZXRoZXIgdGhlIGdpdmVuIGNvbnN0cnVjdCBpcyBhIFdvcmtsb2FkLlxuICAgKlxuICAgKiBAcGFyYW0geCAtIFRoZSB2YWx1ZSB0byB0ZXN0XG4gICAqIEByZXR1cm5zIFdoZXRoZXIgdGhlIHZhbHVlIGlzIGEgV29ya2xvYWQgb2JqZWN0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBpc1dvcmtsb2FkKHg6IGFueSk6IHggaXMgV29ya2xvYWQge1xuICAgIHJldHVybiB4ICE9PSBudWxsICYmIHR5cGVvZiB4ID09PSBcIm9iamVjdFwiICYmIFdPUktMT0FEX1NZTUJPTCBpbiB4O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IHJlZ2lvbiBmb3IgYWxsIHJlc291cmNlcyBkZWZpbmVkIHdpdGhpbiB0aGlzIHdvcmtsb2FkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlZ2lvbj86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IGFjY291bnQgZm9yIGFsbCByZXNvdXJjZXMgZGVmaW5lZCB3aXRoaW4gdGhpcyB3b3JrbG9hZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhY2NvdW50Pzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIHByZWZpeCB1c2VkIGluIHRoZSBkZWZhdWx0IGBzdGFja05hbWVgIHByb3ZpZGVkIHRvIGNoaWxkIFN0YWNrcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB3b3JrbG9hZE5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBkZXBsb3ltZW50IHRpZXIuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdGllcjogVGllcjtcbiAgLyoqXG4gICAqIFRoZSBkb21haW4gbmFtZSB0byB1c2UgZm9yIHJlc291cmNlcyB0aGF0IGV4cG9zZSBwdWJsaWMgZW5kcG9pbnRzLlxuICAgKlxuICAgKiBZb3UgY2FuIHVzZSBgV29ya2xvYWQub2YodGhpcykucHVibGljRG9tYWluTmFtZWAgYXMgdGhlIGB6b25lTmFtZWAgb2YgYVxuICAgKiBSb3V0ZSA1MyBob3N0ZWQgem9uZS5cbiAgICpcbiAgICogQW55IGNvbnN0cnVjdCB0aGF0IGNyZWF0ZXMgcHVibGljIEROUyByZXNvdXJjZXMgKGUuZy4gdGhvc2Ugb2YgQVBJIEdhdGV3YXksXG4gICAqIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jaW5nLCBDbG91ZEZyb250KSBjYW4gdXNlIHRoaXMgcHJvcGVydHkgdG8gZm9ybWF0XG4gICAqIGEgRlFETiBmb3IgaXRzZWxmIGJ5IGFkZGluZyBhIHN1YmRvbWFpbi5cbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICpcbiAgICogY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgKiBjb25zdCB3b3JrbG9hZCA9IG5ldyBXb3JrbG9hZChhcHAsIFwiRm9vYmFyXCIsIHtcbiAgICogICB0aWVyOiBUaWVyLlBST0RVQ1RJT04sXG4gICAqICAgYmFzZURvbWFpbk5hbWU6ICdleGFtcGxlLmNvbSdcbiAgICogfSk7XG4gICAqIGFzc2VydC5zdHJpY3RFcXVhbCh3b3JrbG9hZC5wdWJsaWNEb21haW5OYW1lLCAncHJvZC5mb29iYXIuZXhhbXBsZS5jb20nKTtcbiAgICogY29uc3Qgc3RhY2sgPSB3b3JrbG9hZC5jcmVhdGVTdGFjayhcIkROU1wiKTtcbiAgICogY29uc3QgaG9zdGVkWm9uZSA9IG5ldyBIb3N0ZWRab25lKHN0YWNrLCBcIkhvc3RlZFpvbmVcIiwge1xuICAgKiAgIHpvbmVOYW1lOiBgJHt3b3JrbG9hZC5wdWJsaWNEb21haW5OYW1lfWBcbiAgICogfSk7XG4gICAqIGNvbnN0IGFwaSA9IG5ldyBSZXN0QXBpKHN0YWNrLCBcIkFQSVwiLCB7XG4gICAqICAgcmVzdEFwaU5hbWU6IFwiZm9vYmFyXCIsXG4gICAqICAgZG9tYWluTmFtZTogeyBkb21haW5OYW1lOiBgYXBpLiR7d29ya2xvYWQucHVibGljRG9tYWluTmFtZX1gIH0sXG4gICAqIH0pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIElmIGBiYXNlRG9tYWluTmFtZWAgd2FzIGVtcHR5LCB0aGlzIHdpbGwgYmUgYHVuZGVmaW5lZGBcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwdWJsaWNEb21haW5OYW1lPzogc3RyaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX3N0YWNrczogTWFwPHN0cmluZywgU3RhY2s+O1xuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgbmV3IFdvcmtsb2FkLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgLSBUaGUgY29uc3RydWN0IHNjb3BlLlxuICAgKiBAcGFyYW0gaWQgLSBUaGUgY29uc3RydWN0IElELlxuICAgKiBAcGFyYW0gcHJvcHMgLSBUaGUgY29uc3RydWN0b3Igb3B0aW9ucy5cbiAgICovXG4gIHB1YmxpYyBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogV29ya2xvYWRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBPYmplY3QuZGVmaW5lUHJvcGVydHkodGhpcywgV09SS0xPQURfU1lNQk9MLCB7IHZhbHVlOiB0cnVlIH0pO1xuXG4gICAgY29uc3Qge1xuICAgICAgZW52OiB7IHJlZ2lvbiwgYWNjb3VudCB9ID0ge30sXG4gICAgICB0aWVyLFxuICAgICAgd29ya2xvYWROYW1lLFxuICAgICAgY29udGV4dEZpbGUsXG4gICAgICBiYXNlRG9tYWluTmFtZSxcbiAgICB9ID0gcHJvcHM7XG5cbiAgICB0aGlzLndvcmtsb2FkTmFtZSA9IHdvcmtsb2FkTmFtZSA/PyBgJHtpZH1gLnRvTG93ZXJDYXNlKCk7XG4gICAgdGhpcy50aWVyID0gdGllcjtcbiAgICB0aGlzLnB1YmxpY0RvbWFpbk5hbWUgPSBiYXNlRG9tYWluTmFtZVxuICAgICAgPyBgJHt0aWVyLmlkfS4ke3RoaXMud29ya2xvYWROYW1lfS4ke2Jhc2VEb21haW5OYW1lfWAudG9Mb3dlckNhc2UoKVxuICAgICAgOiB1bmRlZmluZWQ7XG4gICAgdGhpcy5fc3RhY2tzID0gbmV3IE1hcCgpO1xuXG4gICAgY29uc3QgeyByZWdpb246IHN0YWdlUmVnaW9uLCBhY2NvdW50OiBzdGFnZUFjY291bnQgfSA9IFN0YWdlLm9mKHRoaXMpITtcbiAgICB0aGlzLnJlZ2lvbiA9IHJlZ2lvbiB8fCBzdGFnZVJlZ2lvbjtcbiAgICB0aGlzLmFjY291bnQgPSBhY2NvdW50IHx8IHN0YWdlQWNjb3VudDtcblxuICAgIGlmIChjb250ZXh0RmlsZSkge1xuICAgICAgQ29udGV4dExvYWRlci5sb2FkQ29udGV4dChjb250ZXh0RmlsZSwgdGhpcy5ub2RlKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQHJldHVybnMgVGhlIHN0YWNrcyBjcmVhdGVkIGJ5IGludm9raW5nIGBjcmVhdGVTdGFja2BcbiAgICovXG4gIHB1YmxpYyBnZXQgc3RhY2tzKCk6IFN0YWNrW10ge1xuICAgIHJldHVybiBbLi4udGhpcy5fc3RhY2tzLnZhbHVlcygpXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWdpc3RlciB0aGUgcHJvdmlkZWQgYFN0YWNrYCBhcyBiZWluZyBwYXJ0IG9mIHRoaXMgYFdvcmtsb2FkYC5cbiAgICpcbiAgICogQHBhcmFtIHN0YWNrIC0gVGhlIHN0YWNrIHRvIHJlZ2lzdGVyLlxuICAgKiBAcmV0dXJucyBUaGUgcHJvdmlkZWQgU3RhY2tcbiAgICovXG4gIHByb3RlY3RlZCByZWdpc3RlclN0YWNrKHN0YWNrOiBTdGFjayk6IFN0YWNrIHtcbiAgICB0aGlzLnRpZXIuYXBwbHlUYWdzKHN0YWNrKTtcbiAgICB0aGlzLl9zdGFja3Muc2V0KHN0YWNrLnN0YWNrTmFtZSwgc3RhY2spO1xuICAgIHJldHVybiBzdGFjaztcbiAgfVxuXG4gIC8qKlxuICAgKiBGb3JjZXMgYSByZXR1cm4gdmFsdWUgZm9yIGBXb3JrbG9hZC5vZmAgZm9yIG9uZSBvciBtb3JlIGBTdGFja2Agb2JqZWN0cy5cbiAgICpcbiAgICogTm9ybWFsbHksIGEgY29uc3RydWN0IG11c3QgYmUgd2l0aGluIHRoZSBzY29wZSBvZiB0aGUgYFdvcmtsb2FkYCBpbnN0YW5jZSxcbiAgICogc3VjaCBhcyBhIGNvbnN0cnVjdCB0aGF0IGlzIGEgZGVzY2VuZGFudCBvZiBhIGBTdGFja2AgcmV0dXJuZWQgZnJvbVxuICAgKiBgY3JlYXRlU3RhY2soKWAuXG4gICAqXG4gICAqIFRoYXQgbWVhbnMgdGhhdCBhbnkgYFN0YWNrYCBpbnN0YW5jZXMgeW91IGNyZWF0ZWQgaW4geW91ciBDREsgYXBwbGljYXRpb25cbiAgICogX2JlZm9yZV8gaW5zdGFsbGluZyB0aGUgYHNoYWR5LWlzbGFuZGAgbGlicmFyeSB3b3VsZCBub3QgYmUgYWJsZSB0byBiZSBwYXJ0XG4gICAqIG9mIGEgYFdvcmtsb2FkYCB1bmxlc3MgeW91IGNoYW5nZWQgdGhlIGBzY29wZWAgYXJndW1lbnQgb2YgdGhlIGBTdGFja2BcbiAgICogY29uc3RydWN0b3IgZnJvbSB0aGUgYEFwcGAgb3IgYFN0YWdlYCB0byB0aGUgZGVzaXJlZCBgV29ya2xvYWRgIGluc3RhbmNlLlxuICAgKiBIb3dldmVyLCB0aGF0J3MgYmFkIG5ld3MgZm9yIGEgYFN0YWNrYCB0aGF0IGhhcyBhbHJlYWR5IGJlZW4gZGVwbG95ZWQgdG9cbiAgICogQ2xvdWRGb3JtYXRpb24gYmVjYXVzZSB0aGUgcmVzb3VyY2UgaWRlbnRpZmllciBvZiBwZXJzaXN0ZW50IGNoaWxkXG4gICAqIGNvbnN0cnVjdHMgKGUuZy4gUkRTIGRhdGFiYXNlcywgUzMgYnVja2V0cykgd291bGQgY2hhbmdlLlxuICAgKlxuICAgKiBBIHN1Y2Nlc3NmdWwgY2FsbCB0byB0aGlzIG1ldGhvZCB3aWxsIHJlZ2lzdGVyIHRoZSBwcm92aWRlZCBgU3RhY2tgIG9iamVjdHNcbiAgICogYW5kIGFsbCB0aGVpciBjb25zdHJ1Y3QgZGVzY2VuZGFudHMgYXMgbWVtYmVycyBvZiB0aGF0IGBXb3JrbG9hZGAgaW5zdGFuY2UuXG4gICAqIENhbGxpbmcgYFdvcmtsb2FkLm9mKClgIHdpdGggYW55IG9mIHRoZSBwcm92aWRlZCBgU3RhY2tgIG9iamVjdHMgb3IgdGhlaXJcbiAgICogZGVzY2VuZGFudCBjb25zdHJ1Y3RzIHdpbGwgcmV0dXJuIHRoYXQgYFdvcmtsb2FkYCBpbnN0YW5jZS5cbiAgICpcbiAgICogSWYgYW55IG9mIHRoZSBgU3RhY2tgIG9iamVjdHMgcHJvdmlkZWQgdG8gdGhpcyBtZXRob2QgYWxyZWFkeSBiZWxvbmcgdG8gYVxuICAgKiBkaWZmZXJlbnQgYFdvcmtsb2FkYCBvYmplY3QsIG9yIHdob3NlIHBhcmVudCBzY29wZSBpcyBub3QgaWRlbnRpY2FsIHRvIHRoZVxuICAgKiBwYXJlbnQgc2NvcGUgb2YgdGhpcyBgV29ya2xvYWRgIChpLmUuIHRoZSBgU3RhZ2VgIG9yIHRoZSBgQXBwYCksIGFuIGVycm9yXG4gICAqIHdpbGwgYmUgdGhyb3duLlxuICAgKlxuICAgKiBAcGFyYW0gc3RhY2tzIC0gVGhlIGBTdGFja2AgaW5zdGFuY2VzIHRvIGltcG9ydCB0byB0aGlzIGBXb3JrbG9hZGBcbiAgICogQHRocm93cyB7RXJyb3J9IGlmIGFueSBvZiB0aGUgc3RhY2tzIGFyZSBhbHJlYWR5IHBhcnQgb2YgYW5vdGhlciB3b3JrbG9hZFxuICAgKiBAdGhyb3dzIHtFcnJvcn0gaWYgYW55IG9mIHRoZSBzdGFja3MgaGF2ZSBhIGRpZmZlcmVudCBwYXJlbnQgc2NvcGVcbiAgICovXG4gIHB1YmxpYyBpbXBvcnQoLi4uc3RhY2tzOiBTdGFja1tdKSB7XG4gICAgZm9yIChjb25zdCBzdGFjayBvZiBzdGFja3MpIHtcbiAgICAgIGlmIChzdGFjay5ub2RlLnNjb3BlID09PSB0aGlzKSB7XG4gICAgICAgIGNvbnRpbnVlO1xuICAgICAgfVxuICAgICAgaWYgKFdvcmtsb2FkLmlzV29ya2xvYWQoc3RhY2subm9kZS5zY29wZSkgJiYgc3RhY2subm9kZS5zY29wZSAhPT0gdGhpcykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgXCJUaGUgU3RhY2sgaXMgYWxyZWFkeSBjb250YWluZWQgd2l0aGluIGEgZGlmZmVyZW50IFdvcmtsb2FkXCJcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICAgIGlmIChzdGFjay5ub2RlLnNjb3BlICE9PSB0aGlzLm5vZGUuc2NvcGUpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIFwiVGhlIFN0YWNrIG11c3QgYmUgY29udGFpbmVkIHdpdGhpbiB0aGUgc2FtZSBzY29wZSBhcyB0aGlzIFdvcmtsb2FkXCJcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICAgIElNUE9SVEVEX1NUQUNLUy5zZXQodGhpcy5yZWdpc3RlclN0YWNrKHN0YWNrKSwgdGhpcyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBzdGFjayB0byB0aGUgV29ya2xvYWQuXG4gICAqXG4gICAqIFRoaXMgbWV0aG9kIHdpbGwgcmV0dXJuIGEgYFN0YWNrYCB3aXRoIHRoaXMgV29ya2xvYWQgYXMgaXRzIHNjb3BlLiBCeVxuICAgKiBkZWZhdWx0LCB0aGUgYHN0YWNrTmFtZWAgcHJvcGVydHkgcHJvdmlkZWQgdG8gdGhlIGBTdGFja2Agd2lsbCBiZSB0aGlzXG4gICAqIFdvcmtsb2FkJ3MgYHdvcmtsb2FkTmFtZWAsIGl0cyBgdGllcmAsIGFuZCB0aGUgdmFsdWUgb2YgdGhlIGBpZGBcbiAgICogcGFyYW1ldGVyIHNlcGFyYXRlZCBieSBoeXBoZW5zLCBhbGwgaW4gbG93ZXJjYXNlLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKlxuICAgKiBjb25zdCBleGFtcGxlRGV2ID0gbmV3IFdvcmtsb2FkKGFwcCwgJ0V4YW1wbGUnLCB7XG4gICAqICAgdGllcjogVGllci5ERVZFTE9QTUVOVCxcbiAgICogICBlbnY6IHsgYWNjb3VudDogJzEyMzQ1Njc4OTAxMicsIHJlZ2lvbjogJ3VzLWVhc3QtMScgfSxcbiAgICogfSk7XG4gICAqIGNvbnN0IG5ldHdvcmtTdGFjayA9IGV4YW1wbGVEZXYuY3JlYXRlU3RhY2soJ05ldHdvcmsnLCB7fSk7XG4gICAqIGFzc2VydC5zdHJpY3RFcXVhbChuZXR3b3JrU3RhY2suc3RhY2tOYW1lLCAnZXhhbXBsZS1kZXYtbmV0d29yaycpLlxuICAgKlxuICAgKiBZb3UgY2FuIG92ZXJyaWRlIHRoZSBgZW52YCBhbmQgYHN0YWNrTmFtZWAgcHJvcGVydGllcyBpbiB0aGUgYHByb3BzYFxuICAgKiBhcmd1bWVudCBpZiBkZXNpcmVkLlxuICAgKlxuICAgKiBUaGUgc3RhY2sgd2lsbCBoYXZlIGEgYERlcGxveW1lbnRUaWVyYCB0YWcgYWRkZWQsIHNldCB0byB0aGUgdGllciBsYWJlbC5cbiAgICpcbiAgICogQHBhcmFtIGlkIC0gVGhlIFN0YWNrIGNvbnN0cnVjdCBpZCAoZS5nLiBcIk5ldHdvcmtcIilcbiAgICogQHBhcmFtIHByb3BzIC0gVGhlIG5ldyBTdGFjayBwcm9wZXJ0aWVzXG4gICAqL1xuICBwdWJsaWMgY3JlYXRlU3RhY2soaWQ6IHN0cmluZywgcHJvcHM/OiBTdGFja1Byb3BzKTogU3RhY2sge1xuICAgIGNvbnN0IHsgc3RhY2tOYW1lLCAuLi5vcHRpb25zIH0gPSBwcm9wcyA/PyB7fTtcbiAgICBjb25zdCBuZXdTdGFja05hbWUgPVxuICAgICAgc3RhY2tOYW1lID8/IGAke3RoaXMud29ya2xvYWROYW1lfS0ke3RoaXMudGllci5pZH0tJHtpZH1gLnRvTG93ZXJDYXNlKCk7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2sodGhpcywgaWQsIHtcbiAgICAgIHN0YWNrTmFtZTogbmV3U3RhY2tOYW1lLFxuICAgICAgZW52OlxuICAgICAgICB0aGlzLmFjY291bnQgJiYgdGhpcy5yZWdpb25cbiAgICAgICAgICA/IHsgYWNjb3VudDogdGhpcy5hY2NvdW50LCByZWdpb246IHRoaXMucmVnaW9uIH1cbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgIC4uLm9wdGlvbnMsXG4gICAgfSk7XG4gICAgcmV0dXJuIHRoaXMucmVnaXN0ZXJTdGFjayhzdGFjayk7XG4gIH1cbn1cbiJdfQ==