"""
This submodule is used to parse the input configuration into an object.
The configuration object is a python dataclass created by ConfigSchema
which is a marshmallow schema. The data class contains fields defined
by ConfigSchema. Subclass ConfigSchema to add fields.
"""

import logging
from dataclasses import make_dataclass
import regex as re
from marshmallow import Schema, fields, post_load, ValidationError
from gitbuilding.buildup.utilities import as_posix

_LOGGER = logging.getLogger('BuildUp')

class _CategorySchema(Schema):
    """
    Marshmallow schema for validating custom BuildUp categories. On load it converts
    the input dictionary to a python dataclass.
    """
    reuse = fields.Bool(required=True, data_key='Reuse')
    display_name = fields.Str(required=True, data_key='DisplayName')

    @post_load
    def make_object(self, data, **_):
        """
        Auto generates a dataclass for the category
        """
        category = make_dataclass('Category', data.keys())
        return category(**data)

def _create_base_categories():
    cat_schema = _CategorySchema()
    return {"part": cat_schema.load({"Reuse": False, "DisplayName": "Parts"}),
            "tool": cat_schema.load({"Reuse": True, "DisplayName": "Tools"})}

class _CategoriesField(fields.Dict):
    """CategoriesField is used by marshmallow to serialise and de-serialise
    custom_categories. All category names are turned lower case."""

    def _deserialize(self, value, attr, data, **kwargs):
        value = super()._deserialize(value, attr, data, **kwargs)
        lowered = _create_base_categories()
        for key in value:
            lowered[key.lower()] = value[key]
        return lowered

class _DefaultCategory(fields.Str):
    """DefaultCategory is used by marshmallow to serialise and de-serialise
    the DefaultCategories. This object will hopefully be removed once we
    can find a better way to compare the field to the de-serialised output
    of custom_categories"""

    def _deserialize(self, value, attr, data, **kwargs):
        value = super()._deserialize(value, attr, data, **kwargs)
        value = value.lower()

        if value in ['part', 'tool']:
            return value
        if 'CustomCategories' in data and isinstance(data['CustomCategories'], dict):
            for key in data['CustomCategories']:
                if str(key).lower() == value:
                    return value
        raise ValidationError(f'{value} is not a valid category.')

class _NavSchema(Schema):
    """
    Marshmallow schema for parsing the navigation
    """
    title = fields.Str(required=True, data_key='Title')
    link = fields.Str(required=True, data_key='Link')
    subnavigation = fields.List(cls_or_instance=fields.Nested("_NavSchema"),
                                data_key='SubNavigation')

class ConfigSchema(Schema):
    """
    This is the schema for the main configuration object that is used in BuildUp.
    The configuration object is generated by passing a dictionary into this marshmallow
    schema. A validation error is returned for extra fields. If you want to use the
    configuration dictionary to hold to add information you can either subclass this schema,
    or you can use  ConfigSchema.validate() to find extra fields and remove them before the
    configuration object is created.
    Extra fields are not allowed to help catch typos the configuration.
    """
    title = fields.Str(load_default=None, allow_none=True, data_key='Title')
    page_bom_title = fields.Str(load_default="", data_key='PageBOMTitle')
    custom_categories = _CategoriesField(load_default=_create_base_categories,
                                         keys=fields.Str(),
                                         values=fields.Nested(_CategorySchema),
                                         data_key='CustomCategories')
    default_category = _DefaultCategory(load_default="part", data_key='DefaultCategory')
    navigation = fields.List(cls_or_instance=fields.Nested(_NavSchema),
                             load_default=list,
                             data_key='Navigation')
    autocompletesubnav = fields.Bool(load_default=True, data_key='AutoCompleteSubNav')
    landing_page = fields.Str(load_default=None, allow_none=True, data_key='LandingPage')
    remove_landing_title = fields.Bool(load_default=False, data_key='RemoveLandingTitle')
    remove_bottom_nav = fields.Bool(load_default=False, data_key='RemoveBottomNav')
    target_format =  fields.Str(load_default='html', data_key='OutputType')
    force_output = fields.List(cls_or_instance=fields.Str,
                               load_default=list,
                               data_key='ForceOutput')
    external_dirs = fields.Dict(load_default=dict,
                                keys=fields.Str(),
                                values=fields.Str(),
                                data_key='ExternalDirs')
    exsource_def = fields.Str(load_default=None, allow_none=True, data_key='ExSourceDef')
    exsource_out = fields.Str(load_default=None, allow_none=True, data_key='ExSourceOut')

    @post_load
    def make_object(self, data, **_):
        """
        Auto generates a dataclass for the configuration.
        """
        data['categories'] = data['custom_categories']
        del data['custom_categories']

        # Doing validation here as I don't understand the marshmallow.Validators
        # documentation
        for doc_dir, path_on_disk in data['external_dirs'].items():
            _validate_doc_dir(doc_dir)
            _validate_external_dir_path(path_on_disk)

        buildup_config = make_dataclass('BuildUpConfig',
                                        data.keys())
        return buildup_config(**data)

def _validate_doc_dir(doc_dir):
    #Note this list doesn't allow windows style paths
    unsafe = re.findall(r'[^a-zA-z0-9\.\/\-_~! ]', doc_dir)
    if len(unsafe) > 0:
        chars = ''.join(set(unsafe))
        raise ValidationError(f'Documentation diectory "{doc_dir}" for external files'
                              f' contains the following unsafe characters: {chars}.')
    if doc_dir.startswith(('.', ' ', r'/', '_')):
        raise ValidationError(f'Documentation diectory "{doc_dir}" for external files'
                              f" must be a relative path that doesn't start with the"
                              ' the a space a . or an _')

def _validate_external_dir_path(path_on_disk):
    if not as_posix(path_on_disk).startswith('..'):
        raise ValidationError('External directories must be specified as a relative path. '
                              f'The directory "{path_on_disk}" is either not external or '
                              'not relative.')
