"""
This submodule constains the PageOrder class which is used to store and manipulate the paths
through the buildup documentation.
"""

import posixpath
from copy import copy, deepcopy
import logging
from gitbuilding.buildup.utilities import explain_mismatched_data

_LOGGER = logging.getLogger('BuildUp')

def _tree2list(tree, depth=0, parent=None):
    """
    Input is a step tree for a page, output is a list detailing the order the pages
    should be displayed in. The list containts tuples, these tuples have three elements,
    the pagepath, the depth in the steptree, and the variables.
    """
    pagepath = None
    for key in tree.keys():
        if key.endswith(".md"):
            pagepath = key
    if pagepath is None:
        raise RuntimeError("Invalid tree list detected, no pagepath")
    variables = tree['variables']
    entry = PageOrderEntry(pagepath, depth, variables, parent)
    pagelist = PageList([entry])
    for subtree in tree[pagepath]:
        pagelist += _tree2list(subtree, depth+1, entry)
    return pagelist

class PageOrderEntry():
    """
    An entry for a page in PageOrder contains information such as the depth in a step tree and
    variables assigned to the page
    """
    def __init__(self, path, depth, variables, parent):
        self.path = path
        self.depth = depth
        self.variables = variables
        self.parent = parent
        self.bom_pages = None

    def set_bom_pages(self, bom_pages):
        """set any bill of material pages that are generated by this page entry"""
        self.bom_pages = bom_pages

    @property
    def md_bom_page(self):
        """Return the url of the markdown bill of materials page. Note that self.bom_pages
        contains all formats of the bill of materials (markdown, csv, and any formats to be
        added)"""
        if self.bom_pages is None:
            return None
        #Markdown is the first entry in bom_pages
        return self.bom_pages[0]

    def generate_duplicate_entry(self, root):
        """
        Generate a duplicate entry for a pagelist with input root. This object
        is used in the PageOrder duplicates lists
        """
        return DuplicatePageEntry(self.path, self.depth, self.variables, self.parent, root)

    def get_breadcrumbs(self):
        """
        Return breadcrumbs of this pagentry in its pagelist
        """
        reverse_order = [self]
        while reverse_order[-1].parent is not None:
            reverse_order.append(reverse_order[-1].parent)
        return [entry.path for entry in reversed(reverse_order)]

    def __eq__(self, other):
        return self.path == other

    def __repr__(self):
        return f"<PageOrderEntry path: {self.path}, variables: {self.variables}>"

class DuplicatePageEntry(PageOrderEntry):
    """A page order entry for a duplicate page"""

    def __init__(self, path, depth, variables, parent, root):
        self.root = root
        super().__init__(path, depth, variables, parent)

    def __repr__(self):
        return f"<DuplicatePageEntry path: {self.path}, variables: {self.variables}>"

class MakeLinkOrderingEntry():
    """
    This acts as a temporary entry in the page ordering due to a make link
    This must be converted into a PageOrderEntry before the pagelist is complete.

    It is initialised with a link object and the entry of the page where that link was called
    This entry will be a PageOrder if it is linked to on a normal page in the page ordering, or
    it will be another MakeLinkOrderingEntry, if page calling it is also a page included by a make link
    """
    def __init__(self, link, entry):
        self._path = link.link_rel_to_root_no_preview
        self._link_and_entry_tuples = [(link, entry)]
        self._linktext = link.linktext

    def __eq__(self, other):
        return self.path == other

    def __repr__(self):
        return (f"<MakeLinkOrderingEntry path: {self._path}, qty:{self.qty}, "
                f"linktext: {self._linktext}>")

    @property
    def path(self):
        """
        The path of the page the make links points to
        """
        return self._path

    @property
    def linktext(self):
        """
        The linktext of the make links
        """
        return self._linktext

    @property
    def qty(self):
        '''
        The quantity needed of the made object. This is calculated from all make links
        when accessed make links
        '''
        if not self.all_links_from_ordered_pages:
            raise RuntimeError('Cannot get quantity until all calling pages are ordered')

        qtys = []
        for link, entry in self._link_and_entry_tuples:
            link_qty = link.buildup_data.qty
            if entry.variables and 'make_qty' in entry.variables:
                page_qty = entry.variables['make_qty']
                qtys.append(link_qty*page_qty)
            else:
                qtys.append(link_qty)
        return sum(qtys)

    @property
    def variables(self):
        if not self.all_links_from_ordered_pages:
            return None
        return {'make_qty': self.qty, 'make_linktext': self.linktext}

    @property
    def all_links_from_ordered_pages(self):
        """
        Returns True if all links to this page are from PageOrderEntries. If
        this page is linked to from anther page that is created from a make link
        this will return False.
        This is used to check if this link can be added to the page ordering or if
        another makelink entry must be added into the ordering first
        Once the make link pages have had an order found `update_entries_for()` will be run
        on this to update all references.
        """
        for _, entry in self._link_and_entry_tuples:
            if isinstance(entry, MakeLinkOrderingEntry):
                return False
        return True

    @property
    def pages_linked_from(self):
        #can't use `set()` and page entries aren't hashable
        entries = []
        for _, entry in self._link_and_entry_tuples:
            if entry not in entries:
                entries.append(entry)
        return entries

    def register_link(self, link, entry):
        """
        Notify the object that another make link to the same page is to be recorded
        """
        if link.make_data:
            _LOGGER.warning("Make data provided for non-templated link [%s]", link.linktext)
            return
        if self.path != link.link_rel_to_root_no_preview:
            raise ValueError('Cannot combine make-links that link to different pages')

        self._link_and_entry_tuples.append((link, entry))
        if self.linktext != link.linktext:
            _LOGGER.warning('The page %s is used with make links with mismatching '
                            'text ("%s" and "%s"). This will cause problems with '
                            'the Bill of Materials',
                            self.path, self.linktext, link.linktext)

    def update_entries_for(self, page_entry):
        for i, (link, entry) in enumerate(self._link_and_entry_tuples):
            if entry.path == page_entry.path:
                self._link_and_entry_tuples[i] = (link, page_entry)

    def get_page_entry(self, depth, parent):
        return PageOrderEntry(path=self.path,
                              depth=depth,
                              variables=self.variables,
                              parent=parent)

class MakeTemplateOrderingEntry(MakeLinkOrderingEntry):
    """
    Subclass of MakeLinkOrderingEntry for make templates. This has to
    handle multiple make links for different items being made on the same
    page.
    """

    def __init__(self, link, entry):
        super().__init__(link, entry)
        self._linktext = "TEMPLATE"

    def __repr__(self):
        return (f"<MakeTemplateOrderingEntry path: {self._path},"
                f"variables: {self._linktext}>")

    @property
    def qty(self):
        raise RuntimeError('This quantity should never be queried')

    @property
    def collated_data(self):
        if not self.all_links_from_ordered_pages:
            raise RuntimeError('Cannot get collated data until all calling pages are ordered')

        data = {}
        for link, entry in self._link_and_entry_tuples:
            link_qty = link.buildup_data.qty
            if entry.variables and 'make_qty' in entry.variables:
                page_qty = entry.variables['make_qty']
            else:
                page_qty = 1
            key = link.linktext.lower()
            if key in data:
                in_data = {k:v for k,v in link.make_data.items() if k != 'template'}
                existing_data = {k:v for k,v in data[key].items() if k != 'qty'}
                if existing_data != in_data:
                    _LOGGER.warning('Make link [%s] has mismatching make_data between references: %s',
                                   link.linktext, explain_mismatched_data(existing_data, in_data))
                data[key]['qty'] += link_qty*page_qty
            else:
                data[key] = {k:v for k,v in link.make_data.items() if k != 'template'}
                data[key]['qty'] = link_qty*page_qty

        return data

    def register_link(self, link, entry):
        """
        Notify the object that another make link to the same page is to be recorded
        """
        if not link.make_data:
            _LOGGER.warning("No make data found for templated link [%s]", link.linktext)
            return
        if self.path != link.link_rel_to_root_no_preview:
            raise ValueError('Cannot combine make-links that link to different pages')

        self._link_and_entry_tuples.append((link, entry))

    @property
    def variables(self):
        if not self.all_links_from_ordered_pages:
            return None
        # Find a way to pass the information through variables
        return {'make_template_data': self.collated_data}

class PageList(list):

    @property
    def has_make_list_entries(self):
        for i in self:
            if isinstance(i, MakeLinkOrderingEntry):
                return True
        return False

    @property
    def enumerated_make_link_entries(self):
        return [(i, j) for i, j in enumerate(self) if isinstance(j, MakeLinkOrderingEntry)]

    def pop_all_make_link_entries(self):
        pops = [i for i, entry in enumerate(self) if isinstance(entry, MakeLinkOrderingEntry)]
        for i in reversed(pops):
            self.pop(i)


class PageOrder():
    """
    A PageOrder is object is initialised with the step trees from every page and the landing page.
    It will calculate the number of paths through the documentation, these can be accessed as
    trees or as lists.
    The lists have two versions, the list of the pages, and the list for navigation, the list for
    navigation also include bill of materials.
    """

    def __init__(self, trees, doc_obj):
        self._doc = doc_obj
        self._trees = deepcopy(trees)
        self._pagelists = [_tree2list(tree) for tree in self._trees]
        self._filenames_of_duplicates = []
        self._remove_empty_trees_and_lists()
        self._remove_non_rooted_trees()
        self._add_pages_from_make_links()
        self._include_boms()
        if len(self.pagelists) > 0:
            self._validate_pagelists()
        self._generate_masterlist_and_duplicates()
        self._calc_filenames_of_duplicates()

    def __repr__(self):
        if self.number_of_paths == 0:
            return "<PageOrder: No page ordering>"
        elif self.number_of_paths == 1:
            list1 = [str(i) for i in self._pagelists[0]]
            return f"<PageOrder: {list1}>"
        else:
            lists = ""
            for i, pagelist in enumerate(self._pagelists):
                lists += f" list {i+1} : " + str([str(j) for j in pagelist]) + ";"
            return f"<PageOrderEntry:{lists}>"

    @property
    def number_of_paths(self):
        """
        Read-only property which returns a the number of paths through the documentation
        """
        return len(self._pagelists)

    @property
    def trees(self):
        """
        Read-only property which returns a copy of the step trees. One for each path through
        the documentation.
        """
        return deepcopy(self._trees)

    @property
    def masterlist(self):
        """
        Read-only property which returns a list of all pages that are on any step list. The items
        in each list are just the page_names.
        """
        return copy(self._masterlist)

    @property
    def duplicates(self):
        """
        Read-only property which returns a dictionary. The keys are the pagepaths of any pages
        that are used in more than one path through the documentation, the value is a list of
        the pagepaths of the roots of those paths.
        """
        return deepcopy(self._duplicates)


    @property
    def filenames_of_duplicates(self):
        """
        Return the output filename of any files that are duplicates
        """
        return copy(self._filenames_of_duplicates)

    def _calc_filenames_of_duplicates(self):
        filenames = []
        for duplicated_page in self.duplicates:
            for duplicate_entry in self.duplicates[duplicated_page]:
                root_pagepath = duplicate_entry.root
                list_no = self.get_list_number(root_pagepath)
                replacement_dict = self.link_replacement_dictionaries[list_no]
                filenames.append(replacement_dict[duplicated_page])
        self._filenames_of_duplicates = filenames

    @property
    def pagelists(self):
        """
        Read-only property which returns a copy of the pagelists. One list for each path through
        the documentation. The items in each list are a tuple of the page_names and their variables.
        """

        return deepcopy(self._pagelists)

    @property
    def link_replacement_dictionaries(self):
        """
        Read-only property which returns a copy of the pagelists. One list for each path through
        the documentation. The items in each list are a tuple of the file name in the input
        documentation and the file name in the output doucumentation.
        """
        out_lists = []
        for pagelist in self._pagelists:
            # first element in the lists is the first page, first element in the nav tuple
            # is the path
            rootpage_path = pagelist[0].path
            # subdirectory is the first page in the page order minus the '.md'
            subdir = rootpage_path[:-3]

            in_paths = []
            out_paths = []
            for page_entry in pagelist[1:]:
                if page_entry.path in self._duplicates:
                    in_paths.append(page_entry.path)
                    out_paths.append(posixpath.join(subdir, page_entry.path))
                    if page_entry.bom_pages is not None:
                        for bom_page in page_entry.bom_pages:
                            in_paths.append(bom_page)
                            out_paths.append(posixpath.join(subdir, bom_page))
            out_lists.append(dict(zip(in_paths, out_paths)))
        return out_lists

    def get_list_number(self, root_pagepath):
        """
        Return the number (index) of the pagelist with the input page name as root. If the input
        page is not a root of any pagelist then reutn None.
        """
        list_roots = [pagelist[0].path for pagelist in self.pagelists]
        try:
            return list_roots.index(root_pagepath)
        except ValueError:
            return None

    def _validate_pagelists(self):
        """
        Checks the page order doesn't have unexpected problems like repeated steps. The
        landing page doesn't need to be in the page order list, but if it is in it must
        be at the start.
        The function does not return anything, it just logs any issues
        """
        landing_page = self._doc.landing_page
        for pagelist in self.pagelists:
            if len(pagelist) == 0:
                return

            if landing_page in pagelist[1:]:
                _LOGGER.warning('The landing page cannot be a step of another page.')

            #warn if steps are repeated
            pagepaths = [page_entry.path for page_entry in pagelist]
            if len(set(pagepaths)) != len(pagelist):
                _LOGGER.warning('The defined page order has the same step repeated: "%s"',
                                '->'.join(pagepaths))

    def _remove_empty_trees_and_lists(self):
        """
        Some page lists contain only the name of the page the tree was read from as
        they have no steps. This function removes those lists and trees.
        """
        poplist = []
        for n, pagelist in enumerate(self._pagelists):
            if len(pagelist) == 1:
                poplist.append(n)
        #pop in reverse
        for n in poplist[::-1]:
            self._pagelists.pop(n)
            self._trees.pop(n)

    def _remove_non_rooted_trees(self):
        """
        This function removes any trees (and corresponding pagelists) that are not the
        root of a step tree.
        """
        #list of pagelists to remove as they are a sub-part of another list
        poplist = []
        for n, pagelist in enumerate(self._pagelists):
            for otherlist in self._pagelists[0:n]+self._pagelists[n+1:]:
                otherlist_names = [page_entry.path for page_entry in otherlist]
                if pagelist[0].path in otherlist_names:
                    poplist.append(n)
                    break
        #pop in reverse
        for n in poplist[::-1]:
            self._pagelists.pop(n)
            self._trees.pop(n)

    def _add_pages_from_make_links(self):
        for pagelist in self._pagelists:
            #This appends temporary entries for each page required due to a make link
            self._append_make_link_entries(pagelist)
            #This orders the temporary entries and converts them to the pageorder entry
            self._order_make_links_in_pagelist(pagelist)

    def _append_make_link_entries(self, pagelist):

        for entry in pagelist:
            page_obj = self._doc.get_page_by_path(entry.path, variables=entry.variables)
            for link in page_obj.make_links:
                link_path = link.link_rel_to_root_no_preview
                if link_path in pagelist:
                    make_link_entry = pagelist[pagelist.index(link_path)]
                    if isinstance(make_link_entry, PageOrderEntry):
                        _LOGGER.warning("The page %s is used as a step link and a make link "
                                        "GitBuilding doesn't know how to handle this.", link_path)
                        continue
                    make_link_entry.register_link(link, entry)
                else:
                    if link.make_data:
                        make_link_entry = MakeTemplateOrderingEntry(link, entry)
                    else:
                        make_link_entry = MakeLinkOrderingEntry(link, entry)
                    #check page exists before adding to page list
                    make_page_obj = self._doc.get_page_by_path(make_link_entry.path)
                    if make_page_obj is None:
                        if not make_link_entry.path:
                            _LOGGER.warning("'%s' is defined as a make link but no page is set",
                                            make_link_entry.linktext)
                        else:
                            _LOGGER.warning("The page '%s' is used as a make link but does not "
                                            "exist. This page will be skipped. This may cause "
                                            "an unexpected problem with your Bill of Materials",
                                            make_link_entry.path)
                        continue
                    pagelist.append(make_link_entry)


    def _order_make_links_in_pagelist(self, pagelist):

        # Convert make Link order entries one at a time
        while pagelist.has_make_list_entries:

            suitable_entry_found = False
            #Find first make list entry
            for j, entry in pagelist.enumerated_make_link_entries:
                if entry.all_links_from_ordered_pages:
                    suitable_entry_found = True
                    break
            if not suitable_entry_found:
                _LOGGER.warning("Make links seem to form a loop. Removing these make links")
                pagelist.pop_all_make_link_entries()
                return
            #pylint is upset that the loop variable may be unefined, but it can't be.
            #pylint: disable=undefined-loop-variable

            #pop the MakeLinkOrderingEntry
            pagelist.pop(j)
            pos, page_entry = self._get_make_link_page_entry_and_position(entry, pagelist)
            #Add PageOrderEntry in correct position
            pagelist.insert(pos, page_entry)
            self._update_nested_make_link_entries(page_entry, pagelist)
        return

    def _update_nested_make_link_entries(self, page_entry, pagelist):
        '''
        After a MakeLinkOrderingEntry is converted into a PageOrderEntry, any remaining
        MakeLinkOrderingEntries need to be updated if they have a nested link that was
        related to the old MakeLinkOrderingEntry
        '''
        for _, entry in pagelist.enumerated_make_link_entries:
            entry.update_entries_for(page_entry)


    def _get_make_link_page_entry_and_position(self, make_link_entry, pagelist):

        linked_from = make_link_entry.pages_linked_from
        #boolean of which pages the make link was used on
        link_used_on_page = [page_entry in linked_from for page_entry in pagelist]
        first_use = link_used_on_page.index(True)
        if first_use == 0:
            raise ValueError('Make links is used in the first page of documentation. '
                            'Cannot add link before first page.')

        times_used = link_used_on_page.count(True)
        if times_used == 1:
            depth = pagelist[first_use].depth
            parent = pagelist[first_use].parent
            pos = first_use
        else:
            depth, parent, pos = self._get_make_link_depth_parent_and_pos(link_used_on_page,
                                                                          pagelist)

        linked_page_entry = make_link_entry.get_page_entry(depth=depth, parent=parent)

        parent_page = self._doc.get_page_by_path(parent.path, variables=parent.variables)
        make_page = self._doc.get_page_by_path(linked_page_entry.path,
                                               variables=linked_page_entry.variables)
        parent_page.register_make_page(make_page)
        return pos, linked_page_entry

    def _get_make_link_depth_parent_and_pos(self, link_used, pagelist):
        '''
        This is only called if the make list is used more than once. In this
        case the parent should be the first common parent of the calling pages.
        The depth is the depth of the parent page + 1.
        The position is the first entry with this depth which either used the make
        or has a child page that did so.

        i.e. If two pages of depth 2 with the same parent call a make link then
        the that parent is also the parent of the make link page.
        If the two pages of depth 2 have different parents, then it is the parent
        those parents who become the parent of the make link page.
        '''
        def all_same_depth(items):
            return all(i.depth == items[0].depth for i in items)

        def all_same_parents(items):
            return all(i.parent == items[0].parent for i in items)


        entries = [entry for used, entry in zip(link_used, pagelist) if used]

        # If all depths are not the same replace highest depth entry with parent
        i = 0
        while not all_same_depth(entries):
            depths = [entry.depth for entry in entries]
            max_depth = max(depths)
            index_of_entry = depths.index(max_depth)
            index_of_parent = pagelist.index(entries[index_of_entry].parent)
            entries[index_of_entry] = pagelist[index_of_parent]
            i += 1
            if i > len(pagelist):# pragma: no cover
                #we seem to be in an infinite loop. This should never happen
                raise RuntimeError("GitBuilding entered and infinite loop trying"
                                   " to calculate page ordering")

        #Now depths are all the same. If parents don't match replace all entries
        # with parents.
        i = 0
        while not all_same_parents(entries):
            for j, entry in enumerate(entries):
                index_of_parent = pagelist.index(entry.parent)
                entries[j] = pagelist[index_of_parent]
            i += 1
            if i > len(pagelist):# pragma: no cover
                #we seem to be in an infinite loop. This should never happen
                raise RuntimeError("GitBuilding entered and infinite loop trying"
                                   " to calculate page ordering")

        return entries[0].depth, entries[0].parent, pagelist.index(entries[0].path)

    def _include_boms(self):
        """
        Add bills of materials to the pagelists tuple.
        """
        for pagelist in self._pagelists:
            for page_entry in pagelist:
                page = self._doc.get_page_by_path(page_entry.path, variables=page_entry.variables)
                page_entry.set_bom_pages(page.get_bom_page())

    def _generate_masterlist_and_duplicates(self):
        """
        Generate a list of all pages that are in step trees
        """
        self._masterlist = []
        reused_pages = []
        for pagelist in self.pagelists:
            for page_entry in pagelist:
                pagepath = page_entry.path
                if pagepath not in self._masterlist:
                    self._masterlist.append(pagepath)
                elif pagepath not in reused_pages:
                    reused_pages.append(page_entry)

        self._duplicates = {}
        for page_entry in reused_pages:
            pagepath = page_entry.path
            self._duplicates[pagepath] = []
            for pagelist in self.pagelists:
                if pagepath in pagelist:
                    root = pagelist[0].path
                    duplicate_entry = page_entry.generate_duplicate_entry(root)
                    self._duplicates[pagepath].append(duplicate_entry)

    def get_pagelist_for_page(self, page):
        """
        return a pagelist rooted at the specific page, and also returns
        the replacement links to go to the correct version of duplicated pages.
        """
        if page not in self.masterlist:
            raise IndexError(f"{page} is not in any page order")

        page_ind = None
        replace_links = None
        this_pagelist = None
        for n, pagelist in enumerate(self.pagelists):
            if page in pagelist:
                page_ind = pagelist.index(page)
                this_pagelist = pagelist
                if self.number_of_paths == 0:
                    replace_links = None
                else:
                    replace_links = self.link_replacement_dictionaries[n]
                break
        if (page_ind is None) or (this_pagelist is None):
            raise RuntimeError("Navigation page not found when in masterlist")

        sub_pageslist = [this_pagelist[page_ind]]
        page_level = sub_pageslist[0].depth
        ind = page_ind + 1
        while ind < len(this_pagelist) and this_pagelist[ind].depth > page_level:
            sub_pageslist.append(this_pagelist[ind])
            ind+=1
        return sub_pageslist, replace_links
