# pylint: disable=line-too-long, unused-wildcard-import
"""
|wq_widths_strickler| is a stateless submodel that requires information on the current
water level or depth from its main model (usually a routing model).  It returns the
corresponding discharge and related properties, such as the kinematic wave celerity.

|wq_widths_strickler| is very similar to |wq_trapeze_strickler| but more flexible and
thus a little more complicated.  The first difference is that it relies on defining
individual height/width pairs, which can be grouped into separate sectors by marking
the transitions between all neighbouring sectors.  The documentation on method
|WidthsModel.plot| of the base class |WidthsModel| visualises this based on some
trapeze-based profiles.

The second difference is that only |wq_widths_strickler| distinguishes between those
parts of a cross-section that are actively involved in water routing and those that are
not but still play a role due to providing storage volume.  This feature can help to
model retention effects of goyles or structured overbanks.  It is also visualised in
the documentation on method |WidthsModel.plot| and some of the following integration
tests show how it modifies 1-dimensional floodwave routing.


Integration tests
=================

.. how_to_understand_integration_tests::

We use two different main models, |musk_mct| and |kinw_impl_euler|, to emphasise
different aspects of |wq_widths_strickler|.  The following setting and input
time series stem from the integration tests of these models (that rely on
|wq_trapeze_strickler|), which allows for direct comparisons:

>>> from hydpy import Element, IntegrationTest, Nodes, pub
>>> pub.timegrids = "2000-01-01", "2000-01-05", "30m"
>>> nodes = Nodes("input1", "input2", "output")
>>> stream = Element("stream", inlets=["input1", "input2"], outlets="output")

>>> import numpy
>>> q_base = 100.0
>>> q_peak = 900.0
>>> t_peak = 24.0
>>> beta = 16.0
>>> ts = pub.timegrids.init.to_timepoints()
>>> series = (q_peak - q_base) * ((ts / t_peak) * numpy.exp(1.0 - ts / t_peak)) ** beta

Musk-MCT
________

We create the same channel configuration as used in the :ref:`musk_mct_long_segments`
example.  Note that the first two height/width pairs would principally suffice for
representing the same simple trapezoidal channel profile.  However, we prefer to add a
third pair at an extreme height to ensure the vertical orientation of the cross
section's-outlines above the highest height does not cause deviations from the original
example's results:

>>> from hydpy.models.musk_mct import *
>>> parameterstep()
>>> stream.model = model
>>> length(100.0)
>>> nmbsegments(4)
>>> bottomslope(0.00025)
>>> catchmentarea(25000.0)
>>> with model.add_wqmodel_v1("wq_widths_strickler") as submodel:
...     control = submodel.parameters.control
...     control.nmbwidths(3)
...     control.nmbsectors(1)
...     control.heights(0.0, 6.0, 100.0)
...     control.flowwidths(15.0, 75.0, 1015.0)
...     control.totalwidths(15.0, 75.0, 1015.0)
...     control.stricklercoefficients(1.0/0.035)
>>> IntegrationTest.plotting_options.axis1 = (states.discharge,)
>>> test = IntegrationTest(stream)
>>> nodes.input1.sequences.sim.series = q_base
>>> nodes.input2.sequences.sim.series = series

Please also note that, due to the presence of the third height/width pair given, the
second pair has absolutely no effect on the results, because (1) it lies in the direct
line between the first and the last pair and (2) we do define a single sector, which is
why the Manning-Strickler equation (and so the same friction coefficient) is always
applied on the complete cross-section.

.. _wq_widths_strickler_musk_mct_long_segments:

Long segments
-------------

The following results are identical to the ones of the :ref:`musk_mct_long_segments`
example (as to be expected due to the quasi-identical configuration):

.. integration-test::

    >>> model.prepare_states(100.0)
    >>> test("wq_widths_strickler_musk_mct_long_segments")
    |                date |                               referencewaterdepth |                                     wettedarea |                                     surfacewidth |                               celerity |                               correctingfactor |                                  coefficient1 |                               coefficient2 |                               coefficient3 |     inflow |                                     referencedischarge |    outflow |                               courantnumber |                               reynoldsnumber |                                                  discharge | input1 |     input2 |     output |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878653  124.878651  124.878652  124.878652 |  52.178282   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 | 100.000003 | 100.000001   99.999999       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 | 100.000003   99.999999  100.000001       100.0       100.0 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878663  124.878648  124.878654  124.878651 |  52.178283    52.17828   52.178282     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 | 100.000029 | 100.000012   99.999995  100.000002           99.999999 | 100.000001 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 | 100.000029   99.999989  100.000004   99.999998  100.000001 |  100.0 |   0.000029 | 100.000001 |
    | 2000-01-01 04:00:00 | 3.717829  3.717828  3.717828             3.717828 | 124.878717  124.878627  124.878662  124.878648 |  52.178294   52.178276   52.178283     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 | 100.000175 | 100.000071   99.999972  100.000011           99.999996 | 100.000004 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 | 100.000175   99.999932  100.000026    99.99999  100.000004 |  100.0 |   0.000175 | 100.000004 |
    | 2000-01-01 04:30:00 | 3.717834  3.717826  3.717829             3.717828 | 124.878958  124.878535  124.878696  124.878635 |   52.17834   52.178259    52.17829     52.178278 | 1.088386  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 | 0.908841   0.90884   0.90884       0.90884 | 100.000826 | 100.000333   99.999873  100.000048           99.999982 | 100.000017 | 0.057656  0.057656  0.057656       0.057656 |  0.20729  0.207289  0.207289        0.207289 | 100.000826   99.999686  100.000119   99.999955  100.000017 |  100.0 |   0.000826 | 100.000017 |
    | 2000-01-01 05:00:00 | 3.717851   3.71782  3.717831             3.717827 |  124.87983  124.878213  124.878815  124.878592 |  52.178507   52.178197   52.178313      52.17827 | 1.088389  1.088384  1.088386  1.088385 |  1.35916  1.359161  1.359161          1.359161 | -0.581094  -0.581097  -0.581096     -0.581096 | 0.672256  0.672255  0.672256      0.672256 | 0.908842   0.90884  0.908841       0.90884 | 100.003194 | 100.001282   99.999523  100.000177           99.999934 | 100.000061 | 0.057656  0.057656  0.057656       0.057656 |  0.20729  0.207289  0.207289        0.207289 | 100.003194    99.99881  100.000442   99.999836  100.000061 |  100.0 |   0.003194 | 100.000061 |
    | 2000-01-01 05:30:00 | 3.717902  3.717801  3.717838             3.717825 | 124.882516  124.877248   124.87916  124.878469 |  52.179022   52.178012   52.178379     52.178246 | 1.088397  1.088381  1.088387  1.088384 |  1.35916  1.359161   1.35916          1.359161 | -0.581091  -0.581098  -0.581095     -0.581096 | 0.672258  0.672255  0.672256      0.672256 | 0.908845  0.908839  0.908841       0.90884 | 100.010515 | 100.004206   99.998472  100.000553           99.999801 | 100.000179 | 0.057657  0.057656  0.057656       0.057656 | 0.207293  0.207288   0.20729        0.207289 | 100.010515   99.996177  100.001384   99.999501  100.000179 |  100.0 |   0.010515 | 100.000179 |
    | 2000-01-01 06:00:00 | 3.718041  3.717753  3.717855             3.717819 | 124.889772  124.874721  124.880033   124.87817 |  52.180412   52.177528   52.178546     52.178189 | 1.088418  1.088373  1.088389  1.088384 | 1.359158  1.359161   1.35916          1.359161 |  -0.58108  -0.581102  -0.581094     -0.581097 | 0.672261  0.672254  0.672256      0.672256 | 0.908853  0.908836  0.908842       0.90884 | 100.030315 | 100.012103   99.995722  100.001503           99.999475 | 100.000456 | 0.057658  0.057655  0.057656       0.057656 |   0.2073  0.207286  0.207291        0.207289 | 100.030315   99.989278   100.00377   99.998684  100.000456 |  100.0 |   0.030315 | 100.000456 |
    | 2000-01-01 06:30:00 | 3.718378   3.71764  3.717892             3.717807 | 124.907324  124.868822  124.881993  124.877528 |  52.183776   52.176397   52.178922     52.178066 |  1.08847  1.088356  1.088395  1.088382 | 1.359155  1.359163   1.35916          1.359161 | -0.581056   -0.58111  -0.581091     -0.581098 | 0.672266  0.672253  0.672257      0.672255 |  0.90887   0.90883  0.908844      0.908839 | 100.078179 | 100.031208   99.989301  100.003636           99.998776 |  100.00102 | 0.057661  0.057654  0.057657       0.057656 | 0.207317   0.20728  0.207293        0.207288 | 100.078179    99.97319  100.009114   99.996933   100.00102 |  100.0 |   0.078179 |  100.00102 |
    | 2000-01-01 07:00:00 | 3.719119    3.7174  3.717968             3.717783 | 124.946002  124.856327  124.885961  124.876294 |  52.191187   52.174003   52.179682     52.177829 | 1.088585  1.088319  1.088407  1.088378 | 1.359147  1.359165  1.359159          1.359161 | -0.581001  -0.581127  -0.581086     -0.581099 | 0.672272  0.672251  0.672257      0.672255 | 0.908905   0.90882  0.908847      0.908838 | 100.183351 | 100.073312   99.975702  100.007955           99.997434 | 100.002028 | 0.057667  0.057652  0.057657       0.057656 | 0.207354  0.207268  0.207296        0.207287 | 100.183351   99.939247  100.019883   99.993591  100.002028 |  100.0 |   0.183351 | 100.002028 |
    | 2000-01-01 07:30:00 | 3.720626  3.716934  3.718109             3.717742 | 125.024672  124.832018  124.893291  124.874154 |  52.206259   52.169343   52.181087     52.177419 | 1.088819  1.088246  1.088429  1.088372 | 1.359132   1.35917  1.359158          1.359161 | -0.580891  -0.581162  -0.581075     -0.581102 | 0.672278  0.672251  0.672257      0.672256 |  0.90897    0.9088  0.908852      0.908837 | 100.396214 | 100.158964   99.949246  100.015934           99.995105 | 100.003618 |  0.05768  0.057648  0.057658       0.057655 | 0.207429  0.207245  0.207303        0.207285 | 100.396214   99.873628  100.039612   99.987857  100.003618 |  100.0 |   0.396214 | 100.003618 |
    | 2000-01-01 08:00:00 | 3.723484  3.716093  3.718347             3.717677 | 125.173935  124.788138   124.90575  124.870788 |  52.234842   52.160931   52.183474     52.176774 | 1.089262  1.088116  1.088466  1.088362 | 1.359102  1.359178  1.359155          1.359162 | -0.580681  -0.581223  -0.581058     -0.581107 | 0.672276  0.672254  0.672254      0.672257 | 0.909083   0.90877   0.90886      0.908835 | 100.797305 | 100.321521   99.901495  100.029495           99.991441 | 100.005804 | 0.057705  0.057641   0.05766       0.057655 | 0.207572  0.207202  0.207315        0.207282 | 100.797305   99.756253  100.072747   99.978988  100.005804 |  100.0 |   0.797305 | 100.005804 |
    | 2000-01-01 08:30:00 | 3.728583  3.714673  3.718723             3.717585 | 125.440382  124.714086  124.925346   124.86599 |  52.285826   52.146733   52.187229     52.175855 | 1.090052  1.087896  1.088524  1.088347 |  1.35905  1.359193  1.359151          1.359163 | -0.580308  -0.581328   -0.58103     -0.581114 | 0.672249  0.672268  0.672249      0.672259 | 0.909266  0.908723  0.908871      0.908833 | 101.507032 | 100.611865   99.820924  100.050826           99.986219 | 100.008338 | 0.057749  0.057629  0.057664       0.057654 | 0.207827  0.207131  0.207334        0.207277 | 101.507032   99.560533  100.124003   99.966666  100.008338 |  100.0 |   1.507032 | 100.008338 |
    | 2000-01-01 09:00:00 | 3.737195  3.712419  3.719271             3.717468 | 125.891047  124.596584  124.953927  124.859857 |  52.371948   52.124195   52.192706     52.174679 | 1.091386  1.087546  1.088609  1.088329 | 1.358962  1.359216  1.359146          1.359164 | -0.579677  -0.581493   -0.58099     -0.581122 | 0.672166    0.6723  0.672237      0.672263 | 0.909547  0.908658  0.908884      0.908832 | 102.694853 | 101.103426   99.693112  100.081939           99.979544 |  100.01054 | 0.057823  0.057609  0.057668       0.057653 | 0.208258  0.207019  0.207361        0.207271 | 102.694853   99.254644  100.197098   99.951541   100.01054 |  100.0 |   2.694853 |  100.01054 |
    | 2000-01-01 09:30:00 | 3.751048  3.709042   3.72001             3.717337 | 126.617518  124.420603   124.99252  124.853035 |  52.510478   52.090422     52.2001     52.173372 | 1.093529  1.087022  1.088723  1.088309 | 1.358822  1.359251  1.359138          1.359166 | -0.578665  -0.581741  -0.580936     -0.581132 | 0.671976  0.672365  0.672215       0.67227 | 0.909953  0.908573  0.908896      0.908831 | 104.586356 | 101.897084   99.501767  100.123954           99.972119 | 100.011152 | 0.057943   0.05758  0.057675       0.057652 |  0.20895   0.20685  0.207398        0.207265 | 104.586356   98.805047  100.292783   99.935895  100.011152 |  100.0 |   4.586356 | 100.011152 |
    | 2000-01-01 10:00:00 | 3.772364  3.704255  3.720927             3.717221 | 127.739105  124.171337  125.040369  124.846996 |  52.723639   52.042547   52.209265     52.172214 | 1.096822  1.086278  1.088866  1.088291 | 1.358607  1.359301  1.359129          1.359167 | -0.577111  -0.582093  -0.580869     -0.581141 | 0.671599  0.672482   0.67218      0.672279 | 0.910509  0.908471  0.908906      0.908834 | 107.466641 | 103.125447   99.230898  100.176052           99.965547 |  100.00827 | 0.058127  0.057538  0.057683       0.057651 | 0.210015   0.20661  0.207444        0.207259 | 107.466641   98.182852  100.406193   99.924325   100.00827 |  100.0 |   7.466641 |  100.00827 |
    | 2000-01-01 10:30:00 | 3.803855  3.697836  3.721948              3.71717 | 129.404414  123.837487  125.093715    124.8443 |  53.038555   51.978358   52.219482     52.171697 | 1.101674  1.085281  1.089024  1.088283 | 1.358296  1.359368  1.359118          1.359167 | -0.574823  -0.582564  -0.580794     -0.581144 | 0.670921  0.672673  0.672129      0.672291 | 0.911232  0.908361  0.908907      0.908841 | 111.678559 | 104.956077   98.868406  100.234143           99.962614 |  99.999413 | 0.058397  0.057483  0.057692        0.05765 | 0.211587  0.206289  0.207495        0.207256 | 111.678559     97.3735  100.523695    99.92429   99.999413 |  100.0 |  11.678559 |  99.999413 |
    | 2000-01-01 11:00:00 | 3.848658  3.689721  3.722918             3.717256 |  131.79072  123.416028  125.144382  124.848781 |  53.486582   51.897211   52.229184     52.172556 | 1.108552   1.08402  1.089174  1.088296 | 1.357863  1.359452  1.359108          1.359166 | -0.571585  -0.583162  -0.580723     -0.581138 | 0.669786  0.672964   0.67206      0.672304 | 0.912117  0.908259  0.908893      0.908855 | 117.614836 |  107.59328   98.411265  100.289324           99.967491 |  99.981791 |  0.05878  0.057412    0.0577       0.057651 |  0.21382  0.205883  0.207544        0.207261 | 117.614836   96.389821   100.61956   99.946338   99.981791 |  100.0 |  17.614836 |  99.981791 |
    | 2000-01-01 11:30:00 | 3.910188  3.680124  3.723569             3.717577 | 135.100656  122.918433  125.178341  124.865567 |  54.101877   51.801242   52.235685     52.175774 | 1.117953  1.082527  1.089275  1.088346 | 1.357285  1.359553  1.359102          1.359163 |  -0.56717  -0.583869  -0.580675     -0.581114 | 0.667997  0.673377  0.671976      0.672315 | 0.913135  0.908193  0.908853      0.908876 | 125.703736 | 111.278182   97.872232  100.326312            99.98576 |  99.952811 | 0.059304  0.057329  0.057706       0.057654 | 0.216882  0.205402  0.207576        0.207277 | 125.703736   95.287979  100.652992  100.003781   99.952811 |  100.0 |  25.703736 |  99.952811 |
    | 2000-01-01 12:00:00 | 3.991919  3.669682  3.723495             3.718253 |  139.55587  122.378063  125.174507  124.900845 |   54.91919   51.696821   52.234951     52.182534 | 1.130361  1.080901  1.089264  1.088451 | 1.356548  1.359663  1.359102          1.359156 |  -0.56136  -0.584639  -0.580681     -0.581065 | 0.665323   0.67393  0.671884      0.672319 | 0.914224    0.9082  0.908779      0.908906 | 136.388647 | 116.286724   97.287706  100.322132          100.024158 |  99.910852 | 0.059995  0.057238  0.057705        0.05766 |  0.22094  0.204879  0.207573        0.207311 | 136.388647    94.18499  100.566289  100.111595   99.910852 |  100.0 |  36.388647 |  99.910852 |
    | 2000-01-01 12:30:00 | 4.097098  3.659613  3.722148             3.719411 | 145.387535  121.858017  125.104158  124.961248 |  55.970981   51.596127   52.221482     52.194108 |   1.1462  1.079331  1.089055   1.08863 | 1.355648   1.35977  1.359116          1.359144 | -0.553974  -0.585383  -0.580779      -0.58098 | 0.661509  0.674624    0.6718      0.672309 | 0.915286  0.908332   0.90866      0.908945 | 150.102726 | 122.925104      96.726  100.245506           100.08991 |  99.856289 | 0.060876  0.057151  0.057693        0.05767 | 0.226147  0.204374  0.207505        0.207368 | 150.102726   93.275474  100.285189  100.284286   99.856289 |  100.0 |  50.102726 |  99.856289 |
    | 2000-01-01 13:00:00 | 4.228418  3.651861  3.718841             3.721164 | 152.823881  121.458358  124.931493  125.052768 |  57.284183    51.51861   52.188407      52.21164 | 1.165779  1.078121  1.088542  1.088902 | 1.354594  1.359853   1.35915          1.359126 | -0.544891  -0.585956  -0.581022     -0.580851 | 0.656307  0.675434   0.67175      0.672277 | 0.916196  0.908649  0.908488      0.908991 | 167.240349 | 131.521964   96.294888  100.057502          100.189557 |  99.792735 | 0.061964  0.057083  0.057665       0.057685 | 0.232626  0.203986   0.20734        0.207456 | 167.240349   92.844349   99.722725  100.532538   99.792735 |  100.0 |  67.240349 |  99.792735 |
    | 2000-01-01 13:30:00 | 4.387894  3.649242  3.712774              3.72359 | 162.086462  121.323445  124.615066  125.179459 |  58.878937   51.492416    52.12774     52.235899 | 1.189277  1.077712  1.087601  1.089278 | 1.353409   1.35988  1.359213          1.359101 | -0.534058   -0.58615  -0.581467     -0.580674 | 0.649517    0.6763  0.671768      0.672211 | 0.916833  0.909216  0.908259      0.909037 | 188.127569 | 142.429764   96.149483   99.713198          100.327536 |  99.727995 | 0.063268   0.05706  0.057612       0.057706 | 0.240463  0.203854  0.207036        0.207578 | 188.127569   93.277116   98.785758  100.859457   99.727995 |  100.0 |  88.127569 |  99.727995 |
    | 2000-01-01 14:00:00 |  4.57581  3.655275  3.703201             3.726652 | 173.327322  121.634305  124.116511  125.339472 |  60.758098   51.552751   52.032012     52.266523 | 1.216596  1.078654  1.086115  1.089753 | 1.352133  1.359815  1.359312           1.35907 | -0.521561  -0.585703   -0.58217     -0.580449 | 0.640958  0.677089  0.671909      0.672101 | 0.917024  0.910075  0.907984      0.909074 | 212.993962 | 155.953114   96.484681   99.171329          100.501875 |   99.67711 | 0.064783  0.057113  0.057529       0.057732 | 0.249657  0.204157  0.206557        0.207731 | 212.993962   95.029868   97.400118  101.250369    99.67711 |  100.0 | 112.993962 |   99.67711 |
    | 2000-01-01 14:30:00 | 4.791858  3.674415   3.68942             3.730214 | 186.687365  122.622837  123.400398  125.525678 |  62.918577   51.744147     51.8942     52.302137 | 1.247541  1.081638  1.083973  1.090305 | 1.350807  1.359612  1.359456          1.359033 | -0.507531  -0.584289  -0.583184     -0.580188 |  0.63066  0.677635  0.672223      0.671939 | 0.916733  0.911275  0.907672      0.909092 | 241.948195 | 172.415486   97.552489   98.394304          100.704846 |  99.659742 | 0.066496   0.05728   0.05741       0.057763 | 0.260177  0.205116  0.205867        0.207909 | 241.948195   98.647581   95.515096  101.673842   99.659742 |  100.0 | 141.948195 |  99.659742 |
    | 2000-01-01 15:00:00 | 5.034228  3.711521  3.671088             3.733928 | 202.230689  124.549766  122.450745      125.72 |  65.342282   52.115212   51.710878     52.339278 | 1.281704  1.087407   1.08112   1.09088 | 1.349476  1.359223  1.359649          1.358996 | -0.492198  -0.581558  -0.584535     -0.579917 | 0.618677   0.67766  0.672778      0.671722 | 0.915896  0.912787  0.907362      0.909072 |  274.95935 | 192.074476   99.642363   97.366248           100.91677 |   99.70227 | 0.068384  0.057601  0.057251       0.057795 | 0.271921  0.206974  0.204949        0.208094 |  274.95935  104.704416   93.142778  102.070179    99.70227 |  100.0 |  174.95935 |   99.70227 |
    | 2000-01-01 15:30:00 |  5.29996  3.771591  3.648412             3.737208 | 219.947254  127.698353  121.280748  125.891743 |  67.999596   52.715909   51.484123     52.372081 | 1.318538  1.096702  1.077583  1.091388 |  1.34818  1.358613   1.35989          1.358962 | -0.475852  -0.577166  -0.586212     -0.579677 | 0.605203  0.676828  0.673634      0.671455 | 0.914523  0.914535  0.907108      0.908996 | 311.845554 | 215.111403  103.080918   96.103399          101.104163 |  99.836036 | 0.070417   0.05812  0.057053       0.057823 | 0.284732  0.209977  0.203813        0.208258 | 311.845554  113.768341   90.386002  102.347498   99.836036 |  100.0 | 211.845554 |  99.836036 |
    | 2000-01-01 16:00:00 | 5.585172  3.859237  3.622409             3.739201 | 239.748304   132.35708  119.945391  125.996158 |  70.851719   53.592365   51.224094     52.392014 | 1.357401  1.110172  1.073518  1.091697 |  1.34695   1.35776  1.360171          1.358942 | -0.458816  -0.570823  -0.588141     -0.579531 | 0.590515  0.674746  0.674838      0.671157 |  0.91267  0.916369  0.906985      0.908844 | 352.270855 | 241.608497   108.22176   94.667148           101.21813 | 100.093373 | 0.072559  0.058871  0.056826       0.057841 | 0.298416  0.214347  0.202508        0.208358 | 352.270855  126.354752   87.465528  102.381498  100.093373 |  100.0 | 252.270855 | 100.093373 |
    | 2000-01-01 16:30:00 | 5.885159  3.978264  3.595198             3.738769 |  261.45286  138.806893  118.555199  125.973499 |  73.851589   54.782642   50.951977     52.387689 | 1.397583  1.128295  1.069252   1.09163 |  1.34581  1.356667  1.360469          1.358947 | -0.441429  -0.562326  -0.590167     -0.579563 | 0.574938  0.671019  0.676407      0.670862 | 0.910417  0.918088  0.907094      0.908598 | 395.750601 | 271.510751  115.441037   93.177717          101.193385 | 100.502931 |  0.07477   0.05988  0.056588       0.057837 | 0.312742  0.220263  0.201142        0.208336 | 395.750601  142.873598   84.746077  102.017647  100.502931 |  100.0 | 295.750601 | 100.502931 |
    | 2000-01-01 17:00:00 | 6.195328  4.130557  3.569896             3.734652 | 284.840396  147.265835  117.269206  125.757915 |  76.953284   56.305565   50.698956     52.346521 | 1.438433  1.151209  1.065276  1.090992 | 1.344771  1.355371  1.360751          1.358989 |  -0.42399  -0.551645  -0.592058     -0.579864 |  0.55885  0.665258  0.678287      0.670635 | 0.907894  0.919413  0.907522      0.908257 | 441.664962 | 304.679205  125.082912   91.805234          100.958102 | 101.075512 | 0.077015  0.061155  0.056366       0.057801 |  0.32749    0.2278   0.19987        0.208131 | 441.664962  163.617961   82.705477  101.098976  101.075512 |  100.0 | 341.664962 | 101.075512 |
    | 2000-01-01 17:30:00 | 6.510569  4.316782  3.551098             3.725397 | 309.596078  157.924771  116.317956  125.273892 |   80.10569   58.167821    50.51098     52.253974 | 1.479279  1.178836  1.062315  1.089559 | 1.343839  1.353924  1.360963          1.359083 | -0.406791  -0.538862  -0.593467     -0.580541 |  0.54256  0.657298  0.680396      0.670541 | 0.905183  0.920178  0.908394      0.907814 | 489.279607 | 340.799058  137.502117   90.793327          100.430369 | 101.804743 | 0.079257  0.062689    0.0562       0.057721 | 0.342419  0.236973  0.198925        0.207668 | 489.279607  188.699206   81.966749   99.465713  101.804743 |  100.0 | 389.279607 | 101.804743 |
    | 2000-01-01 18:00:00 | 6.826008  4.535517  3.544319             3.709716 | 335.362024  170.887314  115.975781  124.455729 |  83.260077   60.355168   50.443192     52.097165 | 1.519519  1.210771  1.061246  1.087126 | 1.343012  1.352396  1.361039          1.359245 |  -0.39008  -0.524216  -0.593976     -0.581692 | 0.526387  0.647063  0.682513      0.670681 | 0.902385  0.920214   0.90977      0.907292 |  537.77215 | 379.437277  152.991805    90.43006           99.539908 | 102.649918 | 0.081463   0.06446  0.056141       0.057586 | 0.357303   0.24769  0.198584        0.206883 |  537.77215  218.055518   83.218981   97.004169  102.649918 |  100.0 |  437.77215 | 102.649918 |
    | 2000-01-01 18:30:00 | 7.137058  4.783577  3.555781              3.68669 | 361.743837  186.166726  116.554603  123.258774 |  86.370578   62.835774   50.557809     51.866902 | 1.558619  1.246364  1.063053  1.083548 | 1.342286  1.350855  1.360908          1.359485 | -0.374063  -0.508061  -0.593115     -0.583385 | 0.510598  0.634697  0.684323       0.67116 | 0.899584  0.919466  0.911659      0.906736 | 586.262607 | 420.045204  171.766371   91.044925           98.240787 | 103.527965 | 0.083604  0.066431  0.056242       0.057386 | 0.371934  0.259775  0.199161        0.205731 | 586.262607  251.445898   87.166367   93.683686  103.527965 |  100.0 | 486.262607 | 103.527965 |
    | 2000-01-01 19:00:00 | 7.439528  5.056408   3.59184              3.65608 | 388.325803  203.682453  118.384179  121.675784 |   89.39528   65.564084   50.918401     51.560796 | 1.596121  1.284803  1.068725   1.07878 | 1.341652  1.349361  1.360503          1.359809 | -0.358903  -0.490815  -0.590416     -0.585645 | 0.495414  0.620517  0.685398       0.67208 | 0.896843  0.917977   0.91398       0.90621 | 633.846018 | 461.978876   193.93754   92.995149           96.529295 | 104.309247 | 0.085656  0.068555  0.056559        0.05712 |  0.38612  0.272993  0.200974        0.204197 | 633.846018  288.461092   94.462252   89.596621  104.309247 |  100.0 | 533.846018 | 104.309247 |
    | 2000-01-01 19:30:00 | 7.729694   5.34855  3.658291             3.618653 | 414.686212  223.263194  121.789846  119.753048 |  92.296935   68.485501   51.582913     51.186531 | 1.631644  1.325207  1.079125   1.07293 | 1.341103  1.347959   1.35978          1.360212 | -0.344725  -0.472914  -0.585479      -0.58842 | 0.481007  0.604951   0.68522      0.673527 | 0.894209  0.915862  0.916538      0.905811 | 679.625318 | 504.525034  219.494726   96.652743            94.46069 | 104.818785 | 0.087598  0.070785  0.057139       0.056793 | 0.399694  0.287068  0.204309        0.202319 | 679.625318  328.544691   105.64603   84.992626  104.818785 |  100.0 | 579.625318 | 104.818785 |
    | 2000-01-01 20:00:00 | 8.004332   5.65411  3.759772             3.576495 | 440.411628  244.656483  127.075988  117.604018 |  95.043319   71.541105   52.597716     50.764952 | 1.664877  1.366696  1.094877  1.066314 |  1.34063  1.346675  1.358729          1.360678 | -0.331621  -0.454774  -0.578026     -0.591564 | 0.467508  0.588473  0.683245      0.675551 | 0.891716  0.913277  0.919055      0.905658 | 722.742733 | 546.930209  248.293743  102.399101           92.162023 | 104.844225 | 0.089414   0.07307  0.058018       0.056424 | 0.412514  0.301714  0.209388        0.200202 | 722.742733  371.021466  121.092917   80.300104  104.844225 |  100.0 | 622.742733 | 104.844225 |
    | 2000-01-01 20:30:00 | 8.260733  5.966973    3.8984             3.533072 | 465.109512  267.528399  134.463609  115.409065 |  97.607327   74.669726   53.983999     50.330719 | 1.695578  1.408424  1.116156  1.059471 | 1.340225  1.345524   1.35739           1.36117 | -0.319652  -0.436778  -0.568011     -0.594822 |  0.45501  0.571542  0.678931      0.678134 | 0.889384  0.910371  0.921131       0.90588 | 762.408181 | 588.430697  280.034809  110.566868           89.829016 |  104.16117 |  0.09109  0.075366  0.059204       0.056041 | 0.424461  0.316638  0.216297        0.198017 | 762.408181  415.130083  140.974098   76.105761   104.16117 |  100.0 | 662.408181 |  104.16117 |
    | 2000-01-01 21:00:00 | 8.496687  6.281784  4.074749             3.493376 | 488.418803  291.530849  144.139111  113.419005 |  99.966875   77.817845   55.747486     49.933757 | 1.723565  1.449701  1.142846  1.053188 |  1.33988  1.344505  1.355832          1.361631 | -0.308854  -0.419222  -0.555534     -0.597818 | 0.443576  0.554601  0.672034      0.681188 | 0.887225  0.907322  0.922527      0.906618 | 797.923464 |  628.28149  314.340692   121.49651            87.72687 | 102.556077 | 0.092618  0.077633   0.06069        0.05569 | 0.435436  0.331589  0.225043        0.196017 | 797.923464  460.057558  165.274061   73.133666  102.556077 |  100.0 | 697.923464 | 102.556077 |
    | 2000-01-01 21:30:00 | 8.710468  6.593308  4.286613             3.463618 | 510.018278  316.258206  156.174461  111.937508 | 102.104679   80.933084   57.866132     49.636178 | 1.748711  1.489894  1.174389  1.048462 | 1.339588  1.343613  1.354148          1.361983 | -0.299245  -0.402361  -0.540912     -0.600075 | 0.433243  0.537971  0.662422      0.684507 | 0.885244  0.904226  0.922988      0.907975 | 828.701418 | 665.782574   350.68953   135.44274           86.170115 |  99.868602 | 0.093989  0.079839  0.062442       0.055426 | 0.445366  0.346328   0.23549        0.194515 | 828.701418  504.973156  193.761421    72.17475   99.868602 |  100.0 | 728.701418 |  99.868602 |
    | 2000-01-01 22:00:00 | 8.900795  6.897033  4.529813             3.450789 |  529.63265  341.300839  170.543249  111.301568 | 104.007947   83.970333   60.298134     49.507892 | 1.770934  1.528496  1.209945   1.04642 | 1.339344   1.34284  1.352433          1.362136 | -0.290824  -0.386384  -0.524593     -0.601051 | 0.424026  0.521939  0.650292      0.687761 |  0.88344  0.901185  0.922453      0.909992 |  854.27953 | 700.301394  388.487969  152.575413           85.504072 |  96.038156 | 0.095201  0.081954  0.064414       0.055312 | 0.454197  0.360648  0.247411        0.193868 |  854.27953  549.059907  226.032088   74.004435   96.038156 |  100.0 |  754.27953 |  96.038156 |
    | 2000-01-01 22:30:00 | 9.066802  7.189125  4.798717             3.462058 | 547.036472  366.254479  187.119152  111.860085 | 105.668015   86.891251   62.987166     49.620577 | 1.790195   1.56511  1.248515  1.048214 | 1.339142  1.342173  1.350767          1.361999 |  -0.28358  -0.371425   -0.50709     -0.600193 | 0.415927  0.506709  0.636058      0.690489 | 0.881809  0.898265  0.920993      0.912608 | 874.327871 | 731.290705  427.090043  172.954491           86.089105 |   91.14655 | 0.096251  0.083959   0.06655       0.055412 | 0.461891  0.374378  0.260511        0.194437 | 874.327871  591.542379  261.536665   79.304348    91.14655 |  100.0 | 774.327871 |   91.14655 |
    | 2000-01-01 23:00:00 | 9.207997  7.466423  5.086901             3.504007 | 562.056028  390.733671  205.686327  113.950433 | 107.079973   89.664226   65.869011     50.040071 |  1.80649  1.599432  1.289056  1.054873 | 1.338977    1.3416  1.349206          1.361501 |  -0.27749  -0.357574   -0.48892     -0.597012 | 0.408933  0.492427  0.620265      0.692115 | 0.880344   0.89551  0.918779      0.915632 | 888.651531 | 758.301449  465.825892   196.51638           88.287315 |  85.451249 | 0.097139  0.085837   0.06879       0.055785 |  0.46843  0.387379  0.274465        0.196554 | 888.651531  631.709843  299.617153   88.596376   85.451249 |  100.0 | 788.651531 |  85.451249 |
    | 2000-01-01 23:30:00 | 9.324227  7.726403  5.387747             3.581976 | 574.569426  414.382579  225.955306  117.882397 | 108.242268   92.264032   68.877472     50.819759 | 1.819844  1.631243  1.330572  1.067176 | 1.338847   1.34111  1.347785           1.36061 | -0.272527  -0.344885  -0.470554     -0.591152 | 0.403021  0.479189  0.603495      0.692008 | 0.879038  0.892945  0.916022      0.918754 | 897.187983 | 780.990593  504.029431  223.069512           92.459425 |   79.40292 | 0.097867  0.087576   0.07108       0.056472 |  0.47381   0.39954  0.288951        0.200479 | 897.187983  668.934343  339.545148  102.199141    79.40292 |  100.0 | 797.187983 |   79.40292 |
    | 2000-01-02 00:00:00 | 9.415632  7.967131  5.694736             3.698593 | 584.505141  436.882893  247.571158  123.876823 | 109.156323   94.671315   71.947364     51.985926 | 1.830309  1.660397  1.372156  1.085399 | 1.338747  1.340692  1.346518          1.359353 | -0.268653  -0.333379  -0.452405     -0.582506 | 0.398162   0.46705  0.586287       0.68951 | 0.877882  0.890582   0.91292      0.921507 |      900.0 | 799.124185  541.062935  252.285099           98.911569 |   73.61812 | 0.098437  0.089169  0.073371        0.05749 | 0.478038  0.410779  0.303655        0.206329 |      900.0  702.683574  380.558409  120.204024    73.61812 |  100.0 |      800.0 |   73.61812 |
    | 2000-01-02 00:30:00 | 9.482616  8.187198  6.002264             3.854722 | 591.839295  457.959014  270.169838  132.115248 | 109.826162   96.871979   75.022642     53.547222 | 1.837958  1.686804  1.413086  1.109481 | 1.338676  1.340338  1.345404            1.3578 | -0.265832   -0.32306  -0.434783     -0.571146 | 0.394322  0.456033  0.569129      0.684267 | 0.876867  0.888424  0.909675      0.923577 | 897.264896 | 812.576082  576.337507  283.761114          107.953588 |  68.854546 | 0.098854  0.090611  0.075622       0.058832 | 0.481135  0.421036  0.318317        0.214124 | 897.264896  732.528757  421.893769  142.501224   68.854546 |  100.0 | 797.264896 |  68.854546 |
    | 2000-01-02 01:00:00 | 9.525805  8.385655  6.305206             4.048225 |  596.59193  477.380872  293.356216  142.663993 | 110.258055    98.85655   78.052062     55.482248 | 1.842881  1.710427  1.452745  1.138858 | 1.338631   1.34004  1.344435          1.356056 |  -0.26402   -0.31391  -0.417937     -0.557392 | 0.391463   0.44614  0.552372      0.676056 | 0.875985  0.886467  0.906411      0.924628 | 889.260928 |  821.32296  609.328938  316.989555          119.813639 |  65.927469 | 0.099122  0.091901    0.0778       0.060468 | 0.483132  0.430273  0.332699         0.22373 | 889.260928  758.147887  462.815168  168.779253   65.927469 |  100.0 | 789.260928 |  65.927469 |
    | 2000-01-02 01:30:00 | 9.546018  8.561956   6.59931             4.274905 | 598.822552  494.964743  316.744125  155.497653 | 110.460179  100.619555   80.993102     57.749052 | 1.845183  1.731263  1.490662   1.17266 |  1.33861  1.339789  1.343598          1.354236 | -0.263174  -0.305904  -0.402041      -0.54171 | 0.389543  0.437356  0.536298       0.66504 | 0.875228  0.884704  0.903233      0.924562 | 876.351729 | 825.436312  639.588887  351.413508          134.648494 |  65.629814 | 0.099247  0.093038  0.079881       0.062346 | 0.484066  0.438469  0.346611        0.234915 | 876.351729  779.324917  502.636265  198.573393   65.629814 |  100.0 | 776.351729 |  65.629814 |
    | 2000-01-02 02:00:00 | 9.544231  8.715894  6.881123             4.528469 | 598.625179  510.572439  339.966124  170.462192 | 110.442309  102.158939   83.811231      60.28469 | 1.844979  1.749347  1.526488   1.20975 | 1.338611  1.339582  1.342879          1.352442 | -0.263249  -0.299003   -0.38721     -0.524682 |  0.38852  0.429651  0.521102        0.6516 | 0.874588  0.883124  0.900208      0.923394 | 858.969595 | 825.072141  666.751615  386.449009          152.477329 |    68.6319 | 0.099236  0.094024  0.081844       0.064404 | 0.483983  0.445618  0.359898        0.247345 | 858.969595  795.945527  540.737537  231.303215     68.6319 |  100.0 | 758.969595 |    68.6319 |
    | 2000-01-02 02:30:00 | 9.521553  8.847553  7.147934             4.802635 | 596.123115  524.109262  362.683824  187.366055 | 110.215527   103.47553   86.479341     63.026352 | 1.842397  1.764732  1.559976  1.249072 | 1.338635  1.339411  1.342263          1.350744 | -0.264198  -0.293167  -0.373511      -0.50684 | 0.388349   0.42299   0.50691      0.636396 | 0.874057  0.881717  0.897377      0.921377 |   837.5984 | 820.459102  690.536734  421.510527          173.262774 |  75.458274 | 0.099095  0.094863  0.083678        0.06658 | 0.482935  0.451727  0.372444        0.260701 |   837.5984   807.99013  576.578143  266.312961   75.458274 |  100.0 |   737.5984 |  75.458274 |
    | 2000-01-02 03:00:00 | 9.479195  8.957256  7.397697             5.090451 | 591.463655  535.521026  384.595039  205.920202 | 109.791954  104.572561   88.976968     65.904507 | 1.837568  1.777497  1.590964   1.28955 | 1.338679  1.339274  1.341737          1.349189 | -0.265975   -0.28835  -0.360976       -0.4887 | 0.388987  0.417331  0.493794       0.62001 | 0.873629   0.88047  0.894761      0.918706 | 812.756821 | 811.885741  710.748529  456.033301          196.817835 |  86.408892 | 0.098832  0.095559  0.085374       0.068817 | 0.480977  0.456814  0.384159        0.274637 | 812.756821   815.52483  609.703051  302.909099   86.408892 |  100.0 | 712.756821 |  86.408892 |
    | 2000-01-02 03:30:00 | 9.418451  9.045525  7.628943             5.385423 | 584.812859  544.790436  405.438012  225.795256 |  109.18451  105.455245   91.289431     68.854231 | 1.830632  1.787732  1.619358  1.330254 | 1.338744  1.339167  1.341289          1.347796 | -0.268534  -0.284503  -0.349609     -0.470694 | 0.390387   0.41263  0.481789      0.603055 | 0.873298  0.879373  0.892366      0.915629 | 784.982427 | 799.687452  727.272474  489.491469           222.85647 | 101.568214 | 0.098455  0.096117  0.086927       0.071063 | 0.478168  0.460905  0.394984        0.288839 | 784.982427  818.690957  639.746032  340.392999  101.568214 |  100.0 | 684.982427 | 101.568214 |
    | 2000-01-02 04:00:00 | 9.340669   9.11304  7.840698             5.681494 | 576.350556  551.933078   424.99323  246.619275 | 108.406693  106.130399   93.406984     71.814939 | 1.821729   1.79554  1.645121  1.370378 | 1.338829  1.339087  1.340908           1.34657 | -0.271828  -0.281579  -0.339393     -0.453177 | 0.392506  0.408839  0.470898      0.586031 | 0.873057  0.878413  0.890191      0.912341 | 754.817086 | 784.233653  740.069587  521.411689          250.979681 | 120.805106 |  0.09797  0.096543  0.088335       0.073273 |  0.47457  0.464033  0.404878        0.303022 | 754.817086  817.693771  666.429205  378.087807  120.805106 |  100.0 | 654.817086 | 120.805106 |
    | 2000-01-02 04:30:00 | 9.247238  9.160612  8.032401             5.973954 | 566.265664  556.993208  443.083353  268.049935 | 107.472384  106.606117   95.324011     74.739539 | 1.811004   1.80103  1.668253  1.409347 | 1.338933  1.339032  1.340585          1.345501 |  -0.27581  -0.279527  -0.330299     -0.436383 | 0.395298  0.405911  0.461104       0.56937 | 0.872903  0.877582  0.888226      0.909032 |  722.79402 | 765.915628  749.169203  551.382426           280.76932 | 143.831832 | 0.097385  0.096842  0.089598       0.075417 | 0.470247  0.466236  0.413823        0.316969 |  722.79402   812.79086  689.559816  415.359329  143.831832 |  100.0 |  622.79402 | 143.831832 |
    | 2000-01-02 05:00:00 | 9.139567  9.189149  8.203832              6.25855 | 554.751903  560.039502  459.571733  289.725497 | 106.395668  106.891487   97.038315     77.585501 | 1.798602  1.804319  1.688791  1.446678 | 1.339056  1.338999  1.340313          1.344577 | -0.280433    -0.2783  -0.322287       -0.4205 | 0.398718  0.403797  0.452377      0.553336 | 0.872829  0.876868  0.886461      0.905793 | 689.426711 | 745.135325  754.660717  579.059321          311.725884 |  170.20398 | 0.096709  0.097021   0.09072       0.077467 | 0.465262  0.467558  0.421811        0.330487 | 689.426711  804.280661   709.02488  451.631237   170.20398 |  100.0 | 589.426711 |  170.20398 |
    | 2000-01-02 05:30:00 | 9.019068  9.199635  8.355049             6.531967 | 542.003966  561.160899  474.359936  311.312464 | 105.190681  106.996346   98.550488     80.319669 | 1.784668  1.805527  1.706795  1.482029 | 1.339199  1.338987  1.340085          1.343782 | -0.285653  -0.277849  -0.315312     -0.405643 | 0.402724  0.402453  0.444678      0.538141 | 0.872833  0.876264  0.884883      0.902703 | 655.199767 | 722.295367  756.684775  604.167195          343.339972 | 199.381888 |  0.09595  0.097087  0.091703       0.079407 | 0.459679  0.468043  0.428849        0.343429 | 655.199767  792.491457    724.7843  486.395168  199.381888 |  100.0 | 555.199767 | 199.381888 |
    | 2000-01-02 06:00:00 | 8.887147  9.193106  8.486337             6.791652 | 528.214162  560.462628  487.384614  332.507461 | 103.871475  106.931064   99.863368      82.91652 | 1.769345  1.804775  1.722343  1.515166 | 1.339361  1.338994  1.339895            1.3431 | -0.291424   -0.27813  -0.309324     -0.391878 | 0.407271   0.40183   0.43796      0.523914 |  0.87291   0.87576  0.883478      0.899808 | 620.561767 | 697.790409  755.424304  626.499299          375.105399 | 230.765266 | 0.095115  0.097046  0.092551       0.081224 | 0.453564  0.467741  0.434955        0.355684 | 620.561767  777.771123  736.862967  519.216443  230.765266 |  100.0 | 520.561767 | 230.765266 |
    | 2000-01-02 06:30:00 |  8.74519  9.170637  8.598158             7.035717 | 513.569559  558.062493  498.613985  353.042308 | 102.451897  106.706372  100.981581     85.357168 | 1.752776  1.802186  1.735525  1.545941 | 1.339543   1.33902   1.33974          1.342515 | -0.297699  -0.279095  -0.304273     -0.379234 | 0.412316  0.401885  0.432175       0.51073 | 0.873057   0.87535  0.882233       0.89713 | 585.920022 | 671.999903  751.095708  645.914413          406.537274 | 263.727396 | 0.094211  0.096905   0.09327        0.08291 | 0.446978    0.4667  0.440151        0.367171 | 585.920022  760.477815  745.342307  549.736153  263.727396 |  100.0 | 485.920022 | 263.727396 |
    | 2000-01-02 07:00:00 | 8.594549   9.13332  8.691115             7.262838 | 498.249603  554.087455  508.044139  372.686692 | 100.945491  106.333198  101.911151     87.628385 | 1.735101  1.797881  1.746443  1.574274 | 1.339744  1.339063  1.339614          1.342016 | -0.304435  -0.280703  -0.300108     -0.367711 | 0.417817  0.402575  0.427274      0.498618 |  0.87327  0.875027  0.881135      0.894678 |  551.63713 | 645.282291   743.94049  662.332342          437.186397 | 297.643833 | 0.093247   0.09667  0.093866       0.084461 | 0.439983  0.464972  0.444468        0.377836 |  551.63713  740.971724  750.351626  577.670353  297.643833 |  100.0 |  451.63713 | 297.643833 |
    | 2000-01-02 07:30:00 | 8.436541  9.082254  8.765918             7.472165 | 482.424211  548.670548  515.695321  391.248755 |  99.365408  105.822545  102.659176     89.721653 | 1.716455  1.791982  1.755201  1.600138 | 1.339966  1.339123  1.339516          1.341589 | -0.311588   -0.28291  -0.296779     -0.357291 | 0.423733  0.403858  0.423206      0.487581 | 0.873546  0.874785  0.880172      0.892448 | 518.029154 | 617.970552  734.217475  675.728322          466.649613 | 331.915055 |  0.09223  0.096349  0.094343       0.085876 |  0.43264  0.462607   0.44794        0.387647 | 518.029154  719.607964  752.059583   602.80704  331.915055 |  100.0 | 418.029154 | 331.915055 |
    | 2000-01-02 08:00:00 | 8.272434  9.018537  8.823355             7.663232 | 466.252304  541.948142  521.608314  408.574127 |  97.724337  105.185374  103.233552     91.632323 | 1.696972  1.784607   1.76191  1.623545 | 1.340208  1.339199  1.339442          1.341226 | -0.319112  -0.285676  -0.294235     -0.347943 | 0.430021  0.405693  0.419921      0.477599 | 0.873883  0.874618  0.879333      0.890432 | 485.365238 |  590.36904  722.195771  686.126733          494.576445 | 365.983315 | 0.091166  0.095947  0.094709       0.087156 | 0.425005  0.459655  0.450605        0.396587 | 485.365238  696.730612  750.665996    625.0015  365.983315 |  100.0 | 385.365238 | 365.983315 |
    | 2000-01-02 08:30:00 | 8.103445  8.943252  8.864275             7.835887 | 449.880747  534.057546  525.840994  424.543892 |  96.034447  104.432518  103.642751     93.358866 | 1.676781  1.775871  1.766681  1.644538 |  1.34047  1.339291   1.33939          1.340917 | -0.326965  -0.288962   -0.29243     -0.339624 | 0.436642  0.408043  0.417373       0.46864 | 0.874278  0.874522  0.878607      0.888618 |  453.86844 | 562.751483   708.14853  693.594492          520.672515 | 399.344293 | 0.090064   0.09547  0.094969       0.088303 | 0.417134  0.456165  0.452503        0.404654 |  453.86844  672.667864  746.394175  644.170556  399.344293 |  100.0 |  353.86844 | 399.344293 |
    | 2000-01-02 09:00:00 | 7.930733  8.857461  8.889563             7.990225 | 433.443657  525.134954  528.465118   439.07186 |  94.307333  103.574606  103.895632     94.902251 | 1.656007  1.765887  1.769627  1.663179 | 1.340753  1.339398  1.339358          1.340655 | -0.335104   -0.29273  -0.291318     -0.332288 | 0.443556  0.410869  0.415516      0.460661 | 0.874728  0.874492  0.877985      0.886992 | 423.717566 | 535.360001  692.347568   698.23439          544.700315 |  431.55428 | 0.088929  0.094926   0.09513       0.089321 | 0.409081  0.452187  0.453676        0.411856 | 423.717566  647.728272  739.483886  660.286121   431.55428 |  100.0 | 323.717566 |  431.55428 |
    | 2000-01-02 09:30:00 | 7.755399    8.7622  8.900129             8.126541 | 417.062036  515.313719  529.563376  452.101437 |  92.553988  102.621998  104.001286     96.265408 |  1.63477  1.754766  1.770856  1.679548 | 1.341058  1.339521  1.339345          1.340434 | -0.343486  -0.296944  -0.290853     -0.325886 | 0.450725  0.414136  0.414303      0.453616 | 0.875232  0.874523   0.87746       0.88554 | 395.049792 | 508.405017  675.058813  700.178602          566.477913 | 462.233722 | 0.087769   0.09432  0.095197       0.090215 | 0.400895  0.447767  0.454166         0.41821 | 395.049792  622.197956  730.185039  673.368423  462.233722 |  100.0 | 295.049792 | 462.233722 |
    | 2000-01-02 10:00:00 | 7.578477  8.658474  8.896891             8.245279 | 400.843683  504.722923  529.226723  463.602319 |  90.784766  101.584735  103.968911     97.452791 | 1.613185  1.742613   1.77048  1.693736 | 1.341383  1.339658  1.339349          1.340249 | -0.352071  -0.301567  -0.290995     -0.320367 | 0.458109  0.417809  0.413695      0.447456 | 0.875786  0.874612  0.877022      0.884248 | 367.963873 | 482.065892  656.538584  699.582521          585.876103 | 491.067842 | 0.086589  0.093657  0.095176        0.09099 | 0.392623  0.442952  0.454016        0.423741 | 367.963873  596.338725  718.752118  683.479171  491.067842 |  100.0 | 267.963873 | 491.067842 |
    | 2000-01-02 10:30:00 | 7.400936  8.547251  8.880771             8.347001 |  384.88335  493.486234  527.551997  473.567171 |  89.009365  100.472507  103.807707     98.470013 | 1.591364  1.729531  1.768603   1.70584 |  1.34173  1.339809  1.339369          1.340097 | -0.360815  -0.306567  -0.291704     -0.315681 | 0.465672  0.421855   0.41365      0.442131 | 0.876388  0.874756  0.876666      0.883103 | 342.523746 | 456.492164  637.030628  696.619034          602.814464 | 517.805057 | 0.085396  0.092943  0.095074        0.09165 | 0.384311  0.437786  0.453268        0.428475 | 342.523746  570.386975  705.439383  690.714868  517.805057 |  100.0 | 242.523746 | 517.805057 |
    | 2000-01-02 11:00:00 | 7.223682  8.429461  8.852679             8.432355 | 369.263099  481.721019  524.639826  482.008394 |  87.236816   99.294614  103.526791     99.323552 |  1.56941  1.715617   1.76533   1.71596 | 1.342098  1.339976  1.339404          1.339972 | -0.369681   -0.31191  -0.292941     -0.311779 | 0.473378  0.426244   0.41413      0.437591 | 0.877037  0.874951  0.876384      0.882092 | 318.762374 |  431.80523   616.76388  691.473321          617.256708 | 542.253764 | 0.084195  0.092184  0.094896       0.092203 |    0.376  0.432311  0.451965        0.432445 | 318.762374  544.553288  690.496802  695.200443  542.253764 |  100.0 | 218.762374 | 542.253764 |
    | 2000-01-02 11:30:00 | 7.047547  8.305995  8.813513              8.50205 |  354.05281  469.537683  520.592797  488.955065 |  85.475471   98.059949  103.135134    100.020504 | 1.547424  1.700966  1.760762  1.724199 | 1.342488  1.340157  1.339454          1.339873 | -0.378627  -0.317565   -0.29467     -0.308612 |  0.48119  0.430943  0.415099      0.433788 |  0.87773  0.875194  0.876173      0.881204 |  296.68568 |  408.10037  595.950855  684.338206          629.205644 | 564.277999 | 0.082991  0.091384  0.094647       0.092652 | 0.367727  0.426567  0.450148        0.435685 |  296.68568  519.022592  674.166705  697.083297  564.277999 |  100.0 |  196.68568 | 564.277999 |
    | 2000-01-02 12:00:00 | 6.873301  8.177698   8.76415             8.556839 |  339.31082  457.039218  515.513896  494.450088 |  83.733007   96.776983  102.641502    100.568393 |   1.5255  1.685669  1.754994  1.730661 | 1.342898  1.340353  1.339518          1.339797 | -0.387617  -0.323501  -0.296857     -0.306134 | 0.489076  0.435923  0.416522      0.430676 | 0.878464  0.875484  0.876025      0.880428 | 276.276442 | 385.448996  574.786611  675.410084          638.697992 | 583.792382 |  0.08179  0.090549  0.094332       0.093005 |  0.35953  0.420594  0.447858        0.438231 | 276.276442  493.954812  656.681098  696.527847  583.792382 |  100.0 | 176.276442 | 583.792382 |
    | 2000-01-02 12:30:00 | 6.701642  8.045375  8.705441             8.597501 | 325.084653  444.320923  509.505179  498.547591 |  82.016419    95.45375  102.054415    100.975006 | 1.503727  1.669812  1.748122  1.735448 | 1.343328  1.340564  1.339595          1.339741 | -0.396612  -0.329689  -0.299469     -0.304303 | 0.497001  0.441157  0.418368      0.428211 | 0.879236  0.875817  0.875938      0.879755 | 257.498082 | 363.901014  553.448205  664.885412          645.799246 | 600.756667 | 0.080597  0.089684  0.093957       0.093266 |  0.35144  0.414428  0.445133         0.44012 | 257.498082  469.485886  638.259582  693.710608  600.756667 |  100.0 | 157.498082 | 600.756667 |
    | 2000-01-02 13:00:00 | 6.533204  7.909785  8.638212             8.624827 | 311.411814   431.47028  502.666669  501.310568 |  80.332038   94.097851  101.382116    101.248266 | 1.482187  1.653478  1.740234  1.738661 | 1.343778  1.340789  1.339685          1.339703 | -0.405577  -0.336099  -0.302475     -0.303075 | 0.504934  0.446616  0.420604      0.426349 | 0.880046   0.87619  0.875907      0.879175 | 240.298231 | 343.487248  532.094567  652.957756          650.598734 | 615.170163 | 0.079416  0.088791  0.093527       0.093441 | 0.343488  0.408104  0.442011         0.44139 | 240.298231  445.729084  619.107829  688.815827  615.170163 |  100.0 | 140.298231 | 615.170163 |
    | 2000-01-02 13:30:00 | 6.368553  7.771645  8.563254             8.639613 | 298.320616  418.566963  495.095452  502.808786 |  78.685528   92.716445  100.632545    101.396133 | 1.460959  1.636744  1.731416  1.740399 | 1.344247  1.341029  1.339787          1.339683 | -0.414477  -0.342705  -0.305845     -0.302412 | 0.512843  0.452276  0.423202      0.425051 | 0.880888  0.876603  0.875928      0.878682 | 224.612057 | 324.221832  510.866723  639.815349          653.204963 |  627.06621 | 0.078251  0.087877  0.093046       0.093536 | 0.335699  0.401654  0.438529        0.442076 | 224.612057  422.776529  599.416529  682.031672   627.06621 |  100.0 | 124.612057 |  627.06621 |
    | 2000-01-02 14:00:00 | 6.208191  7.631626  8.481332             8.642652 |  285.83103  405.682945  486.884962  503.116926 |  77.081908   91.316258   99.813322    101.426518 | 1.440113  1.619686  1.721752  1.740756 | 1.344733  1.341284  1.339902          1.339679 | -0.423279  -0.349478  -0.309551     -0.302276 | 0.520701  0.458109  0.426133      0.424278 | 0.881762  0.877052  0.875997      0.878268 | 210.365288 | 306.104527  489.888294  625.639123          653.741329 | 636.506858 | 0.077107  0.086945  0.092519       0.093556 | 0.328099  0.395109  0.434722        0.442217 | 210.365288  400.700859  579.360777  673.546951  636.506858 |  100.0 | 110.365288 | 636.506858 |
    | 2000-01-02 14:30:00 | 6.052556  7.490358  8.393174             8.634722 | 273.955531  392.882706  478.124416  502.312968 |  75.525563   89.903582   98.931736    101.347222 | 1.419714  1.602375  1.711318  1.739824 | 1.345236  1.341553  1.340028           1.33969 | -0.431953  -0.356394  -0.313566     -0.302631 | 0.528479  0.464093  0.429371      0.423996 | 0.882664  0.877535  0.876112      0.877928 |  197.47692 | 289.122932  469.266228  610.601192          652.342224 | 643.577847 | 0.075986  0.085998  0.091949       0.093505 | 0.320708  0.388499  0.430623        0.441849 |  197.47692  379.556975  559.099811  663.548351  643.577847 |  100.0 |   97.47692 | 643.577847 |
    | 2000-01-02 15:00:00 | 5.902026  7.348429  8.299473             8.616588 | 262.699926  380.223502  468.898399  500.476789 |  74.020257   88.484293   97.994734    101.165882 | 1.399822  1.584878  1.700191  1.737693 | 1.345753  1.341837  1.340167          1.339714 | -0.440469  -0.363427  -0.317865     -0.303445 |  0.53615  0.470203  0.432891      0.424171 |  0.88359  0.878051   0.87627      0.877654 | 185.861607 | 273.254535  449.091688  594.863703          649.149557 | 648.383966 | 0.074893  0.085041  0.091342       0.093388 | 0.313544   0.38185  0.426264        0.441007 | 185.861607  359.383816  538.777055  652.218142  648.383966 |  100.0 |  85.861607 | 648.383966 |
    | 2000-01-02 15:30:00 | 5.756916  7.206385  8.200893             8.588994 | 252.064158  367.755691  459.286578  497.688968 |  72.569161   87.063849   97.008925    100.889937 | 1.380489  1.567258   1.68844  1.734447 | 1.346283  1.342136  1.340317          1.339752 | -0.448799  -0.370553  -0.322423     -0.304685 |  0.54369  0.476417   0.43667      0.424769 | 0.884538  0.878598  0.876467      0.877442 | 175.431719 | 258.468613  429.441067   578.57805            644.3097 | 651.044809 | 0.073829  0.084077  0.090701       0.093211 | 0.306624  0.375188  0.421674        0.439725 | 175.431719  340.206113  518.520423   639.73232  651.044809 |  100.0 |  75.431719 | 651.044809 |
    | 2000-01-02 16:00:00 | 5.617487  7.064731  8.098057             8.552659 | 242.043071  355.523097   449.36351  494.029777 |  71.174865   85.647311   95.980572    100.526591 | 1.361763  1.549576  1.676135  1.730168 | 1.346823  1.342449  1.340479          1.339802 | -0.456919  -0.377748  -0.327217     -0.306323 | 0.551076  0.482713  0.440683      0.425763 | 0.885504  0.879173  0.876703      0.877288 | 166.099094 | 244.727941  410.377081  561.884376           637.97084 | 651.690965 | 0.072799  0.083109  0.090029       0.092978 | 0.299961  0.368535  0.416883        0.438037 | 166.099094  322.036109  498.442811  626.259151  651.690965 |  100.0 |  66.099094 | 651.690965 |
    | 2000-01-02 16:30:00 | 5.483941  6.923935   7.99156              8.50828 | 232.627131  343.563405  439.198556  489.578338 |  69.839406    84.23935     94.9156      100.0828 | 1.343684  1.531889   1.66334  1.724934 | 1.347372  1.342776  1.340652          1.339864 | -0.464806   -0.38499  -0.332224     -0.308329 | 0.558286  0.489071   0.44491      0.427123 | 0.886484  0.879776  0.876974      0.877188 | 157.776495 | 231.990332  391.949899  544.911334          630.280722 | 650.460645 | 0.071803   0.08214   0.08933       0.092692 | 0.293565  0.361913  0.411919        0.435975 | 157.776495  304.875194  478.642758  611.958062  650.460645 |  100.0 |  57.776495 | 650.460645 |
    | 2000-01-02 17:00:00 | 5.356429  6.784426   7.88196             8.456525 | 223.803099  331.908569  428.855867  484.411931 |  68.564291    82.84426     93.8196     99.565248 | 1.326286  1.514249  1.650115  1.718819 | 1.347927  1.343117  1.340836          1.339937 |  -0.47244  -0.392256  -0.337424     -0.310678 | 0.565303  0.495471   0.44933      0.428824 | 0.887474  0.880404  0.877278      0.877137 | 150.378786 | 220.209985  374.198298  527.776065          621.384763 |  647.49672 | 0.070844  0.081174  0.088608       0.092359 | 0.287445  0.355343  0.406805        0.433569 | 150.378786  288.715454  459.205217  596.978846   647.49672 |  100.0 |  50.378786 |  647.49672 |
    | 2000-01-02 17:30:00 | 5.235052  6.646597  7.769784             8.398034 | 215.554658  320.585234  418.394435  478.605434 |  67.350525   81.465972   92.697835     98.980345 | 1.309599  1.496707  1.636518  1.711894 | 1.348485  1.343472  1.341032          1.340021 | -0.479803  -0.399526  -0.342794     -0.313344 | 0.572108  0.501893  0.453922      0.430839 | 0.888471  0.881055  0.877613      0.877132 | 143.823868 | 209.338663  357.150794  510.584351          611.424492 | 642.944166 | 0.069924  0.080212  0.087865       0.091981 | 0.281607  0.348843  0.401567        0.430849 | 143.823868  273.541114  440.202417  581.461135  642.944166 |  100.0 |  43.823868 | 642.944166 |
    | 2000-01-02 18:00:00 | 5.119864  6.510807  7.655525             8.333421 | 207.862978  309.615145  407.868197   472.23089 |  66.198637    80.10807   91.555251     98.334215 | 1.293643   1.47931  1.622605  1.704227 | 1.349044  1.343841  1.341239          1.340116 |  -0.48688  -0.406779  -0.348317     -0.316304 | 0.578686  0.508319  0.458669      0.433146 | 0.889471  0.881728  0.877978       0.87717 | 138.033385 | 199.326703  340.826755  493.430914          600.536311 | 636.947894 | 0.069043  0.079258  0.087104       0.091562 | 0.276055  0.342429  0.396227        0.427843 | 138.033385  259.329855   421.69477  565.534115  636.947894 |  100.0 |  38.033385 | 636.947894 |
    | 2000-01-02 18:30:00 | 5.010871   6.37738  7.539648              8.26327 | 200.707237  299.015568  397.326186  465.357164 |  65.108715   78.773799   90.396481     97.632696 | 1.278436  1.462102  1.608427   1.69588 | 1.349601  1.344221  1.341458          1.340221 | -0.493659  -0.413996  -0.353972     -0.319535 | 0.585024  0.514731  0.463552      0.435722 | 0.890468  0.882421  0.878371      0.877248 | 132.933235 | 190.123869  325.237462  476.399831          588.850518 | 629.650923 | 0.068203  0.078314  0.086329       0.091107 |  0.27079  0.336117  0.390807        0.424578 | 132.933235  246.054029  403.731814  549.316427  629.650923 |  100.0 |  32.933235 | 629.650923 |
    | 2000-01-02 19:00:00 | 4.908043  6.246608  7.422586             8.188134 | 194.065089  288.799687  386.812712  458.049703 |  64.080432   77.466081   89.225861      96.88134 | 1.263988  1.445123  1.594033  1.686916 | 1.350153  1.344614  1.341687          1.340337 |  -0.50013  -0.421158  -0.359742     -0.323016 |  0.59111  0.521113  0.468554      0.438546 | 0.891459  0.883131   0.87879      0.877364 | 128.453911 | 181.680059  310.387121  459.565033          576.490586 | 621.192879 | 0.067405  0.077382  0.085542       0.090618 | 0.265813  0.329921  0.385325         0.42108 | 128.453911  233.681736  386.353159  532.916244  621.192879 |  100.0 |  28.453911 | 621.192879 |
    | 2000-01-02 19:30:00 | 4.811309  6.118753  7.304744             8.108542 | 187.913079  278.976988  376.367567  450.370369 |  63.113086    76.18753   88.047438     96.085417 | 1.250303  1.428411  1.579471  1.677392 | 1.350698  1.345018  1.341927          1.340462 | -0.506286  -0.428248   -0.36561     -0.326727 | 0.596937  0.527449  0.473658      0.441598 | 0.892441  0.883858  0.879233      0.877515 | 124.530701 | 173.945887  296.273804  442.990866          563.572639 | 611.708784 | 0.066648  0.076464  0.084745       0.090097 | 0.261121  0.323853  0.379802        0.417372 | 124.530701  222.177798  369.589423  516.431475  611.708784 |  100.0 |  24.530701 | 611.708784 |
    | 2000-01-02 20:00:00 | 4.720563  5.994046  7.186498             8.024992 | 182.227013  269.553633  366.026256  442.377333 |  62.205629   74.940461   86.864982     95.249917 | 1.237382  1.412001  1.564783  1.667362 | 1.351233  1.345433  1.342179          1.340597 | -0.512123  -0.435248  -0.371558     -0.330648 | 0.602496  0.533723   0.47885      0.444859 | 0.893408    0.8846  0.879699      0.877699 | 121.103761 | 166.873145  282.890332  426.732699          550.205134 |   601.3281 | 0.065934  0.075562  0.083941        0.08955 |  0.25671  0.317925  0.374254        0.413478 | 121.103761   211.50461  353.463144  499.950082    601.3281 |  100.0 |  21.103761 |   601.3281 |
    | 2000-01-02 20:30:00 | 4.635671   5.87269    7.0682             7.937956 | 176.982286  260.532805  355.820232  434.125041 |  61.356709   73.726902   85.681997     94.379557 | 1.225219  1.395926  1.550011  1.656879 | 1.351755  1.345858  1.342441          1.340741 |  -0.51764  -0.442143  -0.377571     -0.334761 | 0.607784  0.539921  0.484113      0.448312 | 0.894357  0.885354  0.880187      0.877914 | 118.118088 | 160.415166  270.225075  410.837555          536.488681 | 590.174011 |  0.06526  0.074679  0.083133       0.088977 | 0.252576  0.312146  0.368698        0.409418 | 118.118088  201.622886  337.989656   483.55048  590.174011 |  100.0 |  18.118088 | 590.174011 |
    | 2000-01-02 21:00:00 |  4.55647  5.754861  6.950173             7.847879 | 172.154159  251.915041  345.777147   425.66421 |  60.564702    72.54861   84.501733      93.47879 | 1.213803  1.380214  1.535194   1.64599 | 1.352262  1.346291  1.342714          1.340895 | -0.522836  -0.448918  -0.383634      -0.33905 | 0.612797   0.54603  0.489434       0.45194 | 0.895284  0.886119  0.880695      0.878158 | 115.523427 | 154.527091  258.262698  395.344753           522.51601 | 578.362897 | 0.064628  0.073814  0.082321       0.088382 | 0.248712  0.306526  0.363148        0.405214 | 115.523427   192.49231  323.177925  467.302002  578.362897 |  100.0 |  15.523427 | 578.362897 |
    | 2000-01-02 21:30:00 | 4.482775  5.640708   6.83272             7.755181 | 167.718009  243.698537  335.921094  417.041873 |  59.827754   71.407078   83.327198      92.55181 |  1.20312  1.364892  1.520369  1.634743 | 1.352753  1.346731  1.342997          1.341058 | -0.527714  -0.455559  -0.389731     -0.343497 | 0.617534  0.552037  0.494798      0.455726 | 0.896185  0.886893  0.881221      0.878429 | 113.274094 | 149.166072  246.984817  380.286551           508.37204 | 566.003986 | 0.064036  0.072971  0.081509       0.087768 | 0.245109  0.301072  0.357619        0.400885 | 113.274094  184.072085  309.031326  451.265411  566.003986 |  100.0 |  13.274094 | 566.003986 |
    | 2000-01-02 22:00:00 | 4.414381  5.530355  6.716117             7.660256 |  163.64953  235.879434  326.272863   408.30146 |  59.143813   70.303547   82.161166     91.602561 | 1.193152  1.349983   1.50557  1.623182 | 1.353225  1.347178  1.343291          1.341231 | -0.532281  -0.462053  -0.395848     -0.348087 | 0.621997   0.55793  0.500193      0.459656 | 0.897058  0.887673  0.881765      0.878725 | 111.328777 | 144.291395  236.370603  365.688769          494.134047 | 553.199158 | 0.063483   0.07215  0.080698       0.087136 |  0.24176   0.29579  0.352123        0.396448 | 111.328777  176.321409  295.548376  435.493434  553.199158 |  100.0 |  11.328777 | 553.199158 |
    | 2000-01-02 22:30:00 | 4.351067  5.423902  6.600619             7.563475 | 159.924917  228.452086   316.85018  399.482893 |  58.510669   69.239019   81.006195      90.63475 | 1.183877  1.335509   1.49083  1.611348 | 1.353677   1.34763  1.343595          1.341412 | -0.536542  -0.468388  -0.401972     -0.352804 | 0.626189  0.563699  0.505607      0.463714 |   0.8979  0.888459  0.882326      0.879046 | 109.650285 | 139.864564  226.397314  351.571411          479.871893 | 540.042868 | 0.062969  0.071352   0.07989       0.086489 | 0.238655  0.290685  0.346672        0.391921 | 109.650285  169.199867  282.723409  420.031329  540.042868 |  100.0 |   9.650285 | 540.042868 |
    | 2000-01-02 23:00:00 | 4.292599  5.321427  6.486463             7.465185 | 156.521003  221.409301  307.667949  390.622713 |  57.925988   68.214266   79.864629      89.65185 | 1.175272  1.321487  1.476179   1.59928 | 1.354108  1.348085  1.343909          1.341602 | -0.540507  -0.474554  -0.408089     -0.357636 | 0.630114  0.569332  0.511026      0.467887 | 0.898708  0.889247  0.882901      0.879389 | 108.205287 | 135.849332  217.040767  337.949242          465.648323 | 526.622182 | 0.062491  0.070579  0.079086       0.085829 | 0.235783  0.285763  0.341278        0.387321 | 108.205287  162.667758  270.547191   404.91744  526.622182 |  100.0 |   8.205287 | 526.622182 |
    | 2000-01-02 23:30:00 | 4.238735  5.222984  6.373862             7.365712 |  153.41538  214.742569   298.73848  381.754216 |  57.387347    67.22984   78.738616     88.657117 | 1.167308  1.307932  1.461646  1.587014 | 1.354517  1.348543  1.344232          1.341802 | -0.544185  -0.480541  -0.414188     -0.362567 | 0.633779  0.574822   0.51644      0.472162 |  0.89948  0.890036  0.883489      0.879754 | 106.964036 | 132.211695  208.275748  324.832351          451.519296 | 513.016893 | 0.062049  0.069832  0.078289       0.085158 | 0.233133  0.281026   0.33595         0.38266 | 106.964036  156.686363  259.007482  390.183759  513.016893 |  100.0 |   6.964036 | 513.016893 |
    | 2000-01-03 00:00:00 | 4.189227  5.128609  6.263011             7.265359 | 150.586493  208.442264  290.071708  372.907602 |  56.892265   66.286087   77.630111     87.653591 | 1.159958  1.294859  1.447259  1.574587 | 1.354904  1.349001  1.344564           1.34201 | -0.547587   -0.48634  -0.420255     -0.367584 | 0.637189   0.58016  0.521837      0.476527 | 0.900215  0.890823   0.88409      0.880138 | 105.900084 | 128.919864  200.076377  312.226675          437.534351 | 499.299712 | 0.061641   0.06911  0.077499       0.084478 | 0.230694  0.276477  0.330698        0.377955 | 105.900084  151.218161  248.089545  375.856477  499.299712 |  100.0 |   5.900084 | 499.299712 |
    | 2000-01-03 00:30:00 | 4.143823  5.038315  6.154089             7.164412 | 148.013711  202.497836  281.675409  364.110134 |  56.438234   65.383153   76.540892     86.644115 | 1.153191  1.282275  1.433041  1.562031 | 1.355267  1.349459  1.344905          1.342227 | -0.550726  -0.491945   -0.42628     -0.372676 | 0.640355  0.585338  0.527208      0.480969 | 0.900911  0.891607  0.884702      0.880541 |  104.99001 | 125.944213  192.416419  300.134488          423.736988 | 485.536517 | 0.061264  0.068415  0.076718       0.083791 | 0.228454  0.272117  0.325531        0.373217 |  104.99001   146.22699  237.776601  361.956514  485.536517 |  100.0 |    4.99001 | 485.536517 |
    | 2000-01-03 01:00:00 | 4.102274    4.9521  6.047256             7.063134 | 145.677378  196.897978   273.55539  355.386298 |  56.022741   64.521001   75.472563     85.631338 | 1.146976  1.270191  1.419016  1.549376 | 1.355608  1.349914  1.345254          1.342453 | -0.553614  -0.497349  -0.432251      -0.37783 | 0.643285  0.590351  0.532542      0.485477 | 0.901568  0.892386  0.885325      0.880961 | 104.213141 | 123.257205   185.26955  288.554864          410.165069 | 471.786633 | 0.060919  0.067748  0.075948       0.083098 | 0.226402  0.267946  0.320455         0.36846 | 104.213141  141.678184  228.050232  348.500027  471.786633 |  100.0 |   4.213141 | 471.786633 |
    | 2000-01-03 01:30:00 |  4.06433  4.869942  5.942657             6.961773 |  143.55885  191.630782  265.715684   346.75797 |  55.643301   63.699416   74.426566     84.617725 | 1.141281  1.258609  1.405207  1.536654 | 1.355925  1.350365  1.345611          1.342687 | -0.556266  -0.502547  -0.438158     -0.383035 |  0.64599  0.595192  0.537829      0.490041 | 0.902186  0.893157  0.885956      0.881398 | 103.551298 | 120.833311   178.60958  277.484089          396.851211 | 458.103159 | 0.060602  0.067108  0.075189       0.082401 | 0.224525  0.263966  0.315479        0.363694 | 103.551298  137.538654  218.890743  335.498901  458.103159 |  100.0 |   3.551298 | 458.103159 |
    | 2000-01-03 02:00:00 | 4.029747  4.791802  5.840418             6.860557 | 141.640513  186.683883  258.158717  338.244574 |  55.297471   62.918023   73.404185     83.605571 | 1.136074  1.247533  1.391633   1.52389 | 1.356221  1.350811  1.345974          1.342929 | -0.558693  -0.507536  -0.443991      -0.38828 |  0.64848  0.599857  0.543062       0.49465 | 0.902765  0.893919  0.886594      0.881849 | 102.988541 | 118.648921   172.41065  266.916054          383.823187 | 444.533304 | 0.060313  0.066495  0.074442       0.081702 | 0.222813  0.260173  0.310608         0.35893 | 102.988541   133.77696  210.277481  322.961202  444.533304 |  100.0 |   2.988541 | 444.533304 |
    | 2000-01-03 02:30:00 | 3.998287  4.717629  5.740656               6.7597 | 139.905791  182.044579  250.885477   329.86325 |  54.982868   62.176294   72.406557     82.597004 | 1.131324  1.236963  1.378313  1.511111 | 1.356494  1.351251  1.346345           1.34318 | -0.560911  -0.512313   -0.44974     -0.393554 | 0.650767  0.604343  0.548231      0.499295 | 0.903305  0.894669  0.887239      0.882315 |  102.51094 | 116.682248  166.647382  256.842607          371.104312 | 431.118745 | 0.060048   0.06591   0.07371       0.081002 | 0.221255  0.256567  0.305848        0.354178 |  102.51094   130.36334  202.189107  310.891607  431.118745 |  100.0 |    2.51094 | 431.118745 |
    | 2000-01-03 03:00:00 | 3.969719  4.647356  5.643468               6.6594 | 138.339137  177.699949  243.895663  321.629011 |  54.697191   61.473563   71.434678     81.593996 |    1.127  1.226897  1.365264  1.498342 | 1.356747  1.351682  1.346721          1.343439 | -0.562933  -0.516877  -0.455398     -0.398847 | 0.652862  0.608648  0.553328      0.503965 | 0.903807  0.895406  0.887889      0.882794 | 102.106358 | 114.913226  161.295013  247.253868          358.713826 | 417.895987 | 0.059808  0.065353  0.072991       0.080302 | 0.219838  0.253145  0.301203        0.349447 | 102.106358  127.269725   194.60384  299.291807  417.895987 |  100.0 |   2.106358 | 417.895987 |
    | 2000-01-03 03:30:00 | 3.943823  4.580904  5.548941             6.559837 | 136.926026  173.636956  237.187825  313.554896 |  54.438227   60.809038   70.489407     80.598374 |  1.12307  1.217332  1.352501  1.485605 |  1.35698  1.352104  1.347102          1.343706 | -0.564772  -0.521228  -0.460954      -0.40415 | 0.654777  0.612769  0.558345      0.508652 | 0.904272  0.896127  0.888543      0.883286 | 101.764251 | 113.323411  156.329491  238.138524          346.667254 | 404.896726 | 0.059589  0.064823  0.072289       0.079603 | 0.218552  0.249904   0.29668        0.344746 | 101.764251  124.469736  187.499666  288.160872  404.896726 |  100.0 |   1.764251 | 404.896726 |
    | 2000-01-03 04:00:00 | 3.920386  4.518181  5.457147             6.461182 | 135.652922  169.842532  230.759494  305.652117 |   54.20386   60.181813   69.571473     79.611823 | 1.119506   1.20826  1.340039  1.472923 | 1.357193  1.352515  1.347487           1.34398 | -0.566442  -0.525365  -0.466402     -0.409453 | 0.656524  0.616705  0.563276      0.513348 | 0.904702  0.896832  0.889199      0.883788 |  101.47548 | 111.895882  151.727562  229.484085          334.976754 | 392.148208 | 0.059391  0.064321  0.071602       0.078908 | 0.217388  0.246841   0.29228        0.340082 |  101.47548  121.938659   180.85452  277.495594  392.148208 |  100.0 |    1.47548 | 392.148208 |
    | 2000-01-03 04:30:00 | 3.899208  4.459088  5.368148              6.36359 | 134.507258  166.303665  224.607301  297.930207 |  53.992084   59.590883   68.681482     78.635896 |  1.11628  1.199673  1.327891  1.460316 | 1.357388  1.352915  1.347875          1.344262 | -0.567956  -0.529292  -0.471735     -0.414747 | 0.658114  0.620457  0.568115      0.518044 | 0.905097  0.897518  0.889857      0.884301 | 101.232151 | 110.615142  147.466825  221.277119          323.651449 | 379.673581 | 0.059211  0.063845  0.070932       0.078216 | 0.216335   0.24395  0.288008        0.335464 | 101.232151  119.653415  174.646433  267.290795  379.673581 |  100.0 |   1.232151 | 379.673581 |
    | 2000-01-03 05:00:00 |   3.8801  4.403515  5.281992             6.267202 | 133.477395  163.007467  218.727085  290.397149 |  53.801003   59.035153   67.819921     77.672022 | 1.113363  1.191563  1.316067  1.447804 | 1.357566  1.353302  1.348266          1.344551 | -0.569325   -0.53301  -0.476945     -0.420025 | 0.659558  0.624027  0.572854      0.522732 | 0.905461  0.898185  0.890516      0.884823 | 101.027458 |  109.46702  143.525777  213.503454          312.697738 | 367.492232 | 0.059048  0.063395   0.07028       0.077529 | 0.215385  0.241228  0.283866        0.330897 | 101.027458  117.592512  168.853669  257.539611  367.492232 |  100.0 |   1.027458 | 367.492232 |
    | 2000-01-03 05:30:00 | 3.862883  4.351346  5.198716             6.172151 | 132.552588  159.941233  213.113999   283.05951 |  53.628833   58.513457   66.987163     76.721511 |  1.11073  1.183918  1.304578  1.435404 | 1.357729  1.353676  1.348659          1.344847 | -0.570562  -0.536524  -0.482026     -0.425277 | 0.660867  0.627414  0.577488      0.527405 | 0.905794   0.89883  0.891173      0.885353 | 100.855551 | 108.438582  139.883841  206.148367          302.119582 | 355.620121 | 0.058902  0.062971  0.069647       0.076848 | 0.214528  0.238669  0.279857        0.326388 | 100.855551  115.735993   163.45483  248.233751  355.620121 |  100.0 |   0.855551 | 355.620121 |
    | 2000-01-03 06:00:00 | 3.847391  4.302456  5.118347             6.078556 | 131.722936  157.092503  207.762594  275.922558 |  53.473907   58.024564   66.183471      75.78556 | 1.108358  1.176725  1.293432  1.423133 | 1.357876  1.354035  1.349052           1.34515 | -0.571677  -0.539837  -0.486974     -0.430495 | 0.662052  0.630622  0.582012      0.532056 | 0.906098  0.899453  0.891828      0.885891 | 100.711411 |  107.51804  136.521382  199.196742          291.918782 | 344.070084 |  0.05877  0.062572  0.069032       0.076174 | 0.213756  0.236267  0.275982        0.321943 | 100.711411  114.065373  158.428951  239.363723  344.070084 |  100.0 |   0.711411 | 344.070084 |
    | 2000-01-03 06:30:00 | 3.833467   4.25672  5.040901             5.986526 | 130.979347  154.449099  202.666908   268.99038 |  53.334669   57.567195   65.409007     74.865263 | 1.106223  1.169971  1.282637  1.411008 | 1.358009   1.35438  1.349446          1.345459 | -0.572681  -0.542954  -0.491784     -0.435673 | 0.663123  0.633654  0.586421      0.536678 | 0.906375  0.900053   0.89248      0.886436 | 100.590745 | 106.694665  133.419714  192.633212          282.095224 | 332.852139 | 0.058651  0.062197  0.068435       0.075508 | 0.213063  0.234018  0.272242        0.317567 | 100.590745  112.563577  153.755573  230.919043  332.852139 |  100.0 |   0.590745 | 332.852139 |
    | 2000-01-03 07:00:00 | 3.820968  4.214003  4.966383             5.896161 | 130.313486  151.999173  197.820546  262.265986 |  53.209677   57.140034    64.66383      73.96161 | 1.104304   1.16364  1.272197  1.399043 |  1.35813  1.354709  1.349837          1.345774 | -0.573585  -0.545882   -0.49645     -0.440803 | 0.664089  0.636515  0.590712      0.541265 | 0.906627  0.900628  0.893128      0.886986 | 100.489885 | 105.958707  130.561096  186.442284          272.647114 | 321.973758 | 0.058544  0.061845  0.067859        0.07485 |  0.21244  0.231915  0.268638        0.313264 | 100.489885  111.214868  149.414807   222.88842  321.973758 |  100.0 |   0.489885 | 321.973758 |
    | 2000-01-03 07:30:00 |  3.80976  4.174174  4.894791              5.80755 |  129.71773  149.731233  193.216743  255.751408 |  53.097595   56.741736   63.947907     73.075496 | 1.102582  1.157717  1.262119  1.387253 | 1.358239  1.355024  1.350227          1.346095 | -0.574396  -0.548626   -0.50097     -0.445879 | 0.664959  0.639207   0.59488       0.54581 | 0.906855   0.90118  0.893769      0.887542 | 100.405708 | 105.301319  127.928719  180.608447          263.571188 | 311.440134 | 0.058448  0.061516  0.067302       0.074201 | 0.211881  0.229951   0.26517         0.30904 | 100.405708  110.004779   145.38738  215.259922  311.440134 |  100.0 |   0.405708 | 311.440134 |
    | 2000-01-03 08:00:00 | 3.799719  4.137095  4.826112             5.720773 | 129.185124  147.634173  188.848434  249.447798 |  52.997193   56.370945   63.261115     72.207728 | 1.101037  1.152186  1.252403  1.375649 | 1.358338  1.355322  1.350614           1.34642 | -0.575124  -0.551193   -0.50534     -0.450893 | 0.665741  0.641736  0.598922      0.550308 | 0.907062  0.901706  0.894404      0.888102 |  100.33556 | 104.714485  125.506694   175.11626          254.862913 | 301.254422 | 0.058362  0.061209  0.066764       0.073563 |  0.21138  0.228122  0.261839        0.304898 |  100.33556  108.920034   141.65468  208.021117  301.254422 |  100.0 |    0.33556 | 301.254422 |
    | 2000-01-03 08:30:00 | 3.790734   4.10263  4.760324             5.635902 | 128.709334  145.697297  184.708314  243.355505 |  52.907341   56.026297   62.603245     71.359023 | 1.099654  1.147029  1.243054  1.364245 | 1.358426  1.355605  1.350996          1.346751 | -0.575776   -0.55359  -0.509559      -0.45584 | 0.666444  0.644107  0.602835      0.554752 | 0.907249  0.902207   0.89503      0.888665 | 100.277186 | 104.190948  123.280027  169.950437          246.516656 | 291.417967 | 0.058284  0.060922  0.066247       0.072935 | 0.210932  0.226419  0.258643        0.300842 | 100.277186  107.948482  138.198777  201.159204  291.417967 |  100.0 |   0.277186 | 291.417967 |
    | 2000-01-03 09:00:00 |   3.7827  4.070643  4.697401             5.553002 | 128.284602   143.91033  180.788885  237.474163 |  52.827001   55.706432   61.974008     70.530017 | 1.098416   1.14223  1.234071  1.353051 | 1.358505  1.355872  1.351374          1.347085 | -0.576359  -0.555824  -0.513623     -0.460714 | 0.667075  0.646326  0.606618      0.559138 | 0.907417  0.902682  0.895647      0.889231 | 100.228681 | 103.724149  121.234594  165.095912          238.525852 | 281.930514 | 0.058215  0.060655   0.06575       0.072319 | 0.210531  0.224838  0.255583        0.296874 | 100.228681  107.079021  135.002451  194.661128  281.930514 |  100.0 |   0.228681 | 281.930514 |
    | 2000-01-03 09:30:00 | 3.775523  4.041001  4.637304             5.472127 | 127.905704  142.263433  177.082511  231.802758 |  52.755228   55.410005   61.373042     69.721268 | 1.097309   1.13777  1.225453  1.342077 | 1.358577  1.356124  1.351745          1.347423 | -0.576881  -0.557902  -0.517533      -0.46551 |  0.66764  0.648399  0.610268      0.563461 | 0.907569  0.903133  0.896254      0.889798 | 100.188431 | 103.308168  119.357116  160.537893          230.883147 | 272.790401 | 0.058154  0.060407  0.065273       0.071714 | 0.210173  0.223371  0.252655        0.292999 | 100.188431  106.301525  132.049202  188.513672  272.790401 |  100.0 |   0.188431 | 272.790401 |
    | 2000-01-03 10:00:00 | 3.769116  4.013569  4.579991             5.393326 | 127.567908  140.747209  173.581462    226.3397 |  52.691158   55.135689   60.799912     68.933258 |  1.09632  1.133633    1.2172  1.331334 | 1.358641  1.356361   1.35211          1.347764 | -0.577348  -0.559833  -0.521288     -0.470221 | 0.668147  0.650331  0.613784      0.567716 | 0.907706  0.903559   0.89685      0.890367 | 100.155078 | 102.937665  117.635123  156.261914          223.580529 | 263.994736 | 0.058099  0.060177  0.064816       0.071122 | 0.209853  0.222012   0.24986        0.289217 | 100.155078  105.606779  129.323261  182.703548  263.994736 |  100.0 |   0.155078 | 263.994736 |
    | 2000-01-03 10:30:00 | 3.763401  3.988218  4.525412              5.31664 | 127.266935  139.352708  170.277951   221.08288 |  52.634007   54.882184    60.25412     68.166396 | 1.095438  1.129801  1.209308   1.32083 | 1.358698  1.356583  1.352468          1.348107 | -0.577764  -0.561623  -0.524887     -0.474844 | 0.668599   0.65213  0.617166      0.571899 | 0.907829   0.90396  0.897433      0.890935 | 100.127477 | 102.607832   116.05693  152.253868          216.609446 | 255.539557 | 0.058049  0.059964  0.064379       0.070543 | 0.209567  0.220755  0.247194        0.285533 | 100.127477  104.986411  126.809588  177.217468  255.539557 |  100.0 |   0.127477 | 255.539557 |
    | 2000-01-03 11:00:00 | 3.758306  3.964823  4.473511             5.242103 | 126.998921  138.071427  167.164176  216.029732 |  52.583062   54.648225   59.735111     67.421025 | 1.094651  1.126258  1.201773  1.310571 | 1.358749  1.356791  1.352817          1.348453 | -0.578136   -0.56328  -0.528331     -0.479373 | 0.669004  0.653801  0.620413      0.576005 | 0.907939  0.904337  0.898003      0.891503 | 100.104666 | 102.314343  114.611597  148.500043          209.960915 | 247.419981 | 0.058005  0.059766  0.063961       0.069977 | 0.209313  0.219595  0.244656        0.281946 | 100.104666  104.432824  124.493873  172.042211  247.419981 |  100.0 |   0.104666 | 247.419981 |
    | 2000-01-03 11:30:00 | 3.753768  3.943258  4.424227             5.169742 | 126.760382  136.895307  164.232348  211.177281 |  52.537678   54.432583   59.242273     66.697418 |  1.09395  1.122985   1.19459  1.300566 | 1.358795  1.356985  1.353157          1.348799 | -0.578467  -0.564812  -0.531622     -0.483805 | 0.669365  0.655352  0.623526       0.58003 | 0.908037   0.90469  0.898559       0.89207 | 100.085838 | 102.053308  113.288898  144.987142          203.625613 | 239.630339 | 0.057966  0.059584  0.063563       0.069425 | 0.209086  0.218524  0.242243         0.27846 | 100.085838   103.93914  122.362528  167.164674  239.630339 |  100.0 |   0.085838 | 239.630339 |
    | 2000-01-03 12:00:00 | 3.749727  3.923407  4.377495             5.099579 | 126.548182  135.816728  161.474728  206.522196 |  52.497273   54.234072   58.774948     65.995787 | 1.093325  1.119966  1.187754  1.290821 | 1.358835  1.357166  1.353487          1.349147 | -0.578762  -0.566227   -0.53476     -0.488136 | 0.669687  0.656788  0.626505      0.583971 | 0.908126  0.905022    0.8991      0.892634 | 100.070317 | 101.821236  112.079289  141.702306          197.593959 | 232.164289 | 0.057932  0.059416  0.063184       0.068887 | 0.208884  0.217538  0.239952        0.275076 | 100.070317  103.499137  120.402677  162.571924  232.164289 |  100.0 |   0.070317 | 232.164289 |
    | 2000-01-03 12:30:00 | 3.746132  3.905155  4.333243             5.031628 |   126.3595  134.828499  158.883648  202.060826 |  52.461319   54.051549   58.332435     65.316281 | 1.092769  1.117186  1.181258  1.281341 | 1.358872  1.357333  1.353808          1.349494 | -0.579024   -0.56753  -0.537748     -0.492362 | 0.669974  0.658115  0.629351      0.587826 | 0.908205  0.905331  0.899625      0.893195 | 100.057539 | 101.614996  110.973872  138.633125          191.856195 | 225.014933 | 0.057901  0.059261  0.062823       0.068364 | 0.208704  0.216631   0.23778        0.271793 | 100.057539  103.107199  118.602143   158.25123  225.014933 |  100.0 |   0.057539 | 225.014933 |
    | 2000-01-03 13:00:00 | 3.742934  3.888392  4.291399             4.965899 | 126.191801  133.923851   156.45154  197.789247 |  52.429343   53.883922   57.913995      64.65899 | 1.092274  1.114629  1.175095  1.272129 | 1.358904  1.357489  1.354118           1.34984 | -0.579258  -0.568731  -0.540589      -0.49648 |  0.67023  0.659342  0.632066      0.591589 | 0.908275  0.905619  0.900134      0.893752 | 100.047032 |  101.43178  109.964362  135.767644          186.402447 | 218.174909 | 0.057873  0.059119  0.062481       0.067855 | 0.208545  0.215797  0.235724        0.268614 | 100.047032  102.758258  116.949431  154.190108  218.174909 |  100.0 |   0.047032 | 218.174909 |
    | 2000-01-03 13:30:00 | 3.740092  3.873015  4.251886             4.902395 | 126.042813  133.096426  154.170958  193.703298 |  52.400919   53.730145   57.518859     64.023948 | 1.091834   1.11228  1.169256  1.263191 | 1.358933  1.357633  1.354417          1.350185 | -0.579466  -0.569834  -0.543285     -0.500488 | 0.670458  0.660472  0.634651       0.59526 | 0.908338  0.905887  0.900626      0.894304 | 100.038402 | 101.269076  109.043055  133.094377          181.222781 | 211.636477 | 0.057848  0.058988  0.062157       0.067361 | 0.208403  0.215032   0.23378        0.265539 | 100.038402  102.447753  115.433709  150.376338  211.636477 |  100.0 |   0.038402 | 211.636477 |
    | 2000-01-03 14:00:00 | 3.737566  3.858923  4.214623             4.841114 | 125.910498   132.34026  152.034593   189.79861 |  52.375662   53.589227   57.146232     63.411136 | 1.091443  1.110124  1.163732   1.25453 | 1.358959  1.357766  1.354705          1.350528 |  -0.57965  -0.570847  -0.545839     -0.504383 | 0.670661  0.661513  0.637109      0.598836 | 0.908394  0.906135  0.901101      0.894851 | 100.031322 | 101.124634  108.202797  130.602298           176.30726 | 205.391599 | 0.057827  0.058868   0.06185       0.066882 | 0.208276  0.214331  0.231945        0.262567 | 100.031322  102.171578  114.044792  146.797995  205.391599 |  100.0 |   0.031322 | 205.391599 |
    | 2000-01-03 14:30:00 | 3.735323  3.846022   4.17953             4.782048 |  125.79303  131.649773  150.035298  186.070641 |  52.353229   53.460223   56.795299     62.820481 | 1.091096  1.108148  1.158515  1.246146 | 1.358981  1.357889  1.354981          1.350869 | -0.579815  -0.571776  -0.548256     -0.508162 | 0.670841   0.66247  0.639442      0.602314 | 0.908444  0.906366  0.901559      0.895392 | 100.025521 | 100.996444  107.436948  128.280849          171.645982 | 199.432007 | 0.057807  0.058758   0.06156       0.066418 | 0.208164  0.213688  0.230215        0.259699 | 100.025521  101.926047  112.773115  143.443466  199.432007 |  100.0 |   0.025521 | 199.432007 |
    | 2000-01-03 15:00:00 | 3.733331  3.834224  4.146523             4.725186 | 125.688779  131.019754  148.166095  182.514705 |  52.333312   53.342245   56.465227      62.25186 | 1.090788  1.106339  1.153594  1.238042 | 1.359002  1.358002  1.355246          1.351206 |  -0.57996  -0.572627  -0.550539     -0.511825 | 0.671001   0.66335  0.641653      0.605693 | 0.908489  0.906579  0.901998      0.895926 | 100.020773 | 100.882712  106.739358   126.11993          167.229121 |  193.74926 |  0.05779  0.058657  0.061287        0.06597 | 0.208065    0.2131  0.228587        0.256935 | 100.020773  101.707849  111.609714  140.301464   193.74926 |  100.0 |   0.020773 |  193.74926 |
    | 2000-01-03 15:30:00 | 3.731564  3.823445  4.115518              4.67051 | 125.596286  130.445341  146.420194  179.125995 |  52.315636   53.234451   56.155177     61.705104 | 1.090514  1.104685  1.148959  1.230219 |  1.35902  1.358106  1.355499          1.351539 |  -0.58009  -0.573406  -0.552692     -0.515369 | 0.671144  0.664156  0.643746       0.60897 | 0.908528  0.906776   0.90242      0.896453 | 100.016892 | 100.781833  106.104333    124.1099           163.04696 | 188.334803 | 0.057775  0.058565  0.061029       0.065537 | 0.207976  0.212563  0.227056        0.254273 | 100.016892  101.514017  110.546201  137.361045  188.334803 |  100.0 |   0.016892 | 188.334803 |
    | 2000-01-03 16:00:00 | 3.729995  3.813605   4.08643                4.618 | 125.514248  129.922009  144.791001   175.89961 |  52.299952   53.136054     55.8643     61.179998 | 1.090271  1.103173    1.1446  1.222677 | 1.359036  1.358202   1.35574          1.351867 | -0.580204  -0.574117   -0.55472     -0.518795 |  0.67127  0.664895  0.645723      0.612146 | 0.908564  0.906958  0.902823      0.896971 | 100.013722 | 100.692378  105.526609  122.241561           159.08992 | 183.180008 | 0.057761  0.058481  0.060787       0.065119 | 0.207898  0.212073  0.225618        0.251714 | 100.013722  101.341891   109.57474  134.611611  183.180008 |  100.0 |   0.013722 | 183.180008 |
    | 2000-01-03 16:30:00 | 3.728604  3.804631  4.059175             4.567628 | 125.441501  129.445555   143.27213  172.830575 |   52.28604   53.046311   55.591749     60.676284 | 1.090055  1.101793  1.140506  1.215415 |  1.35905  1.358289   1.35597           1.35219 | -0.580306  -0.574767  -0.556627     -0.522101 | 0.671382  0.665571  0.647588      0.615219 | 0.908595  0.907125  0.903208       0.89748 | 100.011136 | 100.613072  105.001323  120.506158          155.348584 | 178.276218 | 0.057749  0.058404  0.060559       0.064717 | 0.207828  0.211625   0.22427        0.249256 | 100.011136  101.189093   108.68802  132.042924  178.276218 |  100.0 |   0.011136 | 178.276218 |
    | 2000-01-03 17:00:00 |  3.72737  3.796453  4.033668             4.519366 | 125.377011  129.012076  141.857408  169.913863 |  52.273705   52.964531    55.33668     60.193665 | 1.089864  1.100534  1.136665  1.208431 | 1.359063   1.35837  1.356187          1.352508 | -0.580397   -0.57536  -0.558418     -0.525287 | 0.671482  0.666189  0.649345      0.618188 | 0.908623  0.907279  0.903575      0.897979 | 100.009028 | 100.542779  104.523989  118.895364          151.813716 | 173.614785 | 0.057738  0.058334  0.060346        0.06433 | 0.207767  0.211217  0.223008        0.246898 | 100.009028  101.053496  107.879232  129.645107  173.614785 |  100.0 |   0.009028 | 173.614785 |
    | 2000-01-03 17:30:00 | 3.726277  3.789007  4.009825              4.47318 | 125.319851  128.617958  140.540881  167.144413 |  52.262769   52.890067   55.098254     59.731803 | 1.089695  1.099388  1.133068  1.201725 | 1.359074  1.358443  1.356394          1.352819 | -0.580477  -0.575901  -0.560097     -0.528353 |  0.67157  0.666754  0.650998      0.621052 | 0.908648   0.90742  0.903925      0.898469 | 100.007313 | 100.480488  104.090471  117.401271          148.476279 | 169.187095 | 0.057729   0.05827  0.060145       0.063958 | 0.207712  0.210846  0.221826         0.24464 | 100.007313  100.933198  107.142046  127.408648  169.187095 |  100.0 |   0.007313 | 169.187095 |
    | 2000-01-03 18:00:00 | 3.725308  3.782232  3.987564             4.429033 | 125.269201  128.259855   139.31682   164.51715 |  52.253077   52.822316   54.875645     59.290328 | 1.089545  1.098343  1.129703  1.195292 | 1.359084   1.35851  1.356589          1.353123 | -0.580548  -0.576393  -0.561669       -0.5313 | 0.671649  0.667269   0.65255      0.623813 |  0.90867  0.907549  0.904256      0.898947 | 100.005918 | 100.425298  103.696962  116.016377          145.327452 |   164.9846 | 0.057721  0.058211  0.059958       0.063602 | 0.207663  0.210507  0.220723        0.242478 | 100.005918  100.826502  106.470579  125.324401    164.9846 |  100.0 |   0.005918 |   164.9846 |
    | 2000-01-03 18:30:00 | 3.724449  3.776072  3.966804             4.386884 | 125.224327  127.934679  138.179722  162.026997 |  52.244488    52.76072   54.668039     58.868837 | 1.089412  1.097394  1.126558   1.18913 | 1.359093  1.358571  1.356773           1.35342 | -0.580611  -0.576841   -0.56314     -0.534128 | 0.671718  0.667739  0.654006       0.62647 | 0.908689  0.907668  0.904571      0.899414 | 100.004785 | 100.376408   103.33996  114.733573          142.358637 |  160.99884 | 0.057713  0.058158  0.059783        0.06326 |  0.20762    0.2102  0.219693        0.240412 | 100.004785  100.731896   105.85938  123.383586   160.99884 |  100.0 |   0.004785 |  160.99884 |
    | 2000-01-03 19:00:00 | 3.723688  3.770476  3.947464              4.34669 | 125.184578  127.639579  137.124313  159.668893 |  52.236879   52.704759   54.474639     58.466895 | 1.089294   1.09653  1.123623  1.183235 |   1.3591  1.358627  1.356947           1.35371 | -0.580666  -0.577249  -0.564513     -0.536839 |  0.67178  0.668167   0.65537      0.629023 | 0.908706  0.907776  0.904868       0.89987 | 100.003865 | 100.333107  103.016244  113.546135          139.561472 | 157.221461 | 0.057707   0.05811   0.05962       0.062933 | 0.207582  0.209921  0.218733         0.23844 | 100.003865  100.648029  105.303395  121.577787  157.221461 |  100.0 |   0.003865 | 157.221461 |
    | 2000-01-03 19:30:00 | 3.723014  3.765395  3.929467             4.308404 | 125.149374  127.371928  136.145547  157.437808 |   52.23014   52.653951   54.294668     58.084044 | 1.089189  1.095746  1.120888  1.177602 | 1.359107  1.358678   1.35711          1.353991 | -0.580716  -0.577619  -0.565795     -0.539433 | 0.671835  0.668556  0.656646      0.631474 | 0.908722  0.907875  0.905149      0.900313 | 100.003119 | 100.294761   102.72286  112.447702          136.927835 | 153.644235 | 0.057701  0.058067  0.059467        0.06262 | 0.207549  0.209667  0.217839         0.23656 | 100.003119  100.573698  104.797955   119.89895  153.644235 |  100.0 |   0.003119 | 153.644235 |
    | 2000-01-03 20:00:00 | 3.722417  3.760785  3.912737              4.27198 | 125.118201  127.129309  135.238608  155.328752 |  52.224171   52.607853    54.12737     57.719798 | 1.089097  1.095034  1.118342  1.172227 | 1.359113  1.358724  1.357264          1.354264 |  -0.58076  -0.577955  -0.566989     -0.541912 | 0.671883  0.668911  0.657839      0.633822 | 0.908735  0.907965  0.905413      0.900743 | 100.002515 | 100.260809  102.457095  111.432274          134.449853 | 150.259074 | 0.057696  0.058027  0.059326       0.062322 | 0.207519  0.209437  0.217008        0.234769 | 100.002515  100.507833  104.338742  118.339378  150.259074 |  100.0 |   0.002515 | 150.259074 |
    | 2000-01-03 20:30:00 | 3.721889  3.756605  3.897201             4.237365 |   125.0906  126.909498  134.398904   153.33679 |  52.218885   52.566053   53.972012     57.373651 | 1.089015  1.094388  1.115973  1.167105 | 1.359119  1.358766  1.357407          1.354529 | -0.580798   -0.57826    -0.5681     -0.544279 | 0.671926  0.669232  0.658951       0.63607 | 0.908747  0.908048  0.905662      0.901161 | 100.002026 |  100.23075  102.216465  110.494186          132.119903 | 147.058042 | 0.057691  0.057991  0.059194       0.062038 | 0.207492  0.209228  0.216235        0.233065 | 100.002026  100.449481  103.921774  116.891726  147.058042 |  100.0 |   0.002026 | 147.058042 |
    | 2000-01-03 21:00:00 | 3.721421  3.752817  3.882789             4.204508 | 125.066166  126.710457  133.622067  151.457052 |  52.214206   52.528175   53.827886     57.045079 | 1.088942  1.093803  1.113773   1.16223 | 1.359124  1.358804  1.357541          1.354784 | -0.580833  -0.578536  -0.569132     -0.546535 | 0.671964  0.669524  0.659988      0.638219 | 0.908758  0.908123  0.905896      0.901565 | 100.001631 | 100.204141  101.998693  109.628105          129.930615 | 144.033364 | 0.057687  0.057958  0.059071       0.061767 | 0.207469  0.209039  0.215519        0.231447 | 100.001631  100.397793  103.543381  115.548994  144.033364 |  100.0 |   0.001631 | 144.033364 |
    | 2000-01-03 21:30:00 | 3.721006  3.749387  3.869431             4.173354 | 125.044538  126.530313  132.903947  149.684741 |  52.210064   52.493869    53.69431     56.733542 | 1.088878  1.093272  1.111732  1.157595 | 1.359128  1.358839  1.357667           1.35503 | -0.580863  -0.578787  -0.570091     -0.548683 | 0.671997  0.669789  0.660952      0.640271 | 0.908767  0.908191  0.906116      0.901956 | 100.001312 |  100.18059  101.801698  108.829011          127.874874 |  141.17744 | 0.057683  0.057929  0.058958       0.061509 | 0.207448  0.208867  0.214854        0.229911 | 100.001312  100.352017  103.200185  114.304524   141.17744 |  100.0 |   0.001312 |  141.17744 |
    | 2000-01-03 22:00:00 |  3.72064  3.746282  3.857063             4.143849 | 125.025395  126.367352  132.240612  148.015148 |  52.206397   52.462816   53.570629     56.438488 | 1.088821  1.092792  1.109839  1.153195 | 1.359132   1.35887  1.357784          1.355268 |  -0.58089  -0.579013  -0.570981     -0.550725 | 0.672027   0.67003   0.66185      0.642226 | 0.908776  0.908254  0.906322      0.902333 | 100.001054 | 100.159746  101.623575  108.092184           125.94582 | 138.482845 |  0.05768  0.057902  0.058852       0.061265 |  0.20743  0.208712  0.214238        0.228455 | 100.001054  100.311481  102.889079  113.151987  138.482845 |  100.0 |   0.001054 | 138.482845 |
    | 2000-01-03 22:30:00 | 3.720315  3.743472  3.845621             4.115935 | 125.008454  126.220005  131.628338  146.443652 |  52.203152   52.434722   53.456214     56.159354 | 1.088771  1.092358  1.108087  1.149022 | 1.359135  1.358899  1.357893          1.355495 | -0.580914  -0.579219  -0.571805     -0.552663 | 0.672054  0.670247  0.662683      0.644089 | 0.908783   0.90831  0.906514      0.902696 | 100.000846 | 100.141301  101.462585  107.413191          124.136845 | 135.942344 | 0.057677  0.057878  0.058754       0.061033 | 0.207414  0.208572  0.213668        0.227077 | 100.000846  100.275592  102.607209  112.085378  135.942344 |  100.0 |   0.000846 | 135.942344 |
    | 2000-01-03 23:00:00 | 3.720028  3.740932  3.835047             4.089557 | 124.993463  126.086835  131.063606  144.965733 |   52.20028   52.409319   53.350465     55.895569 | 1.088726  1.091964  1.106465  1.145069 | 1.359138  1.358924  1.357994          1.355714 | -0.580935  -0.579404  -0.572567     -0.554502 | 0.672077  0.670445  0.663456      0.645859 |  0.90879  0.908361  0.906693      0.903045 | 100.000679 | 100.124979  101.317142  106.787873          122.441593 | 133.548889 | 0.057675  0.057856  0.058664       0.060813 | 0.207399  0.208445  0.213141        0.225773 | 100.000679  100.243821  102.351957  111.099008  133.548889 |  100.0 |   0.000679 | 133.548889 |
    | 2000-01-03 23:30:00 | 3.719774  3.738636  3.825281             4.064656 | 124.980199  125.966532  130.543095  143.576977 |  52.197739   52.386359   53.252811     55.646559 | 1.088687  1.091609  1.104967   1.14133 | 1.359141  1.358948  1.358089          1.355923 | -0.580953  -0.579572  -0.573273     -0.556243 | 0.672097  0.670623  0.664172      0.647541 | 0.908796  0.908408  0.906861      0.903381 | 100.000544 | 100.110538  101.185795  106.212333          120.853958 |  131.29563 | 0.057673  0.057836  0.058581       0.060605 | 0.207387   0.20833  0.212655        0.224541 | 100.000544  100.215698  102.120922  110.187493   131.29563 |  100.0 |   0.000544 |  131.29563 |
    | 2000-01-04 00:00:00 | 3.719549  3.736562  3.816271             4.041175 | 124.968463  125.857898  130.063677  142.273084 |  52.195491   52.365618   53.162708     55.411747 | 1.088652  1.091288  1.103583  1.137796 | 1.359143  1.358969  1.358176          1.356123 |  -0.58097  -0.579724  -0.573924      -0.55789 | 0.672116  0.670785  0.664835      0.649136 | 0.908801   0.90845  0.907017      0.903702 | 100.000436 | 100.097762  101.067227   105.68292          119.368079 | 129.175913 | 0.057671  0.057818  0.058503       0.060408 | 0.207375  0.208226  0.212206        0.223379 | 100.000436  100.190807  101.911905  109.345745  129.175913 |  100.0 |   0.000436 | 129.175913 |
    | 2000-01-04 00:30:00 |  3.71935  3.734689  3.807964             4.019055 | 124.958081  125.759843  129.622407  141.049867 |  52.193502    52.34689    53.07964     55.190555 | 1.088621  1.090998  1.102306  1.134461 | 1.359145  1.358988  1.358257          1.356314 | -0.580984  -0.579861  -0.574526     -0.559446 | 0.672132  0.670931  0.665448      0.650647 | 0.908805  0.908488  0.907162       0.90401 | 100.000349 | 100.086459  100.960234  105.196221          117.978338 | 127.183285 | 0.057669  0.057802  0.058432       0.060223 | 0.207365  0.208133  0.211791        0.222284 | 100.000349  100.168779  101.722896   108.56896  127.183285 |  100.0 |   0.000349 | 127.183285 |
    | 2000-01-04 01:00:00 | 3.719174  3.732999  3.800312             3.998241 | 124.948896  125.671371   129.21652  139.903263 |  52.191742   52.329986   53.003117     54.982409 | 1.088594  1.090736  1.101128  1.131317 | 1.359147  1.359005  1.358332          1.356495 | -0.580997  -0.579985  -0.575081     -0.560915 | 0.672146  0.671063  0.666015      0.652076 | 0.908809  0.908523  0.907297      0.904304 | 100.000279 | 100.076461  100.863722  104.749046          116.679354 | 125.311496 | 0.057668  0.057787  0.058367       0.060048 | 0.207357  0.208048   0.21141        0.221252 | 100.000279  100.149287  101.552054  107.852613  125.311496 |  100.0 |   0.000279 | 125.311496 |
    | 2000-01-04 01:30:00 | 3.719019  3.731474  3.793268             3.978674 | 124.940771  125.591576  128.843421  138.829333 |  52.190185   52.314735   52.932678     54.786738 |  1.08857    1.0905  1.100044  1.128357 | 1.359148  1.359021  1.358401          1.356668 | -0.581009  -0.580096  -0.575592     -0.562299 | 0.672159  0.671182  0.666538      0.653427 | 0.908813  0.908554  0.907422      0.904585 | 100.000223 | 100.067616  100.776696  104.338416          115.465981 |   123.5545 | 0.057666  0.057774  0.058306       0.059883 | 0.207349  0.207972  0.211058        0.220282 | 100.000223  100.132039  101.397701  107.192441    123.5545 |  100.0 |   0.000223 |   123.5545 |
    | 2000-01-04 02:00:00 | 3.718881  3.730098  3.786789             3.960298 | 124.933585  125.519634  128.500679  137.824266 |  52.188808   52.300982   52.867888     54.602979 | 1.088548  1.090287  1.099046  1.125571 |  1.35915  1.359035  1.358465          1.356832 | -0.581019  -0.580197  -0.576062     -0.563602 |  0.67217  0.671289  0.667021      0.654702 | 0.908816  0.908583  0.907538      0.904852 | 100.000178 | 100.059793   100.69825  103.961554            114.3333 | 121.906452 | 0.057665  0.057762  0.058251       0.059728 | 0.207342  0.207903  0.210735         0.21937 | 100.000178   100.11678  101.258303  106.584438  121.906452 |  100.0 |   0.000178 | 121.906452 |
    | 2000-01-04 02:30:00 | 3.718759  3.728858  3.780834             3.943058 | 124.927229  125.454796  128.186019   136.88438 |   52.18759   52.288583   52.808336     54.430576 | 1.088529  1.090095  1.098128  1.122954 | 1.359151  1.359047  1.358524          1.356987 | -0.581028  -0.580288  -0.576495     -0.564827 |  0.67218  0.671386  0.667465      0.655904 | 0.908819  0.908608  0.907645      0.905106 | 100.000142 | 100.052874  100.627563   103.61587          113.276615 | 120.361715 | 0.057664  0.057751  0.058199       0.059583 | 0.207336  0.207841  0.210438        0.218514 | 100.000142   100.10328  101.132463  106.024842  120.361715 |  100.0 |   0.000142 | 120.361715 |
    | 2000-01-04 03:00:00 | 3.718651  3.727741  3.775364             3.926898 | 124.921607   125.39638  127.897313  136.006126 |  52.186513    52.27741   52.753638     54.268983 | 1.088513  1.089922  1.097284  1.120497 | 1.359152  1.359059  1.358578          1.357134 | -0.581036   -0.58037  -0.576893     -0.565978 | 0.672189  0.671474  0.667874      0.657035 | 0.908821  0.908631  0.907744      0.905347 | 100.000113 | 100.046754  100.563889  103.298954          112.291449 | 118.914853 | 0.057663  0.057742  0.058152       0.059446 |  0.20733  0.207785  0.210165        0.217712 | 100.000113  100.091337  101.018905  105.510123  118.914853 |  100.0 |   0.000113 | 118.914853 |
    | 2000-01-04 03:30:00 | 3.718556  3.726735  3.770343             3.911767 | 124.916635  125.343768  127.632579  135.186086 |   52.18556   52.267345    52.70343     54.117666 | 1.088498  1.089766   1.09651  1.118194 | 1.359153  1.359069  1.358628          1.357273 | -0.581043  -0.580443  -0.577258     -0.567058 | 0.672196  0.671553  0.668251        0.6581 | 0.908824  0.908652  0.907836      0.905576 |  100.00009 | 100.041342   100.50655  103.008566          111.373536 | 117.560634 | 0.057662  0.057733  0.058109       0.059317 | 0.207326  0.207735  0.209914         0.21696 |  100.00009  100.080773  100.916471  105.036974  117.560634 |  100.0 |    0.00009 | 117.560634 |
    | 2000-01-04 04:00:00 | 3.718472  3.725828  3.765738              3.89761 | 124.912237  125.296399  127.389963  134.420975 |  52.184718   52.258281   52.657376     53.976101 | 1.088485  1.089625  1.095799  1.116036 | 1.359154  1.359078  1.358674          1.357403 | -0.581049   -0.58051  -0.577594      -0.56807 | 0.672203  0.671624  0.668597        0.6591 | 0.908826  0.908671   0.90792      0.905793 | 100.000072 | 100.036556  100.454932   102.74262          110.518814 | 116.294026 | 0.057662  0.057725  0.058069       0.059197 | 0.207321  0.207689  0.209684        0.216256 | 100.000072  100.071428  100.824104  104.602302  116.294026 |  100.0 |   0.000072 | 116.294026 |
    | 2000-01-04 04:30:00 | 3.718397  3.725012  3.761516             3.884378 | 124.908349  125.253763  127.167744  133.707642 |  52.183972   52.250122   52.615158     53.843782 | 1.088473  1.089499  1.095147  1.114016 | 1.359155  1.359087  1.358717          1.357527 | -0.581054  -0.580569  -0.577902     -0.569018 | 0.672209  0.671688  0.668915      0.660039 | 0.908827  0.908688  0.907998      0.905998 | 100.000057 | 100.032323  100.408478  102.499184          109.723425 | 115.110196 | 0.057661  0.057718  0.058033       0.059085 | 0.207318  0.207649  0.209473        0.215598 | 100.000057  100.063163  100.740845  104.203215  115.110196 |  100.0 |   0.000057 | 115.110196 |
    | 2000-01-04 05:00:00 | 3.718331  3.724278  3.757648             3.872021 |  124.90491  125.215401  126.964318  133.043067 |  52.183313    52.24278   52.576481     53.720213 | 1.088463  1.089385  1.094549  1.112128 | 1.359155  1.359094  1.358756          1.357643 | -0.581059  -0.580623  -0.578184     -0.569905 | 0.672215  0.671746  0.669207      0.660919 | 0.908829  0.908703   0.90807      0.906192 | 100.000046 |  100.02858  100.366683  102.276462          108.983701 | 114.004509 |  0.05766  0.057712     0.058        0.05898 | 0.207314  0.207612   0.20928        0.214983 | 100.000046  100.055853  100.665822  103.837011  114.004509 |  100.0 |   0.000046 | 114.004509 |
    | 2000-01-04 05:30:00 | 3.718273  3.723617  3.754107             3.860492 | 124.901869  125.180892  126.778195   132.42436 |  52.182731   52.236174   52.541069     53.604918 | 1.088454  1.089283  1.094002  1.110364 | 1.359156  1.359101  1.358791          1.357752 | -0.581063  -0.580672  -0.578442     -0.570734 | 0.672219  0.671798  0.669475      0.661743 |  0.90883  0.908717  0.908136      0.906374 | 100.000036 |  100.02527  100.329092  102.072791          108.296161 | 112.972522 |  0.05766  0.057706  0.057969       0.058881 | 0.207312  0.207579  0.209103        0.214409 | 100.000036  100.049388  100.598243  103.501174  112.972522 |  100.0 |   0.000036 | 112.972522 |
    | 2000-01-04 06:00:00 | 3.718222  3.723023  3.750866             3.849743 |  124.89918   125.14986  126.607992  131.848762 |  52.182215   52.230233   52.508664     53.497432 | 1.088446  1.089191  1.093501  1.108718 | 1.359157  1.359107  1.358824          1.357854 | -0.581067  -0.580715  -0.578678     -0.571508 | 0.672224  0.671845  0.669721      0.662514 | 0.908831   0.90873  0.908197      0.906546 | 100.000029 | 100.022343   100.29529   101.88663          107.657508 | 112.009987 | 0.057659  0.057701  0.057941        0.05879 | 0.207309  0.207549  0.208941        0.213874 | 100.000029   100.04367  100.537389  103.193355  112.009987 |  100.0 |   0.000029 | 112.009987 |
    | 2000-01-04 06:30:00 | 3.718176  3.722489  3.747903             3.839731 | 124.896802  125.121962  126.452426  131.313641 |   52.18176   52.224891   52.479029     53.397311 | 1.088439  1.089108  1.093043  1.107184 | 1.359157  1.359113  1.358854          1.357949 | -0.581071  -0.580754  -0.578895     -0.572229 | 0.672227  0.671887  0.669945      0.663235 | 0.908832  0.908741  0.908253      0.906708 | 100.000023 | 100.019755  100.264905  101.716553          107.064617 | 111.112838 | 0.057659  0.057696  0.057916       0.058704 | 0.207307  0.207522  0.208793        0.213375 | 100.000023  100.038614  100.482609  102.911372  111.112838 |  100.0 |   0.000023 | 111.112838 |
    | 2000-01-04 07:00:00 | 3.718136  3.722009  3.745194             3.830413 | 124.894699  125.096888  126.310309   130.81649 |  52.181357    52.22009   52.451942     53.304126 | 1.088433  1.089033  1.092624  1.105754 | 1.359157  1.359118  1.358881          1.358039 | -0.581074   -0.58079  -0.579093     -0.572902 | 0.672231  0.671925  0.670151      0.663908 | 0.908833  0.908751  0.908304       0.90686 | 100.000018 | 100.017466  100.237597  101.561242          106.514532 |   110.2772 | 0.057659  0.057692  0.057892       0.058624 | 0.207305  0.207498  0.208658        0.212911 | 100.000018  100.034143   100.43331  102.653193    110.2772 |  100.0 |   0.000018 |   110.2772 |
    | 2000-01-04 07:30:00 |   3.7181  3.721577   3.74272             3.821746 |  124.89284  125.074358  126.180541  130.354925 |     52.181   52.215775   52.427195     53.217464 | 1.088427  1.088966  1.092241  1.104424 | 1.359158  1.359122  1.358906          1.358123 | -0.581076  -0.580821  -0.579274     -0.573528 | 0.672234  0.671959   0.67034      0.664536 | 0.908834   0.90876  0.908351      0.907002 | 100.000014 | 100.015443  100.213062  101.419479          106.004458 | 109.499371 | 0.057658  0.057688  0.057871        0.05855 | 0.207303  0.207477  0.208534        0.212479 | 100.000014  100.030188  100.388958  102.416932  109.499371 |  100.0 |   0.000014 | 109.499371 |
    | 2000-01-04 08:00:00 | 3.718069   3.72119   3.74046             3.813693 | 124.891196  125.054119  126.062105   129.92668 |  52.180685   52.211899     52.4046     53.136932 | 1.088422  1.088906  1.091891  1.103186 | 1.359158  1.359126  1.358929          1.358201 | -0.581078   -0.58085  -0.579439     -0.574111 | 0.672236  0.671989  0.670512      0.665121 | 0.908835  0.908768  0.908395      0.907136 | 100.000011 | 100.013653  100.191023  101.290137          105.531756 | 108.775831 | 0.057658  0.057685  0.057852       0.058481 | 0.207301  0.207457  0.208421        0.212077 | 100.000011  100.026692  100.349068  102.200839  108.775831 |  100.0 |   0.000011 | 108.775831 |
    | 2000-01-04 08:30:00 | 3.718041  3.720842  3.738398             3.806215 | 124.889743  125.035943   125.95406  129.529607 |  52.180407   52.208417   52.383979     53.062153 | 1.088418  1.088852  1.091572  1.102037 | 1.359158   1.35913   1.35895          1.358274 |  -0.58108  -0.580875   -0.57959     -0.574653 | 0.672238  0.672017  0.670669      0.665666 | 0.908835  0.908776  0.908434      0.907261 | 100.000009 | 100.012071  100.171231  101.172181          105.093937 | 108.103227 | 0.057658  0.057682  0.057834       0.058417 |   0.2073   0.20744  0.208318        0.211704 | 100.000009    100.0236  100.313201   102.00329  108.103227 |  100.0 |   0.000009 | 108.103227 |
    | 2000-01-04 09:00:00 | 3.718016  3.720529  3.736517             3.799277 | 124.888458  125.019623   125.85554  129.161671 |  52.180161   52.205291   52.365168     52.992768 | 1.088414  1.088804  1.091281  1.100969 | 1.359159  1.359133  1.358969          1.358342 | -0.581082  -0.580898  -0.579727     -0.575156 |  0.67224  0.672041  0.670813      0.666173 | 0.908836  0.908782   0.90847      0.907378 | 100.000007 | 100.010673  100.153461  101.064653          104.688652 | 107.478376 | 0.057658  0.057679  0.057818       0.058358 | 0.207299  0.207424  0.208224        0.211358 | 100.000007  100.020866  100.280959  101.822781  107.478376 |  100.0 |   0.000007 | 107.478376 |
    | 2000-01-04 09:30:00 | 3.717994  3.720249  3.734802             3.792843 | 124.887321  125.004973  125.765744  128.820947 |  52.179943   52.202485   52.348017     52.928432 | 1.088411   1.08876  1.091015  1.099978 | 1.359159  1.359136  1.358987          1.358405 | -0.581084  -0.580919  -0.579853     -0.575622 | 0.672242  0.672064  0.670944      0.666645 | 0.908837  0.908788  0.908503      0.907487 | 100.000006 | 100.009436  100.137511  100.966671          104.313692 | 106.898254 | 0.057657  0.057677  0.057803       0.058303 | 0.207298   0.20741  0.208138        0.211037 | 100.000006  100.018449  100.251985  101.657923  106.898254 |  100.0 |   0.000006 | 106.898254 |
    | 2000-01-04 10:00:00 | 3.717975  3.719997  3.733239             3.786882 | 124.886317  124.991826  125.683934  128.505615 |   52.17975   52.199967   52.332387     52.868822 | 1.088408  1.088721  1.090773   1.09906 | 1.359159  1.359138  1.359003          1.358464 | -0.581085  -0.580937  -0.579967     -0.576055 | 0.672244  0.672084  0.671064      0.667082 | 0.908837  0.908794  0.908534      0.907589 | 100.000004 | 100.008342  100.123196  100.877425          103.966977 | 106.359998 | 0.057657  0.057675  0.057789       0.058251 | 0.207297  0.207398   0.20806         0.21074 | 100.000004  100.016311  100.225954  101.507427  106.359998 |  100.0 |   0.000004 | 106.359998 |
    | 2000-01-04 10:30:00 | 3.717958  3.719771  3.731815             3.781363 | 124.885429  124.980029  125.609431  128.213959 |   52.17958   52.197707   52.318148     52.813627 | 1.088405  1.088686  1.090553  1.098209 | 1.359159  1.359141  1.359017          1.358519 | -0.581087  -0.580954  -0.580071     -0.576457 | 0.672245  0.672101  0.671173      0.667489 | 0.908837  0.908799  0.908561      0.907684 | 100.000004 | 100.007376  100.110353  100.796167          103.646551 | 105.860892 | 0.057657  0.057673  0.057777       0.058204 | 0.207296  0.207386  0.207989        0.210464 | 100.000004  100.014421  100.202572  101.370104  105.860892 |  100.0 |   0.000004 | 105.860892 |
    | 2000-01-04 11:00:00 | 3.717943  3.719568  3.730518             3.776255 | 124.884643  124.969446   125.54161  127.944359 |   52.17943   52.195679   52.305183     52.762555 | 1.088403  1.088655  1.090352  1.097422 | 1.359159  1.359143   1.35903          1.358569 | -0.581088  -0.580968  -0.580166     -0.576828 | 0.672246  0.672117  0.671273      0.667866 | 0.908838  0.908803  0.908587      0.907773 | 100.000003 | 100.006521  100.098832  100.722211          103.350579 | 105.398371 | 0.057657  0.057671  0.057766        0.05816 | 0.207295  0.207376  0.207924        0.210209 | 100.000003   100.01275  100.181576  101.244859  105.398371 |  100.0 |   0.000003 | 105.398371 |
    | 2000-01-04 11:30:00 |  3.71793  3.719386  3.729338             3.771533 | 124.883949  124.959955  125.479897  127.695291 |  52.179296   52.193861   52.293383     52.715328 | 1.088401  1.088627  1.090169  1.096693 |  1.35916  1.359145  1.359043          1.358616 | -0.581089  -0.580982  -0.580253     -0.577172 | 0.672247  0.672132  0.671364      0.668215 | 0.908838  0.908807   0.90861      0.907855 | 100.000002 | 100.005765    100.0885  100.654926          103.077336 | 104.970009 | 0.057657  0.057669  0.057756       0.058119 | 0.207294  0.207367  0.207865        0.209973 | 100.000002  100.011273  100.162725  101.130677  104.970009 |  100.0 |   0.000002 | 104.970009 |
    | 2000-01-04 12:00:00 | 3.717918  3.719223  3.728265             3.767169 | 124.883335  124.951444  125.423762  127.465323 |  52.179179    52.19223   52.282648     52.671686 | 1.088399  1.088601  1.090003   1.09602 |  1.35916  1.359146  1.359053           1.35866 | -0.581089  -0.580994  -0.580331      -0.57749 | 0.672248  0.672145  0.671446      0.668538 | 0.908838   0.90881  0.908631      0.907932 | 100.000002 | 100.005097  100.079235  100.593734          102.825207 | 104.573516 | 0.057657  0.057668  0.057746       0.058082 | 0.207294  0.207359  0.207811        0.209755 | 100.000002  100.009966  100.145806  101.026625  104.573516 |  100.0 |   0.000002 | 104.573516 |
    | 2000-01-04 12:30:00 | 3.717907  3.719077  3.727288             3.763138 | 124.882792  124.943814  125.372721  127.253107 |  52.179075   52.190768   52.272884      52.63138 | 1.088397  1.088579  1.089852  1.095397 |  1.35916  1.359148  1.359063            1.3587 |  -0.58109  -0.581004  -0.580403     -0.577783 | 0.672249  0.672156  0.671522      0.668838 | 0.908839  0.908813   0.90865      0.908003 | 100.000001 | 100.004506  100.070929  100.538103          102.592679 | 104.206735 | 0.057657  0.057667  0.057738       0.058047 | 0.207293  0.207352  0.207762        0.209554 | 100.000001  100.008811  100.130622  100.931845  104.206735 |  100.0 |   0.000001 | 104.206735 |
    | 2000-01-04 13:00:00 | 3.717898  3.718946  3.726401             3.759418 | 124.882312  124.936975  125.326329   127.05738 |  52.178983   52.189458   52.264008     52.594178 | 1.088396  1.088558  1.089714  1.094823 |  1.35916  1.359149  1.359073          1.358738 | -0.581091  -0.581014  -0.580468     -0.578055 |  0.67225  0.672167   0.67159      0.669115 | 0.908839  0.908816  0.908668      0.908069 | 100.000001 | 100.003984  100.063484  100.487546          102.378335 | 103.867632 | 0.057657  0.057666   0.05773       0.058015 | 0.207293  0.207345  0.207718        0.209368 | 100.000001   100.00779     100.117  100.845546  103.867632 |  100.0 |   0.000001 | 103.867632 |
    | 2000-01-04 13:30:00 |  3.71789  3.718828  3.725594             3.755986 | 124.881888  124.930846  125.284177  126.876956 |  52.178901   52.188283   52.255943     52.559862 | 1.088395   1.08854  1.089589  1.094293 |  1.35916   1.35915  1.359081          1.358772 | -0.581092  -0.581023  -0.580527     -0.578305 | 0.672251  0.672176  0.671652       0.66937 | 0.908839  0.908819  0.908684       0.90813 | 100.000001 | 100.003522  100.056811  100.441615           102.18085 | 103.554294 | 0.057657  0.057665  0.057723       0.057985 | 0.207292  0.207339  0.207678        0.209197 | 100.000001  100.006887   100.10478  100.766998  103.554294 |  100.0 |   0.000001 | 103.554294 |
    | 2000-01-04 14:00:00 | 3.717883  3.718723  3.724862             3.752823 | 124.881513  124.925354  125.245892  126.710727 |   52.17883   52.187231   52.248616     52.528226 | 1.088394  1.088524  1.089476  1.093804 |  1.35916  1.359151  1.359088          1.358804 | -0.581092   -0.58103   -0.58058     -0.578536 | 0.672251  0.672184  0.671709      0.669607 | 0.908839  0.908821  0.908698      0.908187 | 100.000001 | 100.003114  100.050833  100.399901          101.998986 | 103.264924 | 0.057656  0.057664  0.057717       0.057958 | 0.207292  0.207334  0.207641        0.209039 | 100.000001  100.006089  100.093822  100.695534  103.264924 |  100.0 |   0.000001 | 103.264924 |
    | 2000-01-04 14:30:00 | 3.717877  3.718629  3.724196             3.749908 | 124.881181  124.920434   125.21113  126.557653 |  52.178766   52.186288   52.241962     52.499077 | 1.088393  1.088509  1.089372  1.093353 |  1.35916  1.359152  1.359095          1.358834 | -0.581093  -0.581037  -0.580629     -0.578748 | 0.672252  0.672192   0.67176      0.669825 | 0.908839  0.908823  0.908711      0.908239 | 100.000001 | 100.002753  100.045478  100.362031          101.831587 | 102.997832 | 0.057656  0.057663  0.057711       0.057933 | 0.207292  0.207329  0.207608        0.208893 | 100.000001  100.005383  100.083996   100.63054  102.997832 |  100.0 |   0.000001 | 102.997832 |
    | 2000-01-04 15:00:00 | 3.717871  3.718544  3.723592             3.747223 | 124.880888  124.916027  125.179579  126.416765 |   52.17871   52.185444   52.235922     52.472234 | 1.088392  1.088496  1.089279  1.092938 |  1.35916  1.359153  1.359101          1.358861 | -0.581093  -0.581044  -0.580673     -0.578945 | 0.672252  0.672199  0.671807      0.670026 | 0.908839  0.908825  0.908723      0.908288 |      100.0 | 100.002434  100.040681  100.327661          101.677575 | 102.751434 | 0.057656  0.057662  0.057706        0.05791 | 0.207291  0.207325  0.207578        0.208759 |      100.0  100.004759  100.075188  100.571452  102.751434 |  100.0 |        0.0 | 102.751434 |
    | 2000-01-04 15:30:00 | 3.717866  3.718469  3.723044             3.744753 | 124.880629   124.91208   125.15095  126.287156 |   52.17866   52.184687   52.230441     52.447527 | 1.088391  1.088484  1.089194  1.092556 |  1.35916  1.359154  1.359107          1.358886 | -0.581093  -0.581049  -0.580714     -0.579125 | 0.672253  0.672205  0.671849      0.670211 | 0.908839  0.908826  0.908734      0.908333 |      100.0 | 100.002152  100.036385  100.296477          101.535945 | 102.524246 | 0.057656  0.057662  0.057701       0.057889 | 0.207291  0.207321   0.20755        0.208636 |      100.0  100.004208  100.067293  100.517751  102.524246 |  100.0 |        0.0 | 102.524246 |
    | 2000-01-04 16:00:00 | 3.717862  3.718401  3.722547              3.74248 |   124.8804  124.908546  125.124982  126.167981 |  52.178616    52.18401   52.225469       52.4248 |  1.08839  1.088474  1.089117  1.092204 |  1.35916  1.359155  1.359112          1.358909 | -0.581094  -0.581054   -0.58075     -0.579291 | 0.672253   0.67221  0.671888      0.670382 |  0.90884  0.908828  0.908744      0.908374 |      100.0 | 100.001902  100.032538  100.268194          101.405759 | 102.314879 | 0.057656  0.057661  0.057697       0.057869 | 0.207291  0.207318  0.207525        0.208522 |      100.0   100.00372  100.060218  100.468965  102.314879 |  100.0 |        0.0 | 102.314879 |
    | 2000-01-04 16:30:00 | 3.717858   3.71834  3.722096              3.74039 | 124.880197  124.905383  125.101434  126.058451 |  52.178577   52.183404    52.22096     52.403903 |  1.08839  1.088464  1.089047  1.091881 |  1.35916  1.359155  1.359117           1.35893 | -0.581094  -0.581059  -0.580783     -0.579444 | 0.672253  0.672215  0.671923      0.670539 |  0.90884  0.908829  0.908753      0.908413 |      100.0 | 100.001682  100.029094  100.242548          101.286147 |  102.12203 | 0.057656   0.05766  0.057693       0.057851 | 0.207291  0.207315  0.207503        0.208418 |      100.0  100.003289  100.053879  100.424659   102.12203 |  100.0 |        0.0 |  102.12203 |
    | 2000-01-04 17:00:00 | 3.717854  3.718286  3.721687              3.73847 | 124.880018  124.902551  125.080088  125.957833 |  52.178543   52.182861   52.216872     52.384699 | 1.088389  1.088456  1.088983  1.091583 |  1.35916  1.359156  1.359121          1.358949 | -0.581094  -0.581062  -0.580813     -0.579584 | 0.672254  0.672219  0.671955      0.670684 |  0.90884   0.90883  0.908762      0.908448 |      100.0 | 100.001487  100.026012  100.219302          101.176298 | 101.944485 | 0.057656   0.05766  0.057689       0.057834 | 0.207291  0.207312  0.207482        0.208322 |      100.0  100.002908  100.048201  100.384435  101.944485 |  100.0 |        0.0 | 101.944485 |
    | 2000-01-04 17:30:00 | 3.717851  3.718238  3.721317             3.736706 |  124.87986  124.900016  125.060743  125.865442 |  52.178513   52.182376   52.213168     52.367059 | 1.088389  1.088449  1.088926   1.09131 |  1.35916  1.359156  1.359125          1.358967 | -0.581094  -0.581066   -0.58084     -0.579713 | 0.672254  0.672223  0.671983      0.670817 |  0.90884  0.908831  0.908769      0.908481 |      100.0 | 100.001314  100.023253  100.198236          101.075458 |  101.78111 | 0.057656   0.05766  0.057686       0.057819 |  0.20729   0.20731  0.207464        0.208233 |      100.0  100.002571  100.043115  100.347929   101.78111 |  100.0 |        0.0 |  101.78111 |
    | 2000-01-04 18:00:00 | 3.717849  3.718194  3.720981             3.735086 |  124.87972  124.897748  125.043217  125.780645 |  52.178486   52.181941   52.209811     52.350863 | 1.088388  1.088442  1.088874   1.09106 |  1.35916  1.359157  1.359128          1.358984 | -0.581095  -0.581069  -0.580865     -0.579832 | 0.672254  0.672227   0.67201      0.670939 |  0.90884  0.908832  0.908776      0.908511 |      100.0 | 100.001162  100.020784  100.179152          100.982928 | 101.630844 | 0.057656  0.057659  0.057683       0.057805 |  0.20729  0.207308  0.207447        0.208152 |      100.0  100.002273  100.038561  100.314808  101.630844 |  100.0 |        0.0 | 101.630844 |
    | 2000-01-04 18:30:00 | 3.717846  3.718155  3.720677               3.7336 | 124.879596  124.895719  125.027344   125.70285 |  52.178462   52.181552    52.20677     52.336001 | 1.088388  1.088436  1.088827  1.090829 |  1.35916  1.359157  1.359131          1.358999 | -0.581095  -0.581072  -0.580887     -0.579941 | 0.672254   0.67223  0.672033      0.671051 |  0.90884  0.908833  0.908782      0.908538 |      100.0 | 100.001027  100.018576  100.161868          100.898058 |   101.4927 | 0.057656  0.057659  0.057681       0.057792 |  0.20729  0.207306  0.207432        0.208078 |      100.0  100.002009  100.034483  100.284768    101.4927 |  100.0 |        0.0 |   101.4927 |
    | 2000-01-04 19:00:00 | 3.717844   3.71812  3.720402             3.732237 | 124.879487  124.893903  125.012971   125.63151 |  52.178441   52.181204   52.204017     52.322368 | 1.088388   1.08843  1.088784  1.090618 |  1.35916  1.359158  1.359134          1.359013 | -0.581095  -0.581075  -0.580907      -0.58004 | 0.672254  0.672232  0.672055      0.671155 |  0.90884  0.908834  0.908788      0.908564 |      100.0 | 100.000908  100.016599  100.146218          100.820245 |  101.36576 | 0.057656  0.057659  0.057678       0.057781 |  0.20729  0.207304  0.207418         0.20801 |      100.0  100.001776  100.030832   100.25753   101.36576 |  100.0 |        0.0 |  101.36576 |
    | 2000-01-04 19:30:00 | 3.717842  3.718089  3.720152             3.730987 |  124.87939  124.892279   124.99996  125.566117 |  52.178423   52.180893   52.201525     52.309868 | 1.088387  1.088426  1.088746  1.090425 |  1.35916  1.359158  1.359137          1.359026 | -0.581095  -0.581077  -0.580926     -0.580132 | 0.672255  0.672235  0.672074      0.671249 |  0.90884  0.908835  0.908793      0.908587 |      100.0 | 100.000803  100.014832  100.132053          100.748932 | 101.249166 | 0.057656  0.057658  0.057676        0.05777 |  0.20729  0.207302  0.207406        0.207947 |      100.0   100.00157  100.027565  100.232841  101.249166 |  100.0 |        0.0 | 101.249166 |
    | 2000-01-04 20:00:00 | 3.717841  3.718061  3.719927             3.729841 | 124.879304  124.890827  124.988186  125.506199 |  52.178406   52.180615   52.199269     52.298413 | 1.088387  1.088421  1.088711  1.090247 |  1.35916  1.359158  1.359139          1.359037 | -0.581095  -0.581079  -0.580942     -0.580216 | 0.672255  0.672237  0.672092      0.671336 |  0.90884  0.908835  0.908797      0.908609 |      100.0 |  100.00071  100.013251  100.119234          100.683601 | 101.142123 | 0.057656  0.057658  0.057674        0.05776 |  0.20729  0.207301  0.207394         0.20789 |      100.0  100.001388   100.02464  100.210468  101.142123 |  100.0 |        0.0 | 101.142123 |
    | 2000-01-04 20:30:00 | 3.717839  3.718037  3.719723             3.728792 | 124.879229  124.889528  124.977534   125.45132 |  52.178392   52.180366   52.197229     52.287918 | 1.088387  1.088417  1.088679  1.090085 |  1.35916  1.359158  1.359141          1.359048 | -0.581095  -0.581081  -0.580957     -0.580293 | 0.672255  0.672239  0.672107      0.671416 |  0.90884  0.908836  0.908801      0.908628 |      100.0 | 100.000628  100.011838  100.107637          100.623774 | 101.043891 | 0.057656  0.057658  0.057672       0.057751 |  0.20729    0.2073  0.207384        0.207838 |      100.0  100.001227  100.022023    100.1902  101.043891 |  100.0 |        0.0 | 101.043891 |
    | 2000-01-04 21:00:00 | 3.717838  3.718014  3.719538             3.727831 | 124.879162  124.888367  124.967898  125.401076 |  52.178379   52.180143   52.195383     52.278308 | 1.088387  1.088414   1.08865  1.089936 |  1.35916  1.359159  1.359143          1.359058 | -0.581095  -0.581082  -0.580971     -0.580363 | 0.672255  0.672241  0.672122      0.671489 |  0.90884  0.908836  0.908805      0.908647 |      100.0 | 100.000555  100.010574  100.097147          100.569007 | 100.953784 | 0.057656  0.057658  0.057671       0.057742 |  0.20729  0.207299  0.207375         0.20779 |      100.0  100.001085  100.019682  100.171845  100.953784 |  100.0 |        0.0 | 100.953784 |
    | 2000-01-04 21:30:00 | 3.717837  3.717994  3.719371             3.726951 | 124.879103  124.887329  124.959185  125.355093 |  52.178368   52.179944   52.193713     52.269512 | 1.088386  1.088411  1.088624  1.089799 |  1.35916  1.359159  1.359145          1.359067 | -0.581095  -0.581084  -0.580983     -0.580427 | 0.672255  0.672242  0.672135      0.671556 |  0.90884  0.908837  0.908808      0.908663 |      100.0 | 100.000491  100.009444  100.087662          100.518891 | 100.871163 | 0.057656  0.057657  0.057669       0.057735 |  0.20729  0.207298  0.207366        0.207746 |      100.0  100.000959  100.017588  100.155226  100.871163 |  100.0 |        0.0 | 100.871163 |
    | 2000-01-04 22:00:00 | 3.717836  3.717977   3.71922             3.726146 |  124.87905  124.886401  124.951308  125.313026 |  52.178358   52.179766   52.192204     52.261463 | 1.088386  1.088408  1.088601  1.089675 |  1.35916  1.359159  1.359146          1.359075 | -0.581096  -0.581085  -0.580994     -0.580486 | 0.672255  0.672244  0.672147      0.671617 |  0.90884  0.908837  0.908812      0.908679 |      100.0 | 100.000434  100.008434  100.079087          100.473049 | 100.795438 | 0.057656  0.057657  0.057668       0.057728 |  0.20729  0.207297  0.207359        0.207705 |      100.0  100.000848  100.015715  100.140183  100.795438 |  100.0 |        0.0 | 100.795438 |
    | 2000-01-04 22:30:00 | 3.717835  3.717961  3.719084              3.72541 | 124.879004  124.885571  124.944189  125.274555 |  52.178349   52.179607    52.19084     52.254101 | 1.088386  1.088406   1.08858  1.089561 |  1.35916  1.359159  1.359148          1.359083 | -0.581096  -0.581086  -0.581004      -0.58054 | 0.672255  0.672245  0.672157      0.671673 |  0.90884  0.908837  0.908814      0.908693 |      100.0 | 100.000383  100.007531  100.071336           100.43113 | 100.726061 | 0.057656  0.057657  0.057667       0.057722 |  0.20729  0.207296  0.207352        0.207669 |      100.0   100.00075  100.014039  100.126571  100.726061 |  100.0 |        0.0 | 100.726061 |
    | 2000-01-04 23:00:00 | 3.717834  3.717947  3.718961             3.724737 | 124.878963   124.88483  124.937756  125.239386 |  52.178341   52.179465   52.189607      52.24737 | 1.088386  1.088403  1.088561  1.089456 | 1.359161  1.359159  1.359149           1.35909 | -0.581096  -0.581087  -0.581013      -0.58059 | 0.672255  0.672246  0.672167      0.671725 |  0.90884  0.908838  0.908817      0.908705 |      100.0 | 100.000339  100.006724  100.064333          100.392813 | 100.662525 | 0.057656  0.057657  0.057666       0.057716 |  0.20729  0.207295  0.207346        0.207635 |      100.0  100.000663  100.012541  100.114256  100.662525 |  100.0 |        0.0 | 100.662525 |
    | 2000-01-04 23:30:00 | 3.717833  3.717934  3.718849             3.724122 | 124.878927  124.884167  124.931944  125.207246 |  52.178334   52.179338   52.188494     52.241219 | 1.088386  1.088401  1.088543  1.089361 | 1.359161  1.359159   1.35915          1.359096 | -0.581096  -0.581088  -0.581021     -0.580635 | 0.672255  0.672247  0.672176      0.671772 |  0.90884  0.908838  0.908819      0.908717 |      100.0 |   100.0003  100.006003  100.058006            100.3578 | 100.604361 | 0.057656  0.057657  0.057665        0.05771 |  0.20729  0.207295   0.20734        0.207604 |      100.0  100.000586  100.011202  100.103118  100.604361 |  100.0 |        0.0 | 100.604361 |


.. _wq_widths_strickler_musk_mct_no_flow_areas:

No-flow areas
-------------

Next, we demonstrate the effect of introducing "no-flow areas" by widening the total
profile width in the channel's upper range but not the flow width (which marks those
areas that actively contribute to water routing).  |musk_mct| can react a little
sensitive to strong discontinuities, so we prefer introduce this change smoothly by
letting the difference between both widths continually grow, beginning at the second
height/width pair.  Hence, the resulting more pronounced retention effect shows only
after the water level exceeds the second pair's measuring height:

.. integration-test::

    >>> model.wqmodel.parameters.control.totalwidths(15.0, 75.0, 2015.0)
    >>> model.prepare_states(100.0)
    >>> test("wq_widths_strickler_musk_mct_no_flow_areas")
    |                date |                               referencewaterdepth |                                     wettedarea |                                     surfacewidth |                               celerity |                               correctingfactor |                                  coefficient1 |                               coefficient2 |                               coefficient3 |     inflow |                                     referencedischarge |    outflow |                               courantnumber |                               reynoldsnumber |                                                  discharge | input1 |     input2 |     output |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 |      100.0 |      100.0       100.0       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 |      100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878653  124.878651  124.878652  124.878652 |  52.178282   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 | 100.000003 | 100.000001   99.999999       100.0               100.0 |      100.0 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 | 100.000003   99.999999  100.000001       100.0       100.0 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 3.717828  3.717828  3.717828             3.717828 | 124.878663  124.878648  124.878654  124.878651 |  52.178283    52.17828   52.178282     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 | 100.000029 | 100.000012   99.999995  100.000002           99.999999 | 100.000001 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 | 100.000029   99.999989  100.000004   99.999998  100.000001 |  100.0 |   0.000029 | 100.000001 |
    | 2000-01-01 04:00:00 | 3.717829  3.717828  3.717828             3.717828 | 124.878717  124.878627  124.878662  124.878648 |  52.178294   52.178276   52.178283     52.178281 | 1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 |  0.90884   0.90884   0.90884       0.90884 | 100.000175 | 100.000071   99.999972  100.000011           99.999996 | 100.000004 | 0.057656  0.057656  0.057656       0.057656 | 0.207289  0.207289  0.207289        0.207289 | 100.000175   99.999932  100.000026    99.99999  100.000004 |  100.0 |   0.000175 | 100.000004 |
    | 2000-01-01 04:30:00 | 3.717834  3.717826  3.717829             3.717828 | 124.878958  124.878535  124.878696  124.878635 |   52.17834   52.178259    52.17829     52.178278 | 1.088386  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161          1.359161 | -0.581096  -0.581096  -0.581096     -0.581096 | 0.672256  0.672256  0.672256      0.672256 | 0.908841   0.90884   0.90884       0.90884 | 100.000826 | 100.000333   99.999873  100.000048           99.999982 | 100.000017 | 0.057656  0.057656  0.057656       0.057656 |  0.20729  0.207289  0.207289        0.207289 | 100.000826   99.999686  100.000119   99.999955  100.000017 |  100.0 |   0.000826 | 100.000017 |
    | 2000-01-01 05:00:00 | 3.717851   3.71782  3.717831             3.717827 |  124.87983  124.878213  124.878815  124.878592 |  52.178507   52.178197   52.178313      52.17827 | 1.088389  1.088384  1.088386  1.088385 |  1.35916  1.359161  1.359161          1.359161 | -0.581094  -0.581097  -0.581096     -0.581096 | 0.672256  0.672255  0.672256      0.672256 | 0.908842   0.90884  0.908841       0.90884 | 100.003194 | 100.001282   99.999523  100.000177           99.999934 | 100.000061 | 0.057656  0.057656  0.057656       0.057656 |  0.20729  0.207289  0.207289        0.207289 | 100.003194    99.99881  100.000442   99.999836  100.000061 |  100.0 |   0.003194 | 100.000061 |
    | 2000-01-01 05:30:00 | 3.717902  3.717801  3.717838             3.717825 | 124.882516  124.877248   124.87916  124.878469 |  52.179022   52.178012   52.178379     52.178246 | 1.088397  1.088381  1.088387  1.088384 |  1.35916  1.359161   1.35916          1.359161 | -0.581091  -0.581098  -0.581095     -0.581096 | 0.672258  0.672255  0.672256      0.672256 | 0.908845  0.908839  0.908841       0.90884 | 100.010515 | 100.004206   99.998472  100.000553           99.999801 | 100.000179 | 0.057657  0.057656  0.057656       0.057656 | 0.207293  0.207288   0.20729        0.207289 | 100.010515   99.996177  100.001384   99.999501  100.000179 |  100.0 |   0.010515 | 100.000179 |
    | 2000-01-01 06:00:00 | 3.718041  3.717753  3.717855             3.717819 | 124.889772  124.874721  124.880033   124.87817 |  52.180412   52.177528   52.178546     52.178189 | 1.088418  1.088373  1.088389  1.088384 | 1.359158  1.359161   1.35916          1.359161 |  -0.58108  -0.581102  -0.581094     -0.581097 | 0.672261  0.672254  0.672256      0.672256 | 0.908853  0.908836  0.908842       0.90884 | 100.030315 | 100.012103   99.995722  100.001503           99.999475 | 100.000456 | 0.057658  0.057655  0.057656       0.057656 |   0.2073  0.207286  0.207291        0.207289 | 100.030315   99.989278   100.00377   99.998684  100.000456 |  100.0 |   0.030315 | 100.000456 |
    | 2000-01-01 06:30:00 | 3.718378   3.71764  3.717892             3.717807 | 124.907324  124.868822  124.881993  124.877528 |  52.183776   52.176397   52.178922     52.178066 |  1.08847  1.088356  1.088395  1.088382 | 1.359155  1.359163   1.35916          1.359161 | -0.581056   -0.58111  -0.581091     -0.581098 | 0.672266  0.672253  0.672257      0.672255 |  0.90887   0.90883  0.908844      0.908839 | 100.078179 | 100.031208   99.989301  100.003636           99.998776 |  100.00102 | 0.057661  0.057654  0.057657       0.057656 | 0.207317   0.20728  0.207293        0.207288 | 100.078179    99.97319  100.009114   99.996933   100.00102 |  100.0 |   0.078179 |  100.00102 |
    | 2000-01-01 07:00:00 | 3.719119    3.7174  3.717968             3.717783 | 124.946002  124.856327  124.885961  124.876294 |  52.191187   52.174003   52.179682     52.177829 | 1.088585  1.088319  1.088407  1.088378 | 1.359147  1.359165  1.359159          1.359161 | -0.581001  -0.581127  -0.581086     -0.581099 | 0.672272  0.672251  0.672257      0.672255 | 0.908905   0.90882  0.908847      0.908838 | 100.183351 | 100.073312   99.975702  100.007955           99.997434 | 100.002028 | 0.057667  0.057652  0.057657       0.057656 | 0.207354  0.207268  0.207296        0.207287 | 100.183351   99.939247  100.019883   99.993591  100.002028 |  100.0 |   0.183351 | 100.002028 |
    | 2000-01-01 07:30:00 | 3.720626  3.716934  3.718109             3.717742 | 125.024672  124.832018  124.893291  124.874154 |  52.206259   52.169343   52.181087     52.177419 | 1.088819  1.088246  1.088429  1.088372 | 1.359132   1.35917  1.359158          1.359161 | -0.580891  -0.581162  -0.581075     -0.581102 | 0.672278  0.672251  0.672257      0.672256 |  0.90897    0.9088  0.908852      0.908837 | 100.396214 | 100.158964   99.949246  100.015934           99.995105 | 100.003618 |  0.05768  0.057648  0.057658       0.057655 | 0.207429  0.207245  0.207303        0.207285 | 100.396214   99.873628  100.039612   99.987857  100.003618 |  100.0 |   0.396214 | 100.003618 |
    | 2000-01-01 08:00:00 | 3.723484  3.716093  3.718347             3.717677 | 125.173935  124.788138   124.90575  124.870788 |  52.234842   52.160931   52.183474     52.176774 | 1.089262  1.088116  1.088466  1.088362 | 1.359102  1.359178  1.359155          1.359162 | -0.580681  -0.581223  -0.581058     -0.581107 | 0.672276  0.672254  0.672254      0.672257 | 0.909083   0.90877   0.90886      0.908835 | 100.797305 | 100.321521   99.901495  100.029495           99.991441 | 100.005804 | 0.057705  0.057641   0.05766       0.057655 | 0.207572  0.207202  0.207315        0.207282 | 100.797305   99.756253  100.072747   99.978988  100.005804 |  100.0 |   0.797305 | 100.005804 |
    | 2000-01-01 08:30:00 | 3.728583  3.714673  3.718723             3.717585 | 125.440382  124.714086  124.925346   124.86599 |  52.285826   52.146733   52.187229     52.175855 | 1.090052  1.087896  1.088524  1.088347 |  1.35905  1.359193  1.359151          1.359163 | -0.580308  -0.581328   -0.58103     -0.581114 | 0.672249  0.672268  0.672249      0.672259 | 0.909266  0.908723  0.908871      0.908833 | 101.507032 | 100.611865   99.820924  100.050826           99.986219 | 100.008338 | 0.057749  0.057629  0.057664       0.057654 | 0.207827  0.207131  0.207334        0.207277 | 101.507032   99.560533  100.124003   99.966666  100.008338 |  100.0 |   1.507032 | 100.008338 |
    | 2000-01-01 09:00:00 | 3.737195  3.712419  3.719271             3.717468 | 125.891047  124.596584  124.953927  124.859857 |  52.371948   52.124195   52.192706     52.174679 | 1.091386  1.087546  1.088609  1.088329 | 1.358962  1.359216  1.359146          1.359164 | -0.579677  -0.581493   -0.58099     -0.581122 | 0.672166    0.6723  0.672237      0.672263 | 0.909547  0.908658  0.908884      0.908832 | 102.694853 | 101.103426   99.693112  100.081939           99.979544 |  100.01054 | 0.057823  0.057609  0.057668       0.057653 | 0.208258  0.207019  0.207361        0.207271 | 102.694853   99.254644  100.197098   99.951541   100.01054 |  100.0 |   2.694853 |  100.01054 |
    | 2000-01-01 09:30:00 | 3.751048  3.709042   3.72001             3.717337 | 126.617518  124.420603   124.99252  124.853035 |  52.510478   52.090422     52.2001     52.173372 | 1.093529  1.087022  1.088723  1.088309 | 1.358822  1.359251  1.359138          1.359166 | -0.578665  -0.581741  -0.580936     -0.581132 | 0.671976  0.672365  0.672215       0.67227 | 0.909953  0.908573  0.908896      0.908831 | 104.586356 | 101.897084   99.501767  100.123954           99.972119 | 100.011152 | 0.057943   0.05758  0.057675       0.057652 |  0.20895   0.20685  0.207398        0.207265 | 104.586356   98.805047  100.292783   99.935895  100.011152 |  100.0 |   4.586356 | 100.011152 |
    | 2000-01-01 10:00:00 | 3.772364  3.704255  3.720927             3.717221 | 127.739105  124.171337  125.040369  124.846996 |  52.723639   52.042547   52.209265     52.172214 | 1.096822  1.086278  1.088866  1.088291 | 1.358607  1.359301  1.359129          1.359167 | -0.577111  -0.582093  -0.580869     -0.581141 | 0.671599  0.672482   0.67218      0.672279 | 0.910509  0.908471  0.908906      0.908834 | 107.466641 | 103.125447   99.230898  100.176052           99.965547 |  100.00827 | 0.058127  0.057538  0.057683       0.057651 | 0.210015   0.20661  0.207444        0.207259 | 107.466641   98.182852  100.406193   99.924325   100.00827 |  100.0 |   7.466641 |  100.00827 |
    | 2000-01-01 10:30:00 | 3.803855  3.697836  3.721948              3.71717 | 129.404414  123.837487  125.093715    124.8443 |  53.038555   51.978358   52.219482     52.171697 | 1.101674  1.085281  1.089024  1.088283 | 1.358296  1.359368  1.359118          1.359167 | -0.574823  -0.582564  -0.580794     -0.581144 | 0.670921  0.672673  0.672129      0.672291 | 0.911232  0.908361  0.908907      0.908841 | 111.678559 | 104.956077   98.868406  100.234143           99.962614 |  99.999413 | 0.058397  0.057483  0.057692        0.05765 | 0.211587  0.206289  0.207495        0.207256 | 111.678559     97.3735  100.523695    99.92429   99.999413 |  100.0 |  11.678559 |  99.999413 |
    | 2000-01-01 11:00:00 | 3.848658  3.689721  3.722918             3.717256 |  131.79072  123.416028  125.144382  124.848781 |  53.486582   51.897211   52.229184     52.172556 | 1.108552   1.08402  1.089174  1.088296 | 1.357863  1.359452  1.359108          1.359166 | -0.571585  -0.583162  -0.580723     -0.581138 | 0.669786  0.672964   0.67206      0.672304 | 0.912117  0.908259  0.908893      0.908855 | 117.614836 |  107.59328   98.411265  100.289324           99.967491 |  99.981791 |  0.05878  0.057412    0.0577       0.057651 |  0.21382  0.205883  0.207544        0.207261 | 117.614836   96.389821   100.61956   99.946338   99.981791 |  100.0 |  17.614836 |  99.981791 |
    | 2000-01-01 11:30:00 | 3.910188  3.680124  3.723569             3.717577 | 135.100656  122.918433  125.178341  124.865567 |  54.101877   51.801242   52.235685     52.175774 | 1.117953  1.082527  1.089275  1.088346 | 1.357285  1.359553  1.359102          1.359163 |  -0.56717  -0.583869  -0.580675     -0.581114 | 0.667997  0.673377  0.671976      0.672315 | 0.913135  0.908193  0.908853      0.908876 | 125.703736 | 111.278182   97.872232  100.326312            99.98576 |  99.952811 | 0.059304  0.057329  0.057706       0.057654 | 0.216882  0.205402  0.207576        0.207277 | 125.703736   95.287979  100.652992  100.003781   99.952811 |  100.0 |  25.703736 |  99.952811 |
    | 2000-01-01 12:00:00 | 3.991919  3.669682  3.723495             3.718253 |  139.55587  122.378063  125.174507  124.900845 |   54.91919   51.696821   52.234951     52.182534 | 1.130361  1.080901  1.089264  1.088451 | 1.356548  1.359663  1.359102          1.359156 |  -0.56136  -0.584639  -0.580681     -0.581065 | 0.665323   0.67393  0.671884      0.672319 | 0.914224    0.9082  0.908779      0.908906 | 136.388647 | 116.286724   97.287706  100.322132          100.024158 |  99.910852 | 0.059995  0.057238  0.057705        0.05766 |  0.22094  0.204879  0.207573        0.207311 | 136.388647    94.18499  100.566289  100.111595   99.910852 |  100.0 |  36.388647 |  99.910852 |
    | 2000-01-01 12:30:00 | 4.097098  3.659613  3.722148             3.719411 | 145.387535  121.858017  125.104158  124.961248 |  55.970981   51.596127   52.221482     52.194108 |   1.1462  1.079331  1.089055   1.08863 | 1.355648   1.35977  1.359116          1.359144 | -0.553974  -0.585383  -0.580779      -0.58098 | 0.661509  0.674624    0.6718      0.672309 | 0.915286  0.908332   0.90866      0.908945 | 150.102726 | 122.925104      96.726  100.245506           100.08991 |  99.856289 | 0.060876  0.057151  0.057693        0.05767 | 0.226147  0.204374  0.207505        0.207368 | 150.102726   93.275474  100.285189  100.284286   99.856289 |  100.0 |  50.102726 |  99.856289 |
    | 2000-01-01 13:00:00 | 4.228418  3.651861  3.718841             3.721164 | 152.823881  121.458358  124.931493  125.052768 |  57.284183    51.51861   52.188407      52.21164 | 1.165779  1.078121  1.088542  1.088902 | 1.354594  1.359853   1.35915          1.359126 | -0.544891  -0.585956  -0.581022     -0.580851 | 0.656307  0.675434   0.67175      0.672277 | 0.916196  0.908649  0.908488      0.908991 | 167.240349 | 131.521964   96.294888  100.057502          100.189557 |  99.792735 | 0.061964  0.057083  0.057665       0.057685 | 0.232626  0.203986   0.20734        0.207456 | 167.240349   92.844349   99.722725  100.532538   99.792735 |  100.0 |  67.240349 |  99.792735 |
    | 2000-01-01 13:30:00 | 4.387894  3.649242  3.712774              3.72359 | 162.086462  121.323445  124.615066  125.179459 |  58.878937   51.492416    52.12774     52.235899 | 1.189277  1.077712  1.087601  1.089278 | 1.353409   1.35988  1.359213          1.359101 | -0.534058   -0.58615  -0.581467     -0.580674 | 0.649517    0.6763  0.671768      0.672211 | 0.916833  0.909216  0.908259      0.909037 | 188.127569 | 142.429764   96.149483   99.713198          100.327536 |  99.727995 | 0.063268   0.05706  0.057612       0.057706 | 0.240463  0.203854  0.207036        0.207578 | 188.127569   93.277116   98.785758  100.859457   99.727995 |  100.0 |  88.127569 |  99.727995 |
    | 2000-01-01 14:00:00 |  4.57581  3.655275  3.703201             3.726652 | 173.327322  121.634305  124.116511  125.339472 |  60.758098   51.552751   52.032012     52.266523 | 1.216596  1.078654  1.086115  1.089753 | 1.352133  1.359815  1.359312           1.35907 | -0.521561  -0.585703   -0.58217     -0.580449 | 0.640958  0.677089  0.671909      0.672101 | 0.917024  0.910075  0.907984      0.909074 | 212.993962 | 155.953114   96.484681   99.171329          100.501875 |   99.67711 | 0.064783  0.057113  0.057529       0.057732 | 0.249657  0.204157  0.206557        0.207731 | 212.993962   95.029868   97.400118  101.250369    99.67711 |  100.0 | 112.993962 |   99.67711 |
    | 2000-01-01 14:30:00 | 4.791858  3.674415   3.68942             3.730214 | 186.687365  122.622837  123.400398  125.525678 |  62.918577   51.744147     51.8942     52.302137 | 1.247541  1.081638  1.083973  1.090305 | 1.350807  1.359612  1.359456          1.359033 | -0.507531  -0.584289  -0.583184     -0.580188 |  0.63066  0.677635  0.672223      0.671939 | 0.916733  0.911275  0.907672      0.909092 | 241.948195 | 172.415486   97.552489   98.394304          100.704846 |  99.659742 | 0.066496   0.05728   0.05741       0.057763 | 0.260177  0.205116  0.205867        0.207909 | 241.948195   98.647581   95.515096  101.673842   99.659742 |  100.0 | 141.948195 |  99.659742 |
    | 2000-01-01 15:00:00 | 5.034228  3.711521  3.671088             3.733928 | 202.230689  124.549766  122.450745      125.72 |  65.342282   52.115212   51.710878     52.339278 | 1.281704  1.087407   1.08112   1.09088 | 1.349476  1.359223  1.359649          1.358996 | -0.492198  -0.581558  -0.584535     -0.579917 | 0.618677   0.67766  0.672778      0.671722 | 0.915896  0.912787  0.907362      0.909072 |  274.95935 | 192.074476   99.642363   97.366248           100.91677 |   99.70227 | 0.068384  0.057601  0.057251       0.057795 | 0.271921  0.206974  0.204949        0.208094 |  274.95935  104.704416   93.142778  102.070179    99.70227 |  100.0 |  174.95935 |   99.70227 |
    | 2000-01-01 15:30:00 |  5.29996  3.771591  3.648412             3.737208 | 219.947254  127.698353  121.280748  125.891743 |  67.999596   52.715909   51.484123     52.372081 | 1.318538  1.096702  1.077583  1.091388 |  1.34818  1.358613   1.35989          1.358962 | -0.475852  -0.577166  -0.586212     -0.579677 | 0.605203  0.676828  0.673634      0.671455 | 0.914523  0.914535  0.907108      0.908996 | 311.845554 | 215.111403  103.080918   96.103399          101.104163 |  99.836036 | 0.070417   0.05812  0.057053       0.057823 | 0.284732  0.209977  0.203813        0.208258 | 311.845554  113.768341   90.386002  102.347498   99.836036 |  100.0 | 211.845554 |  99.836036 |
    | 2000-01-01 16:00:00 | 5.585172  3.859237  3.622409             3.739201 | 239.748304   132.35708  119.945391  125.996158 |  70.851719   53.592365   51.224094     52.392014 | 1.357401  1.110172  1.073518  1.091697 |  1.34695   1.35776  1.360171          1.358942 | -0.458816  -0.570823  -0.588141     -0.579531 | 0.590515  0.674746  0.674838      0.671157 |  0.91267  0.916369  0.906985      0.908844 | 352.270855 | 241.608497   108.22176   94.667148           101.21813 | 100.093373 | 0.072559  0.058871  0.056826       0.057841 | 0.298416  0.214347  0.202508        0.208358 | 352.270855  126.354752   87.465528  102.381498  100.093373 |  100.0 | 252.270855 | 100.093373 |
    | 2000-01-01 16:30:00 | 5.885159  3.978264  3.595198             3.738769 |  261.45286  138.806893  118.555199  125.973499 |  73.851589   54.782642   50.951977     52.387689 | 1.397583  1.128295  1.069252   1.09163 |  1.34581  1.356667  1.360469          1.358947 | -0.441429  -0.562326  -0.590167     -0.579563 | 0.574938  0.671019  0.676407      0.670862 | 0.910417  0.918088  0.907094      0.908598 | 395.750601 | 271.510751  115.441037   93.177717          101.193385 | 100.502931 |  0.07477   0.05988  0.056588       0.057837 | 0.312742  0.220263  0.201142        0.208336 | 395.750601  142.873598   84.746077  102.017647  100.502931 |  100.0 | 295.750601 | 100.502931 |
    | 2000-01-01 17:00:00 | 6.203778  4.142576  3.564596             3.736783 | 285.711894   147.94332   117.00067  125.869505 |  79.205641    56.42576   50.645961     52.367835 | 1.400137  1.153005  1.064442  1.091322 | 1.308952  1.355272  1.360811          1.358967 | -0.414555   -0.55081  -0.592455     -0.579708 | 0.555155  0.665986  0.677895      0.670782 | 0.901891  0.920418  0.906997      0.908457 | 441.664962 | 305.615374  125.863505   91.519264          101.079872 | 100.958179 | 0.077016  0.061254  0.056319        0.05782 | 0.336857  0.228394  0.199603        0.208237 | 441.664962  165.464782   82.013981  101.389117  100.958179 |  100.0 | 341.664962 | 100.958179 |
    | 2000-01-01 17:30:00 | 6.522671  4.334283  3.545191             3.727335 | 312.019375  158.944314  116.019765  125.375168 |  85.787043   58.342834   50.451911     52.273352 | 1.384854  1.181411  1.061384  1.089859 | 1.262586  1.353795   1.36103          1.359064 | -0.385016  -0.537675   -0.59391     -0.580399 | 0.525588  0.656826  0.680693      0.670392 | 0.894615  0.920452  0.908382      0.907776 | 489.279607 | 342.234959  138.705343   90.476687           100.54073 | 101.701778 | 0.078972  0.062832  0.056148       0.057738 | 0.365054  0.237833  0.198627        0.207765 | 489.279607   191.78094   81.055695    99.78659  101.701778 |  100.0 | 389.279607 | 101.701778 |
    | 2000-01-01 18:00:00 | 6.835159  4.549458  3.543071             3.708922 | 339.834416  171.729731  115.912839  124.414363 |  92.236256   60.494583   50.430713     52.089224 | 1.374198  1.212789  1.061049  1.087003 | 1.227022  1.352304  1.361053          1.359253 | -0.358529  -0.523296  -0.594069      -0.58175 | 0.495155  0.645759  0.683247      0.670368 | 0.891994  0.919719  0.910293      0.907017 |  537.77215 | 380.596139  154.012605   90.363289           99.494931 |  102.75922 | 0.080636  0.064572   0.05613       0.057579 | 0.391545  0.248371  0.198521        0.206843 |  537.77215  220.529884   82.990297   96.914107   102.75922 |  100.0 |  437.77215 |  102.75922 |
    | 2000-01-01 18:30:00 | 7.137676  4.787403  3.561219             3.682734 | 368.681859  186.407158  116.829669  123.053675 |  98.479699   62.874026   50.612186     51.827343 | 1.367136  1.246907   1.06391  1.082933 | 1.199724  1.350833  1.360846          1.359526 | -0.334927  -0.507816  -0.592707     -0.583677 | 0.467991  0.633341  0.684905      0.670988 | 0.890294  0.918582  0.912339      0.906442 | 586.262607 | 420.128534  172.066019   91.337487           98.018603 | 103.870751 | 0.082047  0.066461   0.05629       0.057352 | 0.416162  0.259961  0.199434        0.205532 | 586.262607  251.653906   88.110025   93.035573  103.870751 |  100.0 | 486.262607 | 103.870751 |
    | 2000-01-01 19:00:00 | 7.427901  5.043986  3.604197              3.64984 | 398.132306  202.868792   119.01416  121.354266 | 104.469447   65.439864   51.041975     51.498401 | 1.362812  1.283068  1.070664  1.077806 | 1.178696  1.349426  1.360366          1.359875 | -0.313944  -0.491589  -0.589494     -0.586106 | 0.443864  0.619572  0.685548      0.672167 | 0.889291  0.916874  0.914601      0.906025 | 633.846018 | 460.321815  192.892721   93.669072           96.182596 | 104.742201 | 0.083247  0.068459  0.056667       0.057066 | 0.438889  0.272392  0.201595        0.203884 | 633.846018  285.022256   96.589582   88.555144  104.742201 |  100.0 | 533.846018 | 104.742201 |
    | 2000-01-01 19:30:00 |  7.70363  5.314656   3.67611             3.611836 | 427.722118    220.9477  122.710576  119.404321 |  110.16003   68.146563     51.7611     51.118357 | 1.360552  1.320557  1.081902  1.071862 | 1.162459  1.348113  1.359591          1.360287 | -0.295338  -0.474962  -0.584163     -0.588927 | 0.422458  0.604827  0.684786       0.67388 | 0.888794   0.91471  0.916917      0.905807 | 679.625318 | 500.609505  216.431428   97.647626           94.086716 | 105.130542 | 0.084269  0.070528  0.057294       0.056734 | 0.459729  0.285439  0.205202        0.201977 | 679.625318   320.38042  108.571888   83.917232  105.130542 |  100.0 | 579.625318 | 105.130542 |
    | 2000-01-01 20:00:00 | 7.962848  5.594669  3.779834             3.571336 | 456.970953  240.421612  128.133235  117.342248 | 115.509846   70.946686    52.79834     50.713361 | 1.359815  1.358684  1.097974  1.065503 | 1.149902  1.346913  1.358529          1.360736 | -0.278896  -0.458258  -0.576566      -0.59195 | 0.403503  0.589484    0.6822      0.676105 | 0.888658  0.912214  0.919036      0.905878 | 722.742733 | 540.390284  242.522709  103.558296           91.883009 | 104.841096 | 0.085143  0.072629  0.058191       0.056378 | 0.478705   0.29887  0.210389        0.199942 | 722.742733  357.369569  124.132284   79.620343  104.841096 |  100.0 | 622.742733 | 104.841096 |
    | 2000-01-01 20:30:00 | 8.203797  5.879994   3.91625             3.531816 | 485.401995   261.07157  135.428802  115.345846 | 120.482617   73.799942   54.162497     50.318157 | 1.360167  1.396896  1.118877  1.059272 | 1.140183   1.34583  1.357226          1.361184 | -0.264431  -0.441725  -0.566735     -0.594917 | 0.386768  0.573923  0.677434      0.678755 | 0.888771  0.909543  0.920661      0.906313 | 762.408181 | 579.054171  270.977771  111.645467           89.762062 | 103.733932 | 0.085891  0.074732  0.059356        0.05603 | 0.495847  0.312495  0.217184        0.197954 | 762.408181  395.549147  143.282264   76.191642  103.733932 |  100.0 | 662.408181 | 103.733932 |
    | 2000-01-01 21:00:00 | 8.425013  6.173359  4.095142             3.493394 | 512.559679  283.312067  145.278083  113.419917 | 125.048142   78.577839   55.951422      49.93394 | 1.361259  1.401869  1.145907  1.053191 | 1.132661   1.31402  1.355662           1.36163 | -0.251778  -0.417518  -0.554109     -0.597817 | 0.372053   0.55529  0.670996      0.681366 |  0.88905  0.901711  0.922494      0.906757 | 797.923464 | 616.006597  302.252963  122.799923           87.727839 | 101.956334 | 0.086531  0.076814   0.06086        0.05569 | 0.511196  0.334104  0.226051        0.196018 | 797.923464  434.419761  167.465774   73.633877  101.956334 |  100.0 | 697.923464 | 101.956334 |
    | 2000-01-01 21:30:00 | 8.625341  6.461028  4.299087             3.469631 | 538.024401  306.770429  156.897046  112.236139 | 129.182564    84.51484   57.990869     49.696305 | 1.362812  1.387428  1.176229  1.049418 | 1.126844  1.270669  1.354055          1.361911 | -0.240793  -0.390519  -0.540063     -0.599618 | 0.359188  0.528492  0.660622       0.68497 | 0.889432  0.894654  0.922069      0.908589 | 828.701418 | 650.689548  334.958963  136.291959           86.483379 |  99.017717 | 0.087077  0.078616  0.062544       0.055479 | 0.524796  0.359696  0.236104        0.194819 | 828.701418  473.445793  194.521928   73.472913   99.017717 |  100.0 | 728.701418 |  99.017717 |
    | 2000-01-01 22:00:00 |  8.80393  6.736867  4.521034             3.467432 | 561.424089  330.868032  170.014254  112.126919 | 132.868335   90.207678    60.21034     49.674323 | 1.364602  1.377146  1.208673  1.049069 | 1.122357  1.237311  1.352492          1.361936 | -0.231349  -0.366617  -0.525174     -0.599784 | 0.348023  0.500749  0.648366      0.688194 | 0.889873  0.892311  0.920689      0.911029 |  854.27953 | 682.599294   368.26129  151.935569           86.368867 |  94.769625 |  0.08754  0.080137  0.064344        0.05546 | 0.536695  0.383331  0.246982        0.194708 |  854.27953  512.077004  222.915618   76.697476   94.769625 |  100.0 |  754.27953 |  94.769625 |
    | 2000-01-01 22:30:00 | 8.960223  6.998933  4.759551             3.489217 | 582.442591  355.217087  184.659888  113.211412 | 136.093962   95.616271   62.595509     49.892166 | 1.366455  1.370019  1.242944  1.052529 | 1.118911  1.211483  1.350997          1.361676 | -0.223336  -0.345519  -0.509607     -0.598131 | 0.338431    0.4763  0.635143      0.690197 | 0.890339  0.890793  0.918985      0.913511 | 874.327871 | 711.300315  401.701298  169.890706           87.508543 |  89.636043 | 0.087929  0.081422  0.066241       0.055654 | 0.546945  0.404993  0.258607        0.195808 | 874.327871  549.768385  252.518229   83.382094   89.636043 |  100.0 | 774.327871 |  89.636043 |
    | 2000-01-01 23:00:00 | 9.093939  7.246592  5.010127             3.538508 | 600.825027  379.530262   200.65876  115.682811 | 138.853631  100.727535   65.101269     50.385079 | 1.368236  1.365256  1.278332  1.060329 | 1.116284  1.191303  1.349603          1.361102 | -0.216656  -0.326873  -0.493705     -0.594411 |   0.3303  0.454783  0.620998      0.690845 | 0.890809  0.889875  0.916821       0.91606 | 888.651531 | 736.435269  434.948984  190.062199           90.119507 |  84.169223 | 0.088251  0.082513  0.068198        0.05609 | 0.555599  0.424789  0.270755        0.198292 | 888.651531  585.997668   283.18809   93.448494   84.169223 |  100.0 | 788.651531 |  84.169223 |
    | 2000-01-01 23:30:00 | 9.205048  7.479203  5.268129             3.618072 |  616.38034   403.51897  217.787842  119.723298 | 141.146737  105.528236   67.681289     51.180719 | 1.369846  1.362265  1.314159  1.072839 | 1.114308  1.175423  1.348328          1.360212 | -0.211222  -0.310389  -0.477786     -0.588461 | 0.323532  0.435802  0.606311      0.689777 | 0.891265   0.88938  0.914328      0.918475 | 897.187983 | 757.731031  467.661019  212.268589           94.429203 |  78.931765 | 0.088511  0.083445  0.070175       0.056788 | 0.562713  0.442819    0.2832        0.202292 | 897.187983  620.280052    314.7127  106.777228   78.931765 |  100.0 | 797.187983 |  78.931765 |
    | 2000-01-02 00:00:00 | 9.293749  7.696158  5.529931             3.728661 | 628.981353  426.899579  235.849633  125.444458 | 142.977369   110.00582   70.299308     52.286606 |  1.37121  1.360592  1.349926  1.090064 | 1.112857  1.162858  1.347178          1.359044 | -0.206957  -0.295827  -0.462077       -0.5803 | 0.318038   0.41904  0.591472      0.686611 | 0.891697  0.889181  0.911681      0.920443 |      900.0 | 775.001018   499.49005  236.330624          100.616701 |  74.456263 | 0.088715  0.084243  0.072147        0.05775 | 0.568344  0.459173   0.29577        0.207833 |      900.0  652.179924  346.826043  123.229975   74.456263 |  100.0 |      800.0 |  74.456263 |
    | 2000-01-02 00:30:00 | 9.360441   7.89694  5.791921             3.869901 | 638.562714  449.402701  254.610567  132.929195 | 144.353781  114.149607   72.919211     53.699012 |  1.37228  1.359886  1.385167  1.111804 | 1.111836  1.152874  1.346151          1.357657 | -0.203792  -0.282987  -0.446778     -0.570055 | 0.313741  0.404241  0.576724       0.68121 | 0.892097  0.889186  0.908956      0.921793 | 897.264896 | 788.144175  530.098292  261.989991          108.857519 |  71.286809 | 0.088866  0.084928  0.074087       0.058962 | 0.572551  0.473934  0.308295        0.214879 | 897.264896  681.319522  379.226097  142.603669   71.286809 |  100.0 | 797.264896 |  71.286809 |
    | 2000-01-02 01:00:00 |   9.4057   8.08115  6.054131             4.043328 | 645.117121  470.780387  274.090085  142.392419 | 145.287845  117.951392   76.117178     55.433279 | 1.373027  1.359872  1.409179  1.138121 | 1.111175  1.144916  1.335129          1.356099 | -0.201664  -0.271701  -0.429385     -0.557736 | 0.310572  0.391197   0.56139      0.673764 | 0.892461  0.889329  0.904779      0.922677 | 889.260928 | 797.141087  559.168416  289.291874          119.504358 |  69.724672 | 0.088967  0.085518  0.075993       0.060427 | 0.575392  0.487178   0.32321        0.223488 | 889.260928   707.38464  411.590012  165.187775   69.724672 |  100.0 | 789.260928 |  69.724672 |
    | 2000-01-02 01:30:00 |  9.43025  8.248522  6.312943             4.247409 | 648.690218   490.81129  294.481354  153.913529 | 145.794523  121.405677   81.458621     57.474086 |  1.37344  1.360332  1.394335  1.168593 | 1.110828  1.138562  1.291752          1.354446 | -0.200516  -0.261827  -0.404138     -0.543589 | 0.308466  0.379736    0.5405      0.664049 | 0.892786  0.889563  0.895503      0.922802 | 876.351729 | 802.047763  586.411862  317.867398          132.793969 |  70.144339 | 0.089022  0.086024  0.077718        0.06212 | 0.576929  0.498979  0.346643        0.233562 | 876.351729  730.127608  443.587211  191.119752   70.144339 |  100.0 | 776.351729 |  70.144339 |
    | 2000-01-02 02:00:00 | 9.434942  8.398922  6.559745             4.462907 | 649.374555  509.304091  315.214001  166.531289 | 145.891364  124.509671   86.552181     59.629068 | 1.373519  1.361095  1.383385  1.200229 | 1.110762  1.133486   1.25791          1.352885 | -0.200297  -0.253241  -0.381753     -0.529035 | 0.307367  0.369709  0.514597      0.651348 |  0.89307  0.889851  0.893186      0.920895 | 858.969595 |  802.98766  611.574871  346.656247          147.740394 |  73.918731 | 0.089032  0.086458  0.079182       0.063876 | 0.577223  0.509404  0.368255        0.244138 | 858.969595  749.367844  474.890568  217.683301   73.918731 |  100.0 | 758.969595 |  73.918731 |
    | 2000-01-02 02:30:00 | 9.420729  8.532339  6.792287             4.690762 | 647.302962  526.099403  335.899063  180.377653 | 145.598015  127.263159   91.351459     61.907617 | 1.373279  1.362028  1.375441  1.233121 | 1.110962  1.129436  1.231417          1.351411 |  -0.20096   -0.24584  -0.362031     -0.514052 | 0.307222  0.360995  0.491729      0.638264 | 0.893312  0.890169  0.891616      0.918976 |   837.5984 | 800.142505  634.442488  375.185257          164.589001 |  81.072929 |    0.089  0.086827  0.080421       0.065698 | 0.576333  0.518516  0.387974        0.255261 |   837.5984  764.990353  505.185713  244.714407   81.072929 |  100.0 |   737.5984 |  81.072929 |
    | 2000-01-02 03:00:00 | 9.388641  8.648876  7.010838             4.926799 | 642.641648  541.070439  356.356929  195.268715 | 144.935774  129.668288   95.861984     64.267988 | 1.372744  1.363026  1.369749  1.266631 | 1.111421  1.126218  1.210426          1.350049 | -0.202464   -0.23953  -0.344596     -0.498943 | 0.307984  0.353485  0.471643      0.624535 | 0.893514  0.890498  0.890609      0.916695 | 812.756821 | 793.742433  654.840802  403.262573          183.203248 |   91.37122 | 0.088929  0.087139  0.081477       0.067551 | 0.574322  0.526375  0.405959        0.266722 | 812.756821   776.94254  534.178532  272.136612    91.37122 |  100.0 | 712.756821 |   91.37122 |
    | 2000-01-02 03:30:00 | 9.339771  8.748741  7.215578             5.167382 | 635.583299  554.122637  376.416202  211.019907 | 143.927183  131.729326  100.087455     66.673819 | 1.371944   1.36401  1.365759  1.300239 | 1.112146  1.123679  1.193633          1.348809 | -0.204769  -0.234234  -0.329145      -0.48395 | 0.309609  0.347089   0.45392      0.610522 | 0.893674  0.890826  0.889997      0.914203 | 784.982427 | 784.055917  672.637647  430.696527           203.42124 | 104.575927 | 0.088819  0.087399  0.082383       0.069407 | 0.571249  0.533039  0.422345        0.278347 | 784.982427  785.229733  561.600956  299.827364  104.575927 |  100.0 | 684.982427 | 104.575927 |
    | 2000-01-02 04:00:00 | 9.275253  8.832229  7.406575             5.409056 | 626.340396  565.192495  395.909123  227.425284 | 142.595647  133.452393  104.029323     69.090561 |  1.37092  1.364918  1.363058  1.333483 | 1.113151  1.121702  1.180095          1.347694 | -0.207842  -0.229881  -0.315435     -0.469277 | 0.312056  0.341727  0.438231      0.596494 | 0.893794  0.891143  0.889665      0.911595 | 754.817086 |  771.37988  687.742046  457.291256          225.027197 | 120.427924 | 0.088673  0.087612  0.083163       0.071241 | 0.567173  0.538561  0.437246        0.289973 | 754.817086  789.909754   587.21517  327.628155  120.427924 |  100.0 | 654.817086 | 120.427924 |
    | 2000-01-02 04:30:00 | 9.196247  8.899716  7.583861              5.64903 | 615.138925  574.245787  414.676363  244.293149 | 140.965102  134.845209  107.688197       71.4903 | 1.369714  1.365707  1.361332  1.366012 | 1.114459  1.120195  1.169114          1.346698 | -0.211649  -0.226411  -0.303271     -0.455072 | 0.315289  0.337328  0.424318      0.582697 | 0.893874  0.891442  0.889529      0.908973 |  722.79402 | 756.030308  700.102618  482.854785          247.796828 | 138.660332 | 0.088491   0.08778  0.083838       0.073033 | 0.562152  0.542994  0.450762         0.30147 |  722.79402  791.086892   610.81638  355.357495  138.660332 |  100.0 |  622.79402 | 138.660332 |
    | 2000-01-02 05:00:00 | 9.103925  8.951641  7.747465             5.885001 | 602.212611  581.275345  432.570744  261.441225 |  139.05972  135.916837  111.064697     73.850014 | 1.368376  1.366346  1.360338  1.397562 | 1.116099  1.119089  1.160166          1.345813 | -0.216164   -0.22377   -0.29249      -0.44144 | 0.319275  0.333832  0.411976      0.569298 | 0.893915  0.891721  0.889531      0.906403 | 689.426711 | 738.333614  709.705191  507.205321          271.493949 | 158.993785 | 0.088275  0.087908  0.084423       0.074768 | 0.556241  0.546386  0.462978        0.312733 | 689.426711  788.905531  632.234326  382.821135  158.993785 |  100.0 | 589.426711 | 158.993785 |
    | 2000-01-02 05:30:00 | 8.999455  8.988495   7.89744              6.11893 | 587.797752  586.298551  449.459791  279.065688 |  136.90365  136.677458  114.159928     77.454507 | 1.366959  1.366817  1.359885  1.405101 | 1.118111  1.118332  1.152851          1.323404 |  -0.22136  -0.221911  -0.282956     -0.422885 | 0.323982  0.331184  0.401039      0.554299 | 0.893918  0.891975  0.889628      0.900486 | 655.199767 | 718.618924  716.569822  530.175789          296.292998 |  182.06216 | 0.088024  0.087998   0.08493       0.076445 | 0.549494  0.548783   0.47397        0.329151 | 655.199767  783.543697   651.33371   409.82018   182.06216 |  100.0 | 555.199767 |  182.06216 |
    | 2000-01-02 06:00:00 | 8.883997  9.010818  8.033875             6.344996 | 572.128723  589.354667  465.227348  297.102927 | 134.520798  137.138155  116.975725     82.120135 | 1.365519  1.367108  1.359821   1.39275 | 1.120539  1.117883  1.146857          1.286978 | -0.227216   -0.22079  -0.274555     -0.401142 | 0.329383  0.329336  0.391368      0.533618 | 0.893883  0.892204  0.889788      0.894516 | 620.561767 | 697.211374  720.747424  551.617007          321.520569 | 206.583936 | 0.087741  0.088052   0.08537       0.077917 | 0.541964  0.550231  0.483805         0.34949 | 620.561767  775.206705  668.013724   436.15759  206.583936 |  100.0 | 520.561767 | 206.583936 |
    | 2000-01-02 06:30:00 | 8.758689   9.01918    8.1569             6.558316 | 555.434149  590.502096  479.774469  315.090315 | 131.934643  137.310726  119.514752     86.522683 | 1.364114  1.367219  1.360027  1.383441 | 1.123437  1.117718  1.141943          1.258088 | -0.233712  -0.220372  -0.267189     -0.381879 | 0.335452  0.328246  0.382846      0.511415 | 0.893812  0.892406  0.889989      0.892748 | 585.920022 | 674.426486  722.316147  571.399767          346.485035 | 231.193259 | 0.087425  0.088072   0.08575       0.079174 | 0.533699  0.550773  0.492546        0.368131 | 585.920022  764.121072  682.206844  461.643355  231.193259 |  100.0 | 485.920022 | 231.193259 |
    | 2000-01-02 07:00:00 | 8.624639  9.014178  8.266685             6.758425 | 537.933744  589.815662   493.01971  332.817513 |  129.16808  137.207514  121.780517     90.652596 | 1.362805  1.367153  1.360407   1.37647 | 1.126864  1.117817  1.137918          1.234989 | -0.240831  -0.220622  -0.260776     -0.364825 | 0.342167  0.327875  0.375377      0.491809 | 0.893704   0.89258  0.890213      0.891538 |  551.63713 | 650.565594  721.377657  589.416037          370.945077 | 255.816907 | 0.087075   0.08806  0.086078       0.080248 | 0.524748  0.550449  0.500247        0.385141 |  551.63713   750.52879  693.877057  486.098552  255.816907 |  100.0 |  451.63713 | 255.816907 |
    | 2000-01-02 07:30:00 | 8.482923  8.996432  8.363438             6.945922 | 519.835797  587.383968  504.898975  350.177406 | 126.243302  136.841257  123.777341     94.522222 | 1.361656   1.36692  1.360887  1.371276 | 1.130889  1.118171  1.134633          1.216307 | -0.248559  -0.221512  -0.255244     -0.349669 | 0.349505  0.328189  0.368875      0.474491 |  0.89356  0.892727  0.890448      0.890739 | 518.029154 | 625.912295  718.053431  605.579454          394.793409 | 280.408323 | 0.086692  0.088017  0.086357       0.081173 | 0.515155  0.549298  0.506959        0.400671 | 518.029154  734.682057  703.017661  509.358431  280.408323 |  100.0 | 418.029154 | 280.408323 |
    | 2000-01-02 08:00:00 | 8.334577   8.96657  8.447406             7.121248 |  501.33525  583.306875  515.364995  367.066769 | 123.181702  136.224966  125.510285     98.140643 |  1.36073  1.366535  1.361407  1.367448 | 1.135588   1.11878  1.131967          1.201055 | -0.256883  -0.223016  -0.250529     -0.336163 | 0.357448  0.329159  0.363269      0.459119 | 0.893382  0.892848  0.890684      0.890233 | 485.365238 | 600.729844  712.481138  619.825219          417.919642 | 304.890152 | 0.086275  0.087945  0.086594       0.081975 | 0.504964  0.547358  0.512729        0.414849 | 485.365238   716.83851  709.648769  531.274664  304.890152 |  100.0 | 385.365238 | 304.890152 |
    | 2000-01-02 08:30:00 | 8.180597  8.925232  8.518865             7.284712 | 482.612324  577.693139  524.386541  383.384983 | 120.003804  135.371804   126.98508    101.514266 | 1.360093  1.366018  1.361923  1.364674 | 1.141047  1.119646  1.129827          1.188512 | -0.265793  -0.225111  -0.246578     -0.324107 | 0.365978  0.330758  0.358497      0.445429 |  0.89317  0.892941  0.890916      0.889938 |  453.86844 | 575.259406  704.811202  632.109546          440.210292 | 329.162914 | 0.085822  0.087843  0.086791       0.082672 | 0.494216  0.544663  0.517601         0.42778 |  453.86844  697.256979  713.814624  551.716874  329.162914 |  100.0 |  353.86844 | 329.162914 |
    | 2000-01-02 09:00:00 | 8.021931  8.873057  8.578121             7.436551 | 463.831621  570.658221  531.947444  399.036716 | 116.729214  134.295003  128.208029    104.647964 | 1.359814   1.36539  1.362401  1.362715 | 1.147358  1.120781  1.128138          1.178137 |  -0.27528  -0.227777  -0.243344     -0.313342 |  0.37508  0.332962  0.354503      0.433213 | 0.892925  0.893008  0.891139      0.889794 | 423.717566 | 549.719065  695.203571  642.408733          461.553746 | 353.113513 | 0.085332  0.087714  0.086951        0.08328 |  0.48295  0.541246  0.521614        0.439552 | 423.717566  676.193784  715.580823  570.573526  353.113513 |  100.0 | 323.717566 | 353.113513 |
    | 2000-01-02 09:30:00 | 7.859484  8.810686  8.625504             7.576962 | 445.141658  562.322292  538.045475  413.933893 | 113.376591  133.007778  129.185931    107.545809 | 1.359961  1.364676  1.362813  1.361386 | 1.154627    1.1222  1.126841          1.169515 | -0.285338  -0.230998  -0.240784     -0.303735 | 0.384736   0.33575  0.351241      0.422306 | 0.892648   0.89305  0.891348      0.889759 | 395.049792 | 524.303471  683.824739  650.717969          481.843947 | 376.621636 | 0.084804  0.087557  0.087078       0.083812 | 0.471206  0.537141  0.524806        0.450242 | 395.049792  653.899554  715.031522  587.752308  376.621636 |  100.0 | 295.049792 | 376.621636 |
    | 2000-01-02 10:00:00 | 7.694114  8.738756  8.661362             7.706125 | 426.674747  552.808425  542.691145  427.997015 | 109.963631  131.523264  129.925987    110.211519 | 1.360603  1.363906  1.363143  1.360538 | 1.162968  1.123924   1.12589          1.162328 | -0.295961  -0.234759  -0.238863     -0.295176 | 0.394934  0.339103  0.348667      0.412573 |  0.89234  0.893066  0.891542      0.889801 | 367.963873 |  499.18403  670.845049  657.049934          500.982978 | 399.564654 | 0.084236  0.087374  0.087172       0.084278 | 0.459021  0.532376  0.527212        0.459914 | 367.963873  630.616557  712.266687   603.18006  399.564654 |  100.0 | 267.963873 | 399.564654 |
    | 2000-01-02 10:30:00 | 7.526631  8.657897  8.686061             7.824217 | 408.547181  542.240992  545.906464  441.155991 |  106.50707  129.854462   130.43573    112.648731 | 1.361812   1.36311  1.363379  1.360054 | 1.172504   1.12598   1.12525          1.156323 | -0.307141  -0.239048  -0.237547     -0.287571 | 0.405659  0.343006  0.346746        0.4039 | 0.892003  0.893057  0.891719      0.889896 | 342.523746 | 474.509529  656.436275  661.433268          518.882853 | 421.821407 | 0.083625  0.087163  0.087237       0.084686 | 0.446432   0.52698  0.528864        0.468627 | 342.523746  606.576509   707.39944  616.802352  421.821407 |  100.0 | 242.523746 | 421.821407 |
    | 2000-01-02 11:00:00 | 7.357801  8.568729  8.699978             7.931417 |  390.85966  530.744239  547.723686  453.350582 | 103.022693  128.014197  130.722945    114.861168 |  1.36366  1.362322  1.363515  1.359839 | 1.183373    1.1284  1.124894          1.151302 | -0.318875  -0.243854  -0.236808      -0.28084 | 0.416895  0.347445  0.345444      0.396194 |  0.89164  0.893024  0.891878      0.890028 | 318.762374 | 450.407099  640.769491  663.910964            535.4667 | 443.275159 | 0.082969  0.086926  0.087273       0.085041 | 0.433475   0.52098  0.529793        0.476434 | 318.762374  581.998836  700.553537  628.582769  443.275159 |  100.0 | 218.762374 | 443.275159 |
    | 2000-01-02 11:30:00 | 7.188342  8.471865  8.703499             8.027916 | 373.697899   518.44105  548.184073  464.530564 |  99.525355  126.015082   130.79561    116.852725 | 1.366221  1.361577   1.36355  1.359817 | 1.195723  1.131222  1.124805          1.147107 | -0.331153  -0.249171  -0.236621     -0.274917 | 0.428627  0.352407  0.344732      0.389375 | 0.891252  0.892966  0.892019      0.890184 |  296.68568 | 426.983429  624.013225   664.53872           550.66947 | 463.815861 | 0.082266  0.086662  0.087282       0.085351 |  0.42019  0.514401  0.530028        0.483379 |  296.68568  557.089339  691.861005  638.501978  463.815861 |  100.0 |  196.68568 | 463.815861 |
    | 2000-01-02 12:00:00 | 7.018931  8.367903  8.697017             8.113911 | 357.133391  505.451885  547.336695  474.655658 |  96.029005  123.869497  130.661833    118.627522 | 1.369568  1.360912  1.363486  1.359928 | 1.209713  1.134487   1.12497           1.14361 | -0.343969  -0.254991  -0.236965     -0.269741 | 0.440837  0.357883  0.344585      0.383374 | 0.890844  0.892885   0.89214      0.890353 | 276.276442 | 404.326163  606.331884  663.383302          564.438265 | 483.341867 | 0.081514   0.08637  0.087265       0.085619 | 0.406615  0.507267  0.529596        0.489505 | 276.276442  532.039238  681.459951  646.556617  483.341867 |  100.0 | 176.276442 | 483.341867 |
    | 2000-01-02 12:30:00 | 6.850202  8.257433  8.680929             8.189614 | 341.224266  491.893877  545.237282   483.69529 |  92.546719  121.589571  130.329805    120.189908 | 1.373777  1.360368  1.363329  1.360121 | 1.225515  1.138245  1.125382           1.14071 | -0.357311  -0.261311  -0.237819     -0.265263 | 0.453508  0.363863  0.344981      0.378133 | 0.890419  0.892781  0.892243      0.890528 | 257.498082 | 382.505406  587.884438  660.520937           576.73237 | 501.761183 | 0.080711   0.08605  0.087224       0.085849 | 0.392791  0.499601  0.528521         0.49485 | 257.498082  507.024541  669.492572  652.758082  501.761183 |  100.0 | 157.498082 | 501.761183 |
    | 2000-01-02 13:00:00 | 6.682747  8.141028  8.655633             8.255247 | 326.016221  477.880072  541.947139  491.628207 |  89.090734  119.187171  129.807751    121.544469 | 1.378922  1.359987  1.363089  1.360359 | 1.243312  1.142553   1.12604          1.138323 | -0.371164  -0.268129  -0.239169     -0.261438 | 0.466615  0.370341    0.3459      0.373601 | 0.889983  0.892653  0.892326      0.890703 | 240.298231 | 361.575287  568.823342  656.035769          587.523045 | 518.992319 | 0.079853  0.085702  0.087157       0.086044 | 0.378762  0.491425  0.526828        0.499449 | 240.298231  482.205709  656.103345  657.131224  518.992319 |  100.0 | 140.298231 | 518.992319 |
    | 2000-01-02 13:30:00 |  6.51712  8.019251  8.621529             8.311043 | 311.543494  463.518814  537.532129  498.442004 |  85.672481  116.673895  129.103894    122.695999 | 1.385079  1.359813  1.362777   1.36061 |   1.2633  1.147472  1.126948          1.136383 | -0.385508  -0.275443  -0.240998     -0.258226 | 0.480134  0.377311  0.347324      0.369733 | 0.889542  0.892503  0.892391      0.890874 | 224.612057 | 341.575542  549.293683  650.018388          596.793126 | 534.964822 | 0.078941  0.085324  0.087067       0.086207 | 0.364573  0.482758  0.524539        0.503333 | 224.612057  457.727565  641.437426  659.713011  534.964822 |  100.0 | 124.612057 | 534.964822 |
    | 2000-01-02 14:00:00 | 6.353838   7.89265  8.579012             8.357243 | 297.829852  448.913246  532.061734  504.132574 |  82.302622  114.061079  128.226426    123.649487 | 1.392321  1.359893  1.362408  1.360852 | 1.285682  1.153072  1.128113          1.134836 | -0.400319  -0.283255  -0.243296     -0.255595 | 0.494034  0.384767  0.349239      0.366491 | 0.889103   0.89233  0.892436      0.891038 | 210.365288 | 322.533072  529.432524  642.564449          604.536482 | 549.619517 | 0.077972  0.084914  0.086954        0.08634 | 0.350274  0.473622  0.521674        0.506531 | 210.365288  433.719421  625.639235  660.551181  549.619517 |  100.0 | 110.365288 | 549.619517 |
    | 2000-01-02 14:30:00 | 6.193382  7.761763  8.528478             8.394097 | 284.889585  434.160916  525.608193  508.703512 |  78.991084  111.359794  127.183474    124.410082 | 1.400723  1.360277  1.361998  1.361066 | 1.310671  1.159434  1.129548           1.13364 | -0.415565  -0.291567  -0.246051     -0.253513 | 0.508281  0.392707   0.35163      0.363839 | 0.888674  0.892136  0.892464      0.891193 |  197.47692 | 304.463454  509.368429  633.773399          610.757362 | 562.908534 | 0.076947  0.084472  0.086817       0.086444 | 0.335916  0.464034  0.518254        0.509072 |  197.47692  410.295377  608.851228   659.70291  562.908534 |  100.0 |   97.47692 | 562.908534 |
    | 2000-01-02 15:00:00 |   6.0362  7.627114  8.470315              8.42186 | 272.728485  419.353487  518.245719  512.165491 |  75.747096  108.580862  125.983087    124.983067 | 1.410353  1.361016  1.361566  1.361239 | 1.338485  1.166644  1.131268           1.13276 | -0.431206  -0.300381  -0.249256     -0.251955 | 0.522834  0.401128  0.354488      0.361749 | 0.888267  0.891919  0.892472      0.891338 | 185.861607 | 287.372388  489.221151  623.747301           615.46966 | 574.795131 | 0.075866  0.083996  0.086657       0.086522 | 0.321557  0.454015  0.514295        0.510979 | 185.861607  387.554766  591.212857  657.233505  574.795131 |  100.0 |  85.861607 | 574.795131 |
    | 2000-01-02 15:30:00 | 5.880808  7.487241  8.405352             8.440705 | 261.131655  404.367843  510.105073  514.524495 |  73.808083  105.694121  124.642368    125.372001 | 1.397005  1.362184  1.361135  1.361362 | 1.345828  1.174922  1.133281          1.132174 | -0.441679  -0.309836   -0.25288     -0.250902 | 0.535651   0.41003  0.357801      0.360194 | 0.884591  0.891694  0.892459      0.891471 | 175.431719 |  271.06153  468.817164   612.66503          618.680987 | 585.259321 | 0.074738  0.083476  0.086476       0.086575 | 0.312534  0.443433  0.509846        0.512271 | 175.431719  364.899993  573.030755  653.181843  585.259321 |  100.0 |  75.431719 | 585.259321 |
    | 2000-01-02 16:00:00 | 5.728523  7.345258  8.333143             8.451124 | 250.007697  389.569153  501.158621  515.831792 |  72.285226  102.763844  123.152105     125.58702 | 1.376688  1.363825  1.360722  1.361433 |  1.34639  1.184236  1.135636          1.131853 | -0.450443  -0.319766  -0.256964     -0.250322 | 0.544501  0.419536  0.361541      0.359153 | 0.884252  0.891426  0.892437      0.891588 | 166.099094 | 255.633672  448.647066  600.489545          620.460749 | 594.265823 |  0.07362  0.082919  0.086271       0.086604 | 0.305269  0.432502  0.504865        0.512984 | 166.099094  343.367988  554.106115  647.711937  594.265823 |  100.0 |  66.099094 | 594.265823 |
    | 2000-01-02 16:30:00 | 5.583184  7.202435  8.253964             8.453345 | 239.607444  375.102519  491.472241  516.110804 |  70.831835   99.816202  121.517983    125.632863 | 1.357132  1.365979  1.360354  1.361448 | 1.346961  1.194637  1.138368          1.131785 | -0.458934  -0.330112  -0.261512     -0.250199 | 0.552292  0.429475  0.365746      0.358595 |  0.88531   0.89114  0.892397      0.891692 | 157.776495 | 241.416886  428.902191  587.310868          620.840612 | 601.818008 | 0.072544  0.082327   0.08604        0.08661 |  0.29832  0.421306  0.499359        0.513136 | 157.776495  323.313072  534.524536  640.894021  601.818008 |  100.0 |  57.776495 | 601.818008 |
    | 2000-01-02 17:00:00 | 5.445141  7.059212   8.16838             8.447533 | 229.924896  361.018284  481.147756  515.381029 |  69.451407   96.860337  119.751662    125.512922 | 1.338404  1.368696  1.360058  1.361408 | 1.347537  1.206226  1.141507          1.131963 | -0.467118  -0.340871  -0.266512     -0.250522 | 0.559862  0.439787  0.370417      0.358504 | 0.886391  0.890843  0.892338      0.891784 | 150.378786 | 228.366551   409.64501  573.267477          619.847073 | 607.933064 | 0.071512  0.081698  0.085785       0.086594 | 0.291705   0.40987  0.493355        0.512738 | 150.378786  304.670199  514.512869  632.767183  607.933064 |  100.0 |  50.378786 | 607.933064 |
    | 2000-01-02 17:30:00 | 5.314361  6.916098  8.076919             8.433893 | 220.927578  347.367561  470.281555  513.670928 |   68.14361   93.906707  117.864078    125.231411 | 1.320517  1.372023  1.359866  1.361317 | 1.348117  1.219105  1.145087          1.132385 | -0.474981  -0.352029  -0.271956     -0.251282 | 0.567171   0.45046  0.375541       0.35887 | 0.887472  0.890537  0.892258      0.891862 | 143.823868 | 216.404534  390.939395  558.489918          617.518996 | 612.626674 | 0.070526  0.081031  0.085505       0.086556 | 0.285424  0.398227  0.486877        0.511804 | 143.823868  287.363174  494.274286  623.391801  612.626674 |  100.0 |  43.823868 | 612.626674 |
    | 2000-01-02 18:00:00 |  5.19077  6.773558    7.9801             8.412655 | 212.582011  334.191763  458.966769  511.015851 |  66.907699   90.964924  115.865892    124.793084 | 1.303479  1.376005  1.359808   1.36118 | 1.348696  1.233381  1.149149           1.13305 | -0.482513  -0.363573  -0.277836     -0.252471 | 0.574207  0.461474  0.381106      0.359683 | 0.888549  0.890228   0.89216      0.891927 | 138.033385 | 205.454791  372.836508   543.10374          613.904763 | 615.915592 | 0.069586  0.080326  0.085199       0.086497 | 0.279474  0.386409  0.479947        0.510347 | 138.033385  271.318076  473.983432  612.846703  615.915592 |  100.0 |  38.033385 | 615.915592 |
    | 2000-01-02 18:30:00 | 5.074264  6.632018  7.878424             8.384069 | 204.854755  321.523297   447.29268  507.457016 |  65.742643   88.043775  113.767475    124.203129 | 1.287294  1.380689   1.35992  1.361006 | 1.349273  1.249166  1.153735          1.133962 | -0.489706  -0.375487  -0.284147     -0.254078 | 0.580961   0.47281  0.387103      0.360931 | 0.889619  0.889918  0.892044      0.891977 | 132.933235 | 195.444684  355.375866  527.228684          609.060839 | 617.820495 | 0.068693  0.079581  0.084867       0.086416 | 0.273854  0.374449  0.472587        0.508381 | 132.933235  256.463496   453.78975  601.223587  617.820495 |  100.0 |  32.933235 | 617.820495 |
    | 2000-01-02 19:00:00 | 4.964714  6.491867  7.772378             8.348405 | 197.712604  309.386603  435.344147  503.040566 |  64.647135   85.151306  111.578867    123.467086 | 1.271963  1.386119  1.360234  1.360803 | 1.349845  1.266578  1.158895          1.135128 | -0.496554  -0.387754  -0.290884     -0.256096 | 0.587427  0.484446  0.393523      0.362607 | 0.890677  0.889614   0.89191      0.892013 | 128.453911 | 186.305105  338.586869  510.977914          603.050453 | 618.367667 | 0.067846  0.078795  0.084509       0.086314 | 0.268557  0.362382  0.464817        0.505921 | 128.453911  242.730408  433.820612  588.622495  618.367667 |  100.0 |  28.453911 | 618.367667 |
    | 2000-01-02 19:30:00 | 4.861961  6.353465  7.662432             8.305944 | 191.122748  297.799091  423.201216  497.816658 |  63.619611   82.294908  109.309765    122.590767 |  1.25748  1.392339  1.360787  1.360586 | 1.350409  1.285737  1.164684          1.136557 | -0.503054  -0.400354  -0.298042     -0.258518 | 0.593599  0.496359  0.400357      0.364701 |  0.89172  0.889319  0.891759      0.892034 | 124.530701 | 177.970535   322.49016  494.457522          595.942291 | 617.589794 | 0.067045   0.07797  0.084123       0.086192 | 0.263579  0.350241  0.456659        0.502979 | 124.530701  230.052111  414.184016  575.148241  617.589794 |  100.0 |  24.530701 | 617.589794 |
    | 2000-01-02 20:00:00 | 4.765827  6.217139  7.549038             8.256978 | 185.052924  286.771977  410.938881  491.838581 |  62.658268   79.481382  106.969515    121.580185 | 1.243838  1.399392  1.361616  1.360366 | 1.350963  1.306767   1.17116          1.138261 | -0.509204  -0.413263   -0.30562     -0.261337 | 0.599474  0.508524  0.407598      0.367204 | 0.892745  0.889039  0.891593      0.892041 | 121.103761 | 170.379055  307.098832  477.766249          587.809245 | 615.526129 | 0.066291  0.077103  0.083709       0.086049 | 0.258911  0.338061  0.448131        0.499569 | 121.103761  218.364215  394.970936  560.907634  615.526129 |  100.0 |  21.103761 | 615.526129 |
    | 2000-01-02 20:30:00 | 4.676108  6.083195  7.432634             8.201804 | 179.471561  276.311028  398.626966  485.161949 |  61.761082   76.716998  104.567123    120.441491 | 1.231022  1.407318  1.362759   1.36016 | 1.351503  1.329793   1.17839          1.140257 | -0.515005  -0.426454  -0.313615     -0.264548 | 0.605049  0.520911  0.415239      0.370111 | 0.893747  0.888781  0.891412      0.892033 | 118.118088 | 163.472341  292.419487  460.995392          578.727255 | 612.222236 | 0.065581  0.076197  0.083265       0.085885 | 0.254546   0.32588  0.439252        0.495707 | 118.118088  207.604659  376.257354  546.007333  612.222236 |  100.0 |  18.118088 | 612.222236 |
    | 2000-01-02 21:00:00 | 4.592584   5.95157  7.312903             8.140892 | 174.347897  266.379444  386.254963  477.863929 |   60.92584   74.515696  102.096078    119.184376 | 1.219016  1.406387  1.364267  1.359986 | 1.352028  1.345579  1.186498          1.142558 |  -0.52046  -0.437652  -0.322078     -0.268137 | 0.610323  0.532667  0.423277      0.373413 | 0.894724  0.887179   0.89122      0.892009 | 115.523427 | 157.195613  278.417328  444.126329          568.801408 |  607.71668 | 0.064917  0.075254  0.082788       0.085702 | 0.250475  0.315904  0.429983        0.491415 | 115.523427  197.713735  357.861901  530.614286   607.71668 |  100.0 |  15.523427 |  607.71668 |
    | 2000-01-02 21:30:00 | 4.515016  5.820789  7.190281             8.074453 | 169.652096  256.719767  373.890897  469.990967 |  60.150161   73.207891   99.565368    117.813184 | 1.207801  1.389018  1.366187  1.359862 | 1.352536  1.346045  1.195573          1.145187 | -0.525575  -0.445117   -0.33101     -0.272104 | 0.615299  0.541711  0.431754      0.377095 | 0.895672  0.885065  0.891012      0.891973 | 113.274094 | 151.497568  264.915563  427.246955          558.094777 | 602.081637 | 0.064295  0.074299  0.082275       0.085497 | 0.246686  0.309672  0.420343        0.486702 | 113.274094   188.63413  339.870307  514.791485  602.081637 |  100.0 |  13.274094 | 602.081637 |
    | 2000-01-02 22:00:00 | 4.443153  5.694679   7.06701              8.00228 | 165.355335  247.566996  361.774231  461.541723 |   59.43153   71.946785   97.021275    116.323645 | 1.197352  1.372149  1.368533  1.359808 | 1.353024   1.34652  1.205563          1.148193 | -0.530355  -0.452409  -0.340275     -0.276478 |  0.61998  0.548338  0.440665      0.381159 | 0.896588  0.885926  0.890787      0.891927 | 111.328777 | 146.330288  252.279008  410.679558          546.605205 | 595.421744 | 0.063716   0.07337  0.081733        0.08527 |  0.24317  0.303652  0.410498        0.481541 | 111.328777  180.310953  322.960961  498.442673  595.421744 |  100.0 |  11.328777 | 595.421744 |
    | 2000-01-02 22:30:00 | 4.376732  5.573819  6.943312             7.924848 | 161.429889  238.944595  349.930809  452.596474 |  58.767319   70.738193   94.468363    114.725589 | 1.187642  1.355867   1.37134  1.359847 | 1.353492  1.346999  1.216549          1.151599 | -0.534811  -0.459486  -0.349875     -0.281248 |  0.62437  0.554834  0.449871       0.38563 |  0.89747  0.886821  0.890557      0.891865 | 109.650285 | 141.649121  240.517634  394.455307          534.441236 | 587.763098 | 0.063177  0.072474  0.081161        0.08502 | 0.239915  0.297871  0.400458        0.475958 | 109.650285  172.691762  307.101673  481.735452  587.763098 |  100.0 |   9.650285 | 587.763098 |
    | 2000-01-02 23:00:00 | 4.315482  5.458186  6.819594             7.842586 | 157.849183  230.831795  338.401302  443.228703 |  58.154825   69.581865   91.915031    113.027834 | 1.178645  1.340181  1.374642  1.360003 | 1.353938  1.347482  1.228602          1.155434 | -0.538952   -0.46634  -0.359796     -0.286406 | 0.628475  0.561144  0.459372      0.390495 | 0.898315  0.887712  0.890326      0.891788 | 108.205287 | 137.412562  229.581074  378.625923          521.701818 | 579.145218 | 0.062678   0.07161  0.080558       0.084748 | 0.236908   0.29233   0.39025        0.469972 | 108.205287  165.726576  292.238034  464.829257  579.145218 |  100.0 |   8.205287 | 579.145218 |
    | 2000-01-02 23:30:00 | 4.259129  5.347724  6.696232             7.755911 | 154.587846  223.206625   327.21953  433.509513 |  57.591292   68.477241   89.369052    111.239004 | 1.170328  1.325094  1.378469  1.360302 | 1.354361  1.347966  1.241794          1.159733 | -0.542789  -0.472965  -0.370025     -0.291944 | 0.632303  0.567261  0.469149      0.395743 | 0.899122  0.888599  0.890096      0.891696 | 106.964036 |  133.58211  219.419211  363.234156          508.482305 | 569.623628 | 0.062216  0.070778  0.079925       0.084452 | 0.234137  0.287027  0.379903        0.463602 | 106.964036  159.367883   278.31717  447.861349  569.623628 |  100.0 |   6.964036 | 569.623628 |
    | 2000-01-03 00:00:00 | 4.207394  5.242365  6.573568             7.665232 | 151.621743  216.047444  316.412389   423.50736 |  57.073942   67.423652   86.837465    109.367551 | 1.162659  1.310608  1.382853   1.36077 | 1.354761  1.348451    1.2562          1.164531 | -0.546336  -0.479357  -0.380545     -0.297858 | 0.635862   0.57318  0.479183      0.401366 | 0.899888  0.889478   0.88987       0.89159 | 105.900084 | 130.122122  209.984232  348.313752          494.874108 | 559.265438 | 0.061791  0.069979  0.079259       0.084133 | 0.231589  0.281959  0.369444        0.456868 | 105.900084  153.570623   265.28847  430.948996  559.265438 |  100.0 |   5.900084 | 559.265438 |
    | 2000-01-03 00:30:00 |     4.16  5.142033   6.45191             7.570948 |  148.92799  209.333016  306.000676  413.287528 |  56.599998   66.420331   84.326659    107.421699 | 1.155605  1.296723  1.387823  1.361434 | 1.355137  1.348935  1.271898          1.169867 | -0.549606  -0.485511  -0.391341      -0.30414 | 0.639162  0.578898  0.489456      0.407354 | 0.900614  0.890349  0.889652      0.891471 |  104.99001 | 126.999667  201.230646  333.890636          480.963987 |  548.14551 | 0.061399  0.069213  0.078562        0.08379 | 0.229252  0.277125  0.358899        0.449788 |  104.99001  148.292175  253.103522  414.191957   548.14551 |  100.0 |    4.99001 |  548.14551 |
    | 2000-01-03 01:00:00 | 4.116671  5.046641  6.331541             7.473446 | 146.484965  203.042537  295.999856  402.911746 |   56.16671   65.466409   81.842443    105.409413 | 1.149132  1.283439  1.393409  1.362323 | 1.355489  1.349415  1.288967          1.175785 | -0.552612  -0.491426  -0.402396     -0.310786 | 0.642215  0.584411  0.499948      0.413699 | 0.901299  0.891209  0.889447       0.89134 | 104.213141 | 124.184368  193.115226  319.983989          466.833574 | 536.343415 | 0.061039   0.06848  0.077834       0.083423 | 0.227113  0.272519  0.348297        0.442379 | 104.213141  143.492319  241.716027  397.674658  536.343415 |  100.0 |   4.213141 | 536.343415 |
    | 2000-01-03 01:30:00 | 4.077137  4.956093  6.212717             7.373097 | 144.272306  197.155653  286.420719  392.437955 |  55.771374   64.560925   79.390121    103.338386 | 1.143205  1.270752  1.399638  1.363465 | 1.355817  1.349892  1.307488          1.182333 | -0.555369  -0.497097  -0.413691     -0.317792 |  0.64503  0.589717  0.510638      0.420394 | 0.901942  0.892057  0.889257      0.891198 | 103.551298 | 121.648259  185.596945  306.607184          452.559076 | 523.941043 | 0.060709  0.067779  0.077074        0.08303 | 0.225159   0.26814  0.337662         0.43466 | 103.551298  139.133195  231.081722  381.468098  523.941043 |  100.0 |   3.551298 | 523.941043 |
    | 2000-01-03 02:00:00 | 4.041135  4.870282  6.095674             7.270261 | 142.270892  191.652482  277.269975  381.920171 |  55.411351   63.702823    76.97454    101.216025 |  1.13779  1.258657  1.406535   1.36489 | 1.356123  1.350363  1.327541          1.189561 | -0.557893  -0.502525  -0.425203     -0.325153 |  0.64762  0.594814  0.521505      0.427431 | 0.902544  0.892892  0.889088      0.891046 | 102.988541 | 119.365627  178.636914  293.768606          438.211154 | 511.020741 | 0.060408   0.06711  0.076284       0.082612 | 0.223377  0.263982  0.327024        0.426648 | 102.988541   135.17925  221.158305  365.631511  511.020741 |  100.0 |   2.988541 | 511.020741 |
    | 2000-01-03 02:30:00 | 4.008409  4.789096  5.980625             7.165021 | 140.462829  186.513638  268.548769  371.382509 |  55.084086   62.890959   74.806252     99.044055 | 1.132854  1.247148  1.410229  1.366633 | 1.356406  1.350827  1.345479          1.197547 | -0.560197   -0.50771  -0.436007     -0.332885 | 0.649998    0.5997  0.532193      0.434802 | 0.903106  0.893711   0.88839      0.890886 |  102.51094 | 117.312871  172.198314  281.472366          423.819453 | 497.685466 | 0.060134  0.066474  0.075465       0.082166 | 0.221756  0.260042  0.317286        0.418339 |  102.51094  131.597178  211.905366  350.130007  497.685466 |  100.0 |    2.51094 | 497.685466 |
    | 2000-01-03 03:00:00 | 3.978712  4.712411   5.86595             7.056942 | 138.831415  181.720245  260.036125  360.798461 |  54.787118   62.124109   73.659504     96.813488 | 1.128362  1.236217   1.39503  1.368744 | 1.356667  1.351282  1.345882           1.20642 | -0.562296  -0.512651  -0.442529     -0.341045 | 0.652174  0.604376  0.540792       0.44252 | 0.903627  0.894512  0.885763      0.890718 | 102.106358 | 115.468363  166.246329  269.531996           409.34404 | 484.065412 | 0.059884  0.065869  0.074629       0.081688 | 0.220284  0.256313  0.311825        0.409687 | 102.106358  128.355847  203.284323  334.769793  484.065412 |  100.0 |   2.106358 | 484.065412 |
    | 2000-01-03 03:30:00 | 3.951809  4.640097  5.754551             6.948306 | 137.361105  177.253963  251.892563  350.402841 |   54.51809   61.400971   72.545512     94.571431 | 1.124283  1.225855  1.380173  1.371218 | 1.356908  1.351727  1.346292          1.216086 | -0.564204  -0.517351  -0.448936     -0.349481 | 0.654163  0.608841  0.546575      0.450635 |  0.90411  0.895295  0.886493      0.890539 | 101.764251 | 113.812305  160.748077  258.231695           395.10256 | 470.026515 | 0.059657  0.065295  0.073812       0.081185 | 0.218949  0.252791  0.306511        0.400866 | 101.764251  125.426232  195.258358  320.222656  470.026515 |  100.0 |   1.764251 | 470.026515 |
    | 2000-01-03 04:00:00 | 3.927476  4.572018  5.647124             6.839244 | 136.037476  173.096998   244.15691  340.211405 |  54.274759   60.720178    71.47124     92.320571 | 1.120585  1.216049  1.365756  1.374083 | 1.357128  1.352161  1.346706          1.226612 | -0.565937  -0.521812  -0.455184     -0.358199 | 0.655976  0.613097   0.55228      0.458996 | 0.904557  0.896057   0.88726       0.89036 |  101.47548 | 112.326604  155.672538  247.610663           381.11363 | 455.693802 | 0.059451  0.064752  0.073018       0.080656 |  0.21774  0.249471  0.301378        0.391882 |  101.47548  122.781325  187.792358  306.478198  455.693802 |  100.0 |    1.47548 | 455.693802 |
    | 2000-01-03 04:30:00 |   3.9055   4.50803  5.543658             6.730089 | 134.847172  169.232123  236.815611   330.25711 |  54.055004     60.0803   70.436583     90.067799 | 1.117239  1.206787  1.351786  1.377363 |  1.35733  1.352583  1.347123           1.23805 | -0.567506  -0.526038  -0.461266     -0.367183 | 0.657625  0.617145  0.557846      0.467604 | 0.904967  0.896798  0.888025      0.890182 | 101.232151 | 110.994748  150.990485  237.635285          367.419652 |  441.19324 | 0.059264  0.064239  0.072249       0.080102 | 0.216648  0.246344  0.296427         0.38276 | 101.232151  120.396061  180.852857  293.498267   441.19324 |  100.0 |   1.232151 |  441.19324 |
    | 2000-01-03 05:00:00 | 3.885682  4.447986  5.444114             6.621155 | 133.777866  165.642688  229.853574  320.568094 |  53.856822    59.47986   69.441137     87.819581 | 1.114215  1.198056  1.338264   1.38108 | 1.357514  1.352991  1.347542           1.25045 | -0.568925  -0.530033   -0.46718     -0.376421 | 0.659122  0.620988  0.563267      0.476442 | 0.905344  0.897516  0.888785      0.890008 | 101.027458 | 109.801684  146.674407  228.270958          354.056747 | 426.633746 | 0.059096  0.063755  0.071504       0.079522 | 0.215663  0.243407  0.291655        0.373522 | 101.027458  118.247219  174.407978  281.245382  426.633746 |  100.0 |   1.027458 | 426.633746 |
    | 2000-01-03 05:30:00 | 3.867834  4.391734  5.348445             6.512724 | 132.818201  162.312632  223.255994  311.167052 |  53.678338   58.917337    68.48445     85.581749 | 1.111488   1.18984  1.325192  1.385258 | 1.357682  1.353386  1.347963          1.263866 | -0.570206  -0.533802  -0.472922     -0.385898 | 0.660479  0.624629  0.568541      0.485493 | 0.905689   0.89821  0.889538      0.889842 | 100.855551 |  108.73371  142.698439  219.484583          341.053933 | 412.108518 | 0.058944  0.063299  0.070784       0.078915 | 0.214774   0.24065  0.287062        0.364192 | 100.855551  116.313342  168.427382  269.683678  412.108518 |  100.0 |   0.855551 | 412.108518 |
    | 2000-01-03 06:00:00 |  3.85178  4.339118  5.256603             6.405051 | 131.957743  159.226492  217.008414   302.07187 |  53.517799   58.391179   67.566029     83.359568 |  1.10903  1.182122  1.312571  1.389915 | 1.357834  1.353765  1.348384          1.278352 | -0.571361   -0.53735  -0.478489       -0.3956 | 0.661707  0.628072  0.573665       0.49474 | 0.906004  0.898878  0.890285      0.889684 | 100.711411 | 107.778372  139.038288  211.244591          328.434018 | 397.696956 | 0.058807  0.062871  0.070088       0.078284 | 0.213975  0.238068  0.282645        0.354792 | 100.711411  114.574639  162.882209  258.778879  397.696956 |  100.0 |   0.711411 | 397.696956 |
    | 2000-01-03 06:30:00 | 3.837358  4.289981  5.168534             6.298368 | 131.186926  156.369408  211.096718  293.296225 |  53.373575   57.899811   66.685338     81.157802 |  1.10682  1.174886  1.300399  1.395074 | 1.357972  1.354128  1.348805          1.293963 | -0.572401  -0.540685  -0.483879     -0.405511 | 0.662816  0.631322  0.578637      0.504168 | 0.906291  0.899521  0.891024      0.889539 | 100.590745 | 106.924357  135.671163  203.520893          316.214438 | 383.466381 | 0.058684   0.06247  0.069416       0.077626 | 0.213257  0.235654  0.278402        0.345344 | 100.590745  113.012896  157.745034  248.498231  383.466381 |  100.0 |   0.590745 | 383.466381 |
    | 2000-01-03 07:00:00 | 3.824415  4.244165   5.08418             6.192883 | 130.496997   153.72713  205.507129  284.850122 |  53.244154   57.441645     65.8418     78.980773 | 1.104834  1.168113  1.288676  1.400751 | 1.358097  1.354476  1.349224          1.310754 | -0.573335  -0.543813  -0.489091     -0.415614 | 0.663816  0.634382  0.583455      0.513757 | 0.906552  0.900137  0.891754       0.88941 | 100.489885 | 106.161409  132.575701  196.284819          304.407998 | 369.473528 | 0.058573  0.062093  0.068769       0.076944 | 0.212612    0.2334  0.274332        0.335871 | 100.489885  111.611383  152.989811  238.810438  369.473528 |  100.0 |   0.489885 | 369.473528 |
    | 2000-01-03 07:30:00 | 3.812814  4.201509  5.003479             6.088787 | 129.879966  151.286017  200.226208  276.740348 |  53.128141   57.015089   65.034792     76.832406 | 1.103051  1.161784  1.277401  1.406966 | 1.358209  1.354807   1.34964          1.328781 | -0.574175  -0.546741  -0.494122     -0.425893 | 0.664716  0.637259  0.588118       0.52349 | 0.906788  0.900726  0.892474      0.889299 | 100.405708 | 105.480237    129.7319  189.509069          293.023513 | 355.765865 | 0.058474  0.061742  0.068146       0.076236 | 0.212033  0.231299  0.270433        0.326393 | 100.405708  110.354762  148.591829  229.685604  355.765865 |  100.0 |   0.405708 | 355.765865 |
    | 2000-01-03 08:00:00 | 3.802425  4.161855  4.926365             5.986253 | 129.328554  149.033038  195.240848  268.969888 |   53.02425   56.618555   64.263652     74.862526 | 1.101453  1.155882   1.26657  1.410972 | 1.358311  1.355122  1.350054           1.34546 | -0.574927  -0.549477  -0.498972     -0.435689 | 0.665526  0.639959  0.592624      0.533112 | 0.907003  0.901288  0.893183      0.888817 |  100.33556 | 104.872437  127.121046  183.167651          282.066364 | 342.330242 | 0.058385  0.061414  0.067548       0.075506 | 0.211515  0.229344    0.2667        0.317554 |  100.33556     109.229  144.527658   221.09517  342.330242 |  100.0 |    0.33556 | 342.330242 |
    | 2000-01-03 08:30:00 |  3.79313  4.125047  4.852767             5.884028 | 128.836138  146.955766  190.538269  261.369357 |  52.931302    56.25047   63.527674     73.840281 | 1.100023  1.150385   1.25618  1.397432 | 1.358402   1.35542  1.350462          1.345817 | -0.575602   -0.55203   -0.50364     -0.441496 | 0.666253  0.642487  0.596973      0.540906 | 0.907196  0.901822   0.89388      0.886378 | 100.277186 | 104.330412  124.725652  177.235837          271.393469 | 328.981881 | 0.058305  0.061109  0.066973       0.074761 | 0.211051  0.227527  0.263132        0.312687 | 100.277186  108.221281  140.775112  213.011859  328.981881 |  100.0 |   0.277186 | 328.981881 |
    | 2000-01-03 09:00:00 | 3.784822  4.090928  4.782611             5.784351 | 128.396709  145.042378  186.106023  254.058876 |  52.848218    55.90928   62.826113     72.843514 | 1.098743  1.145275  1.246226  1.384156 | 1.358484  1.355702  1.350865           1.34618 | -0.576205  -0.554406  -0.508126     -0.447215 | 0.666905   0.64485  0.601164      0.546057 |  0.90737  0.902329  0.894565      0.887022 | 100.228681 | 103.847307  122.529388  171.690109           261.22588 | 316.268584 | 0.058234  0.060824  0.066423       0.074031 | 0.210637  0.225841  0.259726        0.307934 | 100.228681  107.319919  137.313193  205.409618  316.268584 |  100.0 |   0.228681 | 316.268584 |
    | 2000-01-03 09:30:00 | 3.777401  4.059346  4.715818             5.687797 | 128.004817  143.281641  181.931989  247.072141 |  52.774012    55.59346   62.158184     71.877972 | 1.097599  1.140531  1.236704  1.371225 | 1.358558  1.355968  1.351262          1.346547 | -0.576745  -0.556615   -0.51243      -0.45281 |  0.66749  0.647056  0.605197      0.551148 | 0.907528  0.902809  0.895237      0.887697 | 100.188431 |  103.41694  120.517019  166.508111          251.600124 | 304.186134 |  0.05817  0.060561  0.065896        0.07332 | 0.210266  0.224279  0.256479        0.303323 | 100.188431   106.51428  134.122058  198.263571  304.186134 |  100.0 |   0.188431 | 304.186134 |
    | 2000-01-03 10:00:00 | 3.770779  4.030152  4.652306             5.594359 | 127.655533   141.66291  178.004372  240.399627 |  52.707786   55.301521   61.523064     70.943587 | 1.096577  1.136135  1.227608  1.358642 | 1.358624  1.356218  1.351651          1.346916 | -0.577227  -0.558665  -0.516555     -0.458278 | 0.668013   0.64911  0.609073       0.55613 | 0.907669  0.903261  0.895893       0.88837 | 100.155078 | 103.033742  118.674347  161.668602          242.492408 | 292.707027 | 0.058113  0.060316  0.065392       0.072627 | 0.209936  0.222833  0.253386        0.298854 | 100.155078  105.794701  131.182965   191.54996  292.707027 |  100.0 |   0.155078 | 292.707027 |
    | 2000-01-03 10:30:00 | 3.764872  4.003201   4.59199             5.504006 | 127.344404  140.176109  174.311703  234.030519 |  52.648723    55.03201   60.919899     70.040063 | 1.095665  1.132067  1.218931  1.346409 | 1.358683  1.356452  1.352033          1.347288 | -0.577657  -0.560564    -0.5205     -0.463615 | 0.668481   0.65102  0.612791      0.560999 | 0.907796  0.903687  0.896535      0.889037 | 100.127477 | 102.692704   116.98815   157.15141          233.877859 | 281.804081 | 0.058062   0.06009  0.064912       0.071953 | 0.209641  0.221498  0.250445        0.294527 | 100.127477  105.152414   128.47824  185.246103  281.804081 |  100.0 |   0.127477 | 281.804081 |
    | 2000-01-03 11:00:00 | 3.759609  3.978352   4.53478             5.416708 | 127.067411  138.811723  170.842838  227.954219 |  52.596086   54.783524   60.347798     69.167076 | 1.094852  1.128308  1.210664  1.334527 | 1.358736  1.356671  1.352405          1.347661 | -0.578041  -0.562321  -0.524267     -0.468819 | 0.668899  0.652794  0.616353      0.565754 |  0.90791  0.904088   0.89716        0.8897 | 100.104666 | 102.389323  115.446127  152.937386          225.732619 | 271.451239 | 0.058017  0.059881  0.064454       0.071298 | 0.209378  0.220266  0.247652         0.29034 | 100.104666  104.579478  125.991229  179.330342  271.451239 |  100.0 |   0.104666 | 271.451239 |
    | 2000-01-03 11:30:00 |  3.75492   3.95547  4.480585             5.332429 | 126.820934  137.560784  167.586963  222.160401 |  52.549203   54.554704   59.805846     68.324286 | 1.094128  1.124839  1.202801  1.322996 | 1.358783  1.356875  1.352768          1.348036 | -0.578383  -0.563944   -0.52786     -0.473889 | 0.669272  0.654438  0.619761      0.570392 | 0.908011  0.904463  0.897769      0.890356 | 100.085838 | 102.119555  114.036842  149.008365          218.033875 | 261.623561 | 0.057976  0.059687  0.064018       0.070663 | 0.209144  0.219131  0.245002        0.286292 | 100.085838   104.06871  123.706258  173.782001  261.623561 |  100.0 |   0.085838 | 261.623561 |
    | 2000-01-03 12:00:00 | 3.750747  3.934425   4.42931             5.251133 | 126.601716  136.414851  164.533587  216.639001 |  52.507469   54.344246     59.2931     67.511333 | 1.093483  1.121642  1.195332  1.311817 | 1.358825  1.357065  1.353121           1.34841 | -0.578687  -0.565441  -0.531281     -0.478823 | 0.669605   0.65596  0.623016      0.574912 | 0.908103  0.904814   0.89836      0.891007 | 100.070317 | 101.879771  112.749674  145.347119          210.759818 | 252.297175 |  0.05794  0.059509  0.063604       0.070046 | 0.208935  0.218085  0.242492        0.282381 | 100.070317  103.613623  121.608596  168.581344  252.297175 |  100.0 |   0.070317 | 252.297175 |
    | 2000-01-03 12:30:00 | 3.747034   3.91509   4.38086             5.172784 | 126.406829  135.365997  161.672547  211.380208 |   52.47034     54.1509   58.808596     66.727837 | 1.092908    1.1187  1.188247  1.300988 | 1.358863  1.357242  1.353463          1.348784 | -0.578958   -0.56682  -0.534533     -0.483618 | 0.669901  0.657365   0.62612      0.579313 | 0.908184  0.905142  0.898933       0.89165 | 100.057539 | 101.666719  111.574769  141.937323          203.889602 | 243.449238 | 0.057908  0.059346  0.063211       0.069449 |  0.20875  0.217125  0.240117        0.278607 | 100.057539  103.208362  119.684411  163.709534  243.449238 |  100.0 |   0.057539 | 243.449238 |
    | 2000-01-03 13:00:00 | 3.743732  3.897347  4.335135              5.09734 | 126.233645  134.406786  158.994009   206.37446 |  52.437324   53.973472   58.351351     65.973398 | 1.092398  1.115996  1.181536  1.290509 | 1.358896  1.357406  1.353794          1.349158 |   -0.5792  -0.568089   -0.53762     -0.488275 | 0.670166  0.658663  0.629076      0.583593 | 0.908257  0.905447  0.899487      0.892286 | 100.047032 | 101.477488  110.502989  138.763511          197.403303 | 235.057889 |  0.05788  0.059195  0.062839        0.06887 | 0.208585  0.216243  0.237873        0.274968 | 100.047032  102.847654  117.920733   159.14859  235.057889 |  100.0 |   0.047032 | 235.057889 |
    | 2000-01-03 13:30:00 | 3.740798  3.881083  4.292037             5.024759 | 126.079807  133.530261  156.488463   201.61243 |  52.407978   53.810828    57.92037     65.247595 | 1.091944  1.113513  1.175189   1.28038 | 1.358926  1.357557  1.354113          1.349529 | -0.579414  -0.569255  -0.540545     -0.492791 | 0.670401  0.659858  0.631886      0.587752 | 0.908322   0.90573  0.900022      0.892914 | 100.038402 | 101.309469  109.525873  135.811039          191.281881 |  227.10221 | 0.057854  0.059057  0.062486       0.068311 | 0.208438  0.215434  0.235755        0.271461 | 100.038402  102.526754  116.305414  154.881361   227.10221 |  100.0 |   0.038402 |  227.10221 |
    | 2000-01-03 14:00:00 | 3.738191  3.866189  4.251465             4.954998 | 125.943203  132.729919  154.146728  197.085027 |  52.381906   53.661889   57.514647     64.549985 |  1.09154  1.111236  1.169193  1.270598 | 1.358952  1.357698   1.35442          1.349899 | -0.579605  -0.570324  -0.543314     -0.497166 |  0.67061  0.660958  0.634554      0.591789 |  0.90838  0.905993  0.900538      0.893533 | 100.031322 | 101.160332  108.635586  133.066051           185.50714 | 219.562183 | 0.057832   0.05893  0.062153        0.06777 | 0.208308  0.214692  0.233759        0.268087 | 100.031322  102.241397  114.827093  150.891479  219.562183 |  100.0 |   0.031322 | 219.562183 |
    | 2000-01-03 14:30:00 | 3.735875  3.852564  4.213317              4.88801 | 125.821945  131.999691  151.959949  192.783384 |  52.358752   53.525637   57.133169     63.880104 | 1.091182   1.10915  1.163538  1.261161 | 1.358976  1.357827  1.354715          1.350265 | -0.579774  -0.571304  -0.545929       -0.5014 | 0.670796  0.661969  0.637084      0.595702 | 0.908432  0.906237  0.901034      0.894143 | 100.025521 | 101.027995  107.824887  130.515443          180.061697 | 212.418655 | 0.057812  0.058814  0.061839       0.067249 | 0.208192  0.214014  0.231881        0.264842 | 100.025521  101.987753  113.475154  147.163339  212.418655 |  100.0 |   0.025521 | 212.418655 |
    | 2000-01-03 15:00:00 |  3.73382  3.840111  4.177492             4.823747 | 125.714343  131.333927  149.919597  188.698851 |  52.338197    53.40111   56.774924     63.237465 | 1.090863  1.107242  1.158212  1.252068 | 1.358997  1.357946  1.354997          1.350627 | -0.579925  -0.572202  -0.548397     -0.505491 | 0.670962  0.662897  0.639478      0.599492 | 0.908478  0.906462  0.901509      0.894742 | 100.020773 | 100.910597  107.087085  128.146828          174.928941 | 205.653294 | 0.057794  0.058707  0.061543       0.066746 | 0.208089  0.213394  0.230115        0.261724 | 100.020773  101.762387  112.239693  143.682056  205.653294 |  100.0 |   0.020773 | 205.653294 |
    | 2000-01-03 15:30:00 | 3.731996   3.82874   4.14389             4.762156 | 125.618886   130.72737  148.017465  184.822997 |  52.319955   53.287404   56.438899     62.621561 | 1.090581  1.105498  1.153201  1.243315 | 1.359015  1.358055  1.355267          1.350985 | -0.580058  -0.573023  -0.550722     -0.509441 | 0.671109  0.663748   0.64174      0.603159 | 0.908518   0.90667  0.901965      0.895332 | 100.016892 |  100.80648  106.416004  125.948504          170.093004 | 199.248556 | 0.057778   0.05861  0.061265       0.066262 | 0.207998  0.212827  0.228457        0.258732 | 100.016892   101.56222  111.111481  140.433447  199.248556 |  100.0 |   0.016892 | 199.248556 |
    | 2000-01-03 16:00:00 | 3.730377  3.818367  4.112409             4.703186 | 125.534228  130.175137  146.245671  181.147596 |  52.303772    53.18367    56.12409     62.031862 |  1.09033  1.103905  1.148494  1.234898 | 1.359032  1.358156  1.355524          1.351339 | -0.580176  -0.573773  -0.552909     -0.513248 | 0.671239  0.664527  0.643876      0.606702 | 0.908555  0.906862  0.902401       0.89591 | 100.013722 | 100.714163  105.805947  123.909423          165.538732 | 193.187647 | 0.057764  0.058521  0.061003       0.065796 | 0.207917   0.21231  0.226902        0.255864 | 100.013722  101.384497  110.081928  137.403997  193.187647 |  100.0 |   0.013722 | 193.187647 |
    | 2000-01-03 16:30:00 | 3.728942  3.808911   4.08295             4.646782 | 125.459166  129.672702  144.596651  177.664631 |  52.289419   53.089114     55.8295     61.467818 | 1.090108  1.102451  1.144078  1.226815 | 1.359047  1.358248  1.355769          1.351686 | -0.580282  -0.574457  -0.554963     -0.516915 | 0.671355   0.66524  0.645888      0.610121 | 0.908587  0.907038  0.902817      0.896477 | 100.011136 | 100.632328  105.251668   122.01916          161.251647 |  187.45449 | 0.057752   0.05844  0.060758       0.065349 | 0.207845  0.211839  0.225446        0.253117 | 100.011136  101.226753  109.143047  134.580834   187.45449 |  100.0 |   0.011136 |  187.45449 |
    | 2000-01-03 17:00:00 | 3.727669  3.800299  4.055415             4.592886 | 125.392628  129.215872  143.063159  174.366283 |  52.276692   53.002995   55.554146     60.928857 | 1.089911  1.101126   1.13994   1.21906 |  1.35906  1.358332  1.356001          1.352027 | -0.580375  -0.575081  -0.556891     -0.520441 | 0.671458  0.665891  0.647782      0.613417 | 0.908616    0.9072  0.903212      0.897032 | 100.009028 |   100.5598  104.748335  120.267885          157.217927 | 182.033692 | 0.057741  0.058367  0.060528       0.064919 | 0.207782  0.211409  0.224084        0.250489 | 100.009028  101.086784  108.287427  131.951704  182.033692 |  100.0 |   0.009028 | 182.033692 |
    | 2000-01-03 17:30:00 | 3.726541  3.792462  4.029706             4.541439 | 125.333658   128.80077  141.638261  171.244936 |  52.265411    52.92462   55.297063     60.414392 | 1.089736   1.09992  1.136068  1.211628 | 1.359071  1.358409  1.356222          1.352362 | -0.580457   -0.57565  -0.558696     -0.523828 | 0.671549  0.666485  0.649561       0.61659 | 0.908641  0.907349  0.903589      0.897574 | 100.007313 | 100.495533  104.291503  118.646332          153.424372 | 176.910512 | 0.057731  0.058299  0.060312       0.064507 | 0.207725  0.211018  0.222811        0.247977 | 100.007313  100.962623  107.508191  129.504952  176.910512 |  100.0 |   0.007313 | 176.910512 |
    | 2000-01-03 18:00:00 | 3.725541  3.785335   4.00573             4.492381 | 125.281408  128.423815  140.315332  168.293166 |  52.255413   52.853347   55.057303     59.923813 | 1.089581  1.098822  1.132449  1.204516 | 1.359081  1.358479   1.35643          1.352689 | -0.580531  -0.576168  -0.560386     -0.527076 |  0.67163  0.667028  0.651231      0.619642 | 0.908664  0.907485  0.903946      0.898102 | 100.005918 | 100.438598  103.877087  117.145778          149.858382 | 172.070833 | 0.057723  0.058238  0.060111       0.064113 | 0.207675  0.210662  0.221623        0.245579 | 100.005918  100.852513  106.798972  127.229494  172.070833 |  100.0 |   0.005918 | 172.070833 |
    | 2000-01-03 18:30:00 | 3.724655  3.778858  3.983394              4.44565 | 125.235119  128.081706  139.088052  165.503744 |  52.246554   52.788579   54.833941     59.456496 | 1.089444  1.097823  1.129071  1.197715 |  1.35909  1.358544  1.356626          1.353008 | -0.580596  -0.576639  -0.561964     -0.530189 | 0.671702  0.667522  0.652796      0.622573 | 0.908684   0.90761  0.904284      0.898618 | 100.004785 | 100.388165  103.501338  115.758009          146.507928 |  167.50113 | 0.057715  0.058182  0.059923       0.063736 | 0.207631  0.210339  0.220516        0.243292 | 100.004785  100.754889  106.153879  125.114796   167.50113 |  100.0 |   0.004785 |  167.50113 |
    | 2000-01-03 19:00:00 | 3.723871  3.772976  3.962607              4.40118 | 125.194119  127.771397    137.9504  162.869634 |  52.238706   52.729763   54.626074     59.011801 | 1.089322  1.096916  1.125922  1.191222 | 1.359098  1.358602  1.356811          1.353319 | -0.580653  -0.577067  -0.563438     -0.533167 | 0.671765  0.667971  0.654261      0.625385 | 0.908702  0.907724  0.904603      0.899119 | 100.003865 |   100.3435  103.160813  114.475302          143.361534 | 163.188445 | 0.057708  0.058132  0.059748       0.063376 | 0.207592  0.210045  0.219485        0.241113 | 100.003865  100.668354   105.56747  123.150858  163.188445 |  100.0 |   0.003865 | 163.188445 |
    | 2000-01-03 19:30:00 | 3.723175  3.767639  3.943283             4.358908 | 125.157809  127.490091  136.896647  160.383989 |  52.231754   52.676388    54.43283     58.589076 | 1.089214  1.096092  1.122988  1.185028 | 1.359106  1.358655  1.356985          1.353621 | -0.580704  -0.577455  -0.564811     -0.536013 | 0.671821  0.668381   0.65563       0.62808 | 0.908718  0.907828  0.904905      0.899606 | 100.003119 | 100.303948  102.852357  113.290396          140.408248 | 159.120363 | 0.057702  0.058086  0.059584       0.063032 | 0.207557  0.209779  0.218525         0.23904 | 100.003119  100.591665  105.034719   121.32819  159.120363 |  100.0 |   0.003119 | 159.120363 |
    | 2000-01-03 20:00:00 |  3.72256  3.762798  3.925336             4.318765 | 125.125658  127.235213   135.92135  158.040152 |  52.225599    52.62798   54.253359     58.187654 | 1.089119  1.095345   1.12026  1.179128 | 1.359112  1.358704  1.357148          1.353915 | -0.580749  -0.577808  -0.566089     -0.538729 | 0.671871  0.668753  0.656908      0.630658 | 0.908732  0.907923  0.905189      0.900079 | 100.002515 |  100.26893  102.573081  112.196472          137.637624 | 155.284986 | 0.057697  0.058044  0.059433       0.062705 | 0.207526  0.209537  0.217634        0.237069 | 100.002515  100.523716  104.550993  119.637795  155.284986 |  100.0 |   0.002515 | 155.284986 |
    | 2000-01-03 20:30:00 | 3.722015   3.75841  3.908685             4.280686 | 125.097193  127.004399  135.019346  155.831656 |  52.220148   52.584104   54.086846      57.80686 | 1.089034  1.094667  1.117724  1.173514 | 1.359118  1.358748  1.357301          1.354198 | -0.580789  -0.578128  -0.567278     -0.541319 | 0.671916   0.66909  0.658099      0.633122 | 0.908744   0.90801  0.905456      0.900536 | 100.002026 | 100.237929  102.320337  111.187131          135.039701 | 151.670908 | 0.057692  0.058006  0.059291       0.062393 | 0.207499  0.209318  0.216806        0.235197 | 100.002026  100.463522  104.112027  118.071148  151.670908 |  100.0 |   0.002026 | 151.670908 |
    | 2000-01-03 21:00:00 | 3.721532  3.754436   3.89325             4.244601 | 125.071994  126.795484  134.185744  153.752218 |  52.215322   52.544359   53.932503     57.446013 | 1.088959  1.094053  1.115371  1.168177 | 1.359122  1.358788  1.357444          1.354473 | -0.580824  -0.578418  -0.568382     -0.543783 | 0.671955  0.669397  0.659209      0.635474 | 0.908755  0.908089  0.905706      0.900978 | 100.001631 | 100.210488  102.091707  110.256369           132.60498 | 148.267199 | 0.057688  0.057972   0.05916       0.062097 | 0.207475  0.209119  0.216039        0.233422 | 100.001631  100.410206  103.713893  116.620183  148.267199 |  100.0 |   0.001631 | 148.267199 |
    | 2000-01-03 21:30:00 | 3.721105  3.750838  3.878957             4.210442 |  125.04969  126.606481  133.415916  151.795747 |  52.211051   52.508377   53.789575     57.104421 | 1.088893  1.093497  1.113188  1.163111 | 1.359127  1.358824  1.357577          1.354738 | -0.580856  -0.578681  -0.569407     -0.546127 | 0.671989  0.669675  0.660241      0.637716 | 0.908765  0.908161  0.905941      0.901405 | 100.001312 |   100.1862  101.884979   109.39856          130.324412 | 145.063381 | 0.057684  0.057941  0.059039       0.061816 | 0.207453   0.20894  0.215328        0.231739 | 100.001312   100.36299  103.352985  115.277267  145.063381 |  100.0 |   0.001312 | 145.063381 |
    | 2000-01-03 22:00:00 | 3.720727  3.747582  3.865734             4.178139 |  125.02995  126.435574  132.705491  149.956336 |   52.20727   52.475818   53.657337     56.781394 | 1.088835  1.092993  1.111166  1.158308 | 1.359131  1.358857  1.357702          1.354992 | -0.580884  -0.578918  -0.570357     -0.548352 |  0.67202  0.669927    0.6612      0.639851 | 0.908774  0.908226  0.906161      0.901817 | 100.001054 | 100.164706  101.698134  108.608437          128.189372 | 142.049412 | 0.057681  0.057913  0.058926       0.061549 | 0.207434  0.208777   0.21467        0.230147 | 100.001054  100.321182  103.025988  114.035193  142.049412 |  100.0 |   0.001054 | 142.049412 |
    | 2000-01-03 22:30:00 | 3.720392  3.744637   3.85351             4.147624 | 125.012481  126.281099  132.050344  148.228264 |  52.203923   52.446372     53.5351     56.476236 | 1.088783  1.092538  1.109295  1.153758 | 1.359134  1.358887  1.357818          1.355237 | -0.580908  -0.579133  -0.571236     -0.550463 | 0.672047  0.670156   0.66209      0.641881 | 0.908781  0.908285  0.906367      0.902213 | 100.000846 | 100.145686  101.529328  107.881069          126.191648 | 139.215665 | 0.057678  0.057888  0.058822       0.061296 | 0.207418   0.20863  0.214061        0.228641 | 100.000846  100.284168  102.729864  112.887155  139.215665 |  100.0 |   0.000846 | 139.215665 |
    | 2000-01-03 23:00:00 | 3.720096  3.741976   3.84222             4.118825 | 124.997024  126.141538   131.44659  146.605996 |  52.200962   52.419755   53.422203     56.188254 | 1.088737  1.092126  1.107565  1.149454 | 1.359137  1.358914  1.357926          1.355472 |  -0.58093  -0.579328   -0.57205     -0.552462 | 0.672071  0.670362  0.662915       0.64381 | 0.908788  0.908339  0.906559      0.902593 | 100.000679 | 100.128856   101.37688  107.211845          124.323421 | 136.552915 | 0.057676  0.057865  0.058725       0.061057 | 0.207403  0.208497  0.213499        0.227219 | 100.000679  100.251402  102.461827  111.826736  136.552915 |  100.0 |   0.000679 | 136.552915 |
    | 2000-01-03 23:30:00 | 3.719834  3.739571  3.831802             4.091676 | 124.983347  126.015504  130.890571  145.084184 |  52.198342   52.395707   53.318021     55.916757 | 1.088696  1.091754  1.105968  1.145387 |  1.35914  1.358938  1.358026          1.355696 | -0.580949  -0.579504  -0.572802     -0.554354 | 0.672093  0.670549  0.663679      0.645639 | 0.908794  0.908388  0.906738      0.902958 | 100.000544 | 100.113965  101.239259  106.596458           122.57725 | 134.052322 | 0.057673  0.057844  0.058636       0.060831 |  0.20739  0.208377   0.21298        0.225878 | 100.000544  100.222399  102.219326  110.847889  134.052322 |  100.0 |   0.000544 | 134.052322 |
    | 2000-01-04 00:00:00 | 3.719602  3.737399  3.822196             4.066106 | 124.971246  125.901735  130.378851  143.657662 |  52.196024   52.373989    53.22196     55.661057 |  1.08866  1.091418  1.104493  1.141548 | 1.359142   1.35896  1.358118          1.355911 | -0.580966  -0.579663  -0.573496     -0.556141 | 0.672111  0.670718  0.664386      0.647372 |   0.9088  0.908432  0.906904      0.903308 | 100.000436 | 100.100792  101.115068  106.030886          120.946057 | 131.705417 | 0.057671  0.057825  0.058554       0.060617 | 0.207378  0.208268  0.212501        0.224613 | 100.000436  100.196732  102.000029  109.944925  131.705417 |  100.0 |   0.000436 | 131.705417 |
    | 2000-01-04 00:30:00 | 3.719397  3.735438  3.813346             4.042047 | 124.960541  125.799076  129.908206  142.321449 |  52.193973   52.354384   53.133456     55.420474 | 1.088628  1.091114  1.103133  1.137928 | 1.359144   1.35898  1.358204          1.356116 | -0.580981  -0.579806  -0.574136     -0.557829 | 0.672128  0.670871  0.665039      0.649013 | 0.908804  0.908472  0.907059      0.903642 | 100.000349 | 100.089137   101.00304  105.511379          119.423114 | 129.504087 |  0.05767  0.057808  0.058478       0.060416 | 0.207368   0.20817   0.21206        0.223422 | 100.000349  100.174017  101.801802  109.112492  129.504087 |  100.0 |   0.000349 | 129.504087 |
    | 2000-01-04 01:00:00 | 3.719216  3.733669  3.805198             4.019434 | 124.951071  125.706479  129.475613  141.070744 |  52.192159   52.336694   53.051977     55.194337 |   1.0886   1.09084   1.10188  1.134519 | 1.359146  1.358998  1.358284           1.35631 | -0.580994  -0.579936  -0.574726     -0.559419 | 0.672143  0.671009  0.665643      0.650563 | 0.908808  0.908509  0.907203      0.903961 | 100.000279 | 100.078828  100.902018  105.034439          118.002024 | 127.440567 | 0.057668  0.057793  0.058409       0.060226 | 0.207359  0.208082  0.211653        0.222302 | 100.000279  100.153917  101.622698  108.345565  127.440567 |  100.0 |   0.000279 | 127.440567 |
    | 2000-01-04 01:30:00 | 3.719055  3.732074  3.797702             3.998198 | 124.942694  125.622989   129.07824  139.900929 |  52.190554   52.320739   52.977021     54.981984 | 1.088575  1.090593  1.100727  1.131311 | 1.359148  1.359015  1.358357          1.356496 | -0.581006  -0.580052   -0.57527     -0.560917 | 0.672156  0.671134    0.6662      0.652027 | 0.908812  0.908541  0.907336      0.904265 | 100.000223 | 100.069709  100.810953  104.596809          116.676716 | 125.507422 | 0.057667  0.057779  0.058344       0.060048 | 0.207351  0.208002  0.211279         0.22125 | 100.000223  100.136133  101.460939  107.639429  125.507422 |  100.0 |   0.000223 | 125.507422 |
    | 2000-01-04 02:00:00 | 3.718913  3.730635  3.790812             3.978276 | 124.935285  125.547736  128.713443  138.807566 |  52.189134   52.306355   52.908117     54.782765 | 1.088553   1.09037  1.099666  1.128296 | 1.359149  1.359029  1.358425          1.356671 | -0.581016  -0.580158   -0.57577     -0.562327 | 0.672167  0.671247  0.666713      0.653407 | 0.908815  0.908571  0.907459      0.904554 | 100.000178 | 100.061644  100.728891  104.195458          115.441422 |  123.69754 | 0.057665  0.057767  0.058285        0.05988 | 0.207344   0.20793  0.210936        0.220262 | 100.000178  100.120399  101.314904  106.989662   123.69754 |  100.0 |   0.000178 |  123.69754 |
    | 2000-01-04 02:30:00 | 3.718788  3.729339  3.784482             3.959604 | 124.928731  125.479933  128.378748  137.786392 |  52.187878    52.29339    52.84482     54.596042 | 1.088534  1.090169   1.09869  1.125466 | 1.359151  1.359043  1.358488          1.356838 | -0.581026  -0.580253   -0.57623     -0.563651 | 0.672178  0.671348  0.667186      0.654707 | 0.908818  0.908598  0.907574       0.90483 | 100.000142 |  100.05451  100.654966  103.827566          114.290673 | 122.004119 | 0.057664  0.057756  0.058231       0.059722 | 0.207337  0.207865   0.21062        0.219336 | 100.000142  100.106479  101.183116  106.392126  122.004119 |  100.0 |   0.000142 | 122.004119 |
    | 2000-01-04 03:00:00 | 3.718677  3.728171  3.778671             3.942119 | 124.922935  125.418862  128.071849   136.83332 |  52.186768    52.28171   52.786712     54.421194 | 1.088517  1.089988  1.097794  1.122812 | 1.359152  1.359054  1.358545          1.356995 | -0.581034  -0.580338  -0.576652     -0.564894 | 0.672187   0.67144  0.667621      0.655931 | 0.908821  0.908622  0.907679      0.905091 | 100.000113 | 100.048201  100.588394  103.490516          113.219281 |  120.42066 | 0.057663  0.057745  0.058181       0.059575 | 0.207332  0.207807   0.21033        0.218468 | 100.000113  100.094165  101.064231  105.842947   120.42066 |  100.0 |   0.000113 |  120.42066 |
    | 2000-01-04 03:30:00 | 3.718579  3.727119   3.77334             3.925761 | 124.917809  125.363873  127.790598  135.944436 |  52.185785   52.271191   52.733405     54.257614 | 1.088501  1.089825  1.096972  1.120325 | 1.359153  1.359065  1.358598          1.357144 | -0.581041  -0.580415   -0.57704     -0.566059 | 0.672195  0.671522  0.668021      0.657081 | 0.908823  0.908644  0.907777      0.905338 |  100.00009 | 100.042621   100.52846  103.181874          112.222332 | 118.940953 | 0.057662  0.057736  0.058135       0.059436 | 0.207327  0.207754  0.210064        0.217655 |  100.00009  100.083273  100.957024  105.338505  118.940953 |  100.0 |    0.00009 | 118.940953 |
    | 2000-01-04 04:00:00 | 3.718492  3.726172  3.768453             3.910471 | 124.913276  125.314375  127.532994  135.115998 |  52.184917   52.261721   52.684532     54.104713 | 1.088488  1.089679  1.096218  1.117996 | 1.359154  1.359075  1.358647          1.357284 | -0.581047  -0.580484  -0.577396      -0.56715 | 0.672202  0.671597  0.668388       0.65816 | 0.908825  0.908664  0.907867      0.905572 | 100.000072 | 100.037686   100.47452  102.899384          111.295171 | 117.559071 | 0.057662  0.057728  0.058093       0.059306 | 0.207322  0.207707   0.20982        0.216895 | 100.000072  100.073639  100.860382  104.875423  117.559071 |  100.0 |   0.000072 | 117.559071 |
    | 2000-01-04 04:30:00 | 3.718415   3.72532  3.763975             3.896192 | 124.909267  125.269835  127.297175  134.344427 |  52.184148   52.253198   52.639752     53.961918 | 1.088476  1.089547  1.095527  1.115819 | 1.359155  1.359084  1.358692          1.357417 | -0.581053  -0.580547  -0.577722     -0.568172 | 0.672208  0.671664  0.668726      0.659172 | 0.908827  0.908681   0.90795      0.905793 | 100.000057 | 100.033322  100.425988  102.640955          110.433393 | 116.269359 | 0.057661  0.057721  0.058054       0.059185 | 0.207319  0.207664  0.209596        0.216185 | 100.000057  100.065118  100.773295  104.450547  116.269359 |  100.0 |   0.000057 | 116.269359 |
    | 2000-01-04 05:00:00 | 3.718347  3.724553  3.759875             3.882867 | 124.905721  125.229767  127.081413  133.626312 |  52.183469   52.245529   52.598748     53.828675 | 1.088465  1.089428  1.094893  1.113785 | 1.359155  1.359091  1.358733          1.357541 | -0.581058  -0.580603  -0.578021     -0.569127 | 0.672213  0.671724  0.669035      0.660121 | 0.908828  0.908698  0.908026      0.906002 | 100.000046 | 100.029463  100.382334  102.404649          109.632831 | 115.066427 |  0.05766  0.057714  0.058019       0.059072 | 0.207315  0.207626  0.209391        0.215522 | 100.000046  100.057581  100.694845  104.060941  115.066427 |  100.0 |   0.000046 | 115.066427 |
    | 2000-01-04 05:30:00 | 3.718287  3.723863  3.756122             3.870445 | 124.902586  125.193733  126.884104  132.958399 |  52.182868   52.238632   52.561222      53.70445 | 1.088456  1.089321  1.094314  1.111887 | 1.359156  1.359099  1.358771          1.357657 | -0.581062  -0.580654  -0.578295     -0.570018 | 0.672218  0.671778  0.669319      0.661009 |  0.90883  0.908712  0.908096      0.906199 | 100.000036 | 100.026051   100.34308  102.188673          108.889549 | 113.945139 |  0.05766  0.057708  0.057987       0.058966 | 0.207312  0.207591  0.209204        0.214904 | 100.000036  100.050916  100.624197   103.70387  113.945139 |  100.0 |   0.000036 | 113.945139 |
    | 2000-01-04 06:00:00 | 3.718234  3.723243   3.75269             3.858873 | 124.899814  125.161337  126.703761  132.337593 |  52.182337    52.23243     52.5269     53.588729 | 1.088448  1.089225  1.093783  1.110116 | 1.359156  1.359105  1.358805          1.357767 | -0.581066  -0.580699  -0.578545      -0.57085 | 0.672223  0.671827  0.669579      0.661839 | 0.908831  0.908725  0.908161      0.906384 | 100.000029 | 100.023033  100.307791  101.991368          108.199828 | 112.900609 | 0.057659  0.057703  0.057957       0.058868 |  0.20731   0.20756  0.209032        0.214328 | 100.000029  100.045021  100.560596  103.376791  112.900609 |  100.0 |   0.000029 | 112.900609 |
    | 2000-01-04 06:30:00 | 3.718187  3.722685  3.749552             3.848102 | 124.897363  125.132217  126.539005  131.760949 |  52.181867   52.226855   52.495525     53.481015 | 1.088441  1.089138  1.093298  1.108467 | 1.359157  1.359111  1.358837          1.357869 |  -0.58107   -0.58074  -0.578774     -0.571626 | 0.672227  0.671871  0.669818      0.662614 | 0.908832  0.908737   0.90822      0.906558 | 100.000023 | 100.020365  100.276074    101.8112          107.560159 | 111.928187 | 0.057659  0.057698   0.05793       0.058776 | 0.207307  0.207532  0.208875        0.213792 | 100.000023  100.039808  100.503356  103.077341  111.928187 |  100.0 |   0.000023 | 111.928187 |
    | 2000-01-04 07:00:00 | 3.718145  3.722184  3.746686             3.838083 | 124.895195  125.106051  126.388562  131.225674 |  52.181452   52.221844   52.466858     53.380834 | 1.088434  1.089061  1.092855  1.106931 | 1.359157  1.359116  1.358866          1.357965 | -0.581073  -0.580777  -0.578984     -0.572348 |  0.67223  0.671911  0.670036      0.663337 | 0.908833  0.908747  0.908274      0.906722 | 100.000018 | 100.018006  100.247576  101.646753          106.967232 | 111.023456 | 0.057659  0.057694  0.057905        0.05869 | 0.207305  0.207507  0.208732        0.213293 | 100.000018  100.035198  100.451857  102.803325  111.023456 |  100.0 |   0.000018 | 111.023456 |
    | 2000-01-04 07:30:00 | 3.718108  3.721734  3.744068             3.828773 | 124.893279  125.082544  126.251252  130.729115 |  52.181084   52.217343   52.440681     53.287731 | 1.088429  1.088991   1.09245  1.105503 | 1.359158   1.35912  1.358893          1.358055 | -0.581076   -0.58081  -0.579175      -0.57302 | 0.672233  0.671946  0.670235      0.664012 | 0.908834  0.908757  0.908324      0.906875 | 100.000014 |  100.01592  100.221977   101.49672          106.417927 | 110.182222 | 0.057658   0.05769  0.057883        0.05861 | 0.207303  0.207485  0.208601        0.212829 | 100.000014  100.031122  100.405536  102.552708  110.182222 |  100.0 |   0.000014 | 110.182222 |
    | 2000-01-04 08:00:00 | 3.718076   3.72133  3.741679             3.820127 | 124.891584  125.061432  126.125986  130.268762 |   52.18076   52.213299   52.416789     53.201271 | 1.088423  1.088928   1.09208  1.104175 | 1.359158  1.359125  1.358917          1.358139 | -0.581078  -0.580839   -0.57935     -0.573645 | 0.672236  0.671978  0.670417       0.66464 | 0.908835  0.908765   0.90837      0.907019 | 100.000011 | 100.014075  100.198986  101.359896          105.909309 | 109.400506 | 0.057658  0.057686  0.057862       0.058536 | 0.207302  0.207464  0.208482        0.212398 | 100.000011  100.027517  100.363884  102.323605  109.400506 |  100.0 |   0.000011 | 109.400506 |
    | 2000-01-04 08:30:00 | 3.718047  3.720967  3.739499             3.812104 | 124.890085  125.042475  126.011759  129.842239 |  52.180472   52.209669   52.394992     53.121039 | 1.088419  1.088872  1.091743  1.102942 | 1.359158  1.359128  1.358939          1.358217 |  -0.58108  -0.580866  -0.579509     -0.574226 | 0.672238  0.672007  0.670584      0.665225 | 0.908835  0.908773  0.908412      0.907153 | 100.000009 | 100.012444  100.178344  101.235169          105.438613 | 108.674536 | 0.057658  0.057683  0.057843       0.058468 |   0.2073  0.207446  0.208373        0.211998 | 100.000009  100.024329  100.326441  102.114269  108.674536 |  100.0 |   0.000009 | 108.674536 |
    | 2000-01-04 09:00:00 | 3.718022  3.720641  3.737512             3.804664 | 124.888761  125.025457  125.907643    129.4473 |  52.180219   52.206409   52.375117      53.04664 | 1.088415  1.088821  1.091435  1.101798 | 1.359159  1.359132  1.358959          1.358289 | -0.581082   -0.58089  -0.579654     -0.574765 |  0.67224  0.672033  0.670736      0.665769 | 0.908836   0.90878   0.90845      0.907279 | 100.000007 | 100.011002  100.159813  101.121517          105.003241 | 108.000742 | 0.057658   0.05768  0.057826       0.058404 | 0.207299   0.20743  0.208274        0.211627 | 100.000007  100.021511   100.29279  101.923083  108.000742 |  100.0 |   0.000007 | 108.000742 |
    | 2000-01-04 09:30:00 | 3.717999  3.720348    3.7357              3.79777 | 124.887589  125.010183  125.812784  129.081824 |  52.179994   52.203483   52.357002     52.977698 | 1.088412  1.088776  1.091155  1.100737 | 1.359159  1.359135  1.358978          1.358357 | -0.581084  -0.580911  -0.579787     -0.575265 | 0.672242  0.672056  0.670874      0.666274 | 0.908836  0.908786  0.908485      0.907396 | 100.000006 | 100.009727  100.143183  101.017996          104.600752 | 107.375746 | 0.057657  0.057678   0.05781       0.058345 | 0.207298  0.207415  0.208183        0.211283 | 100.000006  100.019019  100.262556  101.748555  107.375746 |  100.0 |   0.000006 | 107.375746 |
    | 2000-01-04 10:00:00 |  3.71798  3.720086   3.73405             3.791386 | 124.886554  124.996478  125.726394  128.743812 |  52.179796   52.200858   52.340499     52.913857 | 1.088409  1.088735  1.090899  1.099754 | 1.359159  1.359137  1.358994           1.35842 | -0.581085  -0.580931  -0.579908     -0.575728 | 0.672243  0.672076  0.671001      0.666743 | 0.908837  0.908792  0.908517      0.907505 | 100.000004 |   100.0086  100.128262  100.923742          104.228854 | 106.796353 | 0.057657  0.057676  0.057796        0.05829 | 0.207297  0.207402  0.208101        0.210964 | 100.000004  100.016815  100.235397  101.589305  106.796353 |  100.0 |   0.000004 | 106.796353 |
    | 2000-01-04 10:30:00 | 3.717962   3.71985  3.732547             3.785478 | 124.885638  124.984183  125.647749  128.431381 |   52.17962   52.198502   52.325472     52.854779 | 1.088406  1.088699  1.090666  1.098844 | 1.359159   1.35914   1.35901          1.358478 | -0.581086  -0.580948  -0.580018     -0.576157 | 0.672245  0.672095  0.671116      0.667178 | 0.908837  0.908797  0.908546      0.907607 | 100.000004 | 100.007603  100.114875  100.837957          103.885397 | 106.259549 | 0.057657  0.057673  0.057783       0.058239 | 0.207296   0.20739  0.208025         0.21067 | 100.000004  100.014867  100.211007  101.444059  106.259549 |  100.0 |   0.000004 | 106.259549 |
    | 2000-01-04 11:00:00 | 3.717946  3.719639  3.731179             3.780014 | 124.884828  124.973155  125.576183  128.142759 |  52.179465    52.19639   52.311793     52.800144 | 1.088403  1.088666  1.090454  1.098002 | 1.359159  1.359142  1.359024          1.358532 | -0.581087  -0.580963  -0.580118     -0.576555 | 0.672246  0.672112  0.671221      0.667581 | 0.908838  0.908801  0.908573      0.907702 | 100.000003 | 100.006722   100.10287   100.75991          103.568366 | 105.762488 | 0.057657  0.057672  0.057771       0.058192 | 0.207295   0.20738  0.207957        0.210397 | 100.000003  100.013144   100.18911  101.311642  105.762488 |  100.0 |   0.000003 | 105.762488 |
    | 2000-01-04 11:30:00 | 3.717933   3.71945  3.729935             3.774965 | 124.884112  124.963266  125.511084  127.876279 |  52.179328   52.194495   52.299347      52.74965 | 1.088401  1.088636  1.090262  1.097223 | 1.359159  1.359144  1.359036          1.358582 | -0.581088  -0.580977  -0.580209     -0.576922 | 0.672247  0.672127  0.671317      0.667955 | 0.908838  0.908805  0.908598       0.90779 | 100.000002 | 100.005943  100.092104  100.688928          103.275873 | 105.302489 | 0.057657   0.05767  0.057761       0.058149 | 0.207295   0.20737  0.207895        0.210145 | 100.000002  100.011621  100.169454  101.190972  105.302489 |  100.0 |   0.000002 | 105.302489 |
    | 2000-01-04 12:00:00 | 3.717921   3.71928  3.728803             3.770301 |  124.88348  124.954399   125.45189  127.630378 |  52.179206   52.192796   52.288027     52.703013 | 1.088399   1.08861  1.090086  1.096503 |  1.35916  1.359146  1.359048          1.358629 | -0.581089   -0.58099  -0.580292     -0.577261 | 0.672248   0.67214  0.671404      0.668301 | 0.908838  0.908809   0.90862      0.907872 | 100.000002 | 100.005254  100.082451  100.624396          103.006152 | 104.877026 | 0.057657  0.057669  0.057751       0.058109 | 0.207294  0.207362  0.207838        0.209912 | 100.000002  100.010274  100.151814  101.081052  104.877026 |  100.0 |   0.000002 | 104.877026 |
    | 2000-01-04 12:30:00 |  3.71791  3.719127  3.727774             3.765996 |  124.88292  124.946452  125.398085  127.403587 |  52.179099   52.191274   52.277736     52.659963 | 1.088398  1.088587  1.089927  1.095839 |  1.35916  1.359147  1.359059          1.358672 |  -0.58109  -0.581001  -0.580367     -0.577575 | 0.672249  0.672152  0.671484       0.66862 | 0.908838  0.908812   0.90864      0.907949 | 100.000001 | 100.004645    100.0738  100.565747          102.757549 | 104.483724 | 0.057657  0.057667  0.057742       0.058072 | 0.207293  0.207354  0.207787        0.209697 | 100.000001  100.009084  100.135987  100.980965  104.483724 |  100.0 |   0.000001 | 104.483724 |
    | 2000-01-04 13:00:00 |   3.7179  3.718991  3.726838             3.762025 | 124.882425  124.939329  125.349196  127.194533 |  52.179004   52.189909   52.268384     52.620249 | 1.088396  1.088565  1.089782  1.095225 |  1.35916  1.359149  1.359068          1.358712 | -0.581091  -0.581011  -0.580436     -0.577864 |  0.67225  0.672163  0.671556      0.668916 | 0.908839  0.908815  0.908659      0.908019 | 100.000001 | 100.004107  100.066046  100.512465          102.528522 | 104.120348 | 0.057657  0.057666  0.057734       0.058038 | 0.207293  0.207347   0.20774        0.209499 | 100.000001  100.008031  100.121789  100.889866  104.120348 |  100.0 |   0.000001 | 104.120348 |
    | 2000-01-04 13:30:00 | 3.717892  3.718869  3.725989             3.758363 | 124.881988  124.932946  125.304789  127.001927 |  52.178921   52.188686   52.259887     52.583634 | 1.088395  1.088546   1.08965   1.09466 |  1.35916   1.35915  1.359077          1.358748 | -0.581091   -0.58102  -0.580498     -0.578131 |  0.67225  0.672173  0.671621      0.669189 | 0.908839  0.908818  0.908675      0.908085 | 100.000001 | 100.003631  100.059098  100.464074          102.317628 | 103.784803 | 0.057657  0.057665  0.057727       0.058006 | 0.207292  0.207341  0.207697        0.209316 | 100.000001    100.0071  100.109056  100.806981  103.784803 |  100.0 |   0.000001 | 103.784803 |
    | 2000-01-04 14:00:00 | 3.717885  3.718759  3.725217             3.754989 | 124.881601  124.927228  125.264468  126.824564 |  52.178847    52.18759   52.252171     52.549893 | 1.088394  1.088529  1.089531  1.094138 |  1.35916  1.359151  1.359085          1.358782 | -0.581092  -0.581028  -0.580554     -0.578378 | 0.672251  0.672182  0.671681      0.669441 | 0.908839   0.90882  0.908691      0.908145 | 100.000001 |  100.00321  100.052873   100.42014          102.123522 | 103.475119 | 0.057656  0.057664   0.05772       0.057977 | 0.207292  0.207336  0.207659        0.209147 | 100.000001  100.006277  100.097639  100.731598  103.475119 |  100.0 |   0.000001 | 103.475119 |
    | 2000-01-04 14:30:00 | 3.717878  3.718661  3.724517             3.751882 | 124.881259  124.922106  125.227868  126.661321 |  52.178781   52.186609   52.245166      52.51882 | 1.088393  1.088514  1.089422  1.093658 |  1.35916  1.359152  1.359092          1.358814 | -0.581092  -0.581035  -0.580606     -0.578604 | 0.672252  0.672189  0.671735      0.669674 | 0.908839  0.908822  0.908705      0.908201 | 100.000001 | 100.002838  100.047298  100.380266          101.944949 | 103.189453 | 0.057656  0.057663  0.057714        0.05795 | 0.207292  0.207331  0.207624        0.208992 | 100.000001   100.00555  100.087403  100.663062  103.189453 |  100.0 |   0.000001 | 103.189453 |
    | 2000-01-04 15:00:00 | 3.717872  3.718573  3.723881             3.749022 | 124.880957  124.917519  125.194658  126.511145 |  52.178723    52.18573   52.238809     52.490217 | 1.088392  1.088501  1.089324  1.093216 |  1.35916  1.359153  1.359098          1.358842 | -0.581093  -0.581041  -0.580652     -0.578813 | 0.672252  0.672196  0.671784      0.669888 | 0.908839  0.908824  0.908717      0.908253 |      100.0 | 100.002509  100.042304  100.344087          101.780741 | 102.926076 | 0.057656  0.057662  0.057708       0.057925 | 0.207292  0.207327  0.207592        0.208849 |      100.0  100.004906  100.078228  100.600774  102.926076 |  100.0 |        0.0 | 102.926076 |
    | 2000-01-04 15:30:00 | 3.717867  3.718494  3.723304              3.74639 |  124.88069  124.913411  125.164532  126.373058 |  52.178672   52.184942   52.233042     52.463903 | 1.088391  1.088488  1.089234  1.092809 |  1.35916  1.359154  1.359104          1.358869 | -0.581093  -0.581047  -0.580695     -0.579005 | 0.672253  0.672203  0.671829      0.670086 | 0.908839  0.908826  0.908729      0.908301 |      100.0 | 100.002218  100.037833  100.311271          101.629808 | 102.683371 | 0.057656  0.057662  0.057703       0.057903 | 0.207291  0.207323  0.207563        0.208717 |      100.0  100.004338  100.070006  100.544184  102.683371 |  100.0 |        0.0 | 102.683371 |
    | 2000-01-04 16:00:00 | 3.717863  3.718424  3.722781             3.743971 | 124.880454  124.909733  125.137214  126.246144 |  52.178627   52.184238   52.227811     52.439707 |  1.08839  1.088477  1.089153  1.092435 |  1.35916  1.359154   1.35911          1.358894 | -0.581094  -0.581052  -0.580733     -0.579182 | 0.672253  0.672208   0.67187      0.670267 |  0.90884  0.908827  0.908739      0.908345 |      100.0 | 100.001961   100.03383  100.281516          101.491139 | 102.459826 | 0.057656  0.057661  0.057699       0.057882 | 0.207291  0.207319  0.207537        0.208596 |      100.0  100.003835  100.062639  100.492789  102.459826 |  100.0 |        0.0 | 102.459826 |
    | 2000-01-04 16:30:00 | 3.717859  3.718361  3.722307             3.741747 | 124.880245  124.906441  125.112448  126.129554 |  52.178587   52.183607   52.223069     52.417469 |  1.08839  1.088468   1.08908  1.092091 |  1.35916  1.359155  1.359115          1.358916 | -0.581094  -0.581057  -0.580768     -0.579345 | 0.672253  0.672213  0.671906      0.670435 |  0.90884  0.908829  0.908749      0.908386 |      100.0 | 100.001734  100.030247  100.254543          101.363792 | 102.254029 | 0.057656  0.057661  0.057695       0.057863 | 0.207291  0.207316  0.207513        0.208485 |      100.0   100.00339   100.05604  100.446128  102.254029 |  100.0 |        0.0 | 102.254029 |
    | 2000-01-04 17:00:00 | 3.717855  3.718304  3.721877             3.739704 |  124.88006  124.903495  125.090004  126.022497 |  52.178551   52.183042   52.218771     52.397041 | 1.088389  1.088459  1.089013  1.091774 |  1.35916  1.359156  1.359119          1.358937 | -0.581094  -0.581061  -0.580799     -0.579494 | 0.672254  0.672218   0.67194      0.670589 |  0.90884   0.90883  0.908758      0.908424 |      100.0 | 100.001533   100.02704    100.2301          101.246891 | 102.064661 | 0.057656   0.05766  0.057691       0.057845 | 0.207291  0.207313  0.207492        0.208383 |      100.0  100.002997  100.050128  100.403778  102.064661 |  100.0 |        0.0 | 102.064661 |
    | 2000-01-04 17:30:00 | 3.717852  3.718254  3.721488             3.737828 | 124.879897  124.900858  125.069669  125.924235 |   52.17852   52.182537   52.214877     52.378285 | 1.088389  1.088451  1.088953  1.091484 |  1.35916  1.359156  1.359123          1.358956 | -0.581094  -0.581065  -0.580828     -0.579631 | 0.672254  0.672222   0.67197       0.67073 |  0.90884  0.908831  0.908765      0.908459 |      100.0 | 100.001355   100.02417  100.207955          101.139626 | 101.890493 | 0.057656   0.05766  0.057688       0.057829 |  0.20729  0.207311  0.207472        0.208289 |      100.0   100.00265  100.044835  100.365353  101.890493 |  100.0 |        0.0 | 101.890493 |
    | 2000-01-04 18:00:00 | 3.717849  3.718208  3.721135             3.736107 | 124.879753  124.898499   125.05125  125.834086 |  52.178492   52.182085   52.211349     52.361071 | 1.088388  1.088444  1.088898  1.091218 |  1.35916  1.359157  1.359127          1.358973 | -0.581095  -0.581068  -0.580854     -0.579757 | 0.672254  0.672225  0.671997       0.67086 |  0.90884  0.908832  0.908773      0.908491 |      100.0 | 100.001198  100.021602  100.187899          101.041241 | 101.730378 | 0.057656  0.057659  0.057685       0.057814 |  0.20729  0.207308  0.207455        0.208203 |      100.0  100.002343  100.040095  100.330502  101.730378 |  100.0 |        0.0 | 101.730378 |
    | 2000-01-04 18:30:00 | 3.717847  3.718168  3.720815             3.734528 | 124.879625  124.896388  125.034572  125.751415 |  52.178468    52.18168   52.208155      52.34528 | 1.088388  1.088438  1.088848  1.090973 |  1.35916  1.359157   1.35913           1.35899 | -0.581095  -0.581071  -0.580877     -0.579873 | 0.672254  0.672229  0.672022       0.67098 |  0.90884  0.908833  0.908779       0.90852 |      100.0 | 100.001059  100.019304  100.169738          100.951038 |  101.58325 | 0.057656  0.057659  0.057682         0.0578 |  0.20729  0.207306  0.207439        0.208124 |      100.0  100.002071  100.035851    100.2989   101.58325 |  100.0 |        0.0 |  101.58325 |
    | 2000-01-04 19:00:00 | 3.717845  3.718132  3.720526              3.73308 | 124.879512    124.8945  125.019474  125.675633 |  52.178446   52.181318   52.205263       52.3308 | 1.088388  1.088432  1.088803  1.090749 |  1.35916  1.359157  1.359133          1.359004 | -0.581095  -0.581074  -0.580898     -0.579979 | 0.672254  0.672231  0.672045       0.67109 |  0.90884  0.908834  0.908785      0.908547 |      100.0 | 100.000936  100.017249  100.153299           100.86837 | 101.448115 | 0.057656  0.057659  0.057679       0.057788 |  0.20729  0.207304  0.207424        0.208052 |      100.0  100.001831  100.032052  100.270254  101.448115 |  100.0 |        0.0 | 101.448115 |
    | 2000-01-04 19:30:00 | 3.717843  3.718099  3.720265             3.731753 | 124.879413  124.892811  125.005811  125.606194 |  52.178427   52.180995   52.202646     52.317529 | 1.088387  1.088427  1.088763  1.090543 |  1.35916  1.359158  1.359136          1.359018 | -0.581095  -0.581076  -0.580917     -0.580076 | 0.672255  0.672234  0.672065       0.67119 |  0.90884  0.908834   0.90879      0.908572 |      100.0 | 100.000828  100.015411  100.138422          100.792636 |  101.32405 | 0.057656  0.057658  0.057677       0.057776 |  0.20729  0.207303  0.207411        0.207986 |      100.0  100.001619  100.028653  100.244295   101.32405 |  100.0 |        0.0 |  101.32405 |
    | 2000-01-04 20:00:00 | 3.717841  3.718071  3.720028             3.730537 | 124.879324  124.891301  124.993448  125.542592 |   52.17841   52.180705   52.200277     52.305371 | 1.088387  1.088423  1.088726  1.090355 |  1.35916  1.359158  1.359138           1.35903 | -0.581095  -0.581078  -0.580935     -0.580165 | 0.672255  0.672236  0.672084      0.671283 |  0.90884  0.908835  0.908795      0.908595 |      100.0 | 100.000732  100.013768  100.124963          100.723281 | 101.210197 | 0.057656  0.057658  0.057675       0.057766 |  0.20729  0.207301  0.207399        0.207925 |      100.0  100.001431  100.025611  100.220778  101.210197 |  100.0 |        0.0 | 101.210197 |
    | 2000-01-04 20:30:00 |  3.71784  3.718045  3.719814             3.729424 | 124.879246  124.889951  124.982266  125.484361 |  52.178395   52.180447   52.198135     52.294237 | 1.088387  1.088419  1.088693  1.090182 |  1.35916  1.359158   1.35914          1.359042 | -0.581095   -0.58108   -0.58095     -0.580246 | 0.672255  0.672238    0.6721      0.671367 |  0.90884  0.908836  0.908799      0.908616 |      100.0 | 100.000647  100.012298  100.112788          100.659793 | 101.105759 | 0.057656  0.057658  0.057673       0.057756 |  0.20729    0.2073  0.207389        0.207869 |      100.0  100.001265  100.022889  100.199479  101.105759 |  100.0 |        0.0 | 101.105759 |
    | 2000-01-04 21:00:00 | 3.717838  3.718022   3.71962             3.728404 | 124.879178  124.888744  124.972153  125.431066 |  52.178382   52.180215   52.196198     52.284045 | 1.088387  1.088415  1.088663  1.090025 |  1.35916  1.359159  1.359142          1.359052 | -0.581095  -0.581082  -0.580965     -0.580321 | 0.672255   0.67224  0.672115      0.671445 |  0.90884  0.908836  0.908803      0.908635 |      100.0 | 100.000572  100.010984  100.101779          100.601696 | 101.009998 | 0.057656  0.057658  0.057671       0.057747 |  0.20729  0.207299  0.207379        0.207818 |      100.0  100.001119  100.020454  100.180194  101.009998 |  100.0 |        0.0 | 101.009998 |
    | 2000-01-04 21:30:00 | 3.717837  3.718001  3.719445             3.727472 | 124.879117  124.887665  124.963011  125.382308 |   52.17837   52.180009   52.194446     52.274718 | 1.088386  1.088412  1.088636   1.08988 |  1.35916  1.359159  1.359144          1.359062 | -0.581095  -0.581083  -0.580978     -0.580389 | 0.672255  0.672242  0.672129      0.671516 |  0.90884  0.908837  0.908807      0.908653 |      100.0 | 100.000506  100.009809  100.091826          100.548551 | 100.922228 | 0.057656  0.057657   0.05767       0.057739 |  0.20729  0.207298   0.20737        0.207772 |      100.0  100.000989  100.018276  100.162737  100.922228 |  100.0 |        0.0 | 100.922228 |
    | 2000-01-04 22:00:00 | 3.717836  3.717982  3.719286             3.726619 | 124.879063    124.8867  124.954747  125.337716 |   52.17836   52.179824   52.192863     52.266187 | 1.088386  1.088409  1.088611  1.089748 |  1.35916  1.359159  1.359146           1.35907 | -0.581095  -0.581085  -0.580989     -0.580452 | 0.672255  0.672243  0.672141      0.671581 |  0.90884  0.908837   0.90881      0.908669 |      100.0 | 100.000447  100.008759   100.08283          100.499955 | 100.841815 | 0.057656  0.057657  0.057669       0.057732 |  0.20729  0.207297  0.207362        0.207729 |      100.0  100.000874  100.016328   100.14694  100.841815 |  100.0 |        0.0 | 100.841815 |
    | 2000-01-04 22:30:00 | 3.717835  3.717966  3.719143             3.725839 | 124.879015  124.885838   124.94728  125.296951 |  52.178351   52.179658   52.191432     52.258387 | 1.088386  1.088406  1.088589  1.089627 |  1.35916  1.359159  1.359147          1.359078 | -0.581096  -0.581086     -0.581     -0.580509 | 0.672255  0.672245  0.672153       0.67164 |  0.90884  0.908837  0.908813      0.908684 |      100.0 | 100.000395  100.007821  100.074701          100.455533 | 100.768171 | 0.057656  0.057657  0.057667       0.057725 |  0.20729  0.207296  0.207355         0.20769 |      100.0  100.000773  100.014586  100.132648  100.768171 |  100.0 |        0.0 | 100.768171 |
    | 2000-01-04 23:00:00 | 3.717834  3.717951  3.719014             3.725126 | 124.878973  124.885068  124.940533  125.259695 |  52.178343   52.179511    52.19014     52.251257 | 1.088386  1.088404  1.088569  1.089516 | 1.359161  1.359159  1.359148          1.359086 | -0.581096  -0.581087  -0.581009     -0.580561 | 0.672255  0.672246  0.672163      0.671695 |  0.90884  0.908838  0.908816      0.908698 |      100.0 | 100.000349  100.006983  100.067357           100.41494 | 100.700751 | 0.057656  0.057657  0.057666       0.057719 |  0.20729  0.207295  0.207349        0.207654 |      100.0  100.000683  100.013029  100.119721  100.700751 |  100.0 |        0.0 | 100.700751 |
    | 2000-01-04 23:30:00 | 3.717834  3.717938  3.718897             3.724474 | 124.878936  124.884379  124.934439   125.22566 |  52.178336   52.179379   52.188972     52.244743 | 1.088386  1.088402  1.088551  1.089416 | 1.359161  1.359159   1.35915          1.359092 | -0.581096  -0.581088  -0.581018     -0.580609 | 0.672255  0.672247  0.672172      0.671744 |  0.90884  0.908838  0.908818       0.90871 |      100.0 | 100.000309  100.006234  100.060723           100.37786 | 100.639054 | 0.057656  0.057657  0.057665       0.057713 |  0.20729  0.207295  0.207343        0.207622 |      100.0  100.000604  100.011636  100.108031  100.639054 |  100.0 |        0.0 | 100.639054 |


KinW-Implicit-Euler
___________________

Next, we use |kinw_impl_euler| as the main model.  Its better stability regarding
discontinuous channel profiles eases defining more comlex cross sections.  Hence, we
can reuse the test configuration of the :ref:`kinw_impl_euler_overbank_flow` example:

>>> from hydpy import reverse_model_wildcard_import
>>> reverse_model_wildcard_import()

>>> from hydpy.models.kinw_impl_euler import *
>>> parameterstep()
>>> stream.model = model
>>> length(100.0)
>>> nmbsegments(8)
>>> with model.add_wqmodel_v1("wq_widths_strickler") as submodel:
...     control = submodel.parameters.control
...     control.nmbwidths(5)
...     control.nmbsectors(3)
...     control.heights(0.0, 6.0, 6.0, 8.0, 10.0)
...     control.flowwidths(15.0, 75.0, 275.0, 315.0, 715.0)
...     control.totalwidths(15.0, 75.0, 275.0, 315.0, 715.0)
...     control.bottomslope(0.00025)
...     control.transitions(1, 3)
...     control.stricklercoefficients(1.0/0.035, 10.0, 10.0)
>>> IntegrationTest.plotting_options.axis1 = (
...     fluxes.inflow, fluxes.internalflow, fluxes.outflow
... )
>>> test = IntegrationTest(stream)
>>> nodes.input1.sequences.sim.series = q_base
>>> nodes.input2.sequences.sim.series = series

Overbank flow
-------------

Due to the practically identical configuration, the following results are equal to
those of the :ref:`kinw_impl_euler_overbank_flow` example:

.. integration-test::

    >>> model.prepare_states(100.0)
    >>> test("wq_widths_strickler_kinw_impl_euler_overbank_flow")
    |                date |                                                                       waterdepth |     inflow |                                                                         internalflow |    outflow |                                                                         watervolume | input1 |     input2 |     output |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000003 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000029 | 100.000004  100.000001       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000029 |      100.0 |
    | 2000-01-01 04:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000175 | 100.000027  100.000004  100.000001       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000175 |      100.0 |
    | 2000-01-01 04:30:00 | 3.717835  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000826 | 100.000136  100.000022  100.000004  100.000001       100.0       100.0         100.0 |      100.0 |  1.560988   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000826 |      100.0 |
    | 2000-01-01 05:00:00 | 3.717842  3.717834  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.003194 |  100.00055  100.000094  100.000016  100.000003       100.0       100.0         100.0 |      100.0 |  1.560992   1.560987  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.003194 |      100.0 |
    | 2000-01-01 05:30:00 | 3.717866  3.717839  3.717834  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.010515 |   100.0019  100.000338  100.000059   100.00001  100.000002       100.0         100.0 |      100.0 |  1.561008    1.56099  1.560987  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.010515 |      100.0 |
    | 2000-01-01 06:00:00 | 3.717934  3.717852  3.717836  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.030315 |  100.00575  100.001072  100.000197  100.000036  100.000006  100.000001         100.0 |      100.0 |  1.561052   1.560998  1.560988  1.560987  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.030315 |      100.0 |
    | 2000-01-01 06:30:00 | 3.718107  3.717886  3.717843  3.717835  3.717833  3.717833  3.717833    3.717833 | 100.078179 | 100.015564  100.003035  100.000581   100.00011   100.00002  100.000004    100.000001 |      100.0 |  1.561165   1.561021  1.560993  1.560987  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.078179 |      100.0 |
    | 2000-01-01 07:00:00 | 3.718507   3.71797   3.71786  3.717838  3.717834  3.717833  3.717833    3.717833 | 100.183351 | 100.038299  100.007813  100.001561  100.000306  100.000059  100.000011    100.000002 |      100.0 |  1.561426   1.561076  1.561004   1.56099  1.560987  1.560986  1.560986     1.560986 |  100.0 |   0.183351 |      100.0 |
    | 2000-01-01 07:30:00 | 3.719361  3.718159  3.717901  3.717847  3.717836  3.717833  3.717833    3.717833 | 100.396214 |   100.0868  100.018516  100.003858  100.000787  100.000158  100.000031    100.000006 | 100.000001 |  1.561983   1.561199   1.56103  1.560995  1.560988  1.560987  1.560986     1.560986 |  100.0 |   0.396214 | 100.000001 |
    | 2000-01-01 08:00:00 | 3.721055  3.718551  3.717989  3.717866   3.71784  3.717834  3.717833    3.717833 | 100.797305 | 100.183096  100.040817  100.008866  100.001882  100.000391   100.00008    100.000016 | 100.000003 |  1.563089   1.561455  1.561088  1.561008  1.560991  1.560987  1.560986     1.560986 |  100.0 |   0.797305 | 100.000003 |
    | 2000-01-01 08:30:00 | 3.724211  3.719319  3.718169  3.717907  3.717849  3.717836  3.717833    3.717833 | 101.507032 | 100.362585   100.08442  100.019103  100.004215   100.00091  100.000192     100.00004 | 100.000008 |  1.565149   1.561956  1.561206  1.561035  1.560997  1.560988  1.560987     1.560986 |  100.0 |   1.507032 | 100.000008 |
    | 2000-01-01 09:00:00 | 3.729764  3.720737  3.718517   3.71799  3.717868   3.71784  3.717834    3.717833 | 102.694853 | 100.678946  100.164995  100.038872  100.008911  100.001994  100.000436    100.000094 |  100.00002 |  1.568777   1.562881  1.561433  1.561089  1.561009  1.560991  1.560987     1.560986 |  100.0 |   2.694853 |  100.00002 |
    | 2000-01-01 09:30:00 | 3.739054  3.723226  3.719156  3.718148  3.717906  3.717849  3.717836    3.717834 | 104.586356 | 101.209449  100.306583  100.075149  100.017886  100.004147  100.000939    100.000208 | 100.000045 |  1.574856   1.564506  1.561849  1.561192  1.561034  1.560997  1.560989     1.560987 |  100.0 |   4.586356 | 100.000045 |
    | 2000-01-01 10:00:00 | 3.753893  3.727404  3.720275  3.718436  3.717978  3.717867  3.717841    3.717835 | 107.466641 | 102.060236  100.544408  100.138745  100.034262  100.008227  100.001927    100.000441 | 100.000099 |  1.584587   1.567234  1.562579   1.56138  1.561081  1.561008  1.560991     1.560987 |  100.0 |   7.466641 | 100.000099 |
    | 2000-01-01 10:30:00 | 3.776614  3.734129  3.722157   3.71894  3.718108  3.717899  3.717849    3.717836 | 111.678559 | 103.371034  100.928003  100.245728  100.062917  100.015638  100.003784    100.000894 | 100.000207 |  1.599541   1.571632  1.563807  1.561709  1.561166   1.56103  1.560996     1.560989 |  100.0 |  11.678559 | 100.000207 |
    | 2000-01-01 11:00:00 | 3.810064  3.744558  3.725204   3.71979  3.718336  3.717959  3.717863     3.71784 | 117.614836 | 105.318823    101.5245  100.419116  100.111189  100.028585  100.007145    100.001741 | 100.000415 |  1.621674   1.578462  1.565797  1.562263  1.561314  1.561068  1.561006     1.560991 |  100.0 |  17.614836 | 100.000415 |
    | 2000-01-01 11:30:00 | 3.857536   3.76018  3.729972  3.721172  3.718721  3.718062   3.71789    3.717847 | 125.703736 | 108.119985   102.42193  100.690808  100.189748  100.050423  100.013009    100.003268 | 100.000801 |  1.653324   1.588718  1.568913  1.563165  1.561565  1.561136  1.561024     1.560995 |  100.0 |  25.703736 | 100.000801 |
    | 2000-01-01 12:00:00 |  3.92261  3.782847  3.737205  3.723351  3.719348  3.718236  3.717937    3.717859 | 136.388647 | 112.030547  103.732402  101.103719  100.313651  100.086094  100.022912    100.005929 | 100.001496 |  1.697169   1.603655  1.573645  1.564587  1.561975  1.561249  1.561054     1.561003 |  100.0 |  36.388647 | 100.001496 |
    | 2000-01-01 12:30:00 | 4.008915  3.814777  3.747865  3.726687  3.720344  3.718522  3.718016     3.71788 | 150.102726 |  117.34411  105.594985  101.714099  100.503569  100.142672  100.039139    100.010429 | 100.002706 |  1.756134   1.624803   1.58063  1.566766  1.562624  1.561436  1.561106     1.561017 |  100.0 |  50.102726 | 100.002706 |
    | 2000-01-01 13:00:00 | 4.119816  3.858513  3.763165  3.731662  3.721881  3.718977  3.718147    3.717916 | 167.240349 | 124.386985  108.178138  102.593982  100.787199  100.230036  100.065007    100.017824 | 100.004755 |   1.83327   1.653979  1.590682  1.570018  1.563627  1.561733  1.561191     1.561041 |  100.0 |  67.240349 | 100.004755 |
    | 2000-01-01 13:30:00 | 4.258065  3.916856  3.784593  3.738904  3.724194  3.719685  3.718355    3.717976 | 188.127569 | 133.509905  111.681431  103.833731  101.200867   100.36166  100.105209    100.029665 |  100.00813 |  1.931582    1.69327  1.604808  1.574757  1.565138  1.562194  1.561327      1.56108 |  100.0 |  88.127569 |  100.00813 |
    | 2000-01-01 14:00:00 | 4.425484  3.992724  3.813918  3.749211  3.727599  3.720759  3.718681    3.718071 | 212.993962 | 145.075607  116.336168  105.544616  101.791309  100.555559  100.166246    100.048172 | 100.013555 |  2.053835   1.745001  1.624233  1.581513  1.567362  1.562895  1.561539     1.561142 |  100.0 | 112.993962 | 100.013555 |
    | 2000-01-01 14:30:00 | 4.622708   4.08898  3.853182  3.763576  3.732507  3.722354  3.719179    3.718222 | 241.948195 | 159.441754  122.404309  107.861294  102.617642  100.835375  100.256949    100.076463 | 100.022079 |  2.202347   1.811669   1.65041  1.590952   1.57057  1.563936  1.561864      1.56124 |  100.0 | 141.948195 | 100.022079 |
    | 2000-01-01 15:00:00 | 4.849057  4.208197  3.904666  3.783211  3.739442  3.724677  3.719925    3.718452 |  274.95935 | 176.939223  130.174918  110.943993  103.753491   101.23164  100.389104    100.118833 | 100.035189 |  2.378783   1.895845  1.685026  1.603895   1.57511  1.565453  1.562351      1.56139 |  100.0 |  174.95935 | 100.035189 |
    | 2000-01-01 15:30:00 | 5.102552  4.352412  3.970809  3.809558  3.749069  3.727997   3.72102      3.7188 | 311.845554 | 197.846626  139.957227  114.980056  105.289248  101.783201  100.578191    100.181093 |  100.05496 |  2.583981   2.000045  1.729985  1.621338  1.581421  1.567622  1.563066     1.561617 |  100.0 | 211.845554 |  100.05496 |
    | 2000-01-01 16:00:00 | 5.380047  4.522888  4.054102  3.844295  3.762209  3.732662  3.722601    3.719316 | 352.270855 | 222.362996  152.069479  120.184367  107.334352  102.538833  100.844252    100.270989 | 100.084235 |  2.817815   2.126574  1.787378  1.644468  1.590053  1.570672  1.564097     1.561953 |  100.0 | 252.270855 | 100.084235 |
    | 2000-01-01 16:30:00 | 5.677459  4.719921  4.156922  3.889315  3.779858  3.739115  3.724845    3.720066 | 395.750601 |  250.58136  166.823051  126.797986  110.019455  103.559018  101.212899    100.398703 | 100.126853 |   3.07912   2.277339  1.859423   1.67467  1.601681  1.574895  1.565563     1.562443 |  100.0 | 295.750601 | 100.126853 |
    | 2000-01-01 17:00:00 | 5.990049  4.942732  4.281349  3.946686  3.803208  3.747906  3.727984     3.72114 | 441.664962 | 282.466342  184.501989  135.084236  113.498205  104.917869  101.716478     100.57744 | 100.187925 |  3.365677   2.453675  1.948375  1.713524  1.617126  1.580658  1.567613     1.563144 |  100.0 | 341.664962 | 100.187925 |
    | 2000-01-01 17:30:00 |  6.09782  5.150883  4.422017  4.017479  3.833489  3.759696  3.732307    3.722656 | 489.279607 |  296.62657  201.992874  144.830235  117.879318  106.695649  102.394099    100.823956 | 100.274145 |  3.712453   2.624015  2.051268  1.762036  1.637257    1.5884  1.570439     1.564134 |  100.0 | 389.279607 | 100.274145 |
    | 2000-01-01 18:00:00 | 6.213979  5.350808  4.575159   4.10207  3.871876  3.775234  3.738171    3.724765 |  537.77215 | 313.425001  219.698205  155.903976  123.243691  108.974701  103.291154    101.158958 | 100.394117 |  4.116278   2.792723  2.166097  1.820825  1.662941  1.598631  1.574277      1.56551 |  100.0 |  437.77215 | 100.394117 |
    | 2000-01-01 18:30:00 | 6.343729   5.54841  4.738486  4.200418  3.919437  3.795339  3.746004    3.727654 | 586.262607 | 333.452069  238.086986  168.255414  129.659205   111.83798  104.458763     101.60739 | 100.558676 |  4.571337   2.964381  2.291794  1.890298  1.695019  1.611913   1.57941     1.567398 |  100.0 | 486.262607 | 100.558676 |
    | 2000-01-01 19:00:00 | 6.484685  5.747832  4.910749  4.312258  3.977114  3.820882    3.7563    3.731557 | 633.846018 | 356.550837  257.558344  181.898529  137.191096  115.369207  105.953387     102.19865 | 100.781201 |  5.070468   3.142567  2.427982  1.970771  1.734298  1.628862  1.586168     1.569949 |  100.0 | 533.846018 | 100.781201 |
    | 2000-01-01 19:30:00 | 6.634207  5.951663  5.091347  4.437229  4.045711  3.852768  3.769625    3.736754 | 679.625318 | 382.460797  278.425803  196.892729   145.90849  119.653891  107.836699    102.966796 | 101.077927 |  5.605364    3.32983  2.574741  2.062543  1.781557  1.650133  1.594934     1.573349 |  100.0 | 579.625318 | 101.077927 |
    | 2000-01-01 20:00:00 | 6.789505  6.049992  5.260613  4.571457  4.125339  3.891831  3.786596    3.743574 | 722.742733 | 410.812739  290.072989  211.599564  155.630553  124.744112  110.170677    103.950086 | 101.468151 |  6.166838   3.547162  2.715993  2.163287  1.837152  1.676365  1.606131     1.577817 |  100.0 | 622.742733 | 101.468151 |
    | 2000-01-01 20:30:00 | 6.947738  6.123323  5.417322  4.711275  4.215488  3.938754  3.807863    3.752398 | 762.408181 | 441.137766  300.213659   225.78839  166.158435  130.659407  113.013409    105.190044 | 101.974308 |  6.745125   3.800825  2.849959  2.270621   1.90105  1.708128  1.620213     1.583605 |  100.0 | 662.408181 | 101.974308 |
    | 2000-01-01 21:00:00 | 7.106112  6.206417  5.566583  4.854421  4.315312  3.994029  3.834078     3.76365 | 797.923464 | 472.884662  312.296967  239.823058  177.367134   137.40033  116.417209    106.730435 | 102.621909 |  7.330195   4.089883  2.980412  2.383042  1.972991  1.745897   1.63765     1.591001 |  100.0 | 697.923464 | 102.621909 |
    | 2000-01-01 21:30:00 | 7.261961  6.298514  5.712409  4.999692  4.423835  4.057953  3.865872    3.777794 | 828.701418 | 505.442628  326.332488  254.031911  189.193876  144.958861  120.428303    108.616343 | 103.439381 |  7.912061   4.412281  3.110553   2.49975  2.052614  1.790052  1.658911     1.600319 |  100.0 | 728.701418 | 103.439381 |
    | 2000-01-01 22:00:00 | 7.412817  6.398553  5.857846  5.146631  4.540104   4.13064  3.903829    3.795323 |  854.27953 | 538.166911  342.277085  268.698795   201.62599  153.326111  125.087494    110.893476 | 104.457865 |  8.481063   4.764883  3.242994  2.620481  2.139553  1.840882   1.68446     1.611903 |  100.0 |  754.27953 | 104.457865 |
    | 2000-01-01 22:30:00 | 7.556453  6.505211  6.001556   5.29465  4.663167  4.212041  3.948473    3.816754 | 874.327871 | 570.405101  360.024209  283.716661   214.63421   162.48948  130.430235    113.607573 | 105.710973 |  9.028124   5.143568  3.380347   2.74483  2.233414  1.898589  1.714741     1.626117 |  100.0 | 774.327871 | 105.710973 |
    | 2000-01-01 23:00:00 | 7.690925  6.616962  6.048361  5.427198  4.789204  4.301473  4.000173    3.842598 | 888.651531 | 601.522255  379.401511  289.853788  226.701223  172.206077  136.453737    116.799248 | 107.233896 |  9.544957   5.543386  3.541533  2.858504  2.331505  1.962943  1.750119     1.643335 |  100.0 | 788.651531 | 107.233896 |
    | 2000-01-01 23:30:00 | 7.814587   6.73214  6.101984  5.548925  4.915763  4.397906  4.059104    3.873342 | 897.187983 | 630.923383  400.173747  297.208105   238.13603  182.305175  143.131025    120.501315 | 109.062279 | 10.024233   5.958735  3.726871  2.964834  2.432001  2.033456  1.790853     1.663925 |  100.0 | 797.187983 | 109.062279 |
    | 2000-01-02 00:00:00 |   7.9261   6.84901  6.162275  5.663589  5.041439  4.500189  4.125242    3.909417 |      900.0 | 658.072291  422.050007  305.805342  249.219882  192.677656  150.422013    124.737854 | 111.231094 | 10.459703   6.383575  3.936112  3.066688  2.533777  2.109516  1.837084     1.688237 |  100.0 |      800.0 | 111.231094 |
    | 2000-01-02 00:30:00 | 8.024228  6.965862  6.228896  5.774216  5.165602  4.607192  4.198383    3.951185 | 897.264896 | 682.582901  444.701124  315.663093  260.204069  203.265897  158.282128    129.524486 | 113.773692 | 10.846131   6.811762   4.16838  3.166514  2.636266  2.190487  1.888848     1.716589 |  100.0 | 797.264896 | 113.773692 |
    | 2000-01-02 01:00:00 | 8.105723  7.080836  6.301311  5.883193   5.28816  4.717916  4.278182    3.998917 | 889.260928 | 703.635207  467.726364  326.768597  271.306132  214.053577  166.668764     134.86945 | 116.721076 | 11.180257   7.236398  4.422104  3.266346   2.73932   2.27578  1.946087     1.749256 |  100.0 | 789.260928 | 116.721076 |
    | 2000-01-02 01:30:00 | 8.171091  7.192126   6.37879   5.99232  5.409376  4.831537  4.364195    4.052791 | 876.351729 |  720.79486  490.694547  339.071914  282.706509  225.055512  175.545754    140.775151 | 120.101451 | 11.460259   7.650579  4.695025  3.367804  2.843092  2.364897  2.008674     1.786468 |  100.0 | 776.351729 | 120.101451 |
    | 2000-01-02 02:00:00 | 8.221772  7.298271  6.460459  6.031821  5.517865  4.945308  4.455563    4.112833 | 858.969595 | 734.276827  513.211595  352.486654  287.649581  235.185958  184.712633    147.214388 | 123.936388 | 11.684706   8.048496   4.98433  3.484511  2.937526  2.455749  2.076171     1.828369 |  100.0 | 758.969595 | 123.936388 |
    | 2000-01-02 02:30:00 | 8.258805  7.398087   6.54534   6.07045  5.616175  5.057348   4.55121    4.178896 |   837.5984 | 744.229759  534.920037  366.892274  292.846899  244.599676  194.015339    154.140028 | 128.238711 |  11.85277   8.425254   5.28678  3.617793  3.024371  2.546801  2.147946     1.874991 |  100.0 |   737.5984 | 128.238711 |
    | 2000-01-02 03:00:00 | 8.282981   7.49062  6.632391  6.113766  5.707327  5.166621  4.650015    4.250667 | 812.756821 | 750.774816  555.499992  382.137687   298.86229  253.528362  203.354213    161.494991 | 133.011957 | 11.964338   8.776748  5.598832  3.767688  3.105972  2.637115  2.223293     1.926261 |  100.0 | 712.756821 | 133.011957 |
    | 2000-01-02 03:30:00 | 8.294939  7.575117  6.720541  6.161636   5.79379  5.272704  4.750934    4.327699 | 784.982427 | 754.026056  574.670357  398.045957   305.71251  262.177441  212.674539    169.219982 | 138.250928 | 12.020059   9.099589  5.916756  3.933888  3.184336   2.72622  2.301511     1.982005 |  100.0 | 684.982427 | 138.250928 |
    | 2000-01-02 04:00:00 | 8.295215  7.650998  6.808724   6.21382  5.877562  5.375612  4.853069    4.409449 | 754.817086 | 754.101116  592.189602  414.419821  313.401207  270.725597  221.957543    177.259225 |    143.943 | 12.021348   9.391029  6.236742  4.115722  3.261152  2.814002  2.381968     2.041974 |  100.0 | 654.817086 |    143.943 |
    | 2000-01-02 04:30:00 | 8.284275  7.717832  6.895903  6.269978  5.960231  5.475649  4.955706    4.495317 |  722.79402 | 751.126179   607.85592  431.047714  321.915296  279.324567  231.211959    185.564436 | 150.069861 |  11.97035   9.648916  6.554996   4.31216  3.337815  2.900605  2.464133     2.105864 |  100.0 |  622.79402 | 150.069861 |
    | 2000-01-02 05:00:00 | 8.262536   7.77532  6.981094  6.329673  6.013265   5.56839  5.057449    4.584544 | 689.426711 | 745.237277  621.506605   447.70996  331.223207   285.22012   239.99611    194.023812 | 156.598511 | 11.869891   9.871631   6.86783  4.521836  3.420621  2.982008  2.546884      2.17323 |  100.0 | 589.426711 | 156.598511 |
    | 2000-01-02 05:30:00 | 8.230367  7.823274  7.063383  6.392387  6.040675  5.651084  5.156631    4.676182 | 655.199767 | 736.579129  633.016651  464.184831  341.274166   288.82539   247.99634    202.489448 | 163.477272 | 11.723408  10.058044  7.171728  4.743076  3.515028  3.055501  2.628796     2.243452 |  100.0 | 555.199767 | 163.477272 |
    | 2000-01-02 06:00:00 | 8.188101  7.861602  7.141935  6.457533  6.071452   5.72634  5.252319    4.769252 | 620.561767 | 725.302205  642.296593  480.254189  351.998375  292.983915   255.41522    210.863996 |   170.6454 | 11.534875  10.207454  7.463404  4.973936  3.621254  3.123124  2.708988     2.315845 |  100.0 | 520.561767 |   170.6454 |
    | 2000-01-02 06:30:00 | 8.136025  7.890294  7.216005  6.524472  6.105581  5.796304  5.344122    4.862845 | 585.920022 | 711.558343  649.289643  495.708487  363.307981  297.711685  262.431589    219.091529 | 178.040442 | 11.308726  10.319537   7.73985  5.212257  3.739328  3.186628     2.787     2.389737 |  100.0 | 485.920022 | 178.040442 |
    | 2000-01-02 07:00:00 | 8.074379  7.909399  7.284939  6.592531  6.142986  5.862745  5.432035    4.956184 |  551.63713 | 695.494848  653.968124  510.350971  375.098761  303.019751  269.201552    227.149194 |  185.60364 | 11.049782  10.394285  7.998361  5.455711   3.86907  3.247501  2.862695     2.464519 |  100.0 |  451.63713 |  185.60364 |
    | 2000-01-02 07:30:00 | 8.003347  7.919014  7.348176  6.661015  6.183527  5.927106  5.516318    5.048655 | 518.029154 | 677.245911   656.32912  524.000941  387.252398  308.910837   275.85941    235.039674 |  193.28366 | 10.763192  10.431936  8.236552  5.701858  4.010085  3.306994   2.93617      2.53968 |  100.0 | 418.029154 |  193.28366 |
    | 2000-01-02 08:00:00 | 7.924408  7.919561  7.405275  6.729233  6.227001  5.990544  5.597398    5.139816 | 485.365238 | 657.655691  656.463751  536.503057  399.639792  315.377731  282.518596    242.784476 | 201.038858 | 10.453069  10.434081  8.452481  5.948212  4.161756   3.36614  3.007692     2.614822 |  100.0 | 385.365238 | 201.038858 |
    | 2000-01-02 08:30:00 |  7.84003   7.91155  7.455913  6.796509  6.273144  6.016652  5.669739    5.228316 |  453.86844 | 637.064815  654.495862  547.729949  412.124757   322.40234  285.659763    249.823054 | 208.744007 | 10.123316  10.402705   8.64466  6.192302  4.323257  3.432277  3.072198     2.688764 |  100.0 |  353.86844 | 208.744007 |
    | 2000-01-02 09:00:00 | 7.751096  7.895527  7.499878  6.862198  6.321637  6.038311  5.734322    5.313198 | 423.717566 | 615.734918  650.569421  557.582992  424.567859  329.955261  288.510454    256.209912 | 216.298863 |  9.777685  10.340003  8.812035  6.431729  4.493559  3.506877  3.130339     2.760604 |  100.0 | 323.717566 | 216.298863 |
    | 2000-01-02 09:30:00 | 7.658445  7.872076  7.537062  6.925703  6.372117  6.062436  5.793097    5.394015 | 395.049792 | 593.924397  644.844835  565.992241  436.830195  337.995805  291.754796    262.107423 |  223.64292 |   9.41971  10.248346   8.95397  6.664221  4.671461  3.590111  3.183704      2.82984 |  100.0 | 295.049792 |  223.64292 |
    | 2000-01-02 10:00:00 | 7.562869  7.841802  7.567447  6.986481   6.42418  6.089031  5.847679    5.470685 | 367.963873 | 571.869329  637.493703  572.915483  448.776953  346.472467  295.404618    267.657219 | 230.747343 |  9.052681  10.130223  9.070211   6.88767  4.855609  3.682033  3.233649     2.896278 |  100.0 | 267.963873 | 230.747343 |
    | 2000-01-02 10:30:00 | 7.465105  7.805319  7.591099  7.044048  6.477394   6.11806  5.899405    5.543381 | 342.523746 | 549.782011  628.694052  578.336573  460.280604  355.323814  299.468399    272.981819 | 237.607874 |  8.679616   9.988181  9.160854  7.100171  5.044532  3.782573  3.281325     2.959951 |  100.0 | 242.523746 | 237.607874 |
    | 2000-01-02 11:00:00 | 7.365838  7.763245  7.608156  7.097991  6.531312   6.14945  5.949375    5.612447 | 318.762374 | 527.850309  618.626087  582.263213  471.223621  364.479734  303.949713    278.186113 | 244.238672 |  8.303257   9.824784  9.226307  7.300042  5.236671  3.891527  3.327699     3.021057 |  100.0 | 218.762374 | 244.238672 |
    | 2000-01-02 11:30:00 | 7.265692  7.716191  7.618815  7.147961  6.585475  6.183089  5.998482     5.67833 |  296.68568 | 506.237697  607.468531  584.724365   481.50067  373.862987  308.846374    283.358727 | 250.667034 |  7.926064   9.642569  9.267247  7.485845  5.430419  4.008557  3.373577     3.079902 |  100.0 |  196.68568 | 250.667034 |
    | 2000-01-02 12:00:00 | 7.165233  7.664751  7.623325  7.193681   6.63943  6.218823  6.014634    5.736261 | 276.276442 | 485.083853  595.395547  585.767411  491.020227  383.390983  314.149874    285.397587 |  256.40312 |   7.55021   9.444008  9.284577  7.656389  5.624151  4.133191  3.425331     3.132092 |  100.0 | 176.276442 |  256.40312 |
    | 2000-01-02 12:30:00 | 7.064968  7.609504  7.621976  7.234939  6.692735  6.256468  6.031488    5.787868 | 257.498082 | 464.505726  582.574245  585.455231  499.705638  392.977694  319.845067     287.60552 | 261.579447 |  7.177597   9.231485  9.279392  7.810739  5.816261   4.26483  3.483363     3.178939 |  100.0 | 157.498082 | 261.579447 |
    | 2000-01-02 13:00:00 | 6.965344  7.551002  7.615088  7.271588  6.744965  6.295802  6.050184    5.834683 | 240.298231 | 444.598954  569.162733  583.863279  507.495651  402.535616  325.910064    290.098762 | 266.329424 |  6.809855    9.00727  9.252931    7.9482   6.00519  4.402756  3.547823     3.221723 |  100.0 | 140.298231 | 266.329424 |
    | 2000-01-02 13:30:00 | 6.866752  7.489768  7.603009  7.303539  6.795724  6.336578  6.070741    5.877979 | 224.612057 | 425.439541  555.308697  581.076767  514.344463  411.977701  332.316346    292.886621 | 270.768533 |  6.448366   8.773505  9.206548  8.068319   6.18945  4.546146  3.618796     3.261536 |  100.0 | 124.612057 | 270.768533 |
    | 2000-01-02 14:00:00 | 6.769527  7.426299  7.586098  7.330762  6.844647  6.378525  6.093151    5.918815 | 210.365288 | 407.085728   541.14843  577.188018  520.221347  421.219179   339.02909    295.976507 |  274.99618 |  6.094269   8.532192  9.141677  8.170859  6.367654  4.694088  3.696291     3.299301 |  100.0 | 110.365288 |  274.99618 |
    | 2000-01-02 14:30:00 | 6.673951  7.361055  7.564728  7.353272  6.891405  6.421354  6.117385    5.958066 |  197.47692 | 389.579987  526.806278  572.294044  525.109939  430.179215  346.007686    299.373027 | 279.097337 |  5.748484   8.285185  9.059799   8.25579  6.538529  4.845597  3.780233     3.335797 |  100.0 |   97.47692 | 279.097337 |
    | 2000-01-02 15:00:00 | 6.580257  7.294467  7.539272  7.371132  6.935706  6.464766  6.143388    5.996455 | 185.861607 | 372.951102  512.394456  566.494375  529.007239  438.782351  353.206432    303.077452 | 283.144008 |  5.411722   8.034187  8.962419  8.323267  6.700934  4.999634  3.870466     3.371677 |  100.0 |  85.861607 | 283.144008 |
    | 2000-01-02 15:30:00 |  6.48863  7.226933  7.510106  7.384443  6.977302  6.508454  6.171079    6.010656 | 175.431719 | 357.216329  498.013158  559.889165  531.922423  446.959701  360.575371     307.08734 | 284.882829 |   5.08451   7.780753  8.851042  8.373607  6.853867  5.155126  3.966744     3.411645 |  100.0 |  75.431719 | 284.882829 |
    | 2000-01-02 16:00:00 | 6.399208  7.158817  7.477598  7.393337  7.015984  6.552111  6.200356    6.023639 | 166.099094 | 342.383683  483.750953  552.577579  533.875512  454.649886  368.061244    311.396274 | 286.572201 |  4.767198   7.526292  8.727154  8.407271  6.996473  5.310985  4.068741     3.456328 |  100.0 |  66.099094 | 286.572201 |
    | 2000-01-02 16:30:00 | 6.312088  7.090455  7.442108  7.397977  7.051583  6.595436   6.23109    6.038018 | 157.776495 |  328.45449  469.685421  544.656466   534.89597  461.799706  375.608511    315.993707 | 288.471511 |  4.459978   7.272076  8.592206   8.42484  7.128046  5.466129  4.176048     3.505868 |  100.0 |  57.776495 | 288.471511 |
    | 2000-01-02 17:00:00 | 6.227326  7.022154  7.403987  7.398547  7.083972  6.638137  6.263134    6.053819 | 150.378786 | 315.426609  455.884046  536.219304  535.021277  468.364545   383.16041    320.864921 | 290.588361 |  4.162891   7.019253  8.447603  8.426996  7.248028  5.619497   4.28818     3.560366 |  100.0 |  50.378786 | 290.588361 |
    | 2000-01-02 17:30:00 | 6.144934  6.954197  7.363572  7.395248  7.113059  6.679935  6.296322    6.071053 | 143.823868 | 303.299565  442.405489  527.355439  534.295542  474.308554   390.66001    325.991087 | 292.929399 |  3.875835   6.768862  8.294693  8.414504  7.356005  5.770064  4.404584     3.619877 |  100.0 |  43.823868 | 292.929399 |
    | 2000-01-02 18:00:00 | 6.064877  6.886854  7.321186  7.388297  7.138791   6.72057  6.330471    6.089717 | 138.033385 | 292.086645  429.301761  518.149647  532.768169  479.604604  398.051231    331.349436 | 295.499713 |  3.598539   6.521875  8.134767  8.388191  7.451699   5.91686  4.524647     3.684407 |  100.0 |  38.033385 | 295.499713 |
    | 2000-01-02 18:30:00 | 5.957751  6.819148  7.277038   7.37791  7.161143  6.759801  6.365387    6.109789 | 132.933235 | 279.064302  416.385268   508.66015  530.490586  484.233884  405.279774    336.913535 | 298.302466 |  3.335503   6.274697  7.968672  8.348896  7.534961  6.058978  4.647706     3.753907 |  100.0 |  32.933235 | 298.302466 |
    | 2000-01-02 19:00:00 |   5.7114  6.745476  7.230867  7.364263  7.180118   6.79741  6.400865    6.131235 | 128.453911 | 253.931868  402.629886  498.844418   527.50631  488.184541   412.29388     342.65364 | 301.338631 |  3.109643   6.007041  7.795486  8.297304   7.60574  6.195581  4.773058     3.828274 |  100.0 |  28.453911 | 301.338631 |
    | 2000-01-02 19:30:00 | 5.491977  6.667908  7.182624   7.34751  7.195737  6.833202  6.436694    6.154003 | 124.530701 | 232.744148  388.491316  488.707802  523.856082  491.450848  419.044953    348.537128 |  304.60678 |  2.914859   5.726696  7.615096  8.234037   7.66407  6.325911  4.899973     3.907348 |  100.0 |  24.530701 |  304.60678 |
    | 2000-01-02 20:00:00 | 5.296307  6.588022  7.132405  7.327804   7.20804  6.867005  6.472661    6.178023 | 121.103761 |   214.7826  374.308788  478.286988  519.580946  494.032679  425.488075    354.528985 |  308.10293 |  2.746237   5.439549  7.427935  8.159708  7.710057  6.449292  5.027699     3.990915 |  100.0 |  21.103761 |  308.10293 |
    | 2000-01-02 20:30:00 | 5.121673  6.507021    7.0804  7.305302  7.217081  6.898669  6.508554    6.203212 | 118.118088 | 199.480745  360.331669  467.637826  514.723882  495.935148  431.582419    360.592333 | 311.820433 |  2.599784   5.150017  7.234784  8.074953  7.743876  6.565127  5.155481     4.078705 |  100.0 |  18.118088 | 311.820433 |
    | 2000-01-02 21:00:00 | 4.965729  6.425809  7.026858  7.280172   7.22293   6.92807  6.544161    6.229473 | 115.523427 | 186.387781  346.740622  456.826542  509.330436  497.168316  437.291578    366.688993 | 315.749933 |  2.472228   4.861382   7.03663  7.980446  7.765768  6.672905  5.282566     4.170395 |  100.0 |  15.523427 | 315.749933 |
    | 2000-01-02 21:30:00 | 4.826442  6.345064  6.972059   7.25259  7.225672  6.955104  6.579278    6.256692 | 113.274094 | 175.141761  333.664491  445.923544  503.448689  497.746897  442.583813    372.780058 | 319.879368 |  2.360867   4.576041  6.834563  7.876901  7.776031  6.772198  5.408213     4.265616 |  100.0 |  13.274094 | 319.879368 |
    | 2000-01-02 22:00:00 | 4.702031  6.265281  6.916295  7.222743  7.225402  6.979691  6.613709    6.284747 | 111.328777 | 165.449622  321.194041   434.99926  497.128847  497.689948  447.432212    378.826464 | 324.194035 |  2.263449   4.295701  6.629714  7.765068  7.775021  6.862662  5.531703     4.363955 |  100.0 |  11.328777 | 324.194035 |
    | 2000-01-02 22:30:00 |  4.59093   6.18681  6.859864  7.190827   7.22223  7.001771  6.647267    6.313505 | 109.650285 | 157.072176  309.393889  424.121566  490.422617  497.020496  451.814779    384.789557 | 328.676705 |   2.17809   4.021522  6.423204  7.645726  7.763145  6.944033  5.652348     4.464958 |  100.0 |   9.650285 | 328.676705 |
    | 2000-01-02 23:00:00 | 4.491756  6.109879   6.80306  7.157042  7.216274  7.021306  6.679777    6.342822 | 108.205287 |  149.81273   298.31508  413.354671   483.38256  495.765148  455.714439    390.631622 | 333.307803 |  2.103196   3.754218  6.216133  7.519676  7.740857  7.016124  5.769497     4.568141 |  100.0 |   8.205287 | 333.307803 |
    | 2000-01-02 23:30:00 | 4.403276  6.034604  6.746181  7.121596  7.207664  7.038276   6.71108    6.372551 | 106.964036 | 143.508391  288.018059  402.760106  476.061595  493.953668  459.118982     396.31638 | 338.065625 |  2.037416     3.4941  6.009597  7.387733  7.708651  7.078826  5.882542     4.672992 |  100.0 |   6.964036 | 338.065625 |
    | 2000-01-03 00:00:00 | 4.324389   5.87222  6.685695  7.084392   7.19651  7.052677  6.741027    6.402539 | 105.900084 | 138.023356  270.175454  391.701847   468.45008  491.612824  462.020419    401.809402 | 342.926605 |  1.979594   3.256226   5.79085  7.249586  7.666958  7.132093  5.990922     4.778981 |  100.0 |   5.900084 | 342.926605 |
    | 2000-01-03 00:30:00 | 4.254112  5.645633  6.619513  7.045192  7.182894  7.064516  6.769487    6.432631 |  104.99001 | 133.243702  247.464084  379.852937  460.511033  488.764097  464.414175    407.078409 | 347.865611 |  1.928738    3.05063   5.55255  7.104402  7.616102  7.175922  6.094126     4.885564 |  100.0 |    4.99001 | 347.865611 |
    | 2000-01-03 01:00:00 | 4.191558  5.443057  6.549477  7.003944  7.166887   7.07381  6.796342    6.462671 | 104.213141 | 129.073312  228.171788  367.606167  452.247544  485.427808  466.298676    412.093503 | 352.856257 |   1.88399   2.872252  5.301568  6.952047  7.556378  7.210355  6.191696     4.992191 |  100.0 |   4.213141 | 352.856257 |
    | 2000-01-03 01:30:00 |  4.13593  5.261763   6.47701  6.960727  7.148565  7.080584  6.821488    6.492507 | 103.551298 | 125.430656  211.701614  355.259198  443.689606  481.625646  467.675201    416.827364 | 357.871238 |  1.844607   2.716965  5.043164  6.792873  7.488093  7.235466  6.283222     5.098312 |  100.0 |   3.551298 | 357.871238 |
    | 2000-01-03 02:00:00 | 4.086509  5.099387  6.403205  6.915703  7.128013  7.084872  6.844838    6.521985 | 102.988541 | 122.246241  197.576901   343.03533   434.88427  477.382025  468.547847      421.2554 | 362.882664 |  1.809943    2.58137  4.781339  6.627544  7.411597  7.251367  6.368348     5.203383 |  100.0 |   2.988541 | 362.882664 |
    | 2000-01-03 02:30:00 | 4.042647  4.953871  6.328897  6.869094   7.10533  7.086717  6.866315     6.55096 |  102.51094 | 119.460565  185.413991  331.100485  425.888402  472.724635  468.923549    425.355879 | 367.862403 |  1.779434   2.462653  4.519103  6.456926  7.327292  7.258209   6.44677     5.306871 |  100.0 |    2.51094 | 367.862403 |
    | 2000-01-03 03:00:00 | 4.003758  4.823411  6.254713  6.821151   7.08063   7.08617  6.885859    6.579291 | 102.106358 | 117.022481   174.90175  319.577259  416.763521  467.684448  468.812079    429.110035 |  372.78242 |  1.752584   2.358471  4.258687  6.281991  7.235634  7.256179  6.518234     5.408261 |  100.0 |   2.106358 |  372.78242 |
    | 2000-01-03 03:30:00 | 3.969314  4.706421   6.18112  6.772142  7.054039  7.083291  6.903423    6.606844 | 101.764251 |  114.88786  165.786045  308.557166  407.572322  462.295395  468.226016    432.502135 | 377.615099 |  1.728962   2.266854  4.001699  6.103764  7.137133  7.245504  6.582537     5.507058 |  100.0 |   1.764251 | 377.615099 |
    | 2000-01-03 04:00:00 | 3.938837  4.601499  6.108442  6.722347  7.025697  7.078152  6.918971    6.633492 |  101.47548 | 113.018506  157.857897  298.113192    398.3767   456.59389  467.180662    435.519536 | 382.333565 |  1.708185   2.186143   3.74924  5.923289  7.032342  7.226448  6.639527     5.602792 |  100.0 |    1.47548 | 382.333565 |
    | 2000-01-03 04:30:00 |   3.9119  4.507401  6.036885  6.672053  6.995757   7.07083  6.932483    6.659119 | 101.232151 | 111.381268  150.944348  288.320846  389.237789   450.61839  465.693923    438.152691 | 386.911973 |  1.689916    2.11493  3.501962  5.741639  6.921857  7.199312  6.689101     5.695026 |  100.0 |   1.232151 | 386.911973 |
    | 2000-01-03 05:00:00 | 3.888113  4.423022  5.890484   6.61825  6.964118  7.061391  6.943947    6.683617 | 101.027458 | 109.947297  144.901382  272.058953  379.629367  444.357422  463.781529    440.394738 | 391.325752 |   1.67386   2.052012  3.273079  5.548012  6.805346  7.164349  6.731197      5.78335 |  100.0 |   1.027458 | 391.325752 |
    | 2000-01-03 05:30:00 | 3.867131  4.347374  5.672064  6.558441  6.930507  7.049871  6.953358    6.706888 | 100.855551 | 108.691441  139.608387  250.051262  369.156595  437.766948  461.454296    442.240854 | 395.551765 |  1.659756   1.996362  3.074281  5.333622  6.681848  7.121711  6.765781      5.86739 |  100.0 |   0.855551 | 395.551765 |
    | 2000-01-03 06:00:00 | 3.848639  4.279574  5.476062  6.494419  6.894834  7.036301  6.960719    6.728845 | 100.711411 | 107.591725  134.963802  231.250628  358.194715   430.84128  458.722006    443.687994 | 399.568436 |  1.647371   1.947092  2.900965  5.105123  6.551084  7.071526  6.792842     5.946805 |  100.0 |   0.711411 | 399.568436 |
    | 2000-01-03 06:30:00 | 3.832358  4.218828  5.299973  6.427581  6.857141  7.020714  6.966033    6.749407 | 100.590745 | 106.628928  130.881667  215.111129   347.03265  423.601272   455.59586    444.734866 | 403.355853 |  1.636503   1.903437  2.749352  4.867664   6.41326  7.013936  6.812392     6.021288 |  100.0 |   0.590745 | 403.355853 |
    | 2000-01-03 07:00:00 | 3.818036  4.164425  5.141623  6.359011  6.817559  7.003153  6.969315    6.768505 | 100.489885 | 105.786211  127.288872   201.19444  335.891072  416.085315  452.089878    445.382017 | 406.895863 |  1.626969   1.864732  2.616322   4.62521  6.268911  6.949128  6.824466     6.090563 |  100.0 |   0.489885 | 406.895863 |
    | 2000-01-03 07:30:00 | 3.805449  4.115726  4.999115  6.289544  6.776278  6.983678  6.970582     6.78608 | 100.405708 | 105.048813  124.122956  189.145931  324.937476   408.34261  448.221552    445.631972 | 410.172164 |  1.618612   1.830399  2.499281  4.380786  6.118781  6.877346  6.829128      6.15439 |  100.0 |   0.405708 | 410.172164 |
    | 2000-01-03 08:00:00 | 3.794395  4.072154  4.870786  6.219818  6.733525  6.962364  6.969859    6.802079 |  100.33556 | 104.403781  121.330328  178.676598  314.298972   400.42828  444.011977    445.489362 |  413.17039 |  1.611289   1.799931  2.396057  4.136665  5.963749  6.798895  6.826468     6.212565 |  100.0 |    0.33556 |  413.17039 |
    | 2000-01-03 08:30:00 | 3.784697  4.033188  4.755174  6.150312   6.68955  6.939303   6.96718    6.816462 | 100.277186 | 103.839744  118.864828  169.549303   304.07393  392.400031  439.485656    444.961022 | 415.878192 |  1.604876   1.772886  2.304825  3.894521  5.804762  6.714141  6.816613     6.264914 |  100.0 |   0.277186 | 415.878192 |
    | 2000-01-03 09:00:00 | 3.776194  3.998362  4.650985  6.081366  6.644617  6.914601  6.962588    6.829194 | 100.228681 | 103.346714  116.686547    161.5682  294.345259  384.316312  434.670155    444.056042 | 418.285306 |  1.599264   1.748874  2.224038  3.655522  5.642814  6.623504  6.799718     6.311301 |  100.0 |   0.228681 | 418.285306 |
    | 2000-01-03 09:30:00 | 3.768744  3.967253   4.55707  6.013192  6.599015   6.88838  6.956132    6.840254 | 100.188431 | 102.915922   114.76087  154.570544  285.210691  376.237164   429.59584    442.785777 | 420.383607 |  1.594354   1.727553  2.152381   3.42037  5.478966  6.527458  6.775976     6.351625 |  100.0 |   0.188431 | 420.383607 |
    | 2000-01-03 10:00:00 | 3.762223  3.939479  4.472408  5.822322  6.547633  6.860353  6.947835    6.849622 | 100.155078 |  102.53966  113.057677  148.420307  265.070237  367.287762  424.215003    441.156768 | 422.166548 |  1.590062   1.708621  2.088728    3.2104  5.294974  6.424989  6.745481     6.385807 |  100.0 |   0.155078 | 422.166548 |
    | 2000-01-03 10:30:00 | 3.756518  3.914698  4.396085   5.61914  6.490771  6.830302  6.937702    6.857285 | 100.127477 |  102.21116  111.550688  143.003158  244.887044  357.577846  418.495141    439.172513 | 423.628775 |  1.586311    1.69181  2.032114  3.027009  5.092131  6.315338  6.708262     6.413786 |  100.0 |   0.127477 | 423.628775 |
    | 2000-01-03 11:00:00 |  3.75153    3.8926  4.327286  5.436144  6.430018  6.798168  6.925736    6.863234 | 100.104666 | 101.924477  110.216917  138.222496  227.530052  347.434477  412.435977    436.836574 | 424.766044 |  1.583036   1.676883  1.981704  2.866256  4.876303  6.198335  6.664341     6.435513 |  100.0 |   0.104666 | 424.766044 |
    | 2000-01-03 11:30:00 | 3.747173  3.872905  4.265276  5.271129  6.366622  6.764005  6.911946     6.86746 | 100.085838 | 101.674391  109.036209  133.996298  212.534364  337.111863  406.059714    434.154541 | 425.575307 |  1.580176   1.663632  1.936776  2.724887  4.652064  6.074229   6.61377     6.450956 |  100.0 |   0.085838 | 425.575307 |
    | 2000-01-03 12:00:00 | 3.743368  3.855363  4.209397  5.122177  6.301553  6.727947  6.896355    6.869962 | 100.070317 | 101.456318   107.99086  130.254583  199.524008  326.806399  399.403566    431.135139 | 426.054885 |  1.577681    1.65187  1.896701  2.600202  4.422955  5.943554  6.556653       6.4601 |  100.0 |   0.070317 | 426.054885 |
    | 2000-01-03 12:30:00 | 3.740047  3.839748  4.159054   4.98761  6.235567  6.690179  6.879003    6.870743 | 100.057539 | 101.266238  107.065289  126.937385  188.192706  316.669501  392.514144    427.790725 | 426.204665 |  1.575506   1.641432  1.860931  2.489943  4.191697  5.807034  6.493155     6.462955 |  100.0 |   0.057539 | 426.204665 |
    | 2000-01-03 13:00:00 | 3.737151  3.825854  4.113713  4.865953   6.16924  6.650923  6.859947    6.869813 | 100.047032 | 101.100623  106.245764  123.993102  178.289228  306.818996  385.443399     424.13737 | 426.026287 |  1.573609    1.63217  1.828986   2.39221  3.960343   5.66551  6.423506     6.459555 |  100.0 |   0.047032 | 426.026287 |
    | 2000-01-03 13:30:00 | 3.734626    3.8135  4.072889  4.755906  6.103004  6.610421  6.839259    6.867189 | 100.038402 |  100.95638  105.520162  121.377155  169.606152  297.350772  378.245979    420.194681 | 425.523289 |  1.571957   1.623955  1.800443  2.305397  3.730403  5.519899  6.347998     6.449963 |  100.0 |   0.038402 | 425.523289 |
    | 2000-01-03 14:00:00 | 3.732426  3.802522  4.036146  4.656318  6.037159  6.568937  6.817031    6.862895 | 100.031322 |   100.8308  104.877774  119.050889  161.971161  288.357315  370.978392    415.985553 | 424.701214 |  1.570518   1.616671  1.774932  2.228141  3.502908  5.371181  6.266986     6.434275 |  100.0 |   0.031322 | 424.701214 |
    | 2000-01-03 14:30:00 | 3.730511   3.79277  4.003087  4.566167  5.908081  6.523945  6.793149    6.856947 | 100.025521 | 100.721508  104.309124  116.980676  155.240258   273.88115  363.217749    411.495084 | 423.564169 |  1.569265   1.610213  1.752123  2.159274  3.289354  5.210375  6.180086     6.412551 |  100.0 |   0.025521 | 423.564169 |
    | 2000-01-03 15:00:00 | 3.728843  3.784112  3.973356  4.484539  5.703394  6.472706  6.767314    6.849336 | 100.020773 | 100.626426  103.805827  115.137162   149.29244  253.139136  354.536564    406.674392 | 422.112028 |  1.568175    1.60449  1.731726  2.097794   3.10243   5.02786  6.086238     6.384763 |  100.0 |   0.020773 | 422.112028 |
    | 2000-01-03 15:30:00 | 3.727392  3.776431  3.946628  4.410619  5.518643  6.416903  6.739398    6.840044 | 100.016892 | 100.543734  103.360453  113.494653  144.025482  235.259589  345.276607    401.509192 | 420.343799 |  1.567227    1.59942  1.713485  2.042839  2.938209  4.829829   5.98502     6.350861 |  100.0 |   0.016892 | 420.343799 |
    | 2000-01-03 16:00:00 |  3.72613  3.769618  3.922611  4.343676  5.351655   6.35786  6.709401     6.82906 | 100.013722 | 100.471842  102.966418  112.030599  139.352593  219.775096  335.706851    396.010059 | 418.259852 |  1.566402    1.59493  1.697169  1.993659  2.793449  4.621152  5.876474     6.310811 |  100.0 |   0.013722 | 418.259852 |
    | 2000-01-03 16:30:00 | 3.725032   3.76358  3.901039  4.283051  5.200543  6.296622  6.677414    6.816383 | 100.011136 | 100.409361  102.617882  110.725155  135.199729  206.307243  326.037781    390.204834 | 415.863287 |  1.565685   1.590955  1.682576  1.949605  2.665455  4.405637  5.760973     6.264626 |  100.0 |   0.011136 | 415.863287 |
    | 2000-01-03 17:00:00 | 3.724079   3.75823  3.881671  4.228152  5.063658  6.234006   6.64359    6.802028 | 100.009028 | 100.355074  102.309667  109.560824  131.503436  194.547442  316.433758    384.132812 | 413.160647 |  1.565062   1.587436  1.669524  1.910108  2.551976  4.186241  5.639115     6.212376 |  100.0 |   0.009028 | 413.160647 |
    | 2000-01-03 17:30:00 |  3.72325  3.753492  3.864291  4.178446  4.939554  6.170646  6.608122    6.786027 | 100.007313 | 100.307921  102.037178  108.522142  128.209102  184.242315  307.024024    377.840418 | 410.162198 |  1.564521   1.584324  1.657851  1.874672  2.451116  3.965234  5.511645     6.154197 |  100.0 |   0.007313 | 410.162198 |
    | 2000-01-03 18:00:00 |  3.72253  3.749298  3.848701  4.133447  4.826955  6.107021  6.571235     6.76843 | 100.005918 | 100.266977   101.79634  107.595423  125.269535  175.182461   297.91374    371.378256 | 406.881898 |  1.564051   1.581571  1.647413  1.842858  2.361273  3.744318  5.379409      6.09029 |  100.0 |   0.005918 | 406.881898 |
    | 2000-01-03 18:30:00 | 3.721905  3.745588  3.834724   4.09272  4.724735  6.043482  6.533177    6.749306 | 100.004785 | 100.231432  101.583536  106.768529  122.643796  167.193755  289.200153    364.799765 | 403.337244 |  1.563643   1.579137   1.63808  1.814283  2.281083  3.524707   5.24333     6.020923 |  100.0 |   0.004785 | 403.337244 |
    | 2000-01-03 19:00:00 | 3.721363  3.742306  3.822198  4.055868  4.631892  5.935524  6.492242    6.728587 | 100.003865 | 100.200584  101.395555  106.030684  120.296249  160.130555  276.737585    357.826502 | 399.521112 |  1.563289   1.576986  1.629737  1.788605  2.209381  3.314814   5.09737     5.945872 |  100.0 |   0.003865 | 399.521112 |
    | 2000-01-03 19:30:00 | 3.720892  3.739406  3.810977  4.022532  4.547534  5.734566   6.44482    6.705945 | 100.003119 | 100.173818   101.22955  105.372305  118.195765  153.870365  256.234201     349.88341 | 395.379733 |  1.562982   1.575086   1.62228  1.765522  2.145167  3.130559  4.928801     5.863979 |  100.0 |   0.003119 | 395.379733 |
    | 2000-01-03 20:00:00 | 3.720483  3.736842   3.80093  3.992385  4.470865  5.552759  6.392579    6.681225 | 100.002515 |   100.1506  101.082996  104.784865  116.315068  148.309613  238.501738    341.305365 | 390.893276 |  1.562715   1.573407  1.615616  1.744768  2.087577  2.968213  4.743755     5.774721 |  100.0 |   0.002515 | 390.893276 |
    | 2000-01-03 20:30:00 | 3.720129  3.734578  3.791939   3.96513  4.401169  5.388038  6.336843    6.654404 | 100.002026 | 100.130463  100.953651  104.260765  114.630188  143.360283  223.094642     332.35839 | 386.066725 |  1.562484   1.571926  1.609664  1.726103  2.035862  2.824691   4.54708     5.678046 |  100.0 |   0.002026 | 386.066725 |
    | 2000-01-03 21:00:00 | 3.719822  3.732579  3.783895    3.9405  4.337806  5.238604  6.278659    6.625549 | 100.001631 | 100.113003  100.839529   103.79323  113.120007  138.947225  209.651136     323.25273 | 380.922673 |  1.562284   1.570618  1.604347  1.709315  1.989373  2.697424  4.342597      5.57424 |  100.0 |   0.001631 | 380.922673 |
    | 2000-01-03 21:30:00 | 3.719556  3.730815  3.776703  3.918248  4.280195   5.10289  6.218853    6.594794 | 100.001312 | 100.097866  100.738868  103.376217  111.765867  135.005978  197.875369    314.154332 | 375.495742 |   1.56211   1.569464    1.5996  1.694214  1.947541  2.584259  4.133295     5.463825 |  100.0 |   0.001312 | 375.495742 |
    | 2000-01-03 22:00:00 | 3.719325  3.729258  3.770275  3.898154  4.227817  4.979515  6.158069    6.562316 | 100.001054 | 100.084746  100.650107  103.004327  110.551251  131.481015  187.523776    305.195264 | 369.828393 |  1.561959   1.568447  1.595362  1.680629  1.909868  2.483382  3.921486     5.347486 |  100.0 |   0.001054 | 369.828393 |
    | 2000-01-03 22:30:00 | 3.719125  3.727886  3.764531  3.880014  4.180197  4.867267  6.096801    6.528327 | 100.000846 | 100.073377  100.571861  102.672741  109.461506  128.324308  178.394553    296.484487 | 363.968023 |  1.561829   1.567549  1.591581  1.668409  1.875915  2.393256  3.708924     5.226015 |  100.0 |   0.000846 | 363.968023 |
    | 2000-01-03 23:00:00 | 3.718951  3.726675  3.759402  3.863645  4.136908  4.765072  6.035409    6.493065 | 100.000679 | 100.063525  100.502905  102.377151  108.483605  125.494153  170.319483    288.124846 | 357.965712 |  1.561716   1.566758  1.588207  1.657418  1.845296   2.31257  3.496875     5.100302 |  100.0 |   0.000679 | 357.965712 |
    | 2000-01-03 23:30:00 | 3.718801  3.725608  3.754823  3.848879  4.097562  4.671975  5.915462    6.454192 | 100.000544 | 100.054991  100.442154  102.113709  107.605949  122.954208  163.157543    274.647633 | 351.441616 |  1.561618   1.566061  1.585198  1.647532  1.817669  2.240204  3.296193     4.962073 |  100.0 |   0.000544 | 351.441616 |
    | 2000-01-04 00:00:00 | 3.718671  3.724669  3.750738  3.835565  4.061806  4.587125   5.72222    6.409018 | 100.000436 | 100.047598  100.388647  101.878973  106.818195  120.672692  156.789858    255.005678 | 343.984873 |  1.561533   1.565448  1.582516  1.638641  1.792731  2.175193  3.119404      4.80191 |  100.0 |   0.000436 | 343.984873 |
    | 2000-01-04 00:30:00 | 3.718558  3.723841  3.747094  3.823565   4.02932  4.509764  5.546962    6.359117 | 100.000349 | 100.041196  100.341533  101.669865  106.111105  118.621725  151.115701    237.948915 | 335.908015 |  1.561459   1.564907  1.580125  1.630647  1.770212  2.116704  2.963104     4.625584 |  100.0 |   0.000349 | 335.908015 |
    | 2000-01-04 01:00:00 |  3.71846  3.723112  3.743844  3.812754  3.999812   4.43921  5.387776    6.305744 | 100.000279 | 100.035651  100.300059  101.483637  105.476421  116.776772  146.049304    223.070652 | 327.461112 |  1.561396   1.564431  1.577994   1.62346  1.749871  2.064014  2.824466     4.437681 |  100.0 |   0.000279 | 327.461112 |
    | 2000-01-04 01:30:00 | 3.718376   3.72247  3.740949  3.803018  3.973016   4.37485  5.242998    6.249899 | 100.000223 | 100.030851  100.263562  101.317827   104.90675  115.116184  141.517288    210.039238 |  318.84344 |   1.56134   1.564012  1.576096     1.617  1.731494  2.016492  2.701126     4.241833 |  100.0 |   0.000223 |  318.84344 |
    | 2000-01-04 02:00:00 | 3.718303  3.721906  3.738369  3.794252  3.948689  4.316133  5.111172     6.19237 | 100.000178 | 100.026695  100.231451  101.170236  104.395469  113.620807  137.456594    198.582347 | 310.213929 |  1.561293   1.563644  1.574407  1.611194  1.714888  1.973587    2.5911     4.040896 |  100.0 |   0.000178 | 310.213929 |
    | 2000-01-04 02:30:00 | 3.718239  3.721409  3.736071  3.786365  3.926611  4.262558  4.991019    6.133775 | 100.000142 | 100.023097  100.203207  101.038899  103.936639  112.273653  133.812801    188.474904 | 301.700992 |  1.561251   1.563319  1.572902  1.605978  1.699882  1.934817  2.492708     3.837089 |  100.0 |   0.000142 | 301.700992 |
    | 2000-01-04 03:00:00 | 3.718185  3.720972  3.734025   3.77927   3.90658  4.213674   4.88141     6.07459 | 100.000113 | 100.019983  100.178371  100.922057  103.524931  111.059619  130.538753    179.529731 | 293.413712 |  1.561216   1.563034  1.571564  1.601293  1.686319  1.899754  2.404525     3.632098 |  100.0 |   0.000113 | 293.413712 |
    | 2000-01-04 03:30:00 | 3.718137  3.720588  3.732205   3.77289  3.888412  4.169072  4.781344    6.015155 |  100.00009 | 100.017287  100.156538   100.81814   103.15556  109.965255  127.593431    171.590254 | 285.465173 |  1.561185   1.562784  1.570373  1.597086  1.674062  1.868024   2.32533     3.427123 |  100.0 |    0.00009 | 285.465173 |
    | 2000-01-04 04:00:00 | 3.718096   3.72025  3.730585  3.767156   3.87194   4.12838  4.689933    5.854782 | 100.000072 | 100.014955  100.137348  100.725744   102.82423  108.978557  124.941023    164.524764 | 268.384578 |  1.561158   1.562563  1.569314  1.593309  1.662984  1.839291   2.25408     3.240176 |  100.0 |   0.000072 | 268.384578 |
    | 2000-01-04 04:30:00 | 3.718061  3.719954  3.729145  3.762005  3.857011  4.091259  4.606384    5.672624 | 100.000057 | 100.012936  100.120487  100.643615  102.527077  108.088792   122.55016     158.22188 | 250.106292 |  1.561135    1.56237  1.568372  1.589918  1.652973  1.813261   2.18987     3.074784 |  100.0 |   0.000057 | 250.106292 |
    | 2000-01-04 05:00:00 |  3.71803  3.719693  3.727864  3.757378  3.843484    4.0574  4.529987    5.506953 | 100.000046 | 100.011189  100.105674  100.570632  102.260631  107.286352  120.393271    152.586938 | 234.154831 |  1.561115     1.5622  1.567535  1.586876  1.643927  1.789668  2.131922     2.927962 |  100.0 |   0.000046 | 234.154831 |
    | 2000-01-04 05:30:00 | 3.718003  3.719464  3.726726  3.753224  3.831233  4.026523  4.460107    5.356054 | 100.000036 | 100.009678  100.092665  100.505796   102.02177  106.562621  118.446052    147.539089 | 220.174858 |  1.561097    1.56205  1.566792  1.584148  1.635753  1.768278  2.079554     2.797217 |  100.0 |   0.000036 | 220.174858 |
    | 2000-01-04 06:00:00 |  3.71798  3.719263  3.725715  3.749496  3.820142   3.99837  4.396168    5.218433 | 100.000029 | 100.008371  100.081241  100.448213  101.807686  105.909865  116.687017    143.008972 | 207.874895 |  1.561082   1.561919  1.566131  1.581701  1.628369  1.748879  2.032175     2.680459 |  100.0 |   0.000029 | 207.874895 |
    | 2000-01-04 06:30:00 |  3.71796  3.719086  3.724817  3.746152  3.810103  3.972707  4.337655    5.092777 | 100.000023 |  100.00724  100.071213  100.397085  101.615856  105.321132  115.097117    138.936818 | 197.014271 |  1.561069   1.561804  1.565544  1.579507    1.6217  1.731283  1.989263     2.575919 |  100.0 |   0.000023 | 197.014271 |
    | 2000-01-04 07:00:00 | 3.717943  3.718932  3.724019  3.743153  3.801021   3.94932  4.284099    4.977931 | 100.000018 | 100.006261  100.062411  100.351703  101.444008  104.790168  113.659428     135.27091 | 187.393014 |  1.561058   1.561703  1.565024  1.577541  1.615677  1.715318  1.950363     2.482099 |  100.0 |   0.000018 | 187.393014 |
    | 2000-01-04 07:30:00 | 3.717928  3.718796  3.723312  3.740464  3.792808  3.928012  4.235077    4.872873 | 100.000014 | 100.005415  100.054688  100.311431  101.290101   104.31134  112.358871    131.966319 | 178.843991 |  1.561048   1.561614  1.564562  1.575779  1.610238  1.700832  1.915069      2.39772 |  100.0 |   0.000014 | 178.843991 |
    | 2000-01-04 08:00:00 | 3.717915  3.718676  3.722684  3.738055  3.785383  3.908605  4.190203    4.776696 | 100.000011 | 100.004683  100.047912  100.275703  101.152295  103.879574  111.181988    128.983871 |  171.22672 |   1.56104   1.561536  1.564152  1.574201  1.605329  1.687688  1.883026     2.321682 |  100.0 |   0.000011 |  171.22672 |
    | 2000-01-04 08:30:00 | 3.717904  3.718572  3.722126  3.735897  3.778672  3.890934  4.149128    4.688593 | 100.000009 | 100.004049  100.041968  100.244016  101.028941  103.490294  110.116743    126.289284 | 164.422493 |  1.561033   1.561468  1.563788  1.572788  1.600899   1.67576  1.853915     2.253043 |  100.0 |   0.000009 | 164.422493 |
    | 2000-01-04 09:00:00 | 3.717894   3.71848  3.721632  3.733964  3.772611  3.874848  4.111531     4.60784 | 100.000007 | 100.003502  100.036756   100.21592  100.918551  103.139371  109.152347    123.852454 | 158.330486 |  1.561026   1.561408  1.563465  1.571524  1.596901  1.664937  1.827455     2.190982 |  100.0 |   0.000007 | 158.330486 |
    | 2000-01-04 09:30:00 | 3.717886  3.718399  3.721194  3.732234  3.767137  3.860211  4.077122    4.533792 | 100.000006 | 100.003028  100.032186  100.191014   100.81979  102.823078  108.279121    121.646866 |  152.86465 |  1.561021   1.561356  1.563179  1.570392  1.593295  1.655116  1.803393      2.13479 |  100.0 |   0.000006 |  152.86465 |
    | 2000-01-04 10:00:00 | 3.717879  3.718329  3.720806  3.730685  3.762195  3.846895  4.045635    4.465865 | 100.000004 | 100.002618  100.028179  100.168943  100.731458   102.53805  107.488361    119.649092 | 147.951205 |  1.561016    1.56131  1.562926  1.569379  1.590044  1.646206  1.781504     2.083846 |  100.0 |   0.000004 | 147.951205 |
    | 2000-01-04 10:30:00 | 3.717873  3.718267  3.720462    3.7293  3.757736  3.834787  4.016825    4.403535 | 100.000004 | 100.002264  100.024668  100.149388  100.652474  102.281247  106.772232    117.838372 | 143.526612 |  1.561012   1.561269  1.562702  1.568474  1.587112  1.638122  1.761585     2.037608 |  100.0 |   0.000004 | 143.526612 |
    | 2000-01-04 11:00:00 | 3.717867  3.718213  3.720157  3.728061  3.753714  3.823779  3.990471    4.346327 | 100.000003 | 100.001958   100.02159  100.132067  100.581869  102.049922  106.123676    116.196263 | 139.535921 |  1.561009   1.561234  1.562503  1.567664  1.584469  1.630789  1.743454     1.995596 |  100.0 |   0.000003 | 139.535921 |
    | 2000-01-04 11:30:00 | 3.717863  3.718165  3.719888  3.726954  3.750087  3.813776  3.966367    4.293812 | 100.000002 | 100.001693  100.018894  100.116729  100.518771  101.841594   105.53632    114.706337 |  135.93142 |  1.561006   1.561203  1.562327   1.56694  1.582088  1.624139  1.726948     1.957391 |  100.0 |   0.000002 |  135.93142 |
    | 2000-01-04 12:00:00 | 3.717859  3.718124  3.719649  3.725964  3.746817  3.804689  3.944328    4.245599 | 100.000002 | 100.001464  100.016533   100.10315  100.462398   101.65402  105.004412     113.35393 |  132.67152 |  1.561003   1.561176  1.562171  1.566294  1.579943  1.618108  1.711919     1.922619 |  100.0 |   0.000002 |  132.67152 |
    | 2000-01-04 12:30:00 | 3.717855  3.718087  3.719437   3.72508  3.743871  3.796437   3.92418    4.201334 | 100.000001 | 100.001265  100.014464  100.091131  100.412046  101.485173  104.522749     112.12592 | 129.719843 |  1.561001   1.561152  1.562033  1.565716  1.578012   1.61264  1.698233      1.89095 |  100.0 |   0.000001 | 129.719843 |
    | 2000-01-04 13:00:00 | 3.717852  3.718056   3.71925   3.72429  3.741218  3.788946  3.905768    4.160691 | 100.000001 | 100.001094  100.012652  100.080496  100.367086  101.333221  104.086627    111.010544 | 127.044452 |  1.560999   1.561131  1.561911    1.5652  1.576273  1.607684   1.68577     1.862089 |  100.0 |   0.000001 | 127.044452 |
    | 2000-01-04 13:30:00 | 3.717849  3.718028  3.719084  3.723584  3.738828  3.782147  3.888945    4.123377 | 100.000001 | 100.000946  100.011066  100.071087   100.32695  101.196507  103.691787    109.997236 | 124.617222 |  1.560997   1.561113  1.561803   1.56474  1.574707  1.603193   1.67442     1.835773 |  100.0 |   0.000001 | 124.617222 |
    | 2000-01-04 14:00:00 | 3.717847  3.718003  3.718938  3.722955  3.736677   3.77598  3.873579    4.089121 | 100.000001 | 100.000818  100.009678  100.062765  100.291131  101.073535  103.334369    109.076488 | 122.413305 |  1.560996   1.561097  1.561707  1.564329  1.573299  1.599123  1.664085     1.811767 |  100.0 |   0.000001 | 122.413305 |
    | 2000-01-04 14:30:00 | 3.717845  3.717982  3.718808  3.722393  3.734741  3.770388  3.859549    4.057675 | 100.000001 | 100.000707  100.008462  100.055407  100.259173  100.962953  103.010877    108.239733 | 120.410686 |  1.560994   1.561083  1.561622  1.563962  1.572032  1.595437  1.654673     1.789859 |  100.0 |   0.000001 | 120.410686 |
    | 2000-01-04 15:00:00 | 3.717844  3.717963  3.718694  3.721892     3.733  3.765318  3.846741    4.028812 |      100.0 | 100.000611  100.007398  100.048901  100.230668  100.863538  102.718143    107.479241 | 118.589801 |  1.560993   1.561071  1.561548  1.563635  1.570893  1.592099  1.646103      1.76986 |  100.0 |        0.0 | 118.589801 |
    | 2000-01-04 15:30:00 | 3.717842  3.717947  3.718593  3.721445  3.731434  3.760724  3.835054    4.002325 |      100.0 | 100.000529  100.006468  100.043151  100.205249  100.774187   102.45329    106.788023 | 116.933223 |  1.560992    1.56106  1.561482  1.563343  1.569869  1.589076    1.6383     1.751599 |  100.0 |        0.0 | 116.933223 |
    | 2000-01-04 16:00:00 | 3.717841  3.717932  3.718503  3.721046  3.730027  3.756562  3.824393    3.978023 |      100.0 | 100.000457  100.005653   100.03807  100.182588  100.693903  102.213712    106.159759 | 115.425389 |  1.560991   1.561051  1.561423  1.563083  1.568949   1.58634  1.631197     1.734921 |  100.0 |        0.0 | 115.425389 |
    | 2000-01-04 16:30:00 |  3.71784   3.71792  3.718424  3.720691  3.728762  3.752794   3.81467    3.955729 |      100.0 | 100.000395  100.004941  100.033581  100.162391  100.621785  101.997041    105.588725 | 114.052366 |  1.560991   1.561043  1.561372  1.562851  1.568122  1.583865  1.624732     1.719686 |  100.0 |        0.0 | 114.052366 |
    | 2000-01-04 17:00:00 | 3.717839  3.717909  3.718354  3.720374  3.727625  3.749382  3.805806    3.935283 |      100.0 | 100.000342  100.004318  100.029616  100.144395   100.55702  101.801132    105.069729 | 112.801656 |   1.56099   1.561036  1.561326  1.562644  1.567379  1.581626  1.618849     1.705769 |  100.0 |        0.0 | 112.801656 |
    | 2000-01-04 17:30:00 | 3.717838  3.717899  3.718293  3.720092  3.726604  3.746294  3.797729    3.916536 |      100.0 | 100.000295  100.003773  100.026114  100.128364  100.498874  101.624035    104.598064 | 111.662027 |   1.56099    1.56103  1.561286   1.56246  1.566712    1.5796  1.613496     1.693054 |  100.0 |        0.0 | 111.662027 |
    | 2000-01-04 18:00:00 | 3.717837  3.717891  3.718238  3.719841  3.725688  3.743501   3.79037     3.89935 |      100.0 | 100.000255  100.003296  100.023022  100.114088  100.446685  101.463983    104.169453 |  110.62336 |  1.560989   1.561024  1.561251  1.562296  1.566113  1.577769  1.608626     1.681437 |  100.0 |        0.0 |  110.62336 |
    | 2000-01-04 18:30:00 | 3.717837  3.717883   3.71819  3.719617  3.724865  3.740976  3.783668    3.883602 |      100.0 | 100.000221  100.002879  100.020293  100.101376  100.399855  101.319369    103.780013 |  109.67653 |  1.560989   1.561019  1.561219   1.56215  1.565576  1.576114  1.604197     1.670823 |  100.0 |        0.0 |  109.67653 |
    | 2000-01-04 19:00:00 | 3.717836  3.717877  3.718148  3.719418  3.724127  3.738692  3.777566    3.869173 |      100.0 | 100.000191  100.002515  100.017884  100.090061  100.357846  101.188737    103.426211 | 108.813291 |  1.560988   1.561015  1.561192   1.56202  1.565094  1.574618  1.600169     1.661126 |  100.0 |        0.0 | 108.813291 |
    | 2000-01-04 19:30:00 | 3.717836  3.717871   3.71811  3.719241  3.723465  3.736628  3.772013    3.855958 |      100.0 | 100.000165  100.002197  100.015758  100.079992  100.320171  101.070766    103.104836 |  108.02618 |  1.560988   1.561011  1.561167  1.561905  1.564662  1.573267  1.596508     1.652268 |  100.0 |        0.0 |  108.02618 |
    | 2000-01-04 20:00:00 | 3.717835  3.717867  3.718077  3.719083  3.722872  3.734764  3.766961    3.843858 |      100.0 | 100.000143  100.001918  100.013883  100.071033  100.286393  100.964256    102.812965 | 107.308437 |  1.560988   1.561008  1.561146  1.561802  1.564274  1.572047   1.59318     1.644176 |  100.0 |        0.0 | 107.308437 |
    | 2000-01-04 20:30:00 | 3.717835  3.717862  3.718048  3.718943  3.722339   3.73308  3.762367    3.832782 |      100.0 | 100.000123  100.001675  100.012229  100.063065  100.256116  100.868119    102.547938 | 106.653928 |  1.560988   1.561005  1.561127   1.56171  1.563927  1.570946  1.590156     1.636785 |  100.0 |        0.0 | 106.653928 |
    | 2000-01-04 21:00:00 | 3.717835  3.717859  3.718022  3.718818  3.721862   3.73156  3.758189    3.822647 |      100.0 | 100.000107  100.001463   100.01077  100.055978  100.228985  100.781367    102.307333 | 106.057084 |  1.560987   1.561003   1.56111  1.561629  1.563615  1.569951   1.58741     1.630035 |  100.0 |        0.0 | 106.057084 |
    | 2000-01-04 21:30:00 | 3.717834  3.717855     3.718  3.718707  3.721435  3.730188  3.754393    3.813376 |      100.0 | 100.000092  100.001277  100.009484  100.049678  100.204679  100.703105    102.088945 | 105.512841 |  1.560987   1.561001  1.561095  1.561557  1.563336  1.569054  1.584915     1.623872 |  100.0 |        0.0 | 105.512841 |
    | 2000-01-04 22:00:00 | 3.717834  3.717852   3.71798  3.718609  3.721052   3.72895  3.750943    3.804897 |      100.0 |  100.00008  100.001115   100.00835  100.044078  100.182909   100.63252    101.890766 | 105.016591 |  1.560987   1.560999  1.561082  1.561492  1.563086  1.568245   1.58265     1.618246 |  100.0 |        0.0 | 105.016591 |
    | 2000-01-04 22:30:00 | 3.717834   3.71785  3.717962  3.718521  3.720709  3.727833   3.74781    3.797147 |      100.0 | 100.000069  100.000973   100.00735  100.039101  100.163417  100.568878    101.710966 |  104.56414 |  1.560987   1.560997  1.561071  1.561435  1.562863  1.567515  1.580595      1.61311 |  100.0 |        0.0 |  104.56414 |
    | 2000-01-04 23:00:00 | 3.717834  3.717848  3.717947  3.718443  3.720402  3.726826  3.744966    3.790064 |      100.0 |  100.00006  100.000849   100.00647  100.034679  100.145968   100.51151    101.547879 | 104.151664 |  1.560987   1.560996   1.56106  1.561384  1.562662  1.566857  1.578729     1.608423 |  100.0 |        0.0 | 104.151664 |
    | 2000-01-04 23:30:00 | 3.717834  3.717846  3.717933  3.718374  3.720127  3.725919  3.742384    3.783593 |      100.0 | 100.000051  100.000741  100.005693  100.030752  100.130353  100.459812    101.399986 | 103.775679 |  1.560987   1.560995  1.561052  1.561339  1.562483  1.566264  1.577037     1.604147 |  100.0 |        0.0 | 103.775679 |


.. _wq_widths_strickler_kinw_impl_euler_no_flow_areas:

No-flow areas
-------------

In this example, we demonstrate the effect of a more sudden difference between the
cross-section's total width and flow width, which comes into play as soon as the water
depth exceeds the main channel's depth:

.. integration-test::

    >>> model.wqmodel.parameters.control.totalwidths(15.0, 75.0, 375.0, 415.0, 815.0)
    >>> model.prepare_states(100.0)
    >>> test("wq_widths_strickler_kinw_impl_euler_no_flow_areas")
    |                date |                                                                       waterdepth |     inflow |                                                                         internalflow |    outflow |                                                                         watervolume | input1 |     input2 |     output |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000003 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000029 | 100.000004  100.000001       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000029 |      100.0 |
    | 2000-01-01 04:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000175 | 100.000027  100.000004  100.000001       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000175 |      100.0 |
    | 2000-01-01 04:30:00 | 3.717835  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000826 | 100.000136  100.000022  100.000004  100.000001       100.0       100.0         100.0 |      100.0 |  1.560988   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000826 |      100.0 |
    | 2000-01-01 05:00:00 | 3.717842  3.717834  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.003194 |  100.00055  100.000094  100.000016  100.000003       100.0       100.0         100.0 |      100.0 |  1.560992   1.560987  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.003194 |      100.0 |
    | 2000-01-01 05:30:00 | 3.717866  3.717839  3.717834  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.010515 |   100.0019  100.000338  100.000059   100.00001  100.000002       100.0         100.0 |      100.0 |  1.561008    1.56099  1.560987  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.010515 |      100.0 |
    | 2000-01-01 06:00:00 | 3.717934  3.717852  3.717836  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.030315 |  100.00575  100.001072  100.000197  100.000036  100.000006  100.000001         100.0 |      100.0 |  1.561052   1.560998  1.560988  1.560987  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.030315 |      100.0 |
    | 2000-01-01 06:30:00 | 3.718107  3.717886  3.717843  3.717835  3.717833  3.717833  3.717833    3.717833 | 100.078179 | 100.015564  100.003035  100.000581   100.00011   100.00002  100.000004    100.000001 |      100.0 |  1.561165   1.561021  1.560993  1.560987  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.078179 |      100.0 |
    | 2000-01-01 07:00:00 | 3.718507   3.71797   3.71786  3.717838  3.717834  3.717833  3.717833    3.717833 | 100.183351 | 100.038299  100.007813  100.001561  100.000306  100.000059  100.000011    100.000002 |      100.0 |  1.561426   1.561076  1.561004   1.56099  1.560987  1.560986  1.560986     1.560986 |  100.0 |   0.183351 |      100.0 |
    | 2000-01-01 07:30:00 | 3.719361  3.718159  3.717901  3.717847  3.717836  3.717833  3.717833    3.717833 | 100.396214 |   100.0868  100.018516  100.003858  100.000787  100.000158  100.000031    100.000006 | 100.000001 |  1.561983   1.561199   1.56103  1.560995  1.560988  1.560987  1.560986     1.560986 |  100.0 |   0.396214 | 100.000001 |
    | 2000-01-01 08:00:00 | 3.721055  3.718551  3.717989  3.717866   3.71784  3.717834  3.717833    3.717833 | 100.797305 | 100.183096  100.040817  100.008866  100.001882  100.000391   100.00008    100.000016 | 100.000003 |  1.563089   1.561455  1.561088  1.561008  1.560991  1.560987  1.560986     1.560986 |  100.0 |   0.797305 | 100.000003 |
    | 2000-01-01 08:30:00 | 3.724211  3.719319  3.718169  3.717907  3.717849  3.717836  3.717833    3.717833 | 101.507032 | 100.362585   100.08442  100.019103  100.004215   100.00091  100.000192     100.00004 | 100.000008 |  1.565149   1.561956  1.561206  1.561035  1.560997  1.560988  1.560987     1.560986 |  100.0 |   1.507032 | 100.000008 |
    | 2000-01-01 09:00:00 | 3.729764  3.720737  3.718517   3.71799  3.717868   3.71784  3.717834    3.717833 | 102.694853 | 100.678946  100.164995  100.038872  100.008911  100.001994  100.000436    100.000094 |  100.00002 |  1.568777   1.562881  1.561433  1.561089  1.561009  1.560991  1.560987     1.560986 |  100.0 |   2.694853 |  100.00002 |
    | 2000-01-01 09:30:00 | 3.739054  3.723226  3.719156  3.718148  3.717906  3.717849  3.717836    3.717834 | 104.586356 | 101.209449  100.306583  100.075149  100.017886  100.004147  100.000939    100.000208 | 100.000045 |  1.574856   1.564506  1.561849  1.561192  1.561034  1.560997  1.560989     1.560987 |  100.0 |   4.586356 | 100.000045 |
    | 2000-01-01 10:00:00 | 3.753893  3.727404  3.720275  3.718436  3.717978  3.717867  3.717841    3.717835 | 107.466641 | 102.060236  100.544408  100.138745  100.034262  100.008227  100.001927    100.000441 | 100.000099 |  1.584587   1.567234  1.562579   1.56138  1.561081  1.561008  1.560991     1.560987 |  100.0 |   7.466641 | 100.000099 |
    | 2000-01-01 10:30:00 | 3.776614  3.734129  3.722157   3.71894  3.718108  3.717899  3.717849    3.717836 | 111.678559 | 103.371034  100.928003  100.245728  100.062917  100.015638  100.003784    100.000894 | 100.000207 |  1.599541   1.571632  1.563807  1.561709  1.561166   1.56103  1.560996     1.560989 |  100.0 |  11.678559 | 100.000207 |
    | 2000-01-01 11:00:00 | 3.810064  3.744558  3.725204   3.71979  3.718336  3.717959  3.717863     3.71784 | 117.614836 | 105.318823    101.5245  100.419116  100.111189  100.028585  100.007145    100.001741 | 100.000415 |  1.621674   1.578462  1.565797  1.562263  1.561314  1.561068  1.561006     1.560991 |  100.0 |  17.614836 | 100.000415 |
    | 2000-01-01 11:30:00 | 3.857536   3.76018  3.729972  3.721172  3.718721  3.718062   3.71789    3.717847 | 125.703736 | 108.119985   102.42193  100.690808  100.189748  100.050423  100.013009    100.003268 | 100.000801 |  1.653324   1.588718  1.568913  1.563165  1.561565  1.561136  1.561024     1.560995 |  100.0 |  25.703736 | 100.000801 |
    | 2000-01-01 12:00:00 |  3.92261  3.782847  3.737205  3.723351  3.719348  3.718236  3.717937    3.717859 | 136.388647 | 112.030547  103.732402  101.103719  100.313651  100.086094  100.022912    100.005929 | 100.001496 |  1.697169   1.603655  1.573645  1.564587  1.561975  1.561249  1.561054     1.561003 |  100.0 |  36.388647 | 100.001496 |
    | 2000-01-01 12:30:00 | 4.008915  3.814777  3.747865  3.726687  3.720344  3.718522  3.718016     3.71788 | 150.102726 |  117.34411  105.594985  101.714099  100.503569  100.142672  100.039139    100.010429 | 100.002706 |  1.756134   1.624803   1.58063  1.566766  1.562624  1.561436  1.561106     1.561017 |  100.0 |  50.102726 | 100.002706 |
    | 2000-01-01 13:00:00 | 4.119816  3.858513  3.763165  3.731662  3.721881  3.718977  3.718147    3.717916 | 167.240349 | 124.386985  108.178138  102.593982  100.787199  100.230036  100.065007    100.017824 | 100.004755 |   1.83327   1.653979  1.590682  1.570018  1.563627  1.561733  1.561191     1.561041 |  100.0 |  67.240349 | 100.004755 |
    | 2000-01-01 13:30:00 | 4.258065  3.916856  3.784593  3.738904  3.724194  3.719685  3.718355    3.717976 | 188.127569 | 133.509905  111.681431  103.833731  101.200867   100.36166  100.105209    100.029665 |  100.00813 |  1.931582    1.69327  1.604808  1.574757  1.565138  1.562194  1.561327      1.56108 |  100.0 |  88.127569 |  100.00813 |
    | 2000-01-01 14:00:00 | 4.425484  3.992724  3.813918  3.749211  3.727599  3.720759  3.718681    3.718071 | 212.993962 | 145.075607  116.336168  105.544616  101.791309  100.555559  100.166246    100.048172 | 100.013555 |  2.053835   1.745001  1.624233  1.581513  1.567362  1.562895  1.561539     1.561142 |  100.0 | 112.993962 | 100.013555 |
    | 2000-01-01 14:30:00 | 4.622708   4.08898  3.853182  3.763576  3.732507  3.722354  3.719179    3.718222 | 241.948195 | 159.441754  122.404309  107.861294  102.617642  100.835375  100.256949    100.076463 | 100.022079 |  2.202347   1.811669   1.65041  1.590952   1.57057  1.563936  1.561864      1.56124 |  100.0 | 141.948195 | 100.022079 |
    | 2000-01-01 15:00:00 | 4.849057  4.208197  3.904666  3.783211  3.739442  3.724677  3.719925    3.718452 |  274.95935 | 176.939223  130.174918  110.943993  103.753491   101.23164  100.389104    100.118833 | 100.035189 |  2.378783   1.895845  1.685026  1.603895   1.57511  1.565453  1.562351      1.56139 |  100.0 |  174.95935 | 100.035189 |
    | 2000-01-01 15:30:00 | 5.102552  4.352412  3.970809  3.809558  3.749069  3.727997   3.72102      3.7188 | 311.845554 | 197.846626  139.957227  114.980056  105.289248  101.783201  100.578191    100.181093 |  100.05496 |  2.583981   2.000045  1.729985  1.621338  1.581421  1.567622  1.563066     1.561617 |  100.0 | 211.845554 |  100.05496 |
    | 2000-01-01 16:00:00 | 5.380047  4.522888  4.054102  3.844295  3.762209  3.732662  3.722601    3.719316 | 352.270855 | 222.362996  152.069479  120.184367  107.334352  102.538833  100.844252    100.270989 | 100.084235 |  2.817815   2.126574  1.787378  1.644468  1.590053  1.570672  1.564097     1.561953 |  100.0 | 252.270855 | 100.084235 |
    | 2000-01-01 16:30:00 | 5.677459  4.719921  4.156922  3.889315  3.779858  3.739115  3.724845    3.720066 | 395.750601 |  250.58136  166.823051  126.797986  110.019455  103.559018  101.212899    100.398703 | 100.126853 |   3.07912   2.277339  1.859423   1.67467  1.601681  1.574895  1.565563     1.562443 |  100.0 | 295.750601 | 100.126853 |
    | 2000-01-01 17:00:00 | 5.990049  4.942732  4.281349  3.946686  3.803208  3.747906  3.727984     3.72114 | 441.664962 | 282.466342  184.501989  135.084236  113.498205  104.917869  101.716478     100.57744 | 100.187925 |  3.365677   2.453675  1.948375  1.713524  1.617126  1.580658  1.567613     1.563144 |  100.0 | 341.664962 | 100.187925 |
    | 2000-01-01 17:30:00 | 6.073156   5.14466  4.420903  4.017303  3.833463  3.759693  3.732306    3.722656 | 489.279607 | 293.217241  201.456148  144.751481  117.868261  106.694129  102.393893    100.823928 | 100.274141 |   3.71859   2.618845  2.050443  1.761914  1.637239  1.588398  1.570439     1.564134 |  100.0 | 389.279607 | 100.274141 |
    | 2000-01-01 18:00:00 | 6.161709  5.332116  4.570818  4.101214  3.871727   3.77521  3.738168    3.724764 |  537.77215 | 305.723102   218.00469  155.583337  123.188647  108.965787  103.289758    101.158745 | 100.394085 |  4.136278   2.776738  2.162802  1.820225   1.66284  1.598615  1.574275      1.56551 |  100.0 |  437.77215 | 100.394085 |
    | 2000-01-01 18:30:00 | 6.262517   5.51143  4.728017  4.197946  3.918936  3.795247  3.745988    3.727652 | 586.262607 | 320.770367  234.577613   167.44678  129.495586  111.807542  104.453391    101.606475 | 100.558524 |  4.614164   2.931885  2.283637  1.888537  1.694679  1.611852  1.579399     1.567396 |  100.0 | 486.262607 | 100.558524 |
    | 2000-01-01 19:00:00 | 6.374106  5.687401  4.890705   4.30675  3.975834  3.820619   3.75625    3.731548 | 633.846018 | 338.316279   251.56001  180.278183  136.814277  115.290162  105.937891    102.195749 | 100.780676 |  5.146118   3.088046  2.411944  1.966772  1.733422  1.628686  1.586135     1.569943 |  100.0 | 533.846018 | 100.780676 |
    | 2000-01-01 19:30:00 | 6.494721  5.863491  5.057983  4.426779  4.042971  3.852141  3.769494    3.736728 | 679.625318 |  358.24582  269.278152  194.068801  145.167308  119.480982  107.799466    102.959209 | 101.076443 |  5.724601   3.248188  2.547321  2.054795  1.779658  1.649713  1.594848     1.573332 |  100.0 | 579.625318 | 101.076443 |
    | 2000-01-01 20:00:00 | 6.622392  6.009658  5.223687  4.556162  4.120545  3.890581  3.786307    3.743512 | 722.742733 | 380.362897  284.754081  208.336665  154.503808  124.434144  110.095539    103.933253 | 101.464579 |  6.340884   3.420284  2.684873  2.151694  1.833783  1.675523   1.60594     1.577776 |  100.0 | 622.742733 | 101.464579 |
    | 2000-01-01 20:30:00 |    6.755  6.053311  5.370025  4.690015  4.207932  3.936543  3.807298    3.752266 | 762.408181 | 404.390376    290.5198  221.447439  164.531047  130.157361  112.878474    105.156991 | 101.966721 |  6.985316   3.625251  2.809203  2.254143  1.895656  1.706625  1.619839     1.583518 |  100.0 | 662.408181 | 101.966721 |
    | 2000-01-01 21:00:00 | 6.890353  6.103975  5.502398  4.825078  4.304015  3.990403  3.833069    3.763395 | 797.923464 |  429.97634  297.486616   233.72522  175.033699  136.627373  116.192101    106.670889 | 102.607231 |  7.647621   3.863732  2.923973  2.359788  1.964787  1.743408  1.636977     1.590833 |  100.0 | 697.923464 | 102.607231 |
    | 2000-01-01 21:30:00 | 7.026254  6.161553  5.625176   4.95936  4.407451  4.052298  3.864178    3.777337 | 828.701418 | 456.705389  305.700471  245.472767  185.864343  143.802295  120.070235    108.515399 | 103.412905 |  8.317214   4.135541  3.032383  2.467083  2.040499  1.786126  1.657775     1.600018 |  100.0 | 728.701418 | 103.412905 |
    | 2000-01-01 22:00:00 | 7.160565  6.225756  5.741894  5.091831  4.516867   4.12213  3.901115    3.794546 |  854.27953 | 484.113816  315.190512  256.965087  196.933814  151.631453   124.53657    110.729745 | 104.412561 |  8.983512   4.439603  3.137189  2.575139  2.122043  1.834897  1.682628     1.611388 |  100.0 |  754.27953 | 104.412561 |
    | 2000-01-01 22:30:00 | 7.291266  6.296105  5.855365  5.222145  4.630992  4.199607  3.944279    3.815486 | 874.327871 | 511.707326    325.9573   268.44439  208.201129   160.06302  129.605526    113.350958 | 105.636577 |  9.636229   4.773953  3.240712  2.683577  2.208692   1.88972  1.711886     1.625274 |  100.0 | 774.327871 | 105.636577 |
    | 2000-01-01 23:00:00 | 7.416497  6.371943  5.967765  5.350423  4.748739  4.284287  3.993964    3.840612 | 888.651531 | 538.979932  337.967748  280.116403  219.663173  169.049648  135.283697    116.413182 | 107.116405 | 10.265638   5.135775  3.344845  2.792393  2.299796  1.950499  1.745853     1.642009 |  100.0 | 788.651531 | 107.116405 |
    | 2000-01-01 23:30:00 | 7.534597  6.452456  6.018609  5.466275  4.867303  4.375305  4.050303    3.870352 | 897.187983 | 565.432705  351.152577  285.914571  230.335012  178.397358  141.549024     119.94408 | 108.883664 | 10.862798   5.521479  3.462273  2.892436  2.393284  2.016826  1.784742     1.661917 |  100.0 | 797.187983 | 108.883664 |
    | 2000-01-02 00:00:00 | 7.644126  6.536705  6.047609   5.56838  4.984052  4.471446  4.113225    3.905078 |      900.0 | 590.591234  365.405849  289.752798  239.995207  187.898491  148.351339    123.961617 | 110.968892 | 11.419733   5.926813  3.598449     2.982  2.487058  2.088011  1.828643     1.685304 |  100.0 |      800.0 | 110.968892 |
    | 2000-01-02 00:30:00 | 7.743879  6.623654   6.08065  5.660192   5.09739  4.571369  4.182451    3.945085 | 897.264896 |  614.02091  380.586509  294.246647   248.88717  197.406839  155.624014    128.472722 | 113.400264 | 11.929573   6.346995   3.75386  3.063647  2.579723   2.16322  1.877516     1.712435 |  100.0 | 797.264896 | 113.400264 |
    | 2000-01-02 01:00:00 | 7.832895  6.712205   6.11774   5.74455    5.2065  4.673761  4.257522     3.99057 | 889.260928 | 635.339371  396.521799   299.42315  257.230347  206.828402  163.293268    133.473329 | 116.202421 | 12.386631   6.776866  3.928638  3.139594  2.670446  2.241583  1.931191     1.743522 |  100.0 | 789.260928 | 116.202421 |
    | 2000-01-02 01:30:00 | 7.910449  6.801238  6.158798  5.823787  5.311124  4.777447  4.337837    4.041616 | 876.351729 | 654.225779  413.012324  305.300889   265.21926  216.112362  171.285419     138.94939 | 119.395539 | 12.786458   7.211051  4.122518  3.211741  2.758838  2.322272  1.989396     1.778719 |  100.0 | 776.351729 | 119.395539 |
    | 2000-01-02 02:00:00 | 7.976051  6.889638  6.203658  5.899806  5.411404  4.881442  4.422699    4.098193 | 858.969595 | 670.426796  429.838404  311.886655  273.023255  225.242438  179.532282    144.878534 | 122.994686 | 13.125835    7.64411  4.334832  3.281695  2.844844   2.40455  2.051773      1.81811 |  100.0 | 758.969595 | 122.994686 |
    | 2000-01-02 02:30:00 | 8.029205  6.976363  6.252068  5.974133  5.507737  4.984976  4.511369    4.160157 |   837.5984 | 683.858341  446.773965  319.173863  280.786753  234.228812  187.974873    151.232137 | 127.009476 | 13.402567   8.070862  4.564512  3.350792  2.928648  2.487807   2.11791     1.861711 |  100.0 |   737.5984 | 127.009476 |
    | 2000-01-02 03:00:00 | 8.069387  7.060363  6.303693   6.01104  5.594403  5.086382  4.602915    4.227232 | 812.756821 | 694.203495  463.573547  327.140577  284.932404  242.495625  196.470886    157.963381 | 131.442012 | 13.615963   8.485995  4.810091  3.426767  3.005034  2.570651  2.187223     1.909449 |  100.0 | 712.756821 | 131.442012 |
    | 2000-01-02 03:30:00 | 8.097259   7.14068  6.358123  6.029603   5.67011  5.184028   4.69625    4.298999 | 784.982427 | 701.431482  479.994831  335.748966  287.356621  249.859389  204.866303    165.007308 | 136.284979 | 13.766355   8.884581  5.069734  3.513873  3.072529  2.651639   2.25897      1.96115 |  100.0 | 684.982427 | 136.284979 |
    | 2000-01-02 04:00:00 | 8.113491  7.216496  6.414886  6.050598  5.737358     5.277  4.790278    4.374908 | 754.817086 |  705.66124  495.812032  344.945785  290.154398  256.512572  213.057339    172.290386 | 141.521296 | 13.854836    9.26231  5.341293  3.612497  3.133085  2.729858   2.33235     2.016534 |  100.0 | 654.817086 | 141.521296 |
    | 2000-01-02 04:30:00 | 8.118676  7.287121  6.473462  6.074067  5.798219  5.364916  4.883997    4.454309 |  722.79402 | 707.015511  510.818529  354.663301  293.342042  262.625231  220.981559     179.73788 | 147.124867 | 13.883237   9.615465  5.622372  3.722876  3.188375  2.804817  2.406589     2.075237 |  100.0 |  622.79402 | 147.124867 |
    | 2000-01-02 05:00:00 | 8.113353  7.351982  6.533298  6.100016  5.854407  5.447768  4.976553    4.536494 | 689.426711 | 705.625173  524.829116  364.820655  296.932954  268.346166  228.609739     187.27932 | 153.062071 |  13.85408   9.940897  5.910388  3.845073  3.239831  2.876343  2.480984     2.136828 |  100.0 | 589.426711 | 153.062071 |
    | 2000-01-02 05:30:00 | 8.098023  7.410619  6.593824  6.128408  5.907341  5.525811  5.067269    4.620731 | 655.199767 | 701.630227  537.681694  375.325613  300.936029  273.804272  235.938549    194.852296 | 159.293703 | 13.770505  10.236005  6.202629  3.978975  3.288668  2.944501  2.554939     2.200834 |  100.0 | 555.199767 | 159.293703 |
    | 2000-01-02 06:00:00 | 8.073157  7.462671   6.65446   6.15917  5.958186  5.599468  5.155652    4.706305 | 620.561767 | 695.178734  549.238285  386.076633  305.354726  279.109956  242.984104    202.404826 | 165.777128 | 13.636194  10.498698   6.49632  4.124274  3.335909  3.009527  2.627982     2.266764 |  100.0 | 520.561767 | 165.777128 |
    | 2000-01-02 06:30:00 | 8.039203  7.507874  6.714632  6.192183  6.001812  5.668245  5.241203    4.792516 | 585.920022 | 686.424291  559.385344  396.965157    310.1864  283.749295  249.676436    209.880659 |  172.46602 | 13.455287  10.727368  6.788676  4.280476  3.383496  3.070859  2.699614      2.33411 |  100.0 | 485.920022 |  172.46602 |
    | 2000-01-02 07:00:00 | 7.996581  7.546047  6.773784  6.227294  6.013398  5.727646  5.322714    4.878558 |  551.63713 |  675.53907    568.0346    407.8781  315.421812    285.2374  255.545195    217.155966 | 179.300455 | 13.232263  10.920876  7.076958  4.446897  3.437828  3.124305  2.768715      2.40225 |  100.0 |  451.63713 | 179.300455 |
    | 2000-01-02 07:30:00 | 7.946198  7.577199  6.831393  6.264309  6.026479  5.779648  5.399532     4.96363 | 518.029154 |  663.02878  575.147137  418.701996  321.044923  286.945123  260.750748    224.149661 | 186.215176 | 12.971264  11.079063  7.358559   4.62268  3.499207  3.171454  2.834596     2.470532 |  100.0 | 418.029154 | 186.215176 |
    | 2000-01-02 08:00:00 | 7.889186  7.601382  6.886975  6.303004  6.041102  5.825886  5.471409    5.047014 | 485.365238 |  649.01883   580.70209  429.325114  327.032859  288.882332  265.432953    230.815042 | 193.145737 | 12.676687  11.202033  7.631037  4.806806  3.567878  3.213663  2.896909     2.538337 |  100.0 | 385.365238 | 193.145737 |
    | 2000-01-02 08:30:00 | 7.826336  7.618729  6.940092  6.343125  6.057302  5.867725  5.538385    5.128111 |  453.86844 | 633.755616  584.704379  439.640015  333.356085  291.058801   269.71314    237.132472 | 200.032857 |  12.35289  11.290325  7.892153  4.998117  3.644013  3.252085  2.955554     2.605116 |  100.0 |  353.86844 | 200.032857 |
    | 2000-01-02 09:00:00 | 7.758424  7.629435  6.990357  6.384398  6.075094  5.906297    5.6007    5.206464 | 423.717566 | 617.478019  587.182097  449.545833  339.978786  293.482798   273.69598    243.103056 | 206.825282 | 12.004122  11.344858  8.139898  5.195338  3.727706  3.287702  3.010621     2.670416 |  100.0 | 323.717566 | 206.825282 |
    | 2000-01-02 09:30:00 | 7.686199  7.633755  7.037437   6.42653  6.094474  5.942545  5.658722    5.281755 | 395.049792 | 600.413576  588.183556  458.950196   346.85944  296.160538  277.471256    248.743271 | 213.481423 | 11.634467  11.366872  8.372519  5.397101  3.818964  3.321342  3.062332     2.733887 |  100.0 | 295.049792 | 213.481423 |
    | 2000-01-02 10:00:00 | 7.610377  7.631989  7.081054  6.469217  6.115422  5.977253  5.712899      5.3538 | 367.963873 | 582.775534  587.774156  467.770746  353.951554  299.095806  281.115483    254.080479 | 219.970015 | 11.247806  11.357874  8.588525  5.601976  3.917705  3.353707  3.110995     2.795286 |  100.0 | 267.963873 | 219.970015 |
    | 2000-01-02 10:30:00 | 7.531635  7.624473  7.120986  6.512151  6.137893  6.002541  5.762309    5.422289 | 342.523746 | 564.760708  586.033216   475.93626  361.204545  302.289663  283.842029    259.008011 |  226.24727 | 10.847779  11.319584  8.786699  5.808493  4.023751  3.386913  3.155696     2.854256 |  100.0 | 242.523746 |  226.24727 |
    | 2000-01-02 11:00:00 | 7.450609   7.61157  7.157065  6.555022  6.161827  6.010553  5.804941    5.486614 | 318.762374 | 546.548081  583.050867  483.387354   368.56472  305.740221  284.869474    263.305763 | 232.240232 | 10.437765  11.253879  8.966094  6.015174  4.136835   3.42448   3.19451     2.910174 |  100.0 | 218.762374 | 232.240232 |
    | 2000-01-02 11:30:00 | 7.367888   7.59366  7.189174   6.59753   6.18714  6.019534  5.842182    5.546517 |  296.68568 | 528.298042  578.925111  490.076802  375.976318  309.442481  286.035223    267.095143 | 237.906579 | 10.020863   11.16275   9.12602  6.220554  4.256596  3.466613  3.228603     2.962713 |  100.0 |  196.68568 | 237.906579 |
    | 2000-01-02 12:00:00 | 7.284011  7.571136  7.217244  6.639383  6.213734  6.029521  5.875185    5.601992 | 276.276442 | 510.152195  573.759104   495.96951  383.382584  313.388247  287.345849    270.480737 | 243.227817 |  9.599886  11.048258  9.266042  6.423211  4.382586   3.51349   3.25896     3.011768 |  100.0 | 176.276442 | 243.227817 |
    | 2000-01-02 12:30:00 | 7.199471  7.544394  7.241251  6.680304   6.24149  6.040544  5.904912    5.653205 | 257.498082 | 492.233624  567.658711  501.042187  390.726822  317.566104  288.807844     273.55236 |  248.20355 |  9.177362  10.912492  9.385951  6.621779  4.514275  3.565254   3.28642     3.057396 |  100.0 | 157.498082 |  248.20355 |
    | 2000-01-02 13:00:00 | 7.114712  7.513831  7.261211  6.720035  6.270278  6.052623  5.932165    5.700437 | 240.298231 | 474.647535  560.730358  505.282754  397.953409  321.961473  290.427058    276.386921 | 252.846723 |  8.755534  10.757543  9.485757  6.814971  4.651061  3.622016  3.311692     3.099768 |  100.0 | 140.298231 | 252.846723 |
    | 2000-01-02 13:30:00 | 7.030127  7.479836  7.277176  6.758339  6.299953  6.065771  5.957616    5.744043 | 224.612057 | 457.482192    553.0792  508.689566  405.008727  326.556739  292.208455    279.050078 | 257.179724 |  8.336367  10.585469  9.565659  7.001597  4.792274  3.683843  3.335377     3.139135 |  100.0 | 124.612057 | 257.179724 |
    | 2000-01-02 14:00:00 | 6.946066  7.442788  7.289229  6.795001  6.330358  6.079991  5.981825    5.784416 | 210.365288 | 440.810068  544.807592  511.270473  411.841997  331.331448  294.155924    281.597707 | 261.231254 |  7.921567  10.398273  9.626025  7.180568  4.937193  3.750759  3.357982     3.175795 |  100.0 | 110.365288 | 261.231254 |
    | 2000-01-02 14:30:00 | 6.862832  7.403055  7.297481  6.829831  6.361329  6.095276  6.001208    5.821301 |  197.47692 | 424.689146  536.013863  513.041793  418.405999  336.262565  296.272131    283.672492 | 264.966386 |  7.512585  10.197889  9.667375  7.350913  5.085052  3.822742  3.380661     3.209466 |  100.0 |   97.47692 | 264.966386 |
    | 2000-01-02 15:00:00 | 6.780684  7.360987  7.302065  6.862667  6.392699  6.111608  6.006656    5.853024 | 185.861607 | 409.164314  526.791364  514.027218  424.657655  341.324795  298.558391    284.367997 | 268.204472 |   7.11064    9.98616  9.690351  7.511778  5.235051  3.899722  3.406204      3.23856 |  100.0 |  85.861607 | 268.204472 |
    | 2000-01-02 15:30:00 | 6.699842  7.316917  7.303132   6.89337  6.424292   6.12896  6.012743     5.88061 | 175.431719 | 394.268811  517.227775  514.256715   430.55849  346.490938  301.014557    285.152594 | 271.039756 |  6.716733   9.764834  9.695698  7.662435  5.386372  3.981579  3.434755     3.263963 |  100.0 |  75.431719 | 271.039756 |
    | 2000-01-02 16:00:00 | 6.620488  7.271162  7.300848  6.921826  6.455937  6.147293  6.019499    5.904916 | 166.099094 | 380.025679   507.40464  513.765437  436.074951  351.732293  303.638938    286.030691 | 273.552775 |  6.331665   9.535552  9.684249  7.802277  5.538189  4.068147   3.46645     3.286423 |  100.0 |  66.099094 | 273.552775 |
    | 2000-01-02 16:30:00 | 6.542769  7.224015   7.29539  6.947946  6.487461  6.166558  6.026948    5.926653 | 157.776495 | 366.449217  497.397102  512.592672  441.178603  357.019077  306.428239    287.006815 | 275.812254 |  5.956054   9.299846  9.656897  7.930823  5.689676  4.159211  3.501409     3.306574 |  100.0 |  57.776495 | 275.812254 |
    | 2000-01-02 17:00:00 | 6.466798  7.175752  7.286945  6.971667  6.518697  6.186699  6.035111    5.946421 | 150.378786 | 353.546419  487.273816  510.780864  445.846203  362.320855   309.37753    288.085268 | 277.876754 |  5.590353   9.059136  9.614584  8.047705  5.840022  4.254509  3.539735     3.324949 |  100.0 |  50.378786 | 277.876754 |
    | 2000-01-02 17:30:00 | 6.392659  7.126629  7.275703  6.992945  6.549482  6.207647  6.044005    5.964719 | 143.823868 | 341.318402  477.097012  508.374709  450.059672  367.606977  312.480237    289.269994 | 279.796123 |  5.234862   8.814735  9.558284  8.152672  5.988437  4.353737  3.581513     3.342002 |  100.0 |  43.823868 | 279.796123 |
    | 2000-01-02 18:00:00 | 6.320409  7.076882  7.261857  7.011759  6.579661  6.229328  6.053642    5.981968 | 138.033385 | 329.761896  466.922692  505.420333  453.805969  372.847002  315.728169    290.564475 | 281.612738 |  4.889751   8.567845  9.488989  8.245578  6.134163  4.456551  3.626808     3.358115 |  100.0 |  38.033385 | 281.612738 |
    | 2000-01-02 18:30:00 | 6.250079  7.026731  7.245601  7.028107   6.60909  6.251659  6.064031    5.998519 | 132.933235 | 318.870895  456.800954  501.964577  457.076884  378.011093  319.111567    291.971654 | 283.362583 |  4.555063   8.319571  9.407694  8.326376  6.276481   4.56257   3.67566     3.373611 |  100.0 |  32.933235 | 283.362583 |
    | 2000-01-02 19:00:00 | 6.181676  6.976375  7.227128  7.042003  6.637631  6.274551  6.075174    6.003369 | 128.453911 | 308.638806  446.776446  498.054379  459.868778  383.070397  322.619185    293.493855 | 283.947551 |  4.230731   8.070923  9.315394   8.39511  6.414718  4.671382  3.728085     3.390795 |  100.0 |  28.453911 | 283.947551 |
    | 2000-01-02 19:30:00 | 6.115182  6.926005  7.206629  7.053478  6.665162   6.29791  6.087067    6.007462 | 124.530701 | 299.061992   436.88904  493.736265  462.182261  387.997379  326.238393    295.132717 |  284.47151 |  3.916574   7.822835  9.213069  8.451907  6.548251  4.782548  3.784076     3.409985 |  100.0 |  24.530701 |  284.47151 |
    | 2000-01-02 20:00:00 | 6.050554  6.875799  7.184291  7.062578  6.691569  6.321638  6.099702    6.012003 | 121.103761 | 290.148566  427.175011  489.055985  464.021852  392.766122  329.955307    296.889135 | 285.056828 |  3.612294   7.576187  9.101683  8.496969  6.676512  4.895608  3.843595     3.431283 |  100.0 |  21.103761 | 285.056828 |
    | 2000-01-02 20:30:00 | 5.946223  6.824537  7.160209  7.069353  6.716751  6.345632  6.113063    6.017013 | 118.118088 | 277.856086  417.403803  484.039805  465.394349  397.352498  333.754934    298.763205 | 285.706697 |  3.324765   7.325001  8.981738   8.53053  6.798987  5.010083   3.90658     3.454785 |  100.0 |  18.118088 | 285.706697 |
    | 2000-01-02 21:00:00 | 5.679824  6.767136  7.134152  7.073836  6.740618   6.36979   6.12713     6.02251 | 115.523427 | 250.813963  406.641306  478.647314  466.304049  401.734018   337.62131    300.754178 | 286.424273 |  3.081242   7.044512  8.852127  8.552748  6.915213  5.125486  3.972941     3.480579 |  100.0 |  15.523427 | 286.424273 |
    | 2000-01-02 21:30:00 | 5.445157  6.705459  7.106041  7.076057   6.76309  6.394008  6.141876    6.028511 | 113.274094 | 228.366922  395.291269  472.870257  466.755066  405.889823  341.537653    302.860428 | 287.212589 |  2.874075   6.744048  8.712485  8.563756   7.02477   5.24132   4.04256     3.508745 |  100.0 |  13.274094 | 287.212589 |
    | 2000-01-02 22:00:00 | 5.238275  6.640944  7.075899  7.076046  6.784093  6.418182  6.157269    6.035029 | 111.328777 | 209.622091  383.660905  466.723011  466.752864  409.800756  345.486508    305.079421 | 288.074497 |  2.697147   6.430778  8.562974  8.563702  7.127284  5.357086  4.115292     3.539354 |  100.0 |  11.328777 | 288.074497 |
    | 2000-01-02 22:30:00 | 5.055793  6.574702  7.043823  7.073842  6.803565  6.442206  6.173273    6.042079 | 109.650285 | 193.884265  371.982086  460.235296  466.305188  413.449483  349.449916    307.407699 | 289.012624 |  2.545526   6.110202  8.404118  8.552776  7.222424  5.472285  4.190968     3.572465 |  100.0 |   9.650285 | 289.012624 |
    | 2000-01-02 23:00:00 | 4.894783  6.507588  7.009959  7.069492  6.821453   6.46598  6.189843    6.049668 | 108.205287 | 180.607096  360.428138  453.446659   465.42255  416.820635  353.409582    309.840876 |  290.02932 |  2.415203   5.786524  8.236684   8.53122  7.309908  5.586425  4.269392     3.608126 |  100.0 |   8.205287 |  290.02932 |
    | 2000-01-02 23:30:00 | 4.752704  6.440258  6.974483  7.063055  6.837713  6.489404  6.206932    6.057803 | 106.964036 | 169.357441  349.127132  446.402341   464.11838  419.900929  357.347051    312.373639 | 291.126623 |  2.302895   5.462939  8.061589  8.499331  7.389499  5.699022  4.350344      3.64637 |  100.0 |   6.964036 | 291.126623 |
    | 2000-01-03 00:00:00 | 4.627341  6.373213  6.937588  7.054601  6.852311  6.512382  6.224487    6.066489 | 105.900084 |  159.78905    338.1724  439.150247  462.408955  422.679286  361.243897    314.999769 | 292.306212 |  2.205894   5.141849  7.879829  8.457465  7.461013  5.809605  4.433584     3.687219 |  100.0 |   5.900084 | 292.306212 |
    | 2000-01-03 00:30:00 |  4.51675   6.30683  6.899478   7.04421  6.865224   6.53482  6.242454    6.075725 |  104.99001 | 151.622953  327.630952  431.738769  460.313167  425.146904  365.081899    317.712169 | 293.569374 |  2.121955   4.825035  7.692435  8.406031  7.524312  5.917722  4.518849     3.730676 |  100.0 |    4.99001 | 293.569374 |
    | 2000-01-03 01:00:00 | 4.419223  6.241387  6.860355  7.031971  6.876436  6.556631   6.26077    6.085507 | 104.213141 | 144.632724  317.550377  424.215295  457.852197  427.297297  368.843224    320.502909 | 294.916961 |    2.0492   4.513783  7.500438  8.345485  7.579311   6.02294  4.605862      3.77673 |  100.0 |   4.213141 | 294.916961 |
    | 2000-01-03 01:30:00 | 4.333253  6.177081  6.820419  7.017981  6.885943  6.577734  6.279375    6.095831 | 103.551298 |  138.63329  307.964897  416.625287  455.049149  429.126306  372.510601    323.363278 | 296.349361 |  1.986052   4.208986  7.304849  8.276322  7.625972  6.124848  4.694327     3.825355 |  100.0 |   3.551298 | 296.349361 |
    | 2000-01-03 02:00:00 | 4.257508  6.114045  6.779867  7.002343  6.893751  6.598049  6.298202    6.106685 | 102.988541 | 133.472387  298.901673   409.01187  451.928672  430.632059  376.067482    326.283854 | 297.866467 |  1.931181   3.911213  7.106651  8.199071  7.664306  6.223064  4.783938     3.876507 |  100.0 |   2.988541 | 297.866467 |
    | 2000-01-03 02:30:00 | 4.190811  6.052354  6.738893  6.985167  6.899872  6.617508  6.317184    6.118055 |  102.51094 |  129.02397  290.390841  401.416125   448.51663  431.814917  379.498197    329.254574 | 299.467651 |  1.883458   3.620753  6.906805  8.114291  7.694369  6.317234  4.874376     3.930123 |  100.0 |    2.51094 | 299.467651 |
    | 2000-01-03 03:00:00 | 4.132115  5.965118  6.696778  6.966509  6.904324  6.636046  6.336253    6.129924 | 102.106358 | 125.183094   279.83803  393.711729  444.828697  432.676649  382.788041    332.264825 | 301.151745 |   1.84192   3.342374  6.701833   8.02228  7.716243  6.407034  4.965318     3.986127 |  100.0 |   2.106358 | 301.151745 |
    | 2000-01-03 03:30:00 | 4.080493  5.706772  6.648046  6.946091  6.907107  6.653602  6.355339     6.14227 | 101.764251 | 121.861913  253.473478  384.928932  440.814793  433.215967  385.923079    335.303513 | 302.917019 |  1.805744   3.105473  6.465213  7.921685   7.72992  6.492161  5.056433     4.044422 |  100.0 |   1.764251 | 302.917019 |
    | 2000-01-03 04:00:00 | 4.035123  5.478399  6.594531  6.923777  6.908215  6.670123  6.374372     6.15507 |  101.47548 | 118.986533  231.469606  375.449711  436.454963   433.43071   388.89009    338.359136 | 304.761179 |  1.774224   2.903004  6.206049  7.811876  7.735364  6.572334  5.147389     4.104899 |  100.0 |    1.47548 | 304.761179 |
    | 2000-01-03 04:30:00 | 3.995273  5.276337  6.537661  6.899542  6.907641  6.685556  6.393284    6.168296 | 101.232151 |  116.49451  212.998247  365.570086  431.751178  433.319352  391.676599    341.419855 | 306.681348 |  1.746752   2.729297  5.931419   7.69275  7.732541  6.647291  5.237851     4.167428 |  100.0 |   1.232151 | 306.681348 |
    | 2000-01-03 05:00:00 | 3.960296  5.097433  6.478549  6.873434  6.905384  6.699854  6.412004    6.181916 | 101.027458 | 114.332858  197.410424  355.517952  426.720944  432.881995  394.270973    344.473578 | 308.674072 |  1.722802   2.579757  5.646826  7.564585  7.721451  6.716791  5.327486     4.231867 |  100.0 |   1.027458 | 308.674072 |
    | 2000-01-03 05:30:00 | 3.929616  4.938952  6.418066  6.845557  6.901453  6.712976  6.430464    6.195896 | 100.855551 | 112.456446  184.193124  345.467542  421.392353  432.120962   396.66254     347.50804 | 310.735322 |  1.701921   2.450631  5.356532   7.42792   7.70214  6.780616  5.415964     4.298058 |  100.0 |   0.855551 | 310.735322 |
    | 2000-01-03 06:00:00 | 3.902722  4.798516  6.356887  6.816049  6.895869  6.724884  6.448598    6.210199 | 100.711411 | 110.826704  172.937463  335.551099  415.800298   431.04109  398.841728      350.5109 | 312.860501 |  1.683713   2.338832  5.063827  7.283471  7.674707  6.838575   5.50296     4.365829 |  100.0 |   0.711411 | 312.860501 |
    | 2000-01-03 06:30:00 | 3.879163  4.674047  6.295534  6.785071   6.88866  6.735547   6.46634    6.224785 | 100.590745 | 109.410568  163.315008  325.868337  409.983627  429.649804  400.800185    353.469836 |  315.04446 |  1.667837   2.241804  4.771231  7.132064  7.639307  6.890504  5.588154     4.434994 |  100.0 |   0.590745 |  315.04446 |
    | 2000-01-03 07:00:00 | 3.858538  4.563721  6.234407  6.752799  6.879868   6.74494  6.483627    6.239612 | 100.489885 | 108.179621  155.060005  316.494236  403.983048  427.957037  402.530893     356.37265 | 317.281524 |  1.653996   2.157419   4.48065  6.974584  7.596154  6.936271  5.671239     4.505358 |  100.0 |   0.489885 | 317.281524 |
    | 2000-01-03 07:30:00 | 3.840493   4.46593  6.173807  6.719414  6.869546  6.753043  6.500398    6.254637 | 100.405708 | 107.109376  147.955915  307.485798  397.839675   425.97505  404.028257    359.207367 | 319.565518 |  1.641929   2.083895  4.193496  6.811947  7.545511  6.975776  5.751917     4.576714 |  100.0 |   0.405708 | 319.565518 |
    | 2000-01-03 08:00:00 | 3.824715  4.379261  6.113954    6.6851  6.857754  6.759846  6.516596    6.269812 |  100.33556 | 106.178698  141.825141  298.888767  391.594134  423.718195   405.28817    361.962343 | 321.889804 |  1.631412   2.019732  3.910781  6.645077  7.487687   7.00895  5.829903     4.648844 |  100.0 |    0.33556 | 321.889804 |
    | 2000-01-03 08:30:00 | 3.810926   4.30246  6.054997  6.650045  6.844561  6.765342  6.532166    6.285092 | 100.277186 | 105.369305  136.521115  290.747226  385.286373  421.202679  406.308049    364.626367 | 324.247321 |  1.622246   1.963659  3.633174  6.474907  7.423038   7.03576   5.90493     4.721527 |  100.0 |   0.277186 | 324.247321 |
    | 2000-01-03 09:00:00 | 3.798882  4.234417  5.986957  6.614102  6.830024  6.769531  6.547059    6.300426 | 100.228681 | 104.665362   131.92217  282.139573  378.896055  418.442455  407.086608    367.188739 | 326.630639 |   1.61426   1.914596  3.362783  6.300745  7.351854    7.0562  5.976747     4.794531 |  100.0 |   0.228681 | 326.630639 |
    | 2000-01-03 09:30:00 |  3.78837   4.17415  5.735738  6.571282  6.813839  6.772395  6.561224    6.315768 | 100.188431 | 104.053137   127.92675  256.351063  371.386489  415.383423  407.619402     369.63908 | 329.031987 |  1.607303   1.871624  3.131619  6.093681   7.27266  7.070176  6.045111     4.867624 |  100.0 |   0.188431 | 329.031987 |
    | 2000-01-03 10:00:00 | 3.779197  4.120785  5.512868  6.523373  6.795841  6.773907  6.574617    6.331065 | 100.155078 | 103.520708  124.449646  234.713436   363.11991  411.999622  407.900836    371.967269 | 331.443295 |  1.601245   1.833952  2.933145   5.86255  7.184676  7.077553  6.109792     4.940567 |  100.0 |   0.155078 | 331.443295 |
    | 2000-01-03 10:30:00 | 3.771198  4.073547  5.314931  6.471778  6.775974  6.774039  6.587192    6.346268 | 100.127477 | 103.057715  121.419024   216.45478  354.380894  408.286051  407.925592    374.163461 | 333.856222 |  1.595971   1.800901   2.76208  5.614283  7.087647  7.078202  6.170563      5.01312 |  100.0 |   0.127477 | 333.856222 |
    | 2000-01-03 11:00:00 | 3.764226  4.031745  5.138985  6.417604  6.754258  6.772772  6.598907    6.361327 | 100.104666 |  102.65515  118.774046  200.967323  345.391735   404.25298  407.689607    376.218171 | 336.262203 |   1.59138   1.771887  2.614132  5.354319  6.981697  7.072016  6.227212     5.085041 |  100.0 |   0.104666 | 336.262203 |
    | 2000-01-03 11:30:00 | 3.758152  3.994766  4.982478   6.36172  6.730767  6.770091  6.609721    6.376191 | 100.085838 | 102.305174  116.462973  187.768438  336.325147  399.921483  407.190707    378.122392 | 338.652489 |  1.587385   1.746403  2.485782  5.086917  6.867224  7.058932  6.279535     5.156087 |  100.0 |   0.085838 | 338.652489 |
    | 2000-01-03 12:00:00 | 3.752862  3.962066  4.843189  6.304808  6.705616  6.765993  6.619596     6.39081 | 100.070317 | 102.000966  114.441638  176.471786  327.314703  395.319958  406.428952    379.867721 | 341.018207 |   1.58391    1.72401  2.374128    4.8154  6.744814  7.038936  6.327345     5.226016 |  100.0 |   0.070317 | 341.018207 |
    | 2000-01-03 12:30:00 | 3.748257  3.933159  4.719177  6.247401  6.678946  6.760486  6.628501    6.405135 | 100.057539 | 101.736588  112.672199   166.76575  318.463459  390.481463  405.406796    381.446496 | 343.350412 |  1.580888   1.704326   2.27676  4.542344  6.615182   7.01207  6.370474     5.294589 |  100.0 |   0.057539 | 343.350412 |
    | 2000-01-03 13:00:00 |  3.74425  3.907615  4.608734  6.189913  6.650914  6.753588  6.636405    6.419117 | 100.047032 | 101.506866   111.12213  158.397173  309.851287  385.441741  404.129092    382.851918 | 345.640163 |   1.57826   1.687019  2.191665  4.269726  6.479119  6.978433  6.408773      5.36157 |  100.0 |   0.047032 | 345.640163 |
    | 2000-01-03 13:30:00 | 3.740765   3.88505  4.510361  6.132659  6.621686   6.74533  6.643283    6.432709 | 100.038402 | 101.307297  109.763395  151.158996  301.541639  380.237827  402.602994    384.078165 | 347.878585 |  1.575976   1.671798  2.117153  3.999038  6.337466  6.938175  6.442117     5.426729 |  100.0 |   0.038402 | 347.878585 |
    | 2000-01-03 14:00:00 | 3.737734  3.865124  4.422731   6.07587  6.591439  6.735751  6.649118    6.445865 | 100.031322 | 101.133957  108.571755  144.880766  293.589393  374.907277  400.837812    385.120479 | 350.056948 |  1.573991   1.658409  2.051796  3.731362  6.191094    6.8915  6.470408     5.489843 |  100.0 |   0.031322 | 350.056948 |
    | 2000-01-03 14:30:00 |   3.7351  3.847534   4.34467  6.019705  6.560356  6.724901  6.653894    6.458542 | 100.025521 | 100.983429  107.526213  139.421316  286.057574  369.488475  398.844892    385.975245 | 352.166739 |  1.572267   1.646632  1.994385  3.467417  6.040918  6.838659  6.493574     5.550699 |  100.0 |   0.025521 | 352.166739 |
    | 2000-01-03 15:00:00 | 3.732811  3.832012  4.275136  5.842578  6.524464  6.712591  6.657587    6.470697 | 100.020773 | 100.852738  106.608535  134.663055  267.135618  363.306658    396.5923    386.637234 | 354.199575 |  1.570769   1.636272  1.943887  3.228966   5.86781  6.778745  6.511493     5.609087 |  100.0 |   0.020773 | 354.199575 |
    | 2000-01-03 15:30:00 | 3.730823  3.818319  4.213204   5.61545  6.482698  6.698593  6.660163    6.482288 | 100.016892 | 100.739292  105.802857  130.507505  244.529444   356.21621  394.041659    387.099298 | 356.147094 |  1.569469   1.627158  1.899419  3.023727  5.666774  6.710659  6.523989     5.664801 |  100.0 |   0.016892 | 356.147094 |
    | 2000-01-03 16:00:00 | 3.729096  3.806245  4.158051  5.413166  6.436556  6.682793  6.661581    6.493274 | 100.013722 | 100.640836  105.095357  126.871768  225.404962  348.514275  391.176664    387.353876 | 358.000941 |   1.56834   1.619139  1.860221  2.846367  5.445177  6.633867   6.53087     5.717636 |  100.0 |   0.013722 | 358.000941 |
    | 2000-01-03 16:30:00 | 3.727597    3.7956  4.108943  5.232814  6.387222  6.665163  6.661804    6.503613 | 100.011136 | 100.555406  104.473967  123.685729  209.140294  340.436116  387.997559    387.394074 | 359.752826 |  1.567361   1.612086   1.82564  2.692549  5.208845  6.548256  6.531957      5.76739 |  100.0 |   0.011136 | 359.752826 |
    | 2000-01-03 17:00:00 | 3.726296  3.786219  4.065226  5.071869  6.335637   6.64574  6.660804    6.513266 | 100.009028 | 100.481293  103.928144  120.889818  195.241119   332.16715  384.516727    387.214401 | 361.394611 |   1.56651   1.605882  1.795109  2.558716  4.962378  6.454027  6.527101     5.813866 |  100.0 |   0.009028 | 361.394611 |
    | 2000-01-03 17:30:00 | 3.725167  3.777954  4.026318  4.928139  6.282544  6.624605  6.658557    6.522195 | 100.007313 |  100.41701  103.448663   118.43321  183.311342  323.853008    380.7552    386.811236 | 362.918433 |  1.565773   1.600425  1.768137  2.441936  4.709403  6.351603    6.5162     5.856873 |  100.0 |   0.007313 | 362.918433 |
    | 2000-01-03 18:00:00 | 3.724187  3.770675  3.991697  4.799708   6.22853  6.601874  6.655054    6.530362 | 100.005918 | 100.361264  103.027445  116.272375  173.031159   315.60798  376.739933    386.183097 | 364.316832 |  1.565133   1.595626  1.744296   2.33977  4.452764  6.241565  6.499202     5.896232 |  100.0 |   0.005918 | 364.316832 |
    | 2000-01-03 18:30:00 | 3.723338  3.764265  3.960899  4.684895  6.174057  6.577683  6.650293    6.537735 | 100.004785 | 100.312928  102.657407  114.369895  164.140503  307.522305  372.501735    385.330745 | 365.582878 |  1.564579   1.591406  1.723213  2.250183  4.194677  6.124602   6.47611     5.931778 |  100.0 |   0.004785 | 365.582878 |
    | 2000-01-03 19:00:00 | 3.722601  3.758624  3.933508  4.582221  6.119479  6.552184  6.644286    6.544284 | 100.003865 | 100.271026  102.332339  112.693498  156.426446  299.669034  368.073775    384.257186 | 366.710294 |  1.564098   1.587695  1.704563  2.171463  3.936841  6.001474   6.44698     5.963363 |  100.0 |   0.003865 | 366.710294 |
    | 2000-01-03 19:30:00 | 3.721963  3.753659  3.909155  4.490381  6.065065   6.52554  6.637054    6.549983 | 100.003119 | 100.234707  102.046788  111.215266  149.713531  292.112333  363.490721     382.96761 | 367.693564 |  1.563681   1.584433   1.68806  2.102167  3.680523  5.872993  6.411921     5.990856 |  100.0 |   0.003119 | 367.693564 |
    | 2000-01-03 20:00:00 | 3.721409  3.749292  3.887509  4.408218  6.011004  6.497931  6.628629     6.55481 | 100.002515 | 100.203232  101.795968   109.91098  143.856295  284.927659  358.789148    381.469344 | 368.528035 |   1.56332   1.581567  1.673453  2.041065  3.426594  5.740042  6.371097      6.01415 |  100.0 |   0.002515 | 368.528035 |
    | 2000-01-03 20:30:00 |  3.72093  3.745451  3.868273  4.334706  5.812107  6.464558  6.618763    6.558731 | 100.002026 | 100.175959  101.575672  108.759575  138.733445  264.032291  353.171617    379.720205 | 369.206869 |  1.563007   1.579047  1.660522  1.987112  3.201056  5.579591   6.32331     6.033074 |  100.0 |   0.002026 | 369.206869 |
    | 2000-01-03 21:00:00 | 3.720514  3.742073  3.851184  4.268932  5.595504  6.425514  6.607239      6.5617 | 100.001631 |  100.15233  101.382206  107.742688   134.24329  242.601786  346.692054    377.684687 | 369.721586 |  1.562735   1.576833  1.649073  1.939411  3.006011  5.392229  6.267523     6.047408 |  100.0 |   0.001631 | 369.721586 |
    | 2000-01-03 21:30:00 | 3.720154  3.739104  3.836007  4.210083   5.40195  6.382191  6.593946     6.56367 | 100.001312 | 100.131862   101.21232  106.844279  130.300134  224.371931  339.621619     375.34697 | 370.063368 |    1.5625   1.574888  1.638936   1.89719  2.836682  5.184779  6.203217     6.056918 |  100.0 |   0.001312 | 370.063368 |
    | 2000-01-03 22:00:00 | 3.719842  3.736495  3.822531  4.157433  5.228791  6.335699  6.578855    6.564595 | 100.001054 | 100.114134  101.063158  106.050308  126.831406  208.785832  332.177032    372.706197 | 370.224005 |  1.562297    1.57318  1.629959  1.859784  2.689164  4.962675  6.130265     6.061386 |  100.0 |   0.001054 | 370.224005 |
    | 2000-01-03 22:30:00 | 3.719572  3.734203   3.81057  4.110335  5.073727  6.286922  6.561995    6.564437 | 100.000846 | 100.098781  100.932211  105.348467  123.775373  195.398277  324.530866    369.772663 | 370.196569 |  1.562121    1.57168   1.62201  1.826616  2.560243  4.730237  6.048829     6.060623 |  100.0 |   0.000846 | 370.196569 |
    | 2000-01-03 23:00:00 | 3.719338  3.732189  3.799955  4.068208  4.934755  6.236564   6.54344    6.563166 | 100.000679 | 100.085486   100.81727  104.727951  121.079295  183.850562  316.820324    366.564772 |  369.97584 |  1.561968   1.570363   1.61497  1.797184  2.447254  4.490891  5.959289     6.054483 |  100.0 |   0.000679 |  369.97584 |
    | 2000-01-03 23:30:00 | 3.719135  3.730421  3.790538  4.030533  4.810122  6.185186  6.523295     6.56076 | 100.000544 | 100.073975  100.716392  104.179259  118.697931  173.851556  309.154785    363.106644 | 369.558576 |  1.561836   1.569206  1.608737   1.77105  2.347978  4.247345  5.862176      6.04287 |  100.0 |   0.000544 | 369.558576 |
    | 2000-01-04 00:00:00 |  3.71896  3.728868  3.782186  3.996847   4.69829  6.133227  6.501689    6.557211 | 100.000436 |  100.06401  100.627871  103.694029  116.592333  165.163327  301.622673    359.426308 | 368.943631 |  1.561721   1.568192  1.603218  1.747833   2.26055  4.001718   5.75813     6.025739 |  100.0 |   0.000436 | 368.943631 |
    | 2000-01-04 00:30:00 | 3.718808  3.727506   3.77478  3.966733  4.597901  6.081029  6.478766     6.55252 | 100.000349 | 100.055383  100.550204  103.264896   114.72885  157.590136   294.29881    355.554456 | 368.131994 |  1.561622   1.567301  1.598332  1.727198    2.1834  3.755643  5.647869     6.003099 |  100.0 |   0.000349 | 368.131994 |
    | 2000-01-04 01:00:00 | 3.718676  3.726309  3.768215  3.939817  4.507755  6.028847  6.454687      6.5467 | 100.000279 | 100.047916  100.482072  102.885361  113.078315  150.969954  287.256988    351.524042 | 367.126786 |  1.561536   1.566519  1.594006  1.708851  2.115195  3.510326  5.532189     5.975014 |  100.0 |   0.000279 | 367.126786 |
    | 2000-01-04 01:30:00 | 3.718563   3.72526  3.762397  3.915766  4.426787  5.898448  6.426909    6.539615 | 100.000223 | 100.041453  100.422313  102.549692  111.615374  145.167857  272.882673    346.921878 | 365.906189 |  1.561462   1.565834  1.590176  1.692532    2.0548  3.280439  5.398918     5.940842 |  100.0 |   0.000223 | 365.906189 |
    | 2000-01-04 02:00:00 | 3.718464  3.724339  3.757242  3.894279  4.354052  5.679694  6.392881    6.531001 | 100.000178 | 100.035861  100.369907  102.252825  110.317925  140.070868  250.801112    341.354379 | 364.426438 |  1.561398   1.565233  1.586787  1.678015  2.001245  3.081125  5.235922     5.899313 |  100.0 |   0.000178 | 364.426438 |
    | 2000-01-04 02:30:00 | 3.718379  3.723532  3.752676  3.875087  4.288705  5.483707  6.354052    6.520705 | 100.000142 | 100.031021  100.323956  101.990286  109.166655   135.58388  231.967294    335.097847 | 362.663772 |  1.561342   1.564705  1.583788  1.665098  1.953694  2.907635  5.050287     5.849694 |  100.0 |   0.000142 | 362.663772 |
    | 2000-01-04 03:00:00 | 3.718305  3.722824  3.748632   3.85795  4.229994  5.307888  6.311585    6.508657 | 100.000113 | 100.026833  100.283672   101.75812  108.144648  131.626433  215.821497    328.375597 | 360.609965 |  1.561294   1.564243  1.581134  1.653602  1.911427  2.756084   4.84769     5.791672 |  100.0 |   0.000113 | 360.609965 |
    | 2000-01-04 03:30:00 | 3.718241  3.722203  3.745052  3.842652  4.177245  5.149987  6.266412    6.494856 |  100.00009 | 100.023209  100.248361   101.55283   107.23706  128.130129  201.915501    321.367481 | 358.268658 |  1.561253   1.563838  1.578786   1.64337  1.873819   2.62327  4.632676      5.72525 |  100.0 |    0.00009 | 358.268658 |
    | 2000-01-04 04:00:00 | 3.718186  3.721659  3.741883  3.828997  4.129855  5.008042  6.219279    6.479349 | 100.000072 | 100.020074  100.217414  101.371321  106.430835   125.03657  189.887597    314.218157 | 355.652381 |  1.561217   1.563482  1.576709  1.634263  1.840329  2.506538  4.408881     5.650668 |  100.0 |   0.000072 | 355.652381 |
    | 2000-01-04 04:30:00 | 3.718138  3.721182  3.739079  3.816814  4.087282  4.880341  6.170785    6.462216 | 100.000057 | 100.017362  100.190297  101.210855  105.714477  122.295684  179.443752    307.044352 |  352.78019 |  1.561186   1.563171  1.574872  1.626157  1.810483  2.403672    4.1792     5.568344 |  100.0 |   0.000057 |  352.78019 |
    | 2000-01-04 05:00:00 | 3.718097  3.720764  3.736598  3.805945  4.049042  4.765377  6.121404    6.443569 | 100.000046 | 100.015016  100.166538   101.06901  105.077844  119.864369  170.343288    299.941595 | 349.675837 |  1.561159   1.562898  1.573247  1.618941  1.783867   2.31281  3.945923     5.478822 |  100.0 |   0.000046 | 349.675837 |
    | 2000-01-04 05:30:00 | 3.718061  3.720398  3.734403  3.796252  4.014699  4.661826  6.071509    6.423536 | 100.000036 | 100.012986  100.145726  100.943641  104.511977  117.705386  162.387917    292.991681 | 346.366503 |  1.561135    1.56266  1.571811  1.612518  1.760119  2.232381  3.710837     5.382748 |  100.0 |   0.000036 | 346.366503 |
    | 2000-01-04 06:00:00 | 3.718031  3.720077  3.732462  3.787609  3.983859  4.568513  6.021381    6.402268 | 100.000029 |  100.01123  100.127497  100.832847  104.008956  115.786449  155.413252    286.276588 | 342.882418 |  1.561115    1.56245  1.570541  1.606801  1.738919  2.161053  3.475283     5.280857 |  100.0 |   0.000029 | 342.882418 |
    | 2000-01-04 06:30:00 | 3.718004  3.719796  3.730746  3.779905  3.956171  4.484397  5.873528    6.376537 | 100.000023 | 100.009712  100.111534  100.734949  103.561766   114.07948  149.282216     270.31012 | 338.708288 |  1.561098   1.562267  1.569419  1.601713  1.719988  2.097688  3.257432      5.15774 |  100.0 |   0.000023 | 338.708288 |
    | 2000-01-04 07:00:00 | 3.717981   3.71955   3.72923   3.77304  3.931317  4.408553  5.664645    6.344779 | 100.000018 | 100.008398  100.097557  100.648458  103.164195  112.559987  143.879861    249.323431 | 333.619067 |  1.561083   1.562106  1.568428  1.597184  1.703075  2.041312  3.067634     5.006008 |  100.0 |   0.000018 | 333.619067 |
    | 2000-01-04 07:30:00 | 3.717961  3.719335  3.727889  3.766923  3.909011  4.340154  5.476845     6.30834 | 100.000014 | 100.007262   100.08532  100.572057  102.810732  111.206547  139.109291    231.323929 | 327.867304 |   1.56107   1.561966  1.567551  1.593155  1.687963  1.991087  2.901648      4.83223 |  100.0 |   0.000014 | 327.867304 |
    | 2000-01-04 08:00:00 | 3.717943  3.719146  3.726705  3.761474  3.888997  4.278462  5.307774     6.26831 | 100.000011 |  100.00628  100.074609  100.504578  102.496487  110.000373   134.88841    215.811228 | 321.659038 |  1.561058   1.561843  1.566778  1.589569  1.674456  1.946289  2.755986     4.641704 |  100.0 |   0.000011 | 321.659038 |
    | 2000-01-04 08:30:00 | 3.717928  3.718981  3.725658  3.756622  3.871043  4.222815   5.15539    6.225566 | 100.000009 |  100.00543  100.065234  100.444989  102.217118  108.924956  131.147339    202.382195 | 315.161911 |  1.561049   1.561735  1.566094  1.586379  1.662382  1.906288  2.627764     4.438701 |  100.0 |   0.000009 | 315.161911 |
    | 2000-01-04 09:00:00 | 3.717915  3.718837  3.724734  3.752301  3.854941  4.172621  5.017912    6.180815 | 100.000007 | 100.004695  100.057031  100.392376  101.968767  107.965752  127.826325    190.709624 | 308.512397 |   1.56104   1.561641   1.56549  1.583542  1.651587  1.870539  2.514574     4.226656 |  100.0 |   0.000007 | 308.512397 |
    | 2000-01-04 09:30:00 | 3.717904   3.71871  3.723918  3.748456  3.840503  4.127346  4.893778    6.134625 | 100.000006 |  100.00406  100.049853   100.34593  101.748005   107.10993  124.874061    180.526065 | 301.822403 |  1.561033   1.561559  1.564957  1.581018  1.641935  1.838564    2.4144     4.008322 |  100.0 |   0.000006 | 301.822403 |
    | 2000-01-04 10:00:00 | 3.717895    3.7186  3.723197  3.745034  3.827559   4.08651  4.781615    6.087452 | 100.000004 |  100.00351  100.043573  100.304934  101.551785  106.346147  122.246315    171.611453 | 295.185988 |  1.561026   1.561487  1.564487  1.578774  1.633306  1.809944  2.325543     3.785888 |  100.0 |   0.000004 | 295.185988 |
    | 2000-01-04 10:30:00 | 3.717886  3.718503  3.722561  3.741989   3.81596  4.049682  4.680209    6.039651 | 100.000004 | 100.003035   100.03808  100.268755  101.377396   105.66436  119.904811    163.783569 | 288.688805 |  1.561021   1.561423  1.564072  1.576778  1.625589  1.784311  2.246561     3.561059 |  100.0 |   0.000004 | 288.688805 |
    | 2000-01-04 11:00:00 | 3.717879  3.718419     3.722  3.739281  3.805566  4.016472  4.588484    5.962782 | 100.000003 | 100.002624  100.033276  100.236832  101.222425  105.055664  117.816311    156.890629 | 279.592601 |  1.561016   1.561368  1.563705  1.575004  1.618689  1.761342  2.176227     3.340195 |  100.0 |   0.000003 | 279.592601 |
    | 2000-01-04 11:30:00 | 3.717873  3.718345  3.721505  3.736873  3.796255  3.986529  4.505485     5.75287 | 100.000002 | 100.002269  100.029074  100.208669  101.084727  104.512157  115.951858     150.80548 | 258.062265 |  1.561012    1.56132  1.563382  1.573427   1.61252   1.74075  2.113491     3.147133 |  100.0 |   0.000002 | 258.062265 |
    | 2000-01-04 12:00:00 | 3.717867   3.71828  3.721068  3.734731  3.787917  3.959536   4.43036    5.563548 | 100.000002 | 100.001962  100.025401  100.183827  100.962392  104.026813  114.286148     145.42103 | 239.532627 |  1.561009   1.561278  1.563097  1.572026  1.607004  1.722283  2.057448     2.977732 |  100.0 |   0.000002 | 239.532627 |
    | 2000-01-04 12:30:00 | 3.717863  3.718223  3.720683  3.732828  3.780451  3.935207  4.362346    5.392553 | 100.000001 | 100.001696  100.022189  100.161918  100.853721  103.593385  112.797003    140.646622 |  223.50871 |  1.561006   1.561241  1.562845  1.570781  1.602073  1.705717  2.007319      2.82858 |  100.0 |   0.000001 |  223.50871 |
    | 2000-01-04 13:00:00 | 3.717859  3.718174  3.720343  3.731137  3.773768  3.913282   4.30076    5.237918 | 100.000001 | 100.001467  100.019381  100.142599  100.757202  103.206309  111.464939    136.405135 | 209.590577 |  1.561003   1.561209  1.562624  1.569674  1.597664  1.690851  1.962426     2.696846 |  100.0 |   0.000001 | 209.590577 |
    | 2000-01-04 13:30:00 | 3.717855  3.718131  3.720043  3.729634  3.767787  3.893529   4.24499    5.097929 | 100.000001 | 100.001268  100.016927  100.125566  100.671488  102.860629  110.272792    132.630667 | 197.452706 |  1.561001   1.561181  1.562428  1.568692  1.593724  1.677509  1.922182     2.580167 |  100.0 |   0.000001 | 197.452706 |
    | 2000-01-04 14:00:00 | 3.717852  3.718093  3.719779  3.728298  3.762436  3.875735  4.194484    4.971081 | 100.000001 | 100.001096  100.014782  100.110552  100.595381  102.551923  109.205413    129.266651 | 186.828309 |  1.560999   1.561156  1.562256  1.567819  1.590202  1.665533  1.886072     2.476556 |  100.0 |   0.000001 | 186.828309 |
    | 2000-01-04 14:30:00 | 3.717849   3.71806  3.719546  3.727113  3.757649  3.859711  4.148745    4.856051 | 100.000001 | 100.000948  100.012907  100.097319  100.527816   102.27625    108.2494    126.264337 | 177.497296 |  1.560997   1.561134  1.562104  1.567044  1.587055  1.654782  1.853645     2.384336 |  100.0 |   0.000001 | 177.497296 |
    | 2000-01-04 15:00:00 | 3.717847  3.718031  3.719341   3.72606  3.753369  3.845284  4.107324    4.751668 |      100.0 | 100.000819  100.011269  100.085658  100.467842  102.030091  107.392876    123.581547 | 169.276976 |  1.560996   1.561116   1.56197  1.566356  1.584243  1.645129  1.824505     2.302085 |  100.0 |        0.0 | 169.276976 |
    | 2000-01-04 15:30:00 | 3.717845  3.718006   3.71916  3.725125  3.749542  3.832296  4.069817    4.656895 |      100.0 | 100.000708  100.009838  100.075383  100.414616  101.810302  106.625299    121.181656 | 162.014806 |  1.560994   1.561099  1.561852  1.565746   1.58173  1.636462  1.798304     2.228585 |  100.0 |        0.0 | 162.014806 |
    | 2000-01-04 16:00:00 | 3.717844  3.717984  3.719001  3.724295  3.746121  3.820609  4.035857    4.570809 |      100.0 | 100.000612  100.008588  100.066331  100.367386  101.614077  105.937297    119.032747 | 155.582702 |  1.560993   1.561085  1.561748  1.565204  1.579486   1.62868  1.774732     2.162795 |  100.0 |        0.0 | 155.582702 |
    | 2000-01-04 16:30:00 | 3.717842  3.717965   3.71886  3.723559  3.743064  3.810093  4.005113    4.492585 |      100.0 |  100.00053  100.007496  100.058359  100.325484  101.438908  105.320525    117.106923 | 149.872546 |  1.560992   1.561072  1.561656  1.564723  1.577482  1.621693  1.753517     2.103817 |  100.0 |        0.0 | 149.872546 |
    | 2000-01-04 17:00:00 | 3.717841  3.717948  3.718737  3.722905  3.740332  3.800634  3.977284    4.421485 |      100.0 | 100.000458  100.006543  100.051337  100.288315  101.282554  104.767546    115.379722 | 144.792614 |  1.560991   1.561061  1.561576  1.564296  1.575693   1.61542  1.734415     2.050874 |  100.0 |        0.0 | 144.792614 |
    | 2000-01-04 17:30:00 |  3.71784  3.717933  3.718628  3.722326  3.737892  3.792127  3.952098    4.356846 |      100.0 | 100.000396   100.00571  100.045154  100.255349  101.143012  104.271723    113.829636 | 140.264715 |  1.560991   1.561052  1.561505  1.563918  1.574095  1.609788  1.717211     2.003291 |  100.0 |        0.0 | 140.264715 |
    | 2000-01-04 18:00:00 | 3.717839   3.71792  3.718532  3.721812  3.735714  3.784479  3.929308    4.298074 |      100.0 | 100.000342  100.004983  100.039711  100.226116  101.018489  103.827126    112.437702 |  136.22189 |   1.56099   1.561043  1.561442  1.563582  1.572669  1.604733  1.701712     1.960479 |  100.0 |        0.0 |  136.22189 |
    | 2000-01-04 18:30:00 | 3.717838  3.717909  3.718448  3.721356  3.733768  3.777605  3.908691    4.244631 |      100.0 | 100.000296  100.004348  100.034919  100.200199  100.907387  103.428457    111.187164 | 132.606553 |   1.56099   1.561036  1.561387  1.563285  1.571396  1.600195  1.687746     1.921924 |  100.0 |        0.0 | 132.606553 |
    | 2000-01-04 19:00:00 | 3.717837    3.7179  3.718373  3.720952  3.732032  3.771428  3.890042    4.196032 |      100.0 | 100.000256  100.003793  100.030701  100.177224  100.808272  103.070972    110.063181 | 129.368972 |  1.560989    1.56103  1.561339  1.563021   1.57026  1.596122   1.67516     1.887174 |  100.0 |        0.0 | 129.368972 |
    | 2000-01-04 19:30:00 | 3.717837  3.717891  3.718308  3.720594  3.730482  3.765878  3.873179    4.151838 |      100.0 | 100.000221  100.003309   100.02699  100.156861  100.719865  102.750428    109.052583 | 126.466034 |  1.560989   1.561024  1.561296  1.562787  1.569246  1.592467  1.663816      1.85583 |  100.0 |        0.0 | 126.466034 |
    | 2000-01-04 20:00:00 | 3.717836  3.717884   3.71825  3.720276  3.729099  3.760893  3.857934    4.111652 |      100.0 | 100.000191  100.002887  100.023724  100.138816  100.641021  102.463021    108.143658 | 123.860226 |  1.560988   1.561019  1.561259   1.56258  1.568342  1.589187  1.653591      1.82754 |  100.0 |        0.0 | 123.860226 |
    | 2000-01-04 20:30:00 | 3.717836  3.717877    3.7182  3.719995  3.727866  3.756417  3.844154    4.075113 |      100.0 | 100.000165  100.002518   100.02085  100.122828  100.570718  102.205343    107.325974 | 121.518795 |  1.560988   1.561015  1.561226  1.562397  1.567536  1.586245  1.644374     1.801993 |  100.0 |        0.0 | 121.518795 |
    | 2000-01-04 21:00:00 | 3.717835  3.717871  3.718155  3.719746  3.726765  3.752398  3.831702    4.041894 |      100.0 | 100.000143  100.002196  100.018323  100.108665  100.508041  101.974338    106.590226 |  119.41305 |  1.560988   1.561011  1.561197  1.562234  1.566817  1.583606  1.636065     1.778912 |  100.0 |        0.0 |  119.41305 |
    | 2000-01-04 21:30:00 | 3.717835  3.717866  3.718116  3.719525  3.725784  3.748792  3.820452    4.011697 |      100.0 | 100.000124  100.001915    100.0161   100.09612  100.452171  101.767265    105.928101 |  117.51778 |  1.560988   1.561008  1.561171   1.56209  1.566176  1.581238  1.628576      1.75805 |  100.0 |        0.0 |  117.51778 |
    | 2000-01-04 22:00:00 | 3.717835  3.717862  3.718082  3.719329   3.72491  3.745555  3.810291    3.984252 |      100.0 | 100.000107   100.00167  100.014145  100.085011  100.402377  101.581667    105.332162 | 115.810772 |  1.560987   1.561005  1.561149  1.561962  1.565605  1.579116  1.621825     1.739189 |  100.0 |        0.0 | 115.810772 |
    | 2000-01-04 22:30:00 | 3.717834  3.717858  3.718052  3.719156   3.72413  3.742652  3.801117    3.959312 |      100.0 | 100.000092  100.001456  100.012425  100.075174  100.358007  101.415335    104.795751 | 114.272398 |  1.560987   1.561003  1.561129  1.561849  1.565096  1.577212   1.61574     1.722131 |  100.0 |        0.0 | 114.272398 |
    | 2000-01-04 23:00:00 | 3.717834  3.717855  3.718025  3.719003  3.723435  3.740048  3.792835    3.936654 |      100.0 |  100.00008   100.00127  100.010914  100.066466  100.318477  101.266289    104.312898 | 112.885274 |  1.560987   1.561001  1.561112  1.561749  1.564642  1.575506  1.610256       1.7067 |  100.0 |        0.0 | 112.885274 |
    | 2000-01-04 23:30:00 | 3.717834  3.717852  3.718002  3.718867  3.722817  3.737713   3.78536    3.916073 |      100.0 | 100.000069  100.001107  100.009585  100.058758  100.283264  101.132752    103.878248 | 111.633966 |  1.560987   1.560999  1.561096  1.561661  1.564238  1.573977  1.605314      1.69274 |  100.0 |        0.0 | 111.633966 |
"""

# import...
# ...from HydPy
from hydpy.core import importtools
from hydpy.core import modeltools
from hydpy.interfaces import routinginterfaces
from hydpy.exe.modelimports import *

# ...from wq
from hydpy.models.wq import wq_control
from hydpy.models.wq import wq_model


class Model(wq_model.WidthsModel, routinginterfaces.CrossSectionModel_V1):
    """|wq_widths_strickler.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="WQ-Widths-Strickler",
        description=(
            "tabulated widths river profile submodel including Strickler-based "
            "calculations"
        ),
    )
    __HYDPY_ROOTMODEL__ = False

    INLET_METHODS = ()
    OBSERVER_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = ()
    INTERFACE_METHODS = (
        wq_model.Use_WaterDepth_V3,
        wq_model.Use_WaterLevel_V3,
        wq_model.Get_WettedArea_V2,
        wq_model.Get_SurfaceWidth_V2,
        wq_model.Get_Discharge_V1,
        wq_model.Get_Celerity_V1,
    )
    ADD_METHODS = (
        wq_model.Set_WaterDepth_V1,
        wq_model.Set_WaterLevel_V1,
        wq_model.Calc_WaterDepth_V3,
        wq_model.Calc_WaterLevel_V2,
        wq_model.Calc_Index_Excess_Weight_V1,
        wq_model.Calc_FlowWidths_V1,
        wq_model.Calc_TotalWidths_V1,
        wq_model.Calc_TotalWidth_V1,
        wq_model.Calc_FlowAreas_V1,
        wq_model.Calc_TotalAreas_V1,
        wq_model.Calc_FlowPerimeters_V1,
        wq_model.Calc_FlowPerimeterDerivatives_V1,
        wq_model.Calc_FlowArea_V1,
        wq_model.Calc_TotalArea_V1,
        wq_model.Calc_Discharges_V2,
        wq_model.Calc_Discharge_V3,
        wq_model.Calc_DischargeDerivatives_V2,
        wq_model.Calc_DischargeDerivative_V2,
        wq_model.Calc_Celerity_V2,
    )
    OUTLET_METHODS = ()
    SENDER_METHODS = ()
    SUBMODELINTERFACES = ()
    SUBMODELS = ()

    @importtools.define_targetparameter(wq_control.BottomSlope)
    def prepare_bottomslope(self, bottomslope: float) -> None:
        """Set the bottom's slope (in the longitudinal direction) [-].

        >>> from hydpy.models.wq_widths_strickler import *
        >>> parameterstep()
        >>> model.prepare_bottomslope(0.01)
        >>> bottomslope
        bottomslope(0.01)
        """
        self.parameters.control.bottomslope(bottomslope)


tester = Tester()
cythonizer = Cythonizer()
