from typing import Dict, List, Union

from belvo.resources.base import Resource


class Incomes(Resource):
    endpoint = "/api/incomes/"

    def create(
        self,
        link: str,
        *,
        token: str = None,
        save_data: bool = True,
        date_from: str = None,
        date_to: str = None,
        raise_exception: bool = False,
        **kwargs: Dict,
    ) -> Union[List[Dict], Dict]:
        """Retrieve incomes for a link

        Retrieve income insights for checking and savings accounts from a specific link. You can receive insights for a period of up to 365 days, depending on the transaction history available for each bank.

        Example:
        ```python
        # Fetch incomes for a Link
        incomes = client.Incomes.create("b91835f5-6f83-4d9b-a0ad-a5a249f18b7c")

        # Fetch incomes for a Link with and timeout after 15 seconds
        incomes = client.Incomes.create(
            "b91835f5-6f83-4d9b-a0ad-a5a249f18b7c",
            timeout=15
        )

        ```

        Args:
            link (str): The `link.id` that you want to get information for. (UUID).
            token (str, optional): The MFA token generated by the bank in order to access the institution. Defaults to None.
            save_data (bool, optional): Indicates whether or not to persist the data in Belvo. Defaults to `True`.
            raise_exception (bool, optional): Indicates whether to raise an exception or return the API error. Defaults to `False`.

        Returns:
            Union[List[Dict], Dict]: For more information on the response from the API, see our [Incomes API documentation](https://developers.belvo.com/reference/retrieveincome).
        """

        data = {"link": link, "save_data": save_data}

        if date_from:
            data.update(date_from=date_from)
        if date_to:
            data.update(date_to=date_to)
        if token:
            data.update(token=token)

        return self.session.post(
            self.endpoint, data=data, raise_exception=raise_exception, **kwargs
        )
