# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/benchmarking.ipynb (unless otherwise specified).

__all__ = ['evaluate_mAP_snn_based', 'full_evaluation']

# Cell
import numpy as np
import gc
import os
from fastprogress.fastprogress import progress_bar
from scipy.spatial.distance import cdist, hamming
from sklearn.metrics.pairwise import paired_distances
from sklearn.metrics import average_precision_score

def evaluate_mAP_snn_based(descriptors:np.array,
                           labels:np.array,
                           img_labels:np.array,
                           path_to_save_mAP: str,
                           backend:str ='numpy', distance:str ='euclidean'):
    '''Function to calculate mean average precision, over per-image based matching using Lowe SNN ratio.'''
    if os.path.isfile(path_to_save_mAP):
        print (f"Found saved results {path_to_save_mAP}, loading")
        res = np.load(path_to_save_mAP)
        return res
    backends = ['numpy', 'pytorch-cuda']
    if backend not in backends:
        raise ValueError(f'backend {backend} should one of {backends}')
    possible_distances = ['euclidean', 'hamming']
    if distance == 'euclidean':
        p=2
    elif distance == 'hamming':
        p=0
    else:
        raise ValueError(f'distance {distance} should one of {possible_distances}')
    APs = []
    unique_img_labels = sorted(np.unique(img_labels))
    for img_idx in progress_bar(unique_img_labels):
        current_batch = img_labels == img_idx
        cur_descs = descriptors[current_batch]
        if backend == 'pytorch-cuda':
            import torch
            dev = torch.device('cpu')
            try:
                if torch.cuda.is_available():
                    dev = torch.device('cuda')
            except:
                dev = torch.device('cpu')
            cur_descs = torch.from_numpy(cur_descs).to(dev)
        cur_labels = labels[current_batch]
        NN = cur_labels.shape[0]
        pos_labels_repeat = np.broadcast_to(cur_labels.reshape(1,-1),(NN,NN))
        pos_mask = (pos_labels_repeat == pos_labels_repeat.T)
        pos_mask_not_anchor = pos_mask != np.eye(NN, dtype=np.bool)
        neg_idx = np.zeros((NN), dtype=np.int32)
        if NN > 1000: # To avoid OOM, we will find hard negative in batches
            bs1 = 128
            nb = (NN // bs1)
            for i in range(nb):
                st = i*bs1
                fin = min(NN, (i+1)*bs1)
                if fin == st:
                    break
                if backend == 'pytorch-cuda':
                    dm = torch.cdist(cur_descs[st:fin], cur_descs, p=p) +\
                            1000.0 * torch.from_numpy(pos_mask[st:fin]).to(device=dev, dtype=cur_descs.dtype) + \
                            1000.0 * torch.eye(NN, device=dev, dtype=torch.bool)[st:fin].float()
                    min_neg_idxs = torch.min(dm, axis=1)[1].cpu().numpy()
                else:
                    dm = cdist(cur_descs[st:fin], cur_descs, metric=distance) +\
                            1000.0 * pos_mask[st:fin] + \
                            1000.0 * np.eye(NN, dtype=np.bool)[st:fin]
                    min_neg_idxs = np.argmin(dm, axis=1)
                neg_idx[st:fin] = min_neg_idxs
        # We want to create all possible anchor-positive combinations
        pos_idxs = np.broadcast_to(np.arange(NN).reshape(1,-1),(NN,NN))[pos_mask_not_anchor]
        anc_idxs = np.nonzero(pos_mask_not_anchor)[0]
        pos_mask = None
        neg_idxs = neg_idx[anc_idxs]
        if backend == 'pytorch-cuda':
            pos_dists = torch.nn.functional.pairwise_distance(cur_descs[anc_idxs], cur_descs[pos_idxs], p=p).detach().cpu().numpy()
            neg_dists = torch.nn.functional.pairwise_distance(cur_descs[anc_idxs], cur_descs[neg_idxs], p=2).detach().cpu().numpy()
        else:
            if distance == 'hamming':
                pos_dists = paired_distances(cur_descs[anc_idxs], cur_descs[pos_idxs], metric=hamming)
                neg_dists = paired_distances(cur_descs[anc_idxs], cur_descs[neg_idxs], metric=hamming)
            else:
                pos_dists = paired_distances(cur_descs[anc_idxs], cur_descs[pos_idxs], metric=distance)
                neg_dists = paired_distances(cur_descs[anc_idxs], cur_descs[neg_idxs], metric=distance)
        correct = pos_dists <= neg_dists
        snn = np.minimum(pos_dists,neg_dists) / np.maximum(pos_dists,neg_dists)
        snn[np.isnan(snn)] = 1.0
        ap = average_precision_score(correct, 1-snn)
        APs.append(ap)
        pos_mask = None
        pos_mask_not_anchor = None
        cur_descs = None
        pos_labels_repeat = None
        dm = None
        gc.collect()
    res = np.array(APs).mean()
    if not os.path.isdir(os.path.dirname(path_to_save_mAP)):
        os.makedirs(os.path.dirname(path_to_save_mAP))
    np.save(path_to_save_mAP, res)
    return res

# Cell
from .extraction import *
from collections import defaultdict

def full_evaluation(models,
                    desc_name: str,
                    path_to_save_dataset:str = './dataset/',
                    path_to_save_descriptors: str = './descriptors/',
                    path_to_save_mAP: str = './mAP/',
                    patch_size: int = 32,
                    device: str = 'cpu',
                    backend='numpy',
                    distance='euclidean'):
    '''Function, which performs descriptor extraction and evaluation on all datasets.
    models can be either torch.nn.Module or dict with keys ['liberty', 'notredame', 'yosemite'],
    denoting datasets, each model was trained on resp.'''
    subsets = ['liberty', 'notredame', 'yosemite']
    results = defaultdict(dict)
    if type(models) is dict:
        for learned_on in subsets:
            for subset in subsets:
                if subset == learned_on:
                    continue
                desc_dict = extract_pytorchinput_descriptors(model,
                                    desc_name + '_' + learned_on,
                                    subset= subset,
                                    path_to_save_dataset = path_to_save_dataset,
                                    path_to_save_descriptors = path_to_save_descriptors,
                                    patch_size = patch_size,
                                    device = device)
                mAP = evaluate_mAP_snn_based(desc_dict['descriptors'],
                             desc_dict['labels'],
                             desc_dict['img_idxs'],
                             path_to_save_mAP=f'{path_to_save_mAP}/{desc_name}_learned{learned_on}_{subset}.npy',
                             backend=backend,
                             distance=distance)
                results[learned_on][subset] = mAP
                print (f'{desc_name} trained on {learned_on} PS = {patch_size} mAP on {subset} = {mAP:.5f}')
    else:
        model = models
        for subset in subsets:
            desc_dict = extract_pytorchinput_descriptors(model,
                                desc_name + '_3rdparty' ,
                                subset= subset,
                                path_to_save_dataset = path_to_save_dataset,
                                path_to_save_descriptors = path_to_save_descriptors,
                                patch_size = patch_size,
                                device = device)
            mAP = evaluate_mAP_snn_based(desc_dict['descriptors'],
                         desc_dict['labels'],
                         desc_dict['img_idxs'],
                         path_to_save_mAP=f'{path_to_save_mAP}/{desc_name}_3rdparty_{subset}.npy',
                         backend=backend,
                         distance=distance)
            results[learned_on][subset] = mAP
            print (f'{desc_name} trained on 3rdparty PS = {patch_size} mAP on {subset} = {mAP:.5f}')
    return results