# Pyrogram - Telegram MTProto API Client Library for Python
# Copyright (C) 2017-2019 Dan Tès <https://github.com/delivrance>
#
# This file is part of Pyrogram.
#
# Pyrogram is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Pyrogram is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Pyrogram.  If not, see <http://www.gnu.org/licenses/>.

from io import BytesIO

from pyrogram.api.core import *


class UpdateMessagePoll(Object):
    """Attributes:
        ID: ``0xaca1657b``

    Parameters:
        poll_id: ``int`` ``64-bit``
        results: :obj:`PollResults <pyrogram.api.types.PollResults>`
        poll (optional): :obj:`Poll <pyrogram.api.types.Poll>`
    """

    __slots__ = ["poll_id", "results", "poll"]

    ID = 0xaca1657b
    QUALNAME = "types.UpdateMessagePoll"

    def __init__(self, *, poll_id: int, results, poll=None):
        self.poll_id = poll_id  # long
        self.poll = poll  # flags.0?Poll
        self.results = results  # PollResults

    @staticmethod
    def read(b: BytesIO, *args) -> "UpdateMessagePoll":
        flags = Int.read(b)
        
        poll_id = Long.read(b)
        
        poll = Object.read(b) if flags & (1 << 0) else None
        
        results = Object.read(b)
        
        return UpdateMessagePoll(poll_id=poll_id, results=results, poll=poll)

    def write(self) -> bytes:
        b = BytesIO()
        b.write(Int(self.ID, False))

        flags = 0
        flags |= (1 << 0) if self.poll is not None else 0
        b.write(Int(flags))
        
        b.write(Long(self.poll_id))
        
        if self.poll is not None:
            b.write(self.poll.write())
        
        b.write(self.results.write())
        
        return b.getvalue()
