from .gen_common import TYPE_MAP
from .db_column import DbColumn
from .utils import snake_to_camel

RS_TEMPLATE = """
/**
 * This file was generated by the extract tool.
 * Do not modify this file manually.
 * 
 */

///
/// The struct of the %(table_entity)s
/// based on the database table %(resource_name)s
///
#[derive(Debug, Clone, serde::Serialize, serde::Deserialize)]
pub struct %(table_entity)s {
%(field_defs)s
}

///
/// The source-of-truth resource name (database table name) of the %(table_entity)s
///
pub const RESOURCE_NAME: &str = "%(resource_name)s";

///
/// The data column names of the %(table_entity)s
///
pub const FIELD_NAMES: [&str; %(field_count)s] = [%(field_names)s];

///
/// The data column set of the %(table_entity)s
///
pub fn field_name_set() -> std::collections::HashSet<String> {
    FIELD_NAMES.iter().map(|s| s.to_string()).collect()
}

///
/// Get a new instance of the struct %(table_entity)s with the default values if not provided
/// # Parameters
/// `input`: The input %(table_entity)s to copy values from
///
pub fn build(input: &%(table_entity)s) -> %(table_entity)s {
    let ret: &mut %(table_entity)s = &mut %(table_entity)s {
%(assignments)s
    };
%(set_default_vals)s
    ret.clone()
}
"""


def get_rs_code(resource_name: str, column_defs: list[DbColumn]) -> str:
    """
    Generate Rust code for a database table schema

    :param resource_name: The name of the database table
    :param column_defs: The list of column definitions
    """
    field_defs = []
    field_names = []
    assignments = []
    default_vals = []
    for col in column_defs:
        normal_field_type = TYPE_MAP[col.field_type]["rs"]
        field_type = (
            normal_field_type if col.not_null else "Option<%s>" % normal_field_type
        )
        field_defs.append(f"\t{col.name}: {field_type}")
        field_names.append(f'"{col.name}"')
        assignment_value_display = ""
        if col.field_type == "TEXT":
            assignment_value_display = (
                f"{col.name}.to_string()" if col.not_null else f"{col.name}.clone()"
            )
        else:
            assignment_value_display = col.name

        assignments.append(f"\t\t\t{col.name}: input.{assignment_value_display},")
        if col.default_val is not None:
            default_val_display = (
                f'r#"{col.default_val}"#.to_string()'
                if col.field_type == "TEXT"
                else col.default_val
            )
            default_display = (
                default_val_display if col.not_null else f"Some({default_val_display})"
            )
            default_vals.append(
                f"\t\tif input.{col.name}.is_none() {{ ret.{col.name} = {default_display}; }}"
            )
    return RS_TEMPLATE % {
        "table_entity": snake_to_camel(resource_name, True),
        "field_defs": ",\n".join(field_defs),
        "field_names": ", ".join(field_names),
        "field_count": len(field_names),
        "resource_name": resource_name,
        "assignments": "\n".join(assignments),
        "set_default_vals": "\n".join(default_vals),
    }
