"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workload = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const WORKLOAD_SYMBOL = Symbol.for("@shady-island/core.Workload");
/**
 * A collection of Stacks in an Environment representing a deployment Tier.
 *
 * Derive a subclass of `Workload` and create your stacks within.
 *
 * The difference between this object and a `Stage` is that a `Stage` is meant
 * to be deployed with CDK Pipelines. This class can be used with `cdk deploy`.
 * This class also provides context loading capabilities.
 */
class Workload extends constructs_1.Construct {
    /**
     * Creates a new Workload.
     *
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param props - The constructor options.
     */
    constructor(scope, id, props) {
        super(scope, id);
        Object.defineProperty(this, WORKLOAD_SYMBOL, { value: true });
        const { env: { region, account } = {}, tier, workloadName, contextFile, baseDomainName, } = props;
        this.workloadName = workloadName ?? `${id}`.toLowerCase();
        this.tier = tier;
        this.publicDomainName = baseDomainName
            ? `${tier.id}.${this.workloadName}.${baseDomainName}`.toLowerCase()
            : undefined;
        this._stacks = new Map();
        const { region: stageRegion, account: stageAccount } = aws_cdk_lib_1.Stage.of(this);
        this.region = region || stageRegion;
        this.account = account || stageAccount;
        if (contextFile) {
            this.loadContext(contextFile);
        }
    }
    /**
     * Return the Workload the construct is contained within, fails if there is no
     * workload up the tree.
     *
     * @param construct - The construct whose parent nodes will be searched
     * @returns The Workload containing the construct
     * @throws Error - if none of the construct's parents are a workload
     */
    static of(construct) {
        const workload = construct.node.scopes
            .reverse()
            .slice(1)
            .find(Workload.isWorkload);
        if (!workload) {
            throw new Error("No workload exists in the parentage of the provided construct");
        }
        return workload;
    }
    /**
     * Test whether the given construct is a Workload.
     *
     * @param x - The value to test
     * @returns Whether the value is a Workload object.
     */
    static isWorkload(x) {
        return x !== null && typeof x === "object" && WORKLOAD_SYMBOL in x;
    }
    loadContext(filename) {
        try {
            fs.accessSync(filename, fs.constants.F_OK);
        }
        catch (err) {
            throw new Error(`Context file does not exist: ${filename}`);
        }
        try {
            fs.accessSync(filename, fs.constants.R_OK);
        }
        catch (err) {
            throw new Error(`Context file is not readable: ${filename}`);
        }
        const data = fs.readFileSync(filename, { encoding: "utf8" });
        let defaults = {};
        try {
            defaults = JSON.parse(data);
        }
        catch (e) {
            throw new Error(`Context file contains invalid JSON syntax: ${filename}`);
        }
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
    }
    /**
     * @returns The stacks created by invoking `createStack`
     */
    get stacks() {
        return [...this._stacks.values()];
    }
    /**
     * Adds a stack to the Workload.
     *
     * This method will return a `Stack` with this Workload as its scope. By
     * default, the `stackName` property provided to the `Stack` will be this
     * Workload's `workloadName`, its `tier`, and the value of the `id`
     * parameter separated by hyphens, all in lowercase.
     *
     * @example
     *
     * const exampleDev = new Workload(app, 'Example', {
     *   tier: Tier.DEVELOPMENT,
     *   env: { account: '123456789012', region: 'us-east-1' },
     * });
     * const networkStack = exampleDev.createStack('Network', {});
     * assert.strictEqual(networkStack.stackName, 'example-dev-network').
     *
     * You can override the `env` and `stackName` properties in the `props`
     * argument if desired.
     *
     * @param id - The Stack construct id (e.g. "Network")
     * @param props - The new Stack properties
     */
    createStack(id, props) {
        const { stackName, ...options } = props ?? {};
        const newStackName = stackName ?? `${this.workloadName}-${this.tier.id}-${id}`.toLowerCase();
        const stack = new aws_cdk_lib_1.Stack(this, id, {
            stackName: newStackName,
            env: this.account && this.region
                ? { account: this.account, region: this.region }
                : undefined,
            ...options,
        });
        this._stacks.set(newStackName, stack);
        return stack;
    }
}
exports.Workload = Workload;
_a = JSII_RTTI_SYMBOL_1;
Workload[_a] = { fqn: "shady-island.Workload", version: "0.0.8" };
//# sourceMappingURL=data:application/json;base64,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