from __future__ import annotations

import asyncio

from decoder.core.agent import Agent
from decoder.core.config import DecoderConfig
from decoder.core.modes import AgentMode
from decoder.core.output_formatters import create_formatter
from decoder.core.types import AssistantEvent, LLMMessage, OutputFormat, Role
from decoder.core.utils import ConversationLimitException, logger


def run_programmatic(
    config: DecoderConfig,
    prompt: str,
    max_turns: int | None = None,
    max_price: float | None = None,
    output_format: OutputFormat = OutputFormat.TEXT,
    previous_messages: list[LLMMessage] | None = None,
    mode: AgentMode = AgentMode.AUTO_APPROVE,
) -> str | None:
    """Run in programmatic mode: execute prompt and return the assistant response.

    Args:
        config: Configuration for the Decoder agent
        prompt: The user prompt to process
        max_turns: Maximum number of assistant turns (LLM calls) to allow
        max_price: Maximum cost in dollars before stopping
        output_format: Format for the output
        previous_messages: Optional messages from a previous session to continue
        mode: Operational mode (defaults to AUTO_APPROVE for programmatic use)

    Returns:
        The final assistant response text, or None if no response
    """
    formatter = create_formatter(output_format)

    agent = Agent(
        config,
        mode=mode,
        message_observer=formatter.on_message_added,
        max_turns=max_turns,
        max_price=max_price,
        enable_streaming=False,
    )
    logger.info("USER: %s", prompt)

    async def _async_run() -> str | None:
        if previous_messages:
            non_system_messages = [
                msg for msg in previous_messages if not (msg.role == Role.system)
            ]
            agent.messages.extend(non_system_messages)
            logger.info(
                "Loaded %d messages from previous session", len(non_system_messages)
            )

        async for event in agent.act(prompt):
            formatter.on_event(event)
            if isinstance(event, AssistantEvent) and event.stopped_by_middleware:
                raise ConversationLimitException(event.content)

        return formatter.finalize()

    return asyncio.run(_async_run())
