"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TreeMetadata = void 0;
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const annotations_1 = require("../annotations");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
const tree_1 = require("../tree");
const FILE_PATH = 'tree.json';
/**
 * Symbol for accessing jsii runtime information
 *
 * Introduced in jsii 1.19.0, cdk 1.90.0.
 */
const JSII_RUNTIME_SYMBOL = Symbol.for('jsii.rtti');
/**
 * Construct that is automatically attached to the top-level `App`.
 * This generates, as part of synthesis, a file containing the construct tree and the metadata for each node in the tree.
 * The output is in a tree format so as to preserve the construct hierarchy.
 *
 * @experimental
 */
class TreeMetadata extends construct_compat_1.Construct {
    constructor(scope) {
        super(scope, 'Tree');
    }
    /**
     * Create tree.json
     * @internal
     */
    _synthesizeTree(session) {
        const lookup = {};
        const visit = (construct) => {
            const children = construct.node.children.map((c) => {
                try {
                    return visit(c);
                }
                catch (e) {
                    annotations_1.Annotations.of(this).addWarning(`Failed to render tree metadata for node [${c.node.id}]. Reason: ${e}`);
                    return undefined;
                }
            });
            const childrenMap = children
                .filter((child) => child !== undefined)
                .reduce((map, child) => Object.assign(map, { [child.id]: child }), {});
            const jsiiRuntimeInfo = Object.getPrototypeOf(construct).constructor[JSII_RUNTIME_SYMBOL];
            const node = {
                id: construct.node.id || 'App',
                path: construct.node.path,
                children: Object.keys(childrenMap).length === 0 ? undefined : childrenMap,
                attributes: this.synthAttributes(construct),
                constructInfo: constructInfoFromRuntimeInfo(jsiiRuntimeInfo),
            };
            lookup[node.path] = node;
            return node;
        };
        const tree = {
            version: 'tree-0.1',
            tree: visit(this.node.root),
        };
        const builder = session.assembly;
        fs.writeFileSync(path.join(builder.outdir, FILE_PATH), JSON.stringify(tree, undefined, 2), { encoding: 'utf-8' });
        builder.addArtifact('Tree', {
            type: cloud_assembly_schema_1.ArtifactType.CDK_TREE,
            properties: {
                file: FILE_PATH,
            },
        });
    }
    synthAttributes(construct) {
        // check if a construct implements IInspectable
        function canInspect(inspectable) {
            return inspectable.inspect !== undefined;
        }
        const inspector = new tree_1.TreeInspector();
        // get attributes from the inspector
        if (canInspect(construct)) {
            construct.inspect(inspector);
            return stack_1.Stack.of(construct).resolve(inspector.attributes);
        }
        return undefined;
    }
}
exports.TreeMetadata = TreeMetadata;
function constructInfoFromRuntimeInfo(jsiiRuntimeInfo) {
    if (typeof jsiiRuntimeInfo === 'object'
        && jsiiRuntimeInfo !== null
        && typeof jsiiRuntimeInfo.fqn === 'string'
        && typeof jsiiRuntimeInfo.version === 'string') {
        return { fqn: jsiiRuntimeInfo.fqn, version: jsiiRuntimeInfo.version };
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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