import math
from typing import Sequence

def index_price_weighted(prices: list[float], D: float) -> float:
    """
    Calculate price-weighted index value.
    
    A price-weighted index is calculated as the sum of all stock prices
    divided by a divisor (which adjusts for stock splits, etc.).
    
    Args:
        prices: List of current stock prices
        D: Divisor (adjusted for corporate actions)
        
    Returns:
        Price-weighted index value
        
    Example:
        >>> index_price_weighted([100.0, 50.0, 25.0], 3.0)
        58.333...
    """
    return sum(prices) / D

def index_cap_weighted(prev_index: float,
                       caps_now: list[float],
                       caps_prev: list[float],
                       J: float=0.0) -> float:
    """
    Calculate capitalization-weighted index value.
    
    A cap-weighted index reflects changes in total market capitalization
    of the constituent stocks.
    
    Args:
        prev_index: Previous period index value
        caps_now: Current market capitalizations of all stocks
        caps_prev: Previous market capitalizations of all stocks
        J: Adjustment factor for new issues/deletions (default 0.0)
        
    Returns:
        New capitalization-weighted index value
        
    Example:
        >>> index_cap_weighted(1000.0, [500.0, 300.0], [400.0, 250.0], 0.0)
        1230.769...
    """
    return prev_index * (sum(caps_now) / (sum(caps_prev) + J))

def index_value_line_geo(prev_index: float,
                         prices_now: Sequence[float],
                         prices_prev: Sequence[float]) -> float:
    """
    Calculate Value Line geometric index.
    
    The Value Line geometric index uses the geometric mean of individual
    stock price relatives, giving equal weight to each stock's percentage change.
    
    Args:
        prev_index: Previous period index value
        prices_now: Current stock prices
        prices_prev: Previous stock prices
        
    Returns:
        New Value Line geometric index value
        
    Example:
        >>> index_value_line_geo(100.0, [110.0, 55.0], [100.0, 50.0])
        104.880...
    """
    n = len(prices_now)
    product = 1.0
    for c_now, c_prev in zip(prices_now, prices_prev):
        product *= c_now / c_prev
    return prev_index * (product ** (1.0 / n))

def index_value_line_arith(prev_index: float,
                           prices_now: Sequence[float],
                           prices_prev: Sequence[float]) -> float:
    """
    Calculate Value Line arithmetic index.
    
    The Value Line arithmetic index uses the arithmetic mean of individual
    stock price relatives, giving equal weight to each stock's percentage change.
    
    Args:
        prev_index: Previous period index value
        prices_now: Current stock prices
        prices_prev: Previous stock prices
        
    Returns:
        New Value Line arithmetic index value
        
    Example:
        >>> index_value_line_arith(100.0, [110.0, 55.0], [100.0, 50.0])
        105.0
    """
    n = len(prices_now)
    ratios = [(c_now / c_prev) for c_now, c_prev in zip(prices_now, prices_prev)]
    return prev_index * (sum(ratios) / n)