import math

def forward_price_no_div(S: float, r: float, tau: float) -> float:
    """
    Calculate forward price for an asset with no dividends.
    
    This implements the basic forward pricing formula: F = S * e^(r*tau)
    
    Args:
        S: Current spot price of the underlying asset
        r: Risk-free interest rate (as decimal, e.g., 0.05 for 5%)
        tau: Time to expiration in years
        
    Returns:
        Forward price
        
    Example:
        >>> forward_price_no_div(100.0, 0.05, 1.0)
        105.127...
    """
    return S * math.exp(r * tau)

def forward_price_with_div(S: float, D: float, r: float, tau: float) -> float:
    """
    Calculate forward price for an asset with known discrete dividend.
    
    This implements the forward pricing formula with discrete dividends:
    F = (S - D) * e^(r*tau) where D is the present value of dividends.
    
    Args:
        S: Current spot price of the underlying asset
        D: Present value of discrete dividends to be paid during the life of the forward
        r: Risk-free interest rate (as decimal, e.g., 0.05 for 5%)
        tau: Time to expiration in years
        
    Returns:
        Forward price
        
    Example:
        >>> forward_price_with_div(100.0, 2.0, 0.05, 1.0)
        103.024...
    """
    return (S - D) * math.exp(r * tau)

def forward_price_cont_yield(S: float, r: float, q: float, tau: float) -> float:
    """
    Calculate forward price for an asset with continuous dividend yield.
    
    This implements the forward pricing formula with continuous dividend yield:
    F = S * e^((r-q)*tau) where q is the continuous dividend yield.
    
    Args:
        S: Current spot price of the underlying asset
        r: Risk-free interest rate (as decimal, e.g., 0.05 for 5%)
        q: Continuous dividend yield (as decimal, e.g., 0.02 for 2%)
        tau: Time to expiration in years
        
    Returns:
        Forward price
        
    Example:
        >>> forward_price_cont_yield(100.0, 0.05, 0.02, 1.0)
        103.045...
    """
    return S * math.exp((r - q) * tau)
