import math


def roll_forward_cont(P_t:float, r:float, tau:float) -> float:
    """
    Roll a value forward in time using continuous compounding.
    
    This function calculates the future value of a present amount using
    continuous compounding: FV = PV * e^(r*tau)
    
    Args:
        P_t: Present value at time t
        r: Continuous interest rate (as decimal, e.g., 0.05 for 5%)
        tau: Time period to roll forward (in years)
        
    Returns:
        Future value after rolling forward
        
    Example:
        >>> roll_forward_cont(1000.0, 0.05, 2.0)
        1105.170...
    """
    return P_t * math.exp(r * tau)

def roll_back_cont(P_t:float, r:float, tau:float) -> float:
    """
    Discount a value back in time using continuous compounding.
    
    This function calculates the present value of a future amount using
    continuous discounting: PV = FV * e^(-r*tau)
    
    Args:
        P_t: Future value at time t
        r: Continuous interest rate (as decimal, e.g., 0.05 for 5%)
        tau: Time period to discount back (in years)
        
    Returns:
        Present value after discounting back
        
    Example:
        >>> roll_back_cont(1000.0, 0.05, 2.0)
        904.837...
    """
    return P_t * math.exp(-r * tau)