# coding: utf8

import clinica.engine as ce


class T1FreeSurferCLI(ce.CmdParser):

    def define_name(self):
        """Define the sub-command name to run this pipeline."""
        self._name = 't1-freesurfer'

    def define_description(self):
        """Define a description of this pipeline."""
        self._description = ('Cross-sectional pre-processing of T1w images with FreeSurfer:\n'
                             'http://clinica.run/doc/Pipelines/T1_FreeSurfer/')

    def define_options(self):
        """Define the sub-command arguments."""
        from clinica.engine.cmdparser import PIPELINE_CATEGORIES
        # Clinica compulsory arguments (e.g. BIDS, CAPS, group_id)
        clinica_comp = self._args.add_argument_group(PIPELINE_CATEGORIES['CLINICA_COMPULSORY'])
        clinica_comp.add_argument("bids_directory",
                                  help='Path to the BIDS directory.')
        clinica_comp.add_argument("caps_directory",
                                  help='Path to the CAPS directory.')

        # Optional arguments (e.g. FWHM)
        optional = self._args.add_argument_group(PIPELINE_CATEGORIES['OPTIONAL'])
        optional.add_argument("-raa", "--recon_all_args",
                              metavar='flag(s)', type=str, default="-qcache",
                              help='Additional flags for recon-all command line '
                                   '(default: --recon_all_args="%(default)s"). '
                                   'Please note that = is compulsory after --recon_all_args/-raa flag '
                                   '(this is not the case for other flags).')

        # Clinica standard arguments (e.g. --n_procs)
        clinica_opt = self._args.add_argument_group(PIPELINE_CATEGORIES['CLINICA_OPTIONAL'])
        clinica_opt.add_argument("-tsv", "--subjects_sessions_tsv",
                                 help='TSV file containing a list of subjects with their sessions.')
        clinica_opt.add_argument("-wd", "--working_directory",
                                 help='Temporary directory to store pipelines intermediate results.')
        clinica_opt.add_argument("-np", "--n_procs",
                                 metavar='N', type=int,
                                 help='Number of cores used to run in parallel.')

    def run_command(self, args):
        """Run the pipeline with defined args."""
        import os
        from colorama import Fore
        from tempfile import mkdtemp
        from .t1_freesurfer_pipeline import T1FreeSurfer
        from clinica.iotools.grabcaps import CAPSLayout
        from clinica.utils.io import read_participant_tsv, extract_image_ids
        from clinica.utils.exceptions import ClinicaException
        from clinica.utils.stream import cprint
        from clinica.utils.ux import print_end_pipeline, print_failed_images
        from clinica.utils.inputs import clinica_file_reader

        pipeline = T1FreeSurfer(
            bids_directory=self.absolute_path(args.bids_directory),
            caps_directory=self.absolute_path(args.caps_directory),
            tsv_file=self.absolute_path(args.subjects_sessions_tsv)
        )

        if "-dontrun" in args.recon_all_args.split(' '):
            cprint('%s[Warning] Found -dontrun flag for FreeSurfer recon-all. '
                   'Please note that this will not run the segmentation.%s' %
                   (Fore.YELLOW, Fore.RESET))

        pipeline.parameters = {
            'recon_all_args': args.recon_all_args or '-qcache'
        }

        if args.working_directory is None:
            args.working_directory = mkdtemp()
        pipeline.base_dir = self.absolute_path(args.working_directory)

        try:
            if args.n_procs:
                pipeline.run(plugin='MultiProc', plugin_args={'n_procs': args.n_procs})
            else:
                pipeline.run()
        except RuntimeError as e:
            # Check that it is a Nipype error
            if 'Workflow did not execute cleanly. Check log for details' in str(e):

                # Extract subject IDs used by the pipeline
                tsv_file = os.path.join(pipeline.base_dir, pipeline.__class__.__name__, 'participants.tsv')
                [participant_ids, session_ids] = read_participant_tsv(tsv_file)

                # Extract subject IDs generated by the pipeline on CAPS folder
                caps_files = clinica_file_reader(participant_ids,
                                                 session_ids,
                                                 self.absolute_path(args.caps_directory),
                                                 {'pattern': 'sub-*_ses-*/scripts/recon-all.log',
                                                  'description': 'recon all log'},
                                                 raise_exception=False)

                # Extract missing subject IDs
                input_ids = [participant_id + '_' + session_id
                             for participant_id, session_id in zip(participant_ids, session_ids)]
                output_ids = extract_image_ids(caps_files)
                missing_ids = list(set(input_ids) - set(output_ids))

                print_failed_images(self._name, missing_ids, args.logname)

                cprint('%sIf your pipeline crashed due to lack of space of network issues, '
                       're-run the pipeline with the working directory (-wd %s).\n'
                       'Otherwise, you can delete it.%s' %
                       (Fore.YELLOW, os.path.abspath(args.working_directory), Fore.RESET))
                # Force the display of "Documentation can be found..."
                raise ClinicaException('')
            else:
                raise e
        else:
            print_end_pipeline(self._name, os.path.join(pipeline.base_dir, pipeline.__class__.__name__))
