import { IResource, Resource, SecretValue } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IUser } from './user';
/**
 * Valid statuses for an IAM Access Key.
 *
 * @stability stable
 */
export declare enum AccessKeyStatus {
    /**
     * An active access key.
     *
     * An active key can be used to make API calls.
     *
     * @stability stable
     */
    ACTIVE = "Active",
    /**
     * An inactive access key.
     *
     * An inactive key cannot be used to make API calls.
     *
     * @stability stable
     */
    INACTIVE = "Inactive"
}
/**
 * Represents an IAM Access Key.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_access-keys.html
 * @stability stable
 */
export interface IAccessKey extends IResource {
    /**
     * The Access Key ID.
     *
     * @stability stable
     * @attribute true
     */
    readonly accessKeyId: string;
    /**
     * The Secret Access Key.
     *
     * @stability stable
     * @attribute true
     */
    readonly secretAccessKey: SecretValue;
}
/**
 * Properties for defining an IAM access key.
 *
 * @stability stable
 */
export interface AccessKeyProps {
    /**
     * A CloudFormation-specific value that signifies the access key should be replaced/rotated.
     *
     * This value can only be incremented. Incrementing this
     * value will cause CloudFormation to replace the Access Key resource.
     *
     * @default - No serial value
     * @stability stable
     */
    readonly serial?: number;
    /**
     * The status of the access key.
     *
     * An Active access key is allowed to be used
     * to make API calls; An Inactive key cannot.
     *
     * @default - The access key is active
     * @stability stable
     */
    readonly status?: AccessKeyStatus;
    /**
     * The IAM user this key will belong to.
     *
     * Changing this value will result in the access key being deleted and a new
     * access key (with a different ID and secret value) being assigned to the new
     * user.
     *
     * @stability stable
     */
    readonly user: IUser;
}
/**
 * Define a new IAM Access Key.
 *
 * @stability stable
 */
export declare class AccessKey extends Resource implements IAccessKey {
    /**
     * The Access Key ID.
     *
     * @stability stable
     */
    readonly accessKeyId: string;
    /**
     * The Secret Access Key.
     *
     * @stability stable
     */
    readonly secretAccessKey: SecretValue;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: AccessKeyProps);
}
