"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ManagedPolicy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const iam_generated_1 = require("./iam.generated");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * Managed policy.
 *
 * @stability stable
 */
class ManagedPolicy extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.managedPolicyName,
        });
        /**
         * The policy document.
         *
         * @stability stable
         */
        this.document = new policy_document_1.PolicyDocument();
        this.roles = new Array();
        this.users = new Array();
        this.groups = new Array();
        jsiiDeprecationWarnings._aws_cdk_aws_iam_ManagedPolicyProps(props);
        this.description = props.description || '';
        this.path = props.path || '/';
        if (props.document) {
            this.document = props.document;
        }
        const resource = new iam_generated_1.CfnManagedPolicy(this, 'Resource', {
            policyDocument: this.document,
            managedPolicyName: this.physicalName,
            description: this.description,
            path: this.path,
            roles: util_1.undefinedIfEmpty(() => this.roles.map(r => r.roleName)),
            users: util_1.undefinedIfEmpty(() => this.users.map(u => u.userName)),
            groups: util_1.undefinedIfEmpty(() => this.groups.map(g => g.groupName)),
        });
        if (props.users) {
            props.users.forEach(u => this.attachToUser(u));
        }
        if (props.groups) {
            props.groups.forEach(g => this.attachToGroup(g));
        }
        if (props.roles) {
            props.roles.forEach(r => this.attachToRole(r));
        }
        if (props.statements) {
            props.statements.forEach(p => this.addStatements(p));
        }
        // arn:aws:iam::123456789012:policy/teststack-CreateTestDBPolicy-16M23YE3CS700
        this.managedPolicyName = this.getResourceNameAttribute(core_1.Stack.of(this).splitArn(resource.ref, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName);
        this.managedPolicyArn = this.getResourceArnAttribute(resource.ref, {
            region: '',
            service: 'iam',
            resource: 'policy',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a customer managed policy from the managedPolicyName.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     * @stability stable
     */
    static fromManagedPolicyName(scope, id, managedPolicyName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = core_1.Stack.of(scope).formatArn({
                    service: 'iam',
                    region: '',
                    account: core_1.Stack.of(scope).account,
                    resource: 'policy',
                    resourceName: managedPolicyName,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an external managed policy by ARN.
     *
     * For this managed policy, you only need to know the ARN to be able to use it.
     * This can be useful if you got the ARN from a CloudFormation Export.
     *
     * If the imported Managed Policy ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * managed policy has a `path` (like `arn:...:policy/AdminPolicy/AdminAllow`), the
     * `managedPolicyName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Managed Policy ARN should be supplied without the
     * `path` in order to resolve the correct managed policy resource.
     *
     * @param scope construct scope.
     * @param id construct id.
     * @param managedPolicyArn the ARN of the managed policy to import.
     * @stability stable
     */
    static fromManagedPolicyArn(scope, id, managedPolicyArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = managedPolicyArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a managed policy from one of the policies that AWS manages.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     * Some managed policy names start with "service-role/", some start with
     * "job-function/", and some don't start with anything. Include the
     * prefix when constructing this object.
     *
     * @stability stable
     */
    static fromAwsManagedPolicyName(managedPolicyName) {
        class AwsManagedPolicy {
            constructor() {
                this.managedPolicyArn = core_1.Arn.format({
                    partition: core_1.Aws.PARTITION,
                    service: 'iam',
                    region: '',
                    account: 'aws',
                    resource: 'policy',
                    resourceName: managedPolicyName,
                });
            }
        }
        return new AwsManagedPolicy();
    }
    /**
     * Adds a statement to the policy document.
     *
     * @stability stable
     */
    addStatements(...statement) {
        jsiiDeprecationWarnings._aws_cdk_aws_iam_PolicyStatement(statement);
        this.document.addStatements(...statement);
    }
    /**
     * Attaches this policy to a user.
     *
     * @stability stable
     */
    attachToUser(user) {
        jsiiDeprecationWarnings._aws_cdk_aws_iam_IUser(user);
        if (this.users.find(u => u === user)) {
            return;
        }
        this.users.push(user);
    }
    /**
     * Attaches this policy to a role.
     *
     * @stability stable
     */
    attachToRole(role) {
        jsiiDeprecationWarnings._aws_cdk_aws_iam_IRole(role);
        if (this.roles.find(r => r === role)) {
            return;
        }
        this.roles.push(role);
    }
    /**
     * Attaches this policy to a group.
     *
     * @stability stable
     */
    attachToGroup(group) {
        jsiiDeprecationWarnings._aws_cdk_aws_iam_IGroup(group);
        if (this.groups.find(g => g === group)) {
            return;
        }
        this.groups.push(group);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        const result = new Array();
        // validate that the policy document is not empty
        if (this.document.isEmpty) {
            result.push('Managed Policy is empty. You must add statements to the policy');
        }
        result.push(...this.document.validateForIdentityPolicy());
        return result;
    }
}
exports.ManagedPolicy = ManagedPolicy;
_a = JSII_RTTI_SYMBOL_1;
ManagedPolicy[_a] = { fqn: "@aws-cdk/aws-iam.ManagedPolicy", version: "1.142.0" };
//# sourceMappingURL=data:application/json;base64,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