from .core import Sparse

class Observer(object):
    '''
    Container attached to a ``Model`` that allows to retrieve output spikes and
    potentials for a given layer.

    '''
    def __init__(self, model, layer):
        self._model = model
        def observer(layer, source_id, spikes, potentials):
            try:
                self._spikes[source_id] = spikes.chip(0, source_id)
                # Convert CHW potentials Col Major Tensor
                # to a WHC Row Major numpy array
                self._potentials[source_id] = potentials.to_numpy()
            except Exception as e:
                # We swallow any python exception because otherwise it would
                # crash the calling library
                print("Exception in observer callback: " + str(e))
        self._id = model.register_observer(layer, observer)
        self._spikes = {}
        self._potentials = {}

    def __del__(self):
        self._model.unregister_observer(self._id)

    @property
    def spikes(self):
        """Get generated spikes.

        Returns a dictionary of spikes generated by the attached layer indexed
        by their source id.

        :return: a dictionary of ``Sparse`` objects

        """
        return self._spikes

    @property
    def potentials(self):
        """Get generated potentials.

        Returns a dictionary of potentials generated by the attached layer
        indexed by their source id.

        :return: a dictionary of numpy.ndarray objects of size (w, h, c)

        """
        return self._potentials

    def clear(self):
        """Clear spikes and potentials lists.

        """
        self._spikes = {}
        self._potentials = {}
