"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomLambdaDeploymentConfig = exports.CustomLambdaDeploymentConfigType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const utils_1 = require("../utils");
/**
 * Lambda Deployment config type.
 *
 * @stability stable
 */
var CustomLambdaDeploymentConfigType;
(function (CustomLambdaDeploymentConfigType) {
    CustomLambdaDeploymentConfigType["CANARY"] = "Canary";
    CustomLambdaDeploymentConfigType["LINEAR"] = "Linear";
})(CustomLambdaDeploymentConfigType = exports.CustomLambdaDeploymentConfigType || (exports.CustomLambdaDeploymentConfigType = {}));
/**
 * A custom Deployment Configuration for a Lambda Deployment Group.
 *
 * @stability stable
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class CustomLambdaDeploymentConfig extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.validateParameters(props);
        // In this section we make the argument for the AWS API call
        const deploymentType = 'TimeBased' + props.type.toString();
        const intervalMinutes = props.interval.toMinutes().toString();
        const percentage = props.percentage.toString();
        let routingConfig; // The argument to the AWS API call
        if (props.type == CustomLambdaDeploymentConfigType.CANARY) {
            routingConfig = {
                type: deploymentType,
                timeBasedCanary: {
                    canaryInterval: intervalMinutes,
                    canaryPercentage: percentage,
                },
            };
        }
        else if (props.type == CustomLambdaDeploymentConfigType.LINEAR) {
            routingConfig = {
                type: deploymentType,
                timeBasedLinear: {
                    linearInterval: intervalMinutes,
                    linearPercentage: percentage,
                },
            };
        }
        // Generates the name of the deployment config. It's also what you'll see in the AWS console
        // The name of the config is <construct unique id>.Lambda<deployment type><percentage>Percent<interval>Minutes
        // Unless the user provides an explicit name
        this.deploymentConfigName = (_b = props.deploymentConfigName) !== null && _b !== void 0 ? _b : `${core_1.Names.uniqueId(this)}.Lambda${props.type}${props.percentage}Percent${props.type === CustomLambdaDeploymentConfigType.LINEAR
            ? 'Every'
            : ''}${props.interval.toMinutes()}Minutes`;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
        // The AWS Custom Resource that calls CodeDeploy to create and delete the resource
        new custom_resources_1.AwsCustomResource(this, 'DeploymentConfig', {
            onCreate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // This `resourceName` is the initial physical ID of the config
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onUpdate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // If `resourceName` is different from the last stack update (or creation),
                // the old config gets deleted and the new one is created
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onDelete: {
                service: 'CodeDeploy',
                action: 'deleteDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                },
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
    }
    // Validate the inputs. The percentage/interval limits come from CodeDeploy
    validateParameters(props) {
        if (!(1 <= props.percentage && props.percentage <= 99)) {
            throw new Error(`Invalid deployment config percentage "${props.percentage.toString()}". \
        Step percentage must be an integer between 1 and 99.`);
        }
        if (props.interval.toMinutes() > 2880) {
            throw new Error(`Invalid deployment config interval "${props.interval.toString()}". \
        Traffic shifting intervals must be positive integers up to 2880 (2 days).`);
        }
    }
}
exports.CustomLambdaDeploymentConfig = CustomLambdaDeploymentConfig;
_a = JSII_RTTI_SYMBOL_1;
CustomLambdaDeploymentConfig[_a] = { fqn: "@aws-cdk/aws-codedeploy.CustomLambdaDeploymentConfig", version: "1.124.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3VzdG9tLWRlcGxveW1lbnQtY29uZmlnLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY3VzdG9tLWRlcGxveW1lbnQtY29uZmlnLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsd0NBQTBEO0FBQzFELGdFQUEyRztBQUUzRyxvQ0FBa0Q7Ozs7OztBQUlsRCxJQUFZLGdDQU1YO0FBTkQsV0FBWSxnQ0FBZ0M7SUFFMUMscURBQWlCLENBQUE7SUFHakIscURBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQU5XLGdDQUFnQyxHQUFoQyx3Q0FBZ0MsS0FBaEMsd0NBQWdDLFFBTTNDOzs7Ozs7O0FBbUJELE1BQWEsNEJBQTZCLFNBQVEsZUFBUTs7OztJQVF4RCxZQUFtQixLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF3Qzs7UUFDdkYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFL0IsNERBQTREO1FBQzVELE1BQU0sY0FBYyxHQUFHLFdBQVcsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQzNELE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDOUQsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUMvQyxJQUFJLGFBQWEsQ0FBQyxDQUFDLG1DQUFtQztRQUN0RCxJQUFJLEtBQUssQ0FBQyxJQUFJLElBQUksZ0NBQWdDLENBQUMsTUFBTSxFQUFFO1lBQ3pELGFBQWEsR0FBRztnQkFDZCxJQUFJLEVBQUUsY0FBYztnQkFDcEIsZUFBZSxFQUFFO29CQUNmLGNBQWMsRUFBRSxlQUFlO29CQUMvQixnQkFBZ0IsRUFBRSxVQUFVO2lCQUM3QjthQUNGLENBQUM7U0FDSDthQUFNLElBQUksS0FBSyxDQUFDLElBQUksSUFBSSxnQ0FBZ0MsQ0FBQyxNQUFNLEVBQUU7WUFDaEUsYUFBYSxHQUFHO2dCQUNkLElBQUksRUFBRSxjQUFjO2dCQUNwQixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFLGVBQWU7b0JBQy9CLGdCQUFnQixFQUFFLFVBQVU7aUJBQzdCO2FBQ0YsQ0FBQztTQUNIO1FBRUQsNEZBQTRGO1FBQzVGLDhHQUE4RztRQUM5Ryw0Q0FBNEM7UUFDNUMsSUFBSSxDQUFDLG9CQUFvQixTQUFHLEtBQUssQ0FBQyxvQkFBb0IsbUNBQ2pELEdBQUcsWUFBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsVUFBVSxLQUFLLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxVQUFVLFVBQVUsS0FBSyxDQUFDLElBQUksS0FBSyxnQ0FBZ0MsQ0FBQyxNQUFNO1lBQy9ILENBQUMsQ0FBQyxPQUFPO1lBQ1QsQ0FBQyxDQUFDLEVBQUUsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUM7UUFDL0MsSUFBSSxDQUFDLG1CQUFtQixHQUFHLDhCQUFzQixDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBRTdFLGtGQUFrRjtRQUNsRixJQUFJLG9DQUFpQixDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUM5QyxRQUFRLEVBQUU7Z0JBQ1IsT0FBTyxFQUFFLFlBQVk7Z0JBQ3JCLE1BQU0sRUFBRSx3QkFBd0I7Z0JBQ2hDLFVBQVUsRUFBRTtvQkFDVixvQkFBb0IsRUFBRSxJQUFJLENBQUMsb0JBQW9CO29CQUMvQyxlQUFlLEVBQUUsUUFBUTtvQkFDekIsb0JBQW9CLEVBQUUsYUFBYTtpQkFDcEM7Z0JBQ0QsK0RBQStEO2dCQUMvRCxrQkFBa0IsRUFBRSxxQ0FBa0IsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDO2FBQ3JFO1lBQ0QsUUFBUSxFQUFFO2dCQUNSLE9BQU8sRUFBRSxZQUFZO2dCQUNyQixNQUFNLEVBQUUsd0JBQXdCO2dCQUNoQyxVQUFVLEVBQUU7b0JBQ1Ysb0JBQW9CLEVBQUUsSUFBSSxDQUFDLG9CQUFvQjtvQkFDL0MsZUFBZSxFQUFFLFFBQVE7b0JBQ3pCLG9CQUFvQixFQUFFLGFBQWE7aUJBQ3BDO2dCQUNELDJFQUEyRTtnQkFDM0UseURBQXlEO2dCQUN6RCxrQkFBa0IsRUFBRSxxQ0FBa0IsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDO2FBQ3JFO1lBQ0QsUUFBUSxFQUFFO2dCQUNSLE9BQU8sRUFBRSxZQUFZO2dCQUNyQixNQUFNLEVBQUUsd0JBQXdCO2dCQUNoQyxVQUFVLEVBQUU7b0JBQ1Ysb0JBQW9CLEVBQUUsSUFBSSxDQUFDLG9CQUFvQjtpQkFDaEQ7YUFDRjtZQUNELE1BQU0sRUFBRSwwQ0FBdUIsQ0FBQyxZQUFZLENBQUM7Z0JBQzNDLFNBQVMsRUFBRSwwQ0FBdUIsQ0FBQyxZQUFZO2FBQ2hELENBQUM7U0FDSCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsMkVBQTJFO0lBQ25FLGtCQUFrQixDQUFDLEtBQXdDO1FBQ2pFLElBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksRUFBRSxDQUFDLEVBQUc7WUFDeEQsTUFBTSxJQUFJLEtBQUssQ0FDYix5Q0FBeUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUU7NkRBQ2YsQ0FBQyxDQUFDO1NBQzFEO1FBQ0QsSUFBSSxLQUFLLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxHQUFHLElBQUksRUFBRTtZQUNyQyxNQUFNLElBQUksS0FBSyxDQUNiLHVDQUF1QyxLQUFLLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRTtrRkFDVSxDQUFDLENBQUM7U0FDL0U7SUFDSCxDQUFDOztBQTlGSCxvRUErRkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBEdXJhdGlvbiwgTmFtZXMsIFJlc291cmNlIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBBd3NDdXN0b21SZXNvdXJjZSwgQXdzQ3VzdG9tUmVzb3VyY2VQb2xpY3ksIFBoeXNpY2FsUmVzb3VyY2VJZCB9IGZyb20gJ0Bhd3MtY2RrL2N1c3RvbS1yZXNvdXJjZXMnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBhcm5Gb3JEZXBsb3ltZW50Q29uZmlnIH0gZnJvbSAnLi4vdXRpbHMnO1xuaW1wb3J0IHsgSUxhbWJkYURlcGxveW1lbnRDb25maWcgfSBmcm9tICcuL2RlcGxveW1lbnQtY29uZmlnJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gQ3VzdG9tTGFtYmRhRGVwbG95bWVudENvbmZpZ1R5cGUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIENBTkFSWSA9ICdDYW5hcnknLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgTElORUFSID0gJ0xpbmVhcidcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnUHJvcHMge1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0eXBlOiBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnVHlwZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcGVyY2VudGFnZTogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbnRlcnZhbDogRHVyYXRpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlcGxveW1lbnRDb25maWdOYW1lPzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQ3VzdG9tTGFtYmRhRGVwbG95bWVudENvbmZpZyBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUxhbWJkYURlcGxveW1lbnRDb25maWcge1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGRlcGxveW1lbnRDb25maWdOYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBkZXBsb3ltZW50Q29uZmlnQXJuOiBzdHJpbmc7XG5cbiAgcHVibGljIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgIHRoaXMudmFsaWRhdGVQYXJhbWV0ZXJzKHByb3BzKTtcblxuICAgIC8vIEluIHRoaXMgc2VjdGlvbiB3ZSBtYWtlIHRoZSBhcmd1bWVudCBmb3IgdGhlIEFXUyBBUEkgY2FsbFxuICAgIGNvbnN0IGRlcGxveW1lbnRUeXBlID0gJ1RpbWVCYXNlZCcgKyBwcm9wcy50eXBlLnRvU3RyaW5nKCk7XG4gICAgY29uc3QgaW50ZXJ2YWxNaW51dGVzID0gcHJvcHMuaW50ZXJ2YWwudG9NaW51dGVzKCkudG9TdHJpbmcoKTtcbiAgICBjb25zdCBwZXJjZW50YWdlID0gcHJvcHMucGVyY2VudGFnZS50b1N0cmluZygpO1xuICAgIGxldCByb3V0aW5nQ29uZmlnOyAvLyBUaGUgYXJndW1lbnQgdG8gdGhlIEFXUyBBUEkgY2FsbFxuICAgIGlmIChwcm9wcy50eXBlID09IEN1c3RvbUxhbWJkYURlcGxveW1lbnRDb25maWdUeXBlLkNBTkFSWSkge1xuICAgICAgcm91dGluZ0NvbmZpZyA9IHtcbiAgICAgICAgdHlwZTogZGVwbG95bWVudFR5cGUsXG4gICAgICAgIHRpbWVCYXNlZENhbmFyeToge1xuICAgICAgICAgIGNhbmFyeUludGVydmFsOiBpbnRlcnZhbE1pbnV0ZXMsXG4gICAgICAgICAgY2FuYXJ5UGVyY2VudGFnZTogcGVyY2VudGFnZSxcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgfSBlbHNlIGlmIChwcm9wcy50eXBlID09IEN1c3RvbUxhbWJkYURlcGxveW1lbnRDb25maWdUeXBlLkxJTkVBUikge1xuICAgICAgcm91dGluZ0NvbmZpZyA9IHtcbiAgICAgICAgdHlwZTogZGVwbG95bWVudFR5cGUsXG4gICAgICAgIHRpbWVCYXNlZExpbmVhcjoge1xuICAgICAgICAgIGxpbmVhckludGVydmFsOiBpbnRlcnZhbE1pbnV0ZXMsXG4gICAgICAgICAgbGluZWFyUGVyY2VudGFnZTogcGVyY2VudGFnZSxcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgfVxuXG4gICAgLy8gR2VuZXJhdGVzIHRoZSBuYW1lIG9mIHRoZSBkZXBsb3ltZW50IGNvbmZpZy4gSXQncyBhbHNvIHdoYXQgeW91J2xsIHNlZSBpbiB0aGUgQVdTIGNvbnNvbGVcbiAgICAvLyBUaGUgbmFtZSBvZiB0aGUgY29uZmlnIGlzIDxjb25zdHJ1Y3QgdW5pcXVlIGlkPi5MYW1iZGE8ZGVwbG95bWVudCB0eXBlPjxwZXJjZW50YWdlPlBlcmNlbnQ8aW50ZXJ2YWw+TWludXRlc1xuICAgIC8vIFVubGVzcyB0aGUgdXNlciBwcm92aWRlcyBhbiBleHBsaWNpdCBuYW1lXG4gICAgdGhpcy5kZXBsb3ltZW50Q29uZmlnTmFtZSA9IHByb3BzLmRlcGxveW1lbnRDb25maWdOYW1lXG4gICAgICA/PyBgJHtOYW1lcy51bmlxdWVJZCh0aGlzKX0uTGFtYmRhJHtwcm9wcy50eXBlfSR7cHJvcHMucGVyY2VudGFnZX1QZXJjZW50JHtwcm9wcy50eXBlID09PSBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnVHlwZS5MSU5FQVJcbiAgICAgICAgPyAnRXZlcnknXG4gICAgICAgIDogJyd9JHtwcm9wcy5pbnRlcnZhbC50b01pbnV0ZXMoKX1NaW51dGVzYDtcbiAgICB0aGlzLmRlcGxveW1lbnRDb25maWdBcm4gPSBhcm5Gb3JEZXBsb3ltZW50Q29uZmlnKHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUpO1xuXG4gICAgLy8gVGhlIEFXUyBDdXN0b20gUmVzb3VyY2UgdGhhdCBjYWxscyBDb2RlRGVwbG95IHRvIGNyZWF0ZSBhbmQgZGVsZXRlIHRoZSByZXNvdXJjZVxuICAgIG5ldyBBd3NDdXN0b21SZXNvdXJjZSh0aGlzLCAnRGVwbG95bWVudENvbmZpZycsIHtcbiAgICAgIG9uQ3JlYXRlOiB7IC8vIFJ1biBvbiBjcmVhdGlvbiBvbmx5LCB0byBtYWtlIHRoZSByZXNvdXJjZVxuICAgICAgICBzZXJ2aWNlOiAnQ29kZURlcGxveScsXG4gICAgICAgIGFjdGlvbjogJ2NyZWF0ZURlcGxveW1lbnRDb25maWcnLFxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgZGVwbG95bWVudENvbmZpZ05hbWU6IHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUsXG4gICAgICAgICAgY29tcHV0ZVBsYXRmb3JtOiAnTGFtYmRhJyxcbiAgICAgICAgICB0cmFmZmljUm91dGluZ0NvbmZpZzogcm91dGluZ0NvbmZpZyxcbiAgICAgICAgfSxcbiAgICAgICAgLy8gVGhpcyBgcmVzb3VyY2VOYW1lYCBpcyB0aGUgaW5pdGlhbCBwaHlzaWNhbCBJRCBvZiB0aGUgY29uZmlnXG4gICAgICAgIHBoeXNpY2FsUmVzb3VyY2VJZDogUGh5c2ljYWxSZXNvdXJjZUlkLm9mKHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUpLFxuICAgICAgfSxcbiAgICAgIG9uVXBkYXRlOiB7IC8vIFJ1biBvbiBzdGFjayB1cGRhdGVcbiAgICAgICAgc2VydmljZTogJ0NvZGVEZXBsb3knLFxuICAgICAgICBhY3Rpb246ICdjcmVhdGVEZXBsb3ltZW50Q29uZmlnJyxcbiAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgIGRlcGxveW1lbnRDb25maWdOYW1lOiB0aGlzLmRlcGxveW1lbnRDb25maWdOYW1lLFxuICAgICAgICAgIGNvbXB1dGVQbGF0Zm9ybTogJ0xhbWJkYScsXG4gICAgICAgICAgdHJhZmZpY1JvdXRpbmdDb25maWc6IHJvdXRpbmdDb25maWcsXG4gICAgICAgIH0sXG4gICAgICAgIC8vIElmIGByZXNvdXJjZU5hbWVgIGlzIGRpZmZlcmVudCBmcm9tIHRoZSBsYXN0IHN0YWNrIHVwZGF0ZSAob3IgY3JlYXRpb24pLFxuICAgICAgICAvLyB0aGUgb2xkIGNvbmZpZyBnZXRzIGRlbGV0ZWQgYW5kIHRoZSBuZXcgb25lIGlzIGNyZWF0ZWRcbiAgICAgICAgcGh5c2ljYWxSZXNvdXJjZUlkOiBQaHlzaWNhbFJlc291cmNlSWQub2YodGhpcy5kZXBsb3ltZW50Q29uZmlnTmFtZSksXG4gICAgICB9LFxuICAgICAgb25EZWxldGU6IHsgLy8gUnVuIG9uIGRlbGV0aW9uLCBvciBvbiByZXNvdXJjZSByZXBsYWNlbWVudFxuICAgICAgICBzZXJ2aWNlOiAnQ29kZURlcGxveScsXG4gICAgICAgIGFjdGlvbjogJ2RlbGV0ZURlcGxveW1lbnRDb25maWcnLFxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgZGVwbG95bWVudENvbmZpZ05hbWU6IHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgcG9saWN5OiBBd3NDdXN0b21SZXNvdXJjZVBvbGljeS5mcm9tU2RrQ2FsbHMoe1xuICAgICAgICByZXNvdXJjZXM6IEF3c0N1c3RvbVJlc291cmNlUG9saWN5LkFOWV9SRVNPVVJDRSxcbiAgICAgIH0pLFxuICAgIH0pO1xuICB9XG5cbiAgLy8gVmFsaWRhdGUgdGhlIGlucHV0cy4gVGhlIHBlcmNlbnRhZ2UvaW50ZXJ2YWwgbGltaXRzIGNvbWUgZnJvbSBDb2RlRGVwbG95XG4gIHByaXZhdGUgdmFsaWRhdGVQYXJhbWV0ZXJzKHByb3BzOiBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnUHJvcHMpOiB2b2lkIHtcbiAgICBpZiAoICEoMSA8PSBwcm9wcy5wZXJjZW50YWdlICYmIHByb3BzLnBlcmNlbnRhZ2UgPD0gOTkpICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgSW52YWxpZCBkZXBsb3ltZW50IGNvbmZpZyBwZXJjZW50YWdlIFwiJHtwcm9wcy5wZXJjZW50YWdlLnRvU3RyaW5nKCl9XCIuIFxcXG4gICAgICAgIFN0ZXAgcGVyY2VudGFnZSBtdXN0IGJlIGFuIGludGVnZXIgYmV0d2VlbiAxIGFuZCA5OS5gKTtcbiAgICB9XG4gICAgaWYgKHByb3BzLmludGVydmFsLnRvTWludXRlcygpID4gMjg4MCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgSW52YWxpZCBkZXBsb3ltZW50IGNvbmZpZyBpbnRlcnZhbCBcIiR7cHJvcHMuaW50ZXJ2YWwudG9TdHJpbmcoKX1cIi4gXFxcbiAgICAgICAgVHJhZmZpYyBzaGlmdGluZyBpbnRlcnZhbHMgbXVzdCBiZSBwb3NpdGl2ZSBpbnRlZ2VycyB1cCB0byAyODgwICgyIGRheXMpLmApO1xuICAgIH1cbiAgfVxufVxuIl19